/*------------------------------------------------------------------------------

*

*   Filename:  705L16.h

*   

*   Current Revision: 1.0

*   Current Revision Release Date: 10/06/98

*

*   Description:    This file contains the memory map definition and the register

*                   bit definition for the MC68HC705L16 microcontroller, as defined

*                   in the databook HC05L16GRS/D Rev 2.0.

*

*   This header file has been validated with the following compilers:

*       Archemedies IDE HC05 compiler, Rev 2.7.10

*       Cosmic MC68HC05 compiler, Rev 4.1c - must use #pragma space [] @near setting

*

*   This file is provided "as is" without warranty of any kind either expressed or

*   implied, including, but not limited to any warranties of merchantability and

*   fitness for a particular purpose.  All risks of using this product including

*   the entire costs of any necessary remedies are those of the user and MOTOROLA

*   assumes no liability of any kind.

*   

*   

*   History:

*

*   10/06/98    rsf     Created

*

------------------------------------------------------------------------------*/



/*--------------------------------------

*   General types that are needed

--------------------------------------*/



typedef char                    BYTE;

typedef unsigned char           UBYTE;

typedef volatile unsigned char  VUBYTE;



/*------------------------------------------------------------------------------

*   Memory Map definition of I/O registers, as a structure.

*   

*   

*   This structure is defined as a type IOREGS.  There is also a label defined, LMREGS, that

*   you can make use of in your code to reference the structure.  This structure is actually

*   a union of two structures, one regular register map and one optional map.  See the example

*   code below the structure definition.

*

------------------------------------------------------------------------------*/



typedef struct ioregs {

union {

	struct reg {

    VUBYTE  PORTA;          /* Port A Data Register */

    VUBYTE  PORTB;          /* Port B Data Register */

    VUBYTE  PORTC;          /* Port C Data Register */

    VUBYTE  PORTD;          /* Port D Data Register */

    VUBYTE  PORTE;          /* Port E Data Register */

    VUBYTE  RESERVED1[3];

    VUBYTE  INTCR;          /* Interrupt Control Register */

    VUBYTE  INTSR;          /* Interrupt Status Register */

    VUBYTE  SPCR;           /* Serial Peripheral Control Register */

    VUBYTE  SPSR;           /* Swrial Peripheral Status Register */

    VUBYTE  SPDR;           /* Serial Peripheral Data Register */

    VUBYTE  RESERVED2[3];

    VUBYTE  TBCR1;          /* Timer Base Control Register 1 */

    VUBYTE  TBCR2;          /* Timer Base Control Register 2 */

    VUBYTE  TCR;            /* Timer Control Register */

    VUBYTE  TSR;            /* Timer Status Register */

    VUBYTE  ICH;            /* Input Capture Register High */

    VUBYTE  ICL;            /* Input Capture Register Low */

    VUBYTE  OC1H;           /* Output Compare Register 1 High */

    VUBYTE  OC1L;           /* Output Compare Register 1 Low */

    VUBYTE  TCNTH;          /* Timer Counter Register High */

    VUBYTE  TCNTL;          /* Timer Counter Register Low */

    VUBYTE  ACNTH;          /* Alternate Timer Counter Register High */

    VUBYTE  ACNTL;          /* Alternate Timer Counter Register Low */

    VUBYTE  TCR2;           /* Timer Control Register 2 */

    VUBYTE  TSR2;           /* Timer Status Register 2 */

    VUBYTE  OC2;            /* Output Compare Register 2 */

    VUBYTE  TCNT2;          /* Timer Counter Register 2 */

    

    VUBYTE  LCDCR;          /* LCD Control Register */

    VUBYTE  LCDR1;          /* LCD Data Register 1 */

    VUBYTE  LCDR2;          /* LCD Data Register 2 */

    VUBYTE  LCDR3;          /* LCD Data Register 3 */

    VUBYTE  LCDR4;          /* LCD Data Register 4 */

    VUBYTE  LCDR5;          /* LCD Data Register 5 */

    VUBYTE  LCDR6;          /* LCD Data Register 6 */

    VUBYTE  LCDR7;          /* LCD Data Register 7 */

    VUBYTE  LCDR8;          /* LCD Data Register 8 */

    VUBYTE  LCDR9;          /* LCD Data Register 9 */

    VUBYTE  LCDR10;         /* LCD Data Register 10 */

    VUBYTE  LCDR11;         /* LCD Data Register 11 */

    VUBYTE  LCDR12;         /* LCD Data Register 12 */

    VUBYTE  LCDR13;         /* LCD Data Register 13 */

    VUBYTE  LCDR14;         /* LCD Data Register 14 */

    VUBYTE  LCDR15;         /* LCD Data Register 15 */

    VUBYTE  LCDR16;         /* LCD Data Register 16 */

    VUBYTE  LCDR17;         /* LCD Data Register 17 */

    VUBYTE  LCDR18;         /* LCD Data Register 18 */

    VUBYTE  LCDR19;         /* LCD Data Register 19 */

    VUBYTE  LCDR20;         /* LCD Data Register 20 */

    VUBYTE  RESERVED3[9];

    VUBYTE  MISC;           /* Miscellaneous Register */

    VUBYTE  RESERVED4;

	} REG;

	struct opt {

    VUBYTE  DDRA;           /* Port A Data Direction Register */

    VUBYTE  RESERVED5;

    VUBYTE  DDRC;           /* Port C Data Direction Register */

    VUBYTE  PDMUX;          /* Port D MUX Register */

    VUBYTE  PEMUX;          /* Port E MUX Register */

    VUBYTE  RESERVED6[3];

    VUBYTE  RCR1;           /* Resistor Control Register 1 */

    VUBYTE  RCR2;           /* Resistor Control Register 2 */

    VUBYTE  WOM1;           /* Open-Drain Output Control Register 1 */

    VUBYTE  WOM2;           /* Open-Drain Output Control Register 2 */

    VUBYTE  RESERVED7[2];

    VUBYTE  KWIEN;          /* Key Wakeup Input Enable Register */

    VUBYTE  MOR;            /* Mask Option Status Register */

	} OPT;

} U;

} IOREGS;



#define LMREGS  (*(IOREGS *)(0x0000))   /* LMREGS (i.e. low memory registers) defines the */

                                        /* IOREGS structure at its natural base in memory */

                                        

/*------------------------------------------------------------------------------

*   Here is an example of how to use this structure definition in your code:

------------------------------------------------------------------------------*/

#ifdef  not_def

   

   UBYTE   tempa, tempb;         /* allocate some variables */

   

   LMREGS.U.REG.MISC &= !OPTM;         /* clear the option map bit so we access the regular register map */

   LMREGS.U.REG.PORTA = PA4 | PA3;     /* set bits 4 and 3 in port a so we can use them as outputs */

   

   LMREGS.U.REG.MISC != OPTM;          /* set the option map bit so we access the option register map */

   LMREGS.U.OPT.DDRA = DDRA4 | DDRA3;  /* make bits 4 and 3 in port a output bits */



   LMREGS.U.REG.MISC &= !OPTM;         /* clear the option map bit so we access the regular register map */

   LMREGS.U.REG.PORTC = PC6 | PC5;     /* set bits 6 and 5 in port c so we can use them as outputs */



   LMREGS.U.REG.MISC != OPTM;          /* set the option map bit so we access the option register map */

   LMREGS.U.OPT.DDRC = DDRC6 | DDRC5;  /* make bits 6 and 5 in port c output bits */

   

   LMREGS.U.REG.MISC &= !OPTM;         /* clear the option map bit so we access the regular register map */

   tempa = LMREGS.U.REG.PORTA          /* read port a */

   tempb = LMREGS.U.REG.PORTB          /* read port b */

   

   /* set PB6:PB5 to the value of PA1:PA0 */

   LMREGS.U.REG.PORTB = (LMREGS.U.REG.PORTA & (PA1 | PA0)) << B_PB5;



#endif       



/*--------------------------------------

*   Memory Map definition, as constants.

*

*   

*   This time each register is defined as a constant pointer value, you can make

*   use of this in your code by simply referencing the value at the pointer.

*   See the example code below the memory definitions.

*   

--------------------------------------*/



/*--------------------------------------

*   I/O Registers

--------------------------------------*/



#define P_PORTA     ((VUBYTE *)(0x0000))    /* Port A Data Register */

#define P_PORTB     ((VUBYTE *)(0x0001))    /* Port B Data Register */

#define P_PORTC     ((VUBYTE *)(0x0002))    /* Port C Data Register */

#define P_PORTD     ((VUBYTE *)(0x0003))    /* Port D Data Register */

#define P_PORTE     ((VUBYTE *)(0x0004))    /* Port E Data Register */

#define P_INTCR     ((VUBYTE *)(0x0008))    /* Interrupt Control Register */

#define P_INTSR     ((VUBYTE *)(0x0009))    /* Interrupt Status Register */

#define P_SPCR      ((VUBYTE *)(0x000a))    /* Serial Peripheral Control Register */

#define P_SPSR      ((VUBYTE *)(0x000b))    /* Swrial Peripheral Status Register */

#define P_SPDR      ((VUBYTE *)(0x000c))    /* Serial Peripheral Data Register */

#define P_TBCR1     ((VUBYTE *)(0x0010))    /* Timer Base Control Register 1 */

#define P_TBCR2     ((VUBYTE *)(0x0011))    /* Timer Base Control Register 2 */

#define P_TCR       ((VUBYTE *)(0x0012))    /* Timer Control Register */

#define P_TSR       ((VUBYTE *)(0x0013))    /* Timer Status Register */

#define P_ICH       ((VUBYTE *)(0x0014))    /* Input Capture Register High */

#define P_ICL       ((VUBYTE *)(0x0015))    /* Input Capture Register Low */

#define P_OC1H      ((VUBYTE *)(0x0016))    /* Output Compare Register 1 High */

#define P_OC1L      ((VUBYTE *)(0x0017))    /* Output Compare Register 1 Low */

#define P_TCNTH     ((VUBYTE *)(0x0018))    /* Timer Counter Register High */

#define P_TCNTL     ((VUBYTE *)(0x0019))    /* Timer Counter Register Low */

#define P_ACNTH     ((VUBYTE *)(0x001a))    /* Alternate Timer Counter Register High */

#define P_ACNTL     ((VUBYTE *)(0x001b))    /* Alternate Timer Counter Register Low */

#define P_TCR2      ((VUBYTE *)(0x001c))    /* Timer Control Register 2 */

#define P_TSR2      ((VUBYTE *)(0x001d))    /* Timer Status Register 2 */

#define P_OC2       ((VUBYTE *)(0x001e))    /* Output Compare Register 2 */

#define P_TCNT2     ((VUBYTE *)(0x001f))    /* Timer Counter Register 2 */

    

#define P_LCDCR     ((VUBYTE *)(0x0020))    /* LCD Control Register */

#define P_LCDR1     ((VUBYTE *)(0x0021))    /* LCD Data Register 1 */

#define P_LCDR2     ((VUBYTE *)(0x0022))    /* LCD Data Register 2 */

#define P_LCDR3     ((VUBYTE *)(0x0023))    /* LCD Data Register 3 */

#define P_LCDR4     ((VUBYTE *)(0x0024))    /* LCD Data Register 4 */

#define P_LCDR5     ((VUBYTE *)(0x0025))    /* LCD Data Register 5 */

#define P_LCDR6     ((VUBYTE *)(0x0026))    /* LCD Data Register 6 */

#define P_LCDR7     ((VUBYTE *)(0x0027))    /* LCD Data Register 7 */

#define P_LCDR8     ((VUBYTE *)(0x0028))    /* LCD Data Register 8 */

#define P_LCDR9     ((VUBYTE *)(0x0029))    /* LCD Data Register 9 */

#define P_LCDR10    ((VUBYTE *)(0x002a))    /* LCD Data Register 10 */

#define P_LCDR11    ((VUBYTE *)(0x002b))    /* LCD Data Register 11 */

#define P_LCDR12    ((VUBYTE *)(0x002c))    /* LCD Data Register 12 */

#define P_LCDR13    ((VUBYTE *)(0x002d))    /* LCD Data Register 13 */

#define P_LCDR14    ((VUBYTE *)(0x002e))    /* LCD Data Register 14 */

#define P_LCDR15    ((VUBYTE *)(0x002f))    /* LCD Data Register 15 */

#define P_LCDR16    ((VUBYTE *)(0x0030))    /* LCD Data Register 16 */

#define P_LCDR17    ((VUBYTE *)(0x0031))    /* LCD Data Register 17 */

#define P_LCDR18    ((VUBYTE *)(0x0032))    /* LCD Data Register 18 */

#define P_LCDR19    ((VUBYTE *)(0x0033))    /* LCD Data Register 19 */

#define P_LCDR20    ((VUBYTE *)(0x0034))    /* LCD Data Register 20 */

#define P_MISC      ((VUBYTE *)(0x003e))    /* Miscellaneous Register */



#define P_DDRA      ((VUBYTE *)(0x0000))    /* Port A Data Direction Register */

#define P_DDRC      ((VUBYTE *)(0x0002))    /* Port C Data Direction Register */

#define P_PDMUX     ((VUBYTE *)(0x0003))    /* Port D MUX Register */

#define P_PEMUX     ((VUBYTE *)(0x0004))    /* Port E MUX Register */

#define P_RCR1      ((VUBYTE *)(0x0008))    /* Resistor Control Register 1 */

#define P_RCR2      ((VUBYTE *)(0x0009))    /* Resistor Control Register 2 */

#define P_WOM1      ((VUBYTE *)(0x000a))    /* Open-Drain Output Control Register 1 */

#define P_WOM2      ((VUBYTE *)(0x000b))    /* Open-Drain Output Control Register 2 */

#define P_KWIEN     ((VUBYTE *)(0x000e))    /* Key Wakeup Input Enable Register */

#define P_MOR       ((VUBYTE *)(0x000f))    /* Mask Option Status Register */

    

/*--------------------------------------

*   Exception Vectors

--------------------------------------*/



#define P_TBIV_H    ((UBYTE *)(0xfff0))     /* Time Base Interrupt Vector High */

#define P_TBIV_L    ((UBYTE *)(0xfff1))     /* Time Base Interrupt Vector Low */

#define P_SSPIV_H   ((UBYTE *)(0xfff2))     /* SSPI Interrupt Vector High */

#define P_SSPIV_L   ((UBYTE *)(0xfff3))     /* SSPI Interrupt Vector Low */

#define P_TMR2IV_H  ((UBYTE *)(0xfff4))     /* Timer 2 Interrupt Vector High */

#define P_TMR2IV_L  ((UBYTE *)(0xfff5))     /* Timer 2 Interrupt Vector Low */

#define P_TMR1IV_H  ((UBYTE *)(0xfff6))     /* Timer 1 Interrupt Vector High */

#define P_TMR1IV_L  ((UBYTE *)(0xfff7))     /* Timer 1 Interrupt Vector Low */

#define P_KWIV_H    ((UBYTE *)(0xfff8))     /* Key Wake Up Interrupt Vector High */

#define P_KWIV_L    ((UBYTE *)(0xfff9))     /* Key Wake Up Interrupt Vector Low */

#define P_EIV_H     ((UBYTE *)(0xfffa))     /* External Interrupt Vector High */

#define P_EIV_L     ((UBYTE *)(0xfffb))     /* External Interrupt Vector Low */

#define P_SIV_H     ((UBYTE *)(0xfffc))     /* Software Interrupt Vector High */

#define P_SIV_L     ((UBYTE *)(0xfffd))     /* Software Interrupt Vector Low */

#define P_RESET_H   ((UBYTE *)(0xfffe))     /* Reset Vector High */

#define P_RESET_L   ((UBYTE *)(0xffff))     /* Reset Vector Low */



/*--------------------------------------

*   Memory Regions

--------------------------------------*/



#define P_RAMLO_BASE    ((UBYTE *)(0x0040))     /* first address of LO RAM */

#define P_RAMLO_TOP     ((UBYTE *)(0x00bf))     /* last address of LO RAM */

#define P_STACK_BASE    ((UBYTE *)(0x00c0))     /* first address of STACK */

#define P_STACK_TOP     ((UBYTE *)(0x00ff))     /* last address of STACK */   

#define P_RAMHI_BASE    ((UBYTE *)(0x0100))     /* first address of HI RAM */

#define P_RAMHI_TOP     ((UBYTE *)(0x023f))     /* last address of HI RAM */

#define P_EPROM_BASE    ((UBYTE *)(0x1000))     /* first address of EPROM */

#define P_EPROM_TOP     ((UBYTE *)(0x4fff))     /* last address of EPROM */



/*------------------------------------------------------------------------------

*   Here is an example of how to use these pointer definitions in your code:

------------------------------------------------------------------------------*/

#ifdef  not_def

   

   UBYTE   tempa, tempb;       /* allocate some variables */

   

   *P_MISC &= !OPTM;           /* clear the OPTM bit to select the regular register map */

   *P_PORTA = PA4 | PA3;       /* set bits 4 and 3 in port a so we can use them as outputs */



   *P_MISC |= OPTM;            /* set the OPTM bit to select the option register map */

   *P_DDRA = DDRA4 | DDRA3;    /* make bits 4 and 3 in port a output bits */



   *P_MISC &= !OPTM;           /* clear the OPTM bit to select the regular register map */

   *P_PORTC = PC6 | PC5;       /* set bits 6 and 5 in port c so we can use them as outputs */



   *P_MISC |= OPTM;            /* set the OPTM bit to select the option register map */

   *P_DDRC = DDRC6 | DDRC5;    /* make bits 6 and 5 in port c output bits */

   

   *P_MISC &= !OPTM;           /* clear the OPTM bit to select the regular register map */

   tempa = *P_PORTA                                /* read port a */

   tempb = *P_PORTB                                /* read port b */

   *P_PORTB = (*P_PORTA & (PA1 | PA0)) << B_PB5;   /* set PB6:PB5 to the value of PA1:PA0 */

   

#endif



/*------------------------------------------------------------------------------

*   Register Definitions

*   

*   The registers are defined in the same order as they seqeuntially appear in

*   the memory map.

------------------------------------------------------------------------------*/



/*--------------------------------------

*   PORTA

*   Port A Data Register

*   

*   Address: 0x0000

*   Manual Page: 75

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PA0 - Port A Data Bit 0

*     |   |   |   |   |   |   |-------- PA1 - Port A Data Bit 1

*     |   |   |   |   |   |------------ PA2 - Port A Data Bit 2

*     |   |   |   |   |---------------- PA3 - Port A Data Bit 3

*     |   |   |   |-------------------- PA4 - Port A Data Bit 4

*     |   |   |------------------------ PA5 - Port A Data Bit 5

*     |   |---------------------------- PA6 - Port A Data Bit 6

*     |-------------------------------- PA7 - Port A Data Bit 7

*     

*   These read/write bits are software programmable. Data direction of each port

*   A pin is under the control of the corresponding bit in data direction register A.

*   Reset has no effect on port A data.  The high nibble or the low nibble can be

*   optionally enabled as open drain outputs in the WOM1 register.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PA0 0x01    /* Port A Data Bit 0 */

#define PA1 0x02    /* Port A Data Bit 1 */

#define PA2 0x04    /* Port A Data Bit 2 */

#define PA3 0x08    /* Port A Data Bit 3 */

#define PA4 0x10    /* Port A Data Bit 4 */

#define PA5 0x20    /* Port A Data Bit 5 */

#define PA6 0x40    /* Port A Data Bit 6 */

#define PA7 0x80    /* Port A Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PA0   0   /* Port A Data Bit 0 */

#define B_PA1   1   /* Port A Data Bit 1 */

#define B_PA2   2   /* Port A Data Bit 2 */

#define B_PA3   3   /* Port A Data Bit 3 */

#define B_PA4   4   /* Port A Data Bit 4 */

#define B_PA5   5   /* Port A Data Bit 5 */

#define B_PA6   6   /* Port A Data Bit 6 */

#define B_PA7   7   /* Port A Data Bit 7 */



/*--------------------------------------

*   PORTB

*   Port B Data Register

*   

*   Address: 0x0001

*   Manual Page: 78

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PB0 - Port B Data Bit 0

*     |   |   |   |   |   |   |-------- PB1 - Port B Data Bit 1

*     |   |   |   |   |   |------------ PB2 - Port B Data Bit 2

*     |   |   |   |   |---------------- PB3 - Port B Data Bit 3

*     |   |   |   |-------------------- PB4 - Port B Data Bit 4

*     |   |   |------------------------ PB5 - Port B Data Bit 5

*     |   |---------------------------- PB6 - Port B Data Bit 6

*     |-------------------------------- PB7 - Port B Data Bit 7

*   

*   These read only bits indicate the state of the Port B inputs.  Each Port B

*   input can be enabled as a KWI in the KWIEN register.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PB0 0x01    /* Port B Data Bit 0 */

#define PB1 0x02    /* Port B Data Bit 1 */

#define PB2 0x04    /* Port B Data Bit 2 */

#define PB3 0x08    /* Port B Data Bit 3 */

#define PB4 0x10    /* Port B Data Bit 4 */

#define PB5 0x20    /* Port B Data Bit 5 */

#define PB6 0x40    /* Port B Data Bit 6 */

#define PB7 0x80    /* Port B Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PB0   0   /* Port B Data Bit 0 */

#define B_PB1   1   /* Port B Data Bit 1 */

#define B_PB2   2   /* Port B Data Bit 2 */

#define B_PB3   3   /* Port B Data Bit 3 */

#define B_PB4   4   /* Port B Data Bit 4 */

#define B_PB5   5   /* Port B Data Bit 5 */

#define B_PB6   6   /* Port B Data Bit 6 */

#define B_PB7   7   /* Port B Data Bit 7 */



/*--------------------------------------

*   PORTC

*   Port C Data Register

*   

*   Address: 0x0002

*   Manual Page: 79

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PC0 - Port C Data Bit 0

*     |   |   |   |   |   |   |-------- PC1 - Port C Data Bit 1

*     |   |   |   |   |   |------------ PC2 - Port C Data Bit 2

*     |   |   |   |   |---------------- PC3 - Port C Data Bit 3

*     |   |   |   |-------------------- PC4 - Port C Data Bit 4

*     |   |   |------------------------ PC5 - Port C Data Bit 5

*     |   |---------------------------- PC6 - Port C Data Bit 6

*     |-------------------------------- PC7 - Port C Data Bit 7

*     

*   These read/write bits are software programmable. Data direction of each port

*   C pin is under the control of the corresponding bit in data direction register C.

*   Reset has no effect on port C data.

*   

*   Note: The port C pins are shared with several on-chip peripherals.  Please

*         refer to the manual for details.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PC0  0x01    /* Port C Data Bit 0 */

#define PC1  0x02    /* Port C Data Bit 1 */

#define PC2  0x04    /* Port C Data Bit 2 */

#define PC3  0x08    /* Port C Data Bit 3 */

#define PC4  0x10    /* Port C Data Bit 4 */

#define PC5  0x20    /* Port C Data Bit 5 */

#define PC6  0x40    /* Port C Data Bit 6 */

#define PC7  0x80    /* Port C Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PC0   0   /* Port C Data Bit 0 */

#define B_PC1   1   /* Port C Data Bit 1 */

#define B_PC2   2   /* Port C Data Bit 2 */

#define B_PC3   3   /* Port C Data Bit 3 */

#define B_PC4   4   /* Port C Data Bit 4 */

#define B_PC5   5   /* Port C Data Bit 5 */

#define B_PC6   6   /* Port C Data Bit 6 */

#define B_PC7   7   /* Port C Data Bit 7 */



/*--------------------------------------

*   PORTD

*   Port D Data Register

*   

*   Address: 0x0003

*   Manual Page: 83

*

*    RW  RW  RW  RW  RW  RW  RW  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- PD1 - Port D Data Bit 1

*     |   |   |   |   |   |------------ PD2 - Port D Data Bit 2

*     |   |   |   |   |---------------- PD3 - Port D Data Bit 3

*     |   |   |   |-------------------- PD4 - Port D Data Bit 4

*     |   |   |------------------------ PD5 - Port D Data Bit 5

*     |   |---------------------------- PD6 - Port D Data Bit 6

*     |-------------------------------- PD7 - Port D Data Bit 7

*   

*   This port is shared  with the LCD controller, pins are either general purpose

*   output pins, or LCD functionality, determined by the PDMUX register.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PD1     0x02    /* Port D Data Bit 1 */

#define PD2     0x04    /* Port D Data Bit 2 */

#define PD3     0x08    /* Port D Data Bit 3 */

#define PD4     0x10    /* Port D Data Bit 4 */

#define PD5     0x20    /* Port D Data Bit 5 */

#define PD6     0x40    /* Port D Data Bit 6 */

#define PD7     0x80    /* Port D Data Bit 7 */



#define PORTD_UNUSED 0x01    /* unused bits in PORTD register */

/*------------------

*   Bit Positions

------------------*/

#define B_PD1   1   /* Port D Data Bit 1 */

#define B_PD2   2   /* Port D Data Bit 2 */

#define B_PD3   3   /* Port D Data Bit 3 */

#define B_PD4   4   /* Port D Data Bit 4 */

#define B_PD5   5   /* Port D Data Bit 5 */

#define B_PD6   6   /* Port D Data Bit 6 */

#define B_PD7   7   /* Port D Data Bit 7 */



/*--------------------------------------

*   PORTE

*   Port E Data Register

*   

*   Address: 0x0004

*   Manual Page: 86

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PE0 - Port E Data Bit 0

*     |   |   |   |   |   |   |-------- PE1 - Port E Data Bit 1

*     |   |   |   |   |   |------------ PE2 - Port E Data Bit 2

*     |   |   |   |   |---------------- PE3 - Port E Data Bit 3

*     |   |   |   |-------------------- PE4 - Port E Data Bit 4

*     |   |   |------------------------ PE5 - Port E Data Bit 5

*     |   |---------------------------- PE6 - Port E Data Bit 6

*     |-------------------------------- PE7 - Port E Data Bit 7

*   

*   This port is shared  with the LCD controller, pins are either general purpose

*   output pins, or LCD functionality, determined by the PEMUX register.

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PE0     0x01    /* Port E Data Bit 0 */

#define PE1     0x02    /* Port E Data Bit 1 */

#define PE2     0x04    /* Port E Data Bit 2 */

#define PE3     0x08    /* Port E Data Bit 3 */

#define PE4     0x10    /* Port E Data Bit 4 */

#define PE5     0x20    /* Port E Data Bit 5 */

#define PE6     0x40    /* Port E Data Bit 6 */

#define PE7     0x80    /* Port E Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PE0   0   /* Port E Data Bit 0 */

#define B_PE1   1   /* Port E Data Bit 1 */

#define B_PE2   2   /* Port E Data Bit 2 */

#define B_PE3   3   /* Port E Data Bit 3 */

#define B_PE4   4   /* Port E Data Bit 4 */

#define B_PE5   5   /* Port E Data Bit 5 */

#define B_PE6   6   /* Port E Data Bit 6 */

#define B_PE7   7   /* Port E Data Bit 7 */



/*--------------------------------------

*   INTCR

*   Interrupt Control Register

*   

*   Address: 0x0008

*   Manual Page: 64

*

*    RW  RW  --  RW  RW  RW  --  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ IRQ2S - IRQ2 Select Edge Sensitive Only

*     |   |   |   |   |---------------- IRQ1S - IRQ1 Select Edge Sensitive Only

*     |   |   |   |-------------------- KWIE - Key Wakeup Interrupt Enable

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- IRQ2E - IRQ2 Interrupt Enable

*     |-------------------------------- IRQ1E - IRQ1 Interrupt Enable

*     

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define IRQ2S   0x04    /* IRQ2 Select Edge Sensitive Only */

#define IRQ1S   0x08    /* IRQ1 Select Edge Sensitive Only */

#define KWIE    0x10    /* Key Wakeup Interrupt Enable */

#define IRQ2E   0x40    /* IRQ2 Interrupt Enable */

#define IRQ1E   0x80    /* IRQ1 Interrupt Enable */



#define INTCR_UNUSED 0x23   /* unused bits in INTCR register */

/*------------------

*   Bit Positions

------------------*/

#define B_IRQ2S   2    /* IRQ2 Select Edge Sensitive Only */

#define B_IRQ1S   3    /* IRQ1 Select Edge Sensitive Only */

#define B_KWIE    4    /* Key Wakeup Interrupt Enable */

#define B_IRQ2E   6    /* IRQ2 Interrupt Enable */

#define B_IRQ1E   7    /* IRQ1 Interrupt Enable */



/*--------------------------------------

*   INTSR

*   Interrupt Status Register

*   

*   Address: 0x0009

*   Manual Page: 66

*

*    RO  RO  --  RO  WO  WO  --  WO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- RKWIF - Reset KWI Flag

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ RIRQ2 - Reset IRQ2 Flag

*     |   |   |   |   |---------------- RIRQ1 - Reset IRQ1 Flag

*     |   |   |   |-------------------- KWIF - Key Wakeup Interrupt Flag

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- IRQ2F - IRQ2 Interrupt Flag

*     |-------------------------------- IRQ1F - IRQ1 Interrupt Flag

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define RKWIF   0x01    /* Reset KWI Flag */

#define RIRQ2   0x04    /* Reset IRQ2 Flag */

#define RIRQ1   0x08    /* Reset IRQ1 Flag */

#define KWIF    0x10    /* Key Wakeup Interrupt Flag */

#define IRQ2F   0x40    /* IRQ2 Interrupt Flag */

#define IRQ1F   0x80    /* IRQ1 Interrupt Flag */



#define INTSR_UNUSED 0x22   /* unused bits in INTSR register */

/*------------------

*   Bit Positions

------------------*/

#define B_RKWIF   0    /* Reset KWI Flag */

#define B_RIRQ2   2    /* Reset IRQ2 Flag */

#define B_RIRQ1   3    /* Reset IRQ1 Flag */

#define B_KWIF    4    /* Key Wakeup Interrupt Flag */

#define B_IRQ2F   6    /* IRQ2 Interrupt Flag */

#define B_IRQ1F   7    /* IRQ1 Interrupt Flag */



/*--------------------------------------

*   SPCR

*   Serial Peripheral Control Register

*   

*   Address: 0x000a

*   Manual Page: 110

*

*    RO  RO  RO  RO  --  --  --  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- SPR - SSPI Clock Rate Select

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- MSTR - Master Mode Select

*     |   |   |------------------------ DORD - Data Transmission Order

*     |   |---------------------------- SPE - SSPI Enable

*     |-------------------------------- SPIE - SSPI Interrupt Enable

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define SPR    0x01    /* SSPI Clock Rate Select */

#define MSTR   0x10    /* Master Mode Select */

#define DORD   0x20    /* Data Transmission Order */

#define SPE    0x40    /* SSPI Enable */

#define SPIE   0x80    /* SSPI Interrupt Enable */



#define SPCR_UNUSED 0x0e   /* unused bits in SPCR register */

/*------------------

*   Bit Positions

------------------*/

#define B_SPR    0    /* SSPI Clock Rate Select */

#define B_MSTR   4    /* Master Mode Select */

#define B_DORD   5    /* Data Transmission Order */

#define B_SPE    6    /* SSPI Enable */

#define B_SPIE   7    /* SSPI Interrupt Enable */



/*--------------------------------------

*   SPSR

*   Serial Peripheral Status Register

*   

*   Address: 0x000b

*   Manual Page: 112

*

*    RO  RO  --  --  --  --  --  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unlimited

*     |   |   |   |   |   |   |-------- Unlimited

*     |   |   |   |   |   |------------ Unlimited

*     |   |   |   |   |---------------- Unlimited

*     |   |   |   |-------------------- Unlimited

*     |   |   |------------------------ Unlimited

*     |   |---------------------------- DCOL - Data Collision

*     |-------------------------------- SPIF - Serial Transfer Complete Flag

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DCOL   0x40    /* Data Collision */

#define SPIF   0x80    /* Serial Transfer Complete Flag */



#define SPSR_UNUSED 0x3f    /* unused bits in the SPSR register */

/*------------------

*   Bit Positions

------------------*/

#define B_DCOL   6    /* Data Collision */

#define B_SPIF   7    /* Serial Transfer Complete Flag */



/*--------------------------------------

*   SPDR

*   Serial Peripheral Data Register

*   

*   Address: 0x000c

*   Manual Page: 113

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- SPDR0 - Serial Peripheral Data Bit 0

*     |   |   |   |   |   |   |-------- SPDR1 - Serial Peripheral Data Bit 1

*     |   |   |   |   |   |------------ SPDR2 - Serial Peripheral Data Bit 2

*     |   |   |   |   |---------------- SPDR3 - Serial Peripheral Data Bit 3

*     |   |   |   |-------------------- SPDR4 - Serial Peripheral Data Bit 4

*     |   |   |------------------------ SPDR5 - Serial Peripheral Data Bit 5

*     |   |---------------------------- SPDR6 - Serial Peripheral Data Bit 6

*     |-------------------------------- SPDR7 - Serial Peripheral Data Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define SPDR0   0x01    /* Serial Peripheral Data Bit 0 */

#define SPDR1   0x02    /* Serial Peripheral Data Bit 1 */

#define SPDR2   0x04    /* Serial Peripheral Data Bit 2 */

#define SPDR3   0x08    /* Serial Peripheral Data Bit 3 */

#define SPDR4   0x10    /* Serial Peripheral Data Bit 4 */

#define SPDR5   0x20    /* Serial Peripheral Data Bit 5 */

#define SPDR6   0x40    /* Serial Peripheral Data Bit 6 */

#define SPDR7   0x80    /* Serial Peripheral Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_SPDR0 0   /* Serial Peripheral Data Bit 0 */

#define B_SPDR1 1   /* Serial Peripheral Data Bit 1 */

#define B_SPDR2 2   /* Serial Peripheral Data Bit 2 */

#define B_SPDR3 3   /* Serial Peripheral Data Bit 3 */

#define B_SPDR4 4   /* Serial Peripheral Data Bit 4 */

#define B_SPDR5 5   /* Serial Peripheral Data Bit 5 */

#define B_SPDR6 6   /* Serial Peripheral Data Bit 6 */

#define B_SPDR7 7   /* Serial Peripheral Data Bit 7 */



/*--------------------------------------

*   TBCR1

*   Time Base Control Register 1

*   

*   Address: 0x0010

*   Manual Page: 98

*

*    RW  --  RW  --  --  --  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- T2R0 - Timer 2 Prescale Rate Select Bit 0

*     |   |   |   |   |   |   |-------- T2R1 - Timer 2 Prescale Rate Select Bit 1

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ LCLK - LCD Clock

*     |   |---------------------------- Unimplemented

*     |-------------------------------- TBCLK - Time Base Clock

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define T2R0    0x01    /* Timer 2 Prescale Rate Select Bit 0 */

#define T2R1    0x02    /* Timer 2 Prescale Rate Select Bit 1 */

#define LCLK    0x20    /* LCD Clock */

#define TBCLK   0x80    /* Time Base Clock */



#define TBCR1_UNUSED 0x5c    /* unused bits in TBCR1 register */

/*------------------

*   Bit Positions

------------------*/

#define B_T2R0    0    /* Timer 2 Prescale Rate Select Bit 0 */

#define B_T2R1    1    /* Timer 2 Prescale Rate Select Bit 1 */

#define B_LCLK    5    /* LCD Clock */

#define B_TBCLK   7    /* Time Base Clock */



/*--------------------------------------

*   TBCR2

*   Time Base Control Register 2

*   

*   Address: 0x0011

*   Manual Page: 99

*

*    RW  RW  RW  RW  RW  --  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- COPC - COP Clear

*     |   |   |   |   |   |   |-------- COPE - COP Enable

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- RTBIF - Reset TBS Interrupt Flag

*     |   |   |   |-------------------- TBR0 - Time Base Interrupt Rate Select Bit 0

*     |   |   |------------------------ TBR1 - Time Base Interrupt Rate Select Bit 1

*     |   |---------------------------- TBIE - Time Base Interrupt Enable

*     |-------------------------------- TBIF - Time Base Interrupt Flag

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define COPC   0x01    /* COP Clear */

#define COPE   0x02    /* COP Enable */

#define RTBIF  0x08    /* Reset TBS Interrupt Flag */

#define TBR0   0x10    /* Time Base Interrupt Rate Select Bit 0 */

#define TBR1   0x20    /* Time Base Interrupt Rate Select Bit 1 */

#define TBIE   0x40    /* Time Base Interrupt Enable */

#define TBIF   0x80    /* Time Base Interrupt Flag */



#define TBCR2_UNUSED 0x04   /* unused bits in TBCR2 register */

/*------------------

*   Bit Positions

------------------*/

#define B_COPC   0    /* COP Clear */

#define B_COPE   1    /* COP Enable */

#define B_RTBIF  3    /* Reset TBS Interrupt Flag */

#define B_TBR0   4    /* Time Base Interrupt Rate Select Bit 0 */

#define B_TBR1   5    /* Time Base Interrupt Rate Select Bit 1 */

#define B_TBIE   6    /* Time Base Interrupt Enable */

#define B_TBIF   7    /* Time Base Interrupt Flag */



/*--------------------------------------

*   TCR

*   Timer Control Register

*   

*   Address: 0x0012

*   Manual Page: 121

*

*    RW  RW  RW  --  --  --  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OLVL - Output Level

*     |   |   |   |   |   |   |-------- IEDG - Input Edge

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ TOIE - Timer Overflow Interrupt Enable

*     |   |---------------------------- OC1IE - Output Compare 1 Interrupt Enable

*     |-------------------------------- ICIE - Input Capture Interrupt Enable

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OLVL   0x01    /* Output Level */

#define IEDG   0x02    /* Input Edge */

#define TOIE   0x20    /* Timer Overflow Interrupt Enable */

#define OC1IE  0x40    /* Output Compare 1 Interrupt Enable */

#define ICIE   0x80    /* Input Capture Interrupt Enable */



#define TCR_UNUSED 0x1c   /* unused bits in the TCR register */

/*------------------

*   Bit Positions

------------------*/

#define B_OLVL   0   /* Output Level */

#define B_IEDG   1   /* Input Edge */

#define B_TOIE   5   /* Timer Overflow Interrupt Enable */

#define B_OC1IE  6   /* Output Compare 1 Interrupt Enable */

#define B_ICIE   7   /* Input Capture Interrupt Enable */



/*--------------------------------------

*   TSR

*   Timer Status Register

*   

*   Address: 0x0013

*   Manual Page: 122

*

*    RO  RO  RO  --  --  --  --  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ TOF - Timer Overflow Flag

*     |   |---------------------------- OC1F - Output Compare 1 Flag

*     |-------------------------------- ICF - Input Capture Flag

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define TOF  0x20    /* Timer Overflow Flag */

#define OC1F 0x40    /* Output Compare 1 Flag */

#define ICF1 0x80    /* Input Capture Flag 1 */



#define TSR_UNUSED  0x1f    /* bit mask of unused bits in TSR */



/*------------------

*   Bit Positions

------------------*/

#define B_TOF    5   /* Timer Overflow Flag */

#define B_OC1F   6   /* Output Compare 1 Flag */

#define B_ICF1   7   /* Input Capture Flag 1 */



/*--------------------------------------

*   ICH

*   Input Capture Register High

*   

*   Address: 0x0014

*   Manual Page: 120

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ICH_8  - Input Capture High Data Bit 8 

*     |   |   |   |   |   |   |-------- ICH_9  - Input Capture High Data Bit 9 

*     |   |   |   |   |   |------------ ICH_10 - Input Capture High Data Bit 10

*     |   |   |   |   |---------------- ICH_11 - Input Capture High Data Bit 11

*     |   |   |   |-------------------- ICH_12 - Input Capture High Data Bit 12

*     |   |   |------------------------ ICH_13 - Input Capture High Data Bit 13

*     |   |---------------------------- ICH_14 - Input Capture High Data Bit 14

*     |-------------------------------- ICH_15 - Input Capture High Data Bit 15

*     

*   NOTE:   To prevent interrupts from occurring between readings of ICH and ICL,

*           set the interrupt flag in the condition code register before reading

*           ICH and clear the flag after reading ICL.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ICH_8  0x01    /* Input Capture High Data Bit 8  */

#define ICH_9  0x02    /* Input Capture High Data Bit 9  */

#define ICH_10 0x04    /* Input Capture High Data Bit 10 */

#define ICH_11 0x08    /* Input Capture High Data Bit 11 */

#define ICH_12 0x10    /* Input Capture High Data Bit 12 */

#define ICH_13 0x20    /* Input Capture High Data Bit 13 */

#define ICH_14 0x40    /* Input Capture High Data Bit 14 */

#define ICH_15 0x80    /* Input Capture High Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_ICH_8   0  /* Input Capture High Data Bit 8  */

#define B_ICH_9   1  /* Input Capture High Data Bit 9  */

#define B_ICH_10  2  /* Input Capture High Data Bit 10 */

#define B_ICH_11  3  /* Input Capture High Data Bit 11 */

#define B_ICH_12  4  /* Input Capture High Data Bit 12 */

#define B_ICH_13  5  /* Input Capture High Data Bit 13 */

#define B_ICH_14  6  /* Input Capture High Data Bit 14 */

#define B_ICH_15  7  /* Input Capture High Data Bit 15 */



/*--------------------------------------

*   ICL

*   Input Capture Low

*   

*   Address: 0x0015

*   Manual Page: 120

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ICL_0 - Input Capture Low Data Bit 0 

*     |   |   |   |   |   |   |-------- ICL_1 - Input Capture Low Data Bit 1 

*     |   |   |   |   |   |------------ ICL_2 - Input Capture Low Data Bit 2

*     |   |   |   |   |---------------- ICL_3 - Input Capture Low Data Bit 3

*     |   |   |   |-------------------- ICL_4 - Input Capture Low Data Bit 4

*     |   |   |------------------------ ICL_5 - Input Capture Low Data Bit 5

*     |   |---------------------------- ICL_6 - Input Capture Low Data Bit 6

*     |-------------------------------- ICL_7 - Input Capture Low Data Bit 7

*     

*   NOTE:   To prevent interrupts from occurring between readings of ICH and ICL,

*           set the interrupt flag in the condition code register before reading

*           ICH and clear the flag after reading ICL.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ICL_0  0x01    /* Input Capture Low Data Bit 0 */

#define ICL_1  0x02    /* Input Capture Low Data Bit 1 */

#define ICL_2  0x04    /* Input Capture Low Data Bit 2 */

#define ICL_3  0x08    /* Input Capture Low Data Bit 3 */

#define ICL_4  0x10    /* Input Capture Low Data Bit 4 */

#define ICL_5  0x20    /* Input Capture Low Data Bit 5 */

#define ICL_6  0x40    /* Input Capture Low Data Bit 6 */

#define ICL_7  0x80    /* Input Capture Low Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_ICL_0  0  /* Input Capture Low Data Bit 0 */

#define B_ICL_1  1  /* Input Capture Low Data Bit 1 */

#define B_ICL_2  2  /* Input Capture Low Data Bit 2 */

#define B_ICL_3  3  /* Input Capture Low Data Bit 3 */

#define B_ICL_4  4  /* Input Capture Low Data Bit 4 */

#define B_ICL_5  5  /* Input Capture Low Data Bit 5 */

#define B_ICL_6  6  /* Input Capture Low Data Bit 6 */

#define B_ICL_7  7  /* Input Capture Low Data Bit 7 */



/*--------------------------------------

*   OC1H

*   Output Compare 1 High

*   

*   Address: 0x0016

*   Manual Page: 119

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OC1H_8  - Output Compare 1 High Data Bit 8 

*     |   |   |   |   |   |   |-------- OC1H_9  - Output Compare 1 High Data Bit 9 

*     |   |   |   |   |   |------------ OC1H_10 - Output Compare 1 High Data Bit 10

*     |   |   |   |   |---------------- OC1H_11 - Output Compare 1 High Data Bit 11

*     |   |   |   |-------------------- OC1H_12 - Output Compare 1 High Data Bit 12

*     |   |   |------------------------ OC1H_13 - Output Compare 1 High Data Bit 13

*     |   |---------------------------- OC1H_14 - Output Compare 1 High Data Bit 14

*     |-------------------------------- OC1H_15 - Output Compare 1 High Data Bit 15

*     

*   NOTE:   To prevent OC1F from being set between the time it is read and the time the

*           output compare registers are updated, you should disable interrupts in the

*           condition code register, write to OC1H, clear the OC1F bit by reading the TSR,

*           write OC1L, and then re-enable interrupts in the condition code register.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OC1H_8  0x01    /* Output Compare 1 High Data Bit 8  */

#define OC1H_9  0x02    /* Output Compare 1 High Data Bit 9  */

#define OC1H_10 0x04    /* Output Compare 1 High Data Bit 10 */

#define OC1H_11 0x08    /* Output Compare 1 High Data Bit 11 */

#define OC1H_12 0x10    /* Output Compare 1 High Data Bit 12 */

#define OC1H_13 0x20    /* Output Compare 1 High Data Bit 13 */

#define OC1H_14 0x40    /* Output Compare 1 High Data Bit 14 */

#define OC1H_15 0x80    /* Output Compare 1 High Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_OC1H_8   0  /* Output Compare 1 High Data Bit 8  */

#define B_OC1H_9   1  /* Output Compare 1 High Data Bit 9  */

#define B_OC1H_10  2  /* Output Compare 1 High Data Bit 10 */

#define B_OC1H_11  3  /* Output Compare 1 High Data Bit 11 */

#define B_OC1H_12  4  /* Output Compare 1 High Data Bit 12 */

#define B_OC1H_13  5  /* Output Compare 1 High Data Bit 13 */

#define B_OC1H_14  6  /* Output Compare 1 High Data Bit 14 */

#define B_OC1H_15  7  /* Output Compare 1 High Data Bit 15 */



/*--------------------------------------

*   OC1L

*   Output Compare 1 Low

*   

*   Address: 0x0017

*   Manual Page: 119

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OC1L_0 - Output Compare 1 Low Data Bit 0 

*     |   |   |   |   |   |   |-------- OC1L_1 - Output Compare 1 Low Data Bit 1 

*     |   |   |   |   |   |------------ OC1L_2 - Output Compare 1 Low Data Bit 2

*     |   |   |   |   |---------------- OC1L_3 - Output Compare 1 Low Data Bit 3

*     |   |   |   |-------------------- OC1L_4 - Output Compare 1 Low Data Bit 4

*     |   |   |------------------------ OC1L_5 - Output Compare 1 Low Data Bit 5

*     |   |---------------------------- OC1L_6 - Output Compare 1 Low Data Bit 6

*     |-------------------------------- OC1L_7 - Output Compare 1 Low Data Bit 7

*     

*   NOTE:   To prevent OC1F from being set between the time it is read and the time the

*           output compare registers are updated, you should disable interrupts in the

*           condition code register, write to OC1H, clear the OC1F bit by reading the TSR,

*           write OC1L, and then re-enable interrupts in the condition code register.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OC1L_0  0x01    /* Output Compare 1 Low Data Bit 0 */

#define OC1L_1  0x02    /* Output Compare 1 Low Data Bit 1 */

#define OC1L_2  0x04    /* Output Compare 1 Low Data Bit 2 */

#define OC1L_3  0x08    /* Output Compare 1 Low Data Bit 3 */

#define OC1L_4  0x10    /* Output Compare 1 Low Data Bit 4 */

#define OC1L_5  0x20    /* Output Compare 1 Low Data Bit 5 */

#define OC1L_6  0x40    /* Output Compare 1 Low Data Bit 6 */

#define OC1L_7  0x80    /* Output Compare 1 Low Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_OC1L_0  0  /* Output Compare 1 Low Data Bit 0 */

#define B_OC1L_1  1  /* Output Compare 1 Low Data Bit 1 */

#define B_OC1L_2  2  /* Output Compare 1 Low Data Bit 2 */

#define B_OC1L_3  3  /* Output Compare 1 Low Data Bit 3 */

#define B_OC1L_4  4  /* Output Compare 1 Low Data Bit 4 */

#define B_OC1L_5  5  /* Output Compare 1 Low Data Bit 5 */

#define B_OC1L_6  6  /* Output Compare 1 Low Data Bit 6 */

#define B_OC1L_7  7  /* Output Compare 1 Low Data Bit 7 */



/*--------------------------------------

*   TCNTH

*   Timer Counter High

*   

*   Address: 0x0018

*   Manual Page: 118

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- TCNTH8  - Timer High Data Bit 8 

*     |   |   |   |   |   |   |-------- TCNTH9  - Timer High Data Bit 9 

*     |   |   |   |   |   |------------ TCNTH10 - Timer High Data Bit 10

*     |   |   |   |   |---------------- TCNTH11 - Timer High Data Bit 11

*     |   |   |   |-------------------- TCNTH12 - Timer High Data Bit 12

*     |   |   |------------------------ TCNTH13 - Timer High Data Bit 13

*     |   |---------------------------- TCNTH14 - Timer High Data Bit 14

*     |-------------------------------- TCNTH15 - Timer High Data Bit 15

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define TCNTH8  0x01    /* Timer High Data Bit 8  */

#define TCNTH9  0x02    /* Timer High Data Bit 9  */

#define TCNTH10 0x04    /* Timer High Data Bit 10 */

#define TCNTH11 0x08    /* Timer High Data Bit 11 */

#define TCNTH12 0x10    /* Timer High Data Bit 12 */

#define TCNTH13 0x20    /* Timer High Data Bit 13 */

#define TCNTH14 0x40    /* Timer High Data Bit 14 */

#define TCNTH15 0x80    /* Timer High Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_TCNTH8   0  /* Timer High Data Bit 8  */

#define B_TCNTH9   1  /* Timer High Data Bit 9  */

#define B_TCNTH10  2  /* Timer High Data Bit 10 */

#define B_TCNTH11  3  /* Timer High Data Bit 11 */

#define B_TCNTH12  4  /* Timer High Data Bit 12 */

#define B_TCNTH13  5  /* Timer High Data Bit 13 */

#define B_TCNTH14  6  /* Timer High Data Bit 14 */

#define B_TCNTH15  7  /* Timer High Data Bit 15 */



/*--------------------------------------

*   TCNTL

*   Timer Counter Low

*   

*   Address: 0x0019

*   Manual Page: 118

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- TCNTL0 - Timer Low Data Bit 0 

*     |   |   |   |   |   |   |-------- TCNTL1 - Timer Low Data Bit 1 

*     |   |   |   |   |   |------------ TCNTL2 - Timer Low Data Bit 2

*     |   |   |   |   |---------------- TCNTL3 - Timer Low Data Bit 3

*     |   |   |   |-------------------- TCNTL4 - Timer Low Data Bit 4

*     |   |   |------------------------ TCNTL5 - Timer Low Data Bit 5

*     |   |---------------------------- TCNTL6 - Timer Low Data Bit 6

*     |-------------------------------- TCNTL7 - Timer Low Data Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define TCNTL0  0x01    /* Timer Low Data Bit 0 */

#define TCNTL1  0x02    /* Timer Low Data Bit 1 */

#define TCNTL2  0x04    /* Timer Low Data Bit 2 */

#define TCNTL3  0x08    /* Timer Low Data Bit 3 */

#define TCNTL4  0x10    /* Timer Low Data Bit 4 */

#define TCNTL5  0x20    /* Timer Low Data Bit 5 */

#define TCNTL6  0x40    /* Timer Low Data Bit 6 */

#define TCNTL7  0x80    /* Timer Low Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_TCNTL0  0   /* Timer Low Data Bit 0 */

#define B_TCNTL1  1   /* Timer Low Data Bit 1 */

#define B_TCNTL2  2   /* Timer Low Data Bit 2 */

#define B_TCNTL3  3   /* Timer Low Data Bit 3 */

#define B_TCNTL4  4   /* Timer Low Data Bit 4 */

#define B_TCNTL5  5   /* Timer Low Data Bit 5 */

#define B_TCNTL6  6   /* Timer Low Data Bit 6 */

#define B_TCNTL7  7   /* Timer Low Data Bit 7 */



/*--------------------------------------

*   ACNTH

*   Alternate Timer Counter Register High

*   

*   Address: 0x001a

*   Manual Page: 118

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ACNTH8  - Alternate Timer High Data Bit 8 

*     |   |   |   |   |   |   |-------- ACNTH9  - Alternate Timer High Data Bit 9 

*     |   |   |   |   |   |------------ ACNTH10 - Alternate Timer High Data Bit 10

*     |   |   |   |   |---------------- ACNTH11 - Alternate Timer High Data Bit 11

*     |   |   |   |-------------------- ACNTH12 - Alternate Timer High Data Bit 12

*     |   |   |------------------------ ACNTH13 - Alternate Timer High Data Bit 13

*     |   |---------------------------- ACNTH14 - Alternate Timer High Data Bit 14

*     |-------------------------------- ACNTH15 - Alternate Timer High Data Bit 15

*     

*   NOTE:   To prevent interrupts from occurring between readings of ACNTH and ACNTL,

*           set the interrupt flag in the condition code register before reading

*           ACNTH and clear the flag after reading ACNTL.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ACNTH8  0x01    /* Alternate Timer High Data Bit 8  */

#define ACNTH9  0x02    /* Alternate Timer High Data Bit 9  */

#define ACNTH10 0x04    /* Alternate Timer High Data Bit 10 */

#define ACNTH11 0x08    /* Alternate Timer High Data Bit 11 */

#define ACNTH12 0x10    /* Alternate Timer High Data Bit 12 */

#define ACNTH13 0x20    /* Alternate Timer High Data Bit 13 */

#define ACNTH14 0x40    /* Alternate Timer High Data Bit 14 */

#define ACNTH15 0x80    /* Alternate Timer High Data Bit 15 */



/*------------------

*   Bit Positions

------------------*/

#define B_ACNTH8   0  /* Alternate Timer High Data Bit 8  */

#define B_ACNTH9   1  /* Alternate Timer High Data Bit 9  */

#define B_ACNTH10  2  /* Alternate Timer High Data Bit 10 */

#define B_ACNTH11  3  /* Alternate Timer High Data Bit 11 */

#define B_ACNTH12  4  /* Alternate Timer High Data Bit 12 */

#define B_ACNTH13  5  /* Alternate Timer High Data Bit 13 */

#define B_ACNTH14  6  /* Alternate Timer High Data Bit 14 */

#define B_ACNTH15  7  /* Alternate Timer High Data Bit 15 */



/*--------------------------------------

*   ACNTL

*   Alternate Timer Counter Register Low

*   

*   Address: 0x001b

*   Manual Page: 118

*

*    RO  RO  RO  RO  RO  RO  RO  RO     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- ACNTL0 - Alternate Timer Low Data Bit 0 

*     |   |   |   |   |   |   |-------- ACNTL1 - Alternate Timer Low Data Bit 1 

*     |   |   |   |   |   |------------ ACNTL2 - Alternate Timer Low Data Bit 2

*     |   |   |   |   |---------------- ACNTL3 - Alternate Timer Low Data Bit 3

*     |   |   |   |-------------------- ACNTL4 - Alternate Timer Low Data Bit 4

*     |   |   |------------------------ ACNTL5 - Alternate Timer Low Data Bit 5

*     |   |---------------------------- ACNTL6 - Alternate Timer Low Data Bit 6

*     |-------------------------------- ACNTL7 - Alternate Timer Low Data Bit 7

*     

*   NOTE:   To prevent interrupts from occurring between readings of ACNTH and ACNTL,

*           set the interrupt flag in the condition code register before reading

*           ACNTH and clear the flag after reading ACNTL.

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ACNTL0  0x01    /* Alternate Timer Low Data Bit 0 */

#define ACNTL1  0x02    /* Alternate Timer Low Data Bit 1 */

#define ACNTL2  0x04    /* Alternate Timer Low Data Bit 2 */

#define ACNTL3  0x08    /* Alternate Timer Low Data Bit 3 */

#define ACNTL4  0x10    /* Alternate Timer Low Data Bit 4 */

#define ACNTL5  0x20    /* Alternate Timer Low Data Bit 5 */

#define ACNTL6  0x40    /* Alternate Timer Low Data Bit 6 */

#define ACNTL7  0x80    /* Alternate Timer Low Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_ACNTL0  0  /* Alternate Timer Low Data Bit 0 */

#define B_ACNTL1  1  /* Alternate Timer Low Data Bit 1 */

#define B_ACNTL2  2  /* Alternate Timer Low Data Bit 2 */

#define B_ACNTL3  3  /* Alternate Timer Low Data Bit 3 */

#define B_ACNTL4  4  /* Alternate Timer Low Data Bit 4 */

#define B_ACNTL5  5  /* Alternate Timer Low Data Bit 5 */

#define B_ACNTL6  6  /* Alternate Timer Low Data Bit 6 */

#define B_ACNTL7  7  /* Alternate Timer Low Data Bit 7 */





/*--------------------------------------

*   TCR2

*   Timer Control Register 2

*   

*   Address: 0x001c

*   Manual Page: 128

*

*    RW  RW  --  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OL2 - Timer Output 2 Edge Select

*     |   |   |   |   |   |   |-------- OE2 - Timer Output 2 Output Enable

*     |   |   |   |   |   |------------ IL2 - Timer Input 2 Active Edge Select

*     |   |   |   |   |---------------- IM2 - Timer Input 2 Mode Select

*     |   |   |   |-------------------- T2CLK - Timer 2 Clock Select

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- OC2IE - Output Compare 2 Interrupt Enable

*     |-------------------------------- TI2IE - Timer Input 2 Interrupt Enable

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OL2   0x01    /* Timer Output 2 Edge Select */

#define OE2   0x02    /* Timer Output 2 Output Enable */

#define IL2   0x04    /* Timer Input 2 Active Edge Select */

#define IM2   0x08    /* Timer Input 2 Mode Select */

#define T2CLK 0x10    /* Timer 2 Clock Select */

#define OC2IE 0x40    /* Output Compare 2 Interrupt Enable */

#define TI2IE 0x80    /* Timer Input 2 Interrupt Enable */



#define TCR2_UNUSED 0x20   /* unused bits in the TCR2 register */

/*------------------

*   Bit Positions

------------------*/

#define B_OL2   0    /* Timer Output 2 Edge Select */

#define B_OE2   1    /* Timer Output 2 Output Enable */

#define B_IL2   2    /* Timer Input 2 Active Edge Select */

#define B_IM2   3    /* Timer Input 2 Mode Select */

#define B_T2CLK 4    /* Timer 2 Clock Select */

#define B_OC2IE 6    /* Output Compare 2 Interrupt Enable */

#define B_TI2IE 7    /* Timer Input 2 Interrupt Enable */



/*--------------------------------------

*   TSR2

*   Timer Status Register 2

*   

*   Address: 0x001d

*   Manual Page: 130

*

*    RO  RO  --  --  WO  WO  --  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ ROC2F - Reset Output Compare 2 Flag

*     |   |   |   |   |---------------- RTI2F -  Reset Timer Input 2 Flag

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- OC2F - Output Compare Flag 2

*     |-------------------------------- TI2F - Timer 2 Input Capture Flag

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define ROC2F 0x04    /* Reset Output Compare 2 Flag */

#define RTI2F 0x08    /* Reset Timer Input 2 Flag */

#define OC2F  0x40    /* Output Compare Flag 2 */

#define TI2F  0x80    /* Timer 2 Input Capture Flag */



#define TSR2_UNUSED  0x33    /* bit mask of unused bits in TSR2 */



/*------------------

*   Bit Positions

------------------*/

#define B_ROC2F 2    /* Reset Output Compare 2 Flag */

#define B_RTI2F 3    /* Reset Timer Input 2 Flag */

#define B_OC2F  6    /* Output Compare Flag 2 */

#define B_TI2F  7    /* Timer 2 Input Capture Flag */



/*--------------------------------------

*   OC2

*   Output Compare Register 2

*   

*   Address: 0x001e

*   Manual Page: 131

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OC2_0 - Output Compare 2 Data Bit 0

*     |   |   |   |   |   |   |-------- OC2_1 - Output Compare 2 Data Bit 1 

*     |   |   |   |   |   |------------ OC2_2 - Output Compare 2 Data Bit 2

*     |   |   |   |   |---------------- OC2_3 - Output Compare 2 Data Bit 3

*     |   |   |   |-------------------- OC2_4 - Output Compare 2 Data Bit 4

*     |   |   |------------------------ OC2_5 - Output Compare 2 Data Bit 5

*     |   |---------------------------- OC2_6 - Output Compare 2 Data Bit 6

*     |-------------------------------- OC2_7 - Output Compare 2 Data Bit 7

*

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OC2_0 0x01    /* Output Compare 2 Data Bit 0 */

#define OC2_1 0x02    /* Output Compare 2 Data Bit 1 */

#define OC2_2 0x04    /* Output Compare 2 Data Bit 2 */

#define OC2_3 0x08    /* Output Compare 2 Data Bit 3 */

#define OC2_4 0x10    /* Output Compare 2 Data Bit 4 */

#define OC2_5 0x20    /* Output Compare 2 Data Bit 5 */

#define OC2_6 0x40    /* Output Compare 2 Data Bit 6 */

#define OC2_7 0x80    /* Output Compare 2 Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_OC2_0  0  /* Output Compare 2 Data Bit 0 */

#define B_OC2_1  1  /* Output Compare 2 Data Bit 1 */

#define B_OC2_2  2  /* Output Compare 2 Data Bit 2 */

#define B_OC2_3  3  /* Output Compare 2 Data Bit 3 */

#define B_OC2_4  4  /* Output Compare 2 Data Bit 4 */

#define B_OC2_5  5  /* Output Compare 2 Data Bit 5 */

#define B_OC2_6  6  /* Output Compare 2 Data Bit 6 */

#define B_OC2_7  7  /* Output Compare 2 Data Bit 7 */



/*--------------------------------------

*   TCNT2

*   Timer Counter Register 2

*   

*   Address: 0x001f

*   Manual Page: 131

*

*    RW  RW  RW  RW  RW  RW  RW  RW   <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- TCNT2_0 - Timer 2 Data Bit 0 

*     |   |   |   |   |   |   |-------- TCNT2_1 - Timer 2 Data Bit 1 

*     |   |   |   |   |   |------------ TCNT2_2 - Timer 2 Data Bit 2

*     |   |   |   |   |---------------- TCNT2_3 - Timer 2 Data Bit 3

*     |   |   |   |-------------------- TCNT2_4 - Timer 2 Data Bit 4

*     |   |   |------------------------ TCNT2_5 - Timer 2 Data Bit 5

*     |   |---------------------------- TCNT2_6 - Timer 2 Data Bit 6

*     |-------------------------------- TCNT2_7 - Timer 2 Data Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define TCNT2_0  0x01    /* Timer 2 Data Bit 0 */

#define TCNT2_1  0x02    /* Timer 2 Data Bit 1 */

#define TCNT2_2  0x04    /* Timer 2 Data Bit 2 */

#define TCNT2_3  0x08    /* Timer 2 Data Bit 3 */

#define TCNT2_4  0x10    /* Timer 2 Data Bit 4 */

#define TCNT2_5  0x20    /* Timer 2 Data Bit 5 */

#define TCNT2_6  0x40    /* Timer 2 Data Bit 6 */

#define TCNT2_7  0x80    /* Timer 2 Data Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_TCNT2_0  0   /* Timer 2 Data Bit 0 */

#define B_TCNT2_1  1   /* Timer 2 Data Bit 1 */

#define B_TCNT2_2  2   /* Timer 2 Data Bit 2 */

#define B_TCNT2_3  3   /* Timer 2 Data Bit 3 */

#define B_TCNT2_4  4   /* Timer 2 Data Bit 4 */

#define B_TCNT2_5  5   /* Timer 2 Data Bit 5 */

#define B_TCNT2_6  6   /* Timer 2 Data Bit 6 */

#define B_TCNT2_7  7   /* Timer 2 Data Bit 7 */



/*--------------------------------------

*   LCDCR

*   LCD Control Register

*   

*   Address: 0x0020

*   Manual Page:146

*

*    RW  RW  RW  --  RW  RW  RW  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- PDH - Select Port D (H)

*     |   |   |   |   |   |------------ PEL - Select Port E (L)

*     |   |   |   |   |---------------- PEH - Select Port E (H)

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ DUTY0 - LCD Duty Select Bit 0

*     |   |---------------------------- DUTY1 - LCD Duty Select Bit 1

*     |-------------------------------- LCDE - LCD Output Enable

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PDH   0x02    /* Select Port D (H) */

#define PEL   0x04    /* Select Port E (L) */

#define PEH   0x08    /* Select Port E (H) */

#define DUTY0 0x20    /* LCD Duty Select Bit 0 */

#define DUTY1 0x40    /* LCD Duty Select Bit 1 */

#define LCDE  0x80    /* LCD Output Enable */



#define LCDCR_UNUSED 0x11   /* unused bits in LCDCR register */

/*------------------

*   Bit Positions

------------------*/

#define B_PDH   1    /* Select Port D (H) */

#define B_PEL   2    /* Select Port E (L) */

#define B_PEH   3    /* Select Port E (H) */

#define B_DUTY0 5    /* LCD Duty Select Bit 0 */

#define B_DUTY1 6    /* LCD Duty Select Bit 1 */

#define B_LCDE  7    /* LCD Output Enable */



/*--------------------------------------

*   LCDRx

*   LCD Data Registers

*   

*   Address: 0x0021 - 0x0034

*   Manual Page:148

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- LBP0 - Backplane Timing Bit 0 for FP(2x-2)

*     |   |   |   |   |   |   |-------- LBP1 - Backplane Timing Bit 1 for FP(2x-2)

*     |   |   |   |   |   |------------ LBP2 - Backplane Timing Bit 2 for FP(2x-2)

*     |   |   |   |   |---------------- LBP3 - Backplane Timing Bit 3 for FP(2x-2)

*     |   |   |   |-------------------- HBP0 - Backplane Timing Bit 0 for FP(2x-1)

*     |   |   |------------------------ HBP1 - Backplane Timing Bit 1 for FP(2x-1)

*     |   |---------------------------- HBP2 - Backplane Timing Bit 2 for FP(2x-1)

*     |-------------------------------- HBP3 - Backplane Timing Bit 3 for FP(2x-1)

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define LBP0 0x01    /* Backplane Timing Bit 0 for FP(2x-2) */

#define LBP1 0x02    /* Backplane Timing Bit 1 for FP(2x-2) */

#define LBP2 0x04    /* Backplane Timing Bit 2 for FP(2x-2) */

#define LBP3 0x08    /* Backplane Timing Bit 3 for FP(2x-2) */

#define HBP0 0x10    /* Backplane Timing Bit 0 for FP(2x-1) */

#define HBP1 0x20    /* Backplane Timing Bit 1 for FP(2x-1) */

#define HBP2 0x40    /* Backplane Timing Bit 2 for FP(2x-1) */

#define HBP3 0x80    /* Backplane Timing Bit 3 for FP(2x-1) */



/*------------------

*   Bit Positions

------------------*/

#define B_LBP0 0    /* Backplane Timing Bit 0 for FP(2x-2) */

#define B_LBP1 1    /* Backplane Timing Bit 1 for FP(2x-2) */

#define B_LBP2 2    /* Backplane Timing Bit 2 for FP(2x-2) */

#define B_LBP3 3    /* Backplane Timing Bit 3 for FP(2x-2) */

#define B_HBP0 4    /* Backplane Timing Bit 0 for FP(2x-1) */

#define B_HBP1 5    /* Backplane Timing Bit 1 for FP(2x-1) */

#define B_HBP2 6    /* Backplane Timing Bit 2 for FP(2x-1) */

#define B_HBP3 7    /* Backplane Timing Bit 3 for FP(2x-1) */



/*--------------------------------------

*   MISC

*   Miscellaneous Register

*   

*   Address: 0x003e

*   Manual Page:101

*

*    RO  RO  --  --  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- OPTM - Option Map Select

*     |   |   |   |   |   |   |-------- FOSCE - Fast Oscillator Enable

*     |   |   |   |   |   |------------ SYS0 - System Clock Select Bit 0

*     |   |   |   |   |---------------- SYS1 - System Clock Select Bit 1

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- STUP - XOSC Time Up Flag

*     |-------------------------------- FTUP - OSC Time Up Flag

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define OPTM  0x01    /* Option Map Select */

#define FOSCE 0x02    /* Fast Oscillator Enable */

#define SYS0  0x04    /* System Clock Select Bit 0 */

#define SYS1  0x08    /* System Clock Select Bit 1 */

#define STUP  0x40    /* XOSC Time Up Flag */

#define FTUP  0x80    /* OSC Time Up Flag */



#define UNUSED_MISC 0x30    /* unused bits in MISC register */

/*------------------

*   Bit Positions

------------------*/

#define B_OPTM  0    /* Option Map Select */

#define B_FOSCE 1    /* Fast Oscillator Enable */

#define B_SYS0  2    /* System Clock Select Bit 0 */

#define B_SYS1  3    /* System Clock Select Bit 1 */

#define B_STUP  6    /* XOSC Time Up Flag */

#define B_FTUP  7    /* OSC Time Up Flag */



/*--------------------------------------

*   DDRA

*   Port A Data Direction Register

*   

*   Address: Option Map - 0x0000

*   Manual Page: 77

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- DDRA0 - Port A Data Direction Bit 0

*     |   |   |   |   |   |   |-------- DDRA1 - Port A Data Direction Bit 1

*     |   |   |   |   |   |------------ DDRA2 - Port A Data Direction Bit 2

*     |   |   |   |   |---------------- DDRA3 - Port A Data Direction Bit 3

*     |   |   |   |-------------------- DDRA4 - Port A Data Direction Bit 4

*     |   |   |------------------------ DDRA5 - Port A Data Direction Bit 5

*     |   |---------------------------- DDRA6 - Port A Data Direction Bit 6

*     |-------------------------------- DDRA7 - Port A Data Direction Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DDRA0 0x01    /* Port A Data Direction Bit 0 */

#define DDRA1 0x02    /* Port A Data Direction Bit 1 */

#define DDRA2 0x04    /* Port A Data Direction Bit 2 */

#define DDRA3 0x08    /* Port A Data Direction Bit 3 */

#define DDRA4 0x10    /* Port A Data Direction Bit 4 */

#define DDRA5 0x20    /* Port A Data Direction Bit 5 */

#define DDRA6 0x40    /* Port A Data Direction Bit 6 */

#define DDRA7 0x80    /* Port A Data Direction Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_DDRA0   0   /* Port A Data Direction Bit 0 */

#define B_DDRA1   1   /* Port A Data Direction Bit 1 */

#define B_DDRA2   2   /* Port A Data Direction Bit 2 */

#define B_DDRA3   3   /* Port A Data Direction Bit 3 */

#define B_DDRA4   4   /* Port A Data Direction Bit 4 */

#define B_DDRA5   5   /* Port A Data Direction Bit 5 */

#define B_DDRA6   6   /* Port A Data Direction Bit 6 */

#define B_DDRA7   7   /* Port A Data Direction Bit 7 */



/*--------------------------------------

*   DDRC

*   Port C Data Direction Register

*   

*   Address: Option Map - 0x0002

*   Manual Page: 82

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- DDRC0 - Port C Data Direction Bit 0

*     |   |   |   |   |   |   |-------- DDRC1 - Port C Data Direction Bit 1

*     |   |   |   |   |   |------------ DDRC2 - Port C Data Direction Bit 2

*     |   |   |   |   |---------------- DDRC3 - Port C Data Direction Bit 3

*     |   |   |   |-------------------- DDRC4 - Port C Data Direction Bit 4

*     |   |   |------------------------ DDRC5 - Port C Data Direction Bit 5

*     |   |---------------------------- DDRC6 - Port C Data Direction Bit 6

*     |-------------------------------- DDRC7 - Port C Data Direction Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define DDRC0 0x01    /* Port C Data Direction Bit 0 */

#define DDRC1 0x02    /* Port C Data Direction Bit 1 */

#define DDRC2 0x04    /* Port C Data Direction Bit 2 */

#define DDRC3 0x08    /* Port C Data Direction Bit 3 */

#define DDRC4 0x10    /* Port C Data Direction Bit 4 */

#define DDRC5 0x20    /* Port C Data Direction Bit 5 */

#define DDRC6 0x40    /* Port C Data Direction Bit 6 */

#define DDRC7 0x80    /* Port C Data Direction Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_DDRC0   0   /* Port C Data Direction Bit 0 */

#define B_DDRC1   1   /* Port C Data Direction Bit 1 */

#define B_DDRC2   2   /* Port C Data Direction Bit 2 */

#define B_DDRC3   3   /* Port C Data Direction Bit 3 */

#define B_DDRC4   4   /* Port C Data Direction Bit 4 */

#define B_DDRC5   5   /* Port C Data Direction Bit 5 */

#define B_DDRC6   6   /* Port C Data Direction Bit 6 */

#define B_DDRC7   7   /* Port C Data Direction Bit 7 */



/*--------------------------------------

*   PDMUX

*   Port D MUX Register

*   

*   Address: Option Map - 0x0003

*   Manual Page: 85

*

*    RW  RW  RW  RW  --  --  --  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- PDM4 - Port D MUX Control Bit 4

*     |   |   |------------------------ PDM5 - Port D MUX Control Bit 5

*     |   |---------------------------- PDM6 - Port D MUX Control Bit 6

*     |-------------------------------- PDM7 - Port D MUX Control Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PDM4 0x10    /* Port D MUX Control Bit 4 */

#define PDM5 0x20    /* Port D MUX Control Bit 5 */

#define PDM6 0x40    /* Port D MUX Control Bit 6 */

#define PDM7 0x80    /* Port D MUX Control Bit 7 */



#define PDMUX_UNUSED 0x0f    /* unused bits in PDMUX register */

/*------------------

*   Bit Positions

------------------*/

#define B_PDM4 4    /* Port D MUX Control Bit 4 */

#define B_PDM5 5    /* Port D MUX Control Bit 5 */

#define B_PDM6 6    /* Port D MUX Control Bit 6 */

#define B_PDM7 7    /* Port D MUX Control Bit 7 */



/*--------------------------------------

*   PEMUX

*   Port E MUX Register

*   

*   Address: Option Map - 0x0004

*   Manual Page: 88

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- PEM0 - Port E MUX Control Bit 0

*     |   |   |   |   |   |   |-------- PEM1 - Port E MUX Control Bit 1

*     |   |   |   |   |   |------------ PEM2 - Port E MUX Control Bit 2

*     |   |   |   |   |---------------- PEM3 - Port E MUX Control Bit 3

*     |   |   |   |-------------------- PEM4 - Port E MUX Control Bit 4

*     |   |   |------------------------ PEM5 - Port E MUX Control Bit 5

*     |   |---------------------------- PEM6 - Port E MUX Control Bit 6

*     |-------------------------------- PEM7 - Port E MUX Control Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define PEM0 0x01    /* Port E MUX Control Bit 0 */

#define PEM1 0x02    /* Port E MUX Control Bit 1 */

#define PEM2 0x04    /* Port E MUX Control Bit 2 */

#define PEM3 0x08    /* Port E MUX Control Bit 3 */

#define PEM4 0x10    /* Port E MUX Control Bit 4 */

#define PEM5 0x20    /* Port E MUX Control Bit 5 */

#define PEM6 0x40    /* Port E MUX Control Bit 6 */

#define PEM7 0x80    /* Port E MUX Control Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_PEM0 0    /* Port E MUX Control Bit 0 */

#define B_PEM1 1    /* Port E MUX Control Bit 1 */

#define B_PEM2 2    /* Port E MUX Control Bit 2 */

#define B_PEM3 3    /* Port E MUX Control Bit 3 */

#define B_PEM4 4    /* Port E MUX Control Bit 4 */

#define B_PEM5 5    /* Port E MUX Control Bit 5 */

#define B_PEM6 6    /* Port E MUX Control Bit 6 */

#define B_PEM7 7    /* Port E MUX Control Bit 7 */



/*--------------------------------------

*   RCR1

*   Resistor Control Register 1

*   

*   Address: Option Map - 0x0008

*   Manual Page: 45

*

*    --  --  --  --  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- RAL - Port A Pullup Resistor (L)

*     |   |   |   |   |   |   |-------- RAH - Port A Pullup Resistor (H)

*     |   |   |   |   |   |------------ RBL - Port B Pullup Resistor (L)

*     |   |   |   |   |---------------- RBH - Port B Pullup Resistor (H)

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ Unimplemented

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define RAL 0x01    /* Port A Pullup Resistor (L) */

#define RAH 0x02    /* Port A Pullup Resistor (H) */

#define RBL 0x04    /* Port B Pullup Resistor (L) */

#define RBH 0x08    /* Port B Pullup Resistor (H) */



#define RCR1_UNUSED 0xf0    /* unused bits in RCR1 register */

/*------------------

*   Bit Positions

------------------*/

#define B_RAL 0    /* Port A Pullup Resistor (L) */

#define B_RAH 1    /* Port A Pullup Resistor (H) */

#define B_RBL 2    /* Port B Pullup Resistor (L) */

#define B_RBH 3    /* Port B Pullup Resistor (H) */



/*--------------------------------------

*   RCR2

*   Resistor Control Register 2

*   

*   Address: Option Map - 0x0009

*   Manual Page: 46

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- RC0 - Port C Pullup Resistor Bit 0

*     |   |   |   |   |   |   |-------- RC1 - Port C Pullup Resistor Bit 1

*     |   |   |   |   |   |------------ RC2 - Port C Pullup Resistor Bit 2

*     |   |   |   |   |---------------- RC3 - Port C Pullup Resistor Bit 3

*     |   |   |   |-------------------- RC4 - Port C Pullup Resistor Bit 4

*     |   |   |------------------------ RC5 - Port C Pullup Resistor Bit 5

*     |   |---------------------------- RC6 - Port C Pullup Resistor Bit 6

*     |-------------------------------- RC7 - Port C Pullup Resistor Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define RC0 0x01    /* Port C Pullup Resistor Bit 0 */

#define RC1 0x02    /* Port C Pullup Resistor Bit 1 */

#define RC2 0x04    /* Port C Pullup Resistor Bit 2 */

#define RC3 0x08    /* Port C Pullup Resistor Bit 3 */

#define RC4 0x10    /* Port C Pullup Resistor Bit 4 */

#define RC5 0x20    /* Port C Pullup Resistor Bit 5 */

#define RC6 0x40    /* Port C Pullup Resistor Bit 6 */

#define RC7 0x80    /* Port C Pullup Resistor Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_RC0 0    /* Port C Pullup Resistor Bit 0 */

#define B_RC1 1    /* Port C Pullup Resistor Bit 1 */

#define B_RC2 2    /* Port C Pullup Resistor Bit 2 */

#define B_RC3 3    /* Port C Pullup Resistor Bit 3 */

#define B_RC4 4    /* Port C Pullup Resistor Bit 4 */

#define B_RC5 5    /* Port C Pullup Resistor Bit 5 */

#define B_RC6 6    /* Port C Pullup Resistor Bit 6 */

#define B_RC7 7    /* Port C Pullup Resistor Bit 7 */



/*--------------------------------------

*   WOM1

*   Open-Drain Output Control Register 1

*   

*   Address: Option Map - 0x000a

*   Manual Page: 46

*

*    RW  RW  RW  RW  --  --  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- AWOML - Port A Open-Drain Mode (L)

*     |   |   |   |   |   |   |-------- AWOMH - Port A Open-Drain Mode (H)

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- EWOML - Port E Open-Drain Mode (L)

*     |   |   |------------------------ EWOMH - Port E Open-Drain Mode (H)

*     |   |---------------------------- DWOML - Port D Open-Drain Mode (L)

*     |-------------------------------- DWOMH - Port D Open-Drain Mode (H)

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define AWOML 0x01    /* Port A Open-Drain Mode (L) */

#define AWOMH 0x02    /* Port A Open-Drain Mode (H) */

#define EWOML 0x10    /* Port E Open-Drain Mode (L) */

#define EWOMH 0x20    /* Port E Open-Drain Mode (H) */

#define DWOML 0x40    /* Port D Open-Drain Mode (L) */

#define DWOMH 0x80    /* Port D Open-Drain Mode (H) */



#define WOM1_UNUSED 0x0c    /* unused bits in WOM1 register */

/*------------------

*   Bit Positions

------------------*/

#define B_AWOML 0    /* Port A Open-Drain Mode (L) */

#define B_AWOMH 1    /* Port A Open-Drain Mode (H) */

#define B_EWOML 4    /* Port E Open-Drain Mode (L) */

#define B_EWOMH 5    /* Port E Open-Drain Mode (H) */

#define B_DWOML 6    /* Port D Open-Drain Mode (L) */

#define B_DWOMH 7    /* Port D Open-Drain Mode (H) */



/*--------------------------------------

*   WOM2

*   Open-Drain Output Control Register 2

*   

*   Address: Option Map - 0x000b

*   Manual Page: 48

*

*    --  --  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- CWOM0 - Port C Open-Drain Mode Bit 0

*     |   |   |   |   |   |   |-------- CWOM1 - Port C Open-Drain Mode Bit 1

*     |   |   |   |   |   |------------ CWOM2 - Port C Open-Drain Mode Bit 2

*     |   |   |   |   |---------------- CWOM3 - Port C Open-Drain Mode Bit 3

*     |   |   |   |-------------------- CWOM4 - Port C Open-Drain Mode Bit 4

*     |   |   |------------------------ CWOM5 - Port C Open-Drain Mode Bit 5

*     |   |---------------------------- Unimplemented

*     |-------------------------------- Unimplemented

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define CWOM0 0x01    /* Port C Open-Drain Mode Bit 0 */

#define CWOM1 0x02    /* Port C Open-Drain Mode Bit 1 */

#define CWOM2 0x04    /* Port C Open-Drain Mode Bit 2 */

#define CWOM3 0x08    /* Port C Open-Drain Mode Bit 3 */

#define CWOM4 0x10    /* Port C Open-Drain Mode Bit 4 */

#define CWOM5 0x20    /* Port C Open-Drain Mode Bit 5 */



#define WOM2_UNUSED 0xc0    /* unused bits in WOM2 register */

/*------------------

*   Bit Positions

------------------*/

#define B_CWOM0 0    /* Port C Open-Drain Mode Bit 0 */

#define B_CWOM1 1    /* Port C Open-Drain Mode Bit 1 */

#define B_CWOM2 2    /* Port C Open-Drain Mode Bit 2 */

#define B_CWOM3 3    /* Port C Open-Drain Mode Bit 3 */

#define B_CWOM4 4    /* Port C Open-Drain Mode Bit 4 */

#define B_CWOM5 5    /* Port C Open-Drain Mode Bit 5 */



/*--------------------------------------

*   KWIEN

*   Key Wakeup Input Enable Register

*   

*   Address: Option Map - 0x000e

*   Manual Page: 48

*

*    RW  RW  RW  RW  RW  RW  RW  RW     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- KWE0 - Key Wakeup Enable Bit 0

*     |   |   |   |   |   |   |-------- KWE1 - Key Wakeup Enable Bit 1

*     |   |   |   |   |   |------------ KWE2 - Key Wakeup Enable Bit 2

*     |   |   |   |   |---------------- KWE3 - Key Wakeup Enable Bit 3

*     |   |   |   |-------------------- KWE4 - Key Wakeup Enable Bit 4

*     |   |   |------------------------ KWE5 - Key Wakeup Enable Bit 5

*     |   |---------------------------- KWE6 - Key Wakeup Enable Bit 6

*     |-------------------------------- KWE7 - Key Wakeup Enable Bit 7

*     

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define KWE0 0x01    /* Key Wakeup Enable Bit 0 */

#define KWE1 0x02    /* Key Wakeup Enable Bit 1 */

#define KWE2 0x04    /* Key Wakeup Enable Bit 2 */

#define KWE3 0x08    /* Key Wakeup Enable Bit 3 */

#define KWE4 0x10    /* Key Wakeup Enable Bit 4 */

#define KWE5 0x20    /* Key Wakeup Enable Bit 5 */

#define KWE6 0x40    /* Key Wakeup Enable Bit 6 */

#define KWE7 0x80    /* Key Wakeup Enable Bit 7 */



/*------------------

*   Bit Positions

------------------*/

#define B_KWE0 0    /* Key Wakeup Enable Bit 0 */

#define B_KWE1 1    /* Key Wakeup Enable Bit 1 */

#define B_KWE2 2    /* Key Wakeup Enable Bit 2 */

#define B_KWE3 3    /* Key Wakeup Enable Bit 3 */

#define B_KWE4 4    /* Key Wakeup Enable Bit 4 */

#define B_KWE5 5    /* Key Wakeup Enable Bit 5 */

#define B_KWE6 6    /* Key Wakeup Enable Bit 6 */

#define B_KWE7 7    /* Key Wakeup Enable Bit 7 */



/*--------------------------------------

*   MOR

*   Mask Option Register

*   

*   Address: Option Map - 0x000f

*   Manual Page: 49

*

*    RO  RO  RO  --  --  --  --  --     <- read/write allowance

*   |---|---|---|---|---|---|---|---|

*   | 7 | 6 | 5 | 4 | 3 | 2 | 1 | 0 |

*   |---|---|---|---|---|---|---|---|

*     |   |   |   |   |   |   |   |

*     |   |   |   |   |   |   |   |---- Unimplemented

*     |   |   |   |   |   |   |-------- Unimplemented

*     |   |   |   |   |   |------------ Unimplemented

*     |   |   |   |   |---------------- Unimplemented

*     |   |   |   |-------------------- Unimplemented

*     |   |   |------------------------ XOSCR - OSC Feedback Resistor

*     |   |---------------------------- OSCR - OSC Feedback Resistor

*     |-------------------------------- RSTR - Reset Pin Pullup Resistor

*   

--------------------------------------*/



/*------------------

*   Bit Masks

------------------*/

#define XOSCR 0x20    /* OSC Feedback Resistor */

#define OSCR  0x40    /* OSC Feedback Resistor */

#define RSTR  0x80    /* Reset Pin Pullup Resistor */



#define MOR_UNUSED 0x1f    /* unused bit mask of MOR register */

/*------------------

*   Bit Positions

------------------*/

#define B_XOSCR 5    /* OSC Feedback Resistor */

#define B_OSCR  6    /* OSC Feedback Resistor */

#define B_RSTR  7    /* Reset Pin Pullup Resistor */



/*--------------------------------------

*   Register Manipulation Macros

--------------------------------------*/



/* PUT_REG writes an immediate value into the register */



#define PUT_REG(reg, value) reg = value



/* GET_REG reads the current value of a register and stores it in a variable */



#define GET_REG(reg, var)   var = reg



/* SET_REG_BITS or's the mask value into the register*/



#define SET_REG_BITS(reg, mask) reg |= mask



/* CLR_REG_BITS and's the inverse of the mask value into the register*/



#define CLR_REG_BITS(reg, mask) reg &= ~(mask)









