/******************************************************************************
*
* Copyright (c) 2005 Freescale Semiconductor, Inc.
* 
* Permission is hereby granted, free of charge, to any person obtaining a
* copy of this software and associated documentation files (the "Software"),
* to deal in the Software without restriction, including without limitation
* the rights to use, copy, modify, merge, publish, distribute, sublicense,
* and/or sell copies of the Software, and to permit persons to whom the
* Software is furnished to do so, subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included
* in all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
* THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR
* OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
* ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
* OTHER DEALINGS IN THE SOFTWARE.
*
******************************************************************************/

/* Description: This is an example that runs on the total5200 Software 	*/
/*	Development Platform. It can be built and downloaded to the dBUG   	*/
/*	bootloader on the total5200 and run. 								*/
/*  The code initializes the PSC1 AC97 interface 						*/
/*	and plays a 1 kHz sine wave using the Bestcomm API to transfer the 	*/
/*	the audio data to PSC1. Connect speakers or headphones to the 	   	*/
/*	headphone output of the total5200 to hear the sine wave. This code 	*/
/*  can be modified to use another PSC or used as initialization for an */
/*  AC97 driver. 														*/

#include <stdio.h>

#include "ppctypes.h"
#include "core5200.h"
#include "exc5xxx.h"
#include "freq5200.h"
#include "time5xxx.h"
#include "mgt5200/mgt5200.h"
#include "mgt5200/psc.h"
#include "mgt5200/gpio_std.h"
#include "mgt5200/gpio_wkup.h"
#include "mgt5200/sdma.h"
#include "mgt5200/int_ctrl.h"

#include "bestcomm_api.h"

int next_buffer;
TaskId	tx_task_id;

psc_regs *ac97PSC;
gpio_std_regs *siu_gpio;
gpio_wkup_regs *siu_gpio_wkup;
sdma_regs *bcomm;
int_ctrl_regs *int_ctrl;

#define AC97_FRAMES			48
#define AC97_FRAME_BYTES 	52
#define RX_FIFO_SIZE 		512
#define RX_ALARM			256
#define TX_ALARM			256
#define RX_GRAN				4
#define TX_GRAN				7

/*defines for psc control messages */
#define RESET_TX		0x30
#define RESET_RX		0x20
#define RESET_ERR	0x40
#define ENABLE_TX	0x04
#define ENABLE_RX	0x01
#define DISABLE_TX	0x08
#define DISABLE_RX	0x02
#define RESET_MODE	0x10

/* defines for status register bits */
#define SR_FFULL	0x0200
#define SR_URERR	0x0800
#define SR_TXRDY	0x0400

/* two buffers to contain sine wave in ac97 frames */
/* one frame is 52 bytes */
uint32 ac97_frame0[AC97_FRAMES * AC97_FRAME_BYTES/4];
uint32 ac97_frame1[AC97_FRAMES * AC97_FRAME_BYTES/4];

/* Interrupt handler that will be called every time the DMA task finishes transfer */
/* of one buffer to the PSC FIFO */
int intHandler (void *arg0, void *arg1)
{
#pragma unused (arg0, arg1)

	TaskBDRelease(tx_task_id);

	/* Assign the next audio buffer to the buffer descriptor ring */
	if(next_buffer)
	{
		TaskBDAssign(tx_task_id, (int *) ac97_frame1,(int *) NULL, AC97_FRAME_BYTES * AC97_FRAMES, 0); 
		next_buffer = 0;
	}
	else
	{
		TaskBDAssign(tx_task_id,(int *) ac97_frame0,(int *) NULL, AC97_FRAME_BYTES * AC97_FRAMES, 0); 
		next_buffer = 1;
	}
	
	/* Clear interrupt */
	TaskIntClear(tx_task_id);
	
	return 1;
}

/* This function reads an AC97 register by writing a read command directly to the */
/* PSC FIFO. Then the receive FIFO is parsed for the reponse from the AC97 device. */

uint32 readRegAC97(uint8 reg)
{
	int i = 0;
	int j = 0;
	
	unsigned int slot[13];						
	unsigned int reg_value;

	ac97PSC->CR = RESET_RX;
	ac97PSC->CR = RESET_TX;
	ac97PSC->CR = RESET_MODE;
	ac97PSC->CR = RESET_ERR;	
	
	/* Write directly to the TX FIFO */
	/* building a register read AC97 Frame */		
	ac97PSC->RB_TB = 0xc0000000; 				
	ac97PSC->RB_TB = (reg << 24) | 0x80000000; 
	for (i = 0; i < 11; i++)
	{
		ac97PSC->RB_TB = 0x00000000;
	}

	/* Transmit read request and get incoming frames */
	ac97PSC->CR = ENABLE_RX | ENABLE_TX;		
	
	/* Wait for FIFO alarm */
	while(!(ac97PSC->SR_CSR & SR_FFULL)){};	

	ac97PSC->CR = DISABLE_RX | DISABLE_TX;
	
	i = 0;
	
	while (i < RX_FIFO_SIZE)				
	{
		
		/* Look for first frame and check for valid data in read slot */
		slot[0] = ac97PSC->RB_TB;			
		if (slot[0] & 0x00000800)
		{
			for (j=1;j<13;j++)
			{
				slot[j]=ac97PSC->RB_TB;
			}
			if ((slot[0] & 0xe0000000) == 0xe0000000)
			{
				if((unsigned int)(slot[1] >> 24) == reg)
				{
					reg_value = (slot[2] >> 16);
					
					/* Exit loop */		
					break;									
				}
			}
			
			i+=52;
		}
		else
		{
			/* Next word in FIFO */
			i+=4;
		}
	}
	
	return reg_value;
		
}

/* This function will write an AC97 register by building the control frame */
/* that sends the command to the transmit FIFO. */

void writeRegAC97 (uint8 reg, uint16 val)
{

	int i = 0;
	
	ac97PSC->CR = RESET_RX;
	ac97PSC->CR = RESET_TX;
	ac97PSC->CR = RESET_MODE;
	ac97PSC->CR = RESET_ERR;			
	
	/* Create reg write ac97 frame and write directly to TX FIFO */
	ac97PSC->RB_TB = 0xe0000000;  							
	ac97PSC->RB_TB = (reg << 24) & ~0x80000000; 			
	ac97PSC->RB_TB = (uint32)(val << 16);					
	for (i = 0; i < 10; i++)
	{
			ac97PSC->RB_TB = 0x00000000;
	}
	
	/* Transmit frame in PSC */	
	ac97PSC->CR = ENABLE_RX | ENABLE_TX;
	
	/* Wait for frame to be transmitted */
	while(!(ac97PSC->SR_CSR & SR_URERR)){};
	
	ac97PSC->CR = DISABLE_RX | DISABLE_TX;
}

/* This function creates the AC97 frames for a sine wave in two seperate buffers */

void createAC97SineWave(void)
{

		int i;

		uint32 sine_wave[48];

/* These sine wave values were created using Excel, but the */
/* calculating the sine function in this program would also work */
/* The native sample rate of AC97 is 48kHz, so a 1 kHz sine */
/* wave will have 48 samples per period. */
	sine_wave[0]=0<<16;
	sine_wave[1]=4280<<16;
	sine_wave[2]=8487<<16;
	sine_wave[3]=12548<<16;
	sine_wave[4]=16395<<16;
	sine_wave[5]=19960<<16;
	sine_wave[6]=23184<<16;
	sine_wave[7]=26010<<16;
	sine_wave[8]=28391<<16;
	sine_wave[9]=30284<<16;
	sine_wave[10]=31659<<16;
	sine_wave[11]=32492<<16;
	sine_wave[12]=32767<<16;
	sine_wave[13]=32482<<16;
	sine_wave[14]=31639<<16;
	sine_wave[15]=30254<<16;
	sine_wave[16]=28351<<16;
	sine_wave[17]=25962<<16;
	sine_wave[18]=23128<<16;
	sine_wave[19]=19898<<16;
	sine_wave[20]=16327<<16;
	sine_wave[21]=12475<<16;
	sine_wave[22]=8411<<16;
	sine_wave[23]=4202<<16;
	sine_wave[24]=(uint32)-78<<16;
	sine_wave[25]=(uint32)-4358<<16;
	sine_wave[26]=(uint32)-8563<<16;
	sine_wave[27]=(uint32)-12621<<16;
	sine_wave[28]=(uint32)-16463<<16;
	sine_wave[29]=(uint32)-20023<<16;
	sine_wave[30]=(uint32)-23240<<16;
	sine_wave[31]=(uint32)-26058<<16;
	sine_wave[32]=(uint32)-28430<<16;
	sine_wave[33]=(uint32)-30315<<16;
	sine_wave[34]=(uint32)-31680<<16;
	sine_wave[35]=(uint32)-32502<<16;
	sine_wave[36]=(uint32)-32767<<16;
	sine_wave[37]=(uint32)-32471<<16;
	sine_wave[38]=(uint32)-31618<<16;
	sine_wave[39]=(uint32)-30224<<16;
	sine_wave[40]=(uint32)-28311<<16;
	sine_wave[41]=(uint32)-25914<<16;
	sine_wave[42]=(uint32)-23072<<16;
	sine_wave[43]=(uint32)-19835<<16;
	sine_wave[44]=(uint32)-16258<<16;
	sine_wave[45]=(uint32)-12402<<16;
	sine_wave[46]=(uint32)-8334<<16;
	sine_wave[47]=(uint32)-4123<<16;

/* The ac97 frames are created in two buffers */
/* with the one sample in both left and right */
/* channels per frame. */
/* Each buffer contains one full period of the */
/* sine wave then. */
		
	for(i=0;i<48;i++)
	{
		ac97_frame0[(i * 13) + 0] = 0x98000000;
		ac97_frame0[(i * 13) + 1] = 0x00000000;
		ac97_frame0[(i * 13) + 2] = 0x00000000;
		ac97_frame0[(i * 13) + 3] = sine_wave[i];  
		ac97_frame0[(i * 13) + 4] = sine_wave[i];		
		ac97_frame0[(i * 13) + 5] = 0x00000000;
		ac97_frame0[(i * 13) + 6] = 0x00000000;
		ac97_frame0[(i * 13) + 7] = 0x00000000;
		ac97_frame0[(i * 13) + 8] = 0x00000000;
		ac97_frame0[(i * 13) + 9] = 0x00000000;
		ac97_frame0[(i * 13) + 10] = 0x00000000;
		ac97_frame0[(i * 13) + 11] = 0x00000000;
		ac97_frame0[(i * 13) + 12] = 0x00000000;
					
	}
	
	for(i=0;i<48;i++)
	{
		ac97_frame1[(i * 13) + 0] = 0x98000000;
		ac97_frame1[(i * 13) + 1] = 0x00000000;
		ac97_frame1[(i * 13) + 2] = 0x00000000;
		ac97_frame1[(i * 13) + 3] = sine_wave[i];  
		ac97_frame1[(i * 13) + 4] = sine_wave[i];			
		ac97_frame1[(i * 13) + 5] = 0x00000000;
		ac97_frame1[(i * 13) + 6] = 0x00000000;
		ac97_frame1[(i * 13) + 7] = 0x00000000;
		ac97_frame1[(i * 13) + 8] = 0x00000000;
		ac97_frame1[(i * 13) + 9] = 0x00000000;
		ac97_frame1[(i * 13) + 10] = 0x00000000;
		ac97_frame1[(i * 13) + 11] = 0x00000000;
		ac97_frame1[(i * 13) + 12] = 0x00000000;
					
	}
		


}


void main()
{

	TaskSetupParamSet_t dma_task_param;
	int i,j;


	/* Establish pointers for PSC1 and port_config register */
	
	printf("*** Start AC97 ***\n");
	
	ac97PSC = (psc_regs *) (readMBAR() + MBAR_PSC1);
	siu_gpio = (gpio_std_regs *) (readMBAR() + MBAR_GPIO_STD);
	siu_gpio_wkup = (gpio_wkup_regs *) (readMBAR() + MBAR_GPIO_WKUP);
	
	bcomm = (sdma_regs *) (readMBAR() + MBAR_SDMA);
	int_ctrl = (int_ctrl_regs *) (readMBAR() + MBAR_INT_CTRL);

	/* Initialize Bestcomm API by calling init function, */
	/* writing the taskbar register with the destination */
	/* of the Bestcomm firmware and calling the load function */
	TasksInitAPI((uint8 *) readMBAR());
	bcomm->taskBar = (uint32)(readMBAR() + MBAR_SRAM);
	TasksLoadImage((sdma_regs *)(bcomm));

	/* Call dBUG helper functions for timer and register interrupt handler*/
	timeInit();
	exceptionRegister (EXC_CRITICAL_INT, 4, intHandler, NULL, NULL);

	/* Unmask Bestcomm interrupt in SIU controller */
	int_ctrl->pimsk &= 0x7fffffff;

	/* Create the memory buffer that contains the sine wave in AC97 frames */
	createAC97SineWave();
	/* Make sure all PSC interrupts disabled. */
	/* Interrupt Event comes from Bestcomm task */
	ac97PSC->ISR_IMR = 0x0;
	
	/* Reset the PSC before writing the registers */
	ac97PSC->CR = RESET_RX;
	ac97PSC->CR = RESET_TX;
	ac97PSC->CR = RESET_MODE;
	ac97PSC->CR = RESET_ERR;
	
	/* PSC register set for AC97 mode and make sure ACRB is set */
	ac97PSC->SICR = 0x83000000;
	/* First write: RxRdy (FIFO Alarm) generates receive FIFO interrupt */		
	/* Second write to mode: register Normal mode for non loopback */
	ac97PSC->ModeReg = 0x00;
	ac97PSC->ModeReg = 0x00;



	/* Set FIFO alarm levels and granularities */
	/* Alarm is set to how many bytes are left before an error happens */
	/*  before notifying the Bestcomm or the core */
	/* Granularity is when the alarm signal goes off. */
	/* For the RX FIFO granualrity is in bytes */
	/* For the TX FIFO granularity is in bytes * 4 */
	/* For this example the TX granularity is set to turn off */
	/*  when enough full AC97 frames are put into the FIFO */
	 
	ac97PSC->RFALARM = RX_ALARM;	/* Set threshold for full rx FIFO */	
	ac97PSC->RFCNTL  = RX_GRAN;
	ac97PSC->TFALARM = TX_ALARM;	/* Set threshold for empty tx FIFO */
	ac97PSC->TFCNTL  = TX_GRAN;

	printf("Reseting AC97 \n");
	/* Reset the AC97 chip by toggling the reset output. */
	/* Wait min reset time. */
	/* Make sure that frame sync is low during reset to not put */
	/* AC97 device into a test mode */
	/* When PSC1 is in AC97 mode the state of the frame sync pin */
	/* will be indeterminate, so the gpio mode is used to make */
	/* sure that the sync will be low when the reset goes high. */
	
	
	/* Enable correct input and output scheme for AC97 interface. */
	/* Four AC97 pins are standard GPIO and one is a wakeup (reset line) */
	
	/* In the MPC5200 there PSC1 has 5 GPIO pins. 4 standard and 1 wakeup */
	
	/* enable CMOS outputs */
	siu_gpio->simple_ode &= 0xFFFFFFF0;
	siu_gpio_wkup->wkup_ode &= 0xFE;
	
	/* enable input on clock and sdatain pin */
	siu_gpio->simple_ddr &= 0xFFFFFFF0;
	siu_gpio->simple_ddr |= 0x00000005;
	siu_gpio_wkup->wkup_ddr |= 0x01;
	
	/* Set output pins low */
	siu_gpio->simple_dvo &= 0xFFFFFFF0;
	siu_gpio_wkup->wkup_dvo &= 0xFE;
	
	/* Enable GPIOs */
	siu_gpio->simple_gpioe |= 0x0000000F;
	siu_gpio_wkup->wkup_gpioe |= 0x01;
	
	/* Enable GPIO mode on PSC1 pins */
	siu_gpio->port_config &= 0xFFFFFFF8;	

	timeWaitms(10);
	
	/* Set reset pin high */
	siu_gpio_wkup->wkup_dvo |= 0x01;

	/* Make sure the AC97 reset pin is high using the PSC interface */
	ac97PSC->OP1 = 0x00;
	ac97PSC->OP0 = 0x02;		

	/* Enable AC97 mode on PSC1 pins */
	siu_gpio->port_config |= 0x00000002;

	printf("After Reset\n");

	/* Wait for AC97 device mixer to fully power up by reading status register */
	while((readRegAC97(0x26) & 0x000F) != 0x000F){};
	
	/* Write AC97 registers to turn on output */

	/* Write Master Out to unmute output */
	writeRegAC97(0x02,0x0000);
	printf("AC97 Register 0x02 = 0x%04x\n",readRegAC97(0x02));

	/* Write to enable external output amplifier */
	writeRegAC97(0x18,0x0808);
	printf("AC97 Register 0x18 = 0x%04x\n",readRegAC97(0x18));
	
	/* Write to enable external output amplifier */
	/* This statement is only necessary on the total5200 dev platform */
	writeRegAC97(0x26,0x8000);
	printf("AC97 Register 0x26 = 0x%04x\n",readRegAC97(0x26));
	
	/* Set up Bestcomm DMA Task for two buffer descriptors in a ring. */
	/* Transfer to PSC1 transmit FIFO in words */
	/* The TASK_GEN_TX_BD task is a buffer descriptor that writes to a FIFO */
	/* Maximum buffer size in bytes */
	dma_task_param.Size.MaxBuf = AC97_FRAME_BYTES * AC97_FRAMES;
	
	/* Number of buffer descriptors to create */
	dma_task_param.NumBD = 2;
	
	/* The initiator is what ties the PSC FIFO to the Bestcomm task */
	/* PSC1 Here */
	dma_task_param.Initiator = INITIATOR_PSC1_TX;
	
	/* Only Destination FIFO register is needed */
	/* The source pointer will be incremented by 4 bytes every transfer */
	dma_task_param.StartAddrDst = (uint32)(&(ac97PSC->RB_TB));
	dma_task_param.IncrSrc = 4;
	dma_task_param.SzSrc = SZ_UINT32;
	dma_task_param.SzDst = SZ_UINT32;
	tx_task_id = TaskSetup(TASK_GEN_TX_BD, &dma_task_param);
	
	/* Prefill the buffers before enabling the tasks */
	/* by assigning the two ac97 buffers created earlier */
	TaskBDAssign(tx_task_id,(int *) ac97_frame0, (int *) NULL, AC97_FRAME_BYTES * AC97_FRAMES, 0); 
	TaskBDAssign(tx_task_id,(int *) ac97_frame1,(int *) NULL, AC97_FRAME_BYTES * AC97_FRAMES, 0); 
	next_buffer = 0;

	/* Reset the PSC1 FIFO controller, error and mode register */
	ac97PSC->CR = RESET_RX;
	ac97PSC->CR = RESET_TX;
	ac97PSC->CR = RESET_MODE;
	ac97PSC->CR = RESET_ERR;
	
	/* Fill the TXFIFO with empty AC97 Frames. This will mean that the PSC FIFO is full */
	/* Turning on the PSC before this will probably underflow the FIFO */
	/* 9 full frames will fit. */
	for(i=0;i<9;i++)
	{
		ac97PSC->RB_TB=0x98000000;
		for(j=0;j<12;j++)
			ac97PSC->RB_TB=0x00000000;
	}
	
	/* Enable Tx Task for autostart when finished and with interrupts enabled */
	TaskStart(tx_task_id, TASK_AUTOSTART_ENABLE, tx_task_id, TASK_INTERRUPT_ENABLE);	
		
	/* Enable both RX and TX to transmit.*/
	ac97PSC->CR = ENABLE_RX | ENABLE_TX;

	printf("Starting infinite loop \n");
	while (1)
	{	
		/* Run in an infinite loop looking for underruns */
		while((ac97PSC->SR_CSR & SR_URERR) == SR_URERR)
		{
			/* If underrun then reset psc2 fifo and restart at the next buffer */
			TaskStop(tx_task_id);
			ac97PSC->CR = RESET_RX;
			ac97PSC->CR = RESET_TX;
			ac97PSC->CR = RESET_ERR;
			
			printf("Underrun\n");
			
			/* Restart at the beginning of the BD ring */
			TaskBDReset(tx_task_id);
			if(next_buffer)
			{
				TaskBDAssign(tx_task_id,(int *) ac97_frame1,(int *) NULL, AC97_FRAME_BYTES * AC97_FRAMES, 0); 
				next_buffer = 0;
			}
			else
			{
				TaskBDAssign(tx_task_id,(int *) ac97_frame0,(int *) NULL, AC97_FRAME_BYTES * AC97_FRAMES, 0); 
				next_buffer = 1;
			}
			TaskStart(tx_task_id, TASK_AUTOSTART_ENABLE, tx_task_id, TASK_INTERRUPT_ENABLE);
			while(ac97PSC->SR_CSR != SR_TXRDY)
			for(i=0;i<9;i++)
			{
				ac97PSC->RB_TB=0x98000000;
				for(j=0;j<12;j++)
					ac97PSC->RB_TB=0x00000000;
			}
			ac97PSC->CR = ENABLE_RX | ENABLE_TX;			
		}
	}
	exceptionRemove (intHandler);

}