/****************************************************************************/
/*
	UART.c

	Implementation file for an abstract UART driver interface.	
*/
/****************************************************************************/

#include "UART.h"		/* abstract interface */

#include "pscuart5200.h"

/****************************************************************************/
/*
	Global variables
*/
/****************************************************************************/

/****************************************************************************/
/*
	Static function declarations
*/
/****************************************************************************/

static unsigned long _strlen(const char * str);

/****************************************************************************/
/*
	Local "string length" function.
*/
/****************************************************************************/
unsigned long _strlen(const char * str)
{
/*
	unsigned long	len = -1;
	
	unsigned char * p = (unsigned char *) str - 1;
	
	do
		len++;
	while (*++p);

	return(len);
*/

	unsigned long len = 0;
	char *p = (char *) str;
	
	while (*p++) {
		len++;
	}
	
	return len;
}


/****************************************************************************/
/*
	InitializeIntDrivenUART
	
	Reset UART with the appropriate baud rate.

	Used instead of InitializeUART() to initialize the library when interrupt-driven
	I/O is desired.  The flags intDrivenInput and intDrivenOutput determine
	whether to use interrupt-driven input, output, or both.  The parameter
	inputPendingPtrRef is used to return a pointer to an InputPending flag.
	The caller can then check the status of InputPending to determine whether
	input has arrived.  Note that the status of this flag can change at 
	any time unless the serial interrupt is masked.  If interrupt-driven
	input is not enabled, the value of this flag is always FALSE (0).
*/
/****************************************************************************/
#if 0
UARTError InitializeIntDrivenUART(UARTBaudRate baudRate,
								  unsigned char intDrivenInput,
								  unsigned char intDrivenOutput,
								  volatile unsigned char **inputPendingPtrRef)
{
	SerialInitIRQ( baudRate, intDrivenInput, intDrivenOutput, inputPendingPtrRef );
	return kUARTNoError;
}
#endif
/****************************************************************************/
/*
	InitializeUART
	
	Reset UART with the appropriate baud rate.
*/
/****************************************************************************/

UARTError InitializeUART(UARTBaudRate baudRate)
{
	pscuartInit (baudRate);

	return kUARTNoError;
}

/****************************************************************************/
/*
	TerminateUART
	
	Turn off the UART (as appropriate).
*/
/****************************************************************************/

UARTError TerminateUART(void)
{
	return kUARTNoError;
}

/****************************************************************************/
/*
	ReadUARTPoll
	
	Read one character from the UART if it is available, otherwise
	return an error.
*/
/****************************************************************************/
UARTError ReadUARTPoll(char* c)
{
	int err;
	
	err = pscuartPollChar ();
	if (err == -1) {
		return kUARTNoData;
	} else {
		*c = (char) err;
		return kUARTNoError;
	}
}

/****************************************************************************/
/*
	ReadUART1
	
	Read one character from the UART.
*/
/****************************************************************************/

UARTError ReadUART1(char* c)
{
	int err;
	
	err = pscuartGetChar ();
	if (err == -1) {
		return kUARTNoData;
	} else {
		*c = (char) err;
		return kUARTNoError;
	}
}

/****************************************************************************/
/*
	ReadUARTN
	
	Read N bytes from the UART.
	
	bytes			pointer to result buffer
	limit			size of buffer and # of bytes to read
*/
/****************************************************************************/
UARTError ReadUARTN(void* bytes, unsigned long limit)
{
	int count;
	UARTError err; 

	for (count = 0, err = kUARTNoError; 
		count < limit && err == kUARTNoError;
		count++)
        {
		err = ReadUART1( (char *)bytes + count );
        }

	return err;
}

/****************************************************************************/
/*
	ReadUARTString
	
	Read a string terminated by a special character into a buffer.  Do not
	include the terminate character (e.g. "0x0D") in the buffer.  Always
	return a zero-terminated buffer.
	
	Example:
	
	limit = 4, termChar = '\r'
	
	input           0  1  2  3   return value
	----------      -----------  ---------------
	a  b \r     ->  a  b \0 ??   kUARTNoError
	a  b  c \r  ->  a  b  c \0   kUARTNoError
	a  b  c  d  ->  a  b  c \0   kUARTBufferOverflow
	
	s				string (result) buffer
	limit			size of buffer
	termChar		character in the UART that terminates string entry
	ReadUARTString	kUARTBufferOverflow (and errors returned by ReadUART1)
*/
/****************************************************************************/
UARTError ReadUARTString(char* s, unsigned long limit, char termChar)
{
	UARTError err = kUARTNoError;
	
	unsigned long l = 0;			/* current length of buffer */
	char c;
		
	do {
		err = ReadUART1(&c);
		if (err)
			break;
		
		if (c == termChar)
			break;
		
		*s++ = c;
		l++;
	} while (l < limit);

	if (l == limit) {
		s--;						/* back up */
		if (err == kUARTNoError)	/* don't mask previous errors */
			err = kUARTBufferOverflow;
	}
	
	*s = 0;							/* always terminate string */
	return err;
}




/****************************************************************************/

UARTError WriteUART1(char c)
{
	pscuartPutChar (c);

	return kUARTNoError;
}

/****************************************************************************/

UARTError WriteUARTN(const void* bytes, unsigned long length)
{
	int count;
	UARTError err;

	for (count = 0, err = kUARTNoError;
		count < length && err == kUARTNoError;
		count++)
        {
		err = WriteUART1( ((char *)bytes)[ count ] );
        } 

	return err;
}

/****************************************************************************/

UARTError WriteUARTString(const char* string)
{
	unsigned long length = _strlen(string);
	
	return WriteUARTN((const void*) string, length);
}
