/******************************************************************************
*
*       COPYRIGHT (c) 2001-2003 MOTOROLA INC.
*       ALL RIGHTS RESERVED
*
*       The code is the property of Motorola.
*
*       The copyright notice above does not evidence any
*       actual or intended publication of such source code.
*
* Filename:     $Source: /proj/cvsroot/mgt/MGT5200/apps/AC97Sample/modules/Frequency/freq5200.c,v $
* Author:       $Author: ra6707 $
* Locker:       $Locker:  $
* State:        $State: Exp $
* Revision:     $Revision: 1.1 $
*
* Functions:    
*
* History:      Use the RCS command rlog to display revision history
*               information.
*
* Description:  
*
* Notes:                
*
******************************************************************************/
#include <stdio.h>

#include "ppctypes.h"

#include "configure.h"

#if !(defined(MGT5200) ^ defined(MPC5200B))
	#error Either MGT5200 or MPC5200B has to be defined.
#endif

#if defined(MPC5200B)
	#include "mpc5200b/mpc5200b.h"
	#include "mpc5200b/cdm.h"
	#include "mpc5200b/rtclock.h"
	#include "mpc5200b/sctmr.h"
	#include "mpc5200b/ipbi.h"
#elif defined(MGT5200)
	#include "mgt5200/mgt5200.h"
	#include "mgt5200/cdm.h"
	#include "mgt5200/rtclock.h"
	#include "mgt5200/sctmr.h"
	#include "mgt5200/ipbi.h"
#endif

#include "core5200.h"
#include "freq5200.h"

#define FREQ_UNINIT 0
#define FREQ_OK     1
#define FREQ_BAD    2

/*
 * Global variables to store frequency status
 */
static unsigned int fquartz = 0;
static unsigned int fvco = 0;
static unsigned int fcore = 0;
static unsigned int fxlb = 0;
static unsigned int fipbi = 0;
static unsigned int fpci = 0;
static unsigned int fusb = 0;
static unsigned int firda = 0;
static int fstatus = FREQ_UNINIT;

/* Clocks in each phase of the fractional dividers */
static const unsigned int fracdiv[] = {
	8, 9, 10, 11, 11, 11, 6, 7
};

/* 2 x Multipy factors for the CPU clock
 * Note - divide by 2 to get the actual multiply factor
 */
static const unsigned cpumult2[] = {
	3, 2, 2, 2, 4, 4, 5, 9, 6, 11, 8, 10, 3, 12, 7, 0,
	6, 5, 13, 2, 14, 4, 15, 9, 0, 11, 8, 10, 16, 12, 7, 0
};

/*! \brief Initialize the Frequency Module.
 *
 * Depending on the settings configured in configure.h a call to this function
 * can take up to two seconds. It is not necessary to call this function before
 * you use any of the other functions. But if it has not been called called
 * before any other function will call it will have a runtime of up to two
 * seconds.
 */
#if (defined USE_SCTMR) && (!defined NO_FREQ_CALIBRATION)
#error "SC Timer usage not supported yet"
void freqInit (void)
{
	cdm_regs *cdm;
	sctmr_regs *sctmr;
	rtclock_regs *rtclock;
	unsigned int ticks;	
	uint8 secs;

	cdm = (cdm_regs *) (readMBAR () + MBAR_CDM);
	rtclock = (rtclock_regs *) (readMBAR () + MBAR_RTCLOCK);
	sctmr = (sctmr_regs *) (readMBAR () + MBAR_SCTMR);

	fstatus = FREQ_OK;
	
	sctmr->tmr1_control = 0x0000;	/* timer1 stop and reset */
	sctmr->tmr1_prescale = 1024;	/* prescale gives range upto 33.5 MHz clk */
	sctmr->tmr1_count = 0xFFFF;		/* terminal count */

	/*
	 * Wait for the seconds in the RTC to roll over 
	 */
	secs = rtclock->second_status;
	while (secs == rtclock->second_status) ;

	/*
	 * Start timer
	 */
	sctmr->tmr1_control = 0x0100;

	/*
	 * Wait for the seconds in the RTC to roll over 
	 */
	secs = rtclock->second_status;
	while (secs == rtclock->second_status) ;

	/*
	 * Pause timer
	 */
	sctmr->tmr1_control = 0x1100;

	/*
	 * Store the number of ticks for one second 
	 */
	ticks = sctmr->tmr1_cvalue;
	ticks *= sctmr->tmr1_prescale;

	/*
	 * Stop and reset timer
	 */
	sctmr->tmr1_control = 0;

	/*
	 * Calculate quartz frequency from ticks per second
	 */
	fquartz = ticks;
	if (cdm->ipg_clk_sel) {		/* from IPBI to XLB */
		fquartz *= 2;
	}
	if (cdm->rstcfg & 0x20) {	/* from XLB to VCO */
		fquartz *= 8;
	} else {
		fquartz *= 4;
	}
	if (cdm->rstcfg & 0x40) {	/* from VCO to quartz */
		fquartz /= 12;
	} else {
		fquartz /= 16;
	}

	/*
	 * Check that value is reasonable otherwise use a default
	 */
	if (fquartz < QUARTZ_FREQ_MIN || fquartz > QUARTZ_FREQ_MAX) {
		fstatus = FREQ_BAD;
		fquartz = QUARTZ_FREQ;
	}
}
#endif

#if (!defined USE_SCTMR) && (!defined NO_FREQ_CALIBRATION)
void freqInit (void)
{
	cdm_regs *cdm;
	rtclock_regs *rtclock;
	unsigned int ticks;	
	uint8 secs;
#if 0
	uint32 ticks1, ticks2;
#endif

	fstatus = FREQ_OK;

	cdm = (cdm_regs *) (readMBAR () + MBAR_CDM);

	rtclock = (rtclock_regs *) (readMBAR () + MBAR_RTCLOCK);

	/*
	 * Wait for the seconds in the RTC to roll over 
	 */
	secs = rtclock->second_status;
	while (secs == rtclock->second_status) ;

	/*
	 * Start counting decrementer ticks 
	 */
	writeDEC (0xFFFFFFFFUL);

	/*
	 * Wait for the seconds in the RTC to roll over 
	 */
	secs = rtclock->second_status;
	while (secs == rtclock->second_status) ;

	/*
	 * Store the number of ticks for one second 
	 */
	ticks = 0xFFFFFFFFUL - readDEC ();

	/*
	 * Calculate quartz frequency from ticks per second
	 */
	fquartz = ticks * 4;		/* from DEC to XLB */
	if (cdm->rstcfg & 0x20) {	/* from XLB to VCO */
		fquartz *= 8;	
	} else {
		fquartz *= 4;
	}
	if (cdm->rstcfg & 0x40) {	/* from VCO to quartz */
		fquartz /=  12;
	} else {
		fquartz /=  16;
	}

	/*
	 * Check that value is reasonable otherwise use a default
	 */
	if (fquartz < QUARTZ_FREQ_MIN || fquartz > QUARTZ_FREQ_MAX) {
		fstatus = FREQ_BAD;
		fquartz = QUARTZ_FREQ;
	}
}
#endif

#if (defined NO_FREQ_CALIBRATION)
void freqInit (void)
{
	fquartz = QUARTZ_FREQ;
	fstatus = FREQ_OK;
}
#endif

/*! \brief Calculate the various frequencies used by the processor.
 *
 * This function must be called every time the settings of the Clock
 * Distribution Module (CDM) have been changed. Otherwise values returned by
 * other functions of this module might not reflect reality.
 */
void freqCalculate (void)
{
	cdm_regs *cdm;
	uint8 phase[4], phase_buf, i;

	cdm = (cdm_regs *) (readMBAR () + MBAR_CDM);

	if (fstatus == FREQ_UNINIT) {
		freqInit ();
	}

	/*
	 * Calculate VCO frequency
	 */
	if (cdm->rstcfg & 0x40) {
		fvco = fquartz * 12;
	} else {
		fvco = fquartz * 16;
	}	

	/*
	 * Calculate USB and IrDA frequencies
	 */	
	if (cdm->fd_enable == 0) {
		if (cdm->ext_48mhz_en & 0x01) {
			for(i = 0; i < 4; i++)
			{
				phase_buf = (uint8)((cdm->fd_counters & (0x7 << (i * 4))) >> (i * 4));
				if(phase_buf < 4)
					phase[i] = (uint8)(phase_buf + 0x8);
				else
					if(phase_buf < 6)	
						phase[i] = 11;
					else
						phase[i] = phase_buf;
			}
			firda = (fvco * 4) / (phase[0] + phase[1] + phase[2] + phase[3]);
		} else {
			firda = 0;
		}
		
		if (cdm->ext_48mhz_en & 0x02) {
			for(i = 0; i < 4; i++)
			{
				phase_buf = (uint8)((cdm->fd_counters & (0x7 << (i * 4))) >> (i * 4));
				if(phase_buf < 4)
					phase[i] = (uint8)(phase_buf + 8);
				else
					if(phase_buf < 6)	
						phase[i] = 11;
					else
						phase[i] = phase_buf;
			}
			fusb = (fvco * 4) / (phase[0] + phase[1] + phase[2] + phase[3]);			
		} else {
			fusb = 0;
		}
	} else {
		fusb = (4 * fvco) /
			(fracdiv[((cdm->fd_counters >> 12) & 7)] +
			fracdiv[((cdm->fd_counters >> 8) & 7)] + fracdiv[((cdm->fd_counters >> 4) & 7)] + fracdiv[((cdm->fd_counters) & 7)]);
		firda = fusb;		
	}

	/*
	 * Calculate XLB bus frequency
	 */
	if (cdm->rstcfg & 0x20) {
		fxlb = fvco / 8;
	} else {
		fxlb = fvco / 4;
	}

	/*
	 * Calculate Core frequency
	 */
	fcore = (fxlb * cpumult2[cdm->rstcfg & 0x1F]) / 2;

	/*
	 * Calculate IPBI bus frequency
	 */
	if (cdm->ipg_clk_sel) {
		fipbi = fxlb / 2;
	} else {
		fipbi = fxlb;
	}

	/*
	 * Calculate PCI bus frequency
	 */
	switch (cdm->pci_clk_sel) {
		case 0:
			fpci = fipbi;
			break;
			
		case 1:
			fpci = fipbi / 2;
			break;
		
		case 2:
			fpci = fxlb / 4;
			break;
			
		default:
			fpci = fipbi;
			break;
	}
}

/*! \brief Return frequency of the external quartz.
 *
 * \return Quartz frequency.
 */
unsigned int freqQuartz (void)
{
	if (fstatus == FREQ_UNINIT) {
		freqInit ();
	}
	
	return fquartz;
}

/*! \brief Return frequency of the system VCO.
 *
 * \return System VCO frequency.
 */
unsigned int freqVCO (void)
{
	if (fvco == 0) {
		freqCalculate ();
	}
	
	return fvco;
}

/*! \brief Return frequency of the core.
 *
 * \return Core frequency.
 */
unsigned int freqCore (void)
{
	if (fcore == 0) {
		freqCalculate ();
	}
	
	return fcore;
}

/*! \brief Return frequency of the XLB bus.
 *
 * \return XLB bus frequency.
 */
unsigned int freqXLB (void)
{
	if (fxlb == 0) {
		freqCalculate ();
	}
	
	return fxlb;
}

/*! \brief Return frequency of the IPBI bus.
 *
 * \return IPBI bus frequency.
 */
unsigned int freqIPBI (void)
{
	if (fipbi == 0) {
		freqCalculate ();
	}
	
	return fipbi;
}

/*! \brief Return frequency of the PCI bus.
 *
 * \return PCI bus frequency.
 */
unsigned int freqPCI (void)
{
	if (fpci == 0) {
		freqCalculate ();
	}
	
	return fpci;
}

/*! \brief Return frequency of the USB module.
 *
 * \return USB module frequency.
 */
unsigned int freqUSB (void)
{
	if (fusb == 0) {
		freqCalculate ();
	}
	
	return fusb;
}

/*! \brief Return frequency of the IrDA module.
 *
 * \return IrDA module frequency.
 */
unsigned int freqIrDA (void)
{
	if (firda == 0) {
		freqCalculate ();
	}
	
	return firda;
}

/*! \brief Set frequency ration (Core:IPBI:PCI).
 *
 * \param ratio		Either FREQ_RATIO_442, FREQ_RATIO_441, FREQ_RATIO_422 or
 *             		FREQ_RATIO_421.
 */
void freqSetRatio (int ratio)
{
	cdm_regs *cdm;
	ipbi_regs *ipbi;

	cdm = (cdm_regs *) (readMBAR () + MBAR_CDM);
	ipbi = (ipbi_regs *) (readMBAR () + MBAR_CS);

	switch (ratio) {
		case FREQ_RATIO_421:
	    	cdm->pci_clk_sel = 0x01;
			EIEIO;
		    cdm->ipg_clk_sel = 0x01;
			EIEIO;
			clrbit (ipbi-> ipbi_wait_state_en, IPBI_WAIT_STATE_EN);
			EIEIO;
			break;
			
		case FREQ_RATIO_422:
	    	cdm->pci_clk_sel = 0x00;
			EIEIO;
		    cdm->ipg_clk_sel = 0x01;
			EIEIO;
			clrbit (ipbi-> ipbi_wait_state_en, IPBI_WAIT_STATE_EN);
			EIEIO;
			break;
			
		case FREQ_RATIO_441:
			setbit (ipbi-> ipbi_wait_state_en, IPBI_WAIT_STATE_EN);
			EIEIO;
	    	cdm->pci_clk_sel = 0x02;
			EIEIO;
		    cdm->ipg_clk_sel = 0x00;
			EIEIO;
			break;
			
		case FREQ_RATIO_442:
			setbit (ipbi-> ipbi_wait_state_en, IPBI_WAIT_STATE_EN);
			EIEIO;
	    	cdm->pci_clk_sel = 0x01;
			EIEIO;
		    cdm->ipg_clk_sel = 0x00;
			EIEIO;
			break;
			
		default:
			break;
	}

	freqCalculate ();
}
