// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2015 Freescale Semiconductor, Inc. 
// All rights reserved
//
// This is unpublished proprietary source code of Freescale Semiconductor.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:      	    power.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@freescale.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION DATE AUTHOR DESCRIPTION
// 1.0 2015-06-30 	    Gary Milliorn
// -----------------------------------------------------------------------------
// KEYWORDS:  		    POWER PSEQ PWR
// -----------------------------------------------------------------------------
// PURPOSE:             Power sequencing: on a power switch (or "auto-on"), apply
//                      power to the ATX PSU, then each "tier" in order as required
//                      by the LS2085A.  On the next switch assertion, power down in
//                      the reverse order.
// -----------------------------------------------------------------------------
// PARAMETERS
//                      CLK_FREQ            core clock in kHz.
//                      SIM_MODE            set to 1 to accelerate simulation
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:   asynchronous, active low
// Clock Domains:        clk:     system main clock
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  y
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module power #(
   parameter CLK_FREQ = 66666,
   parameter SIM_MODE = 0
)
( 
    // General controls
    //
    input   wire			sw_pwr_b, 
    input   wire			sw_auto_on, 
    input   wire			sw_bypass_b, 
    input   wire			force_off_b,

	// -PB LTC power controller
	//
	input	wire			is_pb,
	input	wire			ltc_pwr_off,
	input	wire			ltc_pwr_on,

	// Power registers.
    //
	input	wire			pwr_ctl2_pwr,
    output  reg  [7:0]		pwr_reason,
    output  wire [7:0]		pwr_mstat,
    output  wire [7:0]		pwr_stat1,
    output  wire [7:0]		pwr_stat2,

    // ATX Power Controls
    //
    output  wire			ps_on_b, 
    input   wire			pwrgd, 
    output  reg				pwr_preload, 

    // PSU Power Controls
    //
    output  wire			ps_vdd_en,
    input   wire			ps_vdd_pg, 
    output  wire			ps_gvdd_en,
    input   wire			ps_gvdd_pg, 

    output  wire			ps_vtt_en,
    input   wire			ps_vtt_pg_b, 
    output  wire			ps_vpp_en,
    input   wire			ps_vpp_pg,
    output  wire			ps_ovdd_en,
    input   wire			ps_ovdd_pg, 
    output  wire			ps_xvdd_en,
    input   wire			ps_xvdd_pg_b, 

    output  wire			ps_2v5_en,
    input   wire			ps_2v5_pg_b, 
    output  wire			ps_1v2_en,
    input   wire			ps_1v2_pg_b, 
    output  wire			ps_0v85_en,
    input   wire			ps_0v85_pg_b, 
    output  wire			ps_dvdd_en,
    output  wire			ps_evdd_en,
    input   wire			ps_ta_bb_pg_b, 

    // Status outputs.
    //
    output  reg				disable_io_b, 
    output  reg  [1:0]		sps,
    output  reg				pwr_is_off, 
    output  wire			ps_faulted,
    output  wire [7:0]		pwr_state, 
	output	wire [3:0]		pwr_mon,
    output  reg  [7:0]		tier_report, 


	// LS1088ARDB-PB Additions.
	//
	input	wire			ps_5v0_pg,
	input	wire			ps_3v3_pg,
	output	wire			ps_pcie3v3_en,
	input	wire			ps_pcie3v3_pg,

    // Control.
    //
    input   wire			rst_b, 
    input   wire			clk_10kHz_ena,
    input   wire			clk
);


//===========================================================================
// Include standardized definitions and values.
//
`include <qixis_defs.h>


//===========================================================================
// Definitions.
//

// Number of tiers in use, excluding ATX/Block PSU.
//
	localparam	NO_TIERS	= 4'd4;


// FSM States

    localparam  [7:0]	PSEQ_IDLE           = 8'h00,
                        PSEQ_FORCEON        = 8'h08,
                        PSEQ_ENABLE         = 8'h10,
                        PSEQ_ENA_CHK        = 8'h20,
                        PSEQ_WAIT_ALL       = 8'h60,
                        PSEQ_STABLE         = 8'h80,
                        PSEQ_POWERUP        = 8'hA0,
                        PSEQ_PUP_DELAY      = 8'hB0,
                        PSEQ_PWAIT		    = 8'hC0,
                        PSEQ_DISABLE        = 8'hD0,
                        PSEQ_DIS_CHK        = 8'hDC,
                        PSEQ_AUTOSTALL      = 8'hE0,
                        PSEQ_FAULT          = 8'hF0; 



//===========================================================================
// Declarations.
//
    reg  [NO_TIERS:0]	tier_en,		next_tier_en;
    wire [NO_TIERS:0]	tier_pg;
    reg  [		 3:0]	tier_no,		next_tier_no;

	reg					tier_range_pg;
	wire				int_ps_on_b;


//===========================================================================
// TIER ASSIGNMENT
//		Assign  PS_xyz_EN to a tier to group power supplies
//		Monitor PS_xyz_PG for a tier to allow the FSM to proceed.

// TIER0:
//		ATX (always first)
//
    assign	ps_on_b			=							   ~tier_en[0];
	assign	int_ps_on_b		=							   ~tier_en[0];

	assign	tier_pg[0]		= (pwrgd);


// TIER1:
//
    assign ps_2v5_en      = (!disable_io_b) ? 1'b0 : tier_en[1];
    assign ps_1v2_en      = (!disable_io_b) ? 1'b0 : tier_en[1];
    assign ps_0v85_en     = (!disable_io_b) ? 1'b0 : tier_en[1];
    assign ps_ovdd_en     = (!disable_io_b) ? 1'b0 : tier_en[1];
    assign ps_dvdd_en     = (!disable_io_b) ? 1'b0 : tier_en[1];
    assign ps_evdd_en     = (!disable_io_b) ? 1'b0 : tier_en[1];
    assign ps_xvdd_en     = (!disable_io_b) ? 1'b0 : tier_en[1];
	assign ps_pcie3v3_en  = (!disable_io_b) ? 1'b0 : tier_en[1];

    assign tier_pg[1]     = (~ps_2v5_pg_b  & ~ps_1v2_pg_b 
                          &  ~ps_0v85_pg_b &  ps_ovdd_pg
                          //  DVDD has no PG
                          //  EVDD has no PG
                          &  ~ps_xvdd_pg_b &  ps_pcie3v3_pg);

// TIER2:
//
    assign ps_vdd_en      = (!disable_io_b) ? 1'b0 : tier_en[2];

    assign tier_pg[2]     = (ps_vdd_pg);

// TIER3:
//
    assign ps_vpp_en      = (!disable_io_b) ? 1'b0 : tier_en[3];

    assign tier_pg[3]     = (ps_vpp_pg);

// TIER4
//
    assign ps_gvdd_en     = (!disable_io_b) ? 1'b0 : tier_en[4];
    assign ps_vtt_en      = (!disable_io_b) ? 1'b0 : tier_en[4];

    assign tier_pg[4]     = (ps_gvdd_pg & ~ps_vtt_pg_b);
    

//---------------------------------------------------------------------------
// tier_report 	-- export info regarding PG for monitor to show, if error
//
	always @*
		case (tier_no)				// pragma parallel_case 
		'd0:		tier_report		<= { 7'b1111_111, pwrgd };
		'd1:		tier_report		<= { ~ps_2v5_pg_b,  ~ps_1v2_pg_b,
										 ~ps_0v85_pg_b,  ps_ovdd_pg,
									     1'b1,			 1'b1,
									     ~ps_xvdd_pg_b, ps_pcie3v3_pg
									   };
		'd2:		tier_report		<= { 7'b1111_111, ps_vdd_pg }; 
		'd3:		tier_report		<= { 7'b1111_111, ps_vpp_pg };
		'd4:		tier_report		<= { 6'b1111_11,  ps_gvdd_pg, ~ps_vtt_pg_b };
		default:	tier_report		<= 8'd0;
		endcase


//---------------------------------------------------------------------------
// tier_range_pg 	-- return 1 if <tier_no> and all lower tiers are all
//					   reporting power-good.
	always @*
		case (tier_no)				// pragma parallel_case 
		'd0:		tier_range_pg	<= tier_pg[ 0 ] ==      1'b1;
		'd1:		tier_range_pg	<= tier_pg[1:0] ==     2'b11;
		'd2:		tier_range_pg	<= tier_pg[2:0] ==    3'b111;
		'd3:		tier_range_pg	<= tier_pg[3:0] ==   4'b1111;
		'd4:		tier_range_pg	<= tier_pg[4:0] ==  5'b11111;
		'd5:		tier_range_pg	<= tier_pg[4:0] ==  5'b11111;		// FIXME: testing.

		default:	tier_range_pg	<= 1'b0;
		endcase


//---------------------------------------------------------------------------
// PWR_MSTAT/PWR_STATx -- register bit assignments.
//
    assign pwr_mstat = { ~int_ps_on_b,  
                         pwrgd, 
                         1'b0,
                         1'b0,
                         1'b1, 
                         1'b0, 
                         (pwr_is_off) ? 2'b00 : 2'b11       // Only S0/S3 supported.
                       };

    assign pwr_stat1 = { 1'b1, 
                         ps_pcie3v3_pg,
                         ~ ps_xvdd_pg_b,
                         ~ ps_ta_bb_pg_b,
                         1'b1,
                         ~ ps_0v85_pg_b, 
                         ps_vpp_pg,
                         ps_vdd_pg 
                       }; 
 
    assign pwr_stat2 = { ps_5v0_pg,
                         ps_3v3_pg,
                         ~ ps_1v2_pg_b, 
                         ps_ovdd_pg, 
                         1'b1, 
                         ~ ps_2v5_pg_b, 
                         ~ ps_vtt_pg_b, 
                         ps_gvdd_pg 
                       };


//===========================================================================
// Power Event Conditioning.

// SWITCH:		On the rising (deassertion) edge, drive <pwr_sig_b> low
//				for one clock cycle.

    reg  [1:0]  sw_edge;
    reg         pwr_clr_b, pwr_sig_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                     sw_edge     <= 2'b11;
        else if (clk_10kHz_ena)         sw_edge     <= { sw_edge[0], sw_pwr_b };

    always @(negedge rst_b  or  negedge pwr_clr_b  or  posedge clk)
        if (!rst_b  ||  !pwr_clr_b)     pwr_sig_b   <= 1'b1;
        else if (clk_10kHz_ena)         pwr_sig_b   <= ~(sw_edge == 2'b01);


// PWR_CTL2[7]:		Software power-off/-on (where available)
//					On the rising (assertion) edge, drive <pwr_reg_b> low
//					for one clock cycle.
//  Notes:
//  00 - idle
//  01 - t+0:   bit set
//  11 - t+1:   bit shifted, assert power-cycle flag now.
//  10 - t+n:   bit cleared, likely power-cycle caused reset
//  00 - t+n+1: "

    reg  [1:0]  pwrbit_edge;
    reg         pwr_reg_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)                     pwrbit_edge    <= 2'b00;
        else if (clk_10kHz_ena)         pwrbit_edge    <= { pwrbit_edge[0], pwr_ctl2_pwr };

    always @(negedge rst_b  or  negedge pwr_clr_b  or  posedge clk)
        if (!rst_b  ||  !pwr_clr_b)     pwr_reg_b   <= 1'b1;
        else if (clk_10kHz_ena)         pwr_reg_b   <= ~(pwrbit_edge == 2'b11);


//===========================================================================
// Timers:
//		Units=100us
//		The timers count down to zero, and stay there when expired.  The
//		time-out trigger is at t=1, which gives us a 1-cycle window to
//		cascade timer events without danger of a fall through.

    wire [15:0]  pseq_timer_tier,		pseq_timer_atx_off;
    wire [15:0]  pseq_timer_disable,	pseq_timer_wait_all;
    wire [15:0]  pseq_timer_startup;

	assign  pseq_timer_tier     =  (SIM_MODE) ? 16'd1	:	16'd_500_0;     //  500ms stall for slow PSUs
	assign  pseq_timer_wait_all =  (SIM_MODE) ? 16'd1	:	16'd_100_0;		//  100ms for untracked PSUs
	assign  pseq_timer_disable  =  (SIM_MODE) ? 16'd1	:	16'd__50_0;		//   50ms stall for all PSUs
	assign  pseq_timer_startup  =  (SIM_MODE) ? 16'd1	:	16'd2000_0;		// 1000ms stall before (re)power
	assign  pseq_timer_atx_off  =  (SIM_MODE) ? 16'd1	:	16'd2000_0;		// 2000ms stall for ATX off




//===========================================================================
// Sync FSM elements.
//
    reg  [ 7:0] pseq_state,     next_pseq_state;

    reg  [15:0]	pseq_timer,     next_pseq_timer;
	reg  [ 1:0]					next_sps;

    reg                         next_disable_io_b;
    reg                         next_pwr_preload;
    reg                         next_pwr_is_off;
    reg                         next_pwr_clr_b;
    reg  [ 7:0]                 next_pwr_reason;


    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             pseq_state      <= PSEQ_POWERUP;
        else if (clk_10kHz_ena) pseq_state      <= next_pseq_state;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             pseq_timer		<= 16'd0;
        else if (clk_10kHz_ena)	pseq_timer		<= next_pseq_timer;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             sps				<= SPS_OFF;
        else if (clk_10kHz_ena) sps				<= next_sps;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             tier_en			<= { NO_TIERS{1'b0} };
        else if (clk_10kHz_ena) tier_en			<= next_tier_en; 

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             tier_no			<= 4'b1111;
        else if (clk_10kHz_ena) tier_no			<= next_tier_no; 

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             pwr_reason      <= { PWREVT_NONE, PWREVT_NONE };
        else if (clk_10kHz_ena) pwr_reason      <= next_pwr_reason; 

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             disable_io_b    <= 1'b0;
        else if (clk_10kHz_ena) disable_io_b    <= next_disable_io_b;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             pwr_preload     <= 1'b1;
        else if (clk_10kHz_ena) pwr_preload     <= next_pwr_preload;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             pwr_is_off      <= 1'b1;
        else if (clk_10kHz_ena) pwr_is_off      <= next_pwr_is_off;

    always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             pwr_clr_b       <= 1'b0;
        else if (clk_10kHz_ena) pwr_clr_b       <= next_pwr_clr_b;


// Timer-timeout (non-registered).
//
    reg         pseq_timer_sig;

	always @*
		pseq_timer_sig	<= (pseq_timer == 16'd0);


//---------------------------------------------------------------------------
// Next state decoding.
//
    always @* begin
        next_pseq_state     <= pseq_state;
        next_pseq_timer     <= (|pseq_timer) ? pseq_timer - 16'd1 : pseq_timer;

		next_sps			<= sps;

        next_tier_en        <= tier_en;
        next_tier_no        <= tier_no;

        next_pwr_clr_b      <= pwr_clr_b;
        next_pwr_reason     <= pwr_reason;
        next_pwr_is_off     <= pwr_is_off;

        next_disable_io_b   <= disable_io_b;
		next_pwr_preload	<= pwr_preload;


        case (pseq_state)                       // pragma parallel_case   


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// F0|FAULT -- fault detected.
//      Force all power off immediately.  Stay in this state until the fault clears,
//      then a power-event resets the FSM and goes to IDLE state to await a power-up event.
//
//		If <sw_bypass_b> is on, power remains on for debug support.
//
//      Note that <sw_pwr_b> serves as a "clear power fault status" here and
//      so it takes a double-power tap to restart.

        PSEQ_FAULT: begin
            next_pwr_clr_b		<= 1'b1;							// Allow pwr sigs

			next_pwr_preload	<= 1'b0;

			// Unless in debug-mode, clear <tier_en> to power-off all PSUs, but leave 
			// <tier_no> set so we can detect the error source.
			//
			if (~sw_bypass_b) begin
				next_disable_io_b   <= 1'b1;						// Remain on.

			end else begin
				next_pwr_is_off     <= 1'b1;						// All off
				next_disable_io_b   <= 1'b0;						// "
				next_tier_en        <= 'd0;							// "
			end

			// May exit fault state if fault cleared.
			//
            if (!pwr_sig_b  &&  force_off_b)
				next_pseq_state <= PSEQ_IDLE;

			next_sps			<= SPS_OFF;
        end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// C0|PWAIT -- For the -PB series, external power might stay on a while,
//			   and since we use that as a signal, wait for it to go away.

        PSEQ_PWAIT: begin
			if (~ltc_pwr_on)
				next_pseq_state	<= PSEQ_IDLE;
			if (!pwr_sig_b)
				next_pseq_state	<= PSEQ_IDLE;
		end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// DC|DIS_CHK -- power down next rail and continue until done.
//				 Unlike power-up, on power-down a timeout means
//				 keep powering down even if PG is still high.

        PSEQ_DIS_CHK: begin
			if ((tier_range_pg == 1'b0)  ||  (pseq_timer_sig)) begin
				if (tier_no == 4'd0  &&  sw_auto_on)
					next_pseq_state	<= PSEQ_POWERUP;

				// Special logic for -PB
				//
				else if (tier_no == 4'd0  &&  (pwr_reason[3:0] == PWREVT_PWROFF))
					next_pseq_state	<= PSEQ_PWAIT;

				else if (tier_no == 4'd0)
					next_pseq_state	<= PSEQ_IDLE;

				else
					next_pseq_state	<= PSEQ_DISABLE;
			end

			next_sps				<= SPS_DEPOWERING;
		end


// D0|DISABLE:	Power Down
//				Remove enables from tierN..1, in that order.
//				Note that IOs are enabled until we reach POWERUP or IDLE, otherwise we
//				could not control order of powerdown.
//				When done, if we are in the AUTO_ON mode, use POWERUP to stall before
//				reapplying power.

        PSEQ_DISABLE: begin
            next_pwr_is_off			<= 1'b1;							// Report powering down
            next_pwr_clr_b			<= 1'b0;							// No pwr sigs

			next_pwr_preload		<= 1'b1;							// Enable preload

			next_tier_en			<= { 1'b0, tier_en[NO_TIERS:1] };
			next_tier_no			<= tier_no - 4'd1;

			next_pseq_timer			<= pseq_timer_disable;
			next_pseq_state			<= PSEQ_DIS_CHK;
			next_sps				<= SPS_DEPOWERING;
		end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// 80|STABLE -- normal idle.
//      All power supplies are up; wait for power down events or faults.
//
        PSEQ_STABLE: begin
            next_pwr_is_off         <= 1'b0;							// Report fully on now.
            next_pwr_clr_b			<= 1'b1;							// Allow pwr sigs
			next_pseq_timer			<= 'd0;

			next_pwr_preload		<= 1'b0;							// Done with preload

            if (!force_off_b) begin                         // Fault?
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_FORCE };
                next_pseq_state     <= PSEQ_FAULT;
            end

            else if (!pwrgd) begin                          // Lost mains
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_ATXOFF };
                next_pseq_timer     <= pseq_timer_disable;
                next_pseq_state     <= PSEQ_DISABLE;
            end

            else if (!pwr_sig_b) begin                  // Start powering down.
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_PWROFF };
                next_pseq_timer     <= pseq_timer_disable;
                next_pseq_state     <= PSEQ_DISABLE;
            end                    

            else if (ltc_pwr_off) begin                 // Start powering down.
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_PWROFF };
                next_pseq_timer     <= pseq_timer_disable;
                next_pseq_state     <= PSEQ_DISABLE;
            end                    

            else if (!pwr_reg_b) begin                  // Start powering down.
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_REGOFF };
                next_pseq_timer     <= pseq_timer_disable;
                next_pseq_state     <= PSEQ_DISABLE;
            end                    

			next_sps				<= SPS_ON;
        end


// 60|WAIT_ALL
//      Wait for non-tracked power supplies to stabilize.
//      If any power supplies go down, that's a fault.  Same if external
//      faults force a shutdown.
//
//      Note that the timer is not an error here, we use it to give PSUs
//      without PG a change.  Keep <wait_all> to the minimum accordingly.

        PSEQ_WAIT_ALL: begin
			if (tier_range_pg == 1'b0) begin
                next_pwr_reason		<= { pwr_reason[3:0], PWREVT_PFAULT };
                next_pseq_state     <= PSEQ_FAULT;

            end else if (!force_off_b) begin
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_FORCE };
                next_pseq_state     <= PSEQ_FAULT;

            end else if (pseq_timer_sig) begin
                next_pseq_state     <= PSEQ_STABLE;
            end

			next_sps				<= SPS_POWERING;
        end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// 20|ENA_CHK
//      A tier was enabled.  On PG for this tier and all lower, go to the next 
//      tier (if any).
//      On timeout (a tier would not come up), go to the FAULT state.

        PSEQ_ENA_CHK: begin

			// This tier and all below good?  Either do the next tier, or
			// we are done.
			//
			if (tier_range_pg == 1'b1) begin
				if (tier_no == NO_TIERS) begin
					next_pseq_timer	<= pseq_timer_wait_all;			// May need this later
					next_pseq_state	<= PSEQ_WAIT_ALL;

				end else begin
					next_pseq_state	<= PSEQ_ENABLE;
				end
			end

			// Forced off by alarm/fault handler?
			//
			else if (!force_off_b) begin
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_FORCE };
                next_pseq_state		<= PSEQ_FAULT;

			// Timeout, a problem in this tier.  With -PB, we can't tell
			// ATX power-loss as directly.
			//
			end else if (pseq_timer_sig) begin
				if (!pwrgd) begin                          // Lost mains
					next_pseq_timer     <= pseq_timer_disable;
					next_pwr_reason     <= { pwr_reason[3:0], PWREVT_ATXOFF };
					next_pseq_state     <= PSEQ_DISABLE;
				//end else begin
				//	next_pwr_reason		<= { pwr_reason[3:0], PWREVT_PFAULT };
				//	next_pseq_state		<= PSEQ_FAULT;
				end
			end

			next_sps				<= SPS_POWERING;
        end


// 10|ENABLE: Wait for tiers of power supplies to power-up.  On entry,
//			  <tier_en> was set to 00..001, to enable tier 0 (ATX).  When that tier
//			  is completely powered-up, go to the next tier.
//			  Tier0:	ATX PSU
//			  Tier1-n:  system-defined.
//
//			  <disable_io_b> is de-asserted starting now.

        PSEQ_ENABLE: begin
			next_tier_en		<= { tier_en[NO_TIERS-1:0], 1'b1 };
			next_tier_no		<= tier_no + 4'd1;

            next_disable_io_b   <= 1'b1;				// Enable IO 
			next_pwr_preload	<= 1'b1;				// Enable preload
            next_pwr_clr_b      <= 1'b0;

			next_pseq_timer		<= pseq_timer_tier;
			next_pseq_state		<= PSEQ_ENA_CHK;
			next_sps			<= SPS_POWERING;
		end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// 08|FORCEON
//      SW_AUTO_ON was on, so try to power-up the system.  If the system was
//      shut down due to ATX power loss, this will be unsuccessful until AC
//      is restored.  Otherwise, it is slightly redundant vs. ENABLE.
//      IO is tristated, but ATX_ON_B is driven always.
//
		PSEQ_FORCEON: begin
			next_tier_en		<=  'd1;				// Turn on ATX now
			next_tier_no		<= 4'd0;

			next_pwr_preload	<= 1'b1;				// Preload off
            next_pwr_clr_b      <= 1'b0;

			if ((!is_pb  &&  pwrgd)						// PA powered up
			||  ( is_pb  &&  ltc_pwr_on)) begin			// PB powered up
                next_pseq_timer     <= pseq_timer_tier;
                next_pseq_state     <= PSEQ_ENABLE;
			end
		end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// 00|IDLE
//      Wait for power events, then start power sequencing.
//
        PSEQ_IDLE: begin
            next_tier_en			<= 'd0;			// All off.
            next_tier_no			<= 4'b1111;		// All 1's == no tier number.

            next_disable_io_b		<= 1'b0;        // Tri-state IOs
			next_pwr_preload		<= 1'b0;		// Preload off
            next_pwr_is_off			<= 1'b1;
            next_pwr_clr_b			<= 1'b1;		// Allow pwr sigs now

            if (!pwr_sig_b) begin
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_PWRON };
                next_pseq_state     <= PSEQ_ENABLE;
            end

            else if (!pwr_reg_b) begin
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_REGON };
                next_pseq_state     <= PSEQ_ENABLE;
            end

            else if (sw_auto_on)  begin
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_AUTOON };
                next_pseq_state     <= PSEQ_FORCEON;
            end

            else if (ltc_pwr_on)  begin
                next_pwr_reason     <= { pwr_reason[3:0], PWREVT_PMTOG };
                next_pseq_state     <= PSEQ_ENABLE;
            end

			next_sps				<= SPS_OFF;
        end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// B0|PUP_DELAY
//      Wait for system power-up delay timer to expire.
//
        PSEQ_PUP_DELAY: begin
			if (pseq_timer_sig) begin
				next_pseq_state		<= PSEQ_IDLE;
				next_sps			<= SPS_OFF;
			end
        end

// A0|POWERUP
//      Wait for ATX PSU to stabilize (so we can sample switches safely, etc).
//
        PSEQ_POWERUP: begin
            next_pwr_clr_b			<= 1'b0;
            next_disable_io_b		<= 1'b0;        // Tri-state IOs
			next_pwr_preload		<= 1'b0;		// Preload off

            next_tier_en			<= 'd0;			// All off.
            next_tier_no			<= 'd0;			// "

            next_pseq_timer			<= pseq_timer_startup;
            next_pseq_state			<= PSEQ_PUP_DELAY;
			next_sps				<= SPS_OFF;
        end


//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// XX|(recovery state)
//
        default: begin
            next_pwr_reason			<= { pwr_reason[3:0], PWREVT_INVAL };
            next_pseq_state			<= PSEQ_POWERUP;
			next_sps				<= SPS_OFF;
        end
        endcase
    end


//===========================================================================
// Parallel FSM Functions

// Assert on any fault.
//
    assign  ps_faulted	= (pseq_state == PSEQ_FAULT);


// Export PSEQ state.  The MSB is the most informative, and different IPs
// use the upper or lower 4 bits, so just replicate it.  During power up/
// down states, replace the LSB with the tier in use.
//
	assign	pwr_state	= (ps_faulted)				? { 4'h0,			 tier_no+4'd1    }
						: (sps == SPS_POWERING  )	? { 4'h0,			 tier_no+4'd1    }
						: (sps == SPS_DEPOWERING)	? { 4'h0,			 tier_no+4'd1    }
						:							  { pseq_state[7:4], pseq_state[7:4] };


//---------------------------------------------------------------------------
// Monitoring and unused
//
	assign	pwr_mon[3]	= (pseq_state == PSEQ_ENA_CHK  &&  tier_no == 4'd3);
	assign	pwr_mon[2]	= (pseq_state == PSEQ_ENA_CHK  &&  tier_no == 4'd2);
	assign	pwr_mon[1]	= (pseq_state == PSEQ_ENA_CHK  &&  tier_no == 4'd1);
	assign	pwr_mon[0]	= (pseq_state == PSEQ_ENA_CHK  &&  tier_no == 4'd0);

endmodule
