// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2014 Freescale Semiconductor, Inc. 
// All rights reserved
//
// This is unpublished proprietary source code of Freescale Semiconductor.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:           Fault.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@freescale.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION DATE AUTHOR DESCRIPTION
// 1.0 2012-02-12       Gary Milliorn
// 1.1 2012-08-21       Gary Milliorn	PS_VDD signals are open-drain active
//										high.
// 1.2 2014-10-14       Gary Milliorn	Greatly simplified version.
// -----------------------------------------------------------------------------
// KEYWORDS:            FAULT MONITOR
// -----------------------------------------------------------------------------
// PURPOSE:             Fault Detector
// -----------------------------------------------------------------------------
// PARAMETERS
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:         n/a
// Clock Domains:        clk:           n/a
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  Yes
// Other: 
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module fault (

   // Environment
   //
   input   wire           dut_present_b, 
   input   wire    [7:0]  alarm_mask, 

   // Input faults
   //
   input   wire           vdd_therm_b, 
   input   wire           therm_warn_b, 
   input   wire           therm_fault_b, 
   input   wire           ptempoff_fault,
   output  wire           force_off_b, 
   input   wire           ctl_fail,

   // Output indicators
   //
   output  wire    [7:0]  alarm_stat, 
   output  wire           in_alarm_b
);


//---------------------------------------------------------------------------
// ALARM core:
//   Set a bit to '1' for each alarm:
//
//	    b7:				   unused.
//	    b6:		PSQF	-- power_seq fault
//	    b5:		VCFG	-- power config fault (supply at X when only Y supported, etc.)
//	    b4:		PDOWN	-- powerdown fault
//	    b3:				   unused.
//	    b2:		VDDTMP	-- VDD power supply over-temp
//	    b1:		TWARN	-- DUT overtemp warning
//	    b0:		TFAULT	-- DUT overtemp fault
//
	wire [7:0]		alarms, imask;

	assign	alarms = {  1'b0,				// rsvd
					    1'b0,			    // PWR SEQ fault
						ctl_fail,           // CTL[FAIL] set
						ptempoff_fault,	    // overtemp forced off
						1'b0,          	    // rsvd (rotation error not supported by LS1088A)
						~ vdd_therm_b,	    // VDD PSU overtemp fault
						~ therm_warn_b,	    // ADT7461 thermal warning.
						~ therm_fault_b	    // ADT7461 thermal failure.
					 };


// Mask some alarms if there is no device installed.
//
	assign	imask = (dut_present_b)     ? 8'b0001_1011      // mask: therm
                  :                       8'b0000_0000;     // mask: none.


// An alarm is present if any non-masked alarm has occurred.
// "alarm_mask", from the BCSRs, defaults to 0x02
//
	assign	alarm_stat	    = alarms & ~imask & ~alarm_mask;


// Set "in_alarm_b" if there is a severe problem, to drive status on the LEDS.  
//
	assign	in_alarm_b		= (alarm_stat == 8'h00) ? 1'b1 : 1'b0;


// Assert "force_off_b" for any critical error:
//	 * temperature fault	-- temp > 127C, only when device is present, as the thermal
//							   monitors are erratic when no diode is present.
//
	assign	force_off_b		= (alarm_stat[2])	                  ? 1'b0	// VDD_TEMP
							: (alarm_stat[0]  &&  !dut_present_b) ? 1'b0	// TFAULT
							:										1'b1;


endmodule

