// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2016 NXP Semiconductors.
// All rights reserved
//
// This is unpublished proprietary source code of NXP Semiconductors.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
// -----------------------------------------------------------------------------
// FILE NAME:           clocks.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR'S EMAIL:      gary.milliorn@nxp.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE        AUTHOR DESCRIPTION
// 1.0      2015-01-15  Gary Milliorn         Original
// -----------------------------------------------------------------------------
// KEYWORDS:            CLOCK PRESCALE
// -----------------------------------------------------------------------------
// PURPOSE:             Global prescaled clock enables for slower peripherals.
// -----------------------------------------------------------------------------
// PARAMETERS
//                      CLK_FREQ            core clock in kHz.
//                      SIM_MODE            set to 1 to accelerate simulation
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:      HOT_RST_B:     asynchronous, active low
// Clock Domains:       HOT_CLK:       25 MHz system main clock
// Critical Timing:     <none>
// Test Features:       testmode_b:    If asserted, clock setup is bypassed.
// Asynchronous I/F:    <none>
// Scan Methodology:    <none>
// Instantiations:      <none>
// Synthesizable (y/n): Yes
// Other:
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module clocks #( 
    parameter   CLK_FREQ = 66666,
    parameter   SIM_MODE = 0
) (
    input   wire        clk, 
    input   wire        rst_b,

    output  reg         clk_10kHz_ena,
    output  reg         clk_100Hz_ena,
    output  reg         clk_4Hz_ena
);


//---------------------------------------------------------------------------
// Scale the core clock down to 10kHz
// This is used for the power/reset sequencers.
//
    wire [31:0] clk_param;
    wire [11:0] clk_10kHz_limit;
    reg  [11:0] clk_10kHz_div,      clk_10kHz_next;
    wire        clk_10kHz_zero;

    assign  clk_param       = (CLK_FREQ * 1000 / 10000) - 1;
    assign  clk_10kHz_limit = clk_param[11:0];

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_10kHz_div   <= 12'd0;
        else                    clk_10kHz_div   <= clk_10kHz_next;

    assign  clk_10kHz_zero  = ~(|clk_10kHz_div);

    always @*
        clk_10kHz_next  <=  (clk_10kHz_zero)    ?  clk_10kHz_limit
                        :                          clk_10kHz_div - 12'd1; 

// clk_10kHz_ena is a 1-core-clock pulse.
//
	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_10kHz_ena   <= 1'd0;
        else                    clk_10kHz_ena   <= clk_10kHz_zero;


//---------------------------------------------------------------------------
// Continue the chain to 100 Hz (10ms) for the debouncers.
//
    wire [7:0]  clk_100Hz_limit;
    reg  [7:0]  clk_100Hz_div,      clk_100Hz_next;
    wire        clk_100Hz_zero;
    reg [ 1:0]  clk_100Hz_pulse;

    assign  clk_100Hz_limit = 8'd100;     // 10k/100

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_100Hz_div   <= 8'd0;
        else if (clk_10kHz_ena) clk_100Hz_div   <= clk_100Hz_next;

    assign  clk_100Hz_zero  = ~(|clk_100Hz_div);

    always @*
        clk_100Hz_next  <=  (clk_100Hz_zero)    ?  clk_100Hz_limit
                        :                          clk_100Hz_div - 8'd1; 

// Watch for 0->1 transitions at core clock rate.
//
	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_100Hz_pulse <= 2'b00;
        else                    clk_100Hz_pulse <= { clk_100Hz_pulse[0], clk_100Hz_zero };

// Output a single pulse on that transition.
//
	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_100Hz_ena   <= 1'd0;
        else                    clk_100Hz_ena   <= (clk_100Hz_pulse == 2'b01);


//---------------------------------------------------------------------------
// Continue the chain to 4 Hz (250ms) for the LED display.
//
    wire [7:0]  clk_4Hz_limit;
    reg  [7:0]  clk_4Hz_div,        clk_4Hz_next;
    wire        clk_4Hz_zero;
    reg [ 1:0]  clk_4Hz_pulse;

    assign  clk_4Hz_limit = 8'd25;     // 100/4

	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_4Hz_div   <= 8'd0;
        else if (clk_100Hz_ena) clk_4Hz_div   <= clk_4Hz_next;

    assign  clk_4Hz_zero  = ~(|clk_4Hz_div);

    always @*
        clk_4Hz_next    <=  (clk_4Hz_zero)      ?  clk_4Hz_limit
                        :                          clk_4Hz_div - 8'd1; 

// Watch for 0->1 transitions at core clock rate.
//
	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_4Hz_pulse   <= 2'b00;
        else                    clk_4Hz_pulse   <= { clk_4Hz_pulse[0], clk_4Hz_zero };

// Output a single pulse on that transition.
//
	always @(negedge rst_b  or  posedge clk)
		if (!rst_b)             clk_4Hz_ena     <= 1'd0;
        else                    clk_4Hz_ena     <= (clk_4Hz_pulse == 2'b01);


endmodule
