// vim: ts=4 softtabstop=4 shiftwidth=4 columns=120 lines=48
// +FHDR------------------------------------------------------------------------
// Copyright (c) 2014,2017 NXP Semiconductors.
// All rights reserved
//
// This is unpublished proprietary source code of NXP Semiconductors.
// The copyright notice above does not evidence any actual
// or intended publication of such source code.
//
// Freescale Confidential Proprietary
// -----------------------------------------------------------------------------
// FILE NAME:           WatchDog.v
// DEPARTMENT:          Austin Hardware Design
// AUTHOR:              Gary Milliorn
// AUTHOR�S EMAIL:      gary.milliorn@nxp.com
// -----------------------------------------------------------------------------
// RELEASE HISTORY
// VERSION  DATE		AUTHOR                DESCRIPTION
// 1.0		2012-11-13	Gary Milliorn 
// 1.1		2017-0919   Gary Milliorn		Suppress warnings.
// -----------------------------------------------------------------------------
// KEYWORDS:            WATCHDOG
// -----------------------------------------------------------------------------
// PURPOSE:             Asserts reset if watchdog timer is enabled, then not disabled 
//                      within <rwatch>*2.0s time. 
//                      Note that this is not a conventional watchdog timer, in that you
//                      cannot restart it once it is running.
// -----------------------------------------------------------------------------
// PARAMETERS
//                      CLK_FREQ            core clock freq. in kHz.
// -----------------------------------------------------------------------------
// REUSE ISSUES
// Reset Strategy:       rst_b:             asynchronous, active low
// Clock Domains:        clk:               main clock
// Critical Timing:      <none>
// Test Features:        <none>
// Asynchronous I/F:     <none>
// Scan Methodology:     <none>
// Instantiations:       <none>
// Synthesizable (y/n):  Yes
// Other:
// -FHDR------------------------------------------------------------------------

`resetall
`timescale 1ns/10ps

module WatchDog #(
   parameter CLK_FREQ = 66666
)
( 
	// Config inputs
	//
	input   wire    [7:0]	watch_time, 
	input   wire			watch_enable, 
	input   wire			sys_enable_b, 

	output  reg				wdog_evt_b, 

	// Monitoring.
	//
	output  wire	[7:0]	wdog_mon, 
	output	reg 			wdog_tick,

	input   wire			rst_b,
	input   wire			clk
);



//---------------------------------------------------------------------------
// The WatchDog is very slow, so we need a 0.5Hz clock.  To keep the
// FSM speedy and syncronized, we will use a 2Hz clock and scale the
// watchdog timer by 4X.
//
	wire [31:0] wdog_div_calc;
	wire [28:0] wdog_div_limit;
	reg  [28:0] wdog_div;
	reg		    wdog_clk_en;
   
    assign  wdog_div_calc  = (CLK_FREQ * 1000 / 2);
    assign  wdog_div_limit = wdog_div_calc[28:0];

	always @(negedge rst_b  or  posedge clk) begin
		if (!rst_b) begin
			wdog_clk_en <=  1'b0;
			wdog_div    <= 29'd0;
			wdog_tick	<=	1'b0;

		end else if (wdog_div == wdog_div_limit) begin
			wdog_clk_en	<=  1'b1;
			wdog_div    <= 29'd0;
			wdog_tick	<= ~wdog_tick;

		end else begin
			wdog_clk_en <= 1'b0;
			wdog_div    <= wdog_div + 29'd1;
		end
	end


//---------------------------------------------------------------------------
// If enabled, countdown from WATCH to zero.
//
    localparam [2:0] WDOG_IDLE       = 3'b001,
                     WDOG_COUNT      = 3'b010,
                     WDOG_TIMEOUT    = 3'b100;

	reg   [2:0] wdog_state, wdog_nextstate;                 // pragma attribute wdog_state safe_fsm TRUE 
                                                            // pragma attribute wdog_state fsm_state ONEHOT

    reg   [9:0] wdog,       wdog_next;
    reg                     wdog_next_evt_b;
    
	always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             wdog_state  <= WDOG_IDLE;
        else if (wdog_clk_en)   wdog_state  <= wdog_nextstate;

	always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             wdog        <= 10'd0;
        else if (wdog_clk_en)   wdog        <= wdog_next;

	always @(negedge rst_b  or  posedge clk)
        if (!rst_b)             wdog_evt_b  <= 1'b1;
        else if (wdog_clk_en)   wdog_evt_b  <= wdog_next_evt_b;

    always @* begin
        wdog_next       <= wdog;
        wdog_nextstate  <= wdog_state;
        wdog_next_evt_b <= wdog_evt_b;

        case (wdog_state)									// pragma parallel_case

// 100|TIMEOUT	-- assert reset until "enable" is cleared (generally with a reset).
//
        WDOG_TIMEOUT: begin
            wdog_next       <= wdog;
            wdog_nextstate  <= (!watch_enable) ? WDOG_IDLE
                             : (sys_enable_b)  ? WDOG_IDLE
                             :                   WDOG_TIMEOUT;
            wdog_next_evt_b <= 1'b0;
        end

// 010|COUNT	-- watchdog running, if disabled, or if the system resets, return to IDLE 
//				   (reloading WDOG), 
//				   else on zero go to TIMEOUT.
//
        WDOG_COUNT: begin
            wdog_next       <= wdog - 10'd1;
            wdog_nextstate  <= (!watch_enable) ? WDOG_IDLE
                             : (sys_enable_b)  ? WDOG_IDLE
                             : (wdog == 10'd0) ? WDOG_TIMEOUT
                             :                   WDOG_COUNT;
            wdog_next_evt_b <= 1'b1;
        end

// 001|IDLE		-- keep WDOG tracking WATCH, and start counting when enabled.
//				   "sys_enable_b" keeps the watchdog from running during
//				   power-down or reset, for BootBox cases.
//
        WDOG_IDLE: begin
            wdog_next       <= { watch_time, 2'b01 };
            wdog_nextstate  <= (watch_enable & !sys_enable_b) ? WDOG_COUNT : WDOG_IDLE;
            wdog_next_evt_b <= 1'b1;
        end

// (recovery)
//
        default: begin
            wdog_nextstate  <= WDOG_IDLE;
        end
        endcase
    end


//---------------------------------------------------------------------------
// Monitoring outputs.
//
    assign	wdog_mon	= wdog[9:2];


//---------------------------------------------------------------------------
// Track unused nets.
//
	wire		unused;
	
	assign	unused		= (|wdog_div_calc[31:29]);

endmodule
