/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palI18092mPI
{
    #region DEFINES

    public enum Error : byte
    {
        /// <summary> ISO18092 error recovery failed. </summary>
        ERR_RECOVERY_FAILED = (CustomCodes.ERROR_BEGIN)
    }

    public enum DeselectCommand : byte
    {
        DSL = 0x08,         /**< DSL is sent for deselection of target. */
        RLS = 0x0A          /**< RLS is sent for deselection of target. */
    };

    public enum Config : ushort
    {
        PACKETNO = 0x0000,  /**< Set / Get packet number. */
        DID,                /**< Set / Get Device Identifier. */
        NAD,                /**< Set / Get Node Address. */
        WT,                 /**< Set / Get Waiting Time. */
        FSL,                /**< Set / Get PCD & PICC Frame Length. */
        MAXRETRYCOUNT       /**< Set / Get Maximum Retry Count. */
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_Atr(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte[] pNfcid3i,        /**< [In] NFCID3 - randomly generated in case of 106kbps initial datarate or
                                                   NFCID2 in case of 212/424kbps datarate; uint8_t[10] */
            byte bDid,              /**< [In] Device Identifier; '0' to disable usage else 1-14. */
            byte bLri,              /**< [In] Length Reduction of Initiator; 0-3  */
            byte bNadEnable,        /**< [In] Enable usage of Node Address; Unequal '0' to enable. */
            byte bNad,              /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte[] pGi,             /**< [In] Optional General Information bytes. */
            byte bGiLength,         /**< [In] Number of General Information bytes. */
            byte[] pAtrRes,         /**< [Out] Attribute Response; uint8_t[64] */
            ref byte pAtrResLength  /**< [Out] Attribute Response Length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_Psl(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            byte bDsi,          /**< [In] Divisor Send (target to initiator) Integer; 0-2 */
            byte bDri,          /**< [In] Divisor Receive (initiator to target) Integer; 0-2 */
            byte bFsl           /**< [In] Frame Length Byte; 0-3  */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_ActivateCard(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte[] pNfcid3i,        /**< [In] NFCID3 - randomly generated in case of 106kbps initial datarate or
                                                   NFCID2 in case of 212/424kbps datarate; byte[10] */
            byte bDid,              /**< [In] Device Identifier; '0' to disable usage else 1-14. */
            byte bNadEnable,        /**< [In] Enable usage of Node Address; Unequal '0' to enable. */
            byte bNad,              /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte bDsi,              /**< [In] Divisor Send (target to initiator) Integer; 0-2 */
            byte bDri,              /**< [In] Divisor Receive (initiator to target) Integer; 0-2 */
            byte bFsl,              /**< [In] Frame Length Byte; 0-3  */
            byte[] pGi,             /**< [In] Optional General Information bytes. */
            byte bGiLength,         /**< [In] Number of General Information bytes. */
            byte[] pAtrRes,         /**< [Out] Attribute Response; byte[64] */
            ref byte pAtrResLength  /**< [Out] Attribute Response Length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_ResetProtocol(
            IntPtr pDataParams  /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_Deselect(
            IntPtr pDataParams,                 /**< [In] Pointer to this layers parameter structure. */
            byte bDeselectCommand               /**< [In] Request to send, either deselect or select. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_PresCheck(
            IntPtr pDataParams     /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_Exchange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] Data to transmit. */
            ushort wTxLength,       /**< [In] Length of data to transmit. */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data. */
            ref ushort pRxLength    /**< [Out] number of received data bytes. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_SetConfig(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            ushort wConfig,         /**< [In] Configuration Identifier. */
            ushort wValue           /**< [In] Configuration Value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_GetConfig(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            ushort wConfig,         /**< [In] Configuration Identifier. */
            ref ushort wValue       /**< [Out] Configuration Value. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_GetSerialNo(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure. */
            byte[] pNfcId3Out   /**< [Out] last NFCID3; byte[10]. */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t Atr(
            byte[] pNfcid3i,    /**< [In] NFCID3 - randomly generated in case of 106kbps initial datarate or
                                                    NFCID2 in case of 212/424kbps datarate; byte[10] */
            byte bDid,          /**< [In] Device Identifier; '0' to disable usage else 1-14. */
            byte bLri,          /**< [In] Length Reduction of Initiator; 0-3  */
            byte bNadEnable,    /**< [In] Enable usage of Node Address; Unequal '0' to enable. */
            byte bNad,          /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte[] pGi,         /**< [In] Optional General Information bytes. */
            out byte[] pAtrRes  /**< [Out] Attribute Response; uint8_t[17-64] */
            )
        {
            Status_t status;
            byte bAtrResLen = 0;
            byte bGiLength = 0;
            pAtrRes = new byte[64];
            if (pGi != null)
            {
                bGiLength = (byte)pGi.Length;
            }
            status = phpalI18092mPI_Atr(m_pDataParams, pNfcid3i, bDid, bLri, bNadEnable, bNad, pGi, bGiLength, pAtrRes, ref bAtrResLen);
            Array.Resize<byte>(ref pAtrRes, bAtrResLen);

            return status;
        }

        public Status_t Psl(
            byte bDsi,          /**< [In] Divisor Send (target to initiator) Integer; 0-2 */
            byte bDri,          /**< [In] Divisor Receive (initiator to target) Integer; 0-2 */
            byte bFsl           /**< [In] Frame Length Byte; 0-3  */
            )
        {
            return phpalI18092mPI_Psl(m_pDataParams, bDsi, bDri, bFsl);
        }

        public Status_t ActivateCard(
            byte[] pNfcid3i,    /**< [In] NFCID3 - randomly generated in case of 106kbps initial datarate or
                                                   NFCID2 in case of 212/424kbps datarate; byte[10] */
            byte bDid,          /**< [In] Device Identifier; '0' to disable usage else 1-14. */
            byte bNadEnable,    /**< [In] Enable usage of Node Address; Unequal '0' to enable. */
            byte bNad,          /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
            byte bDsi,          /**< [In] Divisor Send (target to initiator) Integer; 0-2 */
            byte bDri,          /**< [In] Divisor Receive (initiator to target) Integer; 0-2 */
            byte bFsl,          /**< [In] Frame Length Byte; 0-3  */
            byte[] pGi,         /**< [In] Optional General Information bytes. */
            out byte[] pAtrRes  /**< [Out] Attribute Response; byte[17-64] */
            )
        {
            Status_t status;
            byte bAtrResLen = 0;
            byte bGiLength = 0;
            pAtrRes = new byte[64];
            if (pGi != null)
            {
                bGiLength = (byte)pGi.Length;
            }
            status = phpalI18092mPI_ActivateCard(m_pDataParams, pNfcid3i, bDid, bNadEnable, bNad, bDsi, bDri, bFsl, pGi, bGiLength, pAtrRes, ref bAtrResLen);
            Array.Resize<byte>(ref pAtrRes, bAtrResLen);

            return status;
        }

        public Status_t ResetProtocol()
        {
            return phpalI18092mPI_ResetProtocol(m_pDataParams);
        }

        public Status_t Deselect(
            DeselectCommand bDeselectCommand               /**< [In] Request to send, either deselect or select. */
            )
        {
            return phpalI18092mPI_Deselect(m_pDataParams, (byte)bDeselectCommand);
        }

        public Status_t PresCheck()
        {
            return phpalI18092mPI_PresCheck(m_pDataParams);
        }

        public Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] Data to transmit. */
            out byte[] pRxBuffer    /**< [Out] Received data. */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalI18092mPI_Exchange(
                m_pDataParams,
                (ushort)wOption,
                pTxBuffer,
                (ushort)pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength);

            if (wRxLength > 0)
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t SetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            int wValue          /**< [In] Configuration Value */
            )
        {
            return phpalI18092mPI_SetConfig(m_pDataParams, (ushort)wConfig, (ushort)wValue);
        }

        public Status_t GetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            ushort wValue = 0;
            status = phpalI18092mPI_GetConfig(m_pDataParams, (ushort)wConfig, ref wValue);
            pValue = (int)wValue;
            return status;
        }

        public Status_t GetSerialNumber(
            out byte[] pNfcId3Out /**< [Out] Received serial number */
            )
        {
            pNfcId3Out = new byte[10];
            return phpalI18092mPI_GetSerialNo(m_pDataParams, pNfcId3Out);
        }


        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palI18092mPI.Generic
    {
        #region DATA_STRUCTURE

        public const byte NFCID3_LENGTH = 0x0A;

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                          /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;               /**< Pointer to the parameter structure of the underlying layer. */
            public byte bNfcIdValid;                    /**< Whether current NfcID is valid or not. */
            public fixed byte aNfcid3i[NFCID3_LENGTH];  /**< Array holding 10 bytes long NFCID3. */
            public byte bStateNow;                      /**< Current Exchange-State. */
            public byte bDid;                           /**< Device Identifier; '0' to disable usage else 1-14. */
            public byte bNadEnabled;                    /**< Node Address Enabler; Unequal '0' if enabled. */
            public byte bNad;                           /**< Node Address; Ignored if bNadEnabled is equal '0'. */
            public byte bWt;                            /**< Waiting Time; 0-14. */
            public byte bFsl;                           /**< Frame Length Byte; 0-3. */
            public byte bPni;                           /**< Current Packet Number (Packet number information); 0-3. */
            public byte bDsi;                           /**< Divisor Send (target to initiator) Integer; 0-2. */
            public byte bDri;                           /**< Divisor Receive (initiator to target) Integer; 0-2. */
            public byte bMaxRetryCount;                 /**< Maximum retry count for retransmissions. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18092mPI_Sw_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init(Hal.Generic pHal)
        {
            return phpalI18092mPI_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHal.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalI18092mPI_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}