/*
 * Copyright 2013, 2018, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palI18000p3m3
{
    #region DEFINES

    /// <summary>
    /// Targets
    /// </summary>
    public enum Targets : byte
    {
        INVENTORIED_S0 = 0x00,
        INVENTORIED_S2 = 0x02,
        SL = 0x04
    }

    /// <summary>
    /// Memory Banks
    /// </summary>
    public enum MemBanks : byte
    {
        RESERVERD = 0x00,
        UII = 0x01,
        TID = 0x02,
        USER = 0x03
    }

    /// <summary>
    /// Link Frequencies
    /// </summary>
    public enum LF : byte
    {
        LF_423KHZ = 0x00,   /**< 423kHz (fc/32). */
        LF_847KHZ = 0x01    /**< 847kHz (fc/16). */
    }

    /// <summary>
    /// 'M' Values
    /// </summary>
    public enum M : byte
    {
        MANCHESTER_2 = 0x02,    /**< 2 subcarrier pulse Manchester. */
        MANCHESTER_4 = 0x03     /**< 4 subcarrier pulse Manchester. */
    }

    /// <summary>
    /// 'Sel' Values
    /// </summary>
    public enum Sel : byte
    {
        ALL_00 = 0x00,  /**< All. */
        ALL_01 = 0x01,  /**< All. */
        NOT_SL = 0x02,  /**< Select only tags not matching SL. */
        SL = 0x03,      /**< Select only tags matching SL. */
    }

    /// <summary>
    /// 'Session' Values
    /// </summary>
    public enum Session : byte
    {
        S0 = 0x00,  /**< Session S0. */
        S1 = 0x01,  /**< Session S1 (Optional). */
        S2 = 0x02,  /**< Session S2. */
        S3 = 0x03   /**< Session S3 (Optional). */
    }

    /// <summary>
    /// 'UpDn' Values
    /// </summary>
    public enum UpDn : byte
    {
        INCREMENT = 0x06,   /**< Increment Q. */
        NOCHANGE = 0x00,    /**< Do not change Q. */
        DECREMENT = 0x03    /**< Decrement Q. */
    }

    /// <summary>
    /// ACK Options
    /// </summary>
    public enum AckOption : byte
    {
        USE_CRC = 0x00,
        USE_HANDLE = 0x01
    }

    /// <summary>
    /// SetConfig Options
    /// </summary>
    public enum Confic : byte
    {
        TXLASTBITS = 0x0003,        /**< Set number of valid bits of last Tx-Byte. */
        TIMEOUT_VALUE_MS = 0x000E   /**< Set RC Timeout (in [ms]). */
    }
    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_Exchange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit. */
            ushort wTxLength,       /**< [In] length of input data. */
            byte bTxLastBits,       /**< [In] number of valid bits of last byte (Tx). */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data. */
            ref ushort pRxLength,   /**< [Out] number of received data bytes. */
            ref byte pRxLastBits    /**< [Out] number of valid bits of last byte (Rx). */
            );
#if DEPRECATED
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_Select(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bTarget,           /**< [In] Indicates whether the SL flag or the inventoried flags shall be modified. */
            byte bAction,           /**< [In] Indicates the way of changing the SL flag and the inventoried flags. */
            byte bMemBank,          /**< [In] Memory bank where the mask shall be applied. */
            byte[] pPointer,        /**< [In] Memory address where the mask shall be applied. */
            byte bPointerLength,    /**< [In] Length of the pointer in bytes; 1,2,3 or 4. */
            byte[] pMask,           /**< [In] Contains a bit string that a tag compares against the memory location that begins at pPointer. */
            byte bMaskBitLength,    /**< [In] Length of the mask in bits. */
            byte bTruncate          /**< [In] Specifies whether a tag loadmodulates its entire UII (#PH_OFF), or only that portion of the UII immediately following Mask (#PH_ON). */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_BeginRound(
            IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            byte bDr,           /**< [In] Divide ratio. */
            byte bM,            /**< [In] Cycles per symbol. */
            byte bTRext,        /**< [In] Indicates whether a pilot tone shall be used in the T=>R preamble (#PH_ON) or not (#PH_OFF). */
            byte bSel,          /**< [In] Indicates which tags shall take part of the inventory round. */
            byte bSession,      /**< [In] Indicates the session of the inventory round. */
            byte bRfu,          /**< [In] RFU (1 bit). */
            byte bQ,            /**< [In] Indicates the number of slots in a round (2^Q slots). */
            byte[] pStoredCRC   /**< [Out] Stored CRC of a responding tag. This buffer has to be 2 bytes long. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_ResizeRound(
            IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            byte bUpDn,         /**< [In] Indicates whether and how Q shall be ajusted. */
            byte[] pStoredCRC   /**< [Out] Stored CRC of a responding tag. This buffer has to be 2 bytes long. */
            );

         [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_GetSerialNo(
            IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            byte[] pRxBuffer,   /**< [Out] StoredCRC; byte[2]. */
            ref byte pRxLength  /**< [Out] Length of StoredCRC; 2 bytes. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_NextSlot(
            IntPtr pDataParams, /**< [In] Pointer to this layer's parameter structure. */
            byte[] pStoredCRC   /**< [Out] Stored CRC of a responding tag. This buffer has to be 2 bytes long. */
            );
#endif //DEPRECATED


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_Ack(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bOption,           /**< [In] Option parameter. */
            byte[] pHandle,         /**< [In] Handle value, mandatory if \c bOption is #PHPAL_I18000P3M3_ACK_USE_HANDLE; uint8_t[2]. */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to Tag data and, if required, PacketCRC. */
            ref ushort pRxBitLength /**< [Out] Tag response length in bits. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_Nak(
                               IntPtr pDataParams   /**< [In] Pointer to this layer's parameter structure. */
                               );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_ActivateCard(
            IntPtr pDataParams,             /**< [In] Pointer to this layer's parameter structure. */
            byte[] pSelCmd,                 /**< [In] ISO 18000p3m3 Select command frame. */
            byte bSelCmdLen,                /**< [In] Select command length in bytes- 1 to 39 bytes. */
            byte bNumValidBitsinLastByte,   /**< [In] Number of valid bits in last byte of pSelCmd. */
            byte[] pBeginRndCmd,            /**< [In] ISO 18000p3m3 BeginRound command frame. This is 17bits i.e., 3 bytes are expected. CRC5 should not be provided. */
            byte bTSprocessing,             /**< [In] TimeSlot processing behavior. */
            ref IntPtr ppRxBuffer,          /**< [In] Pointer to HAL Rx Buffer containing handle of the activated tag. */
            ref ushort wRxBufferLen,        /**< [In] Length of response in hal Rx Buffer. */
            byte[] pMoreCardsAvailable      /**< [Out] Indicates if more tags are present in the field. */

            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_CreateSelectCmd(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bTarget,           /**< [In] Indicates whether the SL flag or the inventoried flags shall be modified. */
            byte bAction,           /**< [In] Indicates the way of changing the SL flag and the inventoried flags. */
            byte bMemBank,          /**< [In] Memory bank where the mask shall be applied. */
            byte[] pPointer,        /**< [In] Memory address where the mask shall be applied. */
            byte bPointerLength,    /**< [In] Length of the pointer in bytes; 0,1,2 or 3. */
            byte[] pMask,           /**< [In] Contains a bit string that a tag compares against the memory location that begins at pPointer. */
            byte bMaskBitLength,    /**< [In] Length of the mask in bits. */
            byte bTruncate,         /**< [In] Specifies whether a tag load modulates its entire UII (#PH_OFF), or only that portion of the UII immediately following Mask (#PH_ON). */
            byte[] pSelectCmd,      /**< [Out] Select Command frame. (Max 39 bytes) */
            ref ushort wLen,            /**< [Out] Length of Select cmd bytes */
            byte[] bValidBits       /**< [Out] Number of valid bits in the last byte of pSelectCmd. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_CreateBeginRoundCmd(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            byte bDr,               /**< [In] Divide ratio. */
            byte bM,                /**< [In] Cycles per symbol. */
            byte bTRext,            /**< [In] Indicates whether a pilot tone shall be used in the T=>R pre-amble (#PH_ON) or not (#PH_OFF). */
            byte bSel,              /**< [In] Indicates which tags shall take part of the inventory round. */
            byte bSession,          /**< [In] Indicates the session of the inventory round. */
            byte bRfu,              /**< [In] RFU (1 bit). */
            byte bQ,                /**< [In] Indicates the number of slots in a round (2^Q slots). */
            byte[] pBeginRnd        /**< [Out] Begin round command frame (3 bytes) */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_SetConfig(
            IntPtr pDataParams,    /**< [In] Pointer to this layer's parameter structure. */
            ushort wConfig,      /**< [In] Configuration Identifier. */
            ushort wValue        /**< [In] Configuration Value. */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit. */
            byte bTxLastBits,       /**< [In] number of valid bits of last byte (Tx). */
            out byte[] pRxBuffer,   /**< [Out] Pointer to received data. */
            out byte pRxLastBits    /**< [Out] number of valid bits of last byte (Rx). */
            )
        {
            Status_t status;
            ushort wByteLength;
            IntPtr pRxBufferInt = IntPtr.Zero;
            pRxLastBits = 0;

            if (pTxBuffer != null)
            {
                wByteLength = (ushort)pTxBuffer.Length;
            }
            else
            {
                wByteLength = 0;
            }

            status = phpalI18000p3m3_Exchange(
                m_pDataParams,
                (ushort)wOption,
                pTxBuffer,
                wByteLength,
                bTxLastBits,
                ref pRxBufferInt,
                ref wByteLength,
                ref pRxLastBits);

            if ((pRxBufferInt != IntPtr.Zero) && (wByteLength > 0))
            {
                pRxBuffer = new byte[wByteLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wByteLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

#if DEPRECATED
        public Status_t Select(
            Targets bTarget,        /**< [In] Indicates whether the SL flag or the inventoried flags shall be modified. */
            byte bAction,           /**< [In] Indicates the way of changing the SL flag and the inventoried flags. */
            MemBanks bMemBank,      /**< [In] Memory bank where the mask shall be applied. */
            byte[] pPointer,        /**< [In] Memory address where the mask shall be applied. */
            byte[] pMask,           /**< [In] Contains a bit string that a tag compares against the memory location that begins at pPointer. */
            byte bMaskBitLength,    /**< [In] Length of the mask in bits. */
            byte bTruncate          /**< [In] Specifies whether a tag loadmodulates its entire UII (#PH_OFF), or only that portion of the UII immediately following Mask (#PH_ON). */
            )
        {
            byte bPointerLength;
            if ((pPointer != null) && (pPointer.Length != 0))
            {
                bPointerLength = (byte)(pPointer.Length - 1);
            }
            else
            {
                bPointerLength = 0;
            }
            return phpalI18000p3m3_Select(
                m_pDataParams,
                (byte)bTarget,
                bAction,
                (byte)bMemBank,
                pPointer,
                bPointerLength,
                pMask,
                bMaskBitLength,
                bTruncate
                );
        }

        public Status_t BeginRound(
            LF bDr,                 /**< [In] Divide ratio. */
            M bM,                   /**< [In] Cycles per symbol. */
            byte bTRext,            /**< [In] Indicates whether a pilot tone shall be used in the T=>R preamble (#PH_ON) or not (#PH_OFF). */
            Sel bSel,              /**< [In] Indicates which tags shall take part of the inventory round. */
            Session bSession,       /**< [In] Indicates the session of the inventory round. */
            byte bRfu,              /**< [In] RFU (1 bit). */
            byte bQ,                /**< [In] Indicates the number of slots in a round (2^Q slots). */
            out byte[] pStoredCRC   /**< [Out] Stored CRC of a responding tag. This buffer has to be 2 bytes long. */
            )
        {
            Status_t status;
            pStoredCRC = new byte[2];

            status = phpalI18000p3m3_BeginRound(
                m_pDataParams,
                (byte)bDr,
                (byte)bM,
                bTRext,
                (byte)bSel,
                (byte)bSession,
                bRfu,
                bQ,
                pStoredCRC);

            if (!status.Equals(new Status_t()))
            {
                pStoredCRC = null;
            }

            return status;
        }

        public Status_t ResizeRound(
            UpDn bUpDn,             /**< [In] Indicates whether and how Q shall be ajusted. */
            out byte[] pStoredCRC   /**< [Out] Stored CRC of a responding tag. This buffer has to be 2 bytes long. */
            )
        {
            Status_t status;
            pStoredCRC = new byte[2];

            status = phpalI18000p3m3_ResizeRound(
                m_pDataParams,
                (byte)bUpDn,
                pStoredCRC);

            if (!status.Equals(new Status_t()))
            {
                pStoredCRC = null;
            }

            return status;
        }

        /// <summary>
        /// Perform the GetSerialNo command.
        /// </summary>
        /// <returns></returns>
        public Status_t GetSerialNo(
            out byte[] pRxBuffer    /**< [Out] StoredCRC; uint8_t[2]. */
            )
        {
            Status_t status;
            pRxBuffer = new byte[2];
            byte pRxLength = 0;

            status = phpalI18000p3m3_GetSerialNo(m_pDataParams, pRxBuffer, ref pRxLength);

            if (!status.Equals(new Status_t()))
            {
                pRxBuffer = null;
            }

            return status;
        }

         public Status_t NextSlot(
            out byte[] pStoredCRC   /**< [Out] Stored CRC of a responding tag. This buffer has to be 2 bytes long. */
            )
        {
            Status_t status;
            pStoredCRC = new byte[2];

            status = phpalI18000p3m3_NextSlot(
                m_pDataParams,
                pStoredCRC);

            if (!status.Equals(new Status_t()))
            {
                pStoredCRC = null;
            }

            return status;
        }



#endif //DEPRECATED

        public Status_t Ack(
            AckOption bOption,      /**< [In] Option parameter. */
            byte[] pHandle,         /**< [In] Handle value, can be NULL if bSendHandle is #PH_OFF; byte[2]. */
            out byte[] pRxBuffer,   /**< [Out] Pointer to received data */
            out int pRxBitLength    /**< [Out] number of received data bits. */
            )
        {
            Status_t status;
            ushort wRxBitLength = 0;
            int wRxByteLength;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalI18000p3m3_Ack(m_pDataParams, (byte)bOption, pHandle, ref pRxBufferInt, ref wRxBitLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxBitLength > 0))
            {
                pRxBitLength = (int)wRxBitLength;
                wRxByteLength = pRxBitLength >> 3;
                if ((pRxBitLength & 0x07) != 0)
                {
                    ++wRxByteLength;
                }
                pRxBuffer = new byte[wRxByteLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxByteLength);
            }
            else
            {
                pRxBitLength = 0;
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Nak()
        {
            return phpalI18000p3m3_Nak(m_pDataParams);
        }

        public Status_t ActivateCard(
            byte[] pSelCmd,                 /**< [In] ISO 18000p3m3 Select command frame. */
            byte bSelCmdLen,                /**< [In] Select command length in bytes- 1 to 39 bytes. */
            byte bNumValidBitsinLastByte,   /**< [In] Number of valid bits in last byte of pSelCmd. */
            byte[] pBeginRndCmd,            /**< [In] ISO 18000p3m3 BeginRound command frame. This is 17bits i.e., 3 bytes are expected. CRC5 should not be provided. */
            byte bTSprocessing,             /**< [In] TimeSlot processing behavior. */
            out byte[] ppRxBuffer,          /**< [In] Pointer to HAL Rx Buffer containing handle of the activated tag. */
            out int pRxBufferLen,           /**< [In] Length of response in hal Rx Buffer. */
            out byte[] pMoreCardsAvailable      /**< [Out] Indicates if more tags are present in the field. */
            )
        {
            Status_t status;
            ushort wRxBitLength = 0;
            int wRxByteLength;
            IntPtr pRxBufferInt = IntPtr.Zero;
            pMoreCardsAvailable = new byte[2];

            status = phpalI18000p3m3_ActivateCard(
                m_pDataParams,
                pSelCmd,
                bSelCmdLen,
                bNumValidBitsinLastByte,
                pBeginRndCmd,
                bTSprocessing,
                ref pRxBufferInt,
                ref wRxBitLength,
                pMoreCardsAvailable
                );

            if ((pRxBufferInt != IntPtr.Zero) && (wRxBitLength > 0))
            {
                pRxBufferLen = (int)wRxBitLength;
                wRxByteLength = pRxBufferLen >> 3;
                if ((pRxBufferLen & 0x07) != 0)
                {
                    ++wRxByteLength;
                }
                ppRxBuffer = new byte[wRxBitLength];
                Marshal.Copy(pRxBufferInt, ppRxBuffer, 0, wRxBitLength);
            }
            else
            {
                pRxBufferLen = 0;
                ppRxBuffer = null;
            }

            return status;
        }


        public Status_t CreateSelectCmd(
            byte bTarget,           /**< [In] Indicates whether the SL flag or the inventoried flags shall be modified. */
            byte bAction,           /**< [In] Indicates the way of changing the SL flag and the inventoried flags. */
            byte bMemBank,          /**< [In] Memory bank where the mask shall be applied. */
            byte[] pPointer,        /**< [In] Memory address where the mask shall be applied. */
            byte[] pMask,           /**< [In] Contains a bit string that a tag compares against the memory location that begins at pPointer. */
            byte bMaskBitLength,    /**< [In] Length of the mask in bits. */
            byte bTruncate,         /**< [In] Specifies whether a tag load modulates its entire UII (#PH_OFF), or only that portion of the UII immediately following Mask (#PH_ON). */
            out byte[] pSelectCmd,      /**< [Out] Select Command frame. (Max 39 bytes) */
            out ushort pLen,        /**< [Out] Length of Select cmd bytes */
            out byte[] bValidBits       /**< [Out] Number of valid bits in the last byte of pSelectCmd. */
            )
        {
            Status_t status;
            pSelectCmd = new byte[39];
            bValidBits = new byte[2];
            pLen = 0;

            byte bPointerLength;
            if ((pPointer != null) && (pPointer.Length != 0))
            {
                bPointerLength = (byte)(pPointer.Length - 1);
            }
            else
            {
                bPointerLength = 0;
            }

            status = phpalI18000p3m3_CreateSelectCmd(m_pDataParams,
                                                        bTarget,
                                                        bAction,
                                                        bMemBank,
                                                        pPointer,
                                                        bPointerLength,
                                                        pMask,
                                                        bMaskBitLength,
                                                        bTruncate,
                                                        pSelectCmd,
                                                        ref pLen,
                                                        bValidBits
                                                        );
            return status;
        }

        public Status_t CreateBeginRoundCmd(
            byte bDr,               /**< [In] Divide ratio. */
            byte bM,                /**< [In] Cycles per symbol. */
            byte bTRext,            /**< [In] Indicates whether a pilot tone shall be used in the T=>R pre-amble (#PH_ON) or not (#PH_OFF). */
            byte bSel,              /**< [In] Indicates which tags shall take part of the inventory round. */
            byte bSession,          /**< [In] Indicates the session of the inventory round. */
            byte bRfu,              /**< [In] RFU (1 bit). */
            byte bQ,                /**< [In] Indicates the number of slots in a round (2^Q slots). */
            out byte[] pBeginRnd        /**< [Out] Begin round command frame (3 bytes) */
            )
        {
            Status_t status;
            pBeginRnd = new byte[3];

            //pRxBuffer = new byte[2];
            //byte pRxLength = 0;

            status = phpalI18000p3m3_CreateBeginRoundCmd(m_pDataParams,
                                                     bDr,
                                                     bM,
                                                     bTRext,
                                                     bSel,
                                                     bSession,
                                                     bRfu,
                                                     bQ,
                                                     pBeginRnd);

            if (!status.Equals(new Status_t()))
            {
                pBeginRnd = null;
            }

            return status;
        }

        public Status_t SetConfig(
            ushort wConfig,
            ushort wValue
            )
        {
            Status_t status;
            status = phpalI18000p3m3_SetConfig(m_pDataParams, wConfig, wValue);
            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SW

    public class Sw : palI18000p3m3.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                  /**< Layer ID for this HAL layer, NEVER MODIFY! */
            public IntPtr pHalDataParams;       /**< Pointer to the parameter structure of the underlying layer */
            public byte bSession;               /**< Active session of invertory round. */
            public fixed byte abStoredCRC[2];   /**< Stored CRC returned by the tag. */
            public byte bStoredCRCValid;        /**< Whether StoredCRC is valid or not. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalI18000p3m3_Sw_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
            );

        #endregion

        #region INIT

        public Status_t Init(Hal.Generic pHal)
        {
            return phpalI18000p3m3_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHal.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalI18000p3m3_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
