/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.palFelica
{
    #region DEFINES

    public enum Slots : byte
    {
        NUMSLOTS_1 = 0x00,  /**< Value for \c bNumSlots = 1. */
        NUMSLOTS_2 = 0x01,  /**< Value for \c bNumSlots = 2. */
        NUMSLOTS_4 = 0x03,  /**< Value for \c bNumSlots = 4. */
        NUMSLOTS_8 = 0x07,  /**< Value for \c bNumSlots = 8. */
        NUMSLOTS_16 = 0x0F  /**< Value for \c bNumSlots = 16. */
    }

    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalFelica_ReqC(
            IntPtr pDataParams,
            byte[] pSystemCode,
            byte bTimeSlot,
            byte[] pRxBuffer);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalFelica_ActivateCard(
            IntPtr pDataParams,
            byte[] pIDmPMm,
            byte bIDmPMmLen,
            byte[] pSystemCode,
            byte bTimeSlot,
            byte[] pRxBuffer,
            ref byte pRxLength,
            ref byte pMoreCardsAvailable);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalFelica_Exchange(
            IntPtr pDataParams,
            ushort wOption,
            ushort wN,
            byte[] pTxBuffer,
            ushort wTxLength,
            ref IntPtr ppRxBuffer,
            ref ushort pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalFelica_GetSerialNo(
            IntPtr pDataParams,
            byte[] pIDmPMmOut,
            ref byte pLenIDmPMmOut);

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t ReqC(
            byte[] pSystemCode,     /**< [In] System-Code; uint8_t[2]. */
            Slots bNumTimeSlots,    /**< [In] Number of timeslots to use. Refer to e.g. #PHPAL_FELICA_NUMSLOTS_1 for valid values. */
            out byte[] pRxBuffer    /**< [Out] Length of received data. 0 or 16. */
            )
        {
            pRxBuffer = new byte[16];
            return phpalFelica_ReqC(m_pDataParams, pSystemCode, (byte)bNumTimeSlots, pRxBuffer);
        }

        public Status_t ActivateCard(
            byte[] pIDmPMm,                 /**< [In] IDm and PMm; If supplied it is stored and no real activation is done. */
            byte[] pSystemCode,             /**< [In] System-Code; uint8_t[2]. */
            Slots bNumTimeSlots,            /**< [In] Number of timeslots to use. Refer to e.g. #PHPAL_FELICA_NUMSLOTS_1 for valid values. */
            out byte[] pRxBuffer,           /**< [Out] 8 bytes NFCID2 + 8 bytes PAD; uint8_t[16]. */
            out byte pMoreCardsAvailable    /**< [Out] Whether there are more cards in the field or not; uint8_t. */
            )
        {
            Status_t status;
            byte pRxLength = 0;

            pRxBuffer = new byte[16];
            pMoreCardsAvailable = 0;

            status = phpalFelica_ActivateCard(m_pDataParams, pIDmPMm, (byte)((pIDmPMm == null) ? 0 : pIDmPMm.Length), pSystemCode, (byte)bNumTimeSlots, pRxBuffer, ref pRxLength, ref pMoreCardsAvailable);
            Array.Resize<byte>(ref pRxBuffer, pRxLength);
            return status;
        }

        public Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            int wN,                 /**< [In] N to calulate the response timeout. */
            byte[] pTxBuffer,       /**< [In] Data to transmit; Length and IDm is added automatically. */
            out byte[] pRxBuffer    /**< [Out] Received data; Length, response code and IDm are removed automatically. */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phpalFelica_Exchange(
                m_pDataParams,
                (ushort)wOption,
                (ushort)wN,
                pTxBuffer,
                (ushort)pTxBuffer.Length,
                ref pRxBufferInt,
                ref wRxLength);

            if (wRxLength > 0)
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t GetSerialNumber(
            out byte[] pIDmPMm  /**< [Out] Last Uid; uint8_t[8]. */
            )
        {
            Status_t status;
            byte bLenIDmPMm = 0;

            pIDmPMm = new byte[16];
            status = phpalFelica_GetSerialNo(m_pDataParams, pIDmPMm, ref bLenIDmPMm);
            Array.Resize<byte>(ref pIDmPMm, bLenIDmPMm);

            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }
    #endregion

    #region SW

    public class Sw : palFelica.Generic
    {
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL layer, NEVER MODIFY! */
            public IntPtr pHalDataParams;   /**< Pointer to the parameter structure of the underlying layer     */
            public fixed byte pIDm[8];             /**< The Cards Manufacture ID (IDm) of 8 bytes. */
            public fixed byte pPMm[8];             /**< The Cards Manufacture Parameters (PMm) of 8 bytes. */
            public byte bIDmPMmValid;     /**< Whether the stored IDm and PMm are valid (\c 1) or not (\c 0). */
            public byte bLength;          /**< Length byte, used internally for Exchange function */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phpalFelica_Sw_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,
            IntPtr pHalDataParams           /**< [In] Pointer to the parameter structure of the underlying layer */
            );

        #endregion

        #region INIT

        public Status_t Init(Hal.Generic pHal)
        {
            return phpalFelica_Sw_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), pHal.m_pDataParams);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Hal.Generic pHal)
        {
            return phpalFelica_Sw_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, pHal.m_pDataParams);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
    #endregion
}
