/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Text;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.HalContact
{
    #region RdScr01

    /// <summary>
    /// Transparent specific HAL-Contact-Component of Basic Function Library Framework.
    /// </summary>
    public class RdScr01 : HalContact.Generic
    {
        public const int DEFAULT_TIMEOUT_MS = 5;            /**< Default timeout in microseconds. */
        private const int SHADOW_COUNT = 0x0068;            /**< Number of shadowed configurations. */
        private const int RESERVED_BUFFER_LEN = 32;         /**< Amount of needed and reserved memory for the protocol overhead. */
        private const int RESERVED_BUFFER_CRC_LEN = 2;      /**< Amount of needed and reserved memory for the crc overhead. */

        public enum Config : int
        {
            BITRATE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,  /**< Configure communication speed. */
            STOPBITS,                                        /**< Configure number of stop bits. */
            DTR,                                             /**< Configure dtr line. */
            RTS,                                             /**< Configure rts line. */
            PARITY,                                          /**< Configure parity. */
            FI,                                              /**< Configure FI. */
            DI,                                              /**< Configure DI. */
        }

        public enum DTR : int
        {
            DTR_LOW = 0,    /**< dtr low. */
            DTR_HIGH        /**< dtr high. */
        }

        public enum RTS : int
        {
            RTS_LOW = 0,    /**< rts low. */
            RTS_HIGH        /**< rts high. */
        }

        public enum Parity : int
        {
            NONE = 0,
            ODD,
            EVEN
        }

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                  /**< Layer ID for this component, NEVER MODIFY! */
            public IntPtr pBalDataParams;                       /**< pointer to the lower layers parameter structure */
            public IntPtr pTxBuffer;                            /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                           /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                            /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                            /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                           /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                            /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                       /**< Starting position within the global receive buffer. */
            public uint dwNumExpectedBytes;                     /**< Number of expected bytes for Exchanges. */
            public uint dwATRLength;                            /**< Length of the ATR stored internally in data params. */
            public fixed byte pAtr[MAX_ATR_SIZE];               /**< ATR of current card. */
            public IntPtr pIntBuffer;                           /**< Buffer used for exchange with max size Data length */
            public uint dwIntBufferLen;                         /**< Number of valid bytes within the internal temp buffer. */
            public uint dwIntBufferSize;                        /**< Size of the internal buffer. */
            public byte bProtocolType;                          /**< Type of protocol for which the hal is configured for */
            public uint dwI2CRxPollingTimeoutMs;                /**< Timeout Value in ms to be used by I2C exchange function. */
            public ushort wTimingMode;                          /**< Timing mode used to measure FDT */
            public ulong qwTiming_ns;                           /**< Current timing value. */
            public fixed uint wCfgShadow[SHADOW_COUNT];         /**< Configuration shadow; Stores configuration for current cardtype. */
            public byte bCommunicationChannel;                  /**< Current Communication Channel - Standard or I2C. */
            public ushort wSlaveAddress;                        /**< Address of the Slave for I2C Communication. */
            public uint dwBitrate;                              /**< Bitrate used for I2C Communiation. */
            public byte bClkStretch;                            /**< Flag indicating clock streching state. */
            public byte bHsMode;                                /**< Flag indicating the hs mode state. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_RdScr01_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize,              /**< [In] Size of the global receive buffer. */
            IntPtr pInternalBuffer,         /**< [In] Internal buffer used as a scratch buffer */
            uint wInternalBufferLen         /**< [In] Length of innternal buffer used as a scratch buffer */
            );

        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;
        private byte[] m_bIntBuffer;
        private GCHandle m_pIntBuffer;
        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.Generic pBal, int wTxBufferSize, int wRxBufferSize)
        {
            int wIntBufferSize = 0;
            // Adjust IntBuffer length
            if ((2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wIntBufferSize = 0xFFFF + 1;
            }
            else
            {
                wIntBufferSize = 2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_CRC_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_CRC_LEN;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_bIntBuffer = new byte[wIntBufferSize];

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);
            m_pIntBuffer = GCHandle.Alloc(m_bIntBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHwContact_RdScr01_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize,
                m_pIntBuffer.AddrOfPinnedObject(),
                (uint)wIntBufferSize);
        }
        #endregion

        #region CUSTOM_FUNCTIONS

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public RdScr01()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~RdScr01()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

	#endregion
}
#endif
