/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.Text;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.HalContact
{
    #region DEFINES

    public enum Config : int
    {
        REDUNDANCY_MODE = 1,
        CONVENTION,
        FD,
        PROTOCOLTYPE,

        BGT_CLK,
        BWT_CLK,
        CGT_CLK,
        CWT_CLK,
        IFSC,

        WT_CLK,
        GT_CLK,
        EGT_CLK,
        RGT_CLK,
        RGT_1_CLK,
        RGT_2_CLK,

        MAX_CLOCK_ATR,
        PROTOCOL_TIMEOUT,

        ETU_WIDTH_CLK,
        ETU_WIDTH_ATR_CLK,
        INITIAL_WAITING_TIME_CLK,
        PARITY,
        PARITY_CHECK,
        PARITY_ERROR_SIGNAL,
        POWER_OFF_ON_ATR_ERROR,
        RESET_LOW_TIME,
        TIMEOUT_NULL_BYTE,
        NUM_EXPECTED_BYTES,
        COMMUNICATION_CHANNEL,
        POLLING_INTERVAL_MS,

        VCC = 0x0020,
        CLOCK_FREQUENCY_HZ,

        I2C_SLAVE_ADDRESS = 0x002A,
        I2C_BITRATE_KBITSEC,
        I2C_CLK_STRETCH,
        I2C_HS_MODE,

        SPI_BITRATE_BITSEC = 0x002E,
        SPI_RECV_NAD_VALUE = 0x002F,
        SPI_GP_MODE        = 0x0024,

        PIN_VCC_FALL_TIME = 0x0030,
        PIN_VCC_RISE_TIME,
        PIN_RESET_FALL_TIME = 0x0040,
        PIN_RESET_RISE_TIME,
        PIN_RESET_OUTPUT_LOW,
        PIN_RESET_OUTPUT_HIGH,
        PIN_CLOCK_FALL_TIME = 0x0050,
        PIN_CLOCK_RISE_TIME,
        PIN_CLOCK_OUTPUT_LOW,
        PIN_CLOCK_OUTPUT_HIGH,
        PIN_CLOCK_DUTY_CYCLE,
        PIN_IO_FALL_TIME = 0x0060,
        PIN_IO_RISE_TIME,
        PIN_IO_OUTPUT_LOW,
        PIN_IO_OUTPUT_HIGH,
        PIN_IO_INPUT_LOW = 0x0065,
        PIN_IO_INPUT_HIGH,
        PIN_IO_PULL_UP,
        TIMING_MODE = 0x0070,
        TIMING_US,
        TIMING_MS,
        RXBUFFER_STARTPOS = 0x0080,
        RXBUFFER_BUFSIZE,
        TXBUFFER_BUFSIZE,
        TXBUFFER_LENGTH,
        TXBUFFER,
        MAX_PRECACHED_BYTES,
        ADDITIONAL_INFO = 0x0090,
        RESET_ON_TIMEOUT ,
        I2C_RX_POLLING_TIMEOUT_MS
    }

    public enum Pins : int
    {
        C1_VCC = 0x0000,
        C2_RESET,
        C3_CLOCK,
        C7_IO = 0x0006,
    }

    public enum TimingMode : int
    {
        OFF = 0x0000,
        FDT = 0x0001,
        FDT_AUTOCLEAR = 0x0101,
        COMM = 0x0002,
        COMM_AUTOCLEAR = 0x0102
    }

    public enum Protocol : int
    {
        Current,
        T0,
        T1
    }

    public enum DatarateDiValues : int
    {
        Di1,
        Di2,
        Di4,
        Di8,
        Di16,
        Di32,
        Di64,
        Di12,
        Di20
    }

    public enum DatarateFiValues : int
    {
        Fi4MHZ,
        Fi5MHZ
    }

    public enum VoltageClasses : int
    {
        A = 5000,
        B = 3000,
        C = 1800,
        AMin = 4500,
        BMin = 2700,
        CMin = 1620,
        AMax = 5500,
        BMax = 3300,
        CMax = 1980
    }

    public enum ClockStopModes : uint
    {
        High = 12,
        Low = 13,
        Resume = 14,
        Applied = 11,
        AutoOff = 0x80000000
    }

    public enum TimeUnits : int
    {
        Microseconds,
        Milliseconds,
        Etu,
        ClockCycles
    }

    public enum TimeModeSettings : int
    {
        Off,
        Fdt,
        Comm,
        OptionMask = 0xFF00,
        OptionDefault = 0,
        OptionAutoclear = 0x0100
    }

    public enum RedundancyCodeMode : int
    {
        DISABLED            = 0x0000,   /**< Both CRC and LRC disabled */
        TX_LRC              = 0x0001,   /**< LRC for TX */
        RX_LRC              = 0x0002,   /**< LRC for RX */
        LRC                 = 0x0003,   /**< LRC for TX and RX */
        TX_CRC              = 0x0004,   /**< CRC for TX */
        RX_CRC              = 0x0008,   /**< CRC for RX */
        CRC                 = 0x000C,   /**< CRC for TX and RX */
        TX_CRC_SWAPPED      = 0x0014,   /**< CRC for TX */
        RX_CRC_SWAPPED      = 0x0028,   /**< CRC for RX */
        CRC_SWAPPED         = 0x003C,   /**< CRC for TX and RX */
        MASK_TX             = 0x0005,   /**< Mask for Tx Settings. */
        MASK_RX             = 0x000A,   /**< Mask for Rx Settings. */
        MASK_TX_SWAPPED     = 0x0010,   /**< Mask for Tx Swapped Setting. */
        MASK_RX_SWAPPED     = 0x0020,   /**< Mask for Rx Swapped Setting. */
        FULL_RX_BUFFER      = 0x8000    /**< Flag that indicate if the Rx check should be performed on the whole rx buffer. If not set it starts from RxStartPos. */
    }

    public enum CommuniationChannel : int
    {
        STANDARD = 0,
        I2C,
        SPI,
        SPI_EXCHANGE
    }
    #endregion

    #region INTERFACE

    /// <summary>
    /// Interface definition of interfaces a HAL object needs to implement.
    /// </summary>
    public interface IHalContact
    {
        /// <summary>
        /// Perform Data Exchange with Picc.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <param name="wRxBufSize"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            );

        /// <summary>
        /// Apply protocol settings.
        /// </summary>
        /// <param name="bCardType"></param>
        /// <returns></returns>
        Status_t ApplyProtocolSettings(
            Protocol bProtocolType,  /**< [In] Type of protocol for which the hal should be configured for */
            byte[] Atr = null        /**< [In] Atr with Settings which should be applies (if no Atr available - default values) */
            );

        /// <summary>
        /// SetConfig
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="dwValue"></param>
        /// <returns></returns>
        Status_t SetConfig32(
            Config wConfig,      /**< [In] Configuration Identifier */
            uint dwValue         /**< [In] Configuration Value */
            );

        /// <summary>
        /// GetConfig
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="pValue"></param>
        /// <returns></returns>
        Status_t GetConfig32(
            Config wConfig,     /**< [In] Configuration Identifier */
            out uint pValue     /**< [Out] Configuration Value */
            );

        /// <summary>
        /// Blocks until the time given by 	<see cref="wTimeout"/> elapsed.
        /// </summary>
        /// <param name="bUnit"></param>
        /// <param name="wTimeout"></param>
        /// <returns></returns>
        Status_t Wait(
            byte bUnit,     /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS) */
            int wTimeout    /**< [In] Timeout value */
            );

        /// <summary>
        /// Activate
        /// </summary>
        /// <param name="pDataParams"></param>
        /// <param name="pATRBuffer"></param>
        /// <param name="wATRSize"></param>
        /// <param name="pwATRLength"></param>
        /// <returns></returns>
        Status_t Activate(
            out byte[] pATRBuffer     /**< [Out] Data to the received ATR. */
            );

        /// <summary>
        /// Deactivate
        /// </summary>
        /// <param name="pDataParams"></param>
        /// <returns></returns>
        Status_t Deactivate();

        /// <summary>
        /// Clock Stop
        /// </summary>
        /// <param name="pDataParams"></param>
        /// <param name="wTgClockCount"></param>
        /// <param name="wThClockCount"></param>
        /// <param name="dwPinState"></param>
        /// <returns></returns>
        Status_t ClockStop(
                ushort wTgClockCount,     /**< [In] Number of clock cycles while the clock must be kept running after the I/O line remains at Z state before stopping the clock (default is 1,860). Maximum value is 65,535 clock cycles. */
                ushort wThClockCount,     /**< [In] Number of clock cycles before the exchange on the I/O line after a clock resume. */
                uint dwPinState         /**< [In] The desired state of the clock. If CLOCK_STOP_HIGH or CLOCK_CTOP_LOW is ored with CLOCK_AUTO_OFF (0x80000000), the clock will be automatically switched off after each APDU. */
                );

        /// <summary>
        /// Resets Reader to Default State and opens all needed Resources.
        /// </summary>
        /// <returns></returns>
        Status_t InitContactReader();
    }

    #endregion

    #region BASE

    /// <summary>
    /// Abstract Base-Class for every HAL object.
    /// (Uses only the Generic HAL C-Object)
    /// </summary>
    public abstract class Generic : HalContact.IHalContact
    {
        #region DEFINES
        public const ushort MAX_ATR_SIZE = 33;
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Exchange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_ApplyProtocolSettings(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            byte[] pAtr,        /**< [In] Atr Buffer with Settings to apply (if not Atr given default is applied). */
            ushort wAtrLength,  /**< [In] Length of the given Atr Buffer. */
            byte bProtocolType  /**< [In] ProtocolType for which the hal should be configured for */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_SetConfig32(
                                          IntPtr pDataParams,   /**< [In] Pointer to this layers parameter structure */
                                          ushort wConfig,       /**< [In] Configuration Identifier */
                                          uint wValue           /**< [In] Configuration Value */
                                          );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_GetConfig32(
                                          IntPtr pDataParams,   /**< [In] Pointer to this layers parameter structure */
                                          ushort wConfig,       /**< [In] Configuration Identifier */
                                          ref uint pValue       /**< [Out] Configuration Value */
                                          );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Wait(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bUnit,             /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS) */
            ushort wTimeout         /**< [In] Timeout value */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Activate(
            IntPtr pDataParams,      /**< [In] Pointer to this layers parameter structure */
            byte[] pATRBuffer,       /**< [Out] Data to the received ATR. */
		    ref ushort pwATRLength   /**< [Out] Length of the reiceived ATR. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_Deactivate(
            IntPtr pDataParams     /**< [In] Pointer to this layers parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_ClockStop(
            IntPtr pDataParams,     /**< [In] Pointer to this layer's parameter structure. */
            ushort wTgClockCount,   /**< [In] Number of clock cycles while the clock must be kept running after the I/O line remains at Z state before stopping the clock (default is 1,860). Maximum value is 65,535 clock cycles. */
            ushort wThClockCount,   /**< [In] Number of clock cycles before the exchange on the I/O line after a clock resume. */
            uint dwPinState         /**< [In] The desired state of the clock. If CLOCK_STOP_HIGH or CLOCK_CTOP_LOW is ored with CLOCK_AUTO_OFF (0x80000000), the clock will be automatically switched off after each APDU. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHwContact_InitContactReader(
            IntPtr pDataParams      /**< [In] Pointer to this layers parameter structure */
            );
        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        /// <summary>
        /// Perform Data Exchange with Picc.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <param name="wRxBufSize"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        public Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phhalHwContact_Exchange(m_pDataParams, (ushort)wOption, pTxBuffer, (ushort)pTxBuffer.Length, ref pRxBufferInt, ref wRxLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        /// <summary>
        /// Apply protocol settings.
        /// </summary>
        /// <param name="bCardType"></param>
        /// <returns></returns>
        public Status_t ApplyProtocolSettings(
            Protocol bProtocol,  /**< [In] Type of card for which the hal should be configured for */
            byte[] Atr = null    /**< [In] Atr Buffer with Settings to apply (if not Atr given default is applied). */

            )
        {
            return phhalHwContact_ApplyProtocolSettings(m_pDataParams, Atr, ((Atr == null) ? (ushort)0 : (ushort)Atr.Length), (byte)bProtocol);
        }

        /// <summary>
        /// SetConfig32
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="wValue"></param>
        /// <returns></returns>
        public Status_t SetConfig32(
            Config wConfig,     /**< [In] Configuration Identifier */
            uint wValue         /**< [In] Configuration Value */
            )
        {
            return phhalHwContact_SetConfig32(m_pDataParams, (ushort)wConfig, wValue);
        }

        /// <summary>
        /// GetConfig32
        /// </summary>
        /// <param name="wConfig"></param>
        /// <param name="pValue"></param>
        /// <returns></returns>
        public Status_t GetConfig32(
            Config wConfig,     /**< [In] Configuration Identifier */
            out uint pValue     /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            uint wValue = 0;
            status = phhalHwContact_GetConfig32(m_pDataParams, (ushort)wConfig, ref wValue);
            pValue = wValue;
            return status;
        }

        /// <summary>
        /// Set & Get configuration setting
        /// </summary>
        /// <param name="wConfig"></param>
        /// <returns></returns>
        public uint this[Config wConfig]
        {
            get
            {
                uint wValue;
                Status_t status = GetConfig32(wConfig, out wValue);
                if ((Error_Gen)status.Error != Error_Gen.SUCCESS)
                {
                    throw new Exception(status.ToString());
                }
                return wValue;
            }
            set
            {
                Status_t status = SetConfig32(wConfig, value);
                if ((Error_Gen)status.Error != Error_Gen.SUCCESS)
                {
                    throw new Exception(status.ToString());
                }
            }
        }

        /// <summary>
        /// Blocks until the time given by 	<paramref name="wTimeout"/> elapsed.
        /// </summary>
        /// <param name="bUnit"></param>
        /// <param name="wTimeout"></param>
        /// <returns></returns>
        public Status_t Wait(
            byte bUnit,     /**< [In] Unit of given timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS) */
            int wTimeout    /**< [In] Timeout value */
            )
        {
            return phhalHwContact_Wait(m_pDataParams, bUnit, (ushort)wTimeout);
        }

        /// <summary>
        /// Activate
        /// </summary>
        /// <param name="pATRBuffer"></param>
        /// <param name="wATRSize"></param>
        /// <param name="pwATRLength"></param>
        /// <returns></returns>
        public Status_t Activate(
            out byte[] pATRBuffer   /**< [Out] Data to the received ATR. */
            )
        {
            Status_t status;
            pATRBuffer = new byte[MAX_ATR_SIZE];
            ushort pAtrLengthInt = 0;

            status = phhalHwContact_Activate(m_pDataParams, pATRBuffer, ref pAtrLengthInt);

            if (pAtrLengthInt > 0)
                Array.Resize<byte>(ref pATRBuffer, pAtrLengthInt);
            else
                pATRBuffer = null;

            return status;
        }

        /// <summary>
        /// Deactivate
        /// </summary>
        /// <param name="bUnit"></param>
        /// <param name="wTimeout"></param>
        /// <returns></returns>
        public Status_t Deactivate()
        {
            return phhalHwContact_Deactivate(m_pDataParams);
        }

        /// <summary>
        /// ClockStop
        /// </summary>
        /// <param name="wTgClockCount"></param>
        /// <param name="wThClockCount"></param>
        /// <param name="dwPinState"></param>
        /// <returns></returns>
        public Status_t ClockStop(
            ushort wTgClockCount, /**< [In] Number of clock cycles while the clock must be kept running after the I/O line remains at Z state before stopping the clock (default is 1,860). Maximum value is 65,535 clock cycles. */
            ushort wThClockCount, /**< [In] Number of clock cycles before the exchange on the I/O line after a clock resume. */
            uint dwPinState       /**< [In] The desired state of the clock. If CLOCK_STOP_HIGH or CLOCK_CTOP_LOW is ored with CLOCK_AUTO_OFF (0x80000000), the clock will be automatically switched off after each APDU. */
            )
        {
            return phhalHwContact_ClockStop(m_pDataParams, wTgClockCount, wThClockCount, dwPinState);
        }

        /// <summary>
        /// Resets Reader to Default State and opens all needed Resources.
        /// </summary>
        /// <returns></returns>
        public Status_t InitContactReader()
        {
            return phhalHwContact_InitContactReader(m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

	#endregion
}
#endif
