/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// SamAV2 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class SamAV2 : Hal.Generic
    {
        #region DEFINES

        public const int DEFAULT_TIMEOUT = 150;     /**< Default timeout in microseconds    */
        public const int SHADOW_COUNT = 0x000F;     /**< Number of shadowed configurations. */

        public const int RESERVED_TX_BUFFER_LEN = 6;    /**< Amount of needed and reserved memory for the protocol overhead. */
        public const int RESERVED_RX_BUFFER_LEN = 2;    /**< Amount of needed and reserved memory for the protocol overhead. */

        public enum Error : byte
        {
			/// <summary>
			/// EEProm failure.
			/// Custom error code for RESP.ISO6400
			/// </summary>
			HW_EEPROM = ( CustomCodes.ERROR_BEGIN + 0 ),

			/// <summary>
			/// RC failure.
			/// Custom error code for RESP.ISO6401
			/// </summary>
			HW_RC5XX = ( CustomCodes.ERROR_BEGIN + 1 ),

			/// <summary>
			/// Key creation failure.
			/// Custom error code for RESP.ISO6501
			/// </summary>
			KEY_CREATE_FAILED = ( CustomCodes.ERROR_BEGIN + 2 ),

			/// <summary>
			/// SAM key reference error.
			/// Custom error code for RESP.ISO6502
			/// </summary>
			KEY_REF_NO_INVALID = ( CustomCodes.ERROR_BEGIN + 3 ),

			/// <summary>
			/// SAM key usage counter error.
			/// Custom error code for RESP.SO6503
			/// </summary>
			KEY_KUC_NO_INVALID = ( CustomCodes.ERROR_BEGIN + 4 ),

			/// <summary>
			/// Memory failure.
			/// Custom error code for RESP.ISO6581
			/// </summary>
			HW_EE_HIGH_VOLTAGE = ( CustomCodes.ERROR_BEGIN + 5 ),

			/// <summary>
			/// Wrong length; no further indication.
			/// Custom error code for RESP.ISO6700
			/// </summary>
			ISO7816_WRONG_LENGTH_LC = ( CustomCodes.ERROR_BEGIN + 6 ),

			/// <summary>
			/// Command aborted - final chained command expected; running command aborted, new command ignored.
			/// Custom error code for RESP.ISO6883
			/// </summary>
			INCOMPLETE_CHAINING = ( CustomCodes.ERROR_BEGIN + 7 ),

			/// <summary>
			/// Security status not satisfied.
			/// Custom error code for RESP.ISO6982
			/// </summary>
			INTEGRITY_ERROR = ( CustomCodes.ERROR_BEGIN + 8 ),

			/// <summary>
			/// Referenced data invalid.
			/// Custom error code for RESP.ISO6984
			/// </summary>
			KEY_INTEGRITY_ERROR = ( CustomCodes.ERROR_BEGIN + 9 ),

			/// <summary>
			/// Conditions of use not satisfied.
			/// Custom error code for RESP.ISO6985
			/// </summary>
			COND_USE_NOT_SATISFIED = ( CustomCodes.ERROR_BEGIN + 10 ),

			/// <summary>
			/// Command not allowed.
			/// Custom error code for RESP.ISO6986
			/// </summary>
			ISO7816_COMMAND_NOT_ALLOWED = ( CustomCodes.ERROR_BEGIN + 11 ),

			/// <summary>
			/// Wrong datafield parameter.
			/// Custom error code for RESP.ISO6A80
			/// </summary>
			ISO7816_WRONG_PARAMS_FOR_INS = ( CustomCodes.ERROR_BEGIN + 12 ),

			/// <summary>
			/// Key version not found.
			/// Custom error code for RESP.ISO6A82
			/// </summary>
			KEY_VERSION_INVALID = ( CustomCodes.ERROR_BEGIN + 13 ),

			/// <summary>
			/// Host protocol error.
			/// Custom error code for RESP.ISO6A84
			/// </summary>
			HOST_PROTECTION = ( CustomCodes.ERROR_BEGIN + 14 ),

			/// <summary>
			/// Incorrect parameters P1-P2.
			/// Custom error code for RESP.ISO6A86
			/// </summary>
			ISO7816_WRONG_P1P2 = ( CustomCodes.ERROR_BEGIN + 15 ),

			/// <summary>
			/// Incorrect Le value.
			/// Custom error code for RESP.ISO6C00
			/// </summary>
			ISO7816_WRONG_LE = ( CustomCodes.ERROR_BEGIN + 16 ),

			/// <summary>
			/// Instruction code not supported or invalid or not available
			/// in the current state of the SAM.
			/// Custom error code for RESP.ISO6D00
			/// </summary>
			ISO7816_UNKNOWN_INS = ( CustomCodes.ERROR_BEGIN + 17 ),

			/// <summary>
			/// Class not supported.
			/// Custom error code for RESP.ISO6E00
			/// </summary>
			ISO7816_UNKNOWN_CLASS = ( CustomCodes.ERROR_BEGIN + 18 ),

			/// <summary>
			/// Correct execution - authentication failed.
			/// Custom error code for RESP.ISO901E
			/// </summary>
			CRYPTO = ( CustomCodes.ERROR_BEGIN + 19 ),

			/// <summary>
			/// Correct execution - MIFARE Plus PICC protocol error.
			/// Custom error code for RESP.ISO90BE
			/// </summary>
			MIFARE_PLUS_CRYPTO = ( CustomCodes.ERROR_BEGIN + 20 ),

			/// <summary>
			/// Correct execution - error code returned by MIFARE Plus PICC.
			/// Custom error code for RESP.ISO90BF
			/// </summary>
			MIFARE_PLUS_GEN = ( CustomCodes.ERROR_BEGIN + 21 ),

			/// <summary>
			/// Correct execution - UID not complete.
			/// Custom error code for RESP.ISO90C0
			/// </summary>
			ISO_UID_INCOMPLETE = ( CustomCodes.ERROR_BEGIN + 22 ),

			/// <summary>
			/// Correct execution - error code returned by DESFire PICC.
			/// Custom error code for RESP.ISO90DF
			/// </summary>
			DESFIRE_GEN = ( CustomCodes.ERROR_BEGIN + 23 ),

			/// <summary>
			/// UID check byte (BCC) incorrect.
			/// Custom error code for RESP.ISO90EA
			/// </summary>
			ISO_UID_BCC_INCORRECT = ( CustomCodes.ERROR_BEGIN + 24),

			/// <summary>
			/// Incorrect block number.
			/// Custom error code for RESP.ISO90EB
			/// </summary>
			ISO_WRONG_BNR = ( CustomCodes.ERROR_BEGIN + 25 ),

			/// <summary>
			/// Invalid format.
			/// Custom error code for RESP.ISO90EC
			/// </summary>
			ISO_INVALID_FORMAT = ( CustomCodes.ERROR_BEGIN + 26 ),

			/// <summary>
			/// Invalid parameter.
			/// Custom error code for RESP.ISO90ED
			/// </summary>
			ISO_INVALID_PARAMETER = ( CustomCodes.ERROR_BEGIN + 27 ),

			/// <summary>
			/// Correct execution - error code returned by PICC.
			/// Custom error code for RESP.ISO90EF
			/// </summary>
			MIFARE_GEN = ( CustomCodes.ERROR_BEGIN + 28 ),

			/// <summary>
			/// MIFARE (R) NACK 0 received.
			/// Custom error code for RESP.ISO90F0
			/// </summary>
			MIFARE_NAK0 = ( CustomCodes.ERROR_BEGIN + 29 ),

			/// <summary>
			/// MIFARE (R) NACK 1 received.
			/// Custom error code for RESP.ISO90F1
			/// </summary>
			MIFARE_NAK1 = ( CustomCodes.ERROR_BEGIN + 30 ),

			/// <summary>
			/// MIFARE (R) NACK 4 received.
			/// Custom error code for RESP.ISO90F4
			/// </summary>
			MIFARE_NAK4 = ( CustomCodes.ERROR_BEGIN + 31 ),

			/// <summary>
			/// MIFARE (R) NACK 5 received.
			/// Custom error code for RESP.ISO90F5
			/// </summary>
			MIFARE_NAK5 = ( CustomCodes.ERROR_BEGIN + 32 ),

			/// <summary>
			/// MIFARE (R) NACK 2 received.
			/// Custom error code for RESP.ISO90F2
			/// </summary>
			MIFARE_NAK2 = ( CustomCodes.ERROR_BEGIN + 33 ),

			/// <summary>
			/// MIFARE (R) NACK 3 received.
			/// Custom error code for RESP.ISO90F3
			/// </summary>
			MIFARE_NAK3 = ( CustomCodes.ERROR_BEGIN + 34 ),

			/// <summary>
			/// Correct execution - more date expected.
			/// Custom error code for RESP.ISO90AF
			/// </summary>
			OK_CHAINING_ACTIVE = ( CustomCodes.ERROR_BEGIN + 35 )
        }

        public enum LogicalChannel : byte
        {
            LC0 = 0x00,
            LC1,
            LC2,
            LC3
        }

        public enum Config : int
        {
            /**
            * Get / Set the Hostmode; (e.g. #PHHAL_HW_SAMAV2_HC_AV1_MODE).
            */
            HOSTMODE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,
            /**
            * Disables the mapping of standard-configs to the Reader HAL if set to #PH_ON;
            * Default is #PH_OFF; Only applicable in NonX-Mode.
            */
            DISABLE_NONX_CFG_MAPPING
        }

        public enum OpMode : byte
        {
            NON_X = 0x00,
            X_RC523 = 0x01,
            X_RC663 = 0x02
        }

        public enum HcMode : byte
        {
            AV1 = 0x01,
            AV2
        }

        public enum CustomExchangeBit : int
        {
            NO_ENCIPHERING_BIT = 0x0010,
            NO_DECIPHERING_BIT = 0x0020
        }

        public enum DesOption : byte
        {
            DESFIRE4 = 0x00,
            ISO_CRC16,
            ISO_CRC32
        }

        private enum KeySettings : int
        {
            CHANGE_KEY_PICC_DISABLED = 0x0800,
            DECRYPTION_DISABLED = 0x1000,
            ENCRYPTION_DISABLED = 0x2000,
            VERIFY_MAC_DISABLED = 0x4000,
            GENERATE_MAC_DISABLED = 0x8000
        }

        #endregion

        #region DATA_STRUCTURES

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                     	        /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;          	        /**< Pointer to the lower layers parameter structure. */
            public IntPtr pReaderHalDataParams;             /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
            public IntPtr pKeyStoreDataParams;              /**< Pointer to the KeyStore used for Host Authentication */
            public IntPtr pENCCryptoDataParams;             /**< pointer to the ENC crypto layers parameter structure */
            public IntPtr pMACCryptoDataParams;             /**< pointer to the MAC crypto layers parameter structure */
            public IntPtr pCryptoRngDataParams;             /**< Pointer to the parameter structure of the CryptoRng layer. */
            public uint Cmd_Ctr;                            /**< Command counter for Secure Messaging */
            public byte bHostMode;                          /**< Indicates, if we have AV1 or AV2 mode */
            public byte bAuthType;                          /**< the current Authtentication Type used for SM */
            public fixed byte bPendingEncCmdData[16];       /**< Command Data pending for encryption. */
            public byte bPendingEncCmdDataLength;           /**< Length of pending command data to encrypt. */
            public fixed byte bPendingCmdIv[16];            /**< Initvector for pending CMD-Data encryption. */
            public fixed byte bPendingMacCmdData[16];       /**< Command Data pending for MACing. */
            public byte bPendingMacCmdDataLength;           /**< Length of pending command data to MAC. */
            public fixed byte bPendingCmdMac[16];           /**< Initvector for pending CMD-Data MACing. */
            public fixed byte bPendingMacRespData[16];      /**< Response Data pending for MACing. */
            public byte bPendingMacRespDataLength;          /**< Length of pending receive data to MAC. */
            public fixed byte bPendingRespMac[16];          /**< Intermediate response MAC. */
            public fixed byte bPendingRespIv[16];           /**< Initvector for pending receive-Data decryption. */
            public fixed byte bPendingRespData[16];         /**< Pending (unreturned) response data. */
            public byte bPendingRespDataLength;             /**< Length of pending response data. */
            public byte bCmdSM;                             /**< Type of secure messaging for current command. */
            public byte bRespSM;                            /**< Type of secure messaging for current response. */
            public byte bCommandChaining;                   /**< Whether command chaining is active or not. */
            public byte bResponseChaining;                  /**< Whether response chaining is active or not. */
            public fixed byte bUid[7];                      /**< SAM UID. */
            public byte bMasterKeyCmacMode;                 /**< Whether CMAC mode is enabled in the master key. */
            public byte bChangeEntryPlain;                  /**< Whether the next ChangeKeyEntry / ChangeKUCEntry is plain. */
            public byte bOpMode;                            /**< Operation mode; e.g. #PH_HW_SAMAV2_OPMODE_NONX. */
            public byte bLogicalChannel;				    /**< Logical channel number to select for Authenthication*/
            public IntPtr pTxBuffer;               	        /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;              	        /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;               	        /**< Number of valid bytes for exchange within the transmit buffer. */
            public ushort wTxBufLen_Cmd;                    /**< Number of valid bytes for other commands within the transmit buffer. */
            public IntPtr pRxBuffer;               	        /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;              	        /**< Size of the global receive buffer. */
            public ushort wRxBufLen;               	        /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                   /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;                   /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
            public fixed ushort wCfgShadow[SHADOW_COUNT];   /**< Configuration shadow; Stores configuration for current cardtype. */
            public byte bCardType;                 	        /**< Type of card for which the hal is configured for. */
            public byte bTimeoutUnit;              	        /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public ushort wFieldOffTime;           	        /**< Field-Off-Time in milliseconds. */
            public ushort wFieldRecoveryTime;      	        /**< Field-Recovery-Time in milliseconds. */
            public ushort wAdditionalInfo;         	        /**< storage for additional error information. */
            public ushort wTimingMode;         		        /**< storage for timing mode of the hal. */
            public uint dwTimingUs;                         /**< Current timing value. */
            public byte bMifareCryptoDisabled;     	        /**< Contains information about MIFARE Crypto enabled state. */
            public byte b2K3DESEncOption;				    /**< Option for single DES and 2 Key Triple DES encryption keys. Can be set either to #PH_HW_SAMAV2_DES_OPTION_DESFIRE4, #PH_HW_SAMAV2_DES_OPTION_ISO_CRC16 or #PH_HW_SAMAV2_DES_OPTION_ISO_CRC32 */
            public byte bRfResetAfterTo;                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            public byte bDisableNonXCfgMapping;             /**< Storage for #PHHAL_HW_SAMAV2_CONFIG_DISABLE_NONX_CFG_MAPPING setting. */
            public ushort wFdtPc;                           /**< Current timing value backup for PC*/
        };

        public struct ActivateIdleParams
        {
            public byte[] Atqa; /**< ATQA of card */
            public byte Sak;    /**< SAK of card */
            public byte[] Uid;  /**< UID of card */
        };

        public enum AuthenticatePICC_ISOMode : byte
        {
            useNativeCommand = 0x00,
            useISO7816_4_APDU = 0x40,
            useISOCompliantAuth = 0x80
        }

        public class DESFireReadWriteX_Crypto
        {
            public enum WriteMode : byte
            {
                plain = 0x00,
                MACed = 0x01,
                encrypted = 0x03
            }

            private int Option;

            public static implicit operator DESFireReadWriteX_Crypto(byte Option)
            {
                DESFireReadWriteX_Crypto MyInstance = new DESFireReadWriteX_Crypto();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(DESFireReadWriteX_Crypto Option)
            {
                return (byte)Option.Option;
            }

            public byte offset
            {
                set
                {
                    Option &= 0xF0;
                    Option |= (byte)(value & 0x0F);
                }
                get
                {
                    return (byte)(Option & 0x0F);
                }
            }

            public WriteMode writeMode
            {
                set
                {
                    Option &= 0xCF;
                    Option |= (byte)((byte)(value) << 4);
                }
                get
                {
                    return (WriteMode)((Option & 0x30) >> 4);
                }
            }
        }

        public class PKI_ImportKey_Option
        {
            private int Option;

            public static implicit operator PKI_ImportKey_Option(byte Option)
            {
                PKI_ImportKey_Option MyInstance = new PKI_ImportKey_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(PKI_ImportKey_Option Option)
            {
                return (byte)Option.Option;
            }

            public bool importSettingsOnly
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class ChangeKeyPICC_KeyCompMeth
        {
            private int Option;

            public static implicit operator ChangeKeyPICC_KeyCompMeth(byte Option)
            {
                ChangeKeyPICC_KeyCompMeth MyInstance = new ChangeKeyPICC_KeyCompMeth();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(ChangeKeyPICC_KeyCompMeth Option)
            {
                return (byte)Option.Option;
            }

            public bool useNewKeyOnly
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyNewKey
            {
                set
                {
                    if (value)
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ((Option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyCurrentKey
            {
                set
                {
                    if (value)
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ((Option & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool oneDivEncRoundNewKey
            {
                set
                {
                    if (value)
                        Option |= 0x08;
                    else
                        Option &= ~0x08;
                }
                get
                {
                    if ((Option & 0x08) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool oneDivEncRoundCurrentKey
            {
                set
                {
                    if (value)
                        Option |= 0x10;
                    else
                        Option &= ~0x10;
                }
                get
                {
                    if ((Option & 0x10) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool useNewDiversification
            {
                set
                {
                    if (value)
                        Option |= 0x20;
                    else
                        Option &= ~0x20;
                }
                get
                {
                    if ((Option & 0x20) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfpProximityCheck_Option
        {
            private int Option;

            public static implicit operator MfpProximityCheck_Option(byte Option)
            {
                MfpProximityCheck_Option MyInstance = new MfpProximityCheck_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(MfpProximityCheck_Option Option)
            {
                return (byte)Option.Option;
            }

            public bool randomVpcProcessing
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class ActivateIdle_Option
        {
            private int Option;

            public static implicit operator ActivateIdle_Option(byte Option)
            {
                ActivateIdle_Option MyInstance = new ActivateIdle_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(ActivateIdle_Option Option)
            {
                return (byte)Option.Option;
            }

            public bool applyAtqaFilter
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool applySakFilter
            {
                set
                {
                    if (value)
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ((Option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool applyTime
            {
                set
                {
                    if (value)
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ((Option & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class Deselect_FreeCid
        {
            private int Option;

            public static implicit operator Deselect_FreeCid(byte Option)
            {
                Deselect_FreeCid MyInstance = new Deselect_FreeCid();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(Deselect_FreeCid Option)
            {
                return (byte)Option.Option;
            }

            public bool forceFreeing
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class ChangeKeyPICC_Cfg
        {
            private int Option;

            public static implicit operator ChangeKeyPICC_Cfg(byte Option)
            {
                ChangeKeyPICC_Cfg MyInstance = new ChangeKeyPICC_Cfg();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(ChangeKeyPICC_Cfg Option)
            {
                return (byte)Option.Option;
            }

            public byte desfireKeyNo
            {
                set
                {
                    Option &= 0xF8;
                    Option |= (byte)(value & 0x07);
                }
                get
                {
                    return (byte)(Option & 0x07);
                }
            }

            public bool changeMasterKey
            {
                set
                {
                    if (value)
                        Option |= 0x10;
                    else
                        Option &= ~0x10;
                }
                get
                {
                    if ((Option & 0x10) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class DESFireChangeKeyPICC_Cfg
        {
            private int Option;

            public static implicit operator DESFireChangeKeyPICC_Cfg(byte Option)
            {
                DESFireChangeKeyPICC_Cfg MyInstance = new DESFireChangeKeyPICC_Cfg();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(DESFireChangeKeyPICC_Cfg Option)
            {
                return (byte)Option.Option;
            }

            public byte desfireKeyNo
            {
                set
                {
                    Option &= 0xF8;
                    Option |= (byte)(value & 0x07);
                }
                get
                {
                    return (byte)(Option & 0x07);
                }
            }

            public bool changeMasterKey
            {
                set
                {
                    if (value)
                        Option |= 0x10;
                    else
                        Option &= ~0x10;
                }
                get
                {
                    if ((Option & 0x10) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool useISO7816_4_APDU
            {
                set
                {
                    if (value)
                        Option |= 0x40;
                    else
                        Option &= ~0x40;
                }
                get
                {
                    if ((Option & 0x40) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class AuthenticatePICC_AuthMode
        {
            private int Option;

            public static implicit operator AuthenticatePICC_AuthMode(byte Option)
            {
                AuthenticatePICC_AuthMode MyInstance = new AuthenticatePICC_AuthMode();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(AuthenticatePICC_AuthMode Option)
            {
                return (byte)Option.Option;
            }

            public bool useDiversification
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool selectionByDesfireKeyNo
            {
                set
                {
                    if (value)
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ((Option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool useOneDiversificationEncRound
            {
                set
                {
                    if (value)
                        Option |= 0x08;
                    else
                        Option &= ~0x08;
                }
                get
                {
                    if ((Option & 0x08) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool useNewDiversification
            {
                set
                {
                    if (value)
                        Option |= 0x10;
                    else
                        Option &= ~0x10;
                }
                get
                {
                    if ((Option & 0x10) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class ChangeKeyMifare_Option
        {
            private int Option;

            public static implicit operator ChangeKeyMifare_Option(byte Option)
            {
                ChangeKeyMifare_Option MyInstance = new ChangeKeyMifare_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(ChangeKeyMifare_Option Option)
            {
                return (byte)Option.Option;
            }

            public bool retrieveKey
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyKeyA
            {
                set
                {
                    if (value)
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ((Option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyKeyB
            {
                set
                {
                    if (value)
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ((Option & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool retrieveEncrypted
            {
                set
                {
                    if (value)
                        Option |= 0x08;
                    else
                        Option &= ~0x08;
                }
                get
                {
                    if ((Option & 0x08) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class AuthenticateMifare_Option
        {
            private int Option;

            public static implicit operator AuthenticateMifare_Option(byte Option)
            {
                AuthenticateMifare_Option MyInstance = new AuthenticateMifare_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(AuthenticateMifare_Option Option)
            {
                return (byte)Option.Option;
            }

            public bool useKeyDiversification
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class DisableCrypto_ProMas
        {
            private int ProMas;

            public static implicit operator DisableCrypto_ProMas(int ProMas)
            {
                DisableCrypto_ProMas MyInstance = new DisableCrypto_ProMas();
                MyInstance.ProMas = ProMas;
                return MyInstance;
            }

            public static implicit operator int(DisableCrypto_ProMas ProMas)
            {
                return ProMas.ProMas;
            }

            public bool disableChangeKeyPICC
            {
                set
                {
                    if (value)
                        ProMas |= (int)KeySettings.CHANGE_KEY_PICC_DISABLED;
                    else
                        ProMas &= ~(int)KeySettings.CHANGE_KEY_PICC_DISABLED;
                }
                get
                {
                    if ((ProMas & (int)KeySettings.CHANGE_KEY_PICC_DISABLED) != 0x0000)
                        return true;
                    else
                        return false;
                }
            }

            public bool disableDecryption
            {
                set
                {
                    if (value)
                        ProMas |= (int)KeySettings.DECRYPTION_DISABLED;
                    else
                        ProMas &= ~(int)KeySettings.DECRYPTION_DISABLED;
                }
                get
                {
                    if ((ProMas & (int)KeySettings.DECRYPTION_DISABLED) != 0x0000)
                        return true;
                    else
                        return false;
                }
            }

            public bool disableEncryption
            {
                set
                {
                    if (value)
                        ProMas |= (int)KeySettings.ENCRYPTION_DISABLED;
                    else
                        ProMas &= ~(int)KeySettings.ENCRYPTION_DISABLED;
                }
                get
                {
                    if ((ProMas & (int)KeySettings.ENCRYPTION_DISABLED) != 0x0000)
                        return true;
                    else
                        return false;
                }
            }

            public bool disableMACVerify
            {
                set
                {
                    if (value)
                        ProMas |= (int)KeySettings.VERIFY_MAC_DISABLED;
                    else
                        ProMas &= ~(int)KeySettings.VERIFY_MAC_DISABLED;
                }
                get
                {
                    if ((ProMas & (int)KeySettings.VERIFY_MAC_DISABLED) != 0x0000)
                        return true;
                    else
                        return false;
                }
            }

            public bool disableMACGenerate
            {
                set
                {
                    if (value)
                        ProMas |= (int)KeySettings.GENERATE_MAC_DISABLED;
                    else
                        ProMas &= ~(int)KeySettings.GENERATE_MAC_DISABLED;
                }
                get
                {
                    if ((ProMas & (int)KeySettings.GENERATE_MAC_DISABLED) != 0x0000)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class AuthenticateHost_AuthMode
        {
            private int AuthMode;

            public static implicit operator AuthenticateHost_AuthMode(byte AuthMode)
            {
                AuthenticateHost_AuthMode MyInstance = new AuthenticateHost_AuthMode();
                MyInstance.AuthMode = AuthMode;
                return MyInstance;
            }

            public static implicit operator byte(AuthenticateHost_AuthMode AuthMode)
            {
                return (byte)AuthMode.AuthMode;
            }

            public bool useKeyDiversification
            {
                set
                {
                    if (value)
                        AuthMode |= 0x01;
                    else
                        AuthMode &= ~0x01;
                }
                get
                {
                    if ((AuthMode & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool useLCForCMAC
            {
                set
                {
                    if (value)
                        AuthMode |= 0x02;
                    else
                        AuthMode &= ~0x02;
                }
                get
                {
                    if ((AuthMode & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool useSecretKey
            {
                set
                {
                    if (value)
                        AuthMode |= 0x04;
                    else
                        AuthMode &= ~0x04;
                }
                get
                {
                    if ((AuthMode & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyDESWithOneEncRound
            {
                set
                {
                    if (value)
                        AuthMode |= 0x08;
                    else
                        AuthMode &= ~0x08;
                }
                get
                {
                    if ((AuthMode & 0x08) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool useNewDiversification
            {
                set
                {
                    if (value)
                        AuthMode |= 0x10;
                    else
                        AuthMode &= ~0x10;
                }
                get
                {
                    if ((AuthMode & 0x10) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class ChangeKeyEntry_ProMas
        {
            private int promas;

            public static implicit operator ChangeKeyEntry_ProMas(byte promas)
            {
                ChangeKeyEntry_ProMas MyInstance = new ChangeKeyEntry_ProMas();
                MyInstance.promas = promas;
                return MyInstance;
            }

            public static implicit operator byte(ChangeKeyEntry_ProMas promas)
            {
                return (byte)promas.promas;
            }

            public bool updateKeyVa
            {
                set
                {
                    if (value)
                        promas |= 0x80;
                    else
                        promas &= ~0x80;
                }
                get
                {
                    if ((promas & 0x80) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateKeyVb
            {
                set
                {
                    if (value)
                        promas |= 0x40;
                    else
                        promas &= ~0x40;
                }
                get
                {
                    if ((promas & 0x40) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateKeyVc
            {
                set
                {
                    if (value)
                        promas |= 0x20;
                    else
                        promas &= ~0x20;
                }
                get
                {
                    if ((promas & 0x20) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateDFAid_DFKeyNo
            {
                set
                {
                    if (value)
                        promas |= 0x10;
                    else
                        promas &= ~0x10;
                }
                get
                {
                    if ((promas & 0x10) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateKeyNoCEK_KeyVCEK
            {
                set
                {
                    if (value)
                        promas |= 0x08;
                    else
                        promas &= ~0x08;
                }
                get
                {
                    if ((promas & 0x08) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateRefNoKUC
            {
                set
                {
                    if (value)
                        promas |= 0x04;
                    else
                        promas &= ~0x04;
                }
                get
                {
                    if ((promas & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateSET_ExtSET
            {
                set
                {
                    if (value)
                        promas |= 0x02;
                    else
                        promas &= ~0x02;
                }
                get
                {
                    if ((promas & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool keyVersionsSentSeparately
            {
                set
                {
                    if (value)
                        promas |= 0x01;
                    else
                        promas &= ~0x01;
                }
                get
                {
                    if ((promas & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

        }

        public class ChangeKeyEntry_Option
        {
            private int option;

            public static implicit operator ChangeKeyEntry_Option(byte option)
            {
                ChangeKeyEntry_Option MyInstance = new ChangeKeyEntry_Option();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte(ChangeKeyEntry_Option option)
            {
                return (byte)option.option;
            }

            public bool sentPlain
            {
                set
                {
                    if (value)
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ((option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class KillAuthentication_Option
        {
            private int option;

            public static implicit operator KillAuthentication_Option(byte option)
            {
                KillAuthentication_Option MyInstance = new KillAuthentication_Option();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte(KillAuthentication_Option option)
            {
                return (byte)option.option;
            }

            public bool killPICCAuthOnly
            {
                set
                {
                    if (value)
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ((option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class ChangeKUCEntry_ProMas
        {
            private int promas;

            public static implicit operator ChangeKUCEntry_ProMas(byte promas)
            {
                ChangeKUCEntry_ProMas MyInstance = new ChangeKUCEntry_ProMas();
                MyInstance.promas = promas;
                return MyInstance;
            }

            public static implicit operator byte(ChangeKUCEntry_ProMas promas)
            {
                return (byte)promas.promas;
            }

            public bool updateLimit
            {
                set
                {
                    if (value)
                        promas |= 0x80;
                    else
                        promas &= ~0x80;
                }
                get
                {
                    if ((promas & 0x80) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateKeyNoCKUC
            {
                set
                {
                    if (value)
                        promas |= 0x40;
                    else
                        promas &= ~0x40;
                }
                get
                {
                    if ((promas & 0x40) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool updateKeyVCKUC
            {
                set
                {
                    if (value)
                        promas |= 0x20;
                    else
                        promas &= ~0x20;
                }
                get
                {
                    if ((promas & 0x20) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

        }

        public class ChangeKUCEntry_Option
        {
            private int option;

            public static implicit operator ChangeKUCEntry_Option(byte option)
            {
                ChangeKUCEntry_Option MyInstance = new ChangeKUCEntry_Option();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte(ChangeKUCEntry_Option option)
            {
                return (byte)option.option;
            }

            public bool sentPlain
            {
                set
                {
                    if (value)
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ((option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfRead_Option
        {
            private int option;

            public static implicit operator MfRead_Option(byte option)
            {
                MfRead_Option MyInstance = new MfRead_Option();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte(MfRead_Option option)
            {
                return (byte)option.option;
            }

            public bool multiblockRead
            {
                set
                {
                    if (value)
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ((option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfWrite_Option
        {
            private int option;
            private bool first, cont, last;

            public static implicit operator MfWrite_Option(int option)
            {
                MfWrite_Option MyInstance = new MfWrite_Option();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator int(MfWrite_Option option)
            {
                option.option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                option.option &= ~ExchangeOptions.BUFFERED_BIT;

                if (option.first)
                    option.option |= ExchangeOptions.BUFFER_FIRST;
                else if (option.cont)
                    option.option |= ExchangeOptions.BUFFER_CONT;
                else if (option.last)
                    option.option |= ExchangeOptions.BUFFER_LAST;

                return (int)option.option;
            }

            public bool mifareUlWrite
            {
                set
                {
                    if (value)
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ((option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool multiblockWrite
            {
                set
                {
                    if (value)
                        option |= 0x02;
                    else
                        option &= ~0x02;
                }
                get
                {
                    if ((option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if (value)
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if (value)
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if (value)
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }
        }

        public class MfAuthenticateReadWrite_CmdSetting
        {
            private int option;

            public static implicit operator MfAuthenticateReadWrite_CmdSetting(byte option)
            {
                MfAuthenticateReadWrite_CmdSetting MyInstance = new MfAuthenticateReadWrite_CmdSetting();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte(MfAuthenticateReadWrite_CmdSetting option)
            {
                return (byte)option.option;
            }

            public bool keyInformationProvided
            {
                set
                {
                    if (value)
                        option &= ~0x01;
                    else
                        option |= 0x01;
                }
                get
                {
                    if ((option & 0x01) != 0x00)
                        return false;
                    else
                        return true;
                }
            }

            public bool useKeyDiversification
            {
                set
                {
                    if (value)
                        option |= 0x02;
                    else
                        option &= ~0x02;
                }
                get
                {
                    if ((option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfAuthenticate_Option
        {
            private int option;

            public static implicit operator MfAuthenticate_Option(byte option)
            {
                MfAuthenticate_Option MyInstance = new MfAuthenticate_Option();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte(MfAuthenticate_Option option)
            {
                return (byte)option.option;
            }

            public bool useKeyDiversification
            {
                set
                {
                    if (value)
                        option |= 0x01;
                    else
                        option &= ~0x01;
                }
                get
                {
                    if ((option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class MfChangeKey_KeyCompMeth
        {
            private int option;

            public static implicit operator MfChangeKey_KeyCompMeth(byte option)
            {
                MfChangeKey_KeyCompMeth MyInstance = new MfChangeKey_KeyCompMeth();
                MyInstance.option = option;
                return MyInstance;
            }

            public static implicit operator byte(MfChangeKey_KeyCompMeth option)
            {
                return (byte)option.option;
            }

            public bool diversifyKeyA
            {
                set
                {
                    if (value)
                        option |= 0x02;
                    else
                        option &= ~0x02;
                }
                get
                {
                    if ((option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifyKeyB
            {
                set
                {
                    if (value)
                        option |= 0x04;
                    else
                        option &= ~0x04;
                }
                get
                {
                    if ((option & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class DumpSessionKey_Crypto
        {
            private int Crypto;

            public static implicit operator DumpSessionKey_Crypto(byte Crypto)
            {
                DumpSessionKey_Crypto MyInstance = new DumpSessionKey_Crypto();
                MyInstance.Crypto = Crypto;
                return MyInstance;
            }

            public static implicit operator byte(DumpSessionKey_Crypto Crypto)
            {
                return (byte)Crypto.Crypto;
            }

            public bool encrypt
            {
                set
                {
                    if (value)
                        Crypto |= 0x01;
                    else
                        Crypto &= ~0x01;
                }
                get
                {
                    if ((Crypto & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public byte logicalChannelNo
            {
                set
                {
                    Crypto &= 0xF9;
                    Crypto |= (byte)((value & 0x03) << 1);
                }
                get
                {
                    return (byte)((Crypto & 0x06) >> 1);
                }
            }
        }

        public class Buffering_Option
        {
            private int Option;
            private bool first, cont, last;

            public static implicit operator Buffering_Option(int Option)
            {
                Buffering_Option MyInstance = new Buffering_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator int(Buffering_Option Option)
            {
                Option.Option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                Option.Option &= ~ExchangeOptions.BUFFERED_BIT;

                if (Option.first)
                    Option.Option |= ExchangeOptions.BUFFER_FIRST;
                else if (Option.cont)
                    Option.Option |= ExchangeOptions.BUFFER_CONT;
                else if (Option.last)
                    Option.Option |= ExchangeOptions.BUFFER_LAST;

                return Option.Option;
            }

            public bool setTXChaining
            {
                set
                {
                    if (value)
                        Option |= ExchangeOptions.TXCHAINING;
                    else
                        Option &= ~ExchangeOptions.TXCHAINING;
                }
                get
                {
                    if ((Option & ExchangeOptions.TXCHAINING) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool setRXChaining
            {
                set
                {
                    if (value)
                        Option |= ExchangeOptions.RXCHAINING;
                    else
                        Option &= ~ExchangeOptions.RXCHAINING;
                }
                get
                {
                    if ((Option & ExchangeOptions.RXCHAINING) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if (value)
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if (value)
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if (value)
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }
        }

        public class DecipherData_Option
        {
            private int Option;
            private bool first, cont, last;

            public static implicit operator DecipherData_Option(int Option)
            {
                DecipherData_Option MyInstance = new DecipherData_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator int(DecipherData_Option Option)
            {
                Option.Option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                Option.Option &= ~ExchangeOptions.BUFFERED_BIT;

                if (Option.first)
                    Option.Option |= ExchangeOptions.BUFFER_FIRST;
                else if (Option.cont)
                    Option.Option |= ExchangeOptions.BUFFER_CONT;
                else if (Option.last)
                    Option.Option |= ExchangeOptions.BUFFER_LAST;

                return Option.Option;
            }

            public bool setTXChaining
            {
                set
                {
                    if (value)
                        Option |= ExchangeOptions.TXCHAINING;
                    else
                        Option &= ~ExchangeOptions.TXCHAINING;
                }
                get
                {
                    if ((Option & ExchangeOptions.TXCHAINING) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool setRXChaining
            {
                set
                {
                    if (value)
                        Option |= ExchangeOptions.RXCHAINING;
                    else
                        Option &= ~ExchangeOptions.RXCHAINING;
                }
                get
                {
                    if ((Option & ExchangeOptions.RXCHAINING) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if (value)
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if (value)
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if (value)
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }

            public bool withEncDataLength
            {
                set
                {
                    if (value)
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ((Option & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class PKI_GenerateHash_Option
        {
            private int Option;
            private bool first, cont, last;

            public static implicit operator PKI_GenerateHash_Option(int Option)
            {
                PKI_GenerateHash_Option MyInstance = new PKI_GenerateHash_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator int(PKI_GenerateHash_Option Option)
            {
                Option.Option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                Option.Option &= ~ExchangeOptions.BUFFERED_BIT;

                if (Option.first)
                    Option.Option |= ExchangeOptions.BUFFER_FIRST;
                else if (Option.cont)
                    Option.Option |= ExchangeOptions.BUFFER_CONT;
                else if (Option.last)
                    Option.Option |= ExchangeOptions.BUFFER_LAST;

                return Option.Option;
            }

            public bool setTXChaining
            {
                set
                {
                    if (value)
                        Option |= ExchangeOptions.TXCHAINING;
                    else
                        Option &= ~ExchangeOptions.TXCHAINING;
                }
                get
                {
                    if ((Option & ExchangeOptions.TXCHAINING) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool setRXChaining
            {
                set
                {
                    if (value)
                        Option |= ExchangeOptions.RXCHAINING;
                    else
                        Option &= ~ExchangeOptions.RXCHAINING;
                }
                get
                {
                    if ((Option & ExchangeOptions.RXCHAINING) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if (value)
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if (value)
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if (value)
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }

            public bool firstFrame
            {
                set
                {
                    if (value)
                        Option |= 0x04;
                    else
                        Option &= ~0x04;
                }
                get
                {
                    if ((Option & 0x04) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class PKIGenerateKey_Option
        {
            private int Option;

            public static implicit operator PKIGenerateKey_Option(byte Option)
            {
                PKIGenerateKey_Option MyInstance = new PKIGenerateKey_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(PKIGenerateKey_Option Option)
            {
                return (byte)Option.Option;
            }

            public bool fixedExponent
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class AuthenticateMFP_Option
        {
            public enum KDF : byte
            {
                NO_KDF = 0x00,
                SL2_KDF = 0x01,
                SL3_KDF = 0x03
            }

            private int Option;

            public static implicit operator AuthenticateMFP_Option(byte Option)
            {
                AuthenticateMFP_Option MyInstance = new AuthenticateMFP_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator byte(AuthenticateMFP_Option Option)
            {
                return (byte)Option.Option;
            }

            public bool authenticateFirst
            {
                set
                {
                    if (value)
                        Option |= 0x01;
                    else
                        Option &= ~0x01;
                }
                get
                {
                    if ((Option & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool mifareEnciphered
            {
                set
                {
                    if (value)
                        Option |= 0x02;
                    else
                        Option &= ~0x02;
                }
                get
                {
                    if ((Option & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public KDF kdf
            {
                set
                {
                    Option &= 0xF3;
                    Option |= ((byte)value << 2);
                }
                get
                {
                    return (KDF)((Option & 0x0C) >> 2);
                }
            }
        }

        public class KeyEntry
        {
            public enum KeyType : int
            {
                DESFIRE = 0x0000,
                ISO_10116 = 0x0008,
                MIFARE = 0x0010,
                AES_128 = 0x0020,
                THREE_KEY_TRIPLE_DES = 0x0018,
                AES_192 = 0x0028,
                DES_ISO_CRC32 = 0x0030
            }

            public enum KeyClass : int
            {
                HOST = 0x00,
                PICC = 0x01,
                OFFLINECHANGE = 0x02,
                OFFLINECRYPTO = 0x04
            }

            public class KeySet
            {
                private int set;

                public static implicit operator KeySet(int set)
                {
                    KeySet MyInstance = new KeySet();
                    MyInstance.set = set;
                    return MyInstance;
                }

                public static implicit operator int(KeySet set)
                {
                    return set.set;
                }

                public bool allowDumpSessionKey
                {
                    set
                    {
                        if (value)
                            set |= 0x0001;
                        else
                            set &= ~0x0001;
                    }
                    get
                    {
                        if ((set & 0x0001) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool allowCryptoWithSecretKey
                {
                    set
                    {
                        if (value)
                            set |= 0x0002;
                        else
                            set &= ~0x0002;
                    }
                    get
                    {
                        if ((set & 0x0002) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool keepIV
                {
                    set
                    {
                        if (value)
                            set |= 0x0004;
                        else
                            set &= ~0x0004;
                    }
                    get
                    {
                        if ((set & 0x0004) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public KeyType keyType
                {
                    set
                    {
                        set &= 0xFFC7;
                        set |= (int)value;
                    }
                    get
                    {
                        return (KeyType)(set & 0x0038);
                    }
                }

                public bool enableCMACProtectionAV1
                {
                    set
                    {
                        if (value)
                            set |= 0x0100;
                        else
                            set &= ~0x0100;
                    }
                    get
                    {
                        if ((set & 0x0100) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool authKeyAV2
                {
                    set
                    {
                        if (value)
                            set |= 0x0100;
                        else
                            set &= ~0x0100;
                    }
                    get
                    {
                        if ((set & 0x0100) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disabledKeyEntry
                {
                    set
                    {
                        if (value)
                            set |= 0x0200;
                        else
                            set &= ~0x0200;
                    }
                    get
                    {
                        if ((set & 0x0200) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool enableHostAuthAfterResetAV1
                {
                    set
                    {
                        if (value)
                            set |= 0x0400;
                        else
                            set &= ~0x0400;
                    }
                    get
                    {
                        if ((set & 0x0400) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool lockKeyAV2
                {
                    set
                    {
                        if (value)
                            set |= 0x0400;
                        else
                            set &= ~0x0400;
                    }
                    get
                    {
                        if ((set & 0x0400) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disableChangeKeyPICC
                {
                    set
                    {
                        if (value)
                            set |= (int)KeySettings.CHANGE_KEY_PICC_DISABLED;
                        else
                            set &= ~(int)KeySettings.CHANGE_KEY_PICC_DISABLED;
                    }
                    get
                    {
                        if ((set & (int)KeySettings.CHANGE_KEY_PICC_DISABLED) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disableDecryption
                {
                    set
                    {
                        if (value)
                            set |= (int)KeySettings.DECRYPTION_DISABLED;
                        else
                            set &= ~(int)KeySettings.DECRYPTION_DISABLED;
                    }
                    get
                    {
                        if ((set & (int)KeySettings.DECRYPTION_DISABLED) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disableEncryption
                {
                    set
                    {
                        if (value)
                            set |= (int)KeySettings.ENCRYPTION_DISABLED;
                        else
                            set &= ~(int)KeySettings.ENCRYPTION_DISABLED;
                    }
                    get
                    {
                        if ((set & (int)KeySettings.ENCRYPTION_DISABLED) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disableMACVerify
                {
                    set
                    {
                        if (value)
                            set |= (int)KeySettings.VERIFY_MAC_DISABLED;
                        else
                            set &= ~(int)KeySettings.VERIFY_MAC_DISABLED;
                    }
                    get
                    {
                        if ((set & (int)KeySettings.VERIFY_MAC_DISABLED) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disableMACGenerate
                {
                    set
                    {
                        if (value)
                            set |= (int)KeySettings.GENERATE_MAC_DISABLED;
                        else
                            set &= ~(int)KeySettings.GENERATE_MAC_DISABLED;
                    }
                    get
                    {
                        if ((set & (int)KeySettings.GENERATE_MAC_DISABLED) != 0x0000)
                            return true;
                        else
                            return false;
                    }
                }
            }

            public class KeyExtSet
            {
                int extset;

                public static implicit operator KeyExtSet(int extset)
                {
                    KeyExtSet MyInstance = new KeyExtSet();
                    MyInstance.extset = extset;
                    return MyInstance;
                }

                public static implicit operator int(KeyExtSet extset)
                {
                    return extset.extset;
                }

                public KeyClass keyClass
                {
                    set
                    {
                        extset &= 0xF8;
                        extset |= (int)value;
                    }
                    get
                    {
                        return (KeyClass)(extset & 0x07);
                    }
                }

                public bool allowDumpSecret
                {
                    set
                    {
                        if (value)
                            extset |= 0x08;
                        else
                            extset &= ~0x08;
                    }
                    get
                    {
                        if ((extset & 0x08) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

                public bool diversifiedUseOnly
                {
                    set
                    {
                        if (value)
                            extset |= 0x10;
                        else
                            extset &= ~0x10;
                    }
                    get
                    {
                        if ((extset & 0x10) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }
            }

            public byte[] KeyA;
            public byte[] KeyB;
            public byte[] KeyC;
            public byte[] DF_AID;
            public byte DF_KeyNo;
            public byte KeyNoCEK;
            public byte KeyVCEK;
            public byte RefNoKUC;
            public KeySet SET;
            public byte VersionA;
            public byte VersionB;
            public byte VersionC;
            public KeyExtSet ExtSET;

            public KeyEntry()
            {
                SET = new KeySet();
                ExtSET = new KeyExtSet();
            }

            public KeyEntry(byte[] KeyInfo, bool AV2)
            {
                if (AV2)
                {
                    if (KeyInfo.Length < 12)
                        return;
                    SET = (int)((KeyInfo[KeyInfo.Length - 2] << 8) | KeyInfo[KeyInfo.Length - 3]);
                    ExtSET = KeyInfo[KeyInfo.Length - 1];
                }
                else
                {
                    if (KeyInfo.Length < 11)
                        return;
                    SET = (int)((KeyInfo[KeyInfo.Length - 1] << 8) | KeyInfo[KeyInfo.Length - 2]);
                }

                int pointer = 0;

                VersionA = KeyInfo[pointer++];
                VersionB = KeyInfo[pointer++];

                if (SET.keyType != KeyType.AES_192 && SET.keyType != KeyType.THREE_KEY_TRIPLE_DES)
                {
                    VersionC = KeyInfo[pointer++];
                }
                DF_AID = new byte[3];
                Array.Copy(KeyInfo, pointer, DF_AID, 0, 3);
                pointer += 3;
                DF_KeyNo = KeyInfo[pointer++];
                KeyNoCEK = KeyInfo[pointer++];
                KeyVCEK = KeyInfo[pointer++];
                RefNoKUC = KeyInfo[pointer++];
            }

            public byte[] ToByteArray(bool AV2, bool keyVersionsSentSeparately)
            {
                byte[] data;

                if (AV2)
                {
                    data = new byte[61];
                }
                else
                {
                    if (keyVersionsSentSeparately)
                        data = new byte[60];
                    else
                        data = new byte[57];
                }

                if (SET.keyType == KeyType.AES_192 || SET.keyType == KeyType.THREE_KEY_TRIPLE_DES)
                {
                    System.Array.Copy(KeyA, 0, data, 0, 24);
                    System.Array.Copy(KeyB, 0, data, 24, 24);
                }
                else
                {
                    System.Array.Copy(KeyA, 0, data, 0, 16);
                    System.Array.Copy(KeyB, 0, data, 16, 16);
                    System.Array.Copy(KeyC, 0, data, 32, 16);
                    if (keyVersionsSentSeparately)
                        data[59] = VersionC;
                }
                System.Array.Copy(DF_AID, 0, data, 48, 3);
                data[51] = DF_KeyNo;
                data[52] = KeyNoCEK;
                data[53] = KeyVCEK;
                data[54] = RefNoKUC;
                data[55] = (byte)((int)SET);
                data[56] = (byte)((int)SET >> 8);

                if (keyVersionsSentSeparately)
                {
                    data[57] = VersionA;
                    data[58] = VersionB;
                }

                if (AV2)
                    data[60] = (byte)(int)ExtSET;

                return data;
            }
        }

        public class KucEntry
        {
            public long Limit;
            public byte KeyNoCKUC;
            public byte KeyVCKUC;
            public long Value;

            public KucEntry(byte[] Kuc)
            {
                if (Kuc == null)
                    return;

                if (Kuc.Length < 10)
                    return;

                Limit = (long)((Kuc[3] << 24) | (Kuc[2] << 16) | (Kuc[1] << 8) | Kuc[0]);
                KeyNoCKUC = Kuc[4];
                KeyVCKUC = Kuc[5];
                Value = (long)((Kuc[9] << 24) | (Kuc[8] << 16) | (Kuc[7] << 8) | Kuc[6]);
            }

            public byte[] ToByteArray()
            {
                byte[] buffer = new byte[6];
                buffer[0] = (byte)(Limit);
                buffer[1] = (byte)(Limit >> 8);
                buffer[2] = (byte)(Limit >> 16);
                buffer[3] = (byte)(Limit >> 24);
                buffer[4] = KeyNoCKUC;
                buffer[5] = KeyVCKUC;
                return buffer;
            }
        }

        public class PKI_KeyEntry
        {
            public class PKI_Set
            {
                private int set;

                public static implicit operator PKI_Set(int set)
                {
                    PKI_Set MyInstance = new PKI_Set();
                    MyInstance.set = set;
                    return MyInstance;
                }

                public static implicit operator int(PKI_Set set)
                {
                    return set.set;
                }

                public bool privateKeyIncluded
                {
                    set
                    {
                        if (value)
                            set |= 0x01;
                        else
                            set &= ~0x01;
                    }
                    get
                    {
                        if ((set & 0x01) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

                public bool allowPrivateKeyExport
                {
                    set
                    {
                        if (value)
                            set |= 0x02;
                        else
                            set &= ~0x02;
                    }
                    get
                    {
                        if ((set & 0x02) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disabledKeyEntry
                {
                    set
                    {
                        if (value)
                            set |= 0x04;
                        else
                            set &= ~0x04;
                    }
                    get
                    {
                        if ((set & 0x04) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disabledEncryptionHandling
                {
                    set
                    {
                        if (value)
                            set |= 0x08;
                        else
                            set &= ~0x08;
                    }
                    get
                    {
                        if ((set & 0x08) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

                public bool disabledSignatureHandling
                {
                    set
                    {
                        if (value)
                            set |= 0x10;
                        else
                            set &= ~0x10;
                    }
                    get
                    {
                        if ((set & 0x10) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

                public bool enabledUpdateKeyEntries
                {
                    set
                    {
                        if (value)
                            set |= 0x20;
                        else
                            set &= ~0x20;
                    }
                    get
                    {
                        if ((set & 0x20) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

                public bool CRTRepresentation
                {
                    set
                    {
                        if (value)
                            set |= 0x40;
                        else
                            set &= ~0x40;
                    }
                    get
                    {
                        if ((set & 0x40) != 0x00)
                            return true;
                        else
                            return false;
                    }
                }

            }

            public PKI_Set PKISet;
            public byte KeyNoCEK;
            public byte KeyVCEK;
            public byte RefNoKUC;
            public byte[] N, e, P, Q, DP, DQ, IPQ;

            private static void removePaddingBE(ref byte[] data)
            {
                for (int i = 0; i < data.Length; i++)
                {
                    if (data[i] != 0)
                    {
                        Array.Copy(data, i, data, 0, data.Length - i);
                        Array.Resize(ref data, data.Length - i);
                        break;
                    }
                }
            }

            private static byte[] pad2LengthBE(byte[] data, int length)
            {
                if (data.Length >= length)
                    return data;
                else
                {
                    byte[] buffer = new byte[length];
                    Array.Copy(data, 0, buffer, length - data.Length, data.Length);
                    return buffer;
                }
            }

            public static implicit operator PKI_KeyEntry(byte[] key)
            {
                if (key.Length < 9)
                    return null;

                PKI_KeyEntry MyInstance = new PKI_KeyEntry();
                MyInstance.PKISet = (int)((key[0] << 8) | key[1]);
                MyInstance.KeyNoCEK = key[2];
                MyInstance.KeyVCEK = key[3];
                MyInstance.RefNoKUC = key[4];
                int length_N = (int)((key[5] << 8) | key[6]);
                int length_e = (int)((key[7] << 8) | key[8]);

                if (key.Length < (9 + length_N + length_e))
                    return null;

                MyInstance.N = new byte[length_N];
                MyInstance.e = new byte[length_e];

                if (key.Length > (9 + length_N + length_e))
                {
                    Array.Copy(key, 13, MyInstance.N, 0, length_N);
                    Array.Copy(key, 13 + length_N, MyInstance.e, 0, length_e);

                    int pointer = 13 + length_N + length_e;

                    if (key.Length < pointer)
                        return null;

                    int length_P = (int)((key[9] << 8) | key[10]);
                    int length_Q = (int)((key[11] << 8) | key[12]);

                    if (key.Length < (pointer + 2 * length_P + 3 * length_Q))
                        return null;

                    MyInstance.P = new byte[length_P];
                    MyInstance.Q = new byte[length_Q];
                    MyInstance.DP = new byte[length_P];
                    MyInstance.DQ = new byte[length_Q];
                    MyInstance.IPQ = new byte[length_Q];

                    Array.Copy(key, pointer, MyInstance.P, 0, length_P);
                    pointer += length_P;
                    Array.Copy(key, pointer, MyInstance.Q, 0, length_Q);
                    pointer += length_Q;
                    Array.Copy(key, pointer, MyInstance.DP, 0, length_P);
                    pointer += length_P;
                    Array.Copy(key, pointer, MyInstance.DQ, 0, length_Q);
                    pointer += length_Q;
                    Array.Copy(key, pointer, MyInstance.IPQ, 0, length_Q);
                    pointer += length_Q;

                    removePaddingBE(ref MyInstance.DP);
                    removePaddingBE(ref MyInstance.DQ);
                    removePaddingBE(ref MyInstance.IPQ);
                }
                else
                {
                    Array.Copy(key, 9, MyInstance.N, 0, length_N);
                    Array.Copy(key, 9 + length_N, MyInstance.e, 0, length_e);

                    MyInstance.P = MyInstance.Q = MyInstance.DP = MyInstance.DQ = MyInstance.IPQ = null;
                }

                return MyInstance;
            }

            public static implicit operator byte[](PKI_KeyEntry key)
            {
                return key.ToByteArray(key.PKISet.privateKeyIncluded);
            }

            public byte[] ToByteArray(bool privateKeyIncluded)
            {
                byte[] buffer;
                int pointer = 0;

                if (N == null || e == null)
                    return null;

                buffer = new byte[9 + N.Length + e.Length];

                buffer[pointer++] = (byte)((int)PKISet >> 8);
                buffer[pointer++] = (byte)(int)(PKISet);

                buffer[pointer++] = KeyNoCEK;
                buffer[pointer++] = KeyVCEK;
                buffer[pointer++] = RefNoKUC;

                buffer[pointer++] = (byte)(N.Length >> 8);
                buffer[pointer++] = (byte)N.Length;
                buffer[pointer++] = (byte)(e.Length >> 8);
                buffer[pointer++] = (byte)e.Length;

                if (privateKeyIncluded)
                {
                    if (P == null || Q == null || DP == null || DQ == null || IPQ == null)
                        return null;

                    Array.Resize(ref buffer, 13 + N.Length + e.Length + 2 * P.Length + 3 * Q.Length);

                    buffer[pointer++] = (byte)(P.Length >> 8);
                    buffer[pointer++] = (byte)P.Length;
                    buffer[pointer++] = (byte)(Q.Length >> 8);
                    buffer[pointer++] = (byte)Q.Length;
                }

                Array.Copy(N, 0, buffer, pointer, N.Length);
                pointer += N.Length;
                Array.Copy(e, 0, buffer, pointer, e.Length);
                pointer += e.Length;

                if (privateKeyIncluded)
                {
                    Array.Copy(P, 0, buffer, pointer, P.Length);
                    pointer += P.Length;
                    Array.Copy(Q, 0, buffer, pointer, Q.Length);
                    pointer += Q.Length;
                    Array.Copy(pad2LengthBE(DP, P.Length), 0, buffer, pointer, P.Length);
                    pointer += P.Length;
                    Array.Copy(pad2LengthBE(DQ, Q.Length), 0, buffer, pointer, Q.Length);
                    pointer += Q.Length;
                    Array.Copy(pad2LengthBE(IPQ, Q.Length), 0, buffer, pointer, Q.Length);
                }

                return buffer;
            }
        }

        public enum LockUnlock_Mode : byte
        {
            unlock = 0x00,
            lockWithoutUnlockKey = 0x01,
            lockWithUnlockKey = 0x02,
            activateAV2Mode = 0x03
        }

        public enum AuthenticateHost_HostMode : byte
        {
            plain = 0x00,
            MAC = 0x01,
            full = 0x02
        }

        public enum PKI_HashAlgorithm : byte
        {
            SHA_1 = 0x00,
            SHA_224 = 0x01,
            SHA_256 = 0x03
        }

        public class VerifyGenerateMAC_Num
        {
            private int Num;

            public static implicit operator VerifyGenerateMAC_Num(byte Num)
            {
                VerifyGenerateMAC_Num MyInstance = new VerifyGenerateMAC_Num();
                MyInstance.Num = Num;
                return MyInstance;
            }

            public static implicit operator byte(VerifyGenerateMAC_Num Num)
            {
                return (byte)Num.Num;
            }

            public bool useMFPTruncation
            {
                set
                {
                    if (value)
                    {
                        Num = 0x80;
                    }
                    else
                    {
                        Num &= ~0x80;
                    }
                }
                get
                {
                    if ((Num & 0x80) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public byte numMACBytes
            {
                set
                {
                    Num &= 0xE0;
                    Num |= (byte)(value & 0x1F);
                }
                get
                {
                    return (byte)((Num & 0x1F));
                }
            }
        }

        public class DumpSecretKey_Crypto
        {
            private int crypto;

            public static implicit operator DumpSecretKey_Crypto(byte crypto)
            {
                DumpSecretKey_Crypto MyInstance = new DumpSecretKey_Crypto();
                MyInstance.crypto = crypto;
                return MyInstance;
            }

            public static implicit operator byte(DumpSecretKey_Crypto crypto)
            {
                return (byte)crypto.crypto;
            }

            public bool encryptedDump
            {
                set
                {
                    if (value)
                    {
                        crypto = 0x01;
                    }
                    else
                    {
                        crypto &= ~0x01;
                    }
                }
                get
                {
                    if ((crypto & 0x01) != 0x00)
                        return true;
                    else
                        return false;
                }
            }

            public bool diversifiedDump
            {
                set
                {
                    if (value)
                    {
                        crypto = 0x02;
                    }
                    else
                    {
                        crypto &= ~0x02;
                    }
                }
                get
                {
                    if ((crypto & 0x02) != 0x00)
                        return true;
                    else
                        return false;
                }
            }
        }

        public class CombinedReadWriteMfp_Option
        {
            private int Option;
            private bool first, cont, last;

            public enum CmdResp : byte
            {
                command = 0x00,
                response = 0x04,
                command_response = 0x08
            }

            public static implicit operator CombinedReadWriteMfp_Option(int Option)
            {
                CombinedReadWriteMfp_Option MyInstance = new CombinedReadWriteMfp_Option();
                MyInstance.Option = Option;
                return MyInstance;
            }

            public static implicit operator int(CombinedReadWriteMfp_Option Option)
            {
                Option.Option &= ~ExchangeOptions.LEAVE_BUFFER_BIT;
                Option.Option &= ~ExchangeOptions.BUFFERED_BIT;

                if (Option.first)
                    Option.Option |= ExchangeOptions.BUFFER_FIRST;
                else if (Option.cont)
                    Option.Option |= ExchangeOptions.BUFFER_CONT;
                else if (Option.last)
                    Option.Option |= ExchangeOptions.BUFFER_LAST;

                return Option.Option;
            }

            public bool setBufferFirst
            {
                set
                {
                    first = value;
                    if (value)
                    {
                        cont = last = false;
                    }
                }
                get
                {
                    return first;
                }
            }

            public bool setBufferCont
            {
                set
                {
                    cont = value;
                    if (value)
                    {
                        first = last = false;
                    }
                }
                get
                {
                    return cont;
                }
            }

            public bool setBufferLast
            {
                set
                {
                    last = value;
                    if (value)
                    {
                        first = cont = false;
                    }
                }
                get
                {
                    return last;
                }
            }

            public CmdResp commandResponse
            {
                set
                {
                    Option &= 0xFFF3;
                    Option |= (int)value;
                }
                get
                {
                    return (CmdResp)(Option & 0x000C);
                }

            }
        }

        public enum ChangeKeyMfp_Cmd : byte
        {
            writeEnc = 0xA0,
            writeEncWithRespMAC = 0xA1
        }

        #endregion

        #region PRIVATE_FUNCTIONS

        private byte[] MarshalCopy(Status_t status, IntPtr RxBufferPointer, int length)
        {
            if ((status.Equals(Error_Gen.SUCCESS) || status.Equals(Error_Gen.SUCCESS_CHAINING) || status.Equals(Error_Gen.SUCCESS_INCOMPLETE_BYTE)) && length > 0)
            {
                byte[] buffer = new byte[length];
                Marshal.Copy(RxBufferPointer, buffer, 0, length);
                return buffer;
            }
            else
                return null;
        }

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Init(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,              /**< [In] Pointer to the lower layers parameter structure */
            IntPtr pReaderHalDataParams,        /**< [In] Pointer to a Reader-HAL in Non-X Mode. Can be NULL if X-Mode is intended. */
            IntPtr pKeyStoreDataParams,         /**< [In] Pointer to the KeyStore used for Host Authentication. */
            IntPtr pCryptoENCDataParams,        /**< [In] Pointer to the ENC crypto layers parameter structure. */
            IntPtr pCryptoMACDataParams,        /**< [In] Pointer to the MAC crypto layers parameter structure. */
            IntPtr pCryptoRngDataParams,        /**< [In] Pointer to the parameter structure of the CryptoRng layer. */
            byte bOpMode,                       /**< [In] The desired operation mode; e.g. #PH_HW_SAMAV2_OPMODE_NONX. */
            byte bLogicalChannel,               /**< [In] The desired logical channel for this HAL. */
            IntPtr pTxBuffer,                   /**< [In]Pointer to internal transmit buffer.*/
            ushort wTxBufSize,                  /**< [In]internal transmit buffer size.*/
            IntPtr pRxBuffer,                   /**< [In]Pointer to internal receive buffer (can be same as transmit buffer).*/
            ushort wRxBufSize                   /**< [In]internal receive buffer size.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_DetectMode(
            ref DataParams_t m_pDataParams
            );

        #region SAM SECURITY
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_DisableCrypto(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wProMas							/**< [In] Two byte mask to specify the desired settings for cryptography-related features. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_LockUnlock(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bMode,                             /**< [In] Mode for lock/unlock.*/
            ushort wRdKeyNo,					/**< [In] The key reference number for readerLib key storage. */
            ushort wRdKeyV,					/**< [In] The key reference version for readerLib key storage. */
            byte bSamKeyNo,                            /**< [In] Key Number to be used.*/
            byte bSamKeyV,                           /**< [In] Key Version to be used.*/
            byte bUnlockKeyNo,                      /**< [In] Unlock Key Number to be used.*/
            byte bUnlockKeyV,                     /**< [In] Unlock Key Version to be used.*/
            uint bMaxChainBlocks                   /**< [In] Maximum amount of data to be chained.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bAuthType_HostMode,                     /**< [In] the type of Authentication to do*/
            ushort wRdKeyNo,                /**< [In] the key reference number for readerLib KeyStorage */
            ushort wRdKeyV,                /**< [In] the key reference version for readerLib KeyStorage */
            byte bSamKeyNo,                 /**< [In] the key reference number for SAM KeyStorage */
            byte bSamKeyV,                 /**< [In] the key reference version for SAM KeyStorage */
            byte[] pDivInput,                     /**< [In] DivInp data */
            byte bDivInputLength                  /**< [In] DivInp data length */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ActivateOfflineKey(
            ref DataParams_t pDataParams,               /**< [In] Pointer to this layers parameter structure. */
            byte bKeyNo,						        /**< [In] the key reference number */
            byte bKeyV,						            /**< [In] the key reference version */
            byte[] pDivInput,							/**< [In] Encrypted (Random number PCD1 || Random number PICC1) */
            byte bDivInputLength						/**< [In] Encrypted data length */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_SelectApplication(
            ref DataParams_t pDataParams,  	        /**< [In] Pointer to this layers parameter structure. */
            byte[] pDF_Aid							/**< [In] DF_AID data */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_LoadInitVector(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pInitVector,						/**< [In] InitVector data */
            byte bInitVectorLength					/**< [In] InitVector data length */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_KillAuthentication(
            ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure. */
            byte bOption          /**< [In] Only valid in SamAV2 mode. If set to 01, only the PICC authentication is killed. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_GetChallenge(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bExpL,                  /**< [In] Amount of data bytes to be retrieved. */
            byte[] pRnd                    /**< [Out] Pointer containing the challenge */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_IsoExternalAuthenticate(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bKeyNo,                    /**< [In] the key reference number */
            byte bKeyV,                    /**< [In] the key reference version */
            byte[] pEncRnd,                     /**< [In] Encrypted (Random number PCD1 || Random number PICC1) */
            byte bEncRndLength                  /**< [In] Encrypted data length */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_IsoInternalAuthenticate(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte[] pRPcd2,                      /**< [In] Random number PCD2. */
            byte bRPcd2Length,                  /**< [In] Length of RPCD2. */
            ref IntPtr pEncResp,                /**< [Out] Encrypted (Random number PICC2 || Random number PCD2). */
            ref ushort pEncRespLength             /**< [Out] Length of encrypted response. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_GetVersion(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pVersion,           				/**< [Out] Pointer to the version length of the data to be read out.*/
            ref byte pVersionLength      			/**< [Out] Size of the received version.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_Sleep(
            ref DataParams_t m_pDataParams	/**< [In] Pointer to this layers parameter structure. */
            );

        #endregion

        #region SAM KEY HANDLING
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ChangeKeyEntry(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bOption,                       /**< [In] Option mask for this command - only used in AV1. */
            byte bKeyNo,                    /**< [In] number of the key entry to be returned (00h to 7Fh)*/
            byte bProMas,						/**< [In] ProMas byte indicating KeyEntry field that should be taken into account*/
            byte[] bKeyData,                    /**< [In] Key entry structure pointer containing Key Entry*/
            byte bKeyDataLength                 /**< [In] Length of the Key Entry*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_GetKeyEntry(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bKeyNo,                    /**< [In] number of the key entry to be returned (00h to 7Fh)*/
            byte[] bKeyEntry,               /**< [In] Key entry structure pointer containing Key Entry*/
            ref byte bKeyEntryLength            /**< [In] Length of the Key Entry*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ChangeKUCEntry(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte bKucNo,                    /**< [In] number of the key usage counter to be updated (00h to 0Fh)*/
            byte bProMas,						/**< [In] Program mask indicating the fields that should be changed. */
            byte[] pKucData,                    /**< [In] Buffer containing the KUC data: */
            byte KucDataLength                  /**< [In] Length of the KUC data. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_GetKUCEntry(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bKucNo,                    /**< [In] number of the key usage counter to be looked at (00h to 0Fh)*/
            byte[] pKucEntry                    /**< [Out] Buffer containing the KUC entry. This buffer has to be 10 bytes long. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_DumpSecretKey(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bCrypto,                       		/**< [In] Bit mask containing the crypto options for the key dump. */
            byte bKeyNo,								/**< [In] Reference number of the key entry to be dumped. */
            byte bKeyV,								/**< [In] Reference number of the key entry to be dumped. */
            byte[] pDivInput,						/**< [In] Diversification input data. */
            byte bDivInputLength,					/**< [In] Diversification input data length. If set to 00h no diversification is used. */
            byte[] pSecretKey,                     	/**< [Out] Buffer containing the plain secret key. It has to be 24 bytes long. */
            ref byte pSecretKeyLength                	/**< [Out] Amount of valid data in secret key buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_DumpSessionKey(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            ref DataParams_t pDataParamsEnc,     /**< [In] Pointer to this layers parameter structure. */
            byte bCrypto,                       /**< [In] Encrypt the key or not */
            byte[] pSessionKey,                 /**< [Out] Buffer containig the session key. It has to be 32 bytes long. */
            ref byte pSessionKeyLength          /**< [Out] Amount of valid data in session key buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_DisableKeyEntry(
            ref DataParams_t pDataParams,  	    /**< [In] Pointer to this layers parameter structure. */
            byte bKeyNo,                     /**< [In] number of the key entry to be returned (00h to 7Fh)*/
            byte[] pOfflineCryptogram,               /**< [Out] Buffer containing the cryptogram for offline key deactivation. This parameter is only used in AV2. */
            byte bCryptogramLength					/**< [Out] Length of the offline cryptogram. If set to 00h no offline cryptogram is sent. */
            );

        #endregion

        #region SAM DATA PROCESS
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_VerifyMAC(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                     /**< [In] Option parameter. */
            byte bNum,                          /**< [In] Number of MAC bytes to check in the Plain data buffer. */
            byte[] pTxBuffer,    				/**< [In] Plain data including a MAC to be checked */
            byte bTxLength       				/**< [In] length of input data */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_GenerateMAC(
            ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. */
            byte bNum,                          /**< [In] Number of MAC bytes to generate. */
            byte[] pTxBuffer,    					/**< [In] Plain data to be MACed */
            byte bTxLength,     					/**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_DecipherData(
            ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. */
            byte[] pTxBuffer,    					/**< [In] data to transmit */
            ushort bTxLength,     					/**< [In] length of input data */
            byte[] pLenEncInputData,                 /**< [In] Length of encrypted input data [3]. */
            ref IntPtr ppRxBuffer,  				/**< [Out] Pointer to received data */
            ref ushort pRxLength   					/**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_EncipherData(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. */
            byte[] pTxBuffer,    					/**< [In] data to transmit */
            byte bTxLength,     					/**< [In] length of input data */
            byte bOffset,            				/**< [In] Offset indicating first data byte to be encrypted. */
            ref IntPtr ppRxBuffer,  				/**< [Out] Pointer to received data */
            ref ushort pRxLength   					/**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_DecipherOfflineData(
            ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. */
            byte[] pTxBuffer,    					/**< [In] data to transmit */
            byte bTxLength,     					/**< [In] length of input data */
            ref IntPtr ppRxBuffer,  				/**< [Out] Pointer to received data */
            ref ushort pRxLength       				/**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_EncipherOfflineData(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. */
            byte[] pTxBuffer,    					/**< [In] data to transmit */
            byte bTxLength,     					/**< [In] length of input data */
            ref IntPtr ppRxBuffer, 					/**< [Out] Pointer to received data */
            ref ushort pRxLength   					/**< [Out] number of received data bytes */
            );

        #endregion

        #region NON X MODE: PKI
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_GenerateKeyPair(
            ref DataParams_t pDataParams,           /**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte bPKI_KeyNo,                        /**< [In] number of the key entry to be returned (00h to 02h)*/
            ushort wPKI_Set,							/**< [In] PKI_SET bytes representing the configurations settings of the key entry*/
            byte bPKI_KeyNoCEK,			        		/**< [In] Indicates current keyNoCEK of KeyNo */
            byte bPKI_KeyVCEK,	    				/**< [In] Indicates current keyVerCEK of KeyNo */
            byte bPKI_RefNoKUC,     					/**< [In] Indicates current RefNoKUC */
            ushort wPKI_NLen,					/**< [In] Key length for the private exponent of the RSA key (multiple of 8 and in [32;256]). */
            ushort wPKI_eLen,					/**< [In] Key length for the public exponent of the RSA key(multiple of 4 and in [4;256]). */
            byte[] pPKI_e							/**< [In] If bFixed_e is set this buffer holds the fixed exponent e. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_ExportPrivateKey(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. If set to \ref PH_EXCHANGE_RXCHAINING only dummy commands are sent to fetch the remaining key data. */
            byte bPKI_KeyNo,                         /**< [In] number of the key entry to be returned (00h to 01h)*/
            ref IntPtr ppRxBuffer,  					/**< [Out] Pointer to received data */
            ref ushort pRxLength	   					/**< [Out] Length of received data. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_ExportPublicKey(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. If set to \ref PH_EXCHANGE_RXCHAINING only dummy commands are sent to fetch the remaining key data. */
            byte bPKI_KeyNo,                         /**< [In] number of the key entry to be returned (00h to 02h)*/
            ref IntPtr ppRxBuffer,  					/**< [Out] Pointer to received data */
            ref ushort pRxLength	   					/**< [Out] Length of received data. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_SendSignature(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            ref IntPtr ppRxBuffer,  					/**< [Out] Pointer to received data */
            ref ushort pRxLength	   					/**< [Out] Length of received data. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_UpdateKeyEntries(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bNumberOfKeyEntries,				/**< [In] the number of PKI key entries to update*/
            byte bHashingAlg,                       	/**< [In] Hashing algorithm selection (00:SHA-1, 01:SHA-224, 11: SHA-256, 10:RFU)*/
            byte bPKI_KeyNo_Enc,					/**< [In] number of the key entry to be used for decryption (00h to 01h)*/
            byte bPKI_KeyNo_Sign,					/**< [In] number of the key entry to be used for signature verification (00h to 02h)*/
            byte[] pKeyFrame,						/**< [In] Buffer containing the RSA encrypted key entries and the signature. */
            ushort wKeyFrameLength					/**< [In] Key frame length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_GenerateSignature(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bHashingAlg,                     	/**< [In] Hashing algorithm selection (00:SHA-1, 01:SHA-224, 11: SHA-256, 10:RFU)*/
            byte bPKI_KeyNo,                         /**< [In] number of the key entry to be returned (00h to 01h)*/
            byte[] pHashMsgData,					    /**< [In] Hash message to be signed*/
            byte bHashMsgLength						/**< [In] Hash message length */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_GenerateHash(
            ref DataParams_t pDataParams,  	        /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,       					/**< [In] Option parameter. */
            byte bHashingAlg,                       /**< [In] Hashing algorithm selection (00:SHA-1, 01:SHA-224, 11: SHA-256, 10:RFU)*/
            uint dwMsgLength,    					/**< [In] Overall message length. */
            byte[] pTxBuffer,    					/**< [In] Message chunk to be hashed. */
            ushort wTxLength,	     				/**< [In] Length of message chunk. */
            byte[] pHash,		  					/**< [Out] Buffer containing the hash after sending the last message chunk. The buffer has to be 32 bytes long. */
            ref byte pHashLength					/**< [Out] Amount of valid data in hash buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_VerifySignature(
            ref DataParams_t pDataParams,  	        /**< [In] Pointer to this layers parameter structure. */
            byte bPKI_KeyNo,                        /**< [In] Number of the key entry to be used for verification (00h to 02h). */
            byte bHashingAlg,                       /**< [In] Hashing algorithm selection (00:SHA-1, 01:SHA-224, 11: SHA-256, 10:RFU). */
            byte[] pHashData,					    /**< [In] Hash data. */
            byte bHashDataLength,					/**< [In] Hash data length. */
            byte[] pSignature,				    	/**< [In] RSA digital signature. */
            ushort wSignatureLength					/**< [In] RSA digital signature length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_PKI_ImportKey(
            ref DataParams_t pDataParams,  	        /**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte bPKI_KeyNo,                         /**< [In] Number of the key entry to be imported (00h to 02h). */
            ushort wPKI_Set,							/**< [In] Configuration settings of the imported key entry. It indicates wheter a private of public key shall be imported.  */
            byte bPKI_KeyNoCEK,						/**< [In] Reference number to the change key of the imported key entry. */
            byte bPKI_KeyVCEK,						/**< [In] Version of the change key of the imported key entry. */
            byte bPKI_RefNoKUC,						/**< [In] Reference number to the KUC of the created key entry. */
            ushort wPKI_NLen,							/**< [In] Length of Modulus N (multiple of 8 and in [32;256]). */
            ushort wPKI_eLen,							/**< [In] Length of exponent e (multiple of 4 and in [4;256]). */
            ushort wPKI_PLen,							/**< [In] Length of prime P. Only used if bPublicOnly = 00h. */
            ushort wPKI_QLen,							/**< [In] Length of prime Q. Only used if bPublicOnly = 00h. */
            byte[] pPKI_N,							/**< [In] Modulus N. */
            byte[] pPKI_e,							/**< [In] Exponent e. */
            byte[] pPKI_P,							/**< [In] Prime P. */
            byte[] pPKI_Q,							/**< [In] Prime Q. */
            byte[] pPKI_dP,							/**< [In] Parameter dP padded up to a length of wPKI_PLen. */
            byte[] pPKI_dQ,							/**< [In] Parameter dQ padded up to a length of wPKI_QLen. */
            byte[] pPKI_iPQ							/**< [In] Inverse P(-1) mod Q padded up to a length of wPKI_QLen. */
            );

        #endregion

        #region NON X MODE: MIFARE PLUS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_AuthenticateMfp_Part1(
            ref DataParams_t pDataParams,           /**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte bKeyNo,                         /**< [In] Key number to be used in authentication. */
            byte bKeyV,                        /**< [In] Key version to be used in authentication. */
            byte[] pRndB,                            /**< [In] Encrypted RND B received from the card. This buffer has to be 22 bytes long in case of an active MFC authentication; in all other cases 16 bytes. */
            byte[] pDivInput,                        /**< [In] Diversification input. */
            byte bDivInputLength,                   /**< [In] Length of diversification input. If it is set to 00h no diversification is used. */
            byte[] pRndARndB,                      /**< [Out] Pointer to the buffer containing encrypted RndA || RndB to be sent to the card. */
            ref byte pRndARndBLength					/**< [Out] Amount of valid bytes in encrypted RNDA || RndB buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_AuthenticateMfp_Part2(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte[] pEncData,                            /**< [In] Encrypted RND A' including TI */
            byte bEncDataLength,                        /**< [In] Size of RND A' (16 or 32 bytes) */
            byte[] pPdCap,                           /**< [Out] PD Caps received from card */
            byte[] pPcdCap                           /**< [Out] PCD Caps received from card */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_CombinedReadMfp(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,							/**< [In] Implements buffering. In Combination with TX_CHAINING  manual AF handling can be implemented*/
            byte[] pData,                            /**< [In] Input Data Stream */
            byte bDataLength,                        /**< [In] Length of bData */
            ref IntPtr ppOutput,                        /**< [Out] encrypted command or decrypted response */
            ref ushort pOutputLength                     /**< [Out] Length of encrypted command or decrypted response */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_CombinedWriteMfp(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,							/**< [In] Implements buffering. */
            byte[] pData,                            /**< [In] Input Data Stream */
            byte bDataLength,                        /**< [In] Length of bData */
            ref IntPtr ppOutput,                        /**< [Out] encrypted command or decrypted response */
            ref ushort pOutputLength                     /**< [Out] Length of encrypted command or decrypted response */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ChangeKeyMfp_Part1(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bCmd,								/**< [In] Command used to perform the change */
            ushort wBlockNo,							/**< [In] Block Number */
            byte bKeyNo,							/**< [In] Key number to be used */
            byte bKeyV,						/**< [In] Key version to be used */
            byte[] pDivInput,                        /**< [In] Pointer to Div Input Data (could be NULL)*/
            byte bDivInputLength,                     /**< [In] Div input length (could be NULL) */
            ref IntPtr ppOutput,						/**< [Out] Output data from SAM */
            ref ushort pOutputLength						/**< [Out] Output data length */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ChangeKeyMfp_Part2(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pDataIn,							/**< [In] Pointer to Input Data (response from PICC )*/
            byte bDataInLength							/**< [In] Input Data length */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                           /**< [In] Implements the buffering: Note: For BUFFERED_START the parameters are ignored. For BUFFERED_LAST the PCD Caps are used */
            byte[] pData,								/**< [In] Buffer containing the data set field. */
            byte bDataLength,						/**< [In] Length of the data set buffer. */
            byte[] pPcdCapabilities,                 /**< [In] PcdCapabilities sent to the card. This buffer has to be bPcdCapabilitiesLength bytes long. */
            byte bPcdCapabilitiesLength,            /**< [In] Length of PcdCapabilities. */
            ref IntPtr pSupportedCardInfo,              /**< [Out] Card Info of supported cards */
            ref ushort pSupportedCardInfoLength         /**< [Out] Length of SupportedCardInfo */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_SelectVirtualCardMfp(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bKeyNo,                        /**< [In] Key number to be used in SVC command*/
            byte bKeyV,                       /**< [In] Key version to be used in SVC command*/
            byte[] pRndQ,                            /**< [In] RNDQ sent to the card . */
            byte[] pDivInput,                        /**< [In] Diversification input */
            byte bDivInputLength,                   /**< [In] Length if bDivInput */
            byte[] pMac                              /**< [Out] Mac to be sent to the card */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part1(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bKeyNo,                         /**< [In] Key number to be used in SVC command. */
            byte bKeyV,                        /**< [In] Key version to be used in SVC command. */
            byte bPpse,								/**< [In] PPSE byte used. */
            byte[] pRndQ,                            /**< [In] RndQ sent to the card. This buffer has to be 12 bytes long. */
            byte[] pDivInput,                        /**< [In] Diversification input. */
            byte bDivInputLength,                    /**< [In] Length of diversification input. If 00h no diversification is used. */
            byte[] pMac                              /**< [Out] MAC to be sent to the card. This buffer has to be 8 bytes long. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part2(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pMac                      /**< [In] Input in case of response MAC verification, output in case of command MAC generation. The buffer has to be 8 bytes long. */
            );

        #endregion

        #region NON X MODE: MIFARE CLASSIC
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_AuthenticateMifare_Part1(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte[] pUid,								/**< [In] Pointer to the Uid of the Card.*/
            byte bKeyNoM,							/**< [In] Key Number.*/
            byte bKeyVM,							/**< [In] Key Version.*/
            byte bKeyType,							/**< [In] Key type A or B.*/
            byte bBlockNo,							/**< [In] Block Number.*/
            byte[] pRndB,							/**< [In] Random Number received from card [5].*/
            byte bDivBlockNo,						/**< [In] Diversify block number.*/
            byte[] pTokenAB							/**< [Out] Token to be sent to the card [9].*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_AuthenticateMifare_Part2(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pTokenAB							/**< [In] Token received from the card[5].*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ChangeKeyMifare(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bOption,							/**< [In] Option parameter:
												*		  If flag #PHHAL_HW_SAMAV2_CMD_SAM_CHANGEKEY_MIFARE_RETRIEVE_KEY is set the command returns a single MIFARE key
												*		  for transactions in host system. Otherwhise a cryptogram for key change on a MIFARE 1k or 4k card is generated.
												*		  Key diversification can be enabled by setting flag #PHHAL_HW_SAMAV2_CMD_SAM_CHANGEKEY_MIFARE_DIVERSIFY_KEY_A and/or
												*		  flag #PHHAL_HW_SAMAV2_CMD_SAM_CHANGEKEY_MIFARE_DIVERSIFY_KEY_B. In case of key retrieval for transactions in host system
												*		  the encryption with the actual session key can be enabled by setting flag #PHHAL_HW_SAMAV2_CMD_SAM_CHANGEKEY_MIFARE_ENCRYPT. */
            byte bKeyNoM,							/**< [In] Reference number of MIFARE key (00h to 7Fh). */
            byte bKeyVM_KeyVMA,						/**< [In] Key version of MIFARE key in case of key retrieval or key version of MIFARE key A in case of key change. */
            byte bKeyAorB_KeyVMB,					/**< [In] Key 0Ah or 0Bh in case of key retrieval or key version of MIFARE key B in case of key change.. */
            byte[] pAccessConditions,				/**< [In] Access conditions to be stored within MIFARE card. Only used for key change. This buffer has to be 4 bytes long. */
            byte[] pUid,								/**< [In] Uid of the card. Only used if diversification is needed. This buffer has to be 4 bytes long. */
            byte bDivBlockNo,						/**< [In] Block number to be used for diversification. */
            byte[] pKeyData,							/**< [Out] Buffer containing the key data. The buffer has to be 16 bytes long in case of key retrieval and 21 bytes long in case of key change. */
            ref byte pKeyDataLength					/**< [Out] Amount of valid data bytes in pKeyData. */
            );

        #endregion

        #region NON X MODE: DESFIRE ULC

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_AuthenticatePICC_Part1(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bAuthMode,							/**< [In] Authentication mode used */
            byte bKeyNo,							/**< [In] DESFire key number or Key Reference Number (depends on SAm_SelectApplication done or not)*/
            byte bKeyV,							/**< [In] the key version of Key No */
            byte[] pEncRndB,							/**< [In] Encoded Random B (received from DESFire PICC) */
            byte bEncRndBLength,					/**< [In] Encoded Random B length */
            byte[] pDivInput,							/**< [In] DivInp data */
            byte bDivInpLength,						/**< [In] DivInp data length */
            byte[] pEncRndAB,						/**< [Out] Buffer containing the encoded random numbers to be sent to the PICC. This buffer has to be 32 bytes long. */
            ref byte pEncRndABLength					/**< [Out] Amount of valid data bytes in encoded pEncRndAB. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_AuthenticatePICC_Part2(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte[] pEncRndA,							/**< [In] Encoded random A (received from PICC). */
            byte bEncRndALength						/**< [In] Encoded random A length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_IsoAuthenticatePICC_Part1(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bAuthMode,							/**< [In] Authentication mode used */
            byte bKeyNo,							/**< [In] DESFire key number or Key Reference Number (depends on SAm_SelectApplication done or not)*/
            byte bKeyV,							/**< [In] the key version of Key No */
            byte[] pRndCard,						/**< [In] Random Number Card (8 bytes for (3)DES or 16 bytes for 3K3DES and AES) */
            byte bRndCardLength,					/**< [In] Random Number Card length */
            byte[] pDivInput,							/**< [In] DivInp data */
            byte bDivInputLength,						/**< [In] DivInp data length */
            byte[] pEncData,
            ref byte pEncDataLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_IsoAuthenticatePICC_Part2(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte[] pRnd,							/**< [In] Encoded random numbers received from card. */
            byte bRndLength						/**< [In] Encoded random numbers length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_SAM_ChangeKeyPICC(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bKeyCompMeth,						/**< [In] bit mask, determining the key compilation method. */
            byte bCfg,								/**< [In] Bit mask defining the change key configuration. */
            byte bKeyNoCurrent,						/**< [In] Current Key Number. */
            byte bKeyVCurrent,						/**< [In] Current Key Version. */
            byte bKeyNoNew,							/**< [In] New Key Number. */
            byte bKeyVNew,							/**< [In] New Key Version. */
            byte[] pDivInput,						/**< [In] Diversification input data. */
            byte bDivInputLength,					/**< [In] Diversification input data length. No diversification in case of 00h. */
            byte[] pEncStream,						/**< [Out] Buffer containing the stream to be sent to the PICC. The buffer has to be 32 bytes long. */
            ref byte pEncStreamLength				/**< [Out] Amount of valid data bytes in pEncStream. */
            );

        #endregion

        #region X MODE: MFRC52

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_RC_ReadRegister(
                                        ref DataParams_t pDataParams,               /**< [In] Pointer to this layers parameter structure. */
                                        byte[] RegAddress,                          /**< [In] Address of the registers to be read*/
                                        byte RegAddressLength,                            /**< [In] Amount of addresses to be read*/
                                        byte[] Value                                /**< [Out] Contents of the registers*/
                                        );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_RC_WriteRegister(
                                        ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
                                        byte[] pData,                             /**< [In] data containing Address0 || Data0 || Address1 || Data1 ...*/
                                        byte bDataLength                         /**< [In] Length of pData*/
                                        );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_RC_RFControl(
                                        ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
                                        ushort wTime                              /**< [In] Shut down time for HF Field*/
                                        );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_RC_Init(
            ref DataParams_t pDataParams,   	/**< [In] Pointer to this layers parameter structure. */
            byte bLoadReg
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_RC_LoadRegisterValueSet(
                                        ref DataParams_t pDataParams,  	        /**< [In] Pointer to this layers parameter structure. */
                                        byte bStoreReg,							/**< [In] Amount of register value set of the SAM*/
                                        byte[] pData,                           /**< [In] Buffer containing address - value duos. */
                                        byte bDataLength                        /**< [In] Amount of valid data bytes in the pData buffer. It has to be a multiple of 2. */
                                        );

        #endregion

        #region ISO3A LAYER

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_3_RequestA_Wakeup(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bReqaWakeupCode,					/**< [In] bReqaWakeupCode defining the code to be sent (either REQA or WUPA) */
            byte[] pAtqa								/**< [Out] AtqA; byte[2] */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_3_AnticollisionSelect(
            ref DataParams_t pDataParams,                /**< [In] Pointer to this layers parameter structure. */
            byte[] pSelCodes,							 /**< [In] Buffer containing the SEL sequence bytes.  */
            byte bSelCodesLength,
            ref byte pSak,								 /**< [Out] Pointer to the 1 byte Select Acknowledge reveived from card. */
            byte[] pUid,								 /**< [Out] Buffer containing the received UID. This buffer has to be 10 bytes long. */
            ref byte pUidLength                         /**< [Out] Amount of valid bytes in UID buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_3_ActivateIdle(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte bNumCards,                     /**< [In/Out] Number of cards to activate */
            ushort wTime,                       /**< [In] Time to wait for a card */
            byte[] pAtqaIn,                     /**< [In] AtqA, mask and value; byte[4] */
            byte[] pSakIn,                      /**< [In] Sak, mask and value; byte[2] */
            ref IntPtr ppRxBuffer,  			/**< [Out] Pointer to the buffer containing the received data. */
            ref ushort pRxLength				/**< [Out] Amount of valid bytes in ppRxBuffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_3_ActivateWakeUp(
            ref DataParams_t pDataParams,  	            /**< [In] Pointer to this layers parameter structure */
            byte[] pUid,								/**< [In] UID of the card to reactivate. */
            byte bUidLength 							/**< [In] Length of UID (4, 7 or 10 bytes). */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_3_HaltA(
            ref DataParams_t m_pDataParams  /**< [In] Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_3_TransparentExchange(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            byte[] pTxBuffer,						/**< [In] Buffer containing the data to be sent. */
            byte bTxLength,							/**< [In] Number of bytes to be sent to the card. */
            byte bTxBitLength,						/**< [In] Number of bits to be sent to the card in last byte (00h - 07h). */
            ref IntPtr ppRxBuffer,						/**< [Out] Pointer to the buffer containing the received data. */
            ref ushort pRxLength,						/**< [Out] Amount of valid bytes in ppRxBuffer. */
            ref byte pRxBitLength						/**< [Out] Amount of valid bits in the last byte of the received data (00h - 07h). */
            );

        #endregion

        #region ISO4A LAYER

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_4_RATS_PPS(
            ref DataParams_t pDataParams,     /**< [In] Pointer to this layers parameter structure */
            byte bCidIn,
            byte bDsiIn,
            byte bDriIn,
            ref byte pCidOut,
            ref byte pDsiOut,
            ref byte pDriOut,
            byte[] pAts
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_4_Init(
            ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure */
            byte bCid,								  /**< [In] CID to apply. */
            byte bDri,								  /**< [In] Bit rate PCD -> PICC. */
            byte bDsi,								  /**< [In] Bit rate PICC -> PCD */
            byte bFwi,								  /**< [In] Frame waiting time indicator. */
            byte bFsci								  /**< [In] Frame size card indicator. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_4_Exchange(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure */
            ushort wOption,                       /**< [In] Option parameter for TxChaining.
                                                          Can be #PHHAL_HW_SAMAV2_ISO7816_LAST_FRAME or #PHHAL_HW_SAMAV2_ISO7816_CHAINED_FRAME. */
            byte[] pAppDataIn,                  /**< [In] Application Data to exchange; byte[] */
            byte bLenAppData,                   /**< [In] Length of known Data; byte */
            ref IntPtr ppAppDataOut,            /**< [Out] Data returned by the PICC; byte[] */
            ref ushort pAppDataOutLength        /**< [Out] Length of Data returned by the PICC; byte  */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_4_PresenceCheck(
            ref DataParams_t m_pDataParams    /**< [In] Pointer to this layers parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_4_Deselect(
            ref DataParams_t pDataParams,	/**< [In] Pointer to this layers parameter structure */
            byte bFreeCID								/**< [In] FALSE: CID will not be freed if deselect fails */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_X_ISO14443_4_FreeCid(
            ref DataParams_t pDataParams,	/**< [In] Pointer to this layers parameter structure */
            byte[] pCid,								/**< [In] known Uid, can be NULL if bLenUidIn is 0; byte[4/7/10] */
            byte bCidLength							/**< [In] Number of CIDs to free */
            );

        #endregion

        #region X MODE: MIFARE CLASSIC

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfAuthenticate(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte[] pUid,								/**< [In] MIFARE standard UID*/
            byte bKeyNoM,							/**< [In] Key Reference Number of MIFARE key. */
            byte bKeyVM,								/**< [In] Key Reference Version of MIFARE key. */
            byte bKeyType,							/**< [In] Type of Key : MF Key A (0x0A) or MF Key B (0xB)*/
            byte bAuthBlockNo,							/**< [In] Block number to be authenticated within MIFARE PICC */
            byte bDivBlockNo					/**< [In] Block number to be used for Diversification */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfRead(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte bOption,							/**< [In] Option mask for this command. \n
												*		  If flag #PHHAL_HW_SAMAV2_CMD_CHANGE_MFREAD_MULTIBLOCK is set, multi block read is used. */
            byte[] pBlocks,							/**< [In] Pointer to the block numbers to be read. \n
                                                *         In case of single block read -> BNR_1 | BNR_2 | BNR_3 ... \n
                                                *         In case of multi block read -> BNR_1 | NumBlocks_1 | BNR_2 | NumBlocks_2 | BNR_3 | NumBlocks_3 ... */
            byte bBlocksLength,						/**< [In] Length of the pBlocks buffer. */
            ref IntPtr ppData,						    /**< [Out] Pointer to buffer containing the data read from the PICC. */
            ref ushort pDataLength						/**< [Out] Amount of valid bytes in the ppData buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfWrite(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                           /**< [In] Exchange option for buffered transmission. */
            byte[] pData,
            byte bDataLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfValueWrite(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pBlocks,							/**< [In] Pointer to the block numbers and data to be written. bNr || value[4] || address[4] ...*/
            byte bBlockLength						/**< [In] Length of the pBlocks buffer.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfIncrement(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pBlocks,							/**< [In] Pointer to the block numbers and data to be written. srcBnr || destBnr|| value[4] ||  ...*/
            byte bBlockLength						/**< [In] Length of the pBlocks buffer.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfDecrement(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pBlocks,							/**< [In] Pointer to the block numbers and data to be written. srcBnr || destBnr|| value[4] ||  ...*/
            byte bBlockLength						/**< [In] Length of the pBlocks buffer.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfRestore(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pBlocks,							/**< [In] Pointer to the block numbers of source and destination. srcBnr || destBnr||   ...*/
            byte bBlockLength						/**< [In] Length of the pBlocks buffer.*/
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfAuthenticateRead(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            ushort wOption,							/**< [In] Option mask for this command. \n */
            byte[] pUid,								/**< [In] MIFARE standard UID. This buffer has to be 4 bytes long. This buffer is only used if wOption is set to #PH_EXCHANGE_BUFFER_FIRST. */
            byte bCmdSettings,						/**< [In] Command settings of the data cycle. */
            byte bKeyNoM,							/**< [In] Key reference number of MIFARE key. */
            byte bKeyVM,								/**< [In] Key version of MIFARE key. */
            byte bKeyType,							/**< [In] Type of key : MF key A (0x0A) or MF key B (0xB). */
            byte bAuthBlockNo,						/**< [In] Block number to be authenticated within MIFARE PICC. */
            byte bDivBlockNo,   					/**< [In] Block number to be used for diversification. */
            byte[] pBlocks,							/**< [In] Pointer to the block numbers to be read. */
            byte bBlocksLength,						/**< [In] Length of the pBlocks buffer. */
            ref IntPtr ppData,						    /**< [Out] Pointer to buffer containing the data read from the PICC. */
            ref ushort pDataLength						/**< [Out] Amount of valid bytes in the ppData buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfAuthenticateWrite(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            ushort wOption,							/**< [In] Option mask for this command. \n */
            byte[] pUid,								/**< [In] MIFARE standard UID. This buffer has to be 4 bytes long. This buffer is only used if wOption is set to #PH_EXCHANGE_BUFFER_FIRST. */
            byte bCmdSettings,						/**< [In] Command settings of the data cycle. */
            byte bKeyNoM,							/**< [In] Key reference number of MIFARE key. */
            byte bKeyVM,								/**< [In] Key version of MIFARE key. */
            byte bKeyType,							/**< [In] Type of key : MF key A (0x0A) or MF key B (0xB). */
            byte bAuthBlockNo,						/**< [In] Block number to be authenticated within MIFARE PICC. */
            byte bDivBlockNo,   					/**< [In] Block number to be used for diversification. */
            byte[] pBlocks,							/**< [In] Pointer to the block numbers to be read. */
            byte bBlocksLength						/**< [In] Length of the pBlocks buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfChangeKey(
            ref DataParams_t pDataParams,  	        /**< [In] Pointer to this layers parameter structure. */
            byte bKeyCompMeth,						/**< [In] bit mask, determining the key compilation method. */
            byte bKeyNoM,							/**< [In] Key Reference Number of MIFARE key. */
            byte bKeyVMA,							/**< [In] Key Reference Version of MIFARE key A. */
            byte bKeyVMB,							/**< [In] Key Reference Version of MIFARE key B. */
            byte bBlockNo,						    /**< [In] MIFARE block number (block to store the new key). */
            byte[] pAccessConditions,				/**< [In] Access Conditions to be stored (4 bytes). */
            byte[] pUid,							/**< [In] UID standard (4 bytes). */
            byte bDivBlockNo					    /**< [In] Block number to be used for Diversification */
            );

        #endregion

        #region MIFAREPLUS_X
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpWritePerso(
            ref DataParams_t pDataParams,  	/**< [In] Pointer to this layers parameter structure. */
            byte[] pValues,						/**< Values containing from Block ADDR ||DATA ||Block addr || DATA ..... */
            byte bValuesLength,
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpAuthenticate(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bOption,							/**< [In] Option for Authenticate MFP: \n
												*		  If flag #PHHAL_HW_SAMAV2_CMD_MFP_AUTHENTICATE_AUTH_FIRST a first authentication is done.
												*		  The key derivation function can be selected by setting the flags to #PHHAL_HW_SAMAV2_CMD_MFP_AUTHENTICATE_NO_KDF,
												*		  #PHHAL_HW_SAMAV2_CMD_MFP_AUTHENTICATE_SL2_KDF or #PHHAL_HW_SAMAV2_CMD_MFP_AUTHENTICATE_SL3_KDF. */
            byte bKeyNo,								/**< [In] Key reference number to be used in authentication. */
            byte bKeyV,								/**< [In] Key version to be used in authentication. */
            ushort wBlockNo,                          /**< [In] Block Number to be sent to the card. */
            byte[] pPcdCapsIn,						/**< [In] Buffer containing the input PcdCaps. It has to be bDivInputLength bytes long. */
            byte bPcdCapsInLength,					/**< [In] Input PcdCaps length. */
            byte[] pDivInput,						/**< [In] Diversification data. This buffer has to be bDivInputLength bytes long. */
            byte bDivInputLength,					/**< [In] Diversification data length. If set to 00h no diversification is used. */
            byte[] pPcdCapsOut,						/**< [Out] Buffer containing the PcdCaps Out. It has to be 3 bytes long. */
            byte[] pPdCaps,							/**< [Out] Buffer containing the PdCaps Out. It has to be 3 bytes long. */
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpCombinedRead(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                           /**< [In] Implements buffering in RX direction. */
            byte[] pPlainCommand,                     /**< [In] Plain command to be sent to the card */
            byte bPlainCommandLength,                /**< [In] Length of Plain Command */
            ref IntPtr pData,                           /**< [Out] Out Data Stream */
            ref ushort pDataLength,                      /**< [Out] Length of bData */
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpCombinedWrite(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                           /**< [In] Implements buffering. */
            byte[] pData,                            /**< [In] Input Data Stream */
            byte bDataLength,                        /**< [In] Length of bData */
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpChangeKey(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bCmd,                               /**< [In] Command code for MIFARE Plus Write (WRITE_M or WRITE) */
            ushort wBlockNo,                          /**< [In] Block number of the MIFARE Plus Block */
            byte bKeyNo,                        /**< [In] Key number to be used in authentication */
            byte bKeyV,                       /**< [In] Key version to be used in authentication */
            byte[] pDivInput,                        /**< [In] Pointer to Div Input Data (could be NULL)*/
            byte bDivInputLength,                    /**< [In] Div input length (could be NULL) */
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpProximityCheck(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bOption,
            byte bKeyNo,                        /**< [In] Key number to be used in authentication */
            byte bKeyV,                       /**< [In] Key version to be used in authentication */
            byte bBlCnt,                /**< [In] Specifies the maximum amount of random bytes transmitted at once */
            byte[] pDivInput, 						/**< [In] Diversification input data */
            byte bDivInputLength,					/**< [In] Diversification input length */
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpVirtualCardSupport(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                           /**< [In] Implements the buffering: Note: For BUFFERED_START the parameters are ignored. For BUFFERED_LAST the PCD Caps are used */
            byte[] pData,								/**< [In] Buffer containing the data set field. */
            byte bDataLength,						/**< [In] Length of the data set buffer. */
            byte[] pPcdCapabilities,                 /**< [In] PcdCapabilities sent to the card. This buffer has to be bPcdCapabilitiesLength bytes long. */
            byte bPcdCapabilitiesLength,            /**< [In] Length of PcdCapabilities. */
            ref IntPtr pSupportedCardInfo,              /**< [Out] Pointer to buffer containing the card info of supported cards */
            ref ushort pSupportedCardInfoLength,        /**< [Out] Amount of valid bytes in pSupportedCardInfo. */
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_MfpSelectVirtualCard(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bKeyNo,                        /**< [In] Key number to be used in SVC command*/
            byte bKeyV,                       /**< [In] Key version to be used in SVC command*/
            byte[] pIid,                            /**< [In] desired Iid . */
            byte[] pDivInput,                        /**< [In] Diversification input */
            byte bDivInputLength,                  /**< [In] Length if bDivInput */
            ref byte pPiccReturnCode                   /**< [Out] Return Code sent by the MFP card */
            );
        #endregion

        #region X MODE: DESFIRE ULC

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_DESFireAuthenticatePICC(
            ref DataParams_t pDataParams,    /**< [In] Pointer to this layers parameter structure. */
            byte bAuthMode,							/**< [In] Authentication mode to be used. */
            byte bISOMode,							/**< [In] ISO mode to be used. */
            byte bDFKeyNo,							/**< [In] DESFire key number. */
            byte bKeyNo,								/**< [In] Key reference number. */
            byte bKeyV,								/**< [In] Key version. */
            byte[] pDivInput,                        /**< [In] Diversification data. This buffer has to be bDivInputLength bytes long. */
            byte bDivInputLength                    /**< [In] Diversification data length. If set to 00h no diversification is used. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_DESFireChangeKeyPICC(
            ref DataParams_t pDataParams,           /**< [In] Pointer to this layers parameter structure. */
            byte bKeyCompMeth,						/**< [In] bit mask, determining the key compilation method. */
            byte bCfg,								/**< [In] Bit mask defining the change key configuration. */
            byte bKeyNoCurrent,						/**< [In] Current Key Number. */
            byte bKeyVCurrent,						/**< [In] Current Key Version. */
            byte bKeyNoNew,							/**< [In] New Key Number. */
            byte bKeyVNew,							/**< [In] New Key Version. */
            byte[] pDivInput,		                /**< [In] DivInp data */
            byte bDivInputLength	                /**< [In] DivInp data length */
            );


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_DESFireWriteX(
            ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                           /**< [In] Implements buffering. */
            byte bCrypto,							/**< [In] Crypto mode (encrypted, MACed). */
            byte[] pData,							/**< [In] Data to be written. */
            byte bDataLength							/**< [In] Data length. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_DESFireReadX(
            ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure. */
            ushort wOption,                           /**< [In] Implements buffering. */
            byte bCrypto,							/**< [In] Crypto mode (encrypted, MACed). */
            byte[] pData,							/**< [In] Data to send to DESFire. */
            byte bDataLength,							/**< [In] Amount of data to send to DESFire. */
            ref IntPtr ppRxBuffer,  					/**< [Out] Pointer to the received data. */
            ref ushort pRxLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_ULCAuthenticatePICC(
           ref DataParams_t pDataParams,      /**< [In] Pointer to this layers parameter structure. */
           byte bKeyNo,						/**< [In] the key reference number for SAM KeyStorage */
           byte bKeyV,						/**< [In] the key reference version for SAM KeyStorage */
           byte[] pDivInput,		                /**< [In] DivInp data */
           byte bDivInputLength	                /**< [In] DivInp data length */
           );

        #endregion

        #region 7816_EXCHANGE

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_SamAV2_Cmd_7816Exchange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] input data */
            ushort wTxLength,       /**< [In] length of input data */
            ref IntPtr ppRxBuffer,  /**< [Out] Pointer to receive buffer */
            ref ushort pRxLength    /**< [Out] number of output data bytes */
            );

        #endregion

        #endregion

        #region INIT

        private byte[] m_bHalBuffer;
        private GCHandle m_pTxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(
            Bal.Generic pBal,
            Hal.Generic pReaderHal,
            KeyStore.Generic pKeyStore,
            CryptoSym.Generic pENCCrypto,
            CryptoSym.Generic pMACCrypto,
            CryptoRng.Generic pCryptoRng,
            OpMode Mode,
            LogicalChannel Lc

            )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;

            if (pBal != null)
            {
                pBalPtr = pBal.m_pDataParams;
            }
            if (pReaderHal != null)
            {
                pReaderHalPtr = pReaderHal.m_pDataParams;
            }
            if (pKeyStore != null)
            {
                pKeyStorePtr = pKeyStore.m_pDataParams;
            }
            if (pENCCrypto != null)
            {
                pENCCryptoPtr = pENCCrypto.m_pDataParams;
            }
            if (pMACCrypto != null)
            {
                pMACCryptoPtr = pMACCrypto.m_pDataParams;
            }
            if (pCryptoRng != null)
            {
                pCryptoRngPtr = pCryptoRng.m_pDataParams;
            }

            // Free Buffer
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }

            // Use only one Buffer with fixed length
            m_bHalBuffer = new byte[261];
            m_pTxBuffer = GCHandle.Alloc(m_bHalBuffer, GCHandleType.Pinned);

            return phhalHw_SamAV2_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBalPtr,
                pReaderHalPtr,
                pKeyStorePtr,
                pENCCryptoPtr,
                pMACCryptoPtr,
                pCryptoRngPtr,
                (byte)Mode,
                (byte)Lc,
                m_pTxBuffer.AddrOfPinnedObject(),
                261,
                m_pTxBuffer.AddrOfPinnedObject(),
                261);
        }

#if DEBUG
        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(
            int wDataParamSize,
            Bal.Generic pBal,
            Hal.Generic pReaderHal,
            KeyStore.Generic pKeyStore,
            CryptoSym.Generic pENCCrypto,
            CryptoSym.Generic pMACCrypto,
            CryptoRng.Generic pCryptoRng,
            OpMode Mode,
            LogicalChannel Lc,
            int wTxBufferSize,
            byte[] bTxBuffer,
            int wRxBufferSize,
            byte[] bRxBuffer
            )
        {
            IntPtr pBalPtr = IntPtr.Zero;
            IntPtr pReaderHalPtr = IntPtr.Zero;
            IntPtr pKeyStorePtr = IntPtr.Zero;
            IntPtr pENCCryptoPtr = IntPtr.Zero;
            IntPtr pMACCryptoPtr = IntPtr.Zero;
            IntPtr pCryptoRngPtr = IntPtr.Zero;

            if (pBal != null)
            {
                pBalPtr = pBal.m_pDataParams;
            }
            if (pReaderHal != null)
            {
                pReaderHalPtr = pReaderHal.m_pDataParams;
            }
            if (pKeyStore != null)
            {
                pKeyStorePtr = pKeyStore.m_pDataParams;
            }
            if (pENCCrypto != null)
            {
                pENCCryptoPtr = pENCCrypto.m_pDataParams;
            }
            if (pMACCrypto != null)
            {
                pMACCryptoPtr = pMACCrypto.m_pDataParams;
            }
            if (pCryptoRng != null)
            {
                pCryptoRngPtr = pCryptoRng.m_pDataParams;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(bRxBuffer, GCHandleType.Pinned);

            return phhalHw_SamAV2_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBalPtr,
                pReaderHalPtr,
                pKeyStorePtr,
                pENCCryptoPtr,
                pMACCryptoPtr,
                pCryptoRngPtr,
                (byte)Mode,
                (byte)Lc,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#endif

        public Status_t DetectMode()
        {
            return phhalHw_SamAV2_DetectMode(ref m_DataParamsInt[0]);
        }

        public Status_t SetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            int wValue          /**< [In] Configuration Value */
            )
        {
            return SetConfig((Hal.Config)wConfig, wValue);
        }

        public Status_t GetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            return GetConfig((Hal.Config)wConfig, out pValue);
        }

        #endregion

        #region SAM SECURITY

        public Status_t Cmd_SAM_DisableCrypto(
                                DisableCrypto_ProMas ProMas	           /**< [In] Two byte mask to specify the desired settings for cryptography-related features. */
                                )
        {
            return phhalHw_SamAV2_Cmd_SAM_DisableCrypto(
                ref m_DataParamsInt[0],
                (ushort)ProMas);
        }

        public Status_t Cmd_SAM_LockUnlock(
                                LockUnlock_Mode Mode,                 /**< [In] Mode for lock/unlock.*/
                                int RdKeyNo,
                                int RdKeyV,
                                byte SamKeyNo,                        /**< [In] Key Number to be used.*/
                                byte SamKeyV,                       /**< [In] Key Version to be used.*/
                                byte UnlockKeyNo,                     /**< [In] Unlock Key Number to be used.*/
                                byte UnlockKeyV,                      /**< [In] Unlock Key Version to be used.*/
                                long MaxChainBlocks                   /**< [In] Maximum amount of data to be chained. Max value = 0xFFFFFFFF. */
                                )
        {
            return phhalHw_SamAV2_Cmd_SAM_LockUnlock(
                ref m_DataParamsInt[0],
                (byte)Mode,
                (ushort)RdKeyNo,
                (ushort)RdKeyV,
                SamKeyNo,
                SamKeyV,
                UnlockKeyNo,
                UnlockKeyV,
                (uint)MaxChainBlocks);
        }

        public Status_t Cmd_SAM_AuthenticateHost(
                                AuthenticateHost_AuthMode AuthMode, 	/**< [In] the type of Authentication to do*/
                                int RdKeyNo,						/**< [In] the key reference number for readerLib KeyStorage */
                                int RdKeyV,						/**< [In] the key reference version for readerLib KeyStorage */
                                byte SamKeyNo,   					/**< [In] the key reference number for SAM KeyStorage */
                                byte SamKeyV,   					/**< [In] the key reference version for SAM KeyStorage */
                                byte[] DivInput							/**< [In] DivInp data */
                                )
        {
            return phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(
                ref m_DataParamsInt[0],
                (byte)AuthMode,
                (ushort)RdKeyNo,
                (ushort)RdKeyV,
                SamKeyNo,
                SamKeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length));
        }

        public Status_t Cmd_SAM_AuthenticateHost(
                                AuthenticateHost_HostMode HostMode, 	/**< [In] the type of Authentication to do*/
                                int RdKeyNo,						/**< [In] the key reference number for readerLib KeyStorage */
                                int RdKeyV,						/**< [In] the key reference version for readerLib KeyStorage */
                                byte SamKeyNo,   					/**< [In] the key reference number for SAM KeyStorage */
                                byte SamKeyV,   					/**< [In] the key reference version for SAM KeyStorage */
                                byte[] DivInput							/**< [In] DivInp data */
                                )
        {
            return phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(
                ref m_DataParamsInt[0],
                (byte)HostMode,
                (ushort)RdKeyNo,
                (ushort)RdKeyV,
                SamKeyNo,
                SamKeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length));
        }

        public Status_t Cmd_SAM_ActivateOfflineKey(
                                byte KeyNo,						/**< [In] the key reference number */
                                byte KeyV,						/**< [In] the key reference version */
                                byte[] DivInput							/**< [In] Diversification input. If set to null no diversification is used. */
                                )
        {
            return phhalHw_SamAV2_Cmd_SAM_ActivateOfflineKey(
                ref m_DataParamsInt[0],
                KeyNo,
                KeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length));
        }

        public Status_t Cmd_SAM_SelectApplication(
                                byte[] DF_Aid							/**< [In] DF_AID data */
                                )
        {
            if (DF_Aid != null)
            {
                if (DF_Aid.Length == 3)
                {
                    return phhalHw_SamAV2_Cmd_SAM_SelectApplication(
                        ref m_DataParamsInt[0],
                        DF_Aid);
                }
            }
            return new Status_t(Error_CompCode.HAL, Error_Param.INVALID_DATA_PARAMS);
        }

        public Status_t Cmd_SAM_LoadInitVector(
                                byte[] InitVector						/**< [In] InitVector data */
                                )
        {
            return phhalHw_SamAV2_Cmd_SAM_LoadInitVector(ref m_DataParamsInt[0], InitVector, (byte)((InitVector == null) ? 0 : InitVector.Length));
        }

        public Status_t Cmd_SAM_KillAuthentication(
            KillAuthentication_Option Option  /**< [In] Only valid in SamAV2 mode. If set to 01, only the PICC authentication is killed. */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_KillAuthentication(
                ref m_DataParamsInt[0],
                (byte)Option);
        }


        public Status_t Cmd_SAM_GetChallenge(
            byte ExpL,       /**< [In] Amount of data bytes to be retrieved. */
            out byte[] Rnd     /**< [Out] Pointer containing the challenge */
            )
        {
            Rnd = new byte[ExpL];
            return phhalHw_SamAV2_Cmd_SAM_GetChallenge(
                ref m_DataParamsInt[0],
                ExpL, Rnd);
        }

        public Status_t Cmd_SAM_IsoExternalAuthenticate(
            byte KeyNo,          /**< [In] the key reference number */
            byte KeyV,          /**< [In] the key reference version */
            byte[] EncRnd       /**< [In] Encrypted (Random number PCD1 || Random number PICC1) */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_IsoExternalAuthenticate(
                ref m_DataParamsInt[0],
                KeyNo,
                KeyV,
                EncRnd,
                (byte)((EncRnd == null) ? 0 : EncRnd.Length));
        }

        public Status_t Cmd_SAM_IsoInternalAuthenticate(
            byte[] RPcd2,           /**< [In] Random number PCD2. */
            out byte[] pEncResp     /**< [Out] Encrypted (Random number PICC2 || Random number PCD2). */
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;

            status = phhalHw_SamAV2_Cmd_SAM_IsoInternalAuthenticate(
                ref m_DataParamsInt[0],
                RPcd2,
                (byte)((RPcd2 == null) ? 0 : RPcd2.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            pEncResp = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_GetVersion(
                                out byte[] Version           				/**< [Out] Pointer to the version length of the data to be read out.*/
                                )
        {
            Status_t status;
            Version = new byte[31]; // Max Version Lenght is 31 bytes
            byte VersionLength = 0;

            status = phhalHw_SamAV2_Cmd_SAM_GetVersion(ref m_DataParamsInt[0], Version, ref VersionLength);

            /* Resize pVersion buffer to actual bVersion size */
            //if (status.Equals(Error_Gen.SUCCESS))
            //{
            Array.Resize<byte>(ref Version, (int)VersionLength);
            //}
            return status;
        }

        public Status_t Cmd_SAM_Sleep()
        {
            return phhalHw_SamAV2_Cmd_SAM_Sleep(ref m_DataParamsInt[0]);
        }

        #endregion

        #region SAM KEY HANDLING

        public Status_t Cmd_SAM_ChangeKeyEntry(
            ChangeKeyEntry_Option Option,
            byte KeyNo,                             /**< [In] number of the key entry to be changed (00h to 7Fh)*/
            ChangeKeyEntry_ProMas ProMas,           /**< [In] Indicates which KeyEntry parameters should be updated */
            byte[] KeyData                          /**< [In] Key entry structure array containing Key Entry. */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_ChangeKeyEntry(
                ref m_DataParamsInt[0],
                (byte)Option,
                KeyNo,
                (byte)ProMas,
                KeyData,
                (byte)((KeyData == null) ? 0 : KeyData.Length));
        }

        public Status_t Cmd_SAM_GetKeyEntry(
            byte KeyNo,            /**< [In] number of the key entry to be returned (00h to 7Fh)*/
            out byte[] KeyEntry        /**< [Out] Key entry structure pointer containing Key Entry*/
            )
        {
            Status_t status;
            byte KeyEntryLength = 0;

            KeyEntry = new byte[13]; // Max key information length is 13 bytes

            status = phhalHw_SamAV2_Cmd_SAM_GetKeyEntry(ref m_DataParamsInt[0], KeyNo, KeyEntry, ref KeyEntryLength);

            /* Resize Key Entry  buffer to actual size */
            //if (status.Equals(Error_Gen.SUCCESS))
            //{
            Array.Resize<byte>(ref KeyEntry, (int)KeyEntryLength);
            //}
            return status;
        }

        public Status_t Cmd_SAM_ChangeKUCEntry(
            ChangeKUCEntry_Option Option,
            byte KucNo,                     /**< [In] Number of the KUC entry to be returned (00h to 7Fh)*/
            ChangeKUCEntry_ProMas ProMas,		/**< [In] ProMas byte indicating KUC field that should be taken into account*/
            byte[] KucData                      /**< [In] Encrypted KUC data. */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_ChangeKUCEntry(
                ref m_DataParamsInt[0],
                (byte)Option,
                KucNo,
                (byte)ProMas,
                KucData,
                (byte)((KucData == null) ? 0 : KucData.Length));
        }


        public Status_t Cmd_SAM_GetKUCEntry(
            byte KucNo,                /**< [In] number of the key usage counter to be looked at (00h to 0Fh)*/
            out byte[] KucEntry            /**< [In] Reference number of the key usage counter to be returned (00h to 0Fh). */
            )
        {
            KucEntry = new byte[10];
            return phhalHw_SamAV2_Cmd_SAM_GetKUCEntry(ref m_DataParamsInt[0], KucNo, KucEntry);
        }

        public Status_t Cmd_SAM_DumpSecretKey(
            DumpSecretKey_Crypto Crypto,      		/**< [In] Bit mask containing the crypto options for the key dump. */
            byte KeyNo,								/**< [In] Reference number of the key entry to be dumped. */
            byte KeyV,								/**< [In] Reference number of the key entry to be dumped. */
            byte[] DivInput,						/**< [In] Diversification input data. */
            out byte[] SecretKey
        )
        {
            Status_t status;
            SecretKey = new byte[24]; // Max SecretKey Lenght is 24 bytes
            byte SecretKeyLength = 0;

            status = phhalHw_SamAV2_Cmd_SAM_DumpSecretKey(
                ref m_DataParamsInt[0],
                (byte)Crypto,
                KeyNo,
                KeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                SecretKey,
                ref SecretKeyLength);

            /* Resize SecretKey buffer to actual SecretKey size */
            if (status.Equals(Error_Gen.SUCCESS))
            {
                Array.Resize<byte>(ref SecretKey, (int)SecretKeyLength);
            }
            return status;
        }

        public Status_t Cmd_SAM_DumpSessionKey(
            SamAV2 EncChannel,
            DumpSessionKey_Crypto Crypto,   /**< [In] Encrypt the key or not */
            out byte[] SessionKey           /**< [Out] Session Key buffer */
            )
        {
            Status_t status;
            SessionKey = new byte[40]; // Max SessionKey Lenght is 24 bytes but when a Mifare Plus SL3 Session key is dumped the length is 32
            byte SessionKeyLength = 0;

            if (EncChannel != null)
            {
                status = phhalHw_SamAV2_Cmd_SAM_DumpSessionKey(
                    ref m_DataParamsInt[0],
                    ref EncChannel.m_DataParamsInt[0],
                    (byte)Crypto,
                    SessionKey,
                    ref SessionKeyLength);
            }
            else
            {
                DataParams_t[] paramEnc = new DataParams_t[1];
                status = phhalHw_SamAV2_Cmd_SAM_DumpSessionKey(
                    ref m_DataParamsInt[0],
                    ref paramEnc[0],
                    (byte)Crypto,
                    SessionKey,
                    ref SessionKeyLength);
            }

            /* Resize SessionKey buffer to actual SessionKey size */
            if (status.Equals(Error_Gen.SUCCESS))
            {
                Array.Resize<byte>(ref SessionKey, (int)SessionKeyLength);
            }
            return status;

        }

        public Status_t Cmd_SAM_DisableKeyEntry(
            byte KeyNo,                        	/**< [In] number of the key entry to be returned (00h to 7Fh)*/
            byte[] OfflineCryptogram                /**< [Out] Buffer containing the cryptogram for offline key deactivation. This parameter is only used in AV2. */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_DisableKeyEntry(
                ref m_DataParamsInt[0],
                KeyNo,
                OfflineCryptogram,
                (byte)((OfflineCryptogram == null) ? 0 : OfflineCryptogram.Length));
        }

        #endregion

        #region SAM DATA PROCESS

        public Status_t Cmd_SAM_VerifyMAC(
            Buffering_Option Option,		/**< [In] Option parameter. */
            VerifyGenerateMAC_Num Num,    	/**< [In] Number of MAC bytes to check in the Plain data buffer */
            byte[] TxBuffer    			/**< [In] Plain data including a MAC to be checked */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_VerifyMAC(
                ref m_DataParamsInt[0],
                (ushort)Option,
                (byte)Num,
                TxBuffer,
                (byte)((TxBuffer == null) ? 0 : TxBuffer.Length));
        }

        public Status_t Cmd_SAM_GenerateMAC(
            Buffering_Option Option,    			/**< [In] Option parameter. */
            VerifyGenerateMAC_Num Num,    	        /**< [In] Number of MAC bytes to check in the Plain data buffer */
            byte[] TxBuffer,    					/**< [In] Plain data to be MACed */
            out byte[] RxBuffer  					/**< [Out] Pointer to Plain MACed data */
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;

            status = phhalHw_SamAV2_Cmd_SAM_GenerateMAC(
                ref m_DataParamsInt[0],
                (ushort)Option,
                (byte)Num,
                TxBuffer,
                (byte)((TxBuffer == null) ? 0 : TxBuffer.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            RxBuffer = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_DecipherData(
            DecipherData_Option Option,       			/**< [In] Option parameter. */
            byte[] TxBuffer,    					/**< [In] data to transmit */
            int LenEncInputData,                    /**< [In] Length of encrypted input data. Max value is 0x00FFFFFF. */
            out byte[] RxBuffer 					/**< [Out] Pointer to received data */
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;

            byte[] InputLength = new byte[3];
            InputLength[0] = (byte)LenEncInputData;
            InputLength[1] = (byte)(LenEncInputData >> 8);
            InputLength[2] = (byte)(LenEncInputData >> 16);

            status = phhalHw_SamAV2_Cmd_SAM_DecipherData(
                ref m_DataParamsInt[0],
                (ushort)Option,
                TxBuffer,
                (byte)((TxBuffer == null) ? 0 : TxBuffer.Length),
                InputLength,
                ref RxBufferInt,
                ref RxLengthInt);

            RxBuffer = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_EncipherData(
            Buffering_Option Option,    			/**< [In] Option parameter. */
            byte[] TxBuffer,    					/**< [In] data to transmit */
            byte Offset,            				/**< [In] Offset indicating first data byte to be encrypted. */
            out byte[] RxBuffer  					/**< [Out] Pointer to received data */
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;

            status = phhalHw_SamAV2_Cmd_SAM_EncipherData(
                ref m_DataParamsInt[0],
                (ushort)Option,
                TxBuffer,
                (byte)((TxBuffer == null) ? 0 : TxBuffer.Length),
                Offset,
                ref RxBufferInt,
                ref RxLengthInt);

            RxBuffer = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_DecipherOfflineData(
            Buffering_Option Option,    			/**< [In] Option parameter. */
            byte[] TxBuffer,    					/**< [In] data to transmit */
            out byte[] RxBuffer  					/**< [Out] Pointer to received data */
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;

            status = phhalHw_SamAV2_Cmd_SAM_DecipherOfflineData(
                ref m_DataParamsInt[0],
                (ushort)Option,
                TxBuffer,
                (byte)((TxBuffer == null) ? 0 : TxBuffer.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            RxBuffer = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_EncipherOfflineData(
            Buffering_Option Option,				/**< [In] Option parameter. */
            byte[] TxBuffer,    					/**< [In] data to transmit */
            out byte[] RxBuffer  	    			/**< [Out] Pointer to received data */
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;

            status = phhalHw_SamAV2_Cmd_SAM_EncipherOfflineData(
                ref m_DataParamsInt[0],
                (ushort)Option,
                TxBuffer,
                (byte)((TxBuffer == null) ? 0 : TxBuffer.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            RxBuffer = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        #endregion

        #region NON X MODE: PKI

        public Status_t Cmd_PKI_GenerateKeyPair(
            PKIGenerateKey_Option Option,
            byte PKI_KeyNo,                 /**< [In] number of the key entry to be returned (00h to 02h)*/
            PKI_KeyEntry.PKI_Set PKI_Set,	/**< [In] PKI_SET bytes representing the configurations settings of the key entry*/
            byte PKI_KeyNoCEK,					/**< [In] Indicates current keyNoCEK of KeyNo */
            byte PKI_KeyVCEK,					/**< [In] Indicates current keyVerCEK of KeyNo */
            byte PKI_RefNoKUC,					/**< [In] Indicates current RefNoKUC */
            int PKI_NLen,				/**< [In] Key length for the private exponent of the RSA key (multiple of 8 and in [32;256]). */
            int PKI_eLen,				/**< [In] Key length for the public exponent of the RSA key(multiple of 4 and in [4;256]). */
            byte[] PKI_e					/**< [In] public exponent of the RSA key. */
            )
        {
            return phhalHw_SamAV2_Cmd_PKI_GenerateKeyPair(
                ref m_DataParamsInt[0],
                (byte)Option,
                PKI_KeyNo,
                (ushort)PKI_Set,
                PKI_KeyNoCEK,
                PKI_KeyVCEK,
                PKI_RefNoKUC,
                (ushort)PKI_NLen,
                (ushort)PKI_eLen,
                PKI_e);
        }

        public Status_t Cmd_PKI_ExportPrivateKey(
            Buffering_Option Option,       				/**< [In] Option parameter. If set to \ref PH_EXCHANGE_RXCHAINING only dummy commands are sent to fetch the remaining key data. */
            byte PKI_KeyNo,                             /**< [In] number of the key entry to be returned (00h to 01h)*/
            out byte[] KeyEntry					/**< [Out] Key entry structure array containing Key Entry*/
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLength = 0;

            status = phhalHw_SamAV2_Cmd_PKI_ExportPrivateKey(
                ref m_DataParamsInt[0],
                (ushort)Option,
                PKI_KeyNo,
                ref RxBufferInt,
                ref RxLength);

            byte[] RxBuffer = MarshalCopy(status, RxBufferInt, RxLength);
            KeyEntry = RxBuffer;

            return status;
        }

        public Status_t Cmd_PKI_ExportPublicKey(
            Buffering_Option Option,         			/**< [In] Option parameter. If set to \ref PH_EXCHANGE_RXCHAINING only dummy commands are sent to fetch the remaining key data. */
            byte PKI_KeyNo,                             /**< [In] number of the key entry to be returned (00h to 02h)*/
            out byte[] KeyEntry				    /**< [Out] Key entry structure array containing Key Entry*/
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLength = 0;

            status = phhalHw_SamAV2_Cmd_PKI_ExportPublicKey(
                ref m_DataParamsInt[0],
                (ushort)Option,
                PKI_KeyNo,
                ref RxBufferInt,
                ref RxLength);

            byte[] RxBuffer = MarshalCopy(status, RxBufferInt, RxLength);
            KeyEntry = RxBuffer;

            return status;
        }

        public Status_t Cmd_PKI_SendSignature(
            out byte[] Signature					    /**< [Out] Signature*/
            )
        {
            Status_t status;
            ushort RxLength;
            IntPtr RxBufferInt = IntPtr.Zero;

            RxLength = 0;
            status = phhalHw_SamAV2_Cmd_PKI_SendSignature(
                ref m_DataParamsInt[0],
                ref RxBufferInt,
                ref RxLength);

            Signature = MarshalCopy(status, RxBufferInt, RxLength);
            return status;
        }

        public Status_t Cmd_PKI_UpdateKeyEntries(
            byte NoOfKeyEntries,				    /**< [In] the number of PKI key entries to update*/
            PKI_HashAlgorithm HashingAlg,           /**< [In] Hashing algorithm selection (00:SHA-1, 01:SHA-224, 11: SHA-256, 10:RFU)*/
            byte PKI_KeyNo_Enc,					/**< [In] number of the key entry to be used for decryption (00h to 01h)*/
            byte PKI_KeyNo_Sign,				    /**< [In] number of the key entry to be used for signature verification (00h to 02h)*/
            byte[] KeyFrame     					/**< [In] RSA encrypted key frame */
            )
        {
            return phhalHw_SamAV2_Cmd_PKI_UpdateKeyEntries(
                ref m_DataParamsInt[0],
                NoOfKeyEntries,
                (byte)HashingAlg,
                PKI_KeyNo_Enc,
                PKI_KeyNo_Sign,
                KeyFrame,
                (ushort)((KeyFrame == null) ? 0 : KeyFrame.Length));
        }

        public Status_t Cmd_PKI_GenerateSignature(
            PKI_HashAlgorithm bHashingAlg,
            byte PKI_KeyNo,                         /**< [In] number of the key entry to be returned (00h to 01h)*/
            byte[] HashMsgData                      /**< [In] Hash message to be signed*/
            )
        {
            byte hashMsgDataLength = 0;
            if (HashMsgData != null)
                hashMsgDataLength = (byte)(HashMsgData.Length);

            return phhalHw_SamAV2_Cmd_PKI_GenerateSignature(
                ref m_DataParamsInt[0],
                (byte)bHashingAlg,
                PKI_KeyNo,
                HashMsgData,
                hashMsgDataLength);
        }

        public Status_t Cmd_PKI_GenerateHash(
            PKI_GenerateHash_Option Option,       		/**< [In] Option parameter. */
            PKI_HashAlgorithm HashingAlg,                    /**< [In] Hashing algorithm selection (00:SHA-1, 01:SHA-224, 11: SHA-256, 10:RFU)*/
            long MsgLength,					    /**< [In] Overall message length. */
            byte[] TxBuffer,    				/**< [In] Message chunk to be hashed. */
            out byte[] Hash  					/**< [Out] Pointer to received data. Data is returned after sending the last message chunk. */
            )
        {
            Status_t status;
            Hash = new byte[32];
            byte HashLength = 0;
            status = phhalHw_SamAV2_Cmd_PKI_GenerateHash(
                ref m_DataParamsInt[0],
                (ushort)Option,
                (byte)HashingAlg,
                (uint)MsgLength,
                TxBuffer,
                (ushort)((TxBuffer == null) ? 0 : TxBuffer.Length),
                Hash,
                ref HashLength);

            Array.Resize<byte>(ref Hash, (int)HashLength);

            return status;
        }

        public Status_t Cmd_PKI_VerifySignature(
            byte PKI_KeyNo,                        /**< [In] Number of the key entry to be used for verification (00h to 02h). */
            PKI_HashAlgorithm HashingAlg,                       /**< [In] Hashing algorithm selection (00:SHA-1, 01:SHA-224, 11: SHA-256, 10:RFU). */
            byte[] HashData,					    /**< [In] Hash data. */
            byte[] Signature				    	/**< [In] RSA digital signature. */
            )
        {
            return phhalHw_SamAV2_Cmd_PKI_VerifySignature(
                ref m_DataParamsInt[0],
                PKI_KeyNo,
                (byte)HashingAlg,
                HashData,
                (byte)((HashData == null) ? 0 : HashData.Length),
                Signature,
                (ushort)((Signature == null) ? 0 : Signature.Length));
        }

        public Status_t Cmd_PKI_ImportKey(
            PKI_ImportKey_Option Option,
            byte PKI_KeyNo,                        /**< [In] Number of the key entry to be imported (00h to 02h). */
            PKI_KeyEntry key
            )
        {
            if (key.N == null || key.e == null)
                return new Status_t(Error_CompCode.HAL, Error_Param.PARAMETER_OVERFLOW);

            ushort length_N = (ushort)((key.N == null) ? 0 : key.N.Length);
            ushort length_e = (ushort)((key.e == null) ? 0 : key.e.Length);
            ushort length_P = (ushort)((key.PKISet.privateKeyIncluded) ? key.P.Length : 0);
            ushort length_Q = (ushort)((key.PKISet.privateKeyIncluded) ? key.Q.Length : 0);
            byte[] DP = new byte[0]; // null;
            byte[] DQ = new byte[0];
            byte[] IPQ = new byte[0];

            if (key.PKISet.privateKeyIncluded)
            {
                if (key.P == null || key.Q == null || key.DP == null || key.DQ == null || key.IPQ == null)
                    return new Status_t(Error_CompCode.HAL, Error_Param.PARAMETER_OVERFLOW);

                if (DP.Length > key.P.Length || DQ.Length > key.Q.Length || IPQ.Length > key.Q.Length)
                    return new Status_t(Error_CompCode.HAL, Error_Param.PARAMETER_OVERFLOW);

                /* padd DP, DQ and IPQ if needed */
                DP = new byte[key.P.Length];
                Array.Copy(key.DP, 0, DP, key.P.Length - key.DP.Length, key.DP.Length);
                DQ = new byte[key.Q.Length];
                Array.Copy(key.DQ, 0, DQ, key.Q.Length - key.DQ.Length, key.DQ.Length);
                IPQ = new byte[key.Q.Length];
                Array.Copy(key.IPQ, 0, IPQ, key.Q.Length - key.IPQ.Length, key.IPQ.Length);
            }

            return phhalHw_SamAV2_Cmd_PKI_ImportKey(
                ref m_DataParamsInt[0],
                (byte)Option,
                PKI_KeyNo,
                (ushort)(int)key.PKISet,
                key.KeyNoCEK,
                key.KeyVCEK,
                key.RefNoKUC,
                length_N,
                length_e,
                length_P,
                length_Q,
                key.N,
                key.e,
                ((key.PKISet.privateKeyIncluded) ? key.P : null),
                ((key.PKISet.privateKeyIncluded) ? key.Q : null),
                DP,
                DQ,
                IPQ);
        }

        #endregion

        #region NON X MODE: MIFARE PLUS

        public Status_t Cmd_SAM_AuthenticateMfp_Part1(
            AuthenticateMFP_Option Option,
            byte KeyNo,                         /**< [In] Key number to be used in authentication*/
            byte KeyV,                        /**< [In] Key version to be used in authentication*/
            byte[] RndB,                            /**< [In] Encrypted RND B received from the card */
            byte[] DivInput,                        /**< [In] Diversification input */
            out byte[] RndARndB                     /**< [Out] Encrypted RNDA|| RND B to be sent to the card */
            )
        {
            Status_t status;
            byte RxLengthInt;
            RndARndB = new byte[40];

            RxLengthInt = 0;

            status = phhalHw_SamAV2_Cmd_SAM_AuthenticateMfp_Part1(
                ref m_DataParamsInt[0],
                (byte)Option,
                KeyNo,
                KeyV,
                RndB,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                RndARndB,
                ref RxLengthInt);

            /* Resize pKeyData buffer to actual size */
            if (status.Equals(Error_Gen.SUCCESS_CHAINING))
            {
                Array.Resize<byte>(ref RndARndB, (int)RxLengthInt);
            }
            return status;
        }

        public Status_t Cmd_SAM_AuthenticateMfp_Part2(
            byte[] EncData,                            /**< [In] Encrypted RND A' including TI */
            out byte[] PdCap,                           /**< [Out] PD Caps received from card */
            out byte[] PcdCap                           /**< [Out] PCD Caps received from card */
            )
        {
            PdCap = new byte[6];
            PcdCap = new byte[6];

            return phhalHw_SamAV2_Cmd_SAM_AuthenticateMfp_Part2(
                ref m_DataParamsInt[0],
                EncData,
                (byte)((EncData == null) ? 0 : EncData.Length),
                PdCap,
                PcdCap);
        }

        public Status_t Cmd_SAM_CombinedReadMfp(
            CombinedReadWriteMfp_Option Option,				/**< [In] Implements buffering. In Combination with TX_CHAINING  manual AF handling can be implemented*/
            byte[] Data,                            /**< [In] Input Data Stream */
            out byte[] Output                       /**< [Out] encrypted command or decrypted response */
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;

            status = phhalHw_SamAV2_Cmd_SAM_CombinedReadMfp(
                ref m_DataParamsInt[0],
                (ushort)Option,
                Data,
                (byte)((Data == null) ? 0 : Data.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            Output = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_CombinedWriteMfp(
            CombinedReadWriteMfp_Option Option,							/**< [In] Implements buffering. */
            byte[] Data,                                        /**< [In] Input Data Stream */
            out byte[] Output                                   /**< [Out] encrypted command or decrypted response */
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;

            status = phhalHw_SamAV2_Cmd_SAM_CombinedWriteMfp(
                ref m_DataParamsInt[0],
                (byte)Option,
                Data,
                (byte)((Data == null) ? 0 : Data.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            Output = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_ChangeKeyMfp_Part1(
            ChangeKeyMfp_Cmd Cmd,					/**< [In] Command used to perform the change */
            int BlockNo,							/**< [In] Block Number */
            byte KeyNo,							/**< [In] Key number to be used */
            byte KeyV,						/**< [In] Key version to be used */
            byte[] DivInput,                        /**< [In] Pointer to Div Input Data (could be NULL)*/
            out byte[] Output						/**< [Out] Output data from SAM */
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;

            status = phhalHw_SamAV2_Cmd_SAM_ChangeKeyMfp_Part1(
                ref m_DataParamsInt[0],
                (byte)Cmd,
                (ushort)BlockNo,
                KeyNo,
                KeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            Output = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_ChangeKeyMfp_Part2(
            byte[] DataIn							/**< [In] Pointer to Input Data (response from PICC )*/
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_ChangeKeyMfp_Part2(
                ref m_DataParamsInt[0],
                DataIn,
                (byte)((DataIn == null) ? 0 : DataIn.Length));
        }

        public Status_t Cmd_SAM_VirtualCardSupportMfp(
            Buffering_Option Option,							/**< [In] Implements buffering. */
            byte[] Data,
            byte[] PcdCapabilities,                             /**< [In] PcdCapabilities sent to the card */
            out byte[] SupportedCardInfo                        /**< [Out] Card Info of supported cards */
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;
            SupportedCardInfo = null;

            status = phhalHw_SamAV2_Cmd_SAM_VirtualCardSupportMfp(
                ref m_DataParamsInt[0],
                (ushort)Option,
                Data,
                (byte)((Data == null) ? 0 : Data.Length),
                PcdCapabilities,
                (byte)((PcdCapabilities == null) ? 0 : PcdCapabilities.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            SupportedCardInfo = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_SAM_SelectVirtualCardMfp(
            byte KeyNo,                        /**< [In] Key number to be used in SVC command*/
            byte KeyV,                       /**< [In] Key version to be used in SVC command*/
            byte[] RndQ,                            /**< [In] RNDQ sent to the card . */
            byte[] DivInput,                        /**< [In] Diversification input */
            out byte[] Mac                              /**< [Out] Mac to be sent to the card */
            )
        {
            Mac = new byte[8];
            return phhalHw_SamAV2_Cmd_SAM_SelectVirtualCardMfp(
                ref m_DataParamsInt[0],
                KeyNo,
                KeyV,
                RndQ,
                DivInput, (byte)((DivInput == null) ? 0 : DivInput.Length),
                Mac);
        }

        public Status_t Cmd_SAM_ProximityCheckMfp_Part1(
            byte KeyNo,							/**< [In] Key number to be used in authentication */
            byte KeyV,						/**< [In] Key version to be used in authentication */
            byte Ppse,								/**< [In] PPSE byte used. */
            byte[] RndRC,							/**< [In] RND number in correct format for MAC verification */
            byte[] DivInput,						/**< [In] Diversification input */
            out byte[] Mac
            )
        {
            Mac = new byte[8];
            return phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part1(
                ref m_DataParamsInt[0],
                KeyNo,
                KeyV,
                Ppse,
                RndRC,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                Mac);
        }

        public Status_t Cmd_SAM_ProximityCheckMfp_Part2(
            byte[] Mac
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_ProximityCheckMfp_Part2(
                ref m_DataParamsInt[0],
                Mac);
        }

        #endregion

        #region NON X MODE: MIFARE CLASSIC

        public Status_t Cmd_SAM_AuthenticateMifare_Part1(
            AuthenticateMifare_Option Option,
            byte[] Uid,								/**< [In] Pointer to the Uid of the Card.*/
            byte KeyNoM,							/**< [In] Key Number.*/
            byte KeyVM,							/**< [In] Key Version.*/
            byte KeyType,							/**< [In] Key type A or B.*/
            byte BlockNo,							/**< [In] Block Number.*/
            byte[] RndB,							/**< [In] Random Number received from card [5].*/
            byte DivBlockNo,						/**< [In] Diversify block number.*/
            out byte[] TokenAB							/**< [Out] Token to be sent to the card [9].*/
            )
        {
            Status_t status;
            TokenAB = new byte[9];

            status = phhalHw_SamAV2_Cmd_SAM_AuthenticateMifare_Part1(
                ref m_DataParamsInt[0],
                (byte)Option,
                Uid,
                KeyNoM,
                KeyVM,
                KeyType,
                BlockNo,
                RndB,
                DivBlockNo,
                TokenAB);

            if (!(status.Equals(Error_Gen.SUCCESS_CHAINING)))
            {
                TokenAB = null;
            }
            return status;
        }

        public Status_t Cmd_SAM_AuthenticateMifare_Part2(
            byte[] TokenAB							/**< [In] Token received from the card[5].*/
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_AuthenticateMifare_Part2(
                ref m_DataParamsInt[0],
                TokenAB);
        }

        public Status_t Cmd_SAM_ChangeKeyMifare(
            ChangeKeyMifare_Option Option,
            byte KeyNoM,
            byte KeyVM_KeyVMA,
            byte KeyAorB_KeyVMB,					/**< [In] Key 0Ah or 0Bh in case of key retrieval or key version of MIFARE key B in case of key change.. */
            byte[] AccessConditions,				/**< [In] Access conditions to be stored within MIFARE card. Only used for key change. This buffer has to be 4 bytes long. */
            byte[] Uid,								/**< [In] Uid of the card. Only used if diversification is needed. This buffer has to be 4 bytes long. */
            byte DivBlockNo,						/**< [In] Block number to be used for diversification. */
            out byte[] KeyData							/**< [Out] Buffer containing the key data. The buffer has to be 16 bytes long in case of key retrieval and 21 bytes long in case of key change. */
            )
        {
            KeyData = new byte[21];
            byte KeyDataLength = 0;

            Status_t status = phhalHw_SamAV2_Cmd_SAM_ChangeKeyMifare(
                ref m_DataParamsInt[0],
                (byte)Option,
                KeyNoM,
                KeyVM_KeyVMA,
                KeyAorB_KeyVMB,
                AccessConditions,
                Uid,
                DivBlockNo,
                KeyData,
                ref KeyDataLength);

            if (status.Equals(Error_Gen.SUCCESS))
            {
                Array.Resize<byte>(ref KeyData, (int)KeyDataLength);
            }
            return status;
        }

        #endregion

        #region NON X MODE: DESFIRE ULC

        public Status_t Cmd_SAM_AuthenticatePICC_Part1(
            AuthenticatePICC_AuthMode AuthMode,							/**< [In] Authentication mode used */
            byte KeyNo,							/**< [In] DESFire key number or Key Reference Number (depends on SAm_SelectApplication done or not)*/
            byte KeyV,							/**< [In] the key version of Key No */
            byte[] EncRndB,							/**< [In] Encoded Random B (received from DESFire PICC) */
            byte[] DivInput,							/**< [In] DivInp data */
            out byte[] EncRndAB						/**< [Out] Received data */
            )
        {
            Status_t status;
            EncRndAB = new byte[32];
            byte RxLength = 0;

            status = phhalHw_SamAV2_Cmd_SAM_AuthenticatePICC_Part1(
                ref m_DataParamsInt[0],
                (byte)AuthMode,
                KeyNo,
                KeyV,
                EncRndB,
                (byte)((EncRndB == null) ? 0 : EncRndB.Length),
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                EncRndAB,
                ref RxLength);

            /* Resize pEncRndAB buffer to actual size */
            //if (status.Equals(Error_Gen.SUCCESS_CHAINING))
            //{
            Array.Resize<byte>(ref EncRndAB, (int)RxLength);
            //}
            return status;

        }

        public Status_t Cmd_SAM_AuthenticatePICC_Part2(
            byte[] EncRndA					        /**< [In] Input data */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_AuthenticatePICC_Part2(
                ref m_DataParamsInt[0],
                EncRndA,
                (byte)((EncRndA == null) ? 0 : EncRndA.Length));
        }

        public Status_t Cmd_SAM_IsoAuthenticatePICC_Part1(
            AuthenticatePICC_AuthMode AuthMode,							/**< [In] Authentication mode used */
            byte KeyNo,							/**< [In] DESFire key number or Key Reference Number (depends on SAm_SelectApplication done or not)*/
            byte KeyV,							/**< [In] the key version of Key No */
            byte[] RndCard,						/**< [In] Random Number Card (8 bytes for (3)DES or 16 bytes for 3K3DES and AES) */
            byte[] DivInput,							/**< [In] DivInp data */
            out byte[] EncRnd						/**< [Out] Received data */
            )
        {
            Status_t status;
            EncRnd = new byte[48];
            byte RxLength = 0;

            status = phhalHw_SamAV2_Cmd_SAM_IsoAuthenticatePICC_Part1(
                ref m_DataParamsInt[0],
                (byte)AuthMode,
                KeyNo,
                KeyV,
                RndCard,
                (byte)((RndCard == null) ? 0 : RndCard.Length),
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                EncRnd,
                ref RxLength);

            /* Resize pEncRnd buffer to actual size */
            //if (status.Equals(Error_Gen.SUCCESS_CHAINING))
            //{
            Array.Resize<byte>(ref EncRnd, (int)RxLength);
            //}
            return status;
        }

        public Status_t Cmd_SAM_IsoAuthenticatePICC_Part2(
            byte[] Rnd							/**< [In] Input data */
            )
        {
            return phhalHw_SamAV2_Cmd_SAM_IsoAuthenticatePICC_Part2(
                ref m_DataParamsInt[0],
                Rnd,
                (byte)((Rnd == null) ? 0 : Rnd.Length));
        }

        public Status_t Cmd_SAM_ChangeKeyPICC(
            ChangeKeyPICC_KeyCompMeth KeyCompMeth,		/**< [In] bit mask, determining the key compilation method. */
            ChangeKeyPICC_Cfg Cfg,								/**< [In] Bit mask defining the change key configuration. */
            byte KeyNoCurrent,						/**< [In] Current Key Number. */
            byte KeyVCurrent,						/**< [In] Current Key Version. */
            byte KeyNoNew,							/**< [In] New Key Number. */
            byte KeyVNew,							/**< [In] New Key Version. */
            byte[] DivInput,							/**< [In] DivInp data */
            out byte[] EncStream                   /**< [Out] Buffer containing the stream to be sent to the PICC. */
            )
        {
            Status_t status;
            EncStream = new byte[32];
            byte RxLength = 0;

            status = phhalHw_SamAV2_Cmd_SAM_ChangeKeyPICC(
                ref m_DataParamsInt[0],
                (byte)KeyCompMeth,
                (byte)Cfg,
                KeyNoCurrent,
                KeyVCurrent,
                KeyNoNew,
                KeyVNew,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                EncStream,
                ref RxLength);

            /* Resize EncStream buffer to actual size */
            //if (status.Equals(Error_Gen.SUCCESS))
            //{
            Array.Resize<byte>(ref EncStream, (int)RxLength);
            //}
            return status;
        }

        #endregion

        #region X MODE: MFRC52

        public Status_t Cmd_RC_ReadRegister(
                                        byte[] RegAddress,                          /**< [In] Address of the registers to be read*/
                                        out byte[] Value                             /**< [Out] Contents of the registers*/
                                        )
        {
            Value = new byte[RegAddress.Length];
            return phhalHw_SamAV2_Cmd_RC_ReadRegister(
                ref m_DataParamsInt[0],
                RegAddress,
                (byte)((RegAddress == null) ? 0 : RegAddress.Length),
                Value);
        }

        public Status_t Cmd_RC_WriteRegister(
                                        byte[] Data                             /**< [In] data containing Address0 || Data0 || Address1 || Data1 ...*/
                                        )
        {
            return phhalHw_SamAV2_Cmd_RC_WriteRegister(
                ref m_DataParamsInt[0],
                Data,
                (byte)((Data == null) ? 0 : Data.Length));
        }

        public Status_t Cmd_RC_RFControl(
                                        int Time                              /**< [In] Shut down time for HF Field*/
                                        )
        {
            return phhalHw_SamAV2_Cmd_RC_RFControl(
                ref m_DataParamsInt[0],
                (ushort)Time);
        }

        public Status_t Cmd_RC_Init(
            byte LoadReg
            )
        {
            return phhalHw_SamAV2_Cmd_RC_Init(
                ref m_DataParamsInt[0],
                LoadReg);
        }

        public Status_t Cmd_RC_LoadRegisterValueSet(
                                        byte StoreReg,							/**< [In] Number of register value set to be used for storing (00h to 07h). */
                                        byte[] Data							/**< [In] Buffer containing address - value duos. */
                                        )
        {
            return phhalHw_SamAV2_Cmd_RC_LoadRegisterValueSet(
                ref m_DataParamsInt[0],
                StoreReg,
                Data,
                (byte)((Data == null) ? 0 : Data.Length));
        }

        #endregion

        #region ISO3A LAYER

        public Status_t Cmd_X_ISO14443_3_RequestA_Wakeup(
            byte ReqaWakeupCode,					/**< [In] bReqaWakeupCode defining the code to be sent (either REQA or WUPA) */
            out byte[] Atqa								/**< [Out] AtqA; byte[2] */
            )
        {
            Atqa = new byte[2];
            return phhalHw_SamAV2_Cmd_X_ISO14443_3_RequestA_Wakeup(
                ref m_DataParamsInt[0],
                ReqaWakeupCode,
                Atqa);
        }

        public Status_t Cmd_X_ISO14443_3_AnticollisionSelect(
            byte[] SelCodes,							/**< [In] Pointer to the SEL sequence bytes. */
            out byte Sak,								/**< [Out] Select Acknowledge; byte */
            out byte[] Uid                            /**< [Out] complete Uid; byte[5]; */
            )
        {
            Status_t status;
            Uid = new byte[10];
            byte UidLength = 0;
            Sak = 0;

            status = phhalHw_SamAV2_Cmd_X_ISO14443_3_AnticollisionSelect(
                ref m_DataParamsInt[0],
                SelCodes,
                (byte)((SelCodes == null) ? 0 : SelCodes.Length),
                ref Sak,
                Uid,
                ref UidLength);

            /* Resize pUid buffer to actual bVersion size */
            if (status.Equals(Error_Gen.SUCCESS) || status.Equals(Error.ISO_UID_INCOMPLETE))
            {
                Array.Resize<byte>(ref Uid, (int)UidLength);
            }
            return status;
        }

        public Status_t Cmd_X_ISO14443_3_ActivateIdle(
            ActivateIdle_Option Option,
            byte NumCards,
            int Time,                               /**< [In] Time to wait for a card; byte[2] */
            byte[] AtqaIn,                             /**< [In] AtqA, mask and value; byte[4] */
            byte[] SakIn,                              /**< [In] Sak, mask and value; byte[2] */
            out ActivateIdleParams[] CardInfo     /**< [Out] Pointer of structs for cards to be activated, phpalI14443p3a_SamAV2_X_ActivateIdleParams_t[bNumCardsToActivate]*/
            )
        {
            Status_t status;
            IntPtr pRxBufferInt = IntPtr.Zero;
            ushort pRxLength = 0;
            byte[] RxArray;
            int param_begin;

            status = phhalHw_SamAV2_Cmd_X_ISO14443_3_ActivateIdle(
                ref m_DataParamsInt[0],
                (byte)Option,
                NumCards,
                (ushort)Time,
                AtqaIn,
                SakIn,
                ref pRxBufferInt,
                ref pRxLength);

            if (pRxLength > 0 && status.Equals(Error_Gen.SUCCESS))
            {
                RxArray = new byte[pRxLength];
                Marshal.Copy(pRxBufferInt, RxArray, 0, pRxLength);

                if (NumCards > 1)
                {
                    CardInfo = new ActivateIdleParams[RxArray[0]];
                }
                else
                {
                    CardInfo = new ActivateIdleParams[NumCards];
                }

                if (NumCards == 1)
                {
                    CardInfo[0].Atqa = new byte[2];
                    Array.Copy(RxArray, CardInfo[0].Atqa, 2);
                    CardInfo[0].Sak = RxArray[2];
                    CardInfo[0].Uid = new byte[RxArray[3]];
                    Array.Copy(RxArray, 4, CardInfo[0].Uid, 0, RxArray[3]);
                }
                else
                {
                    param_begin = 1;
                    for (int i = 0; i < RxArray[0]; i++)
                    {
                        CardInfo[i].Atqa = new byte[2];
                        Array.Copy(RxArray, param_begin, CardInfo[i].Atqa, 0, 2);
                        CardInfo[i].Sak = RxArray[param_begin + 2];
                        CardInfo[i].Uid = new byte[RxArray[param_begin + 3]];
                        Array.Copy(RxArray, param_begin + 4, CardInfo[i].Uid, 0, RxArray[param_begin + 3]);
                        param_begin = param_begin + 4 + CardInfo[i].Uid.Length;
                    }
                }
            }
            else
            {
                CardInfo = new ActivateIdleParams[0];
            }

            return status;
        }

        public Status_t Cmd_X_ISO14443_3_ActivateWakeUp(
            byte[] Uid							/**< [In] known Uid, can be NULL if bLenUidIn is 0; byte[4/7/10] */
            )
        {
            return phhalHw_SamAV2_Cmd_X_ISO14443_3_ActivateWakeUp(
                ref m_DataParamsInt[0],
                Uid,
                (byte)((Uid == null) ? 0 : Uid.Length));
        }

        public Status_t Cmd_X_ISO14443_3_HaltA(
            )
        {
            return phhalHw_SamAV2_Cmd_X_ISO14443_3_HaltA(
                ref m_DataParamsInt[0]);
        }

        public Status_t Cmd_X_ISO14443_3_TransparentExchange(
            byte[] TxBuffer,						/**< [In] Buffer containing the data to be sent. */
            byte TxBitLength,						/**< [In] Number of bits to be sent to the card in last byte (00h - 07h). */
            out byte[] RxBuffer,						/**< [Out] Pointer to the buffer containing the received data. */
            out byte RxBitLength
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;
            RxBitLength = 0;

            status = phhalHw_SamAV2_Cmd_X_ISO14443_3_TransparentExchange(
                ref m_DataParamsInt[0],
                TxBuffer,
                (byte)((TxBuffer == null) ? 0 : TxBuffer.Length),
                TxBitLength,
                ref RxBufferInt,
                ref RxLengthInt,
                ref RxBitLength);

            RxBuffer = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        #endregion

        #region ISO4A LAYER

        public Status_t Cmd_X_ISO14443_4_RATS_PPS(
            byte CidIn,                              /**< [Int] CID to be sent, and CID used */
            byte DsiIn,                              /**< [In] DSI to be sent, and DSI used */
            byte DriIn,                              /**< [In] DRI to be sent, and DRI used */
            out byte CidOut,                              /**< [In/Out] CID to be sent, and CID used */
            out byte DsiOut,                              /**< [In/Out] DSI to be sent, and DSI used */
            out byte DriOut,                              /**< [In/Out] DRI to be sent, and DRI used */
            out byte[] Ats                               /**< [Out] ATS received */
            )
        {
            Ats = new byte[256];
            CidOut = 0;
            DsiOut = 0;
            DriOut = 0;

            Status_t status = phhalHw_SamAV2_Cmd_X_ISO14443_4_RATS_PPS(
                ref m_DataParamsInt[0],
                CidIn,
                DsiIn,
                DriIn,
                ref CidOut,
                ref DsiOut,
                ref DriOut,
                Ats);

            if (status.Equals(Error_Gen.SUCCESS))
            {
                Array.Resize(ref Ats, Ats[0]);
            }
            else
            {
                Ats = null;
            }
            return status;
        }

        public Status_t Cmd_X_ISO14443_4_Init(
            byte Cid,								  /**< [In] CID to apply. */
            byte Dri,								  /**< [In] Bit rate PCD -> PICC. */
            byte Dsi,								  /**< [In] Bit rate PICC -> PCD */
            byte Fwi,								  /**< [In] Frame waiting time indicator. */
            byte Fsci
            )
        {
            return phhalHw_SamAV2_Cmd_X_ISO14443_4_Init(
                ref m_DataParamsInt[0],
                Cid,
                Dri,
                Dsi,
                Fwi,
                Fsci
                );
        }

        public Status_t Cmd_X_ISO14443_4_Exchange(
            Buffering_Option Option,               /**< [In] Option parameter for TxChaining.
                                         *        Can be #PHHAL_HW_SAMAV2_ISO7816_LAST_FRAME or #PHHAL_HW_SAMAV2_ISO7816_CHAINED_FRAME. */
            byte[] AppDataIn,          /**< [In] Application Data to exchange; byte[] */
            out byte[] AppDataOut     /**< [Out] Data returned by the PICC; byte[] */
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;

            status = phhalHw_SamAV2_Cmd_X_ISO14443_4_Exchange(
                ref m_DataParamsInt[0],
                (ushort)Option,
                AppDataIn,
                (byte)((AppDataIn == null) ? 0 : AppDataIn.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            AppDataOut = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_X_ISO14443_4_PresenceCheck(
            )
        {
            return phhalHw_SamAV2_Cmd_X_ISO14443_4_PresenceCheck(ref m_DataParamsInt[0]);
        }

        public Status_t Cmd_X_ISO14443_4_Deselect(
            Deselect_FreeCid FreeCid								/**< [In] FALSE: CID will not be freed if deselect fails */
            )
        {
            return phhalHw_SamAV2_Cmd_X_ISO14443_4_Deselect(
                ref m_DataParamsInt[0],
                (byte)FreeCid);
        }

        public Status_t Cmd_X_ISO14443_4_FreeCid(
            byte[] Cid								/**< [In] List of CIDs to be freed. */
            )
        {
            return phhalHw_SamAV2_Cmd_X_ISO14443_4_FreeCid(
                ref m_DataParamsInt[0],
                Cid,
                (byte)((Cid == null) ? 0 : Cid.Length));
        }

        #endregion

        #region X MODE: MIFARE CLASSIC

        public Status_t Cmd_MfAuthenticate(
            MfAuthenticate_Option Option,
            byte[] Uid,								/**< [In] MIFARE standard UID*/
            byte KeyNoM,							/**< [In] Key Reference Number of MIFARE key. */
            byte KeyVM,								/**< [In] Key Reference Version of MIFARE key. */
            byte KeyType,							/**< [In] Type of Key : MF Key A (0x0A) or MF Key B (0xB)*/
            byte AuthBlockNo,							/**< [In] Block number to be authenticated within MIFARE PICC */
            byte DivBlockNo     					/**< [In] Block number to be used for Diversification */
            )
        {
            if (Uid.Length != 4)
            {
                return new Status_t(Error_CompCode.HAL, Error_Param.INVALID_DATA_PARAMS);
            }

            return phhalHw_SamAV2_Cmd_MfAuthenticate(
                ref m_DataParamsInt[0],
                (byte)Option,
                Uid,
                KeyNoM,
                KeyVM,
                KeyType,
                AuthBlockNo,
                DivBlockNo
                );
        }

        public Status_t Cmd_MfRead(
            MfRead_Option Option,
            byte[] Blocks,
            out byte[] Data
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;

            status = phhalHw_SamAV2_Cmd_MfRead(
                ref m_DataParamsInt[0],
                (byte)Option,
                Blocks,
                (byte)((Blocks == null) ? 0 : Blocks.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            Data = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_MfWrite(
            MfWrite_Option Option,                 /**< [In] Exchange option for buffered transmission. */
            byte[] Data
            )
        {
            return phhalHw_SamAV2_Cmd_MfWrite(
                ref m_DataParamsInt[0],
                (ushort)Option,
                Data,
                (byte)((Data == null) ? 0 : Data.Length));
        }

        public Status_t Cmd_MfValueWrite(
            byte[] pBlocks							/**< [In] Pointer to the block numbers and data to be written. bNr || value[4] || address[4] ...*/
            )
        {
            return phhalHw_SamAV2_Cmd_MfValueWrite(ref m_DataParamsInt[0], pBlocks, (byte)((pBlocks == null) ? 0 : pBlocks.Length));
        }

        public Status_t Cmd_MfIncrement(
            byte[] pBlocks							/**< [In] Pointer to the block numbers and data to be written. srcBnr || destBnr|| value[4] ||  ...*/
            )
        {
            return phhalHw_SamAV2_Cmd_MfIncrement(ref m_DataParamsInt[0], pBlocks, (byte)((pBlocks == null) ? 0 : pBlocks.Length));
        }

        public Status_t Cmd_MfDecrement(
            byte[] pBlocks							/**< [In] Pointer to the block numbers and data to be written. srcBnr || destBnr|| value[4] ||  ...*/
            )
        {
            return phhalHw_SamAV2_Cmd_MfDecrement(ref m_DataParamsInt[0], pBlocks, (byte)((pBlocks == null) ? 0 : pBlocks.Length));
        }

        public Status_t Cmd_MfRestore(
            byte[] pBlocks							/**< [In] Pointer to the block numbers of source and destination. srcBnr || destBnr||   ...*/
            )
        {
            return phhalHw_SamAV2_Cmd_MfRestore(ref m_DataParamsInt[0], pBlocks, (byte)((pBlocks == null) ? 0 : pBlocks.Length));
        }

        public Status_t Cmd_MfAuthenticateRead(
            Buffering_Option Option,							/**< [In] Option mask for this command. \n */
            byte[] Uid,								/**< [In] MIFARE standard UID. This buffer has to be 4 bytes long. This buffer is only used if wOption is set to #PH_EXCHANGE_BUFFER_FIRST. */
            MfAuthenticateReadWrite_CmdSetting CmdSettings,		/**< [In] Command settings of the data cycle. */
            byte KeyNoM,							/**< [In] Key reference number of MIFARE key. */
            byte KeyVM,								/**< [In] Key version of MIFARE key. */
            byte KeyType,							/**< [In] Type of key : MF key A (0x0A) or MF key B (0xB). */
            byte AuthBlockNo,						/**< [In] Block number to be authenticated within MIFARE PICC. */
            byte DivBlockNo,	    				/**< [In] Block number to be used for diversification. */
            byte[] Blocks,							/**< [In] Pointer to the block numbers to be read. */
            out byte[] Data
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;
            Data = null;

            if (Uid.Length != 4)
            {
                return new Status_t(Error_CompCode.HAL, Error_Param.INVALID_DATA_PARAMS);
            }

            status = phhalHw_SamAV2_Cmd_MfAuthenticateRead(
                ref m_DataParamsInt[0],
                (ushort)Option,
                Uid,
                (byte)CmdSettings,
                KeyNoM,
                KeyVM,
                KeyType,
                AuthBlockNo,
                DivBlockNo,
                Blocks,
                (byte)((Blocks == null) ? 0 : Blocks.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            Data = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_MfAuthenticateWrite(
            Buffering_Option Option,							/**< [In] Option mask for this command. \n */
            byte[] Uid,								/**< [In] MIFARE standard UID. This buffer has to be 4 bytes long. This buffer is only used if wOption is set to #PH_EXCHANGE_BUFFER_FIRST. */
            MfAuthenticateReadWrite_CmdSetting CmdSettings,		/**< [In] Command settings of the data cycle. */
            byte KeyNoM,							/**< [In] Key reference number of MIFARE key. */
            byte KeyVM,								/**< [In] Key version of MIFARE key. */
            byte KeyType,							/**< [In] Type of key : MF key A (0x0A) or MF key B (0xB). */
            byte AuthBlockNo,						/**< [In] Block number to be authenticated within MIFARE PICC. */
            byte DivBlockNo,	    				/**< [In] Block number to be used for diversification. */
            byte[] Blocks							/**< [In] Pointer to the block numbers to be read. */
            )
        {
            if (Uid.Length != 4)
            {
                return new Status_t(Error_CompCode.HAL, Error_Param.INVALID_DATA_PARAMS);
            }

            return phhalHw_SamAV2_Cmd_MfAuthenticateWrite(
                ref m_DataParamsInt[0],
                (ushort)Option,
                Uid,
                (byte)CmdSettings,
                KeyNoM,
                KeyVM,
                KeyType,
                AuthBlockNo,
                DivBlockNo,
                Blocks,
                (byte)((Blocks == null) ? 0 : Blocks.Length));
        }

        public Status_t Cmd_MfChangeKey(
           MfChangeKey_KeyCompMeth KeyCompMeth,		/**< [In] bit mask, determining the key compilation method. */
           byte KeyNoM,							/**< [In] Key Reference Number of MIFARE key. */
           byte KeyVMA,							/**< [In] Key Reference Version of MIFARE key A. */
           byte KeyVMB,							/**< [In] Key Reference Version of MIFARE key B. */
           byte BlockNo,   						/**< [In] MIFARE block number (block to store the new key). */
           byte[] AccessConditions,								/**< [In] Access Conditions to be stored (4 bytes). */
           byte[] Uid,								/**< [In] UID standard (4 bytes). */
           byte DivBlockNo        					/**< [In] Block number to be used for Diversification */
           )
        {
            return phhalHw_SamAV2_Cmd_MfChangeKey(
                ref m_DataParamsInt[0],
                (byte)KeyCompMeth,
                KeyNoM,
                KeyVMA,
                KeyVMB,
                BlockNo,
                AccessConditions,
                Uid,
                DivBlockNo);
        }

        #endregion

        #region MIFAREPLUS_X

        public Status_t Cmd_MfpWritePerso(
            byte[] Values,					/**< Values containing from Block ADDR ||DATA ||Block addr || DATA ..... */
            out byte PiccReturnCode
            )
        {
            PiccReturnCode = 0;

            return phhalHw_SamAV2_Cmd_MfpWritePerso(
                ref m_DataParamsInt[0],
                Values,
                (byte)((Values == null) ? 0 : Values.Length),
                ref PiccReturnCode);
        }

        public Status_t Cmd_MfpAuthenticate(
            AuthenticateMFP_Option Option,
            byte KeyNo,								/**< [In] Key reference number to be used in authentication. */
            byte KeyV,								/**< [In] Key version to be used in authentication. */
            int BlockNo,                          /**< [In] Block Number to be sent to the card. */
            byte[] PcdCapsIn,						/**< [In] Buffer containing the input PcdCaps. It has to be bDivInputLength bytes long. */
            byte[] DivInput,						/**< [In] Diversification data. This buffer has to be bDivInputLength bytes long. */
            out byte[] PcdCapsOut,						/**< [Out] Buffer containing the PcdCaps Out. It has to be 3 bytes long. */
            out byte[] PdCaps,							/**< [Out] Buffer containing the PdCaps Out. It has to be 3 bytes long. */
            out byte PiccReturnCode
            )
        {
            PiccReturnCode = 0;
            PcdCapsOut = new byte[6];
            PdCaps = new byte[6];

            return phhalHw_SamAV2_Cmd_MfpAuthenticate(
                ref m_DataParamsInt[0],
                (byte)Option,
                KeyNo,
                KeyV,
                (ushort)BlockNo,
                PcdCapsIn,
                (byte)((PcdCapsIn == null) ? 0 : PcdCapsIn.Length),
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                PcdCapsOut,
                PdCaps,
                ref PiccReturnCode);
        }

        public Status_t Cmd_MfpCombinedRead(
            Buffering_Option Option,                           /**< [In] Implements buffering in RX direction. */
            byte[] PlainCommand,                     /**< [In] Plain command to be sent to the card */
            out byte[] Data,                          /**< [Out] Out Data Stream */
            out byte pPiccReturnCode
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;
            pPiccReturnCode = 0;

            status = phhalHw_SamAV2_Cmd_MfpCombinedRead(
                ref m_DataParamsInt[0],
                (ushort)Option,
                PlainCommand,
                (byte)((PlainCommand == null) ? 0 : PlainCommand.Length),
                ref RxBufferInt,
                ref RxLengthInt,
                ref pPiccReturnCode);

            Data = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_MfpCombinedWrite(
            Buffering_Option Option,                           /**< [In] Implements buffering. */
            byte[] Data,                            /**< [In] Input Data Stream */
            out byte PiccReturnCode
            )
        {
            PiccReturnCode = 0;

            return phhalHw_SamAV2_Cmd_MfpCombinedWrite(
                ref m_DataParamsInt[0],
                (ushort)Option,
                Data,
                (byte)((Data == null) ? 0 : Data.Length),
                ref PiccReturnCode);
        }

        public Status_t Cmd_MfpChangeKey(
            byte Cmd,                               /**< [In] Command code for MIFARE Plus Write (WRITE_M or WRITE) */
            int BlockNo,                          /**< [In] Block number of the MIFARE Plus Block */
            byte KeyNo,                        /**< [In] Key number to be used in authentication */
            byte KeyV,                       /**< [In] Key version to be used in authentication */
            byte[] DivInput,                        /**< [In] Pointer to Div Input Data (could be NULL)*/
            out byte PiccReturnCode
            )
        {
            PiccReturnCode = 0;

            return phhalHw_SamAV2_Cmd_MfpChangeKey(
                ref m_DataParamsInt[0],
                Cmd,
                (ushort)BlockNo,
                KeyNo,
                KeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                ref PiccReturnCode);
        }

        public Status_t Cmd_MfpProximityCheck(
            MfpProximityCheck_Option Option,
            byte KeyNo,                        /**< [In] Key number to be used in authentication */
            byte KeyV,                       /**< [In] Key version to be used in authentication */
            byte BlCnt,                      /**< [In] Specifies the maximum amount of random bytes transmitted at once */
            byte[] DivInput, 						/**< [In] Diversification input data */
            out byte PiccReturnCode
            )
        {
            PiccReturnCode = 0;

            return phhalHw_SamAV2_Cmd_MfpProximityCheck(
                ref m_DataParamsInt[0],
                (byte)Option,
                KeyNo,
                KeyV,
                BlCnt,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                ref PiccReturnCode);
        }

        public Status_t Cmd_MfpVirtualCardSupport(
            Buffering_Option Option,                           /**< [In] Implements the buffering: Note: For BUFFERED_START the parameters are ignored. For BUFFERED_LAST the PCD Caps are used */
            byte[] Data,
            byte[] PcdCapabilities,                 /**< [In] PcdCapabilities sent to the card */
            out byte[] SupportedCardInfo,              /**< [Out] Card Info of supported cards */
            out byte PiccReturnCode
            )
        {
            Status_t status;
            ushort RxLengthInt = 0;
            IntPtr RxBufferInt = IntPtr.Zero;
            SupportedCardInfo = null;
            PiccReturnCode = 0;

            status = phhalHw_SamAV2_Cmd_MfpVirtualCardSupport(
                ref m_DataParamsInt[0],
                (ushort)Option,
                Data,
                (byte)((Data == null) ? 0 : Data.Length),
                PcdCapabilities,
                (byte)((PcdCapabilities == null) ? 0 : PcdCapabilities.Length),
                ref RxBufferInt,
                ref RxLengthInt,
                ref PiccReturnCode);

            SupportedCardInfo = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_MfpSelectVirtualCard(
            byte KeyNo,                        /**< [In] Key number to be used in SVC command*/
            byte KeyV,                       /**< [In] Key version to be used in SVC command*/
            byte[] Iid,                            /**< [In] desired Iid . */
            byte[] DivInput,                       /**< [In] Diversification input */
            out byte PiccReturnCode
            )
        {
            PiccReturnCode = 0;
            return phhalHw_SamAV2_Cmd_MfpSelectVirtualCard(
                ref m_DataParamsInt[0],
                KeyNo,
                KeyV,
                Iid,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length),
                ref PiccReturnCode);
        }

        #endregion

        #region X MODE: DESFIRE ULC

        public Status_t Cmd_DESFireAuthenticatePICC(
            AuthenticatePICC_AuthMode AuthMode,		/**< [In] Authentication mode used */
            AuthenticatePICC_ISOMode ISOMode,		/**< [In] ISO mode used */
            byte DFKeyNo,							/**< [In] DESFire key number. */
            byte KeyNo,								/**< [In] Key reference number. */
            byte KeyV,								/**< [In] Key version. */
            byte[] DivInput
            )
        {
            return phhalHw_SamAV2_Cmd_DESFireAuthenticatePICC(
                ref m_DataParamsInt[0],
                (byte)AuthMode,
                (byte)ISOMode,
                DFKeyNo,
                KeyNo,
                KeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length));
        }

        public Status_t Cmd_DESFireChangeKeyPICC(
            ChangeKeyPICC_KeyCompMeth KeyCompMeth,						/**< [In] bit mask, determining the key compilation method. */
            DESFireChangeKeyPICC_Cfg Cfg,
            byte KeyNoCurrent,						/**< [In] Current Key Number. */
            byte KeyVCurrent,						/**< [In] Current Key Version. */
            byte KeyNoNew,							/**< [In] New Key Number. */
            byte KeyVNew,							/**< [In] New Key Version. */
            byte[] DivInput							/**< [In] DivInp data */
            )
        {
            return phhalHw_SamAV2_Cmd_DESFireChangeKeyPICC(
                ref m_DataParamsInt[0],
                (byte)KeyCompMeth,
                (byte)Cfg,
                KeyNoCurrent,
                KeyVCurrent,
                KeyNoNew,
                KeyVNew,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length));
        }

        public Status_t Cmd_DESFireWriteX(
            Buffering_Option Option,                         /**< [In] Implements buffering. */
            DESFireReadWriteX_Crypto Crypto,							/**< [In] Crypto mode (encrypted, MACed). */
            byte[] Data							/**< [In] Data to be written. */
            )
        {
            return phhalHw_SamAV2_Cmd_DESFireWriteX(
                ref m_DataParamsInt[0],
                (ushort)Option,
                (byte)Crypto,
                Data,
                (byte)((Data == null) ? 0 : Data.Length));
        }

        public Status_t Cmd_DESFireReadX(
            Buffering_Option Option,                           /**< [In] Implements buffering. */
            DESFireReadWriteX_Crypto Crypto,							/**< [In] Crypto mode (encrypted, MACed). */
            byte[] Data,							/**< [In] Data to send to DESFire. */
            out byte[] ReceiveBuffer					/**< [Out] Data response. */
            )
        {
            Status_t status;
            IntPtr RxBufferInt = IntPtr.Zero;
            ushort RxLengthInt = 0;

            status = phhalHw_SamAV2_Cmd_DESFireReadX(
                ref m_DataParamsInt[0],
                (ushort)Option,
                (byte)Crypto,
                Data,
                (byte)((Data == null) ? 0 : Data.Length),
                ref RxBufferInt,
                ref RxLengthInt);

            ReceiveBuffer = MarshalCopy(status, RxBufferInt, RxLengthInt);
            return status;
        }

        public Status_t Cmd_ULCAuthenticatePICC(
           byte KeyNo,						/**< [In] the key reference number for SAM KeyStorage */
           byte KeyV,						/**< [In] the key reference version for SAM KeyStorage */
           byte[] DivInput						/**< [In] DivInp data */
           )
        {
            return phhalHw_SamAV2_Cmd_ULCAuthenticatePICC(
                ref m_DataParamsInt[0],
                KeyNo,
                KeyV,
                DivInput,
                (byte)((DivInput == null) ? 0 : DivInput.Length));
        }

        #endregion

        #region 7816_EXCHANGE

        /// <summary>
        /// Exchange commands with the SAM.
        /// </summary>
        /// <param name="bCmd"></param>
        /// <param name="pTxBuffer"></param>
        /// <param name="wRxBufSize"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        public Status_t Cmd_7816Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] input data */
            out byte[] pRxBuffer    /**< [Out] output data */
            )
        {
            Status_t status;
            ushort wTxLength = 0;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            if (pTxBuffer != null)
            {
                wTxLength = (ushort)pTxBuffer.Length;
            }
            status = phhalHw_SamAV2_Cmd_7816Exchange(m_pDataParams, (ushort)wOption, pTxBuffer, wTxLength, ref pRxBufferInt, ref wRxLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public SamAV2()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~SamAV2()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
