/*
 * Copyright 2013, 2017, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// RdFpga specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class RdFpga : Hal.Generic
    {
        #region DEFINES

        public const int DEFAULT_TIMEOUT_MS = 5;        /**< Default timeout in microseconds. */
        private const int SHADOW_COUNT = 0x0D;          /**< Number of shadowed configurations. */

        public enum Config : int
        {
            TARGET_JOINER = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN),     /**< Disable PcdReset call in \ref phhalHw_ApplyProtocolSettings function. */
            LOGGING,                                                    /**< Switch the logging on/off */
            FIELD_OFF_TIME_CYCLES,                                      /**< Set the field off time for field-reset in clock cycles */
            FIELD_OFF_TIME_US                                           /**< Set the field off time for field-reset in us */
        }

        public enum Joiner : ushort
        {
            I = 0x8000,
            Q,
            BOTH
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]

        public unsafe struct DataParams_t
        {
            public ushort wId;                                      /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;                           /**< pointer to the lower layers parameter structure */
            public IntPtr pTxBuffer;                                /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                               /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                                /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                                /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                               /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                                /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                           /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;                           /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
            public byte bCardType;                                  /**< Type of card for which the hal is configured for */
            public ushort wJoiner;                                  /**< Currently selected Joiner. */
            public fixed ushort wCfgShadow[SHADOW_COUNT];           /**< Configuration shadow; Stores configuration for current cardtype. */
            public ushort wTimingMode;                              /**< Current timing measurement mode. */
            public uint dwTimeoutUs;                                /**< TO value to be used with phhalHwRd70xPiccExchgBitByte function */
            public uint dwTimeoutETU;								/**< TO value in FPGA ETU to be used by exchange function. */
            public ushort wAdditionalInfo;                          /**< storage for additional error information */
            public ushort wTxRxStartBackup;                         /**< Absolute Start-Position for transmission / reception used by buffer preservation mechanism. */
            public uint dwFieldOffTimeCycles;						/**< Field-Off-Time in clock cycles. */
            public ushort wFieldRecoveryTime;                       /**< Field-Recovery-Time in milliseconds. */
            public byte bSymbolStart;                               /**< Preamble of a frame. */
            public byte bSymbolEnd;                                 /**< Trailer symbol of a frame. */
            public byte bRfResetAfterTo;                            /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            public byte bFieldState;								/**< Mirrored Field State. */
            public byte bTypeBModeActive;                           /**< Flag indicating if TypeBMode is active */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_RdFpgaV6_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize               /**< [In] Size of the global receive buffer. */
            );

        #endregion

        #region CUSTOM COMMANDS


        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.RdFpga pBal, int wTxBufferSize, int wRxBufferSize)
        {
            // Adjust TxBuffer length
            if ((wTxBufferSize) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_RdFpgaV6_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, int wTxBufferSize, int wRxBufferSize)
        {
            // Adjust TxBuffer length
            if ((wTxBufferSize) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];

            return Init(wDataParamSize, pBal, wTxBufferSize, m_bTxBuffer, wRxBufferSize, m_bRxBuffer);
        }
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, int wTxBufferSize, byte[] bTxBuffer, int wRxBufferSize, byte[] bRxBuffer)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_RdFpgaV6_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#endif
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public RdFpga()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~RdFpga()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
