/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// Rd710 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class Rd710 : Hal.Generic
    {
        #region DEFINES

        private const int RESERVED_BUFFER_LEN = 8;  /**< Amount of needed and reserved memory for the protocol overhead. */

        public enum PcscMode : int
        {
            DEFAULT = 0x0000,
            DIRECT,
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;   /**< pointer to the lower layers parameter structure */
            public byte bSlotNumber;        /**< Slot number. */
            public byte bTxLastBits;        /**< TxLastbits for ExchangeRaw function. */
            public IntPtr pTxBuffer;        /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;       /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;        /**< Number of valid bytes within the transmit buffer. */
            public ushort wTxBufStart;      /**< Startposition in case of buffering within the transmit buffer. */
            public IntPtr pRxBuffer;        /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;       /**< Size of the global receive buffer. */
            public ushort wRxBufLen;        /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;   /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;   /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
            public ushort wFdtPc;           /**< Current timing value backup for PC*/
            public byte bCardType;          /**< Type of card for which the hal is configured for */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Init(
                                                IntPtr pDataParams,
                                                ushort wSizeOfDataParams,
                                                IntPtr pBalDataParams,
                                                byte bSlotNumber,
                                                IntPtr pTxBuffer,
                                                ushort wTxBufSize,
                                                IntPtr pRxBuffer,
                                                ushort wRxBufSize);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_InitReader(
                                    IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_LedOff(
                                    IntPtr pDataParams,
                                    byte bLeds);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_LedOn(
                                    IntPtr pDataParams,
                                    byte bLeds);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_ResetReader(
                                    IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_ResetFactoryDefaults(
                                    IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_SetConfigOption(
                                    IntPtr pDataParams,
                                    byte[] pConfig,
                                    byte[] pData,
                                    ushort wDataLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_GetConfigOption(
                                    IntPtr pDataParams,
                                    byte[] pConfig,
                                    IntPtr pData,
                                    ref ushort wDataLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_GetReaderStatus(
                                    IntPtr pDataParams,
                                    IntPtr pReaderStatus);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_RD_SetPcscMode(
                                    IntPtr pDataParams,
                                    PcscMode bMode);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_Activate(
                                    IntPtr pDataParams,
                                    IntPtr pAtr,
                                    ref ushort usAtrLen);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_ColdReset(
                                    IntPtr pDataParams,
                                    IntPtr pAtr,
                                    ref ushort usAtrLen);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_WarmReset(
                                    IntPtr pDataParams,
                                    IntPtr pAtr,
                                    ref ushort usAtrLen);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_StopClock(
                                    IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_StartClock(
                                    IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_Deactivate(
                                    IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_PresenceCheck(
                                    IntPtr pDataParams,
                                    ref byte pCardInserted);


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_TransmitData(
                                    IntPtr pDataParams,
                                    byte[] pTxBuffer,
                                    ushort wTxLength,
                                    ref IntPtr ppRxBuffer,
                                    ref ushort pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_CC_SendPPS(
                                    IntPtr pDataParams,
                                    byte bOption,
                                    byte[] pPPS,
                                    IntPtr pRxBuffer);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_FormatKeyEntry(
                                    IntPtr pDataParams,
                                    ushort wKeyNo,
                                    ushort wNewKeyType);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_SetKey(
                                   IntPtr pDataParams,
                                   ushort wKeyType,
                                   ushort wKeyNo,
                                   ushort wKeyVersion,
                                   ushort wNewKeyVersion,
                                   ushort wNewKeyLen,
                                   byte[] pNewKey);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_SetKeyAtPos(
                                    IntPtr pDataParams,
                                    ushort wKeyType,
                                    ushort wKeyNo,
                                    ushort wNewKeyVersion,
                                    ushort wPos,
                                    ushort wNewKeyLen,
                                    byte[] pNewKey);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_SetKUC(
                                    IntPtr pDataParams,
                                    ushort wKeyNr,
                                    ushort wRefNrKUC);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_SetFullKeyEntry(
                                    IntPtr pDataParams,
                                    ushort wNewKeyType,
                                    ushort wNrOfKeys,
                                    ushort wKeyNr,
                                    ushort wNewRefNoKUC,
                                    ushort wSingleKeyLen,
                                    byte[] pNewKeys,
                                    ushort[] pNewKeyVersionList);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_GetKeyEntry(
                                        IntPtr pDataParams,
                                        ushort wKeyNr,
                                        ushort wKeyVersionBufSize,
                                        IntPtr wKeyVersion,
                                        ref ushort wKeyVersionLength,
                                        ref ushort pKeyType);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_GetKey(
                                        IntPtr pDataParams,
                                        ushort wKeyNr,
                                        ushort wKeyVersion,
                                        byte bKeyBufSize,
                                        IntPtr pKey,
                                        ref ushort pKeyType);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_SetConfig(
                                        IntPtr pDataParams,
                                        ushort wConfig,
                                        ushort wValue);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_SetConfigStr(
                                        IntPtr pDataParams,
                                        ushort wConfig,
                                        byte[] pBuffer,
                                        ushort wBufferLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_GetConfig(
                                        IntPtr pDataParams,
                                        ushort wConfig,
                                        ref ushort pValue);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_GetConfigStr(
                                            IntPtr pDataParams,
                                            ushort wConfig,
                                            ref IntPtr ppBuffer,
                                            ref ushort pBufferLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_ChangeKUC(
                                            IntPtr pDataParams,
                                            ushort wRefNoKUC,
                                            uint dwLimit);


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_KS_GetKUC(
                                            IntPtr pDataParams,
                                            ushort wRefNoKUC,
                                            ref uint pdwLimit,
                                            ref uint pdwCurVal);

        /*
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_CID_GetFreeCid(
                                            IntPtr  pDataParams,
                                            byte* pCid   );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_CID_FreeCid(
                                            IntPtr  pDataParams,
                                            byte bCid );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4_SetProtocol(
                                            IntPtr  pDataParams,
                                            byte bCidEnable,
                                            byte bCid,
                                            byte bNadEnable,
                                            byte bNad,
                                            byte bFwi,
                                            byte bFsdi,
                                            byte bFsci);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4_ResetProtocol(
                                            IntPtr  pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4_Deselect(
                                            IntPtr  pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4_PresCheck(
                                            IntPtr  pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4_Exchange(
                                            IntPtr  pDataParams,
                                            ushort wOption,
                                            byte* pTxBuffer,
                                            ushort wTxLength,
                                            byte** ppRxBuffer,
                                            ushort* pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4_SetConfig(
                                            IntPtr  pDataParams,
                                            ushort wConfig,
                                            ushort wValue);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4_GetConfig(
                                            IntPtr  pDataParams,
                                            ushort wConfig,
                                            ushort* pValue
                                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4A_Rats(
                                            IntPtr  pDataParams,
                                            byte bFsdi,
                                            byte bCid,
                                            byte* pAts);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4A_Pps(
                                            IntPtr  pDataParams,
                                            byte bDri,
                                            byte bDsi);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4A_ActivateCard(
                                            IntPtr  pDataParams,
                                            byte bFsdi,
                                            byte bCid,
                                            byte bDri,
                                            byte bDsi,
                                            byte* pAts);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P4A_GetProtocolParams(
                                            IntPtr  pDataParams,
                                            byte* pCidEnabled,
                                            byte* pCid,
                                            byte* pNadSupported,
                                            byte* pFwi,
                                            byte* pFsdi,
                                            byte* pFsci);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_RequestA(
                                            IntPtr  pDataParams,
                                            byte* pAtqa);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_WakeUpA(
                                            IntPtr  pDataParams,
                                            byte* pAtqa);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_HaltA(
                                            IntPtr  pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_Anticollision(
                                            IntPtr  pDataParams,
                                            byte bCascadeLevel,
                                            byte* pUidIn,
                                            byte bNvbUidIn,
                                            byte* pUidOut,
                                            byte* pNvbUidOut);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_Select(
                                            IntPtr  pDataParams,
                                            byte bCascadeLevel,
                                            byte* pUidIn,
                                            byte* pSak);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_ActivateCard(
                                        IntPtr  pDataParams,
                                        byte* pUidIn,
                                        byte bLenUidIn,
                                        byte* pUidOut,
                                        byte* pLenUidOut,
                                        byte* pSak,
                                        byte* pMoreCardsAvailable);


        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_Exchange(
                                        IntPtr  pDataParams,
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_I14443P3A_GetSerialNo(
                                        IntPtr  pDataParams,
                                        byte* pRxBuffer,
                                        byte* pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_MIFARE_ExchangeL3(
                                        IntPtr  pDataParams,
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_MIFARE_ExchangeL4(
                                        IntPtr  pDataParams,
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_MIFARE_ExchangePc(
                                    IntPtr  pDataParams,
                                    ushort wOption,
                                    byte* pTxBuffer,
                                    ushort wTxLength,
                                    byte** ppRxBuffer,
                                    ushort* pRxLength);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_MIFARE_MfcAuthenticateKeyNo(
                                    IntPtr  pDataParams,
                                    byte bBlockNo,
                                    byte bKeyType,
                                    ushort wKeyNo,
                                    ushort wKeyVersion,
                                    byte* pUid);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort  phhalHw_Rd710_Cmd_MIFARE_MfcAuthenticate(
                                    IntPtr  pDataParams,
                                    byte bBlockNo,
                                    byte bKeyType,
                                    byte* pKey,
                                    byte* pUid);

         */
        #endregion

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd710_Cmd_MIFARE_ExchangeRaw(
                                    IntPtr pDataParams,
                                    ushort wOption,
                                    byte[] pTxBuffer,
                                    ushort wTxLength,
                                    byte bTxLastBits,
                                    ref IntPtr ppRxBuffer,
                                    ref ushort pRxLength,
                                    ref byte pRxLastBits);


        #region CUSTOM FUNCTIONS

        public Status_t Cmd_InitReader()
        {
            return phhalHw_Rd710_Cmd_InitReader(
                                            m_pDataParams);
        }

        public Status_t Cmd_RD_LedOff(
                                    byte bLeds)
        {
            return phhalHw_Rd710_Cmd_RD_LedOff(
                                            m_pDataParams,
                                            bLeds);
        }

        public Status_t Cmd_RD_LedOn(
                                    byte bLeds)
        {
            return phhalHw_Rd710_Cmd_RD_LedOn(
                                            m_pDataParams,
                                            bLeds);
        }

        public Status_t Cmd_RD_ResetReader()
        {
            return phhalHw_Rd710_Cmd_RD_ResetReader(
                                            m_pDataParams);
        }

        public Status_t Cmd_RD_ResetFactoryDefaults()
        {
            return phhalHw_Rd710_Cmd_RD_ResetFactoryDefaults(
                                            m_pDataParams);
        }

        public Status_t Cmd_RD_SetConfigOption(
                                            byte[] pConfig,
                                            byte[] pData)
        {
            return phhalHw_Rd710_Cmd_RD_SetConfigOption(
                                                    m_pDataParams,
                                                    pConfig,
                                                    pData,
                                                    (ushort)pData.Length);
        }

        public Status_t Cmd_RD_GetConfigOption(
                                            byte[] pConfig,
                                            out byte[] pData)
        {
            pData = new byte[4];            // Max type is uing32_t, See Appendix-B of SAD
            ushort usDataLen = 0;
            GCHandle ghData = GCHandle.Alloc(pData, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_RD_GetConfigOption(
                                                            m_pDataParams,
                                                            pConfig,
                                                            ghData.AddrOfPinnedObject(),
                                                            ref usDataLen);
            Array.Resize<byte>(ref pData, usDataLen);
            ghData.Free();
            return status;
        }


        public Status_t Cmd_RD_GetReaderStatus(
                                            out byte[] pReaderStatus)
        {
            pReaderStatus = new byte[9];
            GCHandle ghReaderStatus = GCHandle.Alloc(pReaderStatus, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_RD_GetReaderStatus(
                                                            m_pDataParams,
                                                            ghReaderStatus.AddrOfPinnedObject());
            ghReaderStatus.Free();
            return status;
        }

        public Status_t Cmd_RD_SetPcscMode(
                                        PcscMode bMode)
        {
            return phhalHw_Rd710_Cmd_RD_SetPcscMode(
                                                    m_pDataParams,
                                                    bMode);
        }

        public Status_t Cmd_CC_Activate(
                                        out byte[] pAtr,
                                        out ushort wAtrLen)
        {
            pAtr = new byte[32];
            wAtrLen = 0;
            GCHandle ghAtr = GCHandle.Alloc(pAtr, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_CC_Activate(
                                                    m_pDataParams,
                                                    ghAtr.AddrOfPinnedObject(),
                                                    ref wAtrLen);
            Array.Resize<byte>(ref pAtr, wAtrLen);
            ghAtr.Free();
            return status;
        }

        public Status_t Cmd_CC_ColdReset(
                                        out byte[] pAtr,
                                        out ushort wAtrLen)
        {
            pAtr = new byte[32];
            wAtrLen = 0;
            GCHandle ghAtr = GCHandle.Alloc(pAtr, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_CC_ColdReset(
                                                m_pDataParams,
                                                ghAtr.AddrOfPinnedObject(),
                                                ref wAtrLen);
            Array.Resize<byte>(ref pAtr, wAtrLen);
            ghAtr.Free();
            return status;
        }

        public Status_t Cmd_CC_WarmReset(
                                        out byte[] pAtr,
                                        out ushort wAtrLen)
        {
            pAtr = new byte[32];
            wAtrLen = 0;
            GCHandle ghAtr = GCHandle.Alloc(pAtr, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_CC_WarmReset(
                                                m_pDataParams,
                                                ghAtr.AddrOfPinnedObject(),
                                                ref wAtrLen);
            Array.Resize<byte>(ref pAtr, wAtrLen);
            ghAtr.Free();
            return status;
        }

        public Status_t Cmd_CC_StopClock()
        {
            return phhalHw_Rd710_Cmd_CC_StopClock(
                                                m_pDataParams);
        }

        public Status_t Cmd_CC_StartClock()
        {
            return phhalHw_Rd710_Cmd_CC_StartClock(
                                                m_pDataParams);
        }

        public Status_t Cmd_CC_Deactivate()
        {
            return phhalHw_Rd710_Cmd_CC_Deactivate(
                                                m_pDataParams);
        }

        public Status_t Cmd_CC_PresenceCheck(
                                            out byte CardInserted)
        {
            CardInserted = 0;
            return phhalHw_Rd710_Cmd_CC_PresenceCheck(
                                                      m_pDataParams,
                                                      ref CardInserted);
        }

        public Status_t Cmd_CC_TransmitData(
                                    byte[] pTxBuffer,
                                    out byte[] pRxBuffer,
                                    ushort wRxBufferSize)
        {
            IntPtr pRxBufferInt = IntPtr.Zero;
            ushort wRxLength = 0;
            Status_t status = phhalHw_Rd710_Cmd_CC_TransmitData(
                                                            m_pDataParams,
                                                            pTxBuffer,
                                                            (ushort)pTxBuffer.Length,
                                                            ref pRxBufferInt,
                                                            ref wRxLength);
            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }
            return status;
        }

        public Status_t Cmd_CC_SendPPS(
                                    byte Option,
                                    byte[] pPPS,
                                    out byte[] pRxBuffer)
        {
            pRxBuffer = new byte[4];                //PPS0,1,2,3 only.  PPSS,PCK are ot returned */
            GCHandle ghRxBuf = GCHandle.Alloc(pRxBuffer, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_CC_SendPPS(
                                                    m_pDataParams,
                                                    Option,
                                                    pPPS,
                                                    ghRxBuf.AddrOfPinnedObject());
            ghRxBuf.Free();
            return status;
        }

        public Status_t Cmd_KS_FormatKeyEntry(
                                    ushort wKeyNo,
                                    ushort wNewKeyType)
        {
            return phhalHw_Rd710_Cmd_KS_FormatKeyEntry(
                                                    m_pDataParams,
                                                    wKeyNo,
                                                    wNewKeyType);
        }

        public Status_t Cmd_KS_SetKey(
                                    ushort wKeyType,
                                    ushort wKeyNr,
                                    ushort wKeyVersion,
                                    ushort wNewKeyVersion,
                                    byte[] pNewKey)
        {
            return phhalHw_Rd710_Cmd_KS_SetKey(
                                            m_pDataParams,
                                            wKeyType,
                                            wKeyNr,
                                            wKeyVersion,
                                            wNewKeyVersion,
                                            (ushort)pNewKey.Length,
                                            pNewKey);
        }

        public Status_t Cmd_KS_SetKeyAtPos(
                                    ushort wKeyType,
                                    ushort wKeyNr,
                                    ushort wNewKeyVersion,
                                    ushort wPos,
                                    byte[] pNewKey)
        {
            return phhalHw_Rd710_Cmd_KS_SetKeyAtPos(
                                                    m_pDataParams,
                                                    wKeyType,
                                                    wKeyNr,
                                                    wNewKeyVersion,
                                                    wPos,
                                                    (ushort)pNewKey.Length,
                                                    pNewKey);
        }

        public Status_t Cmd_KS_SetKUC(
                                    ushort wKeyNr,
                                    ushort wRefNrKUC)
        {
            return phhalHw_Rd710_Cmd_KS_SetKUC(
                                            m_pDataParams,
                                            wKeyNr,
                                            wRefNrKUC);
        }

        public Status_t Cmd_KS_SetFullKeyEntry(
                                    ushort wNrOfKeys,
                                    ushort wNewKeyType,
                                    ushort wKeyNr,
                                    ushort wNewRefNrKUC,
                                    ushort wSingleKeyLen,
                                    byte[] pNewKeys,
                                    ushort[] pNewKeyVersionList)
        {
            return phhalHw_Rd710_Cmd_KS_SetFullKeyEntry(
                                                   m_pDataParams,
                                                   wNrOfKeys,
                                                   wNewKeyType,
                                                   wKeyNr,
                                                   wNewRefNrKUC,
                                                   wSingleKeyLen,
                                                   pNewKeys,
                                                   pNewKeyVersionList);
        }

        public Status_t Cmd_KS_GetKeyEntry(
                                        ushort wKeyNr,
                                        ushort wKeyVersionBufSize,
                                        out ushort[] wKeyVersion,
                                        out ushort wKeyVersionLen,
                                        out ushort wKeyType)
        {
            wKeyVersion = new ushort[wKeyVersionBufSize];
            wKeyVersionLen = 0;
            wKeyType = 0;
            GCHandle ghKeyVersion = GCHandle.Alloc(wKeyVersion, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_KS_GetKeyEntry(
                                                        m_pDataParams,
                                                        wKeyNr,
                                                        wKeyVersionBufSize,
                                                        ghKeyVersion.AddrOfPinnedObject(),
                                                        ref wKeyVersionLen,
                                                        ref wKeyType);
            ghKeyVersion.Free();
            return status;
        }

        public Status_t Cmd_KS_GetKey(
                                    ushort wKeyNo,
                                    ushort wKeyVersion,
                                    byte bKeyBufSize,
                                    out byte[] pKey,
                                    out ushort wKeyType)
        {
            pKey = new byte[bKeyBufSize];
            wKeyType = 0;
            GCHandle ghKey = GCHandle.Alloc(pKey, GCHandleType.Pinned);
            Status_t status = phhalHw_Rd710_Cmd_KS_GetKey(
                                                        m_pDataParams,
                                                        wKeyNo,
                                                        wKeyVersion,
                                                        bKeyBufSize,
                                                        ghKey.AddrOfPinnedObject(),
                                                        ref wKeyType);
            ghKey.Free();
            return status;
        }

        public Status_t Cmd_KS_SetConfig(
                                        ushort wConfig,
                                        ushort wValue)
        {
            return phhalHw_Rd710_Cmd_KS_SetConfig(
                                                m_pDataParams,
                                                wConfig,
                                                wValue);
        }

        public Status_t Cmd_KS_SetConfigStr(
                                        ushort wConfig,
                                        byte[] pBuffer,
                                        ushort wBufferLength)
        {
            return phhalHw_Rd710_Cmd_KS_SetConfigStr(
                                                    m_pDataParams,
                                                    wConfig,
                                                    pBuffer,
                                                    wBufferLength);
        }

        public Status_t Cmd_KS_GetConfig(
                                        ushort wConfig,
                                        out ushort pValue)
        {
            pValue = 0;
            return phhalHw_Rd710_Cmd_KS_GetConfig(
                                                m_pDataParams,
                                                wConfig,
                                                ref pValue);
        }

        public Status_t Cmd_KS_GetConfigStr(
                                            ushort wConfig,
                                            out byte[] pRxBuffer,
                                            out ushort wRxLength)
        {
            IntPtr pRxBufferInt = IntPtr.Zero;
            wRxLength = 0;
            Status_t status = phhalHw_Rd710_Cmd_KS_GetConfigStr(
                                                            m_pDataParams,
                                                            wConfig,
                                                            ref pRxBufferInt,
                                                            ref wRxLength);
            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }
            return status;
        }

        public Status_t Cmd_KS_ChangeKUC(
                                         ushort wRefNoKUC,
                                         uint dwLimit)
        {
            return phhalHw_Rd710_Cmd_KS_ChangeKUC(
                                               m_pDataParams,
                                               wRefNoKUC,
                                               dwLimit);
        }

        public Status_t Cmd_KS_GetKUC(
                                       ushort wRefNoKUC,
                                       out uint pdwLimit,
                                       out uint pdwCurVal)
        {
            pdwLimit = 0;
            pdwCurVal = 0;
            return phhalHw_Rd710_Cmd_KS_GetKUC(
                                            m_pDataParams,
                                            wRefNoKUC,
                                            ref pdwLimit,
                                            ref pdwCurVal);
        }


        /*
        public Status_t Cmd_CID_GetFreeCid(
                                            byte* pCid)
        {
             Status_t status = phhalHw_Rd710_Cmd_CID_GetFreeCid(
         *                                  m_pDataParams,
                                            byte* pCid);
        }

        public Status_t Cmd_CID_FreeCid(m_pDataParams,
                                            byte bCid      )
        {
            private static extern ushort  phhalHw_Rd710_Cmd_CID_FreeCid(
         *                                  m_pDataParams,
                                            byte bCid);
            return status;
        }

        public Status_t Cmd_I14443P4_SetProtocol(
                                            byte bCidEnable,
                                            byte bCid,
                                            byte bNadEnable,
                                            byte bNad,
                                            byte bFwi,
                                            byte bFsdi,
                                            byte bFsci)
        {
               Status_t status = phhalHw_Rd710_Cmd_I14443P4_SetProtocol(
                                           m_pDataParams,
                                            byte bCidEnable,
                                            byte bCid,
                                            byte bNadEnable,
                                            byte bNad,
                                            byte bFwi,
                                            byte bFsdi,
                                            byte bFsci);
            return status;
        }

        public Status_t Cmd_I14443P4_ResetProtocol()
        {

            Status_t status = phhalHw_Rd710_Cmd_I14443P4_ResetProtocol(m_pDataParams);
            return status;
        }

        public Status_t Cmd_I14443P4_Deselect()
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P4_Deselect(m_pDataParams);
            return status;
        }

        public Status_t Cmd_I14443P4_PresCheck()
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P4_PresCheck(m_pDataParams);
            return status;
        }

        public Status_t I14443P4_Exchange(
                                            ushort wOption,
                                            byte* pTxBuffer,
                                            ushort wTxLength,
                                            byte** ppRxBuffer,
                                            ushort* pRxLength  )
        {
                Status_t status = phhalHw_Rd710_Cmd_I14443P4_Exchange(
                                            m_pDataParams,
                                            ushort wOption,
                                            byte* pTxBuffer,
                                            ushort wTxLength,
                                            byte** ppRxBuffer,
                                            ushort* pRxLength );
            return status;
        }

        public Status_t Cmd_I14443P4_SetConfig(
                                            ushort wConfig,
                                            ushort wValue  )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P4_SetConfig(
                                            m_pDataParams,
                                            ushort wConfig,
                                            ushort wValue );
            return status;
        }

        public Status_t Cmd_I14443P4_GetConfig(
                                            ushort wConfig,
                                            ushort* pValue        )
        {
             Status_t status = phhalHw_Rd710_Cmd_I14443P4_GetConfig(
                                            m_pDataParams,
                                            ushort wConfig,
                                            ushort* pValue  );
            return status;
        }


        public Status_t Cmd_I14443P4A_Rats(
                                            byte bFsdi,
                                            byte bCid,
                                            byte* pAts   )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P4A_Rats(
                                            m_pDataParams,
                                            byte bFsdi,
                                            byte bCid,
                                            byte* pAts   );
            return status;
        }


        public Status_t Cmd_I14443P4A_Pps(
                                            byte bDri,
                                            byte bDsi)
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P4A_Pps(
                                                    m_pDataParams,
                                                    byte bDri,
                                                    byte bDsi);
            return status;
        }

        public Status_t Cmd_I14443P4A_ActivateCard(
                                            byte bFsdi,
                                            byte bCid,
                                            byte bDri,
                                            byte bDsi,
                                            byte* pAts  )
        {

            Status_t status = phhalHw_Rd710_Cmd_I14443P4A_ActivateCard(m_pDataParams,
                                            byte bFsdi,
                                            byte bCid,
                                            byte bDri,
                                            byte bDsi,
                                            byte* pAts  );
            return status;
        }

        public Status_t Cmd_I14443P4A_GetProtocolParams(
                                            byte* pNadSupported,
                                            byte* pFwi,
                                            byte* pFsdi,
                                            byte* pFsci          )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P4A_GetProtocolParams(
                                            m_pDataParams,
                                            byte* pCidEnabled,
                                            byte* pCid,
                                            byte* pNadSupported,
                                            byte* pFwi,
                                            byte* pFsdi,
                                            byte* pFsci);
            return status;
        }

        public Status_t Cmd_I14443P3A_RequestA(
                                            byte* pAtqa     )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_RequestA(
                                            m_pDataParams,
                                            byte* pAtqa	);
            return status;
        }

        public Status_t Cmd_I14443P3A_WakeUpA(
                                            byte* pAtqa )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_WakeUpA(
                                            m_pDataParams,
                                            byte* pAtqa  );
            return status;
        }

        public Status_t Cmd_I14443P3A_HaltA(
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_HaltA(
                                            m_pDataParams );
            return status;
        }

        public Status_t Cmd_I14443P3A_Anticollision(
                                            byte bCascadeLevel,
                                            byte* pUidIn,
                                            byte bNvbUidIn,
                                            byte* pUidOut,
                                            byte* pNvbUidOut    )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_Anticollision(
                                            m_pDataParams,
                                            byte bCascadeLevel,
                                            byte* pUidIn,
                                            byte bNvbUidIn,
                                            byte* pUidOut,
                                            byte* pNvbUidOut	);
            return status;
        }

        public Status_t Cmd_I14443P3A_Select(
                                            byte bCascadeLevel,
                                            byte* pUidIn,
                                            byte* pSak
                                            )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_Select(
                                            m_pDataParams,
                                            byte bCascadeLevel,
                                            byte* pUidIn,
                                            byte* pSak	);
            return status;
        }

        public Status_t Cmd_I14443P3A_ActivateCard(
                                        byte* pUidIn,
                                        byte bLenUidIn,
                                        byte* pUidOut,
                                        byte* pLenUidOut,
                                        byte* pSak,
                                        byte* pMoreCardsAvailable )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_ActivateCard((
                                        m_pDataParams,
                                        byte* pUidIn,
                                        byte bLenUidIn,
                                        byte* pUidOut,
                                        byte* pLenUidOut,
                                        byte* pSak,
                                        byte* pMoreCardsAvailable  );
            return status;
        }

        public Status_t Cmd_I14443P3A_Exchange(
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength )
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_Exchange(
                                        m_pDataParams,
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength	);
            return status;
        }

        public Status_t Cmd_I14443P3A_GetSerialNo(
                                        byte* pRxBuffer,
                                        byte* pRxLength	)
        {
            Status_t status = phhalHw_Rd710_Cmd_I14443P3A_GetSerialNo(
                                        m_pDataParams,
                                        byte* pRxBuffer,
                                        byte* pRxLength);
            return status;
        }

        public Status_t Cmd_MIFARE_ExchangeL3(
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength    )
        {
             Status_t status = phhalHw_Rd710_Cmd_MIFARE_ExchangeL3(
                                        m_pDataParams,
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength	   );
            return status;
        }

        public Status_t Cmd_MIFARE_ExchangeL4(
                                        m_pDataParams,
                                        ushort wOption,
                                        byte* pTxBuffer,
                                        ushort wTxLength,
                                        byte** ppRxBuffer,
                                        ushort* pRxLength  )
        {
            Status_t status = phhalHw_Rd710_Cmd_MIFARE_ExchangeL4(
                                                m_pDataParams,
                                                ushort wOption,
                                                byte* pTxBuffer,
                                                ushort wTxLength,
                                                byte** ppRxBuffer,
                                                ushort* pRxLength	 );
            return status;
        }

        public Status_t Cmd_MIFARE_ExchangePc(
                                    ushort wOption,
                                    byte* pTxBuffer,
                                    ushort wTxLength,
                                    byte** ppRxBuffer,
                                    ushort* pRxLength  )
        {
        Status_t status = phhalHw_Rd710_Cmd_MIFARE_ExchangePc(
                                    m_pDataParams,
                                    ushort wOption,
                                    byte* pTxBuffer,
                                    ushort wTxLength,
                                    byte** ppRxBuffer,
                                    ushort* pRxLength );
            return status;
        }

        public Status_t Cmd_MIFARE_ExchangeRaw(
                                    ushort wOption,
                                    byte* pTxBuffer,
                                    ushort wTxLength,
                                    byte bTxLastBits,
                                    byte** ppRxBuffer,
                                    ushort* pRxLength,
                                    byte* pRxLastBits )
        {
             Status_t status = phhalHw_Rd710_Cmd_MIFARE_ExchangeRaw(
                                    m_pDataParams,
                                    ushort wOption,
                                    byte* pTxBuffer,
                                    ushort wTxLength,
                                    byte bTxLastBits,
                                    byte** ppRxBuffer,
                                    ushort* pRxLength,
                                    byte* pRxLastBits  );
            return status;
        }

        public Status_t Cmd_MIFARE_MfcAuthenticateKeyNo(
                                    byte bBlockNo,
                                    byte bKeyType,
                                    ushort wKeyNo,
                                    ushort wKeyVersion,
                                    byte* pUid  )
        {
            phStatus_t status = phhalHw_Rd710_Cmd_MIFARE_MfcAuthenticateKeyNo(
                                    m_pDataParams,
                                    byte bBlockNo,
                                    byte bKeyType,
                                    ushort wKeyNo,
                                    ushort wKeyVersion,
                                    byte* pUid  );
            return status;
        }

        public Status_t Cmd_MIFARE_MfcAuthenticate(
                                    byte bBlockNo,
                                    byte bKeyType,
                                    byte* pKey,
                                    byte* pUid    )
        {
            Status_t status = phhalHw_Rd710_Cmd_MIFARE_MfcAuthenticate(
                                    m_pDataParams,
                                    byte bBlockNo,
                                    byte bKeyType,
                                    byte* pKey,
                                    byte* pUid );
            return status;
        }
        */



        // continue: Implementation c

        public Status_t Cmd_MIFARE_ExchangeRaw(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            byte bTxLastBits,       /**< [In] number of valid bits of last byte (Tx) */
            out byte[] pRxBuffer,   /**< [Out] received data */
            out byte pRxLastBits    /**< [Out] number of valid bits of last byte (Rx) */
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            pRxLastBits = 0;
            status = phhalHw_Rd710_Cmd_MIFARE_ExchangeRaw(
                m_pDataParams,
                (ushort)wOption,
                pTxBuffer,
                (ushort)pTxBuffer.Length,
                bTxLastBits,
                ref pRxBufferInt,
                ref wRxLength,
                ref pRxLastBits);

            if (wRxLength > 0)
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.Generic pBal, byte bSlotNumber, int wTxBufferSize, int wRxBufferSize)
        {
            // Adjust TxBuffer length
            if ((wTxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }
            else
            {
                wTxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rd710_Init(
                                    m_pDataParams,
                                    (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                                    pBal.m_pDataParams,
                                    bSlotNumber,
                                    m_pTxBuffer.AddrOfPinnedObject(),
                                    (ushort)wTxBufferSize,
                                    m_pRxBuffer.AddrOfPinnedObject(),
                                    (ushort)wRxBufferSize);
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, byte bSlotNumber, int wTxBufferSize, byte[] m_bTxBuffer, int wRxBufferSize, byte[] m_bRxBuffer)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rd710_Init(
                m_pDataParams,
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                bSlotNumber,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#endif
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rd710()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rd710()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
