/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// Rd70x specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class Rd70x : Hal.Generic
    {
        public const int FIFOSIZE = 256;                /**< Size of RC internal FiFo buffer. */
        public const int DEFAULT_TIMEOUT_MS = 5;        /**< Default timeout in microseconds. */
        private const int SHADOW_COUNT = 0x0D;          /**< Number of shadowed configurations. */
        private const int RESERVED_BUFFER_LEN = 32;     /**< Amount of needed and reserved memory for the protocol overhead. */

        public enum Config : int
        {
            DISABLE_PCDRESET = NxpRdLibNet.CustomCodes.CONFIG_BEGIN  /**< Disable PcdReset call in \ref phhalHw_ApplyProtocolSettings function. */
        }

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                      /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;                           /**< pointer to the lower layers parameter structure */
            public IntPtr pTxBuffer;                                /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                               /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                                /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                                /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                               /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                                /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                           /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;                           /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
            public byte bCardType;                                  /**< Type of card for which the hal is configured for */
            public fixed ushort wCfgShadow[SHADOW_COUNT];           /**< Configuration shadow; Stores configuration for current cardtype. */
            public ushort wTimingMode;                              /**< Current timing measurement mode. */
            public uint dwTimeoutUs;                                /**< TO value to be used with phhalHwRd70xPiccExchgBitByte function */
            public uint dwTimingUs;                                 /**< Current timing value. */
            public ushort wAdditionalInfo;                          /**< storage for additional error information */
            public ushort wTxRxStartBackup;                         /**< Absolute Start-Position for transmission / reception used by buffer preservation mechanism. */
            public ushort wFieldOffTime;                            /**< Field-Off-Time in milliseconds. */
            public ushort wFieldRecoveryTime;                       /**< Field-Recovery-Time in milliseconds. */
            public byte bSymbolStart;                               /**< Preamble of a frame. */
            public byte bSymbolEnd;                                 /**< Trailer symbol of a frame. */
            public byte bDisablePcdReset;                           /**< Whether to disable PcdReset call in \ref phhalHw_ApplyProtocolSettings function or not. */
            public byte bFlags;                                     /**< flags to be used with phhalHwRd70xPiccExchgBitByte function */
            public fixed byte bProtBuffer[RESERVED_BUFFER_LEN];     /**< Protocol buffer. */
            public byte bProtBufferLen;                             /**< Length of protocol data. */
            public ushort wProtRxBufReadPos;                        /**< Read position within \c pRxBuffer for protocol. */
            public byte bRfResetAfterTo;                            /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            public byte bTimerControlPc;                            /**< Storage for Current timing mode value. */
            public ushort wFdtPc;                                   /**< Current timing value backup for PC*/
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Init(
            ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize               /**< [In] Size of the global receive buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_EPCBeginRound(
            IntPtr pDataParams,
            byte[] Mask,
            byte Masklength,
            byte Nbslots,
            byte Hash,
            ref ushort Resplen,
            byte[] pRxBuffer
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ReadMultiple(
            IntPtr pDataParams,
            byte[] pAddrValue,
            ushort wLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdConfig(
                                               IntPtr pDataParams
                                               );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdGetAttrib(
            IntPtr pDataParams,
            ref byte FSCImax,
            ref byte FSDImax,
            ref byte DSsupp,
            ref byte DRsupp,
            ref byte DREQDS
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdSetAttrib(
            IntPtr pDataParams,
            byte DSI,
            byte DRI
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdEnableHighBaudRates(
            IntPtr pDataParams,
            byte[] cryptogram
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdSetDefaultAttrib(
            IntPtr pDataParams
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccRequest(
            IntPtr pDataParams,
            byte req_code,
            byte[] atq
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccCommonRequest(
            IntPtr pDataParams,
            byte req_code,
            byte[] atq
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccAnticoll(
            IntPtr pDataParams,
            byte bcnt,
            byte[] snr
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccCascAnticoll(
            IntPtr pDataParams,
            byte select_code,
            byte bcnt,
            byte[] snr
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccSelect(
            IntPtr pDataParams,
            byte[] snr,
            byte[] sak
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccCascSelect(
            IntPtr pDataParams,
            byte select_code,
            byte[] snr,
            ref byte sak
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccAuth(
                                                IntPtr pDataParams,
                                                byte keyAB,
                                                byte key_addr,
                                                byte block
                                                );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccAuthE2(
            IntPtr pDataParams,
            byte auth_mode,
            byte[] snr,
            byte key_sector,
            byte block
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_HostCodeKey(
            IntPtr pDataParams,
            byte[] uncoded,
            byte[] coded
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdLoadKeyE2(
            IntPtr pDataParams,
            byte auth_mode,
            byte key_sector,
            byte[] keys
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccAuthKey(
            IntPtr pDataParams,
            byte auth_mode,
            byte[] snr,
            byte[] keys,
            byte block
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccRead(
                                                IntPtr pDataParams,
                                                byte addr,
                                                byte[] data
                                                );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccCommonRead(
            IntPtr pDataParams,
            byte cmd,
            byte addr,
            byte datalen,
            byte[] data
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccWrite(
            IntPtr pDataParams,
            byte addr,
            byte[] data
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccWrite4(
                                                IntPtr pDataParams,
                                                byte addr,
                                                byte[] data
                                                );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccCommonWrite(
            IntPtr pDataParams,
            byte cmd,
            byte addr,
            byte datalen,
            byte[] data
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccValue(
            IntPtr pDataParams,
            byte dd_mode,
            byte addr,
            byte[] value,
            byte trans_addr
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccValueDebit(
            IntPtr pDataParams,
            byte dd_mode,
            byte addr,
            byte[] value
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_MfPiccHalt(
                                                IntPtr pDataParams
                                                );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccActivation(
            IntPtr pDataParams,
            byte ctrl_flag,
            byte req_code,
            ref byte br,
            byte[] atq,
            ref byte sak,
            byte[] uid,
            ref byte uid_len,
            byte[] script,
            ushort script_len,
            byte[] resp,
            ref ushort resp_len,
            byte sec
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccActivateIdle(
            IntPtr pDataParams,
            byte br,
            byte[] atq,
            byte[] sak,
            byte[] uid,
            ref byte uid_len
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccActivateIdleLoop(
            IntPtr pDataParams,
            byte br,
            byte[] atq,
            byte[] sak,
            byte[] uid,
            ref byte uid_len,
            byte sec
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccActivateWakeup(
            IntPtr pDataParams,
            byte br,
            byte[] atq,
            byte[] sak,
            byte[] uid,
            byte uid_len
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccExchangeBlock(
            IntPtr pDataParams,
            byte[] send_buffer,
            ushort send_bytelen,
            ref IntPtr rec_data,
            ref ushort rec_bytelen,
            byte append_crc,
            uint timeout
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccExchangeBlockRfCycle(
            IntPtr pDataParams,
            byte[] send_buffer,
            ushort send_bytelen,
            ref IntPtr rec_data,
            ref ushort rec_bytelen,
            byte append_crc,
            uint timeout
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccExchgBitByte(
            IntPtr pDataParams,
            byte[] send_buffer,
            ushort send_bytelen,
            byte send_validbits,
            ref IntPtr rec_data,
            ref ushort rec_bytelen,
            ref byte rec_validbits,
            byte flags,
            uint timeout
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PiccExchgBitByteRfCycle(
            IntPtr pDataParams,
            byte[] send_buffer,
            ushort send_bytelen,
            byte send_validbits,
            ref IntPtr rec_data,
            ref ushort rec_bytelen,
            ref byte rec_validbits,
            byte flags,
            uint timeout
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Exchange(
            IntPtr pDataParams,
            byte[] send_buffer,
            ushort send_bytelen,
            ref IntPtr rec_data,
            ref ushort rec_bytelen
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_EPCUID_Exchange(
            IntPtr pDataParams,
            byte bOption,
            byte[] send_buffer,
            ushort send_bytelen,
            byte bTxLastBits,
            ref IntPtr rec_data,
            ref ushort rec_bytelen
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdSetTmo(
                                               IntPtr pDataParams,
                                               uint tmoLength
                                               );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdReset(
                                              IntPtr pDataParams
                                              );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdRfReset(
                                                IntPtr pDataParams,
                                                ushort ms
                                                );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_HostTransTmrStart(
            IntPtr pDataParams
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_HostTransTmrStop(
            IntPtr pDataParams,
            ref uint us
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_HostTransTmrSetCMDCount(
            IntPtr pDataParams,
            ushort lvtime
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_HostGetExecutionTime(
            IntPtr pDataParams,
            ref uint us
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_WriteRC(
                                             IntPtr pDataParams,
                                             byte addr,
                                             byte value
                                             );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ReadRC(
                                            IntPtr pDataParams,
                                            byte addr,
                                            ref byte value
                                            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_WriteMultiple(
            IntPtr pDataParams,
            byte[] addr_value,
            ushort len
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdReadE2(
                                               IntPtr pDataParams,
                                               ushort startaddr,
                                               byte length,
                                               byte[] data
                                               );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdWriteE2(
                                                IntPtr pDataParams,
                                                ushort startaddr,
                                                byte length,
                                                byte[] data
                                                );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdSetIdleMode(
            IntPtr pDataParams);

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdClearIdleMode(
            IntPtr pDataParams);
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdGetFwVersion(
            IntPtr pDataParams,
            byte[] version,
            ref ushort versionLen
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdGetRicVersion(
            IntPtr pDataParams,
            byte[] version
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdInitializeE2(
            IntPtr pDataParams
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ReadWord(
                                              IntPtr pDataParams,
                                              uint address,
                                              ref ushort value
                                              );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_PcdGetSnr(
                                               IntPtr pDataParams,
                                               byte[] snr
                                               );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_SwitchLED(
                                               IntPtr pDataParams,
                                               byte onoff
                                               );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_DbgTrigger(
                                                IntPtr pDataParams,
                                                byte enableTrigger
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_I1PcdConfig(
            IntPtr pDataParams
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_I1PcdRfReset(
            IntPtr pDataParams,
            ushort ms
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_I2init_StdMode_15693(
            IntPtr pDataParams
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_I2init_FastMode_15693(
            IntPtr pDataParams);
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Read_Sm(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            byte blnr,
            byte nbl,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Inventory(
            IntPtr pDataParams,
            byte flags,
            byte AFI,
            byte masklengh,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Write_Sm(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            byte blnr,
            byte nbl,
            byte[] data,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Stay_Quiet(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Lock_Block(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            byte blnr,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Select(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Reset_To_Ready(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Write_AFI(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            byte AFI,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Lock_AFI(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Write_DSFID(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            byte DSFID,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Lock_DSFID(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Get_System_Information(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Get_Multiple_Block_Security(
            IntPtr pDataParams,
            byte flags,
            byte[] uid,
            byte blnr,
            byte nbl,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Inventory_Read(
            IntPtr pDataParams,
            byte flags,
            byte ManCode,
            byte AFI,
            byte masklengh,
            byte[] uid,
            byte blnr,
            byte nbl,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Fast_Inventory_Read(
            IntPtr pDataParams,
            byte flags,
            byte ManCode,
            byte AFI,
            byte masklengh,
            byte[] uid,
            byte blnr,
            byte nbl,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Set_Eas(
            IntPtr pDataParams,
            byte flags,
            byte ManCode,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Reset_Eas(
            IntPtr pDataParams,
            byte flags,
            byte ManCode,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Lock_Eas(
            IntPtr pDataParams,
            byte flags,
            byte ManCode,
            byte[] uid,
            ref ushort resplen,
            byte[] resp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_Eas_Alarm(
            IntPtr pDataParams,
            byte flags,
            byte ManCode,
            byte[] uid,
            byte bEAS_ID_MaskLength,
            byte[] pbEAS_ID,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_EPCPcdConfig(
            IntPtr pDataParams
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_EPCWrite(
                                              IntPtr pDataParams,
                                              byte blnr,
                                              byte data
                                              );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_EPCDestroy(
                                                IntPtr pDataParams,
                                                byte[] epc,
                                                byte[] destroy_code
                                                );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_UIDPcdConfig(
            IntPtr pDataParams
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_UIDBeginRound(
            IntPtr pDataParams,
            byte[] mask,
            byte masklength,
            byte nbrslots,
            ref ushort resplen,
            byte[] resp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_UIDWrite(
                                              IntPtr pDataParams,
                                              byte blnr,
                                              byte data
                                              );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_UIDDestroy(
                                                IntPtr pDataParams,
                                                byte[] idd,
                                                byte[] destroy_code
                                                );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_PwdProtectEAS(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_WriteEAS_ID(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            ushort wEAS_ID,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_ReadEPC(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_GetRandomNumber(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_SetPwd(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bPwdID,
            byte[] pbPwd,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_WritePwd(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bPwdID,
            byte[] pbPwd,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_LockPWD(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bPWD_ID,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_64BitPWDProtection(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            ref ushort pwRespLen,
            byte[] pbResp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_ProtectPage(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bPageNo,
            byte bProtectionStatus,
            ref ushort pwRespLen,
            byte[] pbResp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_LockPageProtectionCondition(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bPageNo,
            ref ushort pwRespLen,
            byte[] pbResp
            );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_DestroyS(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            ref ushort pwRespLen,
            byte[] pbResp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_InventoryReadS(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte bAFI,
            byte bMaskLen,
            byte[] pbUID,
            byte bBlockNo,
            byte bNoOfBlocks,
            ref ushort pwRespLen,
            byte[] pbResp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_FastInventoryReadS(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte bAFI,
            byte bMaskLen,
            byte[] pbUID,
            byte bBlockNo,
            byte bNoOfBlocks,
            ref ushort pwRespLen,
            byte[] pbResp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_GetMultipleBlockProtStatus(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bFirstBlock,
            byte bNoOfBlocks,
            ref ushort pwRespLen,
            byte[] pbResp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rd70x_Cmd_ISO15693_EnablePrivacy(
            IntPtr pDataParams,
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            ref ushort pwRespLen,
            byte[] pbResp
            );

        #endregion

        #region CUSTOM COMMANDS

        public Status_t Cmd_EPCBeginRound(
            byte[] Mask,
            byte Masklength,
            byte Nbslots,
            byte Hash,
            out byte[] pRxBuffer
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            pRxBuffer = new byte[30];

            status = phhalHw_Rd70x_Cmd_EPCBeginRound(m_pDataParams, Mask, Masklength, Nbslots, Hash, ref wRxLength, pRxBuffer);

            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize(ref pRxBuffer, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Cmd_ReadMultiple(
            byte[] Addresses,
            out byte[] Values
            )
        {
            Status_t status;
            int index;
            byte[] bTmp = new byte[Addresses.Length << 1];

            // Set addresses
            for (index = 0; index < Addresses.Length; ++index)
            {
                bTmp[index << 1] = Addresses[index];
            }

            status = phhalHw_Rd70x_Cmd_ReadMultiple(m_pDataParams, bTmp, (ushort)bTmp.Length);

            // Retrieve values
            Values = new byte[Addresses.Length];
            for (index = 0; index < Addresses.Length; ++index)
            {
                Values[index] = bTmp[(index << 1) + 1];
            }

            return status;
        }

        public Status_t Cmd_PcdConfig()
        {
            Status_t status;

            status = phhalHw_Rd70x_Cmd_PcdConfig(m_pDataParams);
            return status;
        }
        public Status_t Cmd_PcdGetAttrib(
            out byte FSCImax,
            out byte FSDImax,
            out byte DSsupp,
            out byte DRsupp,
            out byte DREQDS
            )
        {
            FSCImax = 0;
            FSDImax = 0;
            DSsupp  = 0;
            DRsupp  = 0;
            DREQDS  = 0;

            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdGetAttrib(
                        m_pDataParams,
                        ref FSCImax,
                        ref FSDImax,
                        ref DSsupp,
                        ref DRsupp,
                        ref DREQDS
                        );

            return status;
        }
        public Status_t Cmd_PcdSetAttrib(byte DSI, byte DRI)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdSetAttrib(
                            m_pDataParams,
                            DSI,
                            DRI
                            );
            return status;
        }
        public Status_t Cmd_PcdEnableHighBaudRates(byte[] cryptogram)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdEnableHighBaudRates(
                            m_pDataParams,
                            cryptogram
                            );
            return status;
        }
        public Status_t Cmd_PcdSetDefaultAttrib()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdSetDefaultAttrib(
                            m_pDataParams
                            );
            return status;
        }

        public Status_t Cmd_MfPiccRequest(
            byte req_code,
            out byte[] atq
            )
        {
            Status_t status;
            atq = new byte[16];
            status = phhalHw_Rd70x_Cmd_MfPiccRequest(
                            m_pDataParams,
                            req_code,
                            atq
                            );
            return status;
        }
        public Status_t Cmd_PiccCommonRequest(
            byte req_code,
            out byte[] atq
            )
        {
            Status_t status;
            atq = new byte[16];
            status = phhalHw_Rd70x_Cmd_PiccCommonRequest(
                            m_pDataParams,
                            req_code,
                            atq
                            );
            return status;
        }
        public Status_t Cmd_MfPiccAnticoll(
            byte bcnt,
            out byte[] snr
            )
        {
            Status_t status;
            snr = new byte[4];
            status = phhalHw_Rd70x_Cmd_MfPiccAnticoll(
                            m_pDataParams,
                            bcnt,
                            snr
                            );
            return status;
        }
        public Status_t Cmd_PiccCascAnticoll(
            byte select_code,
            byte bcnt,
            out byte[] snr
            )
        {
            Status_t status;
            snr = new byte[4];
            status = phhalHw_Rd70x_Cmd_PiccCascAnticoll(
                            m_pDataParams,
                            select_code,
                            bcnt,
                            snr
                            );
            return status;
        }
        public Status_t Cmd_MfPiccSelect(
            byte[] snr,
            out byte[] sak
            )
        {
            Status_t status;
            sak = new byte[1];
            status = phhalHw_Rd70x_Cmd_MfPiccSelect(
                            m_pDataParams,
                            snr,
                            sak
                            );
            return status;
        }
        public Status_t Cmd_PiccCascSelect(
            byte select_code,
            byte[] snr,
            ref byte sak
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PiccCascSelect(
                            m_pDataParams,
                            select_code,
                            snr,
                            ref sak
                            );
            return status;
        }
        public Status_t Cmd_MfPiccAuth(
            byte keyAB,
            byte key_addr,
            byte block
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_MfPiccAuth(
                            m_pDataParams,
                            keyAB,
                            key_addr,
                            block
                            );
            return status;
        }
        public Status_t Cmd_MfPiccAuthE2(
            byte auth_mode,
            byte[] snr,
            byte key_sector,
            byte block
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_MfPiccAuthE2(
                            m_pDataParams,
                            auth_mode,
                            snr,
                            key_sector,
                            block
                            );
            return status;
        }
        public Status_t Cmd_HostCodeKey(
            byte[] uncoded,
            byte[] coded
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_HostCodeKey(
                            m_pDataParams,
                            uncoded,
                            coded
                            );
            return status;
        }
        public Status_t Cmd_PcdLoadKeyE2(
            byte auth_mode,
            byte key_sector,
            byte[] keys
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdLoadKeyE2(
                            m_pDataParams,
                            auth_mode,
                            key_sector,
                            keys
                            );
            return status;
        }
        public Status_t Cmd_MfPiccAuthKey(
            byte auth_mode,
            byte[] snr,
            byte[] keys,
            byte block
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_MfPiccAuthKey(
                            m_pDataParams,
                            auth_mode,
                            snr,
                            keys,
                            block
                            );
            return status;
        }
        public Status_t Cmd_MfPiccRead(
            byte addr,
            out byte[] data
            )
        {
            Status_t status;
            data = new byte[16];
            status = phhalHw_Rd70x_Cmd_MfPiccRead(
                            m_pDataParams,
                            addr,
                            data
                            );
            return status;
        }
        public Status_t Cmd_PiccCommonRead(
            byte cmd,
            byte addr,
            byte datalen,
            out byte[] data
            )
        {
            Status_t status;
            if (datalen != 0)
            {
                data = new byte[datalen];
            }
            else
            {
                data = new byte[16];
            }
            status = phhalHw_Rd70x_Cmd_PiccCommonRead(
                            m_pDataParams,
                            cmd,
                            addr,
                            datalen,
                            data
                            );
            return status;
        }
        public Status_t Cmd_MfPiccWrite(
            byte addr,
            byte[] data
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_MfPiccWrite(
                            m_pDataParams,
                            addr,
                            data
                            );
            return status;
        }
        public Status_t Cmd_PiccWrite4(
            byte addr,
            byte[] data
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PiccWrite4(
                            m_pDataParams,
                            addr,
                            data
                            );
            return status;
        }
        public Status_t Cmd_PiccCommonWrite(
            byte cmd,
            byte addr,
            byte datalen,
            byte[] data
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PiccCommonWrite(
                            m_pDataParams,
                            cmd,
                            addr,
                            datalen,
                            data
                            );
            return status;
        }
        public Status_t Cmd_MfPiccValue(
            byte dd_mode,
            byte addr,
            byte[] value,
            byte trans_addr
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_MfPiccValue(
                            m_pDataParams,
                            dd_mode,
                            addr,
                            value,
                            trans_addr
                            );
            return status;
        }
        public Status_t Cmd_MfPiccValueDebit(
            byte dd_mode,
            byte addr,
            byte[] value
            )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_MfPiccValueDebit(
                            m_pDataParams,
                            dd_mode,
                            addr,
                            value
                            );
            return status;
        }
        public Status_t Cmd_MfPiccHalt()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_MfPiccHalt(m_pDataParams);
            return status;
        }
        public Status_t Cmd_PiccActivation(
            byte ctrl_flag,
            byte req_code,
            ref byte br,
            out byte[] atq,
            ref byte sak,
            out byte[] uid,
            byte[] script,
            out byte[] resp,
            byte sec
            )
        {
            Status_t status;
            byte wUidLen = 0;
            ushort wRespLen = 0;

            atq = new byte[2];
            uid = new byte[16];
            resp = new byte[16];

            status = phhalHw_Rd70x_Cmd_PiccActivation(
                m_pDataParams,
                ctrl_flag,
                req_code,
                ref br,
                atq,
                ref sak,
                uid,
                ref wUidLen,
                script,
                (ushort)script.Length,
                resp,
                ref wRespLen,
                sec
                );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
                Array.Resize<byte>(ref uid, wUidLen);
            }
            else
            {
                resp = null;
                uid = null;
            }
            return status;
        }
        public Status_t Cmd_PiccActivateIdle(
            byte br,
            out byte[] atq,
            out byte[] sak,
            out byte[] uid
            )
        {
            Status_t status;
            byte wUidLen = 0;
            atq = new byte[2];
            sak = new byte[1];
            uid = new byte[16];

            status = phhalHw_Rd70x_Cmd_PiccActivateIdle(
                m_pDataParams,
                br,
                atq,
                sak,
                uid,
                ref wUidLen
                );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref uid, wUidLen);
            }
            else
            {
                uid = null;
            }

            return status;

        }
        public Status_t Cmd_PiccActivateIdleLoop(
                    byte br,
                    out byte[] atq,
                    out byte[] sak,
                    out byte[] uid,
                    byte sec
                    )
        {
            Status_t status;
            byte wUidLen = 0;
            atq = new byte[2];
            sak = new byte[1];
            uid = new byte[16];
            status = phhalHw_Rd70x_Cmd_PiccActivateIdleLoop(
                m_pDataParams,
                br,
                atq,
                sak,
                uid,
                ref wUidLen,
                sec
                );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref uid, wUidLen);
            }
            else
            {
                uid = null;
            }
            return status;

        }
        public Status_t Cmd_PiccActivateWakeup(
                    byte br,
                    out byte[] atq,
                    out byte[] sak,
                    byte[] uid
                    )
        {
            Status_t status;

            atq = new byte[2];
            sak = new byte[1];
            uid = new byte[16];

            status = phhalHw_Rd70x_Cmd_PiccActivateWakeup(
                m_pDataParams,
                br,
                atq,
                sak,
                uid,
                (byte)uid.Length
                );
            return status;
        }
        public Status_t Cmd_PiccExchangeBlock(
                    byte[] send_buffer,
                    out byte[] rec_data,
                    byte append_crc,
                    uint timeout
                    )
        {
            Status_t status;
            IntPtr ppRecData = IntPtr.Zero;
            ushort wRecBufLen = 0;

            status = phhalHw_Rd70x_Cmd_PiccExchangeBlock(
                    m_pDataParams,
                    send_buffer,
                    (ushort)send_buffer.Length,
                    ref ppRecData,
                    ref wRecBufLen,
                    append_crc,
                    timeout
                    );

            if (wRecBufLen != 0)
            {
                rec_data = new byte[wRecBufLen];
                Marshal.Copy(ppRecData, rec_data, 0, wRecBufLen);
            }
            else
            {
                rec_data = null;
            }
            return status;
        }
        public Status_t Cmd_PiccExchangeBlockRfCycle(
                    byte[] send_buffer,
                    out byte[] rec_data,
                    byte append_crc,
                    uint timeout
                    )
        {
            Status_t status;
            IntPtr ppRecData = IntPtr.Zero;
            ushort wRecDataLen = 0;

            status = phhalHw_Rd70x_Cmd_PiccExchangeBlockRfCycle(
                m_pDataParams,
                send_buffer,
                (ushort)send_buffer.Length,
                ref ppRecData,
                ref wRecDataLen,
                append_crc,
                timeout
                );

            if (wRecDataLen != 0)
            {
                rec_data = new byte[wRecDataLen];
                Marshal.Copy(ppRecData, rec_data, 0, wRecDataLen);
            }
            else
            {
                rec_data = null;
            }

            return status;
        }
        public Status_t Cmd_PiccExchgBitByte(
                    byte[] send_buffer,
                    byte send_validbits,
                    out byte[] rec_data,
                    ref byte rec_validbits,
                    byte flags,
                    uint timeout
                    )
        {
            Status_t status;
            IntPtr ppRxData = IntPtr.Zero;
            ushort wRxDataLen = 0;
            status = phhalHw_Rd70x_Cmd_PiccExchgBitByte(
                m_pDataParams,
                send_buffer,
                (ushort)send_buffer.Length,
                send_validbits,
                ref ppRxData,
                ref wRxDataLen,
                ref rec_validbits,
                flags,
                timeout
                );
            if (wRxDataLen != 0)
            {
                rec_data = new byte[wRxDataLen];
                Marshal.Copy(ppRxData, rec_data, 0, wRxDataLen);
            }
            else
            {
                rec_data = null;
            }

            return status;
        }
        public Status_t Cmd_PiccExchgBitByteRfCycle(
                    byte[] send_buffer,
                    byte send_validbits,
                    out byte[] rec_data,
                    ref byte rec_validbits,
                    byte flags,
                    uint timeout
                    )
        {
            Status_t status;
            IntPtr ppRxData = IntPtr.Zero;
            ushort wRxDataLen = 0;

            status = phhalHw_Rd70x_Cmd_PiccExchgBitByteRfCycle(
                m_pDataParams,
                send_buffer,
                (ushort)send_buffer.Length,
                send_validbits,
                ref ppRxData,
                ref wRxDataLen,
                ref rec_validbits,
                flags,
                timeout
                );
            if (wRxDataLen != 0)
            {
                rec_data = new byte[wRxDataLen];
                Marshal.Copy(ppRxData, rec_data, 0, wRxDataLen);
            }
            else
            {
                rec_data = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Exchange(
            byte[] send_buffer,
            out byte[] rec_data
            )
        {
            Status_t status;
            IntPtr ppRxData = IntPtr.Zero;
            ushort wRxDataLen = 0;
            status = phhalHw_Rd70x_Cmd_ISO15693_Exchange(
                m_pDataParams,
                send_buffer,
                (ushort)send_buffer.Length,
                ref ppRxData,
                ref wRxDataLen
                );
            if (wRxDataLen != 0)
            {
                rec_data = new byte[wRxDataLen];
                Marshal.Copy(ppRxData, rec_data, 0, wRxDataLen);
            }
            else
            {
                rec_data = null;
            }
            return status;
        }
        public Status_t Cmd_EPCUID_Exchange(
            byte bOption,
            byte[] send_buffer,
            byte bTxLastBits,
            out byte[] rec_data
            )
        {
            Status_t status;
            IntPtr ppRxData = IntPtr.Zero;
            ushort wRxDataLen = 0;

            status = phhalHw_Rd70x_Cmd_EPCUID_Exchange(
                m_pDataParams,
                bOption,
                send_buffer,
                (ushort)send_buffer.Length,
                bTxLastBits,
                ref ppRxData,
                ref wRxDataLen
                );
            if (wRxDataLen != 0)
            {
                rec_data = new byte[wRxDataLen];
                Marshal.Copy(ppRxData, rec_data, 0, wRxDataLen);
            }
            else
            {
                rec_data = null;
            }
            return status;
        }
        public Status_t Cmd_PcdSetTmo(uint tmoLength)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdSetTmo(m_pDataParams, tmoLength);
            return status;
        }
        public Status_t Cmd_PcdReset()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdReset(m_pDataParams);
            return status;
        }
        public Status_t Cmd_PcdRfReset(ushort ms)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdRfReset(m_pDataParams, ms);
            return status;
        }
        public Status_t Cmd_HostTransTmrStart()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_HostTransTmrStart(m_pDataParams);
            return status;
        }
        public Status_t Cmd_HostTransTmrStop(ref uint us)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_HostTransTmrStop(
                            m_pDataParams,
                            ref us
                            );
            return status;
        }
        public Status_t Cmd_HostTransTmrSetCMDCount(ushort lvtime)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_HostTransTmrSetCMDCount(
                            m_pDataParams,
                            lvtime
                            );
            return status;
        }
        public Status_t Cmd_HostGetExecutionTime(ref uint us)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_HostGetExecutionTime(
                            m_pDataParams,
                            ref us
                            );
            return status;
        }
        public Status_t Cmd_WriteRC(byte addr, byte value)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_WriteRC(
                            m_pDataParams,
                            addr,
                            value
                            );
            return status;
        }
        public Status_t Cmd_ReadRC(byte addr, ref byte value)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_ReadRC(
                            m_pDataParams,
                            addr,
                            ref value
                            );
            return status;
        }
        public Status_t Cmd_WriteMultiple(
                    byte[] addr_value
                    )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_WriteMultiple(
                        m_pDataParams,
                        addr_value,
                        (ushort)addr_value.Length
                        );
            return status;
        }
        public Status_t Cmd_PcdReadE2(
            ushort startaddr,
            byte length,
            out byte[] data
            )
        {
            Status_t status;
            data = new byte[length];
            status = phhalHw_Rd70x_Cmd_PcdReadE2(
                m_pDataParams,
                startaddr,
                length,
                data
                );
            return status;
        }
        public Status_t Cmd_PcdWriteE2(ushort startaddr,
                                                        byte length,
                                                        byte[] data
                                                        )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdWriteE2(
                            m_pDataParams,
                            startaddr,
                            length,
                            data
                            );
            return status;
        }
        public Status_t Cmd_PcdSetIdleMode()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdSetIdleMode(m_pDataParams);
            return status;
        }
        public Status_t Cmd_PcdClearIdleMode()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdClearIdleMode(m_pDataParams);
            return status;
        }
        public Status_t Cmd_PcdGetFwVersion(out byte[] version)
        {
            Status_t status;
            ushort wVerLen = 0;
            version = new byte[256];

            status = phhalHw_Rd70x_Cmd_PcdGetFwVersion(
                m_pDataParams,
                version,
                ref wVerLen
                );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref version, wVerLen);
            }
            else
            {
                version = null;
            }
            return status;
        }

        public Status_t Cmd_PcdGetRicVersion(out byte[] version)
        {
            Status_t status;
            version = new byte[6];
            status = phhalHw_Rd70x_Cmd_PcdGetRicVersion(
                m_pDataParams,
                version
                );
            return status;
        }
        public Status_t Cmd_PcdInitializeE2()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_PcdInitializeE2(m_pDataParams);
            return status;
        }
        public Status_t Cmd_ReadWord(uint address, ref ushort value)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_ReadWord(
                            m_pDataParams,
                            address,
                            ref value
                            );
            return status;
        }
        public Status_t Cmd_PcdGetSnr(out byte[] snr)
        {
            Status_t status;
            snr = new byte[4];
            status = phhalHw_Rd70x_Cmd_PcdGetSnr(m_pDataParams, snr);
            return status;
        }
        public Status_t Cmd_SwitchLED(byte onoff)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_SwitchLED(
                            m_pDataParams,
                            onoff
                            );
            return status;
        }

        public Status_t Cmd_DbgTrigger(byte enableTrigger)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_DbgTrigger(
                            m_pDataParams,
                            enableTrigger
                            );
            return status;
        }
        public Status_t Cmd_I1PcdConfig(
             )
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_I1PcdConfig(
                            m_pDataParams
                            );
            return status;

        }
        public Status_t Cmd_I1PcdRfReset(ushort ms)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_I1PcdRfReset(
                            m_pDataParams,
                            ms
                            );
            return status;
        }
        public Status_t Cmd_I2init_StdMode_15693()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_I2init_StdMode_15693(
                            m_pDataParams
                            );
            return status;
        }
        public Status_t Cmd_I2init_FastMode_15693()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_I2init_FastMode_15693(m_pDataParams);
            return status;
        }
        public Status_t Cmd_ISO15693_Read_Sm(
                    byte flags,
                    byte[] uid,
                    byte blnr,
                    byte nbl,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];
            status = phhalHw_Rd70x_Cmd_ISO15693_Read_Sm(
                m_pDataParams,
                flags,
                uid,
                blnr,
                nbl,
                ref wRespLen,
                resp
                );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Inventory(
                    byte flags,
                    byte AFI,
                    byte masklengh,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Inventory(
                            m_pDataParams,
                            flags,
                            AFI,
                            masklengh,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Write_Sm(
                    byte flags,
                    byte[] uid,
                    byte blnr,
                    byte nbl,
                    byte[] data,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Write_Sm(
                            m_pDataParams,
                            flags,
                            uid,
                            blnr,
                            nbl,
                            data,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Stay_Quiet(
                    byte flags,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Stay_Quiet(
                            m_pDataParams,
                            flags,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Lock_Block(
                    byte flags,
                    byte[] uid,
                    byte blnr,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Lock_Block(
                            m_pDataParams,
                            flags,
                            uid,
                            blnr,
                            ref wRespLen,
                             resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Select(
                    byte flags,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Select(
                            m_pDataParams,
                            flags,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Reset_To_Ready(
                    byte flags,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];


            status = phhalHw_Rd70x_Cmd_ISO15693_Reset_To_Ready(
                            m_pDataParams,
                            flags,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Write_AFI(
                    byte flags,
                    byte[] uid,
                    byte AFI,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Write_AFI(
                            m_pDataParams,
                            flags,
                            uid,
                            AFI,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Lock_AFI(
            byte flags,
            byte[] uid,
            out byte[] resp
            )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Lock_AFI(
                            m_pDataParams,
                            flags,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;

        }
        public Status_t Cmd_ISO15693_Write_DSFID(
                    byte flags,
                    byte[] uid,
                    byte DSFID,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Write_DSFID(
                            m_pDataParams,
                             flags,
                            uid,
                            DSFID,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }

        public Status_t Cmd_ISO15693_Lock_DSFID(
                    byte flags,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Lock_DSFID(
                            m_pDataParams,
                            flags,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }

        public Status_t Cmd_ISO15693_Get_System_Information(
                    byte flags,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Get_System_Information(
                            m_pDataParams,
                            flags,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Get_Multiple_Block_Security(
                    byte flags,
                    byte[] uid,
                    byte blnr,
                    byte nbl,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Get_Multiple_Block_Security(
                            m_pDataParams,
                            flags,
                            uid,
                            blnr,
                            nbl,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Inventory_Read(
                    byte flags,
                    byte ManCode,
                    byte AFI,
                    byte masklengh,
                    byte[] uid,
                    byte blnr,
                    byte nbl,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Inventory_Read(
                            m_pDataParams,
                            flags,
                            ManCode,
                            AFI,
                            masklengh,
                            uid,
                            blnr,
                            nbl,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Fast_Inventory_Read(
                    byte flags,
                    byte ManCode,
                    byte AFI,
                    byte masklengh,
                    byte[] uid,
                    byte blnr,
                    byte nbl,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Fast_Inventory_Read(
                            m_pDataParams,
                            flags,
                            ManCode,
                            AFI,
                            masklengh,
                            uid,
                            blnr,
                            nbl,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }

        public Status_t Cmd_ISO15693_Set_Eas(
                    byte flags,
                    byte ManCode,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Set_Eas(
                            m_pDataParams,
                            flags,
                            ManCode,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Reset_Eas(
                    byte flags,
                    byte ManCode,
                    byte[] uid,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Reset_Eas(
                            m_pDataParams,
                            flags,
                            ManCode,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Lock_Eas(
             byte flags,
             byte ManCode,
             byte[] uid,
             out byte[] resp
             )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Lock_Eas(
                            m_pDataParams,
                            flags,
                            ManCode,
                            uid,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_Eas_Alarm(
             byte flags,
             byte ManCode,
             byte[] uid,
             byte bEAS_ID_MaskLength,
             byte[] pbEAS_ID,
             out byte[] resp
             )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_Eas_Alarm(
                            m_pDataParams,
                            flags,
                            ManCode,
                            uid,
                            bEAS_ID_MaskLength,
                            pbEAS_ID,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_EPCPcdConfig()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_EPCPcdConfig(m_pDataParams);
            return status;
        }

        public Status_t Cmd_EPCWrite(byte blnr, byte data)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_EPCWrite(
                            m_pDataParams,
                            blnr,
                            data
                            );
            return status;
        }

        public Status_t Cmd_EPCDestroy(byte[] epc, byte[] destroy_code)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_EPCDestroy(
                            m_pDataParams,
                            epc,
                            destroy_code
                            );
            return status;
        }
        public Status_t Cmd_UIDPcdConfig()
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_UIDPcdConfig(
                            m_pDataParams
                            );
            return status;
        }
        public Status_t Cmd_UIDBeginRound(
                    byte[] mask,
                    byte nbrslots,
                    out byte[] resp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            resp = new byte[24];

            status = phhalHw_Rd70x_Cmd_UIDBeginRound(
                            m_pDataParams,
                            mask,
                            (byte)mask.Length,
                            nbrslots,
                            ref wRespLen,
                            resp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref resp, wRespLen);
            }
            else
            {
                resp = null;
            }
            return status;
        }
        public Status_t Cmd_UIDWrite(byte blnr, byte data)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_UIDWrite(
                            m_pDataParams,
                            blnr,
                            data
                            );
            return status;
        }
        public Status_t Cmd_UIDDestroy(byte[] idd, byte[] destroy_code)
        {
            Status_t status;
            status = phhalHw_Rd70x_Cmd_UIDDestroy(
                            m_pDataParams,
                            idd,
                            destroy_code
                            );
            return status;
        }
        public Status_t Cmd_ISO15693_PwdProtectEAS(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_PwdProtectEAS(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }

            return status;
        }

        public Status_t Cmd_ISO15693_WriteEAS_ID(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    ushort wEAS_ID,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_WriteEAS_ID(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            wEAS_ID,
                            ref wRespLen,
                            pbResp
                            );

            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }

        public Status_t Cmd_ISO15693_ReadEPC(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_ReadEPC(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_GetRandomNumber(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_GetRandomNumber(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_SetPwd(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    byte bPwdID,
                    byte[] pbPwd,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_SetPwd(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            bPwdID,
                            pbPwd,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }

        public Status_t Cmd_ISO15693_WritePwd(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    byte bPwdID,
                    byte[] pbPwd,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_WritePwd(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            bPwdID,
                            pbPwd,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }

        public Status_t Cmd_ISO15693_LockPWD(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    byte bPWD_ID,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_LockPWD(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            bPWD_ID,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }

        public Status_t Cmd_ISO15693_64BitPWDProtection(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_64BitPWDProtection(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_ProtectPage(
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bPageNo,
            byte bProtectionStatus,
            out byte[] pbResp
            )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_ProtectPage(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            bPageNo,
                            bProtectionStatus,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_LockPageProtectionCondition(
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bPageNo,
            out byte[] pbResp
            )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_LockPageProtectionCondition(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            bPageNo,
                            ref wRespLen,
                             pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_DestroyS(
                    byte bFlags,
                    byte bManCode,
                    byte[] pbUID,
                    out byte[] pbResp
                    )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_DestroyS(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_InventoryReadS(
            byte bFlags,
            byte bManCode,
            byte bAFI,
            byte bMaskLen,
            byte[] pbUID,
            byte bBlockNo,
            byte bNoOfBlocks,
            out byte[] pbResp
            )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_InventoryReadS(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            bAFI,
                            bMaskLen,
                            pbUID,
                            bBlockNo,
                            bNoOfBlocks,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_FastInventoryReadS(
            byte bFlags,
            byte bManCode,
            byte bAFI,
            byte bMaskLen,
            byte[] pbUID,
            byte bBlockNo,
            byte bNoOfBlocks,
            out byte[] pbResp
            )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_FastInventoryReadS(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            bAFI,
                            bMaskLen,
                            pbUID,
                            bBlockNo,
                            bNoOfBlocks,
                            ref wRespLen,
                             pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_GetMultipleBlockProtStatus(
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            byte bFirstBlock,
            byte bNoOfBlocks,
            out byte[] pbResp
            )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_GetMultipleBlockProtStatus(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            bFirstBlock,
                            bNoOfBlocks,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }
        public Status_t Cmd_ISO15693_EnablePrivacy(
            byte bFlags,
            byte bManCode,
            byte[] pbUID,
            out byte[] pbResp
            )
        {
            Status_t status;
            ushort wRespLen = 0;
            pbResp = new byte[24];

            status = phhalHw_Rd70x_Cmd_ISO15693_EnablePrivacy(
                            m_pDataParams,
                            bFlags,
                            bManCode,
                            pbUID,
                            ref wRespLen,
                            pbResp
                            );
            if ((Error_Gen)status.Error == Error_Gen.SUCCESS)
            {
                Array.Resize<byte>(ref pbResp, wRespLen);
            }
            else
            {
                pbResp = null;
            }
            return status;
        }


        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.Rd70xUsbWin pBal, int wTxBufferSize, int wRxBufferSize)
        {
            // Adjust TxBuffer length
            if ((wTxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }
            else
            {
                wTxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rd70x_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.Rd70xUsbWin pBal, int wTxBufferSize, int wRxBufferSize)
        {
            // Adjust TxBuffer length
            if ((wTxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }
            else
            {
                wTxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];

            return Init(wDataParamSize, pBal, wTxBufferSize, m_bTxBuffer, wRxBufferSize, m_bRxBuffer);
        }
        public Status_t Init(int wDataParamSize, Bal.Rd70xUsbWin pBal, int wTxBufferSize, byte[] bTxBuffer, int wRxBufferSize, byte[] bRxBuffer)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rd70x_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#endif
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rd70x()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rd70x()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
