/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// RC663 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class Rc663 : Hal.Generic
    {
        #region DEFINES

        public const int DEFAULT_TIMEOUT = 150;     /**< Default timeout in microseconds    */
        public const int SHADOW_COUNT = 0x0010;     /**< Number of shadowed configurations. */

        public enum Config : int
        {
            /** Configure FIFO-Size. */
            FIFOSIZE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,
            LOADREG_MODE
        }

        public enum FifoSize : byte
        {
            VALUE_255 = 0x00,
            VALUE_512
        }

        public enum LpcdMode : byte
        {
            DEFAULT = 0x00,
            POWERDOWN,
            POWERDOWN_GUARDED
        }

        public enum LpcdOption : byte
        {
            NONE = 0x00,
            TRIMM_LPO = 0x80,
            IGNORE_IQ = 0x40
        }

        public enum RxTxOption : byte
        {
            DEFAULT = 0x00,
            TIMER_NOSTART = 0x00,
            TIMER_START = 0x10
        }

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct LoadRegConfig_t
        {
            public byte EEAddressMsb;
            public byte EEAddressLsb;
            public byte bRegAddress;
            public byte bNumBytes;
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                              /**< Layer ID for this HAL layer, NEVER MODIFY! */
            public IntPtr pBalDataParams;                   /**< pointer to the lower layers parameter structure. */
            public byte bLoadRegMode;                       /**< Whether LoadReg-Mode is enabled or not. */
            public IntPtr pLoadRegConfig;                   /**< Pointer to configuration buffer for LoadReg mode; Refer to \ref phhalHw_Rc663_Init. */
            public IntPtr pTxBuffer;                        /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                       /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                        /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                        /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                       /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                        /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                   /**< Starting position within the global receive buffer. */
            public ushort wMaxPrecachedBytes;               /**< Holds the max. number of bytes which are precached prior to command execution. */
            public byte bCardType;                          /**< Type of card for which the hal is configured for. */
            public fixed ushort wCfgShadow[SHADOW_COUNT];   /**< Configuration shadow; Stores configuration for current cardtype. */
            public ushort wTimingMode;                      /**< Current timing measurement mode. */
            public byte bTimeoutUnit;                       /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public uint dwTimingUs;                         /**< Current timing value. */
            public ushort wFieldOffTime;                    /**< Field-Off-Time in milliseconds. */
            public ushort wFieldRecoveryTime;               /**< Field-Recovery-Time in milliseconds. */
            public byte bSymbolStart;                       /**< Preamble of a frame. */
            public byte bSymbolEnd;                         /**< Trailer symbol of a frame. */
            public byte bFifoSize;                          /**< Fifo-Size setting. */
            public ushort wAdditionalInfo;                  /**< storage for additional error information. */
            public byte bBalConnectionType;                 /**< Type of the underlying BAL connection. Refer to #PHHAL_HW_CONFIG_BALCONNECTION. */
            public byte bRfResetAfterTo;                    /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            public ushort wFdtPc;                           /**< Current timing value backup for PC*/
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Init(
            ref DataParams_t pDataParams,       /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,              /**< [In] Pointer to the lower layers parameter structure. */
            ref LoadRegConfig_t pLoadRegConfig, /**< [In] Pointer to configuration buffer for LoadReg mode; See description above. */
            IntPtr pTxBuffer,                   /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,                  /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,                   /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize                   /**< [In] Size of the global receive buffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_Lpcd(
            IntPtr pDataParams,
            byte bMode,
            byte bI,
            byte bQ,
            ushort wPowerDownTimeMs,
            ushort wDetectionTimeUs
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_LpcdConfig(
            IntPtr pDataParams,
            ref byte pI,
            ref byte pQ
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_LoadKey(
            IntPtr pDataParams,
            byte[] pKey
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_Receive(
            IntPtr pDataParams,
            ushort wOption,
            ref IntPtr ppRxBuffer,
            ref ushort pRxLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_Transmit(
            IntPtr pDataParams,
            ushort wOption,
            byte[] pTxBuffer,
            ushort wTxLength
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_WriteE2(
            IntPtr pDataParams,
            ushort wAddress,
            byte bData
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_WriteE2Page(
            IntPtr pDataParams,
            ushort wAddress,
            byte[] pData,
            byte bDataLen
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_ReadE2(
            IntPtr pDataParams,
            ushort wAddress,
            ushort wNumBytes,
            byte[] pData
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_LoadReg(
            IntPtr pDataParams,
            ushort wEEAddress,
            byte bRegAddress,
            byte bNumBytes
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_LoadProtocol(
            IntPtr pDataParams,
            byte bTxProtocol,
            byte bRxProtocol
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_LoadKeyE2(
            IntPtr pDataParams,
            byte bKeyNo
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_StoreKeyE2(
            IntPtr pDataParams,
            byte bKeyNo,
            byte[] pKeys,
            byte bNumberOfKeys
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_SoftReset(
            IntPtr pDataParams
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Rc663_Cmd_AckReq(
            IntPtr pDataParams,
            byte[] pRxBuffer,
            ref ushort wRxBufferLen
            );

        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(
            Bal.Generic pBal,
            int wTxBufferSize,
            int wRxBufferSize
            )
        {
            // Add 1 SPI reserved byte
            if (wTxBufferSize < 0xFFFF)
            {
                ++wTxBufferSize;
            }
            if (wRxBufferSize < 0xFFFF)
            {
                ++wRxBufferSize;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rc663_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                ref m_LoadRegConfigInt[0],
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }

#if DEBUG
        public Status_t Init(
            int wDataParamSize,
            Bal.Generic pBal,
            int wTxBufferSize,
            byte[] m_bTxBuffer,
            int wRxBufferSize,
            byte[] m_bRxBuffer)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Rc663_Init(
                ref m_DataParamsInt[0],
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                ref m_LoadRegConfigInt[0],
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
        }
#endif

        #endregion

        #region GETTER / SETTER

        public LoadRegConfig_t LoadRegConfig
        {
            set
            {
                m_LoadRegConfigInt[0].EEAddressMsb = value.EEAddressMsb;
                m_LoadRegConfigInt[0].EEAddressLsb = value.EEAddressLsb;
                m_LoadRegConfigInt[0].bRegAddress = value.bRegAddress;
                m_LoadRegConfigInt[0].bNumBytes = value.bNumBytes;
            }
            get
            {
                return m_LoadRegConfigInt[0];
            }
        }

        #endregion

        #region INTERFACE

        Status_t SetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            int wValue          /**< [In] Configuration Value */
            )
        {
            return SetConfig((Hal.Config)wConfig, wValue);
        }

        Status_t GetConfig(
            Config wConfig,     /**< [In] Configuration Identifier */
            out int pValue      /**< [Out] Configuration Value */
            )
        {
            return GetConfig((Hal.Config)wConfig, out pValue);
        }

        #endregion

        #region CUSTOM_FUNCTIONS

        public Status_t Cmd_Lpcd(
            LpcdMode bMode,
            LpcdOption bOption,
            byte bI,
            byte bQ,
            int wPowerDownTimeMs,
            int wDetectionTimeUs
            )
        {
            byte bModeInt;
            bModeInt = (byte)((byte)bMode & 0x0F);
            bModeInt |= (byte)((byte)bOption & 0xF0);
            return phhalHw_Rc663_Cmd_Lpcd(m_pDataParams, bModeInt, bI, bQ, (ushort)wPowerDownTimeMs, (ushort)wDetectionTimeUs);
        }

        public Status_t Cmd_LpcdConfig(
            out byte pI,
            out byte pQ
            )
        {
            pI = 0;
            pQ = 0;
            return phhalHw_Rc663_Cmd_LpcdConfig(m_pDataParams, ref pI, ref pQ);
        }

        public Status_t Cmd_LoadKey(
            byte[] pKey
            )
        {
            return phhalHw_Rc663_Cmd_LoadKey(m_pDataParams, pKey);
        }

        public Status_t Cmd_Receive(
            int wOption,
            out byte[] pRxBuffer
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            status = phhalHw_Rc663_Cmd_Receive(m_pDataParams, (ushort)wOption, ref pRxBufferInt, ref wRxLength);

            if ((pRxBufferInt != IntPtr.Zero) && (wRxLength > 0))
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }

        public Status_t Cmd_Transmit(
            int wOption,
            byte[] pTxBuffer
            )
        {
            ushort wTxLength = 0;

            if (pTxBuffer != null)
            {
                wTxLength = (ushort)pTxBuffer.Length;
            }
            return phhalHw_Rc663_Cmd_Transmit(m_pDataParams, (ushort)wOption, pTxBuffer, wTxLength);
        }

        public Status_t Cmd_WriteE2(
            int wAddress,
            byte bData
            )
        {
            return phhalHw_Rc663_Cmd_WriteE2(m_pDataParams, (ushort)wAddress, bData);
        }

        public Status_t Cmd_WriteE2Page(
            int wAddress,
            byte[] pData
            )
        {
            byte bDataLength = 0;
            if (pData != null)
            {
                bDataLength = (byte)pData.Length;
            }
            return phhalHw_Rc663_Cmd_WriteE2Page(m_pDataParams, (ushort)wAddress, pData, bDataLength);
        }

        public Status_t Cmd_ReadE2(
            int wAddress,
            int wNumBytes,
            out byte[] pData
            )
        {
            if (wNumBytes == 0)
            {
                pData = new byte[256];
            }
            else
            {
                pData = new byte[wNumBytes];
            }
            return phhalHw_Rc663_Cmd_ReadE2(m_pDataParams, (ushort)wAddress, (ushort)wNumBytes, pData);
        }

        public Status_t Cmd_LoadReg(
            int wEEAddress,
            byte bRegAddress,
            byte bNumBytes
            )
        {
            return phhalHw_Rc663_Cmd_LoadReg(m_pDataParams, (ushort)wEEAddress, bRegAddress, bNumBytes);
        }

        public Status_t Cmd_LoadProtocol(
            byte bTxProtocol,
            byte bRxProtocol
            )
        {
            return phhalHw_Rc663_Cmd_LoadProtocol(m_pDataParams, bTxProtocol, bRxProtocol);
        }

        public Status_t Cmd_LoadKeyE2(
            byte bKeyNo
            )
        {
            return phhalHw_Rc663_Cmd_LoadKeyE2(m_pDataParams, bKeyNo);
        }

        public Status_t Cmd_StoreKeyE2(
            byte bKeyNo,
            byte[] pKeys
            )
        {
            byte NumberOfKeys = 0x00;

            if (pKeys != null)
            {
                // Parameter check
                if ((pKeys.Length % 6) != 0)
                {
                    return new Status_t(Error_CompCode.HAL, Error_Param.INVALID_PARAMETER);
                }
                else
                {
                    NumberOfKeys = (byte)(pKeys.Length / 6);
                }
            }
            return phhalHw_Rc663_Cmd_StoreKeyE2(m_pDataParams, bKeyNo, pKeys, NumberOfKeys);
        }

        public Status_t Cmd_SoftReset()
        {
            return phhalHw_Rc663_Cmd_SoftReset(m_pDataParams);
        }

        public Status_t Cmd_AckReq(
            out byte[] pRxBuffer
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            pRxBuffer = new byte[0x0F];

            status = phhalHw_Rc663_Cmd_AckReq(m_pDataParams, pRxBuffer, ref wRxLength);

            Array.Resize<byte>(ref pRxBuffer, wRxLength);

            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;
        private LoadRegConfig_t[] m_LoadRegConfigInt;
        private GCHandle m_pLoadRegConfigInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rc663()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
            this.m_LoadRegConfigInt = new LoadRegConfig_t[1];
            this.m_pLoadRegConfigInt = GCHandle.Alloc(this.m_LoadRegConfigInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rc663()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pLoadRegConfigInt.IsAllocated)
            {
                this.m_pLoadRegConfigInt.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
