/*
 * Copyright 2015, 2017 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// PCSC specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class PCSC : Hal.Generic
    {
        #region DEFINES

        private const int RESERVED_BUFFER_LEN = 8;  /**< Amount of needed and reserved memory for the protocol overhead. */

        public enum PcscMode : int
        {
            DEFAULT = 0x0000,
            DIRECT,
        }

        public enum PcscExchangeMode : int
        {
            TRANSPARENT = 0,
            NORMAL = 1,
			UNKNOWN = 255
        }

        public enum PcscKeyLocation : int
        {
            VOLATILE = 0x0000,
            NONVOLATILE,
        }

        public enum PcscConfig : int
        {
            CUSTOM_BEGIN             = 0xE000,
            FSDI                     = (CUSTOM_BEGIN + 0x01),
            FSCI                     = (CUSTOM_BEGIN + 0x02),
            FWTI                     = (CUSTOM_BEGIN + 0x03),
            MAX_SPEED                = (CUSTOM_BEGIN + 0x04),
            CURRENT_SPEED            = (CUSTOM_BEGIN + 0x05),
            MOD_INDEX                = (CUSTOM_BEGIN + 0x06),
            PCB                      = (CUSTOM_BEGIN + 0x07),
            CID                      = (CUSTOM_BEGIN + 0x08),
            NAD                      = (CUSTOM_BEGIN + 0x09),
            DATA_CODING              = (CUSTOM_BEGIN + 0x0B),
            EXCHANGE                 = (CUSTOM_BEGIN + 0x0C),
            STATUS_BYTES             = (CUSTOM_BEGIN + 0x0D),
            KEY_LOCATION             = (CUSTOM_BEGIN + 0x0E),
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                  /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDataParams;       /**< Pointer to the lower layers parameter structure. */
            public IntPtr pKeyStoreDataParams;  /**< Pointer to the Key Store parameter structure. */
            public byte bTxLastBits;            /**< TxLastbits for ExchangeRaw function (indicates number of valid bits of the last byte in the transmit buffer). */
            public IntPtr pTxBuffer;            /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;           /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;            /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;            /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;           /**< Size of the global receive buffer. */
            public ushort wRxBufLen;            /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;       /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;       /**< Starting position within the global transmit buffer (used if \b TxBuffer equals \b RxBuffer). */
            public ushort dwFdtPc;              /**< Current timing value backup for PC. */
            public byte bCardType;              /**< Type of card for which the hal is configured for (if any). */
            public ushort wErrorCode;           /**< SW1 SW2 error code received in response from reader. */
            public byte bRxLastBits;            /**< Number of valid bits in last byte received. */
            public byte bCollPos;               /**< Collision position in received data. */
            public byte bTransparentSession;    /**< Flag to indicate transparent session is open. */
            public int dwTimeoutUs;             /**< Receive timeout in microseconds. */
            public byte bSak;                   /**< Type A SAK; Updated when switch protocol to layer 3A is success. */
            public ushort wFieldOffTime;        /**< Field-Off-Time in milliseconds. */
            public ushort wFieldRecoveryTime;   /**< Field-Recovery-Time in milliseconds. */
            public ushort wTxRxFlag;            /**< Transmission and Reception Flag  to be used with \ref phhalHw_Pcsc_Cmd_TxRxFlag. */
            public byte bExchangeType;          /**< Type of HAL exchange to use. #PHHAL_HW_PCSC_TRANSPARENT (default) or #PHHAL_HW_PCSC_NORMAL. */
            public ushort wAdditionalInfo;      /**< Additional Information. */
            public byte bKeyLocation;           /**< Key location for PCSC load key command (#PHHAL_HW_PCSC_KEY_LOC_VOLATILE or #PHHAL_HW_PCSC_KEY_LOC_NONVOLATILE). */
            public byte bTxDataRate;            /**< Type A TX data rate. */
            public byte bRxDataRate;            /**< Type A RX data rate. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_Pcsc_Init(
            IntPtr pDataParams,             /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure */
            IntPtr pKeyStoreDataParams,     /**< [In] Pointer to Key Store parameter structure. */
            IntPtr pTxBuffer,               /**< [In]Pointer to internal transmit buffer.*/
            ushort wTxBufSize,              /**< [In]internal transmit buffer size.*/
            IntPtr pRxBuffer,               /**< [In]Pointer to internal receive buffer (can be same as transmit buffer).*/
            ushort wRxBufSize               /**< [In] internal receive buffer size.*/
            );
        #endregion


        #region CUSTOM FUNCTIONS
        // continue: Implementation c
/*
        public Status_t Cmd_MIFARE_ExchangeRaw(
            int wOption,
            byte[] pTxBuffer,
            byte bTxLastBits,
            out byte[] pRxBuffer,
            out byte pRxLastBits
            )
        {
            Status_t status;
            ushort wRxLength = 0;
            IntPtr pRxBufferInt = IntPtr.Zero;

            pRxLastBits = 0;
            status = phhalHw_Rd710_Cmd_MIFARE_ExchangeRaw(
                m_pDataParams,
                (ushort)wOption,
                pTxBuffer,
                (ushort)pTxBuffer.Length,
                bTxLastBits,
                ref pRxBufferInt,
                ref wRxLength,
                ref pRxLastBits);

            if (wRxLength > 0)
            {
                pRxBuffer = new byte[wRxLength];
                Marshal.Copy(pRxBufferInt, pRxBuffer, 0, wRxLength);
            }
            else
            {
                pRxBuffer = null;
            }

            return status;
        }
*/
        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init(Bal.Generic pBal, KeyStore.Generic pKeyStore, int wTxBufferSize, int wRxBufferSize)
        {
            IntPtr pKeyStoreDataParams;
            if (pKeyStore != null)
            {
                pKeyStoreDataParams = pKeyStore.m_pDataParams;
            }
            else
            {
                pKeyStoreDataParams = IntPtr.Zero;
            }

            // Adjust TxBuffer length
            if ((wTxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }
            else
            {
                wTxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            return phhalHw_Pcsc_Init(
                m_pDataParams,
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBal.m_pDataParams,
                pKeyStoreDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);
            /*  Call init function
            return phhalHw_Rd710_Init(
                                    m_pDataParams,
                                    (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                                    pBal.m_pDataParams,
                                    bSlotNumber,
                                    m_pTxBuffer.AddrOfPinnedObject(),
                                    (ushort)wTxBufferSize,
                                    m_pRxBuffer.AddrOfPinnedObject(),
                                    (ushort)wRxBufferSize);
			*/
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, Bal.Generic pBal, KeyStore.Generic pKeyStore, int wTxBufferSize, byte[] m_bTxBuffer, int wRxBufferSize, byte[] m_bRxBuffer)
        {
            IntPtr pKeyStoreDataParams;
            if (pKeyStore != null)
            {
                pKeyStoreDataParams = pKeyStore.m_pDataParams;
            }
            else
            {
                pKeyStoreDataParams = IntPtr.Zero;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);

            // Call init function
            return phhalHw_Pcsc_Init(
	        	m_pDataParams,
		        (ushort)wDataParamSize,
                pBal.m_pDataParams,
                pKeyStoreDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize);

            /* return phhalHw_Rd710_Init(
                m_pDataParams,
                (ushort)wDataParamSize,
                pBal.m_pDataParams,
                bSlotNumber,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize); */
        }
#endif
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public PCSC()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~PCSC()
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }
}
