/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// Rd710 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class LicenseeWrapper : Hal.Generic
    {
        #region DEFINES

        // None

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        private struct DataParams_t
        {
            ushort wId;
            public IntPtr pTxBuffer;            /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;           /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;            /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;            /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wRxBufSize;           /**< Size of the global transmit buffer. */
            public ushort wRxBufLen;            /**< Number of valid bytes within the receive buffer. */
            IntPtr pCbInit;                     /**< Pointer to the Init callback. */
            IntPtr pCbTerm;                     /**< Pointer to the Term callback. */
            IntPtr pCbSetConfig;                /**< Pointer to the SetConfig callback. */
            IntPtr pCbGetConfig;                /**< Pointer to the GetConfig callback. */
            IntPtr pCbFieldReset;               /**< Pointer to the FieldReset callback. */
            IntPtr pCbWait;                     /**< Pointer to the Wait callback. */
            IntPtr pCbExchange;                 /**< Pointer to the Exchange callback. */
            IntPtr pCbApplyProtocolSettings;    /**< Pointer to the ApplyProtocolSettings callback. */
            uint LicenseeHalDll;              /**< Handle to the Licensee HAL DLL. */
            IntPtr pCbContext;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_LicenseeWrapper_Init ( ref DataParams_t pDataParams,   /**< [In] Pointer to this layers parameter structure */
                                                                  ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
                                                                  IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
                                                                  ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
                                                                  IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
                                                                  ushort wRxBufSize,              /**< [In] Size of the global receive buffer. */
                                                                  char[] cLicenseeHalDll );        /**< [In] Array containing the path and the name of the Licensee HAL DLL. */


        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phhalHw_LicenseeWrapper_Term ( ref DataParams_t pDataParams );  /**< [In] Pointer to this layers parameter structure */




        #endregion

        #region CUSTOM FUNCTIONS

        // None

        #endregion

        #region MEMORY_MAPPING
        private DataParams_t[] m_DataParamsInt;
        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;


        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public LicenseeWrapper ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~LicenseeWrapper ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }
            if ( this.m_pTxBuffer.IsAllocated )
            {
                this.m_pTxBuffer.Free ();
            }
        }

        #endregion

        #region INIT
        /// <summary>
        /// Initialise this component.
        /// </summary>

        public Status_t Init ( int wTxBufferSize, int wRxBufferSize, string sLicenseeHalDll )
        {
            char[] cLicenseeHalDll = new char[sLicenseeHalDll.Length + 1];
            //GCHandle pLicenseeHalDll;

            // Allocate the TX and RX buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_pTxBuffer = GCHandle.Alloc ( m_bTxBuffer, GCHandleType.Pinned );
            m_pRxBuffer = GCHandle.Alloc ( m_bRxBuffer, GCHandleType.Pinned );

            /* Add Null character to terminate the string */
            sLicenseeHalDll += "\0";
            cLicenseeHalDll = sLicenseeHalDll.ToCharArray ();
            //pLicenseeHalDll = GCHandle.Alloc(cLicenseeHalDll, GCHandleType.Pinned);

            // Call init function
            return phhalHw_LicenseeWrapper_Init ( ref m_DataParamsInt[0],
                                                ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                                m_pTxBuffer.AddrOfPinnedObject (),
                                                ( ushort ) wTxBufferSize,
                                                m_pRxBuffer.AddrOfPinnedObject (),
                                                ( ushort ) wRxBufferSize,
                                                cLicenseeHalDll );
        }

        #endregion

        #region TERM
        /// <summary>
        /// Initialise this component.
        /// </summary>

        public Status_t Term ()
        {
            return phhalHw_LicenseeWrapper_Term ( ref m_DataParamsInt[0] );
        }

        #endregion

        #region MEMORY_MAPPING

        // None

        #endregion
    }
}
