/*
 * Copyright 2019 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.IO;
using System.Text;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace NxpRdLibNet.Hal
{
    /// <summary>
    /// ISO3 specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    public class ISO3 : Hal.Generic
    {
        #region DEFINES
        private const int SHADOW_COUNT = 0x20;                        /**< Number of shadowed configurations. Fixxme maximum config size if no custom paramters are used */
        private const int SHADOW_COUNT_CUSTOM = 0x30;                 /**< Number of shadowed configurations for custom configs. */
        private const int RESERVED_BUFFER_LEN = 32;                   /**< Amount of needed and reserved memory for the protocol overhead. */
        private const int RESERVED_BUFFER_CRC_LEN = 2;                /**< Amount of needed and reserved memory for the crc overhead. */
        private const int WAVESHAPE_BUF_CONFIG_COUNT = 6;             /**< Number of waveshape buffer configs. */
        private const ushort WAVESHAPE_DELIMITER = 0xFFFF;            /**< Delimiter to the end of a wave shape. */
        private const ushort WAVESHAPE_MAX_VALUE = 0xFFFE;            /**< Highest possible value for a wave shape sample. */
        private const ushort MAX_NUMB_WAVESHAPES_SLOTS = 0x001E;      /**< Max possible number of waveshape partitions */
        private const ushort OVS_FILTER_MAX_SLOT_NUM = 0x8;           /**< Max possible number of overshoot filter slots. */

        /* Waveshape prescaling for IIRFilter */
        public const int WAVESHAPE_PRESCALE_PERCENT_FILTER_OFF = 80;
        public const int WAVESHAPE_PRESCALE_PERCENT_FILTER_ON = 45;

        /* Important common apf file offsets */
        const int HEADER_OFFSET = 0x00;
        const int MOD_TYPE_OFFSET = 0x4A;
        const int MOD_TYPE_LENGTH = 0x04;
        readonly ushort[] ENVELOPE_SEPARATOR = {0xFFFF};

        #region MACROS
        private bool IS_EVEN(int x) { if (x % 2 == 0) { return true; } else { return false; } }
        #endregion

        public enum Config : int
        {
            CONFIG_FIELD_STATE                 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 16,
            CONFIG_WAVESHAPE_SLOTS_NUMB        = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 17,
            CONFIG_WAVESHAPE_SLOTS_SIZE        = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 18,
            CONFIG_WAVESHAPE_SLOT_FIELD_ON     = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 19,
            CONFIG_WAVESHAPE_SLOT_FIELD_OFF    = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 20,
            CONFIG_WAVESHAPE_SLOT_EXCHANGE     = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 21,
            CONFIG_FPGA_VERSION                = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 22,
            CONFIG_BIT_BOUNDARY                = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 23,
            CONFIG_EGT                         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 24,
            CONFIG_DUTY_CYCLE                  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 25,
            CONFIG_RX_NOISE_POWER_TH           = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 26,
            CONFIG_IIR_FILTER_CONTROL          = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 28,
            CONFIG_DOUBLE_DATA_RATE            = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 29,
            CONFIG_SEQ8_MODE                   = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 30,
            CONFIG_FIELD_OFF_TIME_US           = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 31,
            CONFIG_DISABLE_AGC                 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 32,
            CONFIG_IIR_FILTER_OVERFLOW         = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 33,
            CONFIG_FIELD_RECOVERY_TIME_US      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 34,
            CONFIG_DEBUG_OUTPUT0               = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 35, /**< Sets the config for the Output 0 */
            CONFIG_DEBUG_OUTPUT1               = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 36, /**< Sets the config for the Output 1 */
            CONFIG_DEBUG_OUTPUT2               = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 37, /**< Sets the config for the Output 2 */
            CONFIG_DEBUG_OUTPUT3               = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 38, /**< Sets the config for the Output 3 */
            CONFIG_WAVESHAPE_PRESCALE_PERCENT  = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 39, /**< Sets the max waveshape pattern percent for waveshapes */
            CONFIG_TX_LAST_BIT                 = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 40, /**< Gets last TX bit */
            CONFIG_PEAK_SENS_SIGNAL            = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 41, /**< Gets SENS Signal peak value */
            CONFIG_PEAK_CAL_SIGNAL             = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 42, /**< Gets CAL Signal peak value */
            CONFIG_OVS_FILTER_SLOT_ENABLE      = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 43, /**< Enable a certain overshoot filter slot. */
            CONFIG_OVS_FILTER_SLOT_DISABLE     = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 44, /**< Disable a certain overshoot filter slot. */
            CONFIG_MILLER_OFFSET               = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 45, /**< Sets the config for Miller Min/Max test */
            CONFIG_FIELD_OFF_TIME_CYCLES       = NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 46, /**< sets the field of time in cw cycles */
        }

        public enum WaveBufConfig : int
        {
            WAVESHAPE_SLOT_SIZE                  = 0,
            WAVESHAPE_SLOT_NUMB                  = 1,
            WAVESHAPE_SLOT_FIELD_ON              = 2,
            WAVESHAPE_SLOT_FIELD_OFF             = 3,
            WAVESHAPE_SLOT_EXCHANGE              = 4,
            WAVESHAPE_SLOT_RFU                   = 5
        }

        public enum TriggerMode : uint
        {
            TRIG_MODE_TX_FIELD_OFF =             0x0U, /**< Trigger event when switching off carrier */
            TRIG_MODE_TX_IDLE =                  0x1U, /**< Trigger event when transmitter enters IDLE STATE */
            TRIG_MODE_TX_SOC =                   0x2U, /**< Trigger event when transmitter modulates SOC */
            TRIG_MODE_TX_LAST_BIT =              0x3U, /**< Trigger event when transmitter modulates last bit of a frame */
            TRIG_MODE_TX_RX_AFTER_LAST_TX_BIT =  0x4U, /**< Trigger event when transmitter activates receiver after last TX bit */
            TRIG_MODE_TX_SOC_RX =                0x5U, /**< Trigger event when SOC was detected by receiver */
            TRIG_MODE_TX_LMA_RX_EVENT =          0x6U, /**< Trigger event when LMA event (8 subcarrier modulations) was detected */
            TRIG_MODE_RX_1ST_MOD =               0xCU, /**< Trigger event when receiver detected first modulation */
            TRIG_MODE_RX_SOC =                   0xDU, /**< Trigger event when receiver detected SOC */
            TRIG_MODE_RX_BIT_NUM =               0xEU, /**< Trigger event when received a certain bit (see TRIGGER_CTRL1 */
            TRIG_MODE_RX_EOC =                   0xFU  /**< Trigger event when recieved EOC */
        };

        public enum Trigger1SignalId : uint
        {
            ADC_INPUT =        0x0U, /**< Signal on ADC input (@ 135.6 MHz) */
            I_CHANNEL_DEMOD =  0x1U, /**< Signal after I channel demodulator */
        };

        public enum Trigger2SignalId : uint
        {
            ADC_INPUT =        0x0U, /**< Signal on ADC input (@ 135.6 MHz) */
            Q_CHANNEL_DEMOD =  0x1U, /**< Signal after Q channel demodulator */
        };

        public enum TxDutyCycle : uint
        {
            DUTY_CYCLE_25 = 0x0U, /**< 25 percent duty cycle */
            DUTY_CYCLE_33 = 0x1U, /**< 33.33 percent duty cycle */
            DUTY_CYCLE_42 = 0x2U, /**< 41.66 percent duty cycle */
            DUTY_CYCLE_50 = 0x3U, /**< 50 percent duty cycle */
            DUTY_CYCLE_58 = 0x4U, /**< 58.33 percent duty cycle */
            DUTY_CYCLE_67 = 0x5U, /**< 66.66 percent duty cycle */
            DUTY_CYCLE_75 = 0x6U  /**< 75 percent duty cycle */
        };

        public enum Iso3Reg : uint
        {
            FPGA_VERSION_REG                 = 0x09, /**< Version of Fpga bitfile */
            VENUS_CLIF_RX_CONFIG_REG         = 0x16, /**< one of the 6 register CLIF_RX_CONFIG_REG addresses used to configure the data rate and card type for the venus receiver */
            VENUS_CLIF_SIGPRO_RM_CONFIG_REG  = 0x17, /**< one of the 6 register CLIF_SIGPRO_RM_CONFIG_REG addresses used to configure the data rate and card type for the venus receiver */
            VENUS_CLIF_SIGPRO_RM_PATTERN_REG = 0x18, /**< one of the 6 register CLIF_SIGPRO_RM_PATTERN_REG addresses used to configure the data rate and card type for the venus receiver */
            VENUS_CLIF_SIGPRO_RM_ENABLES_REG = 0x19, /**< one of the 6 register CLIF_SIGPRO_RM_ENABLES_REG addresses used to configure the data rate and card type for the venus receiver */
            VENUS_CLIF_SIGPRO_RM_TECH_REG    = 0x1a, /**< one of the 6 register CLIF_SIGPRO_RM_TECH_REG addresses used to configure the data rate and card type for the venus receiver */
            VENUS_RM_REG                     = 0x1b, /**< one of the 6 register parts of CLIF_SIGPRO_CONFIG_REG addresses used to configure the data rate and card type for the venus receiver */
            RX_MATCH_REG                     = 0x48, /**< Register to control Rx match */
            RX_MATCH_MASK_REG                = 0x49, /**< Register to control Rx match mask */
            TX_MATCH_REG                     = 0x4A, /**< Register to control Tx match */
            TX_MATCH_MASK_REG                = 0x4B, /**< Register to control Tx match mask */
            TRIGGER_INPUT_REG                = 0x4C, /**< Register to control Config of the bits that are used for as trigger input */
            TRIGGER1_RISE_MASK_REG           = 0x4D, /**< Register to control Mask used to configure the rising edge of the trigger */
            TRIGGER1_FALL_MASK_REG           = 0x4E, /**< Register to control Mask used to configure the falling edge of the trigger */
            TRIGGER1_RISE_EXPECTED_REG       = 0x4F, /**< Register to control expected value used to configure the rising edge of the trigger */
            TRIGGER1_FALL_EXPECTED_REG       = 0x50, /**< Register to control expected value used to configure the falling edge of the trigger */
            TRIGGER1_MODE_REG                = 0x51, /**< Register to control Mode selection of trigger */
            TRIGGER2_RISE_MASK_REG           = 0x52, /**< Register to control Mask used to configure the rising edge of the trigger */
            TRIGGER2_FALL_MASK_REG           = 0x53, /**< Register to control Mask used to configure the falling edge of the trigger */
            TRIGGER2_RISE_EXPECTED_REG       = 0x54, /**< Register to control expected value used to configure the rising edge of the trigger */
            TRIGGER2_FALL_EXPECTED_REG       = 0x55, /**< Register to control expected value used to configure the falling edge of the trigger */
            TRIGGER2_MODE_REG                = 0x56, /**< Register to control Mode selection of trigger */
            TRIGGER3_RISE_MASK_REG           = 0x57, /**< Register to control Mask used to configure the rising edge of the trigger */
            TRIGGER3_FALL_MASK_REG           = 0x58, /**< Register to control Mask used to configure the falling edge of the trigger */
            TRIGGER3_RISE_EXPECTED_REG       = 0x59, /**< Register to control expected value used to configure the rising edge of the trigger */
            TRIGGER3_FALL_EXPECTED_REG       = 0x5A, /**< Register to control expected value used to configure the falling edge of the trigger */
            TRIGGER3_MODE_REG                = 0x5B, /**< Register to control Mode selection of trigger */
            TRIGGER4_RISE_MASK_REG           = 0x5C, /**< Register to control Mask used to configure the rising edge of the trigger */
            TRIGGER4_FALL_MASK_REG           = 0x5D, /**< Register to control Mask used to configure the falling edge of the trigger */
            TRIGGER4_RISE_EXPECTED_REG       = 0x5E, /**< Register to control expected value used to configure the rising edge of the trigger */
            TRIGGER4_FALL_EXPECTED_REG       = 0x5F, /**< Register to control expected value used to configure the falling edge of the trigger */
            TRIGGER4_MODE_REG                = 0x60, /**< Register to control Mode selection of trigger */
            TRIGGER5_RISE_MASK_REG           = 0x61, /**< Register to control Mask used to configure the rising edge of the trigger */
            TRIGGER5_FALL_MASK_REG           = 0x62, /**< Register to control Mask used to configure the falling edge of the trigger */
            TRIGGER5_RISE_EXPECTED_REG       = 0x63, /**< Register to control expected value used to configure the rising edge of the trigger */
            TRIGGER5_FALL_EXPECTED_REG       = 0x64, /**< Register to control expected value used to configure the falling edge of the trigger */
            TRIGGER5_MODE_REG                = 0x65, /**< Register to control Mode selection of trigger */
            RX_EOF_TIMEOUT_CONFIG_REG        = 0x66  /**< Register to control Rx EOF Timeout detection */
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                                         /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalDirectDataParams;                                        /**< pointer to the lower layers parameter structure used to write fpga in and output buffer */
            public IntPtr pBalDispatchDataParams;                                      /**< pointer to the lower layers parameter structure used to communicate with server dispatcher */
            public IntPtr pTxBuffer;                                                   /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;                                                  /**< Size of the global transmit buffer. */
            public ushort wTxBufLen;                                                   /**< Number of valid bytes within the transmit buffer. */
            public IntPtr pRxBuffer;                                                   /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;                                                  /**< Size of the global receive buffer. */
            public ushort wRxBufLen;                                                   /**< Number of valid bytes within the receive buffer. */
            public ushort wRxBufStartPos;                                              /**< Starting position within the global receive buffer. */
            public ushort wTxBufStartPos;                                              /**< Starting position within the global transmit buffer (used if TxBuffer = RxBuffer). */
            public ushort wRxExpectedInputBytes;                                       /**< Expected size of exchange response in bytes */
            public IntPtr pIntBuffer;                                                  /**< Internal fpga intput buffer */
            public ushort wIntNumValidBytes;                                           /**< Number of valid bytes within the input buffer */
            public ushort wIntNumValidRxBytes;                                         /**< Number of valid bytes within the input buffer after bal receive */
            public ushort wMaxIntBufferSizeBytes;                                      /**< Size of fpga input buffer */
            public byte bCardType;                                                     /**< Type of card for which the HAL is configured for. */
            public fixed ushort wCfgShadow[SHADOW_COUNT];                              /**< Configuration shadow; Stores configuration for current cardtype. */
            public fixed ushort wCfgShadowCustom[SHADOW_COUNT_CUSTOM];                 /**< Configuration shadow for Custom configs; Stores configuration for current cardtype. */
            public ushort wTimingMode;                                                 /**< Current timing measurement mode. */
            public byte bLastBit;                                                      /**< Last bit exchanged (0 if last bit was Zero 1 if last bit was One. */
            public ushort wCurrentFieldStrength;                                       /**< Current Fieldstrength in Permil. */
            public uint dwTxCtrl;                                                      /**< Copy of the current state of the TX CTRL register. */
            public uint dwTxCtrl1;                                                     /**< Copy of the current state of the TX CTRL 1 register. */
            public uint dwRxCtrl2;                                                     /**< Copy of the current state of the RX CTRL 2 register. */
            public uint dwVenusClifRxConfigReg;                                        /**< Copy of the current state of the PHHAL_HW_ISO3_REG_VENUS_CLIF_RX_CONFIG_REG register. */
            public uint dwExchangeCounter;                                             /**< To assign every received packet to a transmitted packet */
            public uint dwNextTxBufferAddress;                                         /**< either address for TX_DATA or TX_DATA_1. changes after every exchange! */
            public uint dwCurrentWaveShape;                                            /**< id of curren wave shape. e.g. type A, field on ramp, field off ramp... */
            public ushort wLastFdtValueT1;                                             /**< measured fdt value of last successfull exchange */
            public ushort wLastFdtValueT2;                                             /**< slot list */
            public uint dwDacGain;                                                     /**< Gain factor for envelope to control field strength */
            public fixed ushort wWaveShapeBufferConfigs[WAVESHAPE_BUF_CONFIG_COUNT];      /**< attenuation values */
            public fixed byte bWaveShapeBufferSlots[(MAX_NUMB_WAVESHAPES_SLOTS + 7) / 8];     /**< slot list */
            public byte bCurrentWaveshapeSlot;                                         /**< Slot ID of current wave shape */
            public uint dwIIRFilterCtrl;                                               /**< register to control the IIR filter */
            public uint dwMlsWhiteNoiseCtrlReg;                                        /**< Register to control the white noise generator extension in the transmitter */
            public uint dwRxStatReg;                                                   /**< Register containing the receiver status */
            public uint dwDgrmSigDetTh;                                                /**< threshold of the signal detector in the venus receiver */
            public uint dwPicc2PcdDelay;                                               /**< inverse FDT delay in ETUs */
            public ushort wAdditionalInfo;                                             /**< Storage for additional error information. */
            public byte bRfResetAfterTo;                                               /**< Storage for #PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT setting. */
            public uint dwFieldOffTimeCycles;                                          /**< Field-Off-Time in carrier periods */
            public byte bDisableAGC;                                                   /**< Disables Automatic Gain control for exchange, rf on/off, field reset */
            public uint dwFieldRecoveryTimeCycles;                                     /**< Field-Recovery time in carrier periods */
            public IntPtr pWaveShapeShadowBuffer;
            public uint dwWaveShapeBufferSizeByte;
            public uint dwPcd2PiccTimeout;                                             /**< max time for duration {Tx Last bit, Rx EOF} in FPGA clock periods */
            public ushort wDebugConfig0;                                               /**< Config for the debug outputs */
            public ushort wDebugConfig1;                                               /**< Config for the debug outputs */
            public ushort wDebugConfig2;                                               /**< Config for the debug outputs */
            public ushort wDebugConfig3;                                               /**< Config for the debug outputs */
            public byte bTimeoutUnit;                                                  /**< Unit of current timeout value (either #PHHAL_HW_TIME_MICROSECONDS or #PHHAL_HW_TIME_MILLISECONDS). */
            public ushort WaveShapePrescalePercent;                                    /**< Sets the max pattern percent for waveshapes */
            public uint dwPeakSensSignal;                                              /**< Peak SENS signal value */
            public uint dwPeakCalSignal;                                               /**< Peak CAL signal value */
            public fixed byte pbOvsFilterSlotsBitmap[(OVS_FILTER_MAX_SLOT_NUM + 7) / 8];     /**< slot list for overshoot filter coefficients. Bitmat indicates if slot is in use. */
        };

        enum ModulationType_e
        {
            None,
            A,
            B,
            F,
            I
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_InitAll(
            IntPtr pDataParams,   /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDirectDataParams,    /**< [In] Pointer to the lower layers parameter structure for direct fpga read write. */
            IntPtr pBalDispatchDataParams,  /**< [In] Pointer to the lower layers parameter structure for communicating with server dispatcher. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,                /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize,                /**< [In] Size of the global receive buffer. */
            IntPtr pInternalBuffer,         /**< [In] Internal buffer used as a scratch buffer */
            uint wInternalBufferLen,       /**< [In] Length of innternal buffer used as a scratch buffer */
            IntPtr pWaveShapeShadowBuffer,
            uint dwWaveShapeBufferSizeByte
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_InitFpga(
            IntPtr pDataParams
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_WriteFpgaRegister(
            IntPtr pDataParams,
            uint dwAddr,
            byte[] dwValue,
            uint size
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_ReadFpgaRegister(
            IntPtr pDataParams,
            uint dwAddr,
            byte[] pValue,
            uint size
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_LoadOVSCoefficients(
            IntPtr pDataParams,
            uint[] pCoefficients,
            ushort wCoefficientsSize,
            byte CoefficientsSlot
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_LoadWaveShape(
            IntPtr pDataParams,
            UInt16[] pWaveShape,
            ushort wWaveShapeSize,
            byte bWaveShapeType
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_ReadFpgaSignalData(
            IntPtr pDataParams,
            UInt16[] pwSig1Data,
            UInt16[] pwSig2Data,
            UInt16[] pwSize
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_WriteFpgaOutBuffer(
            IntPtr pDataParams,
            byte[] pValue,
            UInt16 wSize,
            byte[] pResp
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_PiccRequest(
            IntPtr pDataParams,
            UInt16[] pwSig1Data,
            UInt16[] pwSig2Data,
            UInt16[] pwSize
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_PlaySeq8(
            ref DataParams_t pDataParams
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_GetPLSVersion(
            IntPtr pDataParams,
            ref ushort pwVersion,
            ref ushort pwRevision
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_GetDeviceDna(
            IntPtr pDataParams,
            ref ulong pdna
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_GetAgcStatistic(
            IntPtr pDataParams,
            byte bReset,
            ref UInt32 pdwMinAgc,
            ref UInt32 pdwMaxAgc
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_SetConfig32(
                                          IntPtr pDataParams,   /**< [In] Pointer to this layers parameter structure */
                                          ushort wConfig,       /**< [In] Configuration Identifier */
                                          UInt32 wValue         /**< [In] Configuration Value */
                                          );
        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalHw_ISO3_Cmd_GetConfig32(
                                          IntPtr pDataParams,   /**< [In] Pointer to this layers parameter structure */
                                          ushort wConfig,       /**< [In] Configuration Identifier */
                                          ref UInt32 pValue     /**< [Out] Configuration Value */
                                          );

        #endregion

        #region INIT
        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;
        private byte[] m_bIntBuffer;
        private GCHandle m_pIntBuffer;

        private byte[] m_bWaveShapeShadowBuffer;
        private GCHandle m_pWaveShapeShadowBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>

        public Status_t Init(Bal.Generic pBalDirect, Bal.Generic pBalDispatch, int wTxBufferSize, int wRxBufferSize)
        {
            int wIntBufferSize = 0;
            // Adjust IntBuffer length
            if ((2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wIntBufferSize = 0xFFFF;
            }
            else
            {
                wIntBufferSize = 2 * Math.Max(wTxBufferSize, wRxBufferSize) + RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ((wRxBufferSize + RESERVED_BUFFER_CRC_LEN) > 0xFFFF)
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RESERVED_BUFFER_CRC_LEN;
            }

            // Adjust TxBuffer length
            if ((wTxBufferSize + RESERVED_BUFFER_LEN) > 0xFFFF)
            {
                wTxBufferSize = 0xFFFF;
            }
            else
            {
                wTxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pRxBuffer.IsAllocated)
            {
                this.m_pRxBuffer.Free();
            }
            if (this.m_pIntBuffer.IsAllocated)
            {
                this.m_pIntBuffer.Free();
            }

            if (this.m_pWaveShapeShadowBuffer.IsAllocated)
            {
                this.m_pWaveShapeShadowBuffer.Free();
            }

            // Allocate buffers
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_bIntBuffer = new byte[wIntBufferSize];

            m_bWaveShapeShadowBuffer = new byte[0xffff];

            for (int i = 0; i < m_bTxBuffer.Length; i++)
            {
                m_bTxBuffer[i] = 0x0000;
            }
            for (int i = 0; i < m_bRxBuffer.Length; i++)
            {
                m_bRxBuffer[i] = 0x0000;
            }
            for (int i = 0; i < m_bIntBuffer.Length; i++)
            {
                m_bIntBuffer[i] = 0x0000;
            }

            for (int i = 0; i < m_bWaveShapeShadowBuffer.Length; i++)
            {
                m_bWaveShapeShadowBuffer[i] = 0x0000;
            }

            // Link given buffers
            m_pTxBuffer = GCHandle.Alloc(m_bTxBuffer, GCHandleType.Pinned);
            m_pRxBuffer = GCHandle.Alloc(m_bRxBuffer, GCHandleType.Pinned);
            m_pIntBuffer = GCHandle.Alloc(m_bIntBuffer, GCHandleType.Pinned);

            m_pWaveShapeShadowBuffer = GCHandle.Alloc(m_bWaveShapeShadowBuffer, GCHandleType.Pinned);

            return phhalHw_ISO3_InitAll(m_pDataParams,
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pBalDirect.m_pDataParams,
                pBalDispatch.m_pDataParams,
                m_pTxBuffer.AddrOfPinnedObject(),
                (ushort)wTxBufferSize,
                m_pRxBuffer.AddrOfPinnedObject(),
                (ushort)wRxBufferSize,
                m_pIntBuffer.AddrOfPinnedObject(),
                (uint)wIntBufferSize,
                m_pWaveShapeShadowBuffer.AddrOfPinnedObject(),
                (uint)m_bWaveShapeShadowBuffer.Length);
        }


        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t WriteFpgaRegister(uint addr, byte[] value)
        {
            return phhalHw_ISO3_Cmd_WriteFpgaRegister(m_pDataParams, addr, value, (uint)value.Length);
        }

        public Status_t ReadFpgaRegister(uint addr, out byte[] value)
        {
            value = new byte[4];
            return phhalHw_ISO3_Cmd_ReadFpgaRegister(m_pDataParams, addr, value, 4);
        }

        public Status_t CmdWriteFpgaOutBuffer(byte[] value, UInt16 size, out byte[] response)
        {
            response = new byte[size];
            return phhalHw_ISO3_Cmd_WriteFpgaOutBuffer(m_pDataParams, value, size, response);
        }

        public Status_t InitFpga()
        {
            return phhalHw_ISO3_Cmd_InitFpga(m_pDataParams);
        }

        public Status_t LoadOVSCoefficients(int[] osf_data_apf, byte slot)
        {
            /* Get the coefficients in the right order (how they are stored into the FPGA registers - increasing address) */
            uint[] coefficients = new uint[17];
            int[] address = { 4, 5, 6, -1, 2, 3, 9, 10, 11, -1, 7, 8, 14, 15, 16, -1, 12, 13, 0, 1 };
            for (int i = 0; i < 17; i++)
            {
                coefficients[i] = (uint)osf_data_apf[Array.IndexOf(address, i)];
            }

            /* Load Coefficients to FPGA */
            return phhalHw_ISO3_Cmd_LoadOVSCoefficients(m_pDataParams, coefficients, (ushort)(coefficients.Length * sizeof(uint)), slot);
        }

        public Status_t LoadWaveShape(UInt16[] waveshape, byte slot)
        {
            return phhalHw_ISO3_Cmd_LoadWaveShape(m_pDataParams, waveshape, (ushort)(waveshape.Length * 2), slot);
        }

        public Status_t LoadWaveShapeTxt(String waveShapeFileFall, String waveShapeFileRise, byte slot)
        {
            /* Init status with default value */
            Status_t status = new Status_t();

            int shapeSeparatorOffset = 0;

            try
            {
                List<double> fall = new List<double>();
                List<double> rise = new List<double>();
                using (var reader = new StreamReader(waveShapeFileFall))
                {
                    while (!reader.EndOfStream)
                    {
                        var line = reader.ReadLine();
                        fall.Add(Convert.ToDouble(line));
                    }
                }

                if (!string.IsNullOrEmpty(waveShapeFileRise))
                {
                    using (var reader = new StreamReader(waveShapeFileRise))
                    {
                        while (!reader.EndOfStream)
                        {
                            var line = reader.ReadLine();
                            rise.Add(Convert.ToDouble(line));
                        }
                    }
                }

                double[] envelope = null;

                if (rise.Count == 0)
                {
                    envelope = fall.ToArray();
                }
                else
                {
                    int extraPaddingSize = 0;

                    /* if envelope1 is of odd length, pad last byte to get even length */
                    if (fall.Count % 2 == 1)
                    {
                        extraPaddingSize = 1;
                    }

                    /* remember offset for later on */
                    shapeSeparatorOffset = fall.Count + extraPaddingSize;

                    envelope = new double[fall.Count + extraPaddingSize + ENVELOPE_SEPARATOR.Length + rise.Count];
                    Array.Copy(fall.ToArray(), 0, envelope, 0, fall.Count);
                    if (extraPaddingSize == 1)
                    {
                        envelope[fall.Count] = envelope[fall.Count - 1];
                    }
                    /* skip over one word that will have the separator filled in later */
                    Array.Copy(rise.ToArray(), 0, envelope, fall.Count + extraPaddingSize + ENVELOPE_SEPARATOR.Length, rise.Count);
                }

                ushort[] waveshape;
                status = PrepareWaveshape(envelope, shapeSeparatorOffset, out waveshape);
                if (!status.Equals(new Status_t()))
                    return status;

                /* Load waveshape to FPGA */
                status = phhalHw_ISO3_Cmd_LoadWaveShape(m_pDataParams, waveshape, (ushort)(waveshape.Length * 2), slot);
                if (!status.Equals(new Status_t()))
                    return status;

                /* Disable overshoot filter */
                SetConfig((Hal.Config)Config.CONFIG_OVS_FILTER_SLOT_DISABLE, slot);
            }
            catch (System.IO.FileNotFoundException)
            {
                status = new Status_t(Error_CompCode.HAL, Error_Param.INVALID_PARAMETER);
            }
            catch (Exception)
            {
                status = new Status_t(Error_CompCode.HAL, Error_Comm.INTERNAL_ERROR);
            }

            return status;
        }

        public Status_t LoadWaveShape(String waveShapeFile, byte slot)
        {
            /* Init status with default value */
            Status_t status = new Status_t();

            /* Important common file offsets */
            const int HEADER_OFFSET = 0x00;
            const int MOD_TYPE_OFFSET = 0x4A;
            const int MOD_TYPE_LENGTH = 0x04;
            int shapeSeparatorOffset = 0;

            try
            {
                /* Read file contents into byte array content */
                FileStream fs = new FileStream(waveShapeFile, FileMode.Open);
                BinaryReader br = new BinaryReader(fs);
                byte[] content = br.ReadBytes((int)fs.Length);

                /* Check file length before accessing these offsets later on */
                if (content.Length < (MOD_TYPE_OFFSET + MOD_TYPE_LENGTH))
                {
                    throw new Exception("Wrong file size");
                }

                if ((content[HEADER_OFFSET] != 'A') || (content[HEADER_OFFSET + 1] != 'P') ||
                    (content[HEADER_OFFSET + 2] != 'F') || (content[HEADER_OFFSET + 3] != '#'))
                {
                    throw new Exception("Unknown file format");
                }

                /* Depeding on modulation type, apf files are handled differently */
                ModulationType_e modType = GetModulationType(ref content);

                double[] envelope = null;

                if ((modType == ModulationType_e.A) || (modType == ModulationType_e.I))
                {
                    GetStructFromAPF("envelope", ref content, out envelope);
                }
                else if ((modType == ModulationType_e.B) || (modType == ModulationType_e.F))
                {
                    double[] envelope1, envelope2;
                    int extraPaddingSize = 0;

                    GetStructFromAPF("envelope1", ref content, out envelope1);
                    GetStructFromAPF("envelope2", ref content, out envelope2);

                    /* if envelope1 is of odd length, pad last byte to get even length */
                    if (envelope1.Length % 2 == 1)
                    {
                        extraPaddingSize = 1;
                    }

                    /* remember offset for later on */
                    shapeSeparatorOffset = envelope1.Length + extraPaddingSize;

                    envelope = new double[envelope1.Length + extraPaddingSize + ENVELOPE_SEPARATOR.Length + envelope2.Length];
                    Array.Copy(envelope1, 0, envelope, 0, envelope1.Length);
                    if (extraPaddingSize == 1)
                    {
                        envelope[envelope1.Length] = envelope[envelope1.Length - 1];
                    }
                    /* skip over one word that will have the separator filled in later */
                    Array.Copy(envelope2, 0, envelope, envelope1.Length + extraPaddingSize + ENVELOPE_SEPARATOR.Length, envelope2.Length);
                }

                ushort[] waveshape;
                status = PrepareWaveshape(envelope, shapeSeparatorOffset, out waveshape);
                if (!status.Equals(new Status_t()))
                    return status;

                /* Load waveshape to FPGA */
                status = phhalHw_ISO3_Cmd_LoadWaveShape(m_pDataParams, waveshape, (ushort)(waveshape.Length * 2), slot);
                if (!status.Equals(new Status_t()))
                    return status;

                /* Load overshoot filter */
                int[] osf_data_apf;
                int osf_success = GetOsfCoefsFromAPF(ref content, out osf_data_apf); // 18 lsb's are used

                if ((osf_success != 0) || (osf_data_apf.Length < 17))
                {
                    // disable overshoot filter slot
                    SetConfig((Hal.Config)Config.CONFIG_OVS_FILTER_SLOT_DISABLE, slot);
                }
                else
                {
                    LoadOVSCoefficients(osf_data_apf, slot);
                    SetConfig((Hal.Config)Config.CONFIG_OVS_FILTER_SLOT_ENABLE, slot);
                }

                fs.Close();
            }
            catch (System.IO.FileNotFoundException)
            {
                status = new Status_t(Error_CompCode.HAL, Error_Param.INVALID_PARAMETER);
            }
            catch (Exception)
            {
                status = new Status_t(Error_CompCode.HAL, Error_Comm.INTERNAL_ERROR);
            }

            return status;
        }

        private Status_t PrepareWaveshape(double[] envelope, int shapeSeparatorOffset, out ushort[] waveshape)
        {
            int wSlotSizeBytes;
            /* Convert doubles into byte array */
            /* [double p1][double p2] -> [p2 low][p2 high][p1 low][p1 high] */
            /* Scale to the required % depending on whether the IIRFilter is switched ON/OFF to have head room as some shapes may overshoot */
            int WaveShapePrescalePercent;
            GetConfig((Hal.Config) Config.CONFIG_WAVESHAPE_PRESCALE_PERCENT, out WaveShapePrescalePercent);
            double scale = (double) WaveShapePrescalePercent/100.0;
            ushort maxValue = 0xFFFE;
            ushort defaultValue = (ushort) (maxValue*scale);
            uint waveshape_length = 0;

            /* If envelope has even size add 2 else add 3 bytes of length */
            if (envelope.Length%2 == 0)
            {
                waveshape_length = (uint) (envelope.Length + 2);
            }
            else
            {
                waveshape_length = (uint) (envelope.Length + 3);
            }

            GetConfig((Hal.Config) Config.CONFIG_WAVESHAPE_SLOTS_SIZE, out wSlotSizeBytes);

            /* Check wave shape length */
            if (waveshape_length > ((uint) wSlotSizeBytes/2))
            {
                waveshape = null;
                return new Status_t(Error_CompCode.HAL, Error_Param.INVALID_PARAMETER);
            }

            waveshape = new ushort[waveshape_length];
            ushort index = 0;

            /* Limit waveshape values to 0 - maxValue */
            for (uint i = 0; i < envelope.Length; i++)
            {
                if (envelope[i]*defaultValue > maxValue)
                {
                    waveshape[index++] = maxValue;
                }
                else if (envelope[i] < 0)
                {
                    waveshape[index++] = 0;
                }
                else
                {
                    waveshape[index++] = (ushort) (envelope[i]*defaultValue);
                }
            }

            /* If envelope was of odd length, repeat last value as padding */
            if (envelope.Length%2 == 1)
            {
                waveshape[index++] = (ushort) (envelope[envelope.Length - 1]*defaultValue);
            }

            /* Add delimiter at last two bytes */
            waveshape[index++] = 0xFFFF;
            waveshape[index++] = (ushort) (envelope[envelope.Length - 1]*defaultValue);

            /* If we dealt with separate envelopes, put in separator FFFF now */
            if (shapeSeparatorOffset != 0)
            {
                Array.Copy(ENVELOPE_SEPARATOR, 0, waveshape, shapeSeparatorOffset, ENVELOPE_SEPARATOR.Length);
            }
            return new Status_t();
        }

        public Status_t PlaySeq8()
        {
            return phhalHw_ISO3_Cmd_PlaySeq8(ref m_DataParamsInt[0]);
        }

        private ModulationType_e GetModulationType(ref byte[] content)
        {
            ModulationType_e modType = ModulationType_e.None;

            if ((content[MOD_TYPE_OFFSET] == 'P') && (content[MOD_TYPE_OFFSET + 1] == 'T') && (content[MOD_TYPE_OFFSET + 2] == '_'))
            {
                switch ((char)content[MOD_TYPE_OFFSET + 3])
                {
                    case 'A':
                        modType = ModulationType_e.A;
                        break;
                    case 'B':
                        modType = ModulationType_e.B;
                        break;
                    case 'F':
                        modType = ModulationType_e.F;
                        break;
                    case 'I':
                        modType = ModulationType_e.I;
                        break;
                    default:
                        throw new Exception("Unknown modulation type " + (char)content[MOD_TYPE_OFFSET + 3] + " found");
                }
            }
            else
            {
                throw new Exception("No modulation type found");
            }

            return modType;
        }

        private int GetOsfCoefsFromAPF(ref byte[] content, out int[] coefs_out)
        {
            int return_val = 0;
            /************* COEFFICIENTS *************/

            /* convert the name as string to a byte array */
            byte[] bytes_g1 = Encoding.ASCII.GetBytes("filter_cofs_g1");
            byte[] bytes_g2 = Encoding.ASCII.GetBytes("filter_cofs_g2");
            byte[] bytes_g3 = Encoding.ASCII.GetBytes("filter_cofs_g3");

            int coefs_offset = 56;
            bool[,] coefs = new bool[18, 18];
            int[] cofs = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };

            /* get the position of the g1 filter bitstream in the content byte array */
            int position_g1 = CofsPos(content, bytes_g1, coefs_offset);
            if (position_g1 != -1)
            {
                byte[] cofs_g1 = new byte[108];

                /* copy the bitstream out of the content byte array */
                Array.Copy(content, position_g1, cofs_g1, 0, 108);

                /* calculate the coefficents for filter_cofs_g1 */
                for (int i = 0; i < 6; i++)
                    for (int j = 0; j < 18; j++)
                        if (cofs_g1[(i * 18) + 17 - j] == 49)
                            cofs[i] += 1 << j;
            }
            else
                return_val = -1;

            /* get the position of the g1 filter bitstream in the content byte array */
            int position_g2 = CofsPos(content, bytes_g2, coefs_offset);
            if (position_g2 != -1)
            {
                byte[] cofs_g2 = new byte[108];

                /* copy the bitstream out of the content byte array */
                Array.Copy(content, position_g2, cofs_g2, 0, 108);

                /* calculate the coefficents for filter_cofs_g2 */
                for (int i = 6; i < 12; i++)
                    for (int j = 0; j < 18; j++)
                        if (cofs_g2[((i - 6) * 18) + 17 - j] == 49)
                            cofs[i] += 1 << j;
            }
            else
                return_val = -1;
            /* get the position of the g1 filter bitstream in the content byte array */
            int position_g3 = CofsPos(content, bytes_g3, coefs_offset);
            if (position_g3 != -1)
            {
                byte[] cofs_g3 = new byte[108];

                /* copy the bitstream out of the content byte array */
                Array.Copy(content, position_g3, cofs_g3, 0, 108);

                /* calculate the coefficents for filter_cofs_g3 */
                for (int i = 12; i < 18; i++)
                    for (int j = 0; j < 18; j++)
                        if (cofs_g3[((i - 12) * 18) + 17 - j] == 49)
                            cofs[i] += 1 << j;
            }
            else
                return_val = -1;

            /************* DOWNSAMPLING FACTOR *************/

            /* convert the name as string to a byte array */
            byte[] bytes_ds = Encoding.ASCII.GetBytes("filter_ds_fact");

            int ds_offset = 56;

            /* get the position of the downsampling factor bitstream in the content byte array */
            int position_ds = CofsPos(content, bytes_ds, ds_offset);
            if (position_ds != -1)
            {
                byte[] ds = new byte[18];

                /* copy the bitstream out of the content byte array */
                Array.Copy(content, position_ds, ds, 0, 18);

                /* calculate the downsampling factor */
                for (int j = 0; j < 18; j++)
                    if (ds[17 - j] == 49)
                        cofs[18] += 1 << j;
            }
            else
                return_val = -1;

            /************* G23 DELAY *************/

            /* convert the name as string to a byte array */
            byte[] bytes_g23 = Encoding.ASCII.GetBytes("filter_g23_delay");

            int g23_offset = 54;

            /* get the position of the g23 delay bitstream in the content byte array */
            int position_g23 = CofsPos(content, bytes_g23, g23_offset);
            if (position_g23 != -1)
            {
                byte[] g23 = new byte[18];

                /* copy the bitstream out of the content byte array */
                Array.Copy(content, position_g23, g23, 0, 18);

                /* calculate the delay for g23 */
                for (int j = 0; j < 18; j++)
                    if (g23[17 - j] == 49)
                        cofs[19] += 1 << j;
            }
            else
                return_val = -1;

            coefs_out = cofs;
            return return_val;
        }

        /* get the position where bytes is located in array */
        public static int CofsPos(byte[] array, byte[] bytes, int offset)
        {
            int success = 0;
            for (int i = 0; i < array.Length; i++)
            {
                if (array[i] == bytes[success])
                    success++;
                else
                    success = 0;

                if (bytes.Length == success)
                    return i + offset; // the bitstreams starts 56 bytes after the "filter_cofs_gx" byte array
            }
            return -1;
        }

        private void GetStructFromAPF(String name, ref byte[] content, out double[] data)
        {
            int struct_count = content[4];
            int offset = 5;
            int data_length = 0;
            bool found = false;

            /* common gap used in apf files between markers and data */
            int GAP = 65;

            /* loop over all structs in the file */
            for (int i = 0; i < struct_count; i++)
            {
                /* try to match string 'name' to file data at current offset */
                int j;
                for (j = 0; j < name.Length; j++)
                {
                    if (name[j] != content[offset + j])
                    {
                        /* On first mismatch try next struct after offset incrementation */
                        break;
                    }
                }

                /* read length of data from 4 bytes after the data (at + GAP offset) */
                data_length = BitConverter.ToInt32(content, offset + GAP);

                if (j == name.Length)
                {
                    found = true;
                    break;
                }

                /* for next offset jump over 4 bytes of the data_length field, the actual data_length plus the default GAP */
                offset += sizeof(Int32) + data_length + GAP;
            }

            if (!found)
            {
                data = null;
                return;
            }

            /* convert data to double */
            data = new double[data_length / 8];

            for (int i = 0; i < data.Length; i++)
            {
                data[i] = BitConverter.ToDouble(content, offset + GAP + sizeof(Int32) + i * 8);
            }
        }

        public Status_t ReadFpgaSignalData(ushort[] sig1, ushort[] sig2, ushort[] size)
        {
            return phhalHw_ISO3_Cmd_ReadFpgaSignalData(m_pDataParams, sig1, sig2, size);
        }

        public Status_t Cmd_GetPLSVersion(out ushort version, out ushort revision)
        {
            ushort rev = 0;
            ushort ver = 0;

            Status_t status = phhalHw_ISO3_Cmd_GetPLSVersion(m_pDataParams, ref ver, ref rev);

            version = ver;
            revision = rev;

            return status;
        }

        public Status_t Cmd_GetDeviceDna(out ulong dna)
        {
            ulong ddna = 0;

            Status_t status = phhalHw_ISO3_Cmd_GetDeviceDna(m_pDataParams, ref ddna);
            dna = ddna;

            return status;
        }

        public Status_t Cmd_GetAgcStatistic(bool reset, out UInt32 minAgc, out UInt32 maxAgc)
        {
            minAgc = 0;
            maxAgc = 0;

            return phhalHw_ISO3_Cmd_GetAgcStatistic(m_pDataParams, (byte)(reset ? Value.ON : Value.OFF), ref minAgc, ref maxAgc);
        }

        public Status_t SetConfig32(
            Config wConfig,     /**< [In] Configuration Identifier */
            uint wValue          /**< [In] Configuration Value */
            )
        {
            return phhalHw_ISO3_Cmd_SetConfig32(m_pDataParams, (ushort)wConfig, wValue);
        }

        public Status_t GetConfig32(
            Config wConfig,     /**< [In] Configuration Identifier */
            out uint pValue      /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            uint wValue = 0;
            status = phhalHw_ISO3_Cmd_GetConfig32(m_pDataParams, (ushort)wConfig, ref wValue);
            pValue = wValue;
            return status;
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public ISO3()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~ISO3()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion

        public void Exchange(int p, byte[] tx, byte[] rx)
        {
            throw new NotImplementedException();
        }
    }
}
#endif
