/*
 * Copyright 2019, 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

#if PACKAGE_INTERNAL
using System;
using System.IO;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.dlISO3
{
    #region BASE

    #region DEFINES

    #endregion

    public abstract class Generic
    {

        #region DLLIMPORTS

        #endregion

        #region DLL_WRAPPED_FUNCTIONS


        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region FilterCalibration

    /// <summary>
    /// ISO3 DL Component of Basic Function Library Framework.
    /// </summary>
    public class FilterCalibration : dlISO3.Generic
    {

        #region DEFINES

        public enum AntennaType : ushort
        {
            PCD1 = 0,                                       /**<  */
            PCD2 = 1,                                       /**<  */
            ISO15693 = 2                                    /**<  */
        }

        public static int MLS_NUM_SAMPLES = 0x80000;        /**< Maximum number of samples for Maximum Length Sequence */

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// IIR filter coefficients for one second-order-section (SOS)
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct SOSection
        {
            public uint a1;
            public uint a2;
            public uint b0;
            public uint b1;
            public uint b2;
        };

        /// <summary>
        /// IIR filter coefficients for all four second-order-section plus the 13.56 MHz gain
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct IIRFilterCoefficients
        {
            public SOSection sec1;
            public SOSection sec2;
            public SOSection sec3;
            public SOSection sec4;
            public uint gain;
        };

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                 /**< Layer ID for this DL component, NEVER MODIFY! */
            public IntPtr pHalISO3DataParams;                  /**< Pointer to the parameter structure of HAL ISO3. */
            public IntPtr pDlISO3SoftScopeDataParams;          /**< Pointer to the parameter structure of DL ISO3SoftScope. */
            public byte bUseRemoteBuffers;                     /**< Indicates if data should be stored remotely or fetched. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Init(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,           /**< [In]  Specifies the size of the data parameter structure. */
            IntPtr pHalISO3DataParams,          /**< [In]  Pointer to the parameter structure of HAL ISO3. */
            IntPtr pDlISO3SoftScopeDataParams   /**< [In]  Pointer to the parameter structure of DL ISO3SoftScope. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_Calibrate(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            byte bAntennaType,                  /**< [In]  Type of pcd antenna. */
            long dwTimeStamp                    /**< [In]  Calibration time stamp in UNIX format. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_PlayMaximumLengthSequence(
            IntPtr pDataParams                  /**< [In]  Pointer to this layers parameter structure. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_GetMaximumLengthSequence(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            byte[] pbMlsBuffer,                 /**< [Out] Maximum Length Sequence samples. */
            uint dwBufferSizeBytes,             /**< [In]  Buffer size of MlsBuffer. */
            uint[] pdwBufferValidBytes          /**< [Out] Number of valid bytes within MlsBuffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_GetSystemResponse(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            byte[] pbSystemResponseBuffer,      /**< [Out] System Response. */
            uint dwBufferSizeBytes,             /**< [In]  Buffer size of MlsBuffer. */
            uint[] pdwBufferValidBytes          /**< [Out] Number of valid bytes within MlsBuffer. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_GetFilterCoefficients(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            byte bAntennaType,                  /**< [In]  Type of pcd antenna. */
            ref IIRFilterCoefficients pValue,   /**< [OUT] Filter coefficients of last calibration. */
            ref long dwTimeStamp                /**< [OUT] Time stamp of calibration in UNIX format. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_SetFilterCoefficients(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            byte bAntennaType,                  /**< [In]  Type of pcd antenna. */
            IIRFilterCoefficients pValue,       /**< [In]  Filter coefficients for this pcd antenna. */
            long dwTimeStamp                    /**< [In]  Time stamp of calibration in  UNIX format.  */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_StoreInRemoteBuffer(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            uint dwBufferId,                    /**< [In]  Remote buffer ID. */
            byte[] pData,                       /**< [In]  Data to be moved inside remote buffer. */
            uint dwSizeBytes                    /**< [In]  Size of data in bytes.  */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phdlISO3_FilterCalibration_Cmd_CalculateFilterCoefficients(
            IntPtr pDataParams,                 /**< [In]  Pointer to this layers parameter structure. */
            byte bAntennaType,                  /**< [In]  Type of pcd antenna. */
            long qwTimeStamp,                   /**< [In]  Time stamp of filter calculation. */
            uint dwMlsBufferId,                 /**< [In]  Remote buffer ID where Maximum Length Sequence is located. */
            uint dwSysBufferId                  /**< [In]  Remote buffer ID where System Response is located. */
            );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="bComPort"></param>
        /// <returns></returns>
        public Status_t Init(NxpRdLibNet.Hal.ISO3 pHalISO3, NxpRdLibNet.dlOsci.ISO3SoftScope pDlScope)
        {
            return phdlISO3_FilterCalibration_Init(
                m_pDataParams,
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pHalISO3.m_pDataParams,
                pDlScope.m_pDataParams
                );
        }

        #endregion

        #region CUSTOM_FUNCTIONS

        public Status_t Cmd_Calibrate(
            AntennaType type,                   /**< [In]  Antenna Type. */
            long timeStamp                      /**< [In]  Current time stamp. */
            )
        {
            return phdlISO3_FilterCalibration_Cmd_Calibrate(m_pDataParams, (byte)type, timeStamp);
        }

        public Status_t Cmd_PlayMaximumLengthSequence()
        {
            return phdlISO3_FilterCalibration_Cmd_PlayMaximumLengthSequence(m_pDataParams);
        }

        public Status_t Cmd_GetMaximumLengthSequence(
            ref byte[] mlsBuffer                /**< [Out] Maximum Length sequence. */
            )
        {
            uint[] numValidSamples = { 0 };
            return phdlISO3_FilterCalibration_Cmd_GetMaximumLengthSequence(
                m_pDataParams,
                mlsBuffer,
                (uint)mlsBuffer.Length,
                numValidSamples);
        }

        public Status_t Cmd_GetSystemResponse(
            ref byte[] systemResponseBuffer     /**< [Out] System response after playing Maximum Length Sequence. */
            )
        {
            uint[] numValidSamples = { 0 };
            return phdlISO3_FilterCalibration_Cmd_GetSystemResponse(
                m_pDataParams,
                systemResponseBuffer,
                (uint)systemResponseBuffer.Length,
                numValidSamples);
        }

        public Status_t Cmd_SetFilterCoefficients(
            AntennaType type,                    /**<  [In] Antenna Type */
            IIRFilterCoefficients coeffs,        /**<  [In] IIR Filter Coefficients */
            long timestamp                       /**<  [In] Time stamp of calibration in UNIX format */
            )
        {
            return phdlISO3_FilterCalibration_Cmd_SetFilterCoefficients(m_pDataParams, (byte)type, coeffs, timestamp);
        }

        public Status_t Cmd_GetFilterCoefficients(
            AntennaType type,                   /**< [In]  Antenna Type */
            out IIRFilterCoefficients coeffs,   /**< [Out] IIR Filter Coefficients for antenna type */
            out long timestamp                  /**< [Out] Time stamp of calibration in UNIX format*/
            )
        {
            Status_t status;
            IIRFilterCoefficients pCoeffs = new IIRFilterCoefficients();
            long pTimestamp = new long();
            status = phdlISO3_FilterCalibration_Cmd_GetFilterCoefficients(m_pDataParams, (byte)type, ref pCoeffs, ref pTimestamp);
            coeffs = pCoeffs;
            timestamp = pTimestamp;
            return status;
        }

        public Status_t ReadFilterFile(String filename, ref IIRFilterCoefficients coeffs)
        {
            Status_t status = new Status_t();
            String[] section_name = {"[IIR Filter Coefficients]"};
            try
            {
                using (StreamReader sr = new StreamReader(filename))
                {
                    String line = sr.ReadToEnd();
                    String[] sections = line.Split(section_name, StringSplitOptions.None);
                    /* sections[0] contains everything before the filter coefficients, section[1] contains the filter coefficients as strings */
                    String[] coeff_strings = sections[1].Split('\n');
                    UInt32[] tmp_coeffs = new UInt32[25]; /* 5 coefficients per section. 4 sections -> 20 coefficients. +gain = 21 */
                    for (int line_index = 0; line_index < 25; line_index++)
                    {
                        int numOfChar = 25;
                        String current_line = coeff_strings[line_index + 1];
                        tmp_coeffs[line_index] = 0x0;
                        for (int char_index = 0; char_index < numOfChar; char_index++)
                        {
                            if (current_line[char_index] == '1')
                            {
                                tmp_coeffs[line_index] += (uint)(0x1 << ((numOfChar -1) - char_index));
                            }
                        }
                    }

                    int x = 0;
                    coeffs.sec1.b0 = tmp_coeffs[x++];
                    coeffs.sec2.b0 = tmp_coeffs[x++];
                    coeffs.sec3.b0 = tmp_coeffs[x++];
                    coeffs.sec4.b0 = tmp_coeffs[x++];

                    coeffs.sec1.b1 = tmp_coeffs[x++];
                    coeffs.sec2.b1 = tmp_coeffs[x++];
                    coeffs.sec3.b1 = tmp_coeffs[x++];
                    coeffs.sec4.b1 = tmp_coeffs[x++];

                    coeffs.sec1.b2 = tmp_coeffs[x++];
                    coeffs.sec2.b2 = tmp_coeffs[x++];
                    coeffs.sec3.b2 = tmp_coeffs[x++];
                    coeffs.sec4.b2 = tmp_coeffs[x++];

                    x += 4; /* a1 is set to 1 in all 4 sections */

                    coeffs.sec1.a1 = tmp_coeffs[x++];
                    coeffs.sec2.a1 = tmp_coeffs[x++];
                    coeffs.sec3.a1 = tmp_coeffs[x++];
                    coeffs.sec4.a1 = tmp_coeffs[x++];

                    coeffs.sec1.a2 = tmp_coeffs[x++];
                    coeffs.sec2.a2 = tmp_coeffs[x++];
                    coeffs.sec3.a2 = tmp_coeffs[x++];
                    coeffs.sec4.a2 = tmp_coeffs[x++];

                    coeffs.gain = tmp_coeffs[x];
                }
            }
            catch (Exception)
            {
                status = new Status_t(Error_CompCode.DL_ISO3, Error_Param.INVALID_PARAMETER);
            }
            return status;
        }

        public Status_t Cmd_StoreInRemoteBuffer(uint remoteBufferID, byte[] data)
        {
            return phdlISO3_FilterCalibration_Cmd_StoreInRemoteBuffer(m_pDataParams, remoteBufferID,
                data, (uint)data.Length);
        }

        public Status_t Cmd_CalculateFilterCoefficients(AntennaType antenna, long timestamp,
            uint remoteMlsBufferID, uint remoteSysBufferID)
        {
            return phdlISO3_FilterCalibration_Cmd_CalculateFilterCoefficients(m_pDataParams, (byte)antenna,
                timestamp, remoteMlsBufferID, remoteSysBufferID);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public FilterCalibration()
        {
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~FilterCalibration()
        {
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
}
#endif