/*
 * Copyright 2013, 2015, 2017 - 2020, 2023 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Text;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.Bal
{
    #region BASE

    #region DEFINES

    public enum Error : byte
    {
        ERR_DLL = ( CustomCodes.ERROR_BEGIN )    /**< Used dll threw an non standard error read wAddionalInfo to get error code */
    }

    public enum Config : int
    {
        TIMEOUT_WRITE_MS = 0,   /**< Configure transmission timeout. */
        TIMEOUT_READ_MS         /**< Configure reception timeout. */
    }

    #endregion

    /// <summary>
    /// Abstract Base-Class for every BAL object.
    /// (Uses only the Generic BAL C-Object)
    /// </summary>
    public abstract class Generic
    {
        #region DEFINES

        private const int ReaderStringLen = 256;

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_GetPortList(
            IntPtr m_pDataParams,   /**< [In] Pointer to this layers parameter structure */
            ushort PortBufSize,     /**< [In] Buffer Size of Port Name String */
            byte[] PortNames,       /**< [Out] Port Name as Multi-Unicode String */
            ref ushort NumOfPorts   /**< [Out] Number of found port strings. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_SetPort(
            IntPtr m_pDataParams,   /**< [In] Pointer to this layers parameter structure */
            byte[] PortName         /**< [In] Port Name as String. */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_OpenPort(
            IntPtr m_pDataParams    /**< [In] Pointer to this layers parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_ClosePort(
            IntPtr m_pDataParams    /**< [In] Pointer to this layers parameter structure */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_Exchange(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength,       /**< [In] length of input data */
            ushort wRxBufSize,      /**< [In] size of receive buffer */
            byte[] pRxBuffer,       /**< [Out] received data */
            ref ushort pRxLength    /**< [Out] number of received data bytes */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_SetConfig(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ushort wValue       /**< [In] Configuration Value */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_GetConfig(
            IntPtr pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wConfig,     /**< [In] Configuration Identifier */
            ref ushort pValue   /**< [Out] Configuration Value */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        /// <summary>
        /// Obtain a port list
        /// </summary>
        /// <param name="readers"></param>
        /// <returns></returns>
        public Status_t GetPortList(
            int MaxPorts,
            out string[] Ports)
        {
            Status_t status;
            int iIndex1 = 0;
            int iIndex2 = 0;
            ushort NumOfPortsInt = 0;

            // Get Readerlist
            byte[] pPortNames = new byte[ReaderStringLen * MaxPorts];
            status = phbalReg_GetPortList(m_pDataParams, (ushort)pPortNames.Length, pPortNames, ref NumOfPortsInt);

            // Fill our Readerlist
			string[] aSplitString = Encoding.ASCII.GetString ( pPortNames ).Trim ( '\0' ).Split ( '\0' );
            Ports = new string[NumOfPortsInt];

			do
			{
				if ( !string.IsNullOrEmpty ( aSplitString[iIndex1] ) )
					Ports[iIndex2++] = aSplitString[iIndex1];

				iIndex1++;
			} while ( iIndex2 != NumOfPortsInt );

            return status;
        }

        /// <summary>
        /// Set communication port.
        /// </summary>
        /// <returns></returns>
        public Status_t SetPort(
            string PortName
            )
        {
            // Allocate internal data parameters and pointer to them
            this.m_PortNameInt = new byte[PortName.Length + 1];
            for (int i = 0; i < PortName.Length; i++)
            {
                this.m_PortNameInt[i] = (byte)PortName[i];
            }
            this.m_PortNameInt[PortName.Length] = 0;
            this.m_pPortNameInt = GCHandle.Alloc(this.m_PortNameInt, GCHandleType.Pinned);
            return phbalReg_SetPort(m_pDataParams, m_PortNameInt);
        }

        /// <summary>
        /// Open communication port.
        /// </summary>
        /// <returns></returns>
        public Status_t OpenPort()
        {
            return phbalReg_OpenPort(m_pDataParams);
        }

        /// <summary>
        /// Close communication port.
        /// </summary>
        /// <returns></returns>
        public Status_t ClosePort()
        {
            return phbalReg_ClosePort(m_pDataParams);
        }

        /// <summary>
        /// Perform Data Exchange on the bus.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <param name="wRxBufSize"></param>
        /// <param name="pRxBuffer"></param>
        /// <returns></returns>
        public Status_t Exchange(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            int wRxBufSize,         /**< [In] size of receive buffer */
            out byte[] pRxBuffer    /**< [Out] received data */
            )
        {
            Status_t status;
            ushort wRxLength = 0;

            pRxBuffer = new byte[wRxBufSize];
            status = phbalReg_Exchange(m_pDataParams, (ushort)wOption, pTxBuffer, (ushort)pTxBuffer.Length, (ushort)wRxBufSize, pRxBuffer, ref wRxLength);
            Array.Resize<byte>(ref pRxBuffer, wRxLength);

            return status;
        }

        public Status_t SetConfig(
            Enum wConfig,   /**< [In] Configuration Identifier */
            int wValue      /**< [In] Configuration Value */
            )
        {
            return phbalReg_SetConfig ( m_pDataParams, Convert.ToUInt16 ( wConfig ), ( ushort ) wValue );
        }

        public Status_t GetConfig(
            Enum wConfig,   /**< [In] Configuration Identifier */
            out int pValue  /**< [Out] Configuration Value */
            )
        {
            Status_t status;
            ushort wValue = 0;
            status = phbalReg_GetConfig(m_pDataParams, Convert.ToUInt16 ( wConfig ), ref wValue);
            pValue = (int)wValue;
            return status;
        }

        private byte[] m_PortNameInt;
        private GCHandle m_pPortNameInt;

        private IntPtr m_pPortName
        {
            get
            {
                return this.m_pPortNameInt.AddrOfPinnedObject();
            }
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Generic()
        {
            // Free allocated pointer to reader params
            if (this.m_pPortNameInt.IsAllocated)
            {
                this.m_pPortNameInt.Free();
            }
        }

        /// <summary>
        /// Set & Get configuration setting
        /// </summary>
        /// <param name="wConfig"></param>
        /// <returns></returns>
        public int this[Enum wConfig]
        {
            get
            {
                int wValue;
                Status_t status = GetConfig(wConfig, out wValue);
                if ((Error_Gen)status.Error != Error_Gen.SUCCESS)
                {
                    throw new Exception(status.ToString());
                }
                return wValue;
            }
            set
            {
                Status_t status = SetConfig(wConfig, value);
                if ((Error_Gen)status.Error != Error_Gen.SUCCESS)
                {
                    throw new Exception(status.ToString());
                }
            }
        }

        #endregion

        #region MEMORY_MAPPING

        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion
    }

    #endregion

    #region SERIALWIN

    /// <summary>
    /// RS232 (Windows) BAL Component of Basic Function Library Framework.
    /// </summary>
    public class SerialWin : Bal.Generic
    {
        #region DEFINES

        public enum Config : int
        {
            BITRATE = NxpRdLibNet.CustomCodes.CONFIG_BEGIN,     /**< Configure communication speed. */
            STOPBITS,                                           /**< Configure number of stop bits. */
            DTR,                                                /**< Configure dtr line. */
            RTS,                                                /**< Configure rts line. */
            PARITY,                                             /**< Configure parity. */
            FI = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 7),    /**< Configure FI. */
            DI,                                                 /**< Configure DI. */
            TIMEOUT_MULTIPLIER_MS,                              /**< Configure Timeout Multiplier. */
            ADDITIONAL_RXDELAY_BYTES                            /**< Needed because WriteFile() sometimes returns too early. */
        }

        public enum DTR : int
        {
            DTR_LOW = 0,    /**< dtr low. */
            DTR_HIGH        /**< dtr high. */
        }

        public enum RTS : int
        {
            RTS_LOW = 0,    /**< rts low. */
            RTS_HIGH        /**< rts high. */
        }

        public enum Parity : int
        {
            NONE = 0,
            ODD,
            EVEN
        }

        public enum BitRate : int
        {
            VALUE_9600 = 0x0000,
            VALUE_19200,
            VALUE_38400,
            VALUE_57600,
            VALUE_115200,
            VALUE_230400,
            VALUE_460800
        }

        public enum StopBits : int
        {
            VALUE_ONE = 0x0000,
            VALUE_ONE5,
            VALUE_TWO
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Number of communication port.
            /// </summary>
            public IntPtr pComPortString;

            /// <summary>
            /// Handle to communication port.
            /// </summary>
            public IntPtr pComHandle;

            /// <summary>
            /// Storage for bit rate setting.
            /// </summary>
            public uint dwBitRate;

            /// <summary>
            /// Storage for timeout setting.
            /// </summary>
            public ushort wTimeout;

            /// <summary>
            /// Storage for additional timeout setting.
            /// </summary>
            public ushort wAdditionalRxDelayBytes;

            /// <summary>
            /// Storage for timeout multiplier setting.
            /// </summary>
            public ushort wTimeoutMultiplier;

            /// <summary>
            /// Storage for enumeration buffer size.
            /// </summary>
            public ushort wEnumBufSize;

            /// <summary>
            /// Storage for enumeration buffer.
            /// </summary>
            public IntPtr pEnumBuffer;

            /// <summary>
            /// Storage for stop bits.
            /// </summary>
            public byte bStopBits;

            /// <summary>
            /// DTR line.
            /// </summary>
            public byte bDtr;

            /// <summary>
            /// RTS line.
            /// </summary>
            public byte bRts;

            /// <summary>
            /// Parity.
            /// </summary>
            public byte bParity;

            /// <summary>
            /// FI.
            /// </summary>
            public byte bFI;

            /// <summary>
            /// DI.
            /// </summary>
            public byte bDI;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_SerialWin_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            ushort wEnumBufSize,                /**< [In] Size of the enumeration buffer. Maximum possible size is 255 bytes. */
            IntPtr EnumBuffer                   /**< [In] Pointer to the enumeration buffer for the GetPortList() function. */
            );

        #endregion

        #region INIT

        private byte[] m_bEnumBuffer;
        private GCHandle m_pEnumBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="bComPort"></param>
        /// <returns></returns>
        public Status_t Init(ushort EnumBufSize)
        {
            // Free Buffers
            if (this.m_pEnumBuffer.IsAllocated)
            {
                this.m_pEnumBuffer.Free();
            }

            // Allocate Buffers
            this.m_bEnumBuffer = new byte[EnumBufSize];
            this.m_pEnumBuffer = GCHandle.Alloc(this.m_bEnumBuffer, GCHandleType.Pinned);

            return phbalReg_SerialWin_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), EnumBufSize, this.m_pEnumBuffer.AddrOfPinnedObject());
        }

#if DEBUG
        public Status_t Init(int wDataParamSize, ushort EnumBufSize)
        {
            // Free Buffers
            if (this.m_pEnumBuffer.IsAllocated)
            {
                this.m_pEnumBuffer.Free();
            }

            // Allocate Buffers
            this.m_bEnumBuffer = new byte[EnumBufSize];
            this.m_pEnumBuffer = GCHandle.Alloc(this.m_bEnumBuffer, GCHandleType.Pinned);

            return phbalReg_SerialWin_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize, EnumBufSize, this.m_pEnumBuffer.AddrOfPinnedObject());
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public SerialWin()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~SerialWin()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (this.m_pEnumBuffer.IsAllocated)
            {
                this.m_pEnumBuffer.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region PCSCWIN

    /// <summary>
    /// PCSC (Windows) BAL Component of Basic Function Library Framework.
    /// </summary>
    public class PcscWin : Bal.Generic
    {
        #region DEFINES

        public enum Config : int
        {
            PROTOCOL = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN),    /**< Protocol of card (T=0 or T=1) */
            SHARE,                                                /**< Protocol of card (T=0 or T=1) */
            USE_WARM_ATR,                                        /**< Define if the warm atr should used. Values: PH_ON, PH_OFF. */
            CARD_STATE,                                            /**< Current state of the smart card in the reader. See SCardStatus */
            ACTIVE_PROTOCOL,                                    /**< Get the used Protocol; e.g. #PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_T0. */
            PEGODA2_MODE                                        /**< Define if the Pegoda 2 Mode should used. Values: PH_ON, PH_OFF. */
        }

        public enum Protocol : int
        {
            T0 = 0x0000,    /**< T=0 */
            T1,             /**< T=1 */
            RAW,            /**< RAW protocol. */
            AUTO,           /**< Use T=0|T=1|RAW protocol. */
            UNDEFINED       /**< UNDEFINED */
        }

        public enum Share : int
        {
            SHARED = 0x0000,    /**< SHARED */
            EXCLUSIVE,          /**< EXCLUSIVE */
            DIRECT              /**< DIRECT */
        }

        public static class ExchangeOptions
        {
            public const int SCARDCONTROL = 0x0000;
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Protocol Type that is required during open (T=0,T=1,RAW,Auto,Undefined).
            /// </summary>
            public byte bProtocolType;

            /// <summary>
            /// Define protocol type that is active after open. (T=0,T=1,RAW,Undefined).
            /// </summary>
            public byte bActiveProtocolType;

            /// <summary>
            /// Share Mode (Shared, Exclusive, Direct).
            /// </summary>
            public byte bShare;

            /// <summary>
            /// This is mainly for Pegoda 2 (RD710) reader usage. To be set only if firmware higher that 2.2.7 is used.
            /// </summary>
            public byte bPegoda2Mode;

            /// <summary>
            /// Pointer to buffer used for storing the ATR after card activation.
            /// </summary>
            public IntPtr pAtrBuffer;

            /// <summary>
            /// Indicates that the warm activation should be used.
            /// </summary>
            public byte bUseWarmATR;

            /// <summary>
            /// The maximum size of the ATR buffer.
            /// </summary>
            public ushort wMaxAtrBufSize;

            /// <summary>
            /// Size of the ATR buffer.
            /// </summary>
            public ushort wAtrBufSize;

            /// <summary>
            /// Pointer to buffer used for storing the vendor name.
            /// </summary>
            public IntPtr pVendorName;

            /// <summary>
            /// Size of the VendorName buffer.
            /// </summary>
            public ushort wVendorNameSize;

            /// <summary>
            /// Pointer to buffer used for storing the vendor ifd type.
            /// </summary>
            public IntPtr pVendorIFDType;

            /// <summary>
            /// Size of the VendorIFDType buffer.
            /// </summary>
            public ushort wVendorIFDTypeSize;

            /// <summary>
            /// Pointer to buffer used for storing the vendor ifd version.
            /// </summary>
            public IntPtr pVendorIFDVersion;

            /// <summary>
            /// Size of the VendorIFDVersion buffer.
            /// </summary>
            public ushort wVendorIFDVersionSize;

            /// <summary>
            /// Pointer to buffer used for storing the vendor ifd serial.
            /// </summary>
            public IntPtr pVendorIFDSerial;

            /// <summary>
            /// Size of the VendorIFDSerial buffer.
            /// </summary>
            public ushort wVendorIFDSerialSize;

            /// <summary>
            /// PCSC Context.
            /// </summary>
            public IntPtr hContext;

            /// <summary>
            /// PCSC Card Context.
            /// </summary>
            public IntPtr hCard;

            /// <summary>
            /// Name of the PCSC Reader as UNICODE String
            /// </summary>
            public IntPtr pReaderName;

            /// <summary>
            /// TO value for Write transfer
            /// </summary>
            public ushort wTimeoutWrMs;

            /// <summary>
            /// TO value for Read transfer
            /// </summary>
            public ushort wTimeoutRdMs;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_PcscWin_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            IntPtr pAtrBuffer,                  /**< [In] Pointer to buffer used for storing the ATR after card activation. */
            ushort wAtrBufSize                  /**< [In] Size of the ATR buffer. */
            );

        #endregion

        #region INIT

        private GCHandle m_pAtrBuffer;
        private GCHandle m_pVendorNameBuffer;
        private GCHandle m_pVendorIFDTypeBuffer;
        private GCHandle m_pVendorIFDVersionBuffer;
        private GCHandle m_pVendorIFDSerialBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pReaderName"></param>
        /// <returns></returns>
        public Status_t Init(out byte[] AtrBuffer)
        {
            // Free Buffers
            if (this.m_pAtrBuffer.IsAllocated)
            {
                this.m_pAtrBuffer.Free();
            }

            AtrBuffer = new byte[254];
            m_pAtrBuffer = GCHandle.Alloc(AtrBuffer, GCHandleType.Pinned);

            return phbalReg_PcscWin_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                m_pAtrBuffer.AddrOfPinnedObject(),
                (ushort)AtrBuffer.Length);
        }

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pReaderName"></param>
        /// <returns></returns>
        public Status_t Init(out byte[] AtrBuffer,
            out byte[] VendorNameBuffer,
            out byte[] VendorIFDTypeBuffer,
            out byte[] VendorIFDVersionBuffer,
            out byte[] VendorIFDSerialBuffer)
        {
            // Free Buffers
            if (this.m_pAtrBuffer.IsAllocated)
            {
                this.m_pAtrBuffer.Free();
            }
            if (this.m_pVendorNameBuffer.IsAllocated)
            {
                this.m_pVendorNameBuffer.Free();
            }
            if (this.m_pVendorIFDTypeBuffer.IsAllocated)
            {
                this.m_pVendorIFDTypeBuffer.Free();
            }
            if (this.m_pVendorIFDVersionBuffer.IsAllocated)
            {
                this.m_pVendorIFDVersionBuffer.Free();
            }
            if (this.m_pVendorIFDSerialBuffer.IsAllocated)
            {
                this.m_pVendorIFDSerialBuffer.Free();
            }

            AtrBuffer = new byte[254];
            m_pAtrBuffer = GCHandle.Alloc(AtrBuffer, GCHandleType.Pinned);
            VendorNameBuffer = new byte[256];
            m_pVendorNameBuffer = GCHandle.Alloc(VendorNameBuffer, GCHandleType.Pinned);
            VendorIFDTypeBuffer = new byte[256];
            m_pVendorIFDTypeBuffer = GCHandle.Alloc(VendorIFDTypeBuffer, GCHandleType.Pinned);
            VendorIFDVersionBuffer = new byte[4];
            m_pVendorIFDVersionBuffer = GCHandle.Alloc(VendorIFDVersionBuffer, GCHandleType.Pinned);
            VendorIFDSerialBuffer = new byte[256];
            m_pVendorIFDSerialBuffer = GCHandle.Alloc(VendorIFDSerialBuffer, GCHandleType.Pinned);

            ushort initResult = phbalReg_PcscWin_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                m_pAtrBuffer.AddrOfPinnedObject(),
                (ushort)AtrBuffer.Length);

            m_DataParamsInt[0].pVendorName = m_pVendorNameBuffer.AddrOfPinnedObject();
            m_DataParamsInt[0].wVendorNameSize = (ushort)VendorNameBuffer.Length;
            m_DataParamsInt[0].pVendorIFDType = m_pVendorIFDTypeBuffer.AddrOfPinnedObject();
            m_DataParamsInt[0].wVendorIFDTypeSize = (ushort)VendorIFDTypeBuffer.Length;
            m_DataParamsInt[0].pVendorIFDVersion = m_pVendorIFDVersionBuffer.AddrOfPinnedObject();
            m_DataParamsInt[0].wVendorIFDVersionSize = (ushort)VendorIFDVersionBuffer.Length;
            m_DataParamsInt[0].pVendorIFDSerial = m_pVendorIFDSerialBuffer.AddrOfPinnedObject();
            m_DataParamsInt[0].wVendorIFDSerialSize = (ushort)VendorIFDSerialBuffer.Length;

            return initResult;
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public PcscWin()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~PcscWin()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (this.m_pAtrBuffer.IsAllocated)
            {
                this.m_pAtrBuffer.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region RD70XUSBWIN

    /// <summary>
    /// PEGODA BAL Component of Basic Function Library Framework.
    /// </summary>
    public class Rd70xUsbWin : Bal.Generic
    {
        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;              /**< Layer ID for this BAL component, NEVER MODIFY! */
            public byte ucTxSeq;            /**< Sequence counter for packets. */
            public IntPtr pDeviceName;      /**< Name of currently open USB device. */
            public IntPtr pDeviceHandle;    /**< Handle to the USB device. */
            public IntPtr pPipeOut;         /**< Handle to Usb Out-pipe */
            public IntPtr pPipeIn;          /**< Handle to Usb In-pipe. */
            public ushort wTimeoutWrMs;        /**< TO value for Usb Write pipe transfer */
            public ushort wTimeoutRdMs;        /**< TO value for Usb Read pipe transfer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_Rd70xUsbWin_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams        /**< [In] Specifies the size of the data parameter structure */
            );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pReaderName"></param>
        /// <returns></returns>
        public Status_t Init()
        {
            return phbalReg_Rd70xUsbWin_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rd70xUsbWin()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rd70xUsbWin()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

#if PACKAGE_INTERNAL
    #region MP300USB
    /// <summary>
    /// MP300 USB BAL Component of Basic Function Library Framework.
    /// This should work on Linux too, just need to substitute the library file libusb-win32 for libusb.
    /// </summary>
    public class Mp300Usb : Bal.Generic
    {

        #region DEFINES

        public enum Config : int
        {
            EXCHANGE_DELAY_COUNTER = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN),    /**< An integer value to throttle USB comms speed */
            VENDOR_ID_FILTER       = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN+1),  /**< Filter for ventor ID used in GetPortList to filter USB devices. If Vendor ID and Product ID is set to zero all devices are listed */
            PRODUCT_ID_FILTER      = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN+2),  /**< Filter for Product ID used in GetPortList to filter USB devices. If Vendor ID and Product ID is set to zero all devices are listed */
            EXCHANGE_MODE          = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN+3)   /**< Exchange mode of USB (INTERRUPT or BULK) */
        }

        public enum ExchangeMode : byte
        {
            INTERRUPT = 0,
            BULK = 1
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public UInt16 wId;                  /**< Layer ID for this BAL component, NEVER MODIFY! */
            public UInt16 wExchangeDelayCount;  /**< Counter to throttle the USB comms speed, without this the MP300 cannot keep up. */
            public UInt16 wTxTimeout;
            public UInt16 wRxTimeout;
            public IntPtr pPortName;
            public IntPtr pUsbDevHandle;
            public byte bInterfaceNumber;
            public byte bUsbWriteEndpointAddress;
            public byte bUsbReadEndpointAddress;
            public UInt16 wVendorIdFilter;
            public UInt16 wProductIdFilter;
            public byte bExchangeMode;
            IntPtr pLibraryHandleLibUsb;
            IntPtr pfInit;
            IntPtr pfFindBusses;
            IntPtr pfFindDevices;
            IntPtr pfGetBusses;
            IntPtr pfOpen;
            IntPtr pfClose;
            IntPtr pfClaimInterface;
            IntPtr pfReleaseInterface;
            IntPtr pfInterruptWrite;
            IntPtr pfBulkWrite;
            IntPtr pfInterruptRead;
            IntPtr pfBulkRead;
            IntPtr pfGetStringSimple;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_Mp300Usb_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams        /**< [In] Specifies the size of the data parameter structure */
            );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pReaderName"></param>
        /// <returns></returns>
        public Status_t Init()
        {
            return phbalReg_Mp300Usb_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Mp300Usb()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Mp300Usb()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion
    }

    #endregion
#endif

    #region PIPELIN

    /// <summary>
    /// RS232 (Windows) BAL Component of Basic Function Library Framework.
    /// </summary>
    public class PipeLin : Bal.Generic
    {
        #region DEFINES

        public enum Mode : byte
        {
            RAW = 0x00,
            RS232,
            I2C,
            SPI
        }

        public enum Config : int
        {
            MODE = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN),  /**< Configure communication speed  */
            BITRATE,                                        /**< Configure communication speed. */
            I2C_SLAVE_ADDR,                                 /**< Configure slave address for I2C communication. */
            SETCARD,                                        /**< Set card configuration for simulation. */
            RESET                                           /**< Reset the simulation. */
        }

        public enum BitRate : int
        {
            RS232_9600 = 0x0000,    /**< 9600 kBit/s RS232. */
            RS232_19200,            /**< 19200 kBit/s RS232. */
            RS232_38400,            /**< 38400 kBit/s RS232. */
            RS232_57600,            /**< 57600 kBit/s RS232. */
            RS232_115200,           /**< 115200 kBit/s RS232. */
            RS232_128000,           /**< 128000 kBit/s RS232. */
            RS232_230400,           /**< 230400 kBit/s RS232. */
            RS232_460800,           /**< 460800 kBit/s RS232. */
            RS232_921600,           /**< 921600 kBit/s RS232. */
            RS232_1228800,          /**< 1228800 kBit/s RS232. */
            SPI_10K = 0x0010,       /**< 10 kBit/s SPI. */
            SPI_100K,               /**< 100 kBit/s SPI. */
            SPI_1M,                 /**< 1 MBit/s SPI. */
            SPI_3M,                 /**< 3 MBit/s SPI. */
            SPI_5M,                 /**< 5 MBit/s SPI. */
            SPI_8M,                 /**< 8 MBit/s SPI. */
            I2C_10K = 0x0020,       /**< 10 KBit/s I2C. */
            I2C_100K,               /**< 100 KBit/s I2C. */
            I2C_200K,               /**< 200 KBit/s I2C. */
            I2C_400K,               /**< 400 KBit/s I2C. */
            I2CL_1M = 0x0044,       /**< 1 MBit/s I2CL. */
            I2CL_3M,                /**< 3 MBit/s I2CL. */
            I2CL_5M                 /**< 5 MBit/s I2CL. */
        }

        public enum RxTx : byte
        {
            RXTX_MIFARE_106 = 0x00,     /**< MIFARE(R) Operating mode at 106kbit/s. */
            RXTX_MIFARE_212,            /**< MIFARE(R) Operating mode at 212kbit/s. */
            RXTX_MIFARE_424,            /**< MIFARE(R) Operating mode at 414kbit/s. */
            RXTX_MIFARE_848,            /**< MIFARE(R) Operating mode at 848kbit/s. */

            RXTX_TYPEB_106 = 0x04,          /**< ISO14443B Operating mode at 106kbit/s. */
            RXTX_TYPEB_212,             /**< ISO14443B Operating mode at 212kbit/s. */
            RXTX_TYPEB_424,             /**< ISO14443B Operating mode at 414kbit/s. */
            RXTX_TYPEB_848,             /**< ISO14443B Operating mode at 848kbit/s. */

            RXTX_FELICA_212 = 0x08,     /**< FeliCa Operating mode at 212kbit/s. */
            RXTX_FELICA_424 = 0x09,     /**< FeliCa Operating mode at 424kbit/s. */

            RXTX_I15693_1OO4 = 0x0A,    /**< ISO15693 One-Out-Of-Four Operating mode. */
            RXTX_I15693_1OO256          /**< ISO15693 One-Out-Of-256 Operating mode. */
        }

        private const byte TX_TWO_CARDS = 0x80;     /**< Set this in the MSB(yte) to have two cards instead of one. */

        public class RxTxConfig_t
        {
            private int Config;

            /// <summary>
            /// Create RxTxConfig
            /// </summary>
            /// <param name="CompCode"></param>
            /// <param name="Error"></param>
            public RxTxConfig_t(RxTx Tx, RxTx Rx, bool DualCards)
            {
                Config = (int)Tx << 8;
                Config |= (int)Rx;

                if (DualCards)
                {
                    Config |= (int)(TX_TWO_CARDS << 8);
                }
            }

            /// <summary>
            /// Convert managed status object to raw status for C-Object
            /// </summary>
            /// <param name="status"></param>
            /// <returns></returns>
            public static implicit operator int(RxTxConfig_t status)
            {
                return status.Config;
            }
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;
            public byte bMode;
            public IntPtr pPipeNameRx;
            public IntPtr pPipeNameTx;
            public IntPtr pPipeRx;
            public IntPtr pPipeTx;
            public IntPtr pTxBuffer;
            public ushort wTxBufSize;
            public byte bI2cSlaveAddr;
            public byte bConfigValue;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_PipeLin_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            byte bMode,                         /**< [In] Operating mode; Refer to the PHBAL_REG_PIPELIN_MODE_* defines. */
            ushort wTxBufSize,       /**< [In] Size of the global transmit buffer. */
            IntPtr pTxBuffer       /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            );

        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="bComPort"></param>
        /// <returns></returns>
        public Status_t Init(Mode Mode, int wTxBufferSize)
        {
            // Free Buffer
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }

            // Allocate TxBuffer
            this.m_bTxBuffer = new byte[wTxBufferSize];
            this.m_pTxBuffer = GCHandle.Alloc(this.m_bTxBuffer, GCHandleType.Pinned);

            return phbalReg_PipeLin_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                (byte)Mode,
                (ushort)wTxBufferSize,
                m_pTxBuffer.AddrOfPinnedObject()
                );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public PipeLin()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~PipeLin()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region SOCKETWIN

    /// <summary>
    /// SocketWin (Windows) BAL Component of Basic Function Library Framework.
    /// </summary>
    public class SocketWin : Bal.Generic
    {
        #region DEFINES

        public enum Mode : byte
        {
            RAW = 0x00,
            RS232,
            I2C,
            SPI,
            HEX,
            TELNET
        }

        public enum Config : int
        {
            MODE = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN),  /**< Configure communication speed  */
            BITRATE,                                        /**< Configure communication speed. */
            I2C_SLAVE_ADDR,                                 /**< Configure slave address for I2C communication. */
            SETCARD,                                        /**< Set card configuration for simulation. */
            RESET                                           /**< Reset the simulation. */
        }

        public enum BitRate : int
        {
            RS232_9600 = 0x0000,    /**< 9600 kBit/s RS232. */
            RS232_19200,            /**< 19200 kBit/s RS232. */
            RS232_38400,            /**< 38400 kBit/s RS232. */
            RS232_57600,            /**< 57600 kBit/s RS232. */
            RS232_115200,           /**< 115200 kBit/s RS232. */
            RS232_128000,           /**< 128000 kBit/s RS232. */
            RS232_230400,           /**< 230400 kBit/s RS232. */
            RS232_460800,           /**< 460800 kBit/s RS232. */
            RS232_921600,           /**< 921600 kBit/s RS232. */
            RS232_1228800,          /**< 1228800 kBit/s RS232. */
            SPI_10K = 0x0010,       /**< 10 kBit/s SPI. */
            SPI_100K,               /**< 100 kBit/s SPI. */
            SPI_1M,                 /**< 1 MBit/s SPI. */
            SPI_3M,                 /**< 3 MBit/s SPI. */
            SPI_5M,                 /**< 5 MBit/s SPI. */
            SPI_8M,                 /**< 8 MBit/s SPI. */
            I2C_10K = 0x0020,       /**< 10 KBit/s I2C. */
            I2C_100K,               /**< 100 KBit/s I2C. */
            I2C_200K,               /**< 200 KBit/s I2C. */
            I2C_400K,               /**< 400 KBit/s I2C. */
            I2CL_1M = 0x0044,       /**< 1 MBit/s I2CL. */
            I2CL_3M,                /**< 3 MBit/s I2CL. */
            I2CL_5M                 /**< 5 MBit/s I2CL. */
        }

        public enum RxTx : byte
        {
            RXTX_MIFARE_106 = 0x00,     /**< MIFARE(R) Operating mode at 106kbit/s. */
            RXTX_MIFARE_212,            /**< MIFARE(R) Operating mode at 212kbit/s. */
            RXTX_MIFARE_424,            /**< MIFARE(R) Operating mode at 414kbit/s. */
            RXTX_MIFARE_848,            /**< MIFARE(R) Operating mode at 848kbit/s. */

            RXTX_TYPEB_106 = 0x04,          /**< ISO14443B Operating mode at 106kbit/s. */
            RXTX_TYPEB_212,             /**< ISO14443B Operating mode at 212kbit/s. */
            RXTX_TYPEB_424,             /**< ISO14443B Operating mode at 414kbit/s. */
            RXTX_TYPEB_848,             /**< ISO14443B Operating mode at 848kbit/s. */

            RXTX_FELICA_212 = 0x08,     /**< FeliCa Operating mode at 212kbit/s. */
            RXTX_FELICA_424 = 0x09,     /**< FeliCa Operating mode at 424kbit/s. */

            RXTX_I15693_1OO4 = 0x0A,    /**< ISO15693 One-Out-Of-Four Operating mode. */
            RXTX_I15693_1OO256          /**< ISO15693 One-Out-Of-256 Operating mode. */
        }

        private const byte TX_TWO_CARDS = 0x80;     /**< Set this in the MSB(yte) to have two cards instead of one. */

        public class RxTxConfig_t
        {
            private int Config;

            /// <summary>
            /// Create RxTxConfig
            /// </summary>
            /// <param name="CompCode"></param>
            /// <param name="Error"></param>
            public RxTxConfig_t(RxTx Tx, RxTx Rx, bool DualCards)
            {
                Config = (int)Tx << 8;
                Config |= (int)Rx;

                if (DualCards)
                {
                    Config |= (int)(TX_TWO_CARDS << 8);
                }
            }

            /// <summary>
            /// Convert managed status object to raw status for C-Object
            /// </summary>
            /// <param name="status"></param>
            /// <returns></returns>
            public static implicit operator int(RxTxConfig_t status)
            {
                return status.Config;
            }
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;                    /**< Layer ID for this BAL component, NEVER MODIFY! */
            public IntPtr wsaData;                /**< WsaData */
            public IntPtr pSocketHostName;        /**< Socket Host Name String e.g. "localhost". */
            public IntPtr pSocketPortName;        /**< Socket Port Name String e.g. "1234". */
            public IntPtr pUserName;              /**< User Name String e.g. "nxp12607". */
            public int dwServerSocket;           /**< Server Socket ID */
            public IntPtr pTxBuffer;              /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;              /**< Size of the global transmit buffer. */
            public byte bMode;                    /**< Operating mode. */
            public byte bConfigValue;              /**< current configuration value. */
            public byte bI2cSlaveAddr;              /**< I2C slave address. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_SocketWin_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            byte bMode,                         /**< [In] Operating mode; Refer to the PHBAL_REG_SOCKETWIN_MODE_* defines. */
            ushort wTxBufSize,                  /**< [In] Size of the global transmit buffer. */
            IntPtr pTxBuffer,                   /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wWsaBufferSize,              /**< [In] Size of the Windsocket WSADATA structure buffer. The buffer should be 400 bytes. */
            IntPtr pWsaBuffer                   /**< [In] Pointer to the Winsocket WSADATA structure buffer. */
            );

        #endregion

        #region INIT

        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;

        private byte[] m_bWSADATA;
        private GCHandle m_pWSADATA;

        public Status_t Init(Mode Mode, int wTxBufferSize)
        {
            // Free Buffers
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pWSADATA.IsAllocated)
            {
                this.m_pWSADATA.Free();
            }

            // Allocate TxBuffer
            this.m_bTxBuffer = new byte[wTxBufferSize];
            this.m_pTxBuffer = GCHandle.Alloc(this.m_bTxBuffer, GCHandleType.Pinned);

            // Allocate WSADATA buffer
            ushort WsaSize = 400;
            this.m_bWSADATA = new byte[WsaSize];
            this.m_pWSADATA = GCHandle.Alloc(this.m_bWSADATA, GCHandleType.Pinned);

            return phbalReg_SocketWin_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                (byte)Mode,
                (ushort)wTxBufferSize,
                m_pTxBuffer.AddrOfPinnedObject(),
                WsaSize,
                this.m_pWSADATA.AddrOfPinnedObject()
                );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public SocketWin()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~SocketWin()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (this.m_pTxBuffer.IsAllocated)
            {
                this.m_pTxBuffer.Free();
            }
            if (this.m_pWSADATA.IsAllocated)
            {
                this.m_pWSADATA.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region Rd710Sam

    /// <summary>
    /// Rd710Sam specific HAL-Component of Basic Function Library Framework.
    /// </summary>
    [Obsolete ( "Rd710Sam component is obsolete and will be removed. Use Sam component." )]
    public class Rd710Sam : Bal.Generic
    {
        private const int RESERVED_BUFFER_LEN = 11;        /**< Amount of needed and reserved memory for the protocol overhead. */
        private const int RX_OVERLAP_LEN_EX_MAX = 4;    /**< Max. number of bytes which are to be overlapped for Rx-Buffering. */

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            public ushort wId;              /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pBalClDataParams; /**< Pointer to the BAL which has the link to lower driver layer */
            public IntPtr pAtrBuffer;       /**< Pointer to buffer used for storing the ATR after card activation. */
            public ushort wMaxAtrBufSize;    /**< The maximum size of the ATR buffer. */
            public ushort wAtrBufSize;        /**< Size of the ATR buffer. */
            public IntPtr pTxBuffer;        /**< Pointer to global transmit buffer used by the Exchange() function. */
            public ushort wTxBufSize;       /**< Size of the global transmit buffer. */
            public IntPtr pRxBuffer;        /**< Pointer to global receive buffer used by the Exchange() function. */
            public ushort wRxBufSize;       /**< Size of the global receive buffer. */
            public byte bCardOpened;        /**< Indicator if card was already opened. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phbalReg_Rd710Sam_Init (
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure. */
            ushort wSizeOfDataParams,       /**< [In] Specifies the size of the data parameter structure */
            IntPtr pBalDataParams,          /**< [In] Pointer to the lower layers parameter structure. */
            IntPtr pAtrBuffer,              /**< [In] Pointer to buffer used for storing the ATR after card activation. */
            ushort wAtrBufSize,             /**< [In] Size of the ATR buffer. */
            IntPtr pTxBuffer,               /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
            ushort wTxBufSize,              /**< [In] Size of the global transmit buffer. */
            IntPtr pRxBuffer,               /**< [In] Pointer to global receive buffer used by the Exchange() function. */
            ushort wRxBufSize               /**< [In] Size of the global receive buffer. */
            );

        #endregion

        #region INIT

        private GCHandle m_pAtrBuffer;
        private byte[] m_bTxBuffer;
        private GCHandle m_pTxBuffer;
        private byte[] m_bRxBuffer;
        private GCHandle m_pRxBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pBal"></param>
        /// <returns></returns>
        public Status_t Init ( Bal.Generic pBal, int wTxBufferSize, int wRxBufferSize, out byte[] AtrBuffer )
        {
            // Adjust TxBuffer length
            if ( ( wTxBufferSize + RESERVED_BUFFER_LEN ) > 0xFFFF )
            {
                wTxBufferSize = 0xFFFF;
            }
            else
            {
                wTxBufferSize += RESERVED_BUFFER_LEN;
            }

            // Adjust RxBuffer length
            if ( ( wRxBufferSize + RX_OVERLAP_LEN_EX_MAX ) > 0xFFFF )
            {
                wRxBufferSize = 0xFFFF;
            }
            else
            {
                wRxBufferSize += RX_OVERLAP_LEN_EX_MAX;
            }

            // Free Buffers
            if ( this.m_pAtrBuffer.IsAllocated )
            {
                this.m_pAtrBuffer.Free ();
            }
            if ( this.m_pTxBuffer.IsAllocated )
            {
                this.m_pTxBuffer.Free ();
            }
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }

            // Allocate buffers
            AtrBuffer = new byte[254];
            m_bTxBuffer = new byte[wTxBufferSize];
            m_bRxBuffer = new byte[wRxBufferSize];
            m_pAtrBuffer = GCHandle.Alloc ( AtrBuffer, GCHandleType.Pinned );
            m_pTxBuffer = GCHandle.Alloc ( m_bTxBuffer, GCHandleType.Pinned );
            m_pRxBuffer = GCHandle.Alloc ( m_bRxBuffer, GCHandleType.Pinned );

            // Call init function
            return phbalReg_Rd710Sam_Init (
                                    ref m_DataParamsInt[0],
                                    ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                                    pBal.m_pDataParams,
                                    m_pAtrBuffer.AddrOfPinnedObject (),
                                    ( ushort ) AtrBuffer.Length,
                                    m_pTxBuffer.AddrOfPinnedObject (),
                                    ( ushort ) wTxBufferSize,
                                    m_pRxBuffer.AddrOfPinnedObject (),
                                    ( ushort ) wRxBufferSize );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Rd710Sam ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Rd710Sam ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
            if ( this.m_pAtrBuffer.IsAllocated )
            {
                this.m_pAtrBuffer.Free ();
            }
            if ( this.m_pTxBuffer.IsAllocated )
            {
                this.m_pTxBuffer.Free ();
            }
            if ( this.m_pRxBuffer.IsAllocated )
            {
                this.m_pRxBuffer.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region Sam
    /// <summary>
    /// SAM (Secure Access Module) implementation for Reader Library
    /// </summary>
    public class Sam : Generic
    {
        #region Constants
        /// <summary>
        ///  Amount of needed and reserved memory for the protocol overhead.
        /// </summary>
        private const int RESERVED_BUFFER_LEN = 13;

        /// <summary>
        /// Max. number of bytes which are to be overlapped for Rx-Buffering.
        /// </summary>
        private const int RX_OVERLAP_LEN_EX_MAX = 8;
        #endregion

        #region Enumerations
        /// <summary>
        /// Configuration identifies to be used for <see cref="Generic.GetConfig"/> and <see cref="Generic.SetConfig"/>
        /// interfaces.
        /// </summary>
        public enum Config : ushort
        {
            /// <summary>
            /// Configure communication type. Refer <see cref="CommunicationType"/> for supported values.
            /// </summary>
            COMMUNICATION_TYPE = CustomCodes.CONFIG_BEGIN + 0,

            /// <summary>
            /// Configure I2C Slave Address.
            /// </summary>
            I2C_SLAVE_ADDRESS = CustomCodes.CONFIG_BEGIN + 1,

            /// <summary>
            /// Configure I2C Timeout.
            /// </summary>
            I2C_TIMEOUT_MS = CustomCodes.CONFIG_BEGIN + 2,

            /// <summary>
            /// Configure I2C BitRate. Refer <see cref="I2C_BitRate"/> for supported values.
            /// </summary>
            I2C_BITRATE = CustomCodes.CONFIG_BEGIN + 3
        }

        /// <summary>
        /// Type of communication being enabled for SAM.
        /// </summary>
        public enum CommunicationType : ushort
        {
            /// <summary>
            /// SAM Communication type as TDA. SAM APDU's will be exchanged via TDA interface.
            /// </summary>
            TDA = 0x01,

            /// <summary>
            /// SAM Communication type as I2C. SAM APDU's will be exchanged via I2C interface.
            /// </summary>
            I2C = 0x02
        }

        /// <summary>
        /// Type of I2C Bit Rates to be used.
        /// </summary>
        public enum I2C_BitRate : ushort
        {
            /// <summary>
            /// I2C bit rate configuration value for 100 KBits/s.
            /// </summary>
            STANDARD_MODE = 0x00,

            /// <summary>
            /// I2C bit rate configuration value for 400 KBits/s.
            /// </summary>
            FAST_MODE = 0x01,

            /// <summary>
            /// I2C bit rate configuration value for 1 MBits/s.
            /// </summary>
            FAST_PLUS_MODE = 0x02,

            /// <summary>
            /// I2C bit rate configuration value for 3.4 MBits/s.
            /// </summary>
            HIGH_SPEED_MODE = 0x03
        }
        #endregion

        #region Variables
        private GCHandle pAtrBuffer;

        private GCHandle pTxBuffer;
        private byte[] aTxBuffer;

        private GCHandle pRxBuffer;
        private byte[] aRxBuffer;
        #endregion

        #region Data Structure
        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this HAL component, NEVER MODIFY!
            /// </summary>
            public ushort wId;

            /// <summary>
            /// Pointer to the BAL which actually connects to reader in direct mode.
            /// </summary>
            public IntPtr pLowerBalDataParams;

            /// <summary>
            /// Pointer to buffer used for storing the ATR after card activation.
            /// </summary>
            public IntPtr pAtrBuffer;

            /// <summary>
            /// The maximum size of the ATR buffer.
            /// </summary>
            public ushort wMaxAtrBufSize;

            /// <summary>
            /// Length of bytes available in ATR buffer. This will be filled once ATR is received from SAM.
            /// </summary>
            public ushort wAtrBufSize;

            /// <summary>
            /// Pointer to global transmit buffer used by the Exchange() function.
            /// </summary>
            public IntPtr pTxBuffer;

            /// <summary>
            /// Size of the global transmit buffer.
            /// </summary>
            public ushort wTxBufSize;

            /// <summary>
            /// Pointer to global receive buffer used by the Exchange() function.
            /// </summary>
            public IntPtr pRxBuffer;

            /// <summary>
            /// Size of the global receive buffer.
            /// </summary>
            public ushort wRxBufSize;

            /// <summary>
            /// Communication type to use  for exchanging with SAM. Will be one of the following,
            ///     - <see cref="CommunicationType.TDA"/>
            ///     - <see cref="CommunicationType.I2C"/>
            /// </summary>
            public byte  bCommType;

            /// <summary>
            /// Indicator if card was already opened.
            /// </summary>
            public byte bIsPortOpened;

            /// <summary> Address of I2C Slave. </summary>
            public ushort wI2C_SlaveAddr;

            /// <summary> For saving / configuring I2C communication timeout. </summary>
            public ushort wI2C_TimeoutMs;

            /// <summary>
            /// I2C BitRate. One of the following
            ///     - <see cref="I2C_BitRate.STANDARD_MODE"/>
            ///     - <see cref="I2C_BitRate.FAST_MODE"/>
            ///     - <see cref="I2C_BitRate.FAST_PLUS_MODE"/>
            ///     - <see cref="I2C_BitRate.HIGH_SPEED_MODE"/>
            /// </summary>
            public byte  bI2C_BitRate;

            /// <summary>
            /// Protocol Control Byte.
            /// </summary>
            public byte bI2C_PcbBlockNum;

            /// <summary>
            /// Next Frame Type. Whether its I Frame or S Frame
            /// </summary>
            public byte bI2C_NextFrame;
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phbalReg_Sam_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams,
            IntPtr pBalDataParams, IntPtr pAtrBuffer, ushort wAtrBufSize, IntPtr pTxBuffer, ushort wTxBufSize,
            IntPtr pRxBuffer, ushort wRxBufSize );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialize this component to communicate with SAM using ISO7816 T = 1 protocol.
        /// This component is used to communicate with SAM using Pegoda reader.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the BAL which actually connects to Pegoda or DUT
        ///                    reader in direct mode
        /// </param>
        /// <param name="wTxBufferSize">Size of the global transmit buffer</param>
        /// <param name="wRxBufferSize">Size of the global receive buffer</param>
        /// <param name="aAtrBuffer">Buffer used for storing the ATR after card activation</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if TX and RX Buffer size is zero.
        /// </returns>
        public Status_t Init ( Generic oBal, int wTxBufferSize, int wRxBufferSize, out byte[] aAtrBuffer )
        {
            // Adjust Tx  Buffer length
            wTxBufferSize = ( ( wTxBufferSize + RESERVED_BUFFER_LEN ) > 0xFFFF ) ? 0xFFFF :
                ( wTxBufferSize + RESERVED_BUFFER_LEN );

            // Adjust Rx Buffer length
            wRxBufferSize = ( ( wRxBufferSize + RX_OVERLAP_LEN_EX_MAX ) > 0xFFFF ) ? 0xFFFF :
                ( wRxBufferSize + RX_OVERLAP_LEN_EX_MAX );

            // Free Buffers
            if ( pAtrBuffer.IsAllocated )
            {
                pAtrBuffer.Free ();
                aAtrBuffer = null;
            }

            if ( pTxBuffer.IsAllocated )
            {
                pTxBuffer.Free ();
                aTxBuffer = null;
            }
            if ( pRxBuffer.IsAllocated )
            {
                pRxBuffer.Free ();
                aRxBuffer = null;
            }

            // Allocate buffers
            aAtrBuffer = new byte[254];
            pAtrBuffer = GCHandle.Alloc ( aAtrBuffer, GCHandleType.Pinned );

            aTxBuffer = new byte[wTxBufferSize];
            pTxBuffer = GCHandle.Alloc ( aTxBuffer, GCHandleType.Pinned );

            aRxBuffer = new byte[wRxBufferSize];
            pRxBuffer = GCHandle.Alloc ( aRxBuffer, GCHandleType.Pinned );

            // Call init function
            return phbalReg_Sam_Init ( ref aDataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                oBal.m_pDataParams, pAtrBuffer.AddrOfPinnedObject (), ( ushort ) aAtrBuffer.Length,
                pTxBuffer.AddrOfPinnedObject (), ( ushort ) wTxBufferSize,
                pRxBuffer.AddrOfPinnedObject (), ( ushort ) wRxBufferSize );
        }

#if DEBUG
        /// <summary>
        /// Initialize this component to communicate with SAM using ISO7816 T = 1 or I2C T = 1 protocol.
        /// This component is used to communicate with SAM using Pegoda reader.
        /// </summary>
        ///
        /// <param name="oBal">Pointer to the BAL which actually connects to Pegoda or DUT
        ///                    reader in direct mode
        /// </param>
        /// <param name="bCommType">Communication type to use  for exchanging with SAM. Will be one of the following,
        ///                             - <see cref="CommunicationType.TDA"/>
        ///                             - <see cref="CommunicationType.I2C"/>
        /// </param>
        /// <param name="wTxBufferSize">Size of the global transmit buffer</param>
        /// <param name="wRxBufferSize">Size of the global receive buffer</param>
        /// <param name="aAtrBuffer">Buffer used for storing the ATR after card activation</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if TX and RX Buffer size is zero.
        /// </returns>
        public Status_t Init ( ushort wDataParamSize, Generic oBal, int wTxBufferSize, int wRxBufferSize,
            out byte[] aAtrBuffer )
        {
            // Adjust Tx  Buffer length
            wTxBufferSize = ( ( wTxBufferSize + RESERVED_BUFFER_LEN ) > 0xFFFF ) ? 0xFFFF :
                ( wTxBufferSize + RESERVED_BUFFER_LEN );

            // Adjust Rx Buffer length
            wRxBufferSize = ( ( wRxBufferSize + RX_OVERLAP_LEN_EX_MAX ) > 0xFFFF ) ? 0xFFFF :
                ( wRxBufferSize + RX_OVERLAP_LEN_EX_MAX );

            // Free Buffers
            if ( pAtrBuffer.IsAllocated )
            {
                pAtrBuffer.Free ();
                aAtrBuffer = null;
            }

            if ( pTxBuffer.IsAllocated )
            {
                pTxBuffer.Free ();
                aTxBuffer = null;
            }
            if ( pRxBuffer.IsAllocated )
            {
                pRxBuffer.Free ();
                aRxBuffer = null;
            }

            // Allocate buffers
            aAtrBuffer = new byte[254];
            pAtrBuffer = GCHandle.Alloc ( aAtrBuffer, GCHandleType.Pinned );

            aTxBuffer = new byte[wTxBufferSize];
            pTxBuffer = GCHandle.Alloc ( aTxBuffer, GCHandleType.Pinned );

            aRxBuffer = new byte[wRxBufferSize];
            pRxBuffer = GCHandle.Alloc ( aRxBuffer, GCHandleType.Pinned );

            // Call init function
            return phbalReg_Sam_Init ( ref aDataParamsInt[0], wDataParamSize, oBal.m_pDataParams,
                pAtrBuffer.AddrOfPinnedObject (), ( ushort ) aAtrBuffer.Length, pTxBuffer.AddrOfPinnedObject (),
                ( ushort ) wTxBufferSize, pRxBuffer.AddrOfPinnedObject (), ( ushort ) wRxBufferSize );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] aDataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Sam ()
        {
            // Allocate internal data parameters and pointer to them
            aDataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( aDataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Sam ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
            if ( pAtrBuffer.IsAllocated )
            {
                pAtrBuffer.Free ();
            }
            if ( pTxBuffer.IsAllocated )
            {
                pTxBuffer.Free ();
            }
            if ( pRxBuffer.IsAllocated )
            {
                pRxBuffer.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                aDataParamsInt[0] = value;
            }
            get
            {
                return aDataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public byte CommType
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &aDataParamsInt[0] )
                    {
                        return pDataParams->bCommType;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &aDataParamsInt[0] )
                    {
                        pDataParams->bCommType = value;
                    }
                }
            }
        }

        public byte IsPortOpened
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &aDataParamsInt[0] )
                    {
                        return pDataParams->bIsPortOpened;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &aDataParamsInt[0] )
                    {
                        pDataParams->bIsPortOpened = value;
                    }
                }
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL
    #region MULTISOCKET

    /// <summary>
    /// MultiSocket BAL Component of Basic Function Library Framework.
    /// </summary>
    public class MultiSocket : Bal.Generic
    {
        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;                      /**< Layer ID for this BAL component, NEVER MODIFY! */
            public ushort wTxTimeout;                /**< Tx Timeout in Seconds. */
            public ushort wRxTimeout;                /**< Rx Timeout in Seconds. */
            public IntPtr pHost;                    /**< Socket Host String e.g. "10.0.2.15". */
            public IntPtr pSendPort;                /**< Socket Port String e.g. "22221". */
            public IntPtr pRecvPort;                /**< Socket Port String e.g. "22223". */
            public uint dwSendFd;                   /**< Socket Send File Descriptor. */
            public uint dwRecvFd;                   /**< Socket Recv File Descriptor. */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_MultiSocket_Init(
            ref DataParams_t m_pDataParams      /**< [In] Pointer to this layers parameter structure */
            );

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t Init()
        {
            return phbalReg_MultiSocket_Init(ref m_DataParamsInt[0]);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public MultiSocket()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~MultiSocket()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region PROXILAB
    /// <summary>
    /// ProxiLAB BAL Component of Basic Function Library Framework.
    /// This should work on Linux too, just need to substitute the library file libusb-win32 for libusb.
    /// </summary>
    public class ProxiLAB : Bal.Generic
    {
        #region DEFINES
        private const int VERSION_SIZE = 20;              /**< Size of the reader version and serial strings. */
        private const int ERROR_MESSAGE_SIZE = 1000;      /**< size of the reader error buffer. */
        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public UInt16 wId;                    /**< Layer ID for this BAL component, NEVER MODIFY! */
            public IntPtr pProxiLAB;
            public byte bIsBetaVersion;                        /**< Indicates if the version of the ProxiLAB is a beta or an official version */
            public UInt32 bBoardRevision;                      /**< Indicates ProxiLAB board revision, 1: ProxiLAB V1, 2: ProxiLAB V2, 3: ProxiLAB V3 to 3.3, 34:ProxiLAB V3.4 */
            public fixed byte bHardwareVersion[VERSION_SIZE];  /**< This value stores the hardware version string ("V2.1" for example). */
            public fixed byte bFirmwareVersion[VERSION_SIZE];  /**< This value stores the firmware version string (e.g. "0201"). */
            public fixed byte bSoftwareVersion[VERSION_SIZE];  /**< This value stores the current version of the software as string (1.16.0.598 for example). */
            public fixed byte bProxiLABSerial[VERSION_SIZE];   /**< This value stores the ProxiLABSerial string. */
            public fixed byte bLastErrorMessage[ERROR_MESSAGE_SIZE];   /**< Stores the last error message */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phbalReg_ProxiLAB_Init (
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams        /**< [In] Specifies the size of the data parameter structure */
            );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pReaderName"></param>
        /// <returns></returns>
        public Status_t Init ()
        {
            return phbalReg_ProxiLAB_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ) );
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public ProxiLAB ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~ProxiLAB ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Setter & Getter for DataParams structure
        public string Serial
        {
            get
            {
                char[] serial = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < VERSION_SIZE; i++ )
                        {
                            if ( pDataParams->bProxiLABSerial[i] == 0 )
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        serial = new char[zeroIndex];
                        for ( int i = 0; i < zeroIndex; i++ )
                        {
                            serial[i] = ( char ) pDataParams->bProxiLABSerial[i];
                        }
                    }
                }
                return new string ( serial );
            }
        }

        public string HardwareVersion
        {
            get
            {
                char[] version = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < VERSION_SIZE; i++ )
                        {
                            if ( pDataParams->bHardwareVersion[i] == 0 )
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        version = new char[zeroIndex];
                        for ( int i = 0; i < zeroIndex; i++ )
                        {
                            version[i] = ( char ) pDataParams->bHardwareVersion[i];
                        }
                    }
                }
                return new string ( version );
            }
        }

        public string FirmwareVersion
        {
            get
            {
                char[] version = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < VERSION_SIZE; i++ )
                        {
                            if ( pDataParams->bFirmwareVersion[i] == 0 )
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        version = new char[zeroIndex];
                        for ( int i = 0; i < zeroIndex; i++ )
                        {
                            version[i] = ( char ) pDataParams->bFirmwareVersion[i];
                        }
                    }
                }
                return new string ( version );
            }
        }

        public string SoftwareVersion
        {
            get
            {
                char[] version = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < VERSION_SIZE; i++ )
                        {
                            if ( pDataParams->bSoftwareVersion[i] == 0 )
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        version = new char[zeroIndex];
                        for ( int i = 0; i < zeroIndex; i++ )
                        {
                            version[i] = ( char ) pDataParams->bSoftwareVersion[i];
                        }
                    }
                }
                return new string ( version );
            }
        }

        public string LastErrorMessage
        {
            get
            {
                char[] error = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < ERROR_MESSAGE_SIZE; i++ )
                        {
                            if ( pDataParams->bLastErrorMessage[i] == 0 )
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        error = new char[zeroIndex];
                        for ( int i = 0; i < zeroIndex; i++ )
                        {
                            error[i] = ( char ) pDataParams->bLastErrorMessage[i];
                        }
                    }
                }
                return new string ( error );
            }
        }
        #endregion
    }

    #endregion

    #region RDSCR01

    /// <summary>
    /// RdScr01 BAL Component of Basic Function Library Framework.
    /// </summary>
    public class RdScr01 : Bal.Generic
    {
        #region DEFINES

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;
            public IntPtr pLibraryHandle;                                        /**< Handle of the scr01_dll.dll */
            public uint dwBitRate;
            public uint dwTimeoutValue;
            public uint dwInitVccmv;
            public uint dwInitClockFrequencykHz;
            public IntPtr pfInitVF;
            public IntPtr pfClose;
            public IntPtr pfSleep;
            public IntPtr pfRawSetBaudrate;
            public IntPtr pfRawExchange;
            public IntPtr pfRawGetColdAtr;
            public IntPtr pfRawGetWarmAtr;
            public IntPtr pfSetUartFreq;
            public IntPtr pfSetVcc;
            public IntPtr pf7816Activate;
            public IntPtr pf7816Deactivate;
            public IntPtr pf7816GetAtr;
            public IntPtr pf7816Exchange;
            public IntPtr pf7816ModifyProtocol;
            public IntPtr pf7816ModifyTimeout;
            public IntPtr pf7816GetFdt;
            public IntPtr pfI2CSetControlbyte;
            public IntPtr pfSetIoConfig;
            public IntPtr pfI2CSendData;
            public IntPtr pfI2CReadData;
            public IntPtr pfI2CSetBitrate;
            public IntPtr pfI2CSetParameter;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_RdScr01_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            UInt32 dwInitVccmv,                 /**< [In] Specifies the voltage that should be appled when open. */
            UInt32 dwInitClockFrequencykHz      /**< [In] Specifies the clock frequency that should be appled when open. */
            );

        #endregion

        #region INIT
        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="dwInitVccmv"></param>
        /// <param name="dwInitClockFrequencykHz"></param>
        /// <returns></returns>
        public Status_t Init(uint dwInitVccmv, uint dwInitClockFrequencykHz)
        {
            return phbalReg_RdScr01_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), dwInitVccmv, dwInitClockFrequencykHz);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public RdScr01()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~RdScr01()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region RDSCR2

    /// <summary>
    /// RdScr2 BAL Component of Basic Function Library Framework.
    /// </summary>
    public class RdScr2 : Bal.Generic
    {
        #region DEFINES

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public ushort wId;
            public uint dwBitRate;
            public uint dwTimeoutValue;
            public IntPtr pLibraryHandle;                                        /**< Handle of the scr2_dll.dll */
            public IntPtr pfEnumerate;
            public IntPtr pfInit;
            public IntPtr pfClose;
            public IntPtr pfSleep;
            public IntPtr pf7816RawExchange;
            public IntPtr pf7816RawActivate;
            public IntPtr pf7816SetParameter;
            public IntPtr pfControlIOPort;
            public IntPtr pfSetVcc;
            public IntPtr pfPoweroffReset;
            public IntPtr pf7816Activate;
            public IntPtr pf7816Deactivate;
            public IntPtr pf7816GetAtr;
            public IntPtr pf7816Exchange;
            public IntPtr pf7816GetParameter;
            public IntPtr pfI2CSetSlaveParameter;
            public IntPtr pfSetCommIfConfig;
            public IntPtr pfI2CSendData;
            public IntPtr pfI2CReadData;
            public IntPtr pfI2CSetParameter;
            public IntPtr pfSpiSetParameter;
            public IntPtr pfSpiGetParameter;
            public IntPtr pfSpiProtocolActivate;
            public IntPtr pfSpiProtocolDeactivate;
            public IntPtr pfSpiProtocolExchange;
            public IntPtr pfSpiRawExchange;
            public IntPtr pfSpiRawSend;
            public IntPtr pfSpiRawRead;
            public IntPtr pfTriggerSetPort1;
            public IntPtr pfTriggerSetPort2;
            public IntPtr pfTriggerSetIccValue;
            public IntPtr pfTriggerSetVtpValue;
            public IntPtr pfTriggerSetHwInterfaceWait1;
            public IntPtr pfTriggerSetHwInterfaceWait2;
            public IntPtr pfTdaClearTriggers;
            public IntPtr pfTdaSetTrigger;
            public IntPtr pfTdaStart;
            public IntPtr pfTdaStop;
            public IntPtr pfTdaGetStatus;
            public IntPtr pfExtI2CEnabled;
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_RdScr2_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
            ushort wEnumBufSize,                /**< [In] Size of the enumeration buffer. Maximum possible size is 255 bytes. */
            IntPtr EnumBuffer                   /**< [In] Pointer to the enumeration buffer for the GetPortList() function. */
            );

        #endregion

        #region INIT

        private byte[] m_bEnumBuffer;
        private GCHandle m_pEnumBuffer;

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="bComPort"></param>
        /// <returns></returns>
        public Status_t Init(ushort EnumBufSize)
        {
            // Free Buffers
            if (this.m_pEnumBuffer.IsAllocated)
            {
                this.m_pEnumBuffer.Free();
            }

            // Allocate Buffers
            this.m_bEnumBuffer = new byte[EnumBufSize];
            this.m_pEnumBuffer = GCHandle.Alloc(this.m_bEnumBuffer, GCHandleType.Pinned);

            return phbalReg_RdScr2_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)), EnumBufSize, this.m_pEnumBuffer.AddrOfPinnedObject());
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public RdScr2()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~RdScr2()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
            if (this.m_pEnumBuffer.IsAllocated)
            {
                this.m_pEnumBuffer.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region CONTACTLAB
    /// <summary>
    /// ContactLAB BAL Component of Basic Function Library Framework.
    /// This should work on Linux too, just need to substitute the library file libusb-win32 for libusb.
    /// </summary>
    public class ContactLAB : Bal.Generic
    {
        #region DEFINES
        private const int VERSION_SIZE = 20;              /**< Size of the reader version and serial strings. */
        private const int ERROR_MESSAGE_SIZE = 1000;      /**< size of the reader error buffer. */
        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public UInt16 wId;                    /**< Layer ID for this BAL component, NEVER MODIFY! */
            public IntPtr pProxiLAB;
            public fixed byte bBoardNumber[VERSION_SIZE];               /**< Indicates ContactLAB board number, 1: ContactLAB V1, 2: ContactLAB V2, 3: ContactLAB V3 to 3.3, 34:ContactLAB V3.4 */
            public fixed byte bHardwareVersion[VERSION_SIZE];           /**< This value stores the hardware version string ("V2.1" for example). */
            public fixed byte bFirmwareVersion[VERSION_SIZE];           /**< This value stores the firmware version string (e.g. "0201"). */
            public fixed byte bSoftwareVersion[VERSION_SIZE];           /**< This value stores the current version of the software as string (1.16.0.598 for example). */
            public fixed byte bContactLABSerial[VERSION_SIZE];          /**< This value stores the ContactLABSerial string. */
            public fixed byte bLastErrorMessage[ERROR_MESSAGE_SIZE];    /**< Stores the last error message */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_ContactLAB_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams        /**< [In] Specifies the size of the data parameter structure */
            );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pReaderName"></param>
        /// <returns></returns>
        public Status_t Init()
        {
            return phbalReg_ContactLAB_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public ContactLAB()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~ContactLAB()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        // Setter & Getter for DataParams structure
        public string Serial
        {
            get
            {
                char[] serial = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < VERSION_SIZE; i++)
                        {
                            if (pDataParams->bContactLABSerial[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        serial = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            serial[i] = (char)pDataParams->bContactLABSerial[i];
                        }
                    }
                }
                return new string(serial);
            }
        }

        public string BoardNumber
        {
            get
            {
                char[] boardnumber = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < VERSION_SIZE; i++)
                        {
                            if (pDataParams->bBoardNumber[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        boardnumber = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            boardnumber[i] = (char)pDataParams->bBoardNumber[i];
                        }
                    }
                }
                return new string(boardnumber);
            }
        }

        public string HardwareVersion
        {
            get
            {
                char[] version = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < VERSION_SIZE; i++)
                        {
                            if (pDataParams->bHardwareVersion[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        version = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            version[i] = (char)pDataParams->bHardwareVersion[i];
                        }
                    }
                }
                return new string(version);
            }
        }

        public string FirmwareVersion
        {
            get
            {
                char[] version = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < VERSION_SIZE; i++)
                        {
                            if (pDataParams->bFirmwareVersion[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        version = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            version[i] = (char)pDataParams->bFirmwareVersion[i];
                        }
                    }
                }
                return new string(version);
            }
        }

        public string SoftwareVersion
        {
            get
            {
                char[] version = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < VERSION_SIZE; i++)
                        {
                            if (pDataParams->bSoftwareVersion[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        version = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            version[i] = (char)pDataParams->bSoftwareVersion[i];
                        }
                    }
                }
                return new string(version);
            }
        }

        public string LastErrorMessage
        {
            get
            {
                char[] error = null;
                unsafe
                {
                    int zeroIndex = 0;
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        for (int i = 0; i < ERROR_MESSAGE_SIZE; i++)
                        {
                            if (pDataParams->bLastErrorMessage[i] == 0)
                            {
                                zeroIndex = i;
                                break;
                            }
                        }
                        error = new char[zeroIndex];
                        for (int i = 0; i < zeroIndex; i++)
                        {
                            error[i] = (char)pDataParams->bLastErrorMessage[i];
                        }
                    }
                }
                return new string(error);
            }
        }
        #endregion
    }

    #endregion

    #region SMARTWARE
    /// <summary>
    /// Smartware BAL Component of Basic Function Library Framework.
    /// This should work on Linux too, just need to substitute the library file libusb-win32 for libusb.
    /// </summary>
    public class Smartware : Bal.Generic
    {
        #region DEFINES
        private const int MLOSCFG_BUFFERSIZE      = 316;        /**< Size of internal Buffer to store struct */
        private const int CARD_BUFFERSIZE         = 396;        /**< Size of internal Buffer to store struct */
        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            /// <summary>
            /// Layer ID for this BAL layer, NEVER MODIFY!
            /// </summary>
            public UInt16 wId;                                        /**< Layer ID for this BAL component, NEVER MODIFY! */
            public IntPtr pLibraryHandleMlosCfg;                    /**< Handle of the MLOS.dll */
            public IntPtr pLibraryHandleCard;                       /**< Handle of the CARD.dll */
            public IntPtr pLibraryHandleCardTm;                     /**< Handle of the CARDTM.dll */
            public IntPtr pLibraryHandleCmt;                        /**< Handle of the CMT.dll */
            public fixed byte MlosCfgBuffer[MLOSCFG_BUFFERSIZE];    /**< Buffer for strcut */
            public fixed byte CardBuffer[CARD_BUFFERSIZE];          /**< Buffer for strcut */
            public uint instrumentNode;                             /**< Used Instrument Node */
            public IntPtr pfMlosCfgInit;                            /**< function pointer */
            public IntPtr pfMlosCfgStart;                           /**< function pointer */
            public IntPtr pfMlosCfgEnd;                             /**< function pointer */
            public IntPtr pfCardOpen;                               /**< function pointer */
            public IntPtr pfCardClose;                              /**< function poitner */
            public IntPtr pfCardReset;                              /**< function pointer */
            public IntPtr pfCardResetWarm;                          /**< function pointer */
            public IntPtr pfCardOff;                                /**< function pointer */
            public IntPtr pfCardVcc;                                /**< function pointer */
            public IntPtr pfCardSetClk;                             /**< function pointer */
            public IntPtr pfCardSetFd;                              /**< function pointer */
            public IntPtr pfCardSetGetConfig;                       /**< function pointer */
            public IntPtr pfCardExchange;                           /**< function pointer */
            public IntPtr pfCardTmInit;                             /**< function pointer */
            public IntPtr pfCmtNew;                                 /**< function pointer */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_Smartware_Init(
            ref DataParams_t m_pDataParams, /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams        /**< [In] Specifies the size of the data parameter structure */
            );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="pReaderName"></param>
        /// <returns></returns>
        public Status_t Init()
        {
            return phbalReg_Smartware_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Smartware()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Smartware()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion

    #region VISA

    /// <summary>
    /// VISA BAL Component of Basic Function Library Framework.
    /// </summary>
    public class Visa : Bal.Generic
    {
        #region DEFINES

        public enum Config : int
        {
            TERMCHAR = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN),    /**< Configure communication termination char. */
            ASRL_END_IN = (NxpRdLibNet.CustomCodes.CONFIG_BEGIN + 1) /**< Configure communication end detection mode. */
        }

        public enum EndMode : int
        {
            VI_ASRL_END_NONE      = 0x0000,
            VI_ASRL_END_LAST_BIT  = 0x0001,
            VI_ASRL_END_TERMCHAR  = 0x0002,
            VI_ASRL_END_BREAK     = 0x0003
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            public ushort wId;                    /**< Layer ID for this BAL component, NEVER MODIFY! */
            public IntPtr pVisaPortString;      /**< VISA-Port string. */
            public UInt32 defaultRM;            /**< Default Resource Manager for VISA */
            public UInt32 instr;                /**< instrument address */
            public IntPtr pHandle;              /**< Handle of the visa32.dll. */
            public IntPtr pviOpenDefaultRM;        /**< function pointer for viOpenDefaultRM */
            public IntPtr pviOpen;                /**< function pointer for viOpen */
            public IntPtr pviClose;                /**< function pointer for viClose */
            public IntPtr pviFindRsrc;            /**< function pointer for viFindRsrc */
            public IntPtr pviFindNext;            /**< function pointer for viFindNext */
            public IntPtr pviRead;                /**< function pointer for viRead */
            public IntPtr pviWrite;                /**< function pointer for viWrite */
            public IntPtr pviSetAttribute;        /**< function pointer for viSetAttribute */
            public IntPtr pviGetAttribute;        /**< function pointer for viGetAttribute */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_Visa_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams            /**< [In] Specifies the size of the data parameter structure */
                        );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalReg_Visa_Cmd_Transmit(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            ushort wTxLength        /**< [In] length of input data */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phhalReg_Visa_Cmd_TransmitLarge(
            IntPtr pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wOption,         /**< [In] Option parameter. */
            byte[] pTxBuffer,       /**< [In] data to transmit */
            UInt32 dwTxLength       /**< [In] length of input data */
            );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="bComPort"></param>
        /// <returns></returns>
        public Status_t Init()
        {
            return phbalReg_Visa_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

#if DEBUG
        public Status_t Init(int wDataParamSize)
        {
            return phbalReg_Visa_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize);
        }
#endif

        #endregion

        #region CUSTOM_FUNCTIONS

        /// <summary>
        /// Perform Data Exchange on the bus.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <returns></returns>
        public Status_t Cmd_Transmit(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer        /**< [In] data to transmit */
            )
        {
            if (pTxBuffer.Length > 0xFFFF)
                return new Status_t(Error_CompCode.BAL, Error_Param.PARAMETER_OVERFLOW);
            return phhalReg_Visa_Cmd_Transmit(m_pDataParams, (ushort)wOption, pTxBuffer, (ushort)pTxBuffer.Length);
        }

        /// <summary>
        /// Perform Data Exchange on the bus.
        /// </summary>
        /// <param name="pTxBuffer"></param>
        /// <returns></returns>
        public Status_t Cmd_TransmitLarge(
            int wOption,            /**< [In] Option parameter. */
            byte[] pTxBuffer        /**< [In] data to transmit */
            )
        {
            return phhalReg_Visa_Cmd_TransmitLarge(m_pDataParams, (ushort)wOption, pTxBuffer, (UInt32)pTxBuffer.Length);
        }

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Visa()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Visa()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        #endregion
    }

    #endregion
#endif

    #region RdFpga

    /// <summary>
    /// FPGA BAL Component of Basic Function Library Framework.
    /// </summary>
    public class RdFpga : Bal.Generic
    {
        #region DEFINES

        public enum Config : int
        {
            ADDITIONAL_INFO = (CustomCodes.CONFIG_BEGIN)       /**< Set / Get additional information . */
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public struct DataParams_t
        {
            public ushort wId;                                      /**< Layer ID for this BAL layer, NEVER MODIFY!*/
            public IntPtr pDeviceName;                              /**< Number of communication port. */
            public byte bPortOpen;                                    /**< Internal State if port is opened */
            public ushort wAdditionalInfo;                            /**< Storage for additional error information. */
            public ushort wTimeout;                                    /**< shadow for the timeout. */
            public IntPtr pHandle;                                  /**< Handle of the phCDllReaderLib.dll. */
            public IntPtr pfRdFpga_Open;                            /**< function pointer for pfRdFpga_Open */
            public IntPtr pfRdFpga_Close;                            /**< function pointer for pfRdFpga_Close */
            public IntPtr pfRdFpga_Init;                            /**< function pointer for pfRdFpga_Init */
            public IntPtr pfRdFpga_SetParam;                        /**< function pointer for pfRdFpga_SetParam */
            public IntPtr pfRdFpga_GetParam;                        /**< function pointer for pfRdFpga_GetParam */
            public IntPtr pfRdFpga_ExchangeBits;                    /**< function pointer for pfRdFpga_ExchangeBits */
            public IntPtr pfRdFpga_SetTargetJoiner;                    /**< function pointer for pfRdFpga_SetTargetJoiner */
            public IntPtr pfRdFpga_SetJoinerReg;                    /**< function pointer for pfRdFpga_SetJoinerReg */
            public IntPtr pfRdFpga_GetJoinerReg;                    /**< function pointer for pfRdFpga_GetJoinerReg */
            public IntPtr pfRdFpga_LoadJoinerIniFile;                /**< function pointer for pfRdFpga_LoadJoinerIniFile */
            public IntPtr pfRdFpga_ActivateField;                    /**< function pointer for pfRdFpga_ActivateField */
            public IntPtr pfRdFpga_DeactivateField;                    /**< function pointer for pfRdFpga_DeactivateField */
            public IntPtr pfRdFpga_HFPause;                            /**< function pointer for pfRdFpga_HFPause */
            public IntPtr pfRdFpga_SetPotiPos;                        /**< function pointer for pfRdFpga_SetPotiPos */
            public IntPtr pfRdFpga_SetDefaultPotiPos;                /**< function pointer for pfRdFpga_SetDefaultPotiPos */
            public IntPtr pfRdFpga_GetPotiPos;                        /**< function pointer for pfRdFpga_GetPotiPos */
            public IntPtr pfRdFpga_IncPotiPos;                        /**< function pointer for pfRdFpga_IncPotiPos */
            public IntPtr pfRdFpga_DecPotiPos;                        /**< function pointer for pfRdFpga_DecPotiPos */
            public IntPtr pfRdFpga_SetTimeoutValue;                    /**< function pointer for pfRdFpga_SetTimeoutValue */
            public IntPtr pfRdFpga_GetTimeoutValue;                    /**< function pointer for pfRdFpga_GetTimeoutValue */
            public IntPtr pfRdFpga_LoadPattern;                        /**< function pointer for pfRdFpga_LoadPattern */
            public IntPtr pfRdFpga_PlayPattern;                        /**< function pointer for pfRdFpga_PlayPattern */
            public IntPtr pfRdFpga_PlaySeqA;                        /**< function pointer for pfRdFpga_PlaySeqA */
            public IntPtr pfRdFpga_PlaySeqB;                        /**< function pointer for pfRdFpga_PlaySeqB */
            public IntPtr pfRdFpga_PlaySinus;                        /**< function pointer for pfRdFpga_PlaySinus */
            public IntPtr pfRdFpga_SelectTrigger;                    /**< function pointer for pfRdFpga_SelectTrigger */
            public IntPtr pfRdFpga_LabelReset;                        /**< function pointer for pfRdFpga_LabelReset */
            public IntPtr pfRdFpga_LabelResetCounterReset;            /**< function pointer for pfRdFpga_LabelResetCounterReset */
            public IntPtr pfRdFpga_LabelResetCounterResetSingleShot;/**< function pointer for pfRdFpga_LabelResetCounterResetSingleShot */
            public IntPtr pfRdFpga_LabelResetSingleShot;            /**< function pointer for pfRdFpga_LabelResetSingleShot */
            public IntPtr pfRdFpga_LabelResetWithCommand;            /**< function pointer for pfRdFpga_LabelResetWithCommand */
            public IntPtr pfRdFpga_ActivateFDTMeasurement;            /**< function pointer for pfRdFpga_ActivateFDTMeasurement */
            public IntPtr pfRdFpga_GetFDT;                            /**< function pointer for pfRdFpga_GetFDT */
            public IntPtr pfRdFpga_GetFDTFast;                        /**< function pointer for pfRdFpga_GetFDTFast */
            public IntPtr pfRdFpga_ActivateAntiTearingMode;            /**< function pointer for pfRdFpga_ActivateAntiTearingMode */
            public IntPtr pfRdFpga_ResonanceFrequenceTest;            /**< function pointer for pfRdFpga_ResonanceFrequenceTest */
            public IntPtr pfRdFpga_ActivateFDTMeasurementPICCtoPCD;    /**< function pointer for pfRdFpga_ActivateFDTMeasurementPICCtoPCD */
            public IntPtr pfRdFpga_ActivateLMAMeasurement;            /**< function pointer for pfRdFpga_ActivateLMAMeasurement */
            public IntPtr pfRdFpga_ActivateSequencerMode;            /**< function pointer for pfRdFpga_ActivateSequencerMode */
            public IntPtr pfRdFpga_DeactivateModes;                    /**< function pointer for pfRdFpga_DeactivateModes */
            public IntPtr pfRdFpga_SetJoinerRxGain;                    /**< function pointer for pfRdFpga_SetJoinerRxGain */
            public IntPtr pfRdFpga_GetJoinerRxGain;                    /**< function pointer for pfRdFpga_GetJoinerRxGain */
            public IntPtr pfRdFpga_SetFDTPCDforActiveDS;            /**< function pointer for pfRdFpga_SetFDTPCDforActiveDS */
            public IntPtr pfRdFpga_SetFDTPCDforDS;                    /**< function pointer for pfRdFpga_SetFDTPCDforDS */
            public IntPtr pfRdFpga_GetVersion;                      /**< function pointer for pfRdFpga_GetVersion */
            public IntPtr pfRdFpga_GetRevision;                     /**< function pointer for pfRdFpga_GetRevision */
            public IntPtr pfRdFpga_GetErrorCodes;                   /**< function pointer for pfRdFpga_GetErrorCodes */
            public IntPtr pfRdFpga_SetClockSource;                  /**< function pointer for pfRdFpga_SetClockSource */
            public IntPtr pfRdFpga_GetFirstResponse;                /**< function pointer for pfRdFpga_GetFirstResponse */
            public IntPtr pfRdFpga_GetResponse;                     /**< function pointer for pfRdFpga_GetResponse */
            public IntPtr pfRdFpga_GetIQTimeoutValue;               /**< function pointer for pfRdFpga_GetIQTimeoutValue */
            public IntPtr pfRdFpga_SetIQTimeoutValue;               /**< function pointer for pfRdFpga_SetIQTimeoutValue */
            public IntPtr pfRdFpga_GetTerminalId;                   /**< function pointer for pfRdFpga_GetTerminalId */
            public IntPtr pfRdFpga_ResetJoiner;                     /**< function pointer for pfRdFpga_ResetJoiner */
            public IntPtr pfRdFpga_LoadFilterFile;                    /**< function pointer for pfRdFpga_LoadFilterFile */
            public IntPtr pfRdFpga_ExchangeFPGACommand;                  /**< function pointer for pfRdFpga_ExchangeFPGACommand */
            public IntPtr pfRdFpga_ActivateTypeBMode;                 /**< function pointer for pfRdFpga_ActivateTypeBMode */
        };

        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_RdFpgaV6_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams            /**< [In] Specifies the size of the data parameter structure */
                        );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="bComPort"></param>
        /// <returns></returns>
        public Status_t Init()
        {
            return phbalReg_RdFpgaV6_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

#if DEBUG
        public Status_t Init(int wDataParamSize)
        {
            return phbalReg_RdFpgaV6_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public RdFpga()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~RdFpga()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }


        #endregion
    }

    #endregion

    #region Ps6000Usb

    /// <summary>
    /// PicoScope BAL Component of Basic Function Library Framework.
    /// </summary>
    public class Ps6000Usb : Bal.Generic
    {
        #region DEFINES

        private const int GET_UNIT_INFO_BUFFER_SIZE = 16;          /**< The size of the serial number. */


        public enum Config : int
        {
            ADDITIONAL_INFO = (CustomCodes.CONFIG_BEGIN)       /**< Set / Get additional information . */
        }

        #endregion

        #region DATA_STRUCTURE

        /// <summary>
        /// Private data storage definition of underlying C Object.
        /// </summary>
        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                                   /**< Layer ID for this BAL component, NEVER MODIFY! */
            public IntPtr pDeviceName;                                           /**< Name of PicoScope Osci device. */
            public short pDeviceHandle;                                          /**< Handle of PicoScope Osci device. */
            public byte bPortOpen;                                               /**< Internal State if port is opened. */
            public uint wAdditionalInfo;                                         /**< Storage for additional error information. */
            public ushort wTimeout;                                              /**< Shadow for the timeout. */
            public IntPtr pLibraryHandle;                                        /**< Handle of the PS6000.dll. */

            public fixed byte pDriverVersion[GET_UNIT_INFO_BUFFER_SIZE];         /**< Version number of PicoScope 6000 DLL */
            public fixed byte pUSBVersion[GET_UNIT_INFO_BUFFER_SIZE];            /**< Type of USB connection to device: 1.1, 2.0 or 3.0 */
            public fixed byte pHardwareVersion[GET_UNIT_INFO_BUFFER_SIZE];       /**< Hardware version of device */
            public fixed byte pVariantInfo[GET_UNIT_INFO_BUFFER_SIZE];           /**< Model number of device */
            public fixed byte pBatchAndSerial[GET_UNIT_INFO_BUFFER_SIZE];        /**< Batch and serial number of device */
            public fixed byte pCalibrationDate[GET_UNIT_INFO_BUFFER_SIZE];       /**< Calibration date of device */
            public fixed byte pKernelVersion[GET_UNIT_INFO_BUFFER_SIZE];         /**< Version of kernel driver */
            public fixed byte pDigitalHardwareVersion[GET_UNIT_INFO_BUFFER_SIZE];/**< Hardware version of the digital section */
            public fixed byte pAnalogueHardwareVersion[GET_UNIT_INFO_BUFFER_SIZE];/**< Hardware version of the analogue section */
            public fixed byte pFirmwareVersion1[GET_UNIT_INFO_BUFFER_SIZE];      /**< Version information of Firmware 1 */
            public fixed byte pFirmwareVersion2[GET_UNIT_INFO_BUFFER_SIZE];      /**< Version information of Firmware 2 */

            public IntPtr pfPs6000Usb_OpenUnit;                                  /**< function pointer for pfPs6000Usb_OpenUnit. */
            public IntPtr pfPs6000Usb_OpenUnitAsync;                             /**< function pointer for pfPs6000Usb_OpenUnitAsync. */
            public IntPtr pfPs6000Usb_OpenUnitProgress;                          /**< function pointer for pfPs6000Usb_OpenUnitProgress. */
            public IntPtr pfPs6000Usb_GetUnitInfo;                               /**< function pointer for pfPs6000Usb_GetUnitInfo. */
            public IntPtr pfPs6000Usb_FlashLed;                                  /**< function pointer for pfPs6000Usb_FlashLed. */
            public IntPtr pfPs6000Usb_CloseUnit;                                 /**< function pointer for pfPs6000Usb_CloseUnit. */
            public IntPtr pfPs6000Usb_MemorySegments;                            /**< function pointer for pfPs6000Usb_MemorySegments. */
            public IntPtr pfPs6000Usb_SetChannel;                                /**< function pointer for pfPs6000Usb_SetChannel. */
            public IntPtr pfPs6000Usb_GetTimebase;                               /**< function pointer for pfPs6000Usb_GetTimebase. */
            public IntPtr pfPs6000Usb_GetTimebase2;                              /**< function pointer for pfPs6000Usb_GetTimebase2. */
            public IntPtr pfPs6000Usb_SetSigGenArbitrary;                        /**< function pointer for pfPs6000Usb_SetSigGenArbitrary. */
            public IntPtr pfPs6000Usb_SetSigGenBuiltIn;                          /**< function pointer for pfPs6000Usb_SetSigGenBuiltIn. */
            public IntPtr pfPs6000Usb_SigGenSoftwareControl;                     /**< function pointer for pfPs6000Usb_SigGenSoftwareControl. */
            public IntPtr pfPs6000Usb_SetSimpleTrigger;                          /**< function pointer for pfPs6000Usb_SetSimpleTrigger. */
            public IntPtr pfPs6000Usb_SetEts;                                    /**< function pointer for pfPs6000Usb_SetEts. */
            public IntPtr pfPs6000Usb_SetTriggerChannelProperties;               /**< function pointer for pfPs6000Usb_SetTriggerChannelProperties. */
            public IntPtr pfPs6000Usb_SetTriggerChannelConditions;               /**< function pointer for pfPs6000Usb_SetTriggerChannelConditions. */
            public IntPtr pfPs6000Usb_SetTriggerChannelDirections;               /**< function pointer for pfPs6000Usb_SetTriggerChannelDirections. */
            public IntPtr pfPs6000Usb_SetTriggerDelay;                           /**< function pointer for pfPs6000Usb_SetTriggerDelay. */
            public IntPtr pfPs6000Usb_SetPulseWidthQualifier;                    /**< function pointer for pfPs6000Usb_SetPulseWidthQualifier. */
            public IntPtr pfPs6000Usb_IsTriggerOrPulseWidthQualifierEnabled;     /**< function pointer for pfPs6000Usb_IsTriggerOrPulseWidthQualifierEnabled. */
            public IntPtr pfPs6000Usb_GetTriggerTimeOffset;                      /**< function pointer for pfPs6000Usb_GetTriggerTimeOffset. */
            public IntPtr pfPs6000Usb_GetTriggerTimeOffset64;                    /**< function pointer for pfPs6000Usb_GetTriggerTimeOffset64. */
            public IntPtr pfPs6000Usb_GetValuesTriggerTimeOffsetBulk;            /**< function pointer for pfPs6000Usb_GetValuesTriggerTimeOffsetBulk. */
            public IntPtr pfPs6000Usb_GetValuesTriggerTimeOffsetBulk64;          /**< function pointer for pfPs6000Usb_GetValuesTriggerTimeOffsetBulk64. */
            public IntPtr pfPs6000Usb_SetDataBuffers;                            /**< function pointer for pfPs6000Usb_SetDataBuffers. */
            public IntPtr pfPs6000Usb_SetDataBuffer;                             /**< function pointer for pfPs6000Usb_SetDataBuffer. */
            public IntPtr pfPs6000Usb_SetDataBufferBulk;                         /**< function pointer for pfPs6000Usb_SetDataBufferBulk. */
            public IntPtr pfPs6000Usb_SetDataBuffersBulk;                        /**< function pointer for pfPs6000Usb_SetDataBuffersBulk. */
            public IntPtr pfPs6000Usb_SetEtsTimeBuffer;                          /**< function pointer for pfPs6000Usb_SetEtsTimeBuffer. */
            public IntPtr pfPs6000Usb_SetEtsTimeBuffers;                         /**< function pointer for pfPs6000Usb_SetEtsTimeBuffers. */
            public IntPtr pfPs6000Usb_RunBlock;                                  /**< function pointer for pfPs6000Usb_RunBlock. */
            public IntPtr pfPs6000Usb_IsReady;                                   /**< function pointer for pfPs6000Usb_IsReady. */
            public IntPtr pfPs6000Usb_RunStreaming;                              /**< function pointer for pfPs6000Usb_RunStreaming. */
            public IntPtr pfPs6000Usb_GetStreamingLatestValues;                  /**< function pointer for pfPs6000Usb_GetStreamingLatestValues. */
            public IntPtr pfPs6000Usb_NoOfStreamingValues;                       /**< function pointer for pfPs6000Usb_NoOfStreamingValues. */
            public IntPtr pfPs6000Usb_GetMaxDownSampleRatio;                     /**< function pointer for pfPs6000Usb_GetMaxDownSampleRatio. */
            public IntPtr pfPs6000Usb_GetValues;                                 /**< function pointer for pfPs6000Usb_GetValues. */
            public IntPtr pfPs6000Usb_GetValuesBulk;                             /**< function pointer for pfPs6000Usb_GetValuesBulk. */
            public IntPtr pfPs6000Usb_GetValuesAsync;                            /**< function pointer for pfPs6000Usb_GetValuesAsync. */
            public IntPtr pfPs6000Usb_GetValuesOverlapped;                       /**< function pointer for pfPs6000Usb_GetValuesOverlapped. */
            public IntPtr pfPs6000Usb_GetValuesOverlappedBulk;                   /**< function pointer for pfPs6000Usb_GetValuesOverlappedBulk. */
            public IntPtr pfPs6000Usb_GetValuesBulkAsync;                        /**< function pointer for pfPs6000Usb_GetValuesBulkAsync. */
            public IntPtr pfPs6000Usb_GetNoOfCaptures;                           /**< function pointer for pfPs6000Usb_GetNoOfCaptures. */
            public IntPtr pfPs6000Usb_Stop;                                      /**< function pointer for pfPs6000Usb_Stop. */
            public IntPtr pfPs6000Usb_SetNoOfCaptures;                           /**< function pointer for pfPs6000Usb_SetNoOfCaptures. */
            public IntPtr pfPs6000Usb_SetWaveformLimiter;                        /**< function pointer for pfPs6000Usb_SetWaveformLimiter. */
            public IntPtr pfPs6000Usb_EnumerateUnits;                            /**< function pointer for pfPs6000Usb_EnumerateUnits. */
            public IntPtr pfPs6000Usb_SetExternalClock;                          /**< function pointer for pfPs6000Usb_SetExternalClock. */
            public IntPtr pfPs6000Usb_PingUnit;                                  /**< function pointer for pfPs6000Usb_PingUnit. */
        };
        #endregion

        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phbalReg_Ps6000Usb_Init(
            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure */
            ushort wSizeOfDataParams            /**< [In] Specifies the size of the data parameter structure */
                        );

        #endregion

        #region INIT

        /// <summary>
        /// Initialise this component.
        /// </summary>
        /// <param name="bComPort"></param>
        /// <returns></returns>
        public Status_t Init()
        {
            return phbalReg_Ps6000Usb_Init(ref m_DataParamsInt[0], (ushort)Marshal.SizeOf(typeof(DataParams_t)));
        }

#if DEBUG
        public Status_t Init(int wDataParamSize)
        {
            return phbalReg_Ps6000Usb_Init(ref m_DataParamsInt[0], (ushort)wDataParamSize);
        }
#endif

        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Allocate unmanaged memory for underlying C-Object
        /// </summary>
        public Ps6000Usb()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }

        /// <summary>
        /// Free allocated unmanaged memory.
        /// </summary>
        ~Ps6000Usb()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }

        unsafe private string ByteArrayToString(byte* arr, int maxLength)
        {
            char[] str = null;
            int zeroIndex = maxLength;
            unsafe
            {
                for (int i = 0; i < maxLength; i++)
                {
                    if (arr[i] == 0)
                    {
                        zeroIndex = i;
                        break;
                    }
                }
                str = new char[zeroIndex];
                for (int i = 0; i < zeroIndex; i++)
                {
                    str[i] = (char)arr[i];
                }
            }
            return new string(str);
        }

        public string VersionString
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        string value = "";// Serial;
                        value += " DriverVersion:" + DriverVersion;
                        value += " USBVersion:" + USBVersion;
                        value += " HardwareVersion:" + HardwareVersion;
                        value += " KernelVersion:" + KernelVersion;
                        value += " DigitalHardwareVersion:" + DigitalHardwareVersion;
                        value += " AnalogueHardwareVersion:" + AnalogueHardwareVersion;
                        value += " FirmwareVersion1:" + FirmwareVersion1;
                        value += " FirmwareVersion2:" + FirmwareVersion2;
                        return value;
                    }
                }
            }
        }

        public string DriverVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pDriverVersion, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string USBVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pUSBVersion, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string HardwareVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pHardwareVersion, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string VariantInfo
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pVariantInfo, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string BatchAndSerial
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pBatchAndSerial, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string CalibrationDate
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pCalibrationDate, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string KernelVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pKernelVersion, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string DigitalHardwareVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pDigitalHardwareVersion, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string AnalogueHardwareVersion
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pAnalogueHardwareVersion, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string FirmwareVersion1
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pFirmwareVersion1, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        public string FirmwareVersion2
        {
            get
            {
                unsafe
                {
                    fixed (DataParams_t* pDataParams = &this.m_DataParamsInt[0])
                    {
                        return ByteArrayToString(pDataParams->pFirmwareVersion2, GET_UNIT_INFO_BUFFER_SIZE);
                    }
                }
            }
        }

        #endregion
    }

    #endregion
}
