/*
 * Copyright 2013, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alP40CmdPriv
{
    #region DEFINES

    public enum Error : byte
    {
        /// <summary> P40 Unknown Command Error. </summary>
        CMD = (CustomCodes.ERROR_BEGIN),
        /// <summary> P40 Invalid Parameters Error. </summary>
        PARAM,
        /// <summary> P40 Bad Command Format Error. </summary>
        FORMAT,
        /// <summary> P40 Command Execution Error. </summary>
        EXECUTION,
        /// <summary> P40 Not Authorized Error. </summary>
        NOT_AUTH,
        /// <summary> P40 Response Length Error. </summary>
        RESP,
        /// <summary> P40 Generic Error. </summary>
        UNKNOWN
    }

    public enum CrcType : byte
    {
        Crc8 = 0x00,
        Crc16 = 0x01,
        Crc32 = 0x10
    }

    public enum Random : byte
    {
        OFF = 0x00,
        ON = 0xFF
    }

    public enum KeyType : byte
    {
        WaferTest = 0,
        FinalTest = 1
    }
    #endregion

    #region VERSION_INFO
    public class VersionInfo
    {
        //define the struct used for passing
        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct VersionInfo_t
        {
            public ushort wIcVer;                    /**<  IC version version */
            public ushort wRomCodeVer;      /**< ROM code version */
            public ushort wWaferTestVer;      /**< WaferTest Program version */
            public byte bLifecycle;              /**< Current Lifeycle */
            /* reserved bytes for struct so total size is 16*/
            public fixed byte bReserve[9];
        };

        #endregion

        #region GETTERS/SETTERS
        // Setter & Getter for VersionInfo_t struct
        public VersionInfo_t Data
        {
            get
            {
                return this.m_version;
            }
            set
            {
                this.m_version = value;
            }
        }

        public ushort Ic
        {
            get
            {
                return this.m_version.wIcVer;
            }
        }

        public ushort RomCode
        {
            get
            {
                return this.m_version.wRomCodeVer;
            }
        }

        public ushort WaferTest
        {
            get
            {
                return this.m_version.wWaferTestVer;
            }
        }

        public byte Lifecycle
        {
            get
            {
                return this.m_version.bLifecycle;
            }
        }

        #endregion


        private VersionInfo_t m_version;

        public VersionInfo()
        {
            m_version = new VersionInfo_t();
        }
    }
    #endregion

    #region BASE

    public abstract class Generic
    {
        #region DLLIMPORTS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_GetVersionP(
                                            IntPtr pDataParams,	                   /**< [In] Pointer to this layer's parameter structure. */
                                            ref VersionInfo.VersionInfo_t pData   /**< Out] The data  structure to write to */
            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_WriteMemoryP(
                                            IntPtr pDataParams,                 	/**< [In] Pointer to this layer's parameter structure. */
                                            ushort wStartAddr,			            /**< [In] Specify starting address to write */
                                            byte bUseRandom,	                    /**< [In] Specify if P40 should write random data to memory */
                                            byte bDataLength,		                /**< [In] Number of bytes to write, 0 = 256 bytes  */
                                            byte[] pData					        /**< [In] Bytes to write  if bUseRandomData = 0*/
                                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_ReadMemoryP(
                                            IntPtr pDataParams,	                    /**< [In] Pointer to this layer's parameter structure. */
                                            ushort wStartAddr,			            /**< [In] Specify starting address to read from */
                                            byte bDataLength,		                /**< [In] requested number of bytes to read  */
                                            byte[] pData,				            /**< [Out] Buffer to store read data*/
                                            ref ushort pReadLength		            /**< [Out] Actual number of bytes read*/
                                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_CalculateChecksumP(
                                            IntPtr pDataParams,	                    /**< [In] Pointer to this layer's parameter structure. */
                                            ushort wStartAddr,			            /**< [In] Specify starting address to read from */
                                            ushort wLength,		                    /**< [In] Number of bytes to calculate over  */
                                            byte bCksumType,                        /**< [In] Type of crc to calculate  */
                                            ushort wDestAddr,			            /**< [In] Optional address to store computed checksum on chip. 0 = not specified. */
                                            ref byte[] pCrc		                    /**< [Out] Actual number of bytes read*/
                                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_ExecuteP(
                                            IntPtr pDataParams,                    /**< [In] Pointer to this layer's parameter structure. */
                                            ushort wExeAddr,                      /**< [In] Entry point of routine */
                                            byte bMagic,                               /**< [In] Sys or user call, 0xff = use address  */
                                            byte bParamLength,                   /**< [In] Number of parameter bytes  */
                                            byte[] pParamBuffer,                  /**< [In] Buffer to store read data*/
                                            byte bRespSize,                         /**< [In] Size of response buffer  */
                                            byte[] pRespBuffer,	                    /**< [Out] Buffer to store function response data*/
                                            ref ushort pRespLength	            /**< [Out] Number of returned data bytes written or expected to write in case of overflow. */
                                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_AuthenticateP(
                                            IntPtr pDataParams,                    /**< [In] Pointer to this layer's parameter structure. */
                                            byte bKeyId,                              /**< [In] Ident of Key to authenticate to. */
                                            ushort wKeyNo,                        /**< [In] Key Number in KeyStore */
                                            ushort wKeyVersion                /**< [In] Key version to use*/
                                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_FastTestP(
                                            IntPtr pDataParams,		                /**< [In] Pointer to this layer's parameter structure. */
                                            byte bTestCode		                    /**< [In] Ident of subtest to perform. */
                                            );

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_SwitchLifeCycleP(
                                            IntPtr pDataParams,		                /**< [In] Pointer to this layer's parameter structure. */
                                            byte bLifeCycle		                    /**< [In] Intended LC. */
                                            );


        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        public Status_t GetVersionP(out VersionInfo Ver)
        {
            Status_t status;

            Ver = new VersionInfo();
            VersionInfo.VersionInfo_t vi = new VersionInfo.VersionInfo_t();
            status = phalP40CmdPriv_GetVersionP(m_pDataParams, ref vi);
            Ver.Data = vi;
            return status;
        }

        public Status_t WriteMemoryP(
            int wStart,
            Random eRandom,
            int bLength,
            byte[] bData)
        {
            return phalP40CmdPriv_WriteMemoryP(m_pDataParams, (ushort)wStart, (byte)eRandom, (byte)bLength, bData);
        }

        public Status_t ReadMemoryP(
            int wStart,
            out byte[] bData,
            int bLen)
        {
            Status_t status;
            ushort pReadLen = new ushort();
            byte[] tmpBuf = new byte[bLen];

            status = phalP40CmdPriv_ReadMemoryP(m_pDataParams, (ushort)wStart, (byte)bLen, tmpBuf, ref pReadLen);
            if (!status.Equals(Error_Gen.SUCCESS))
            {
                bData = new byte[0];
            }
            else
            {
                bData = new byte[pReadLen];
                tmpBuf.CopyTo(bData, 0);
            }

            return status;
        }

        public Status_t CalculateChecksumP(
            int wStart,
            int wLength,
            CrcType eType,
            int wDest,
            out byte[] pCrc)
        {
            Status_t status;
            switch (eType)
            {
                case CrcType.Crc8:
                    {
                        pCrc = new byte[1];
                        break;
                    }
                case CrcType.Crc16:
                    {
                        pCrc = new byte[2];
                        break;
                    }
                case CrcType.Crc32:
                    {
                        pCrc = new byte[4];
                        break;
                    }
                default:
                    {
                        pCrc = new byte[0];
                        return new Status_t(Error_CompCode.AL_P40CMDPRIV, Error_Param.INVALID_PARAMETER);
                    }
            }
            status = phalP40CmdPriv_CalculateChecksumP(m_pDataParams, (ushort)wStart, (ushort)wLength, (byte)eType, (ushort)wDest, ref pCrc);
            if (!status.Equals(Error_Gen.SUCCESS))
            {
                pCrc = new byte[0];
            }
            return status;
        }

        public Status_t ExecuteP(
            int  wStartAddr,
            int bMagic,
            byte[] pParams,
            out byte[] pRespBuffer,
            int wRespLen)
        {
            Status_t status;
            // execute based on the expected response length
            ushort wTmpRespLen = new ushort();
            wTmpRespLen = (ushort)wRespLen;

            byte[] pTmpRespBuffer = new byte[wRespLen];

            status = phalP40CmdPriv_ExecuteP(m_pDataParams,
                          (ushort)wStartAddr, (byte)bMagic, (byte)pParams.Length,
                          pParams, (byte)pTmpRespBuffer.Length,
                          pTmpRespBuffer, ref wTmpRespLen);

            if (status.Equals(Error.RESP))
            {
                // user supplied too short length, we can try again with responded length
                pTmpRespBuffer = new byte[wTmpRespLen];
                wRespLen = wTmpRespLen;
                status = phalP40CmdPriv_ExecuteP(m_pDataParams,
                             (ushort)wStartAddr, (byte)bMagic, (byte)pParams.Length,
                              pParams, (byte)pTmpRespBuffer.Length,
                              pTmpRespBuffer, ref wTmpRespLen);
            }

            if (!status.Equals(Error_Gen.SUCCESS))
            {
                pTmpRespBuffer = new byte[0];
            }
            else
            {
                // readjust response buffer size if expected length was too great
                System.Array.Resize<byte>(ref  pTmpRespBuffer, wTmpRespLen);
            }

            pRespBuffer = pTmpRespBuffer;
            return status;
        }


        public Status_t AuthenticateP(
            int bKeyId,
            int wKeyNo,
            int wKeyVersion)
        {
            return phalP40CmdPriv_AuthenticateP(m_pDataParams, (byte)bKeyId, (ushort)wKeyNo, (ushort)wKeyVersion);
        }


        public Status_t FastTestP(
            int bTestCode)
        {
            return phalP40CmdPriv_FastTestP(m_pDataParams, (byte)bTestCode);
        }


        public Status_t SwitchLifeCycleP(
            int bLifeCycle)
        {
            return phalP40CmdPriv_SwitchLifeCycleP(
                                                m_pDataParams,
                                                (byte)bLifeCycle);
        }


        #endregion

        /// <summary>
        /// Abstract definition for private data storage of underlying C Object.
        /// </summary>
        public abstract IntPtr m_pDataParams
        {
            get;
        }

    }
    #endregion

    #region Sw

    public class Sw : Generic
    {

        #region DATA_STRUCTURE

        [StructLayout(LayoutKind.Sequential, Pack = 1)]
        public unsafe struct DataParams_t
        {
            public ushort wId;                                 /**< Layer ID for this HAL component, NEVER MODIFY! */
            public IntPtr pHalDataParams;            /**< Pointer to the underlying HAL layer. */
            public IntPtr pKeyStoreDataParams;     /**< Pointer to the parameter structure of the KeyStore component. */
        };

        #endregion

        #region DLL_WRAPPED_FUNCTIONS

        [DllImport(Common.IMPORT_LIBRARY_NAME)]
        private static extern ushort phalP40CmdPriv_Sw_Init(
                                            ref DataParams_t m_pDataParams,     /**< [In] Pointer to this layers parameter structure. */
                                            ushort wSizeOfDataParams,           /**< [In] Specifies the size of the data parameter structure */
                                            IntPtr pHal,                                       /**< [In] Pointer to the parameter structure of the Hal layer. */
                                            IntPtr pKeyStoreDataParams          /**< [In] Pointer to the parameter structure of the KeyStore component. */
            );
        #endregion

        #region INIT
        public Status_t Init(
            Hal.Generic pHal,
            KeyStore.Generic pKeyStore
            )
        {

            return phalP40CmdPriv_Sw_Init(
                ref m_DataParamsInt[0],
                (ushort)Marshal.SizeOf(typeof(DataParams_t)),
                pHal.m_pDataParams,
                pKeyStore == null ? IntPtr.Zero : pKeyStore.m_pDataParams);
        }
        #endregion

        #region MEMORY_MAPPING

        private DataParams_t[]   m_DataParamsInt;
        private GCHandle          m_pDataParamsInt;

        public Sw()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc(this.m_DataParamsInt, GCHandleType.Pinned);
        }


        ~Sw()
        {
            // Free allocated pointer to data params
            if (this.m_pDataParamsInt.IsAllocated)
            {
                this.m_pDataParamsInt.Free();
            }
        }

        // Setter & Getter for DataParams structure
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }


        // Override m_pDataParams from abstract base class
        public override IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject();
            }
        }

        #endregion

    }

    #endregion

}

