/*
 * Copyright 2017 - 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfdfLight
{
    #region Enumerations
    #region Error Codes
    #region Generic Error Codes
    /// <summary>
    /// Custom error codes equivalent to C library error codes.
    /// </summary>
    public enum Error : byte
    {
        /// <summary> MF DF Light Response - No changes done to backup files. </summary>
        PHAL_MFDFLIGHT_ERR_FORMAT = 0x80,

        /// <summary> MF DF Light Response - Insufficient NV-Memory. </summary>
        PHAL_MFDFLIGHT_ERR_OUT_OF_EEPROM_ERROR,

        /// <summary> MF DF Light Invalid key number specified. </summary>
        PHAL_MFDFLIGHT_ERR_NO_SUCH_KEY,

        /// <summary> MF DF Light Current configuration/status does not allow the requested command. </summary>
        PHAL_MFDFLIGHT_ERR_PERMISSION_DENIED,

        /// <summary> MF DF Light Requested AID not found on PICC. </summary>
        PHAL_MFDFLIGHT_ERR_APPLICATION_NOT_FOUND,

        /// <summary> MF DF Light Attempt to read/write data from/to beyond the files/record's limits. </summary>
        PHAL_MFDFLIGHT_ERR_BOUNDARY_ERROR,

        /// <summary> MF DF Light Previous cmd not fully completed. Not all frames were requested or provided by the PCD. </summary>
        PHAL_MFDFLIGHT_ERR_COMMAND_ABORTED,

        /// <summary> MF DF Light Num. of applns limited to 28. No additional applications possible. </summary>
        PHAL_MFDFLIGHT_ERR_COUNT,

        /// <summary> MF DF Light File/Application with same number already exists. </summary>
        PHAL_MFDFLIGHT_ERR_DUPLICATE,

        /// <summary> MF DF Light Specified file number does not exist. </summary>
        PHAL_MFDFLIGHT_ERR_FILE_NOT_FOUND,

        /// <summary> MF DF Light Crypto error returned by PICC. </summary>
        PHAL_MFDFLIGHT_ERR_PICC_CRYPTO,

        /// <summary> MF DF Light Parameter value error returned by PICC. </summary>
        PHAL_MFDFLIGHT_ERR_PARAMETER_ERROR,

        /// <summary> MF DF Light DesFire Generic error. Check additional Info. </summary>
        PHAL_MFDFLIGHT_ERR_DF_GEN_ERROR,

        /// <summary> MF DF Light ISO 7816 Generic error. Check Additional Info. </summary>
        PHAL_MFDFLIGHT_ERR_DF_7816_GEN_ERROR,

        /// <summary> MF DF LIGHT Invalid Command Error. </summary>
        PHAL_MFDFLIGHT_ERR_CMD_INVALID
    };
    #endregion Generic Error Codes

    #region ISO7816 Error Codes
    /// <summary>
    /// ISO7816 error codes.
    /// </summary>
    public enum ERROR_ISO7816 : uint
    {
        /// <summary> Correct execution. </summary>
        PHAL_MFDFLIGHT_ISO7816_SUCCESS = 0x9000U,

        /// <summary> Wrong length. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_WRONG_LENGTH = 0x6700U,

        /// <summary> Application / file not found. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_INVALID_APPLN = 0x6A82U,

        /// <summary> Wrong parameters P1 and/or P2. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_WRONG_PARAMS = 0x6A86U,

        /// <summary> Lc inconsistent with P1/p2. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_WRONG_LC = 0x6A87U,

        /// <summary> Wrong Le. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_WRONG_LE = 0x6C00U,

        /// <summary> No precise diagnostics. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_NO_PRECISE_DIAGNOSTICS = 0x6F00U,

        /// <summary> End of File reached. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_EOF_REACHED = 0x6282U,

        /// <summary> Limited Functionality. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_LIMITED_FUNCTIONALITY_INS = 0x6283,

        /// <summary> File access not allowed. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_FILE_ACCESS = 0x6982U,

        /// <summary> File empty or access conditions not satisfied. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_FILE_EMPTY = 0x6985U,

        /// <summary> File not found. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_FILE_NOT_FOUND = 0x6A82U,

        /// <summary> Memory failure (unsuccessful update). </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_MEMORY_FAILURE = 0x6581U,

        /// <summary> Wrong parameter p1 or p2. READ RECORDS. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_INCORRECT_PARAMS = 0x6B00U,

        /// <summary> Wrong Class byte. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_WRONG_CLA = 0x6E00U,

        /// <summary> Instruction not supported. </summary>
        PHAL_MFDFLIGHT_ISO7816_ERR_UNSUPPORTED_INS = 0x6D00U
    };
    #endregion ISO7816 Error Codes
    #endregion Error Codes

    #region Auth Types
    /// <summary>
    /// Type of EV2 Authentication to perform.
    /// </summary>
    public enum AuthType : byte
    {
        /// <summary> MIFARE DESFire Light authentication type as EV2 NonFirst. </summary>
        EV2_NON_FIRST_AUTH,

        /// <summary> MIFARE DESFire Light authentication type as EV2 First. </summary>
        EV2_FIRST_AUTH,

        /// <summary> MIFARE DESFire Light authentication type as LRP NonFirst. </summary>
        LRP_NON_FIRST_AUTH,

        /// <summary> MIFARE DESFire Light authentication type as LRP First. </summary>
        LRP_FIRST_AUTH
    }
    #endregion Auth Types

    #region Diversification Options
    /// <summary>
    /// Diversification method to be used for key diversification.
    /// </summary>
    public enum DivOption : ushort
    {
        /// <summary> MIFARE DESFire Light No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary> MIFARE DESFire Light Encryption based method of diversification. </summary>
        AUTH_DIV_METHOD_ENCR = 0x0000,

        /// <summary> MIFARE DESFire Light CMAC based method of diversification. </summary>
        AUTH_DIV_METHOD_CMAC = 0x0001,

        /// <summary> Bit 1. Indicating diversification of new key requred. </summary>
        CHANGE_KEY_DIV_NEW_KEY = 0x0002,

        /// <summary> Bit 2 indicating old key was diversified. </summary>
        CHANGE_KEY_DIV_OLD_KEY = 0x0004,

        /// <summary> Bit 3 indicating new key diversification using one rnd. </summary>
        CHANGE_KEY_DIV_NEW_KEY_ONERND = 0x0008,

        /// <summary> Bit 4 indicating old key diversification using one rnd.. </summary>
        CHANGE_KEY_DIV_OLD_KEY_ONERND = 0x0010,

        /// <summary> Bit 5 indicating key diversification method based on CMAC. </summary>
        CHANGE_KEY_DIV_METHOD_CMAC = 0x0020,
    }
    #endregion Auth Types

    #region SetConfiguration Options
    /// <summary>
    /// Options for SetConfiguration command
    /// </summary>
    public enum SetConfig : byte
    {
        /// <summary> Option 0 for PICC Configuration. </summary>
        OPTION_0 = 0x00,

        /// <summary> Option 1 Reserved. </summary>
        OPTION_1 = 0x01,

        /// <summary> Option 2 for ATS Update. </summary>
        OPTION_2 = 0x02,

        /// <summary> Option 3 for SAK Update. </summary>
        OPTION_3 = 0x03,

        /// <summary> Option 4 for Secure Messaging Configuration. </summary>
        OPTION_4 = 0x04,

        /// <summary> Option 5 for Capability Data. </summary>
        OPTION_5 = 0x05,

        /// <summary> Option 6 for Application Renaming. </summary>
        OPTION_6 = 0x06,

        /// <summary> Option 8 for File Renaming. </summary>
        OPTION_8 = 0x08,

        /// <summary> Option 9 for Value file type configuration. </summary>
        OPTION_9 = 0x09,

        /// <summary> Option 10 for Failed Authentication Counter Configuration. </summary>
        OPTION_10 = 0x0A,

        /// <summary> Option 11 for Hardware Configuration. </summary>
        OPTION_11 = 0x0B,
    }
    #endregion SetConfiguration Options

    #region Communication Options
    /// <summary>
    /// The communication mode to be used.
    /// </summary>
    public enum ComOption : byte
    {
        /// <summary> Plain mode of communication.</summary>
        PLAIN = 0x00,

        /// <summary> Plain mode of communication. </summary>
        PLAIN_1 = 0x20,

        /// <summary> MAC mode of communication. </summary>
        MAC = 0x10,

        /// <summary> Enciphered mode of communication. </summary>
        FULL = 0x30,
    }
    #endregion Communication Options

    #region ChangeFileSettings Options
    /// <summary>
    /// The File option and other flags for ChangeFileSettings command.
    /// </summary>
    public enum ChangeFileSettingsOption : byte
    {
        /// <summary> Plain mode of communication. </summary>
        PLAIN = 0x00,

        /// <summary> Plain mode of communication. </summary>
        PLAIN_1 = 0x02,

        /// <summary> MAC mode of communication. </summary>
        MAC = 0x01,

        /// <summary> Enciphered mode of communication. </summary>
        FULL = 0x03,

        /// <summary> 5th Bit of FileOption indicating TMC limit config. </summary>
        TMC_LIMIT_ENABLED = 0x20,

        /// <summary> 4th Bit of FileOption indicating Excude Unauthentication config. </summary>
        UNAUTHENTICATED_TMI_EXCLUDE = 0x10,
    }
    #endregion ChangeFileSettings Options

    #region CommitTransaction Options
    /// <summary>
    /// Options for CommitTransaction command.
    /// </summary>
    public enum TmcTmv : byte
    {
        /// <summary> Option byte is not exchanged to the PICC. </summary>
        TMC_TMV_NOT_RETURNED = 0x00,

        /// <summary> Option byte exchanged to PICC and represent return of TMC and TMV. </summary>
        TMC_TMV_RETURNED = 0x01,

        /// <summary> Option byte exchanged to PICC. This is to support the exchange of Option byte to PICC. </summary>
        INCLUDE_OPTION = 0x80,
    }
    #endregion CommitTransaction Options

    #region ISOSelect Options
    /// <summary>
    /// Options for ISOSelectFile command.
    /// </summary>
    public enum Selector : byte
    {
        /// <summary> Option to indicate Selection by 2 byte file Id. </summary>
        SELECT_MF_DF_EF_FILE_IDENTIFIER = 0x00,

        /// <summary> Option to indicate Selection by child DF. </summary>
        SELECT_CHILD_DF = 0x01,

        /// <summary> Option to indicate Select EF under current DF. Fid = EF id. </summary>
        SELECT_EF_CURRENT_DF = 0x02,

        /// <summary> Option to indicate Select parent DF of the current DF. </summary>
        SELECT_PARENT_DF_CURRENT_DF = 0x03,

        /// <summary> Option to indicate Selection by DF Name. DFName and len is then valid. </summary>
        SELECT_BY_DF_NAME = 0x04
    }

    /// <summary>
    /// Options for ISOSelectFile command.
    /// </summary>
    public enum FCI : byte
    {
        /// <summary> Option to indicate the return of FCI. </summary>
        RETURNED = 0x00,

        /// <summary> Option to indicate the no return of FCI. </summary>
        NOT_RETURNED = 0x0C
    }
    #endregion ISO Select Options

    #region Configuration
    /// <summary>
    /// The configuration to be used for SetConfig / GetConfig.
    /// </summary>
    public enum CONFIG : uint
    {
        /// <summary> Option for getconfig to get additional info of a generic error. </summary>
        PHAL_MFDFLIGHT_ADDITIONAL_INFO = 0x00A1,

        /// <summary> Option to get / set current status of command wrapping in ISO 7816-4 APDUs. </summary>
        PHAL_MFDFLIGHT_WRAPPED_MODE = 0x00A2,

        /// <summary> Option to get / set Short Length APDU wrapping in ISO 7816-4 APDUs. </summary>
        PHAL_MFDFLIGHT_SHORT_LENGTH_APDU = 0x00A3U,
    }
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    /// </summary>
    public abstract class Generic
    {
        #region Constants
        public const byte READ_BLOCK_LENGTH = 16;
        public const byte WRITE_BLOCK_LENGTH = 4;
        public const byte COMPWRITE_BLOCK_LENGTH = 16;
        public const byte SIG_LENGTH = 56;
        public const byte DES_BLOCK_SIZE = 8;
        public const byte DES_KEY_LENGTH = 16;
        #endregion

        #region DLL Imports
        #region Secure Messaging
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_AuthenticateEv2 ( IntPtr pDataParams, byte bFirstAuth, ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard,
            byte[] pDivInput, byte bDivLen, byte bLenPcdCapsIn, byte[] bPcdCapsIn, byte[] bPcdCapsOut, byte[] bPdCapsOut );
        #endregion Secure Messaging

        #region Memory and Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_SetConfiguration ( IntPtr pDataParams, byte bOption, byte[] pData, byte bDataSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetVersion ( IntPtr pDataParams, byte[] pVerInfo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetCardUID ( IntPtr pDataParams, byte[] pUid );
        #endregion Memory and Configuration

        #region Key Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_ChangeKey ( IntPtr pDataParams, ushort wOption, ushort wKeyNo, ushort wKeyVer, ushort wNewKeyNo, ushort wNewKeyVer,
            byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetKeyVersion ( IntPtr pDataParams, byte bKeyNo, byte bKeySetNo, byte[] pResponse, ref byte bRxLen );
        #endregion Key Management

        #region File Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_CreateTransactionMacFile ( IntPtr pDataParams, byte bFileNo, byte bCommMode, byte[] pAccessRights, byte bKeyType,
            byte[] bTMKey, ushort wKeyNo, byte bTMKeyVer, byte[] pDivInput, byte bDivInputLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_DeleteFile ( IntPtr pDataParams, byte bFileNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetFileIDs ( IntPtr pDataParams, byte[] pFid, ref byte bNumFIDs );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetISOFileIDs ( IntPtr pDataParams, byte[] pFidBuffer, ref byte bNumFIDs );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetFileSettings ( IntPtr pDataParams, byte bFileNo, byte[] pFSBuffer, ref byte bBufferLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_ChangeFileSettings ( IntPtr pDataParams, byte bCommMode, byte bFileNo, byte bFileOption, byte[] pAccessRights,
            byte bAdditionalInfoLen, byte[] bAdditionalInfo );
        #endregion File Management

        #region Data Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_ReadData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pLength,
            ref IntPtr ppRxdata, ref ushort pRxdataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_WriteData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pTxData,
            byte[] pTxDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetValue ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_Credit ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_Debit ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_LimitedCredit ( IntPtr pDataParams, byte bCommOption, byte bFileNo, byte[] pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_ReadRecords ( IntPtr pDataParams, byte bCommOption, byte bIns, byte bFileId, byte[] pRecNo, byte[] pRecCount,
            byte[] pRecSize, ref IntPtr ppRxdata, ref ushort pRxdataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_WriteRecord ( IntPtr pDataParams, byte bCommOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pData,
            byte[] pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_UpdateRecord ( IntPtr pDataParams, byte bCommOption, byte bIns, byte bFileNo, byte[] pRecNo, byte[] pOffset,
            byte[] pData, byte[] pDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_ClearRecordFile ( IntPtr pDataParams, byte bFileNo );
        #endregion data Management

        #region Transaction Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_CommitTransaction ( IntPtr pDataParams, byte bOption, byte[] bTMC, byte[] bTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_AbortTransaction ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_CommitReaderID ( IntPtr pDataParams, byte[] bTMRI, byte[] bEncTMRI );
        #endregion Transaction Management

        #region ISO7816
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_IsoSelectFile ( IntPtr pDataParams, byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, byte bDFnameLen,
            byte bExtendedLenApdu, ref IntPtr ppFCI, ref ushort pwFCILen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_IsoReadBinary ( IntPtr pDataParams, ushort wOption, byte bOffset, byte bSfid, UInt32 dwBytesToRead, byte bExtendedLenApdu,
            ref IntPtr pRxBuffer, ref UInt32 pBytesRead );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_IsoUpdateBinary ( IntPtr pDataParams, byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] pData, UInt32 dwDataLen );
        #endregion ISO7816

        #region Originality Check
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_ReadSign ( IntPtr pDataParams, byte bAddr, ref IntPtr pSignature );
        #endregion Originality Check

        #region Miscellaneous
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_ResetAuthentication ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_CalculateTMV ( IntPtr pDataParams, ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pTMI,
            uint dwTMILen, byte[] pTMV );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_DecryptReaderID ( IntPtr pDataParams, ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pEncTMRI,
            byte[] pTMRIPrev );
        #endregion Miscellaneous
        #endregion

        #region Wrapper Functions
        #region Secure Messaging
        /// <summary>
        /// Performs an Light First or Light Non First Authentication depending upon bFirstAuth Parameter. This will be using the AES128 keys and will
        /// generate and verify the contents based on generic AES algorithm.
        /// </summary>
        ///
        /// <param name="bFirstAuth">One of the below options.
        ///								NxpRdLibNet.alMfdfLight.AuthType.EV2_NON_FIRST_AUTH
        ///								NxpRdLibNet.alMfdfLight.AuthType.EV2__FIRST_AUTH
        ///								NxpRdLibNet.alMfdfLight.AuthType.LRP_NON_FIRST_AUTH
        ///								NxpRdLibNet.alMfdfLight.AuthType.LRP_NON_FIRST_AUTH</param>
        /// <param name="wOption">Diversification option can be one of
        ///								NxpRdLibNet.alMfdfLight.DivOption.NO_DIVERSIFICATION
        ///								NxpRdLibNet.alMfdfLight.DivOption.AUTH_DIV_METHOD_ENCR
        ///								NxpRdLibNet.alMfdfLight.DivOption.AUTH_DIV_METHOD_CMAC</param>
        /// <param name="wKeyNo">Key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Key version in keystore of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card.</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        /// <param name="bPcdCapsIn">PCD Capabilities. Upto 6 bytes.</param>
        /// <param name="bPcdCapsOut">PCD Capabilities. 6 bytes.</param>
        /// <param name="bPdCapsOut">PD Capabilities. 6 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t AuthenticateEv2 ( byte bFirstAuth, ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput, byte[] bPcdCapsIn,
            out byte[] bPcdCapsOut, out byte[] bPdCapsOut )
        {
            bPcdCapsOut = new byte[6];
            bPdCapsOut = new byte[6];

            return phalMfdfLight_AuthenticateEv2 ( m_pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, pDivInput, ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ),
                ( byte ) ( ( bPcdCapsIn == null ) ? 0 : bPcdCapsIn.Length ), bPcdCapsIn, bPcdCapsOut, bPdCapsOut );
        }
        #endregion Secure Messaging

        #region Memory and Configuration
        /// <summary>
        /// Configures the card and pre personalizes the card with a key, defines if the UID or the random ID is sent back during
        /// communication setup and configures the ATS string.
        /// </summary>
        ///
        /// <param name="bOption">Configuration Option. Define length and content of the Data parameter.
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_0
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_1
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_2
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_3
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_4
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_5
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_6
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_8
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_9
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_10
        ///							NxpRdLibNet.alMfdfLight.SetConfig.OPTION_11</param>
        /// <param name="pData">Data for the option specified</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfiguration ( byte bOption, byte[] pData )
        {
            if ( pData != null )
            {
                byte[] pDataTmp = new byte[pData.Length];

                pData.CopyTo ( pDataTmp, 0 );
                return phalMfdfLight_SetConfiguration ( m_pDataParams, bOption, pDataTmp, ( byte ) pDataTmp.Length );
            }
            else
            {
                return phalMfdfLight_SetConfiguration ( m_pDataParams, bOption, null, 0 );
            }
        }

        /// <summary>
        /// Returns manufacturing related data of the PICC
        /// </summary>
        ///
        /// <param name="pVerInfo">The version information of the product.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetVersion ( out byte[] pVerInfo )
        {
            Status_t oStatus, oStatusTemp;
            byte[] aVersion = new byte[33];
            ushort wLength = 0, wVal = 0;

            pVerInfo = null;

            /*Do a Get Config of  ADDITIONAL_INFO to have the backup of current value*/
            oStatusTemp = GetConfig ( 0x00A1, ref wVal );

            oStatus = phalMfdfLight_GetVersion ( m_pDataParams, aVersion );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                /* Do a Get Config of  ADDITIONAL_INFO to read the length(wLength) of the Version string */
                oStatusTemp = GetConfig ( 0x00A1, ref wLength );
                pVerInfo = new byte[wLength];
                /* Copy the version string only to the length retreived */
                Array.Copy ( aVersion, pVerInfo, wLength );
            }

            /*Do a Set Config of  ADDITIONAL_INFO to retain the backup value*/
            oStatusTemp = SetConfig ( 0x00A1, wVal );
            return oStatus;
        }

        /// <summary>
        /// Returns the Unique ID of the PICC
        /// </summary>
        ///
        /// <param name="pUid">The complete UID of the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetCardUID ( out byte[] pUid )
        {
            pUid = new byte[10];
            ushort bLength = 0;

            Status_t oStatus = phalMfdfLight_GetCardUID ( m_pDataParams, pUid );

            /* Get the length of the UID */
            if ( oStatus == new Status_t () )
            {
                oStatus = GetConfig ( 0x00A1, ref bLength );
            }

            Array.Resize<byte> ( ref pUid, bLength );
            return oStatus;
        }
        #endregion Memory and Configuration

        #region Key Management
        /// <summary>
        /// Changes any key on the PICC
        ///
        ///		<remarks>
        ///			The key on the PICC is changed to the new key.
        ///			The key type of the application keys cannot be changed. The key type of only the PICC master key can be changed.
        ///			The keys changeable are PICCDAMAuthKey, PICCDAMMACKey, PICCDAMEncKey, VCConfigurationKey, SelectVCKey, VCProximityKey,
        ///			VCPollingEncKey, VCPollingMACKey.
        ///		</remarks>
        /// </summary>
        ///
        /// <param name="wOption">Diversification option can be one of
        ///								NxpRdLibNet.alMfdfLight.DivOption.NO_DIVERSIFICATION
        ///								NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_METHOD_CMAC
        ///								NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_OLD_KEY | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_METHOD_CMAC
        ///								NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_NEW_KEY_ONERND
        ///								NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_OLD_KEY | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_OLD_KEY_ONERND
        ///								NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_OLD_KEY
        ///								NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_OLD_KEY |
        ///									NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_METHOD_CMAC
        ///								NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_NEW_KEY | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_OLD_KEY |
        ///									NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_NEW_KEY_ONERND | NxpRdLibNet.alMfdfLight.DivOption.CHANGE_KEY_DIV_OLD_KEY_ONERND</param>
        /// <param name="wKeyNo">Old key number in keystore of software or SAM.</param>
        /// <param name="wKeyVer">Old key version in keystore of software or SAM.</param>
        /// <param name="wNewKeyNo">New key number in keystore of software or SAM.</param>
        /// <param name="wNewKeyVer">New key version in keystore of software or SAM.</param>
        /// <param name="bKeyNoCard">Key number on card</param>
        /// <param name="pDivInput">Diversification input. Can be NULL.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeKey ( ushort wOption, ushort wKeyNo, ushort wKeyVer, ushort wNewKeyNo, ushort wNewKeyVer, byte bKeyNoCard, byte[] pDivInput )
        {
            return phalMfdfLight_ChangeKey ( m_pDataParams, wOption, wKeyNo, wKeyVer, wNewKeyNo, wNewKeyVer, bKeyNoCard, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ) );
        }

        /// <summary>
        /// Reads out the current key version of any key stored on the PICC
        /// </summary>
        /// <param name="bKeyNo">Key number on card.</param>
        /// <param name="bKeySetNo">1 byte Key set number. Optional as it is passed only when bit[6] of bKeyNo is set.</param>
        /// <param name="pResponse">The version of the specified key.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetKeyVersion ( byte bKeyNo, byte bKeySetNo, out byte[] pResponse )
        {
            pResponse = new byte[16];
            byte bRxLen = 0;

            Status_t oStatus = phalMfdfLight_GetKeyVersion ( m_pDataParams, bKeyNo, bKeySetNo, pResponse, ref bRxLen );

            Array.Resize<byte> ( ref pResponse, bRxLen );

            return oStatus;
        }
        #endregion Key Management

        #region File Management
        /// <summary>
        /// Creates a Transaction MAC file. An application can have only one Transaction MAC File.
        /// This command is for Software and SAM layer.
        ///
        /// </summary>
        ///
        /// <param name="bFileNo">File number of the file to be created.</param>
        /// <param name="bCommMode">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="pAccessRights">The access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bTMKeyOption">Bit0-1 indicates key type and it should be always 0x02.</param>
        /// <param name="bTMKey">The 16 byte key value to be used as TMKey.</param>
        /// <param name="wKeyNo">The reference key number in SAM keystore. Only valid if initialized for SAM.</param>
        /// <param name="bKeyVer">The version of the TMKey.</param>
        /// <param name="aDivInput">Diversification input to diversify the TMKey.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CreateTransactionMacFile ( byte bFileNo, byte bCommMode, byte[] pAccessRights, byte bTMKeyOption, byte[] bTMKey, ushort wKeyNo,
            byte bKeyVer, byte[] aDivInput )
        {

            byte[] bTMKeyTmp = new byte[bTMKey.Length];
            byte[] bAccessRightsTmp = new byte[pAccessRights.Length];

            bTMKey.CopyTo ( bTMKeyTmp, 0 );
            pAccessRights.CopyTo ( bAccessRightsTmp, 0 );

            return phalMfdfLight_CreateTransactionMacFile ( m_pDataParams, bFileNo, bCommMode, bAccessRightsTmp, bTMKeyOption, bTMKeyTmp, wKeyNo, bKeyVer,
                aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Creates a Transaction MAC file. An application can have only one Transaction MAC File. This command is for Software layer usage only.
        /// </summary>
        ///
        /// <param name="bFileNo">File number of the file to be created.</param>
        /// <param name="bCommMode">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="pAccessRights">The access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bTMKeyOption">Bit0-1 indicates key type and it should be always 0x02.</param>
        /// <param name="bTMKey">The 16 byte key value to be used as TMKey.</param>
        /// <param name="bTMKeyVer">The version of the TMKey.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CreateTransactionMacFile ( byte bFileNo, byte bCommMode, byte[] pAccessRights, byte bTMKeyOption, byte[] bTMKey, byte bTMKeyVer )
        {
            return CreateTransactionMacFile ( bFileNo, bCommMode, pAccessRights, bTMKeyOption, bTMKey, 0, bTMKeyVer, null );
        }

        /// <summary>
        /// Permanently deactivates a file within the file directory of the currently selected application.
        /// </summary>
        ///
        /// <param name="bFileNo">File number of the file to be deleted.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DeleteFile ( byte bFileNo )
        {
            return phalMfdfLight_DeleteFile ( m_pDataParams, bFileNo );
        }

        /// <summary>
        /// Returns the file IDs of all active files within the currently selected application.
        /// </summary>
        ///
        /// <param name="pFid">The buffer containing the available file ID(s). Buffer should be 32 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetFileIDs ( out byte[] pFid )
        {
            byte bNumFIDs = 0;
            pFid = new byte[32];

            Status_t oStatus = phalMfdfLight_GetFileIDs ( m_pDataParams, pFid, ref bNumFIDs );
            Array.Resize<byte> ( ref pFid, bNumFIDs );

            return oStatus;
        }

        /// <summary>
        /// Get the ISO File IDs.
        /// </summary>
        ///
        /// <param name="pFidBuffer">The buffer containing the available ISO file ID(s). Buffer should be 64 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetISOFileIDs ( out byte[] pFidBuffer )
        {
            byte bNumFIDs = 0;

            pFidBuffer = new byte[64];

            Status_t oStatus = phalMfdfLight_GetISOFileIDs ( m_pDataParams, pFidBuffer, ref bNumFIDs );

            Array.Resize<byte> ( ref pFidBuffer, bNumFIDs * 2 );

            return oStatus;
        }

        /// <summary>
        /// Get informtion on the properties of a specific file
        /// </summary>
        ///
        /// <param name="bFileNo">The file number for which the setting to be retrieved.</param>
        /// <param name="pFSBuffer">The buffer containing the settings. The buffer should be 17 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetFileSettings ( byte bFileNo, out byte[] pFSBuffer )
        {
            byte bBufferLen = 0;

            pFSBuffer = new byte[35];

            Status_t oStatus = phalMfdfLight_GetFileSettings ( m_pDataParams, bFileNo, pFSBuffer, ref bBufferLen );
            Array.Resize<byte> ( ref pFSBuffer, bBufferLen );

            return oStatus;
        }

        /// <summary>
        /// Changes the access parameters of an existing file
        /// </summary>
        ///
        /// <param name="bCommMode">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bFileNo">File number for which the setting need to be updated.</param>
        /// <param name="bFileOption">New communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.FULL
        ///
        ///							  Ored with one of the below flags if required.
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.TMC_LIMIT_ENABLED
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.UNAUTHENTICATED_TMI_EXCLUDE</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bTmcLimitLen">Length of bytes available in pTmcLimit buffer. </param>
        /// <param name="pTmcLimit">One of the below inforamtion.
        ///								If Standard AES: 4 byte TMC Limit value.
        ///								If LRP         : 2 byte TMC Limit value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeFileSettings ( byte bCommMode, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte bTmcLimitLen, byte[] pTmcLimit )
        {
            if ( ( pAccessRights == null ) && ( pTmcLimit == null ) )
            {
                return phalMfdfLight_ChangeFileSettings ( m_pDataParams, bCommMode, bFileNo, bFileOption, null, 0, null );
            }
            else if ( pAccessRights == null )
            {
                byte[] pTmcLimitTmp = new byte[pTmcLimit.Length];
                pTmcLimit.CopyTo ( pTmcLimitTmp, 0 );

                return phalMfdfLight_ChangeFileSettings ( m_pDataParams, bCommMode, bFileNo, bFileOption, null, bTmcLimitLen, pTmcLimitTmp );
            }
            else if ( ( pTmcLimit == null ) )
            {
                byte[] pAccessRightsTmp = new byte[pAccessRights.Length];
                pAccessRights.CopyTo ( pAccessRightsTmp, 0 );

                return phalMfdfLight_ChangeFileSettings ( m_pDataParams, bCommMode, bFileNo, bFileOption, pAccessRightsTmp, bTmcLimitLen, null );
            }
            else
            {
                byte[] pAccessRightsTmp = new byte[pAccessRights.Length];
                byte[] pTmcLimitTmp = new byte[pTmcLimit.Length];
                pAccessRights.CopyTo ( pAccessRightsTmp, 0 );
                pTmcLimit.CopyTo ( pTmcLimitTmp, 0 );

                return phalMfdfLight_ChangeFileSettings ( m_pDataParams, bCommMode, bFileNo, bFileOption, pAccessRightsTmp, bTmcLimitLen, pTmcLimit );
            }
        }

        /// <summary>
        /// Changes the access parameters of an existing file.
        /// </summary>
        ///
        /// <param name="bCommMode">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bMfm">If Set to 0x01, Indicates that API should be called to extract MFDFL specific features. Valid for Software mode only.</param>
        /// <param name="bFileNo">File number for which the setting need to be updated.</param>
        /// <param name="bFileOption">New communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.FULL
        ///
        ///							  Ored with one of the below flags if required.
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.TMC_LIMIT_ENABLED
        ///								NxpRdLibNet.alMfdfLight.ChangeFileSettingsOption.UNAUTHENTICATED_TMI_EXCLUDE</param>
        /// <param name="pAccessRights">The new access right to be applied for the file. Should be 2 byte.</param>
        /// <param name="bTmcLimitLen">Length of bytes available in pTmcLimit buffer. </param>
        /// <param name="pTmcLimit">One of the below inforamtion.
        ///								If Standard AES: 4 byte TMC Limit value.
        ///								If LRP         : 2 byte TMC Limit value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ChangeFileSettings ( byte bCommMode, byte bMfm, byte bFileNo, byte bFileOption, byte[] pAccessRights, byte bTmcLimitLen, byte[] pTmcLimit )
        {
            /* bMfm is ORed with bCommMode to ensure that the C interface handles it either in Desfire Ev2 way or in MFMatch way
			 * The bit-0 of bCommMode is cleared in C Interface.
			 */
            if ( bMfm == 1 )
            {
                bCommMode |= 0x01;
            }

            return ChangeFileSettings ( bCommMode, bFileNo, bFileOption, pAccessRights, bTmcLimitLen, pTmcLimit );
        }
        #endregion File Management

        #region Data Management
        /// <summary>
        /// Reads data from standard data files or backup data files
        ///
        ///		<remarks>
        ///			Chaining upto the size of the HAL Rx buffer is handled within this function. If more data is to be read, the user has to call
        ///			this function again with bOption = #PH_EXCHANGE_RXCHAINING | [one of the communication options]
        ///		</remarks>
        /// </summary>
        ///
        /// <param name="bOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///							NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///							NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///							NxpRdLibNet.alMfdfLight.ComOption.FULL
        ///
        ///						  NxpRdLibNet.ExchangeOptions.RXCHAINING: To be Or'd with the above option
        ///																  flag if Chaining status is returned. </param>
        /// <param name="bIns">Uses ISO 14443-4 chaining instead of DESFire application chaining. This should be always Set.</param>
        /// <param name="bFileNo">The file number from where the data to be read.</param>
        /// <param name="pOffset">The offset from where the data should be read. Will be of 3 bytes with LSB first.
        ///							If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pLength">The number of bytes to be read. Will be of 3 bytes with LSB first.
        ///							If 0x10 bytes need to be read then it will be 10 00 00.
        ///							If complete file need to be read then it will be 00 00 00.</param>
        /// <param name="pRxdata">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ReadData ( byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pLength, out byte[] pRxdata )
        {
            IntPtr ppRxdata = IntPtr.Zero;
            byte[] pRxdataLen;
            ushort rxDataLen = 0;

            pRxdataLen = new byte[3];

            Status_t oStatus = phalMfdfLight_ReadData ( m_pDataParams, bOption, bIns, bFileNo, pOffset, pLength, ref ppRxdata, ref rxDataLen );

            if ( rxDataLen != 0 )
            {
                pRxdata = new byte[rxDataLen];
                Marshal.Copy ( ppRxdata, pRxdata, 0, rxDataLen );
            }
            else
            {
                pRxdata = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Writes data to standard data files or backup data files
        /// </summary>
        ///
        /// <param name="bOption">Communication settings for the file.
        ///							NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///							NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///							NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///							NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bIns">Uses ISO 14443-4 chaining instead of DESFire application chaining. This should be always Set.</param>
        /// <param name="bFileNo">The file number from where the data to be read.</param>
        /// <param name="pOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///							If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pTxData">The data to be written to the PICC.</param>
        /// <param name="pTxDataLen">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///								If 0x10 bytes need to be written then it will be 10 00 00.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t WriteData ( byte bOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pTxData, byte[] pTxDataLen )
        {
            byte[] pTxDataTmp = null;
            byte[] pOffsetTmp = null;
            byte[] pTxDataLenTmp = null;

            if ( pTxData != null )
            {
                pTxDataTmp = new byte[pTxData.Length];
                pTxData.CopyTo ( pTxDataTmp, 0 );
            }

            if ( pOffset != null )
            {
                pOffsetTmp = new byte[pOffset.Length];
                pOffset.CopyTo ( pOffsetTmp, 0 );
            }

            if ( pTxDataLen != null )
            {
                pTxDataLenTmp = new byte[pTxDataLen.Length];
                pTxDataLen.CopyTo ( pTxDataLenTmp, 0 );
            }
            else
            {
                pTxDataLenTmp = null;
            }

            return phalMfdfLight_WriteData ( m_pDataParams, bOption, bIns, bFileNo, pOffsetTmp, pTxDataTmp, pTxDataLenTmp );
        }

        /// <summary>
        /// Reads the currently stored value from value files.
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bFileNo">The file number for which the value to the retrieved.</param>
        /// <param name="pValue">The value returned by the PICC. The buffer should be 4 bytes.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetValue ( byte bCommOption, byte bFileNo, out byte[] pValue )
        {
            pValue = new byte[4];

            return phalMfdfLight_GetValue ( m_pDataParams, bCommOption, bFileNo, pValue );
        }

        /// <summary>
        /// Increases a value stored in a Value File
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bFileNo">The file number to which the value should be credited.</param>
        /// <param name="pValue">The value to be credited. Will be of 4 bytes with LSB first.
        ///							If 0x10 bytes need to be credited then it will be 10 00 00 00.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Credit ( byte bCommOption, byte bFileNo, byte[] pValue )
        {
            if ( pValue == null )
            {
                return phalMfdfLight_Credit ( m_pDataParams, bCommOption, bFileNo, null );
            }
            else
            {
                byte[] pValueTmp = new byte[pValue.Length];
                pValue.CopyTo ( pValueTmp, 0 );

                return phalMfdfLight_Credit ( m_pDataParams, bCommOption, bFileNo, pValueTmp );
            }
        }

        /// <summary>
        /// Decreases a value stored in a Value File
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bFileNo">The file number to which the value should be debited.</param>
        /// <param name="pValue">The value to be debited. Will be of 4 bytes with LSB first.
        ///							If 0x10 bytes need to be debited then it will be 10 00 00 00.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Debit ( byte bCommOption, byte bFileNo, byte[] pValue )
        {
            if ( pValue == null )
            {
                return phalMfdfLight_Debit ( m_pDataParams, bCommOption, bFileNo, null );
            }
            else
            {
                byte[] pValueTmp = new byte[pValue.Length];
                pValue.CopyTo ( pValueTmp, 0 );

                return phalMfdfLight_Debit ( m_pDataParams, bCommOption, bFileNo, pValueTmp );
            }
        }

        /// <summary>
        /// Allows a limited increase of a value stored in a Value File without having full credit permissions to the file.
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bFileNo">The file number to which the value should be credited.</param>
        /// <param name="pValue">The value to be credited. Will be of 4 bytes with LSB first.
        ///							If 0x10 bytes need to be credited then it will be 10 00 00 00.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t LimitedCredit ( byte bCommOption, byte bFileNo, byte[] pValue )
        {
            if ( pValue == null )
            {
                return phalMfdfLight_LimitedCredit ( m_pDataParams, bCommOption, bFileNo, null );
            }
            else
            {
                byte[] pValueTmp = new byte[pValue.Length];
                pValue.CopyTo ( pValueTmp, 0 );

                return phalMfdfLight_LimitedCredit ( m_pDataParams, bCommOption, bFileNo, pValueTmp );
            }
        }

        /// <summary>
        /// Writes data to a record in a Cyclic or Linear Record File.
        ///
        ///		<remarks>
        ///			Implements chaining to the card. The data provided on pData will be chained to the card by sending
        ///			data upto the frame size of the DESFire PICC, at a time.
        ///		</remarks>
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bIns">Uses ISO 14443-4 chaining instead of DESFire application chaining. This should be always Set.</param>
        /// <param name="bFileNo">he file number to which the data should be written.</param>
        /// <param name="pOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///							If 0x10 need to be offset then it will be 10 00 00.</param>
        /// <param name="pData">The data to be written to the PICC.</param>
        /// <param name="pDataLen">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///							If 0x10 bytes need to be written then it will be 10 00 00.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t WriteRecord ( byte bCommOption, byte bIns, byte bFileNo, byte[] pOffset, byte[] pData, byte[] pDataLen )
        {
            if ( pData == null )
            {
                return phalMfdfLight_WriteRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pOffset, null, pDataLen );
            }
            if ( pDataLen == null )
            {
                return phalMfdfLight_WriteRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pOffset, null, null );
            }
            else
            {
                byte[] pDataTmp = new byte[pData.Length];
                pData.CopyTo ( pDataTmp, 0 );

                return phalMfdfLight_WriteRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pOffset, pDataTmp, pDataLen );
            }
        }

        /// <summary>
        /// Reads out a set of complete records from a Cyclic or Linear Record File.
        ///
        ///		<remarks>
        ///			The readrecords command reads and stores data in the rxbuffer upto the rxbuffer size before returning
        ///			to the user. The rxbuffer is configured during the HAL init and this is specified by the user.
        ///
        ///			Chaining upto the size of the HAL Rx buffer is handled within this function.
        ///			If more data is to be read, the user has to call this function again with
        ///			bCommOption = PH_EXCHANGE_RXCHAINING | [one of the communication options]
        ///
        ///			If TMI collection is ON, if pRecCount is zero then pRecSize is madatory parameter.
        ///			If pRecSize and RecCount are zero and TMI collection is ON, then  PH_ERR_INVALID_PARAMETER error returned.
        ///			If TMI collection is ON; and if wrong pRecSize is provided, then wrong RecCount value will be calculated and updated for TMI collection.
        ///		</remarks>
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL
        ///
        ///						  NxpRdLibNet.ExchangeOptions.RXCHAINING: To be Or'd with the above option
        ///																  flag if Chaining status is returned. </param>
        /// <param name="bIns">Uses ISO 14443-4 chaining instead of DESFire application chaining. This should be always Set.</param>
        /// <param name="bFileId">The file number from where the data to be read.</param>
        /// <param name="pRecNo">3 bytes LSB first. Record number of the newest record targeted, starting to count
        ///						 from the latest record written.</param>
        /// <param name="pRecCount">3 bytes LSB first. Number of records to be read. If 0x00 00 00, then all the records are read.</param>
        /// <param name="pRecSize">The number of bytes to be read. Will be of 3 bytes with LSB first.</param>
        /// <param name="pRxdata">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ReadRecords ( byte bCommOption, byte bIns, byte bFileId, byte[] pRecNo, byte[] pRecCount, byte[] pRecSize, out byte[] pRxdata )
        {
            IntPtr ppRxdata = IntPtr.Zero;
            ushort pRxDataLen = 0;

            Status_t oStatus = phalMfdfLight_ReadRecords ( m_pDataParams, bCommOption, bIns, bFileId, pRecNo, pRecCount, pRecSize, ref ppRxdata, ref pRxDataLen );

            if ( pRxDataLen != 0 )
            {
                pRxdata = new byte[pRxDataLen];
                Marshal.Copy ( ppRxdata, pRxdata, 0, pRxDataLen );
            }
            else
            {
                pRxdata = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Updates data to a record in a Cyclic or Linear Record File.
        ///
        ///		<remarks>
        ///			Implements chaining to the card. The data provided on pData will be chained to the card by sending
        ///			data upto the frame size of the DESFire PICC, at a time.
        ///		</remarks>
        /// </summary>
        ///
        /// <param name="bCommOption">Communication settings for the file.
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN
        ///								NxpRdLibNet.alMfdfLight.ComOption.PLAIN_1
        ///								NxpRdLibNet.alMfdfLight.ComOption.MAC
        ///								NxpRdLibNet.alMfdfLight.ComOption.FULL</param>
        /// <param name="bIns">Uses ISO 14443-4 chaining instead of DESFire application chaining. This should be always Set.</param>
        /// <param name="bFileNo">The file number to which the data should be written.</param>
        /// <param name="pRecNo">Record number. Records are numbered starting with the latest record written.</param>
        /// <param name="pOffset">Starting position for the update operationwithin the targeted record.</param>
        /// <param name="pData">The data to be written to the PICC.</param>
        /// <param name="pDataLen">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///							If 0x10 bytes need to be written then it will be 10 00 00.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t UpdateRecord ( byte bCommOption, byte bIns, byte bFileNo, byte[] pRecNo, byte[] pOffset, byte[] pData, byte[] pDataLen )
        {
            if ( pData == null )
            {
                return phalMfdfLight_UpdateRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pRecNo, pOffset, null, pDataLen );
            }
            if ( pDataLen == null )
            {
                return phalMfdfLight_UpdateRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pRecNo, pOffset, null, null );
            }
            if ( pRecNo == null )
            {
                byte[] pDataTmp = new byte[pData.Length];
                pData.CopyTo ( pDataTmp, 0 );

                return phalMfdfLight_UpdateRecord ( m_pDataParams, bCommOption, bIns, bFileNo, null, pOffset, pDataTmp, pDataLen );
            }
            else
            {
                byte[] pDataTmp = new byte[pData.Length];
                pData.CopyTo ( pDataTmp, 0 );

                return phalMfdfLight_UpdateRecord ( m_pDataParams, bCommOption, bIns, bFileNo, pRecNo, pOffset, pDataTmp, pDataLen );
            }
        }

        /// <summary>
        /// Resets a Cyclic or Linear Record File.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be cleared.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ClearRecordFile ( byte bFileNo )
        {
            return phalMfdfLight_ClearRecordFile ( m_pDataParams, bFileNo );
        }
        #endregion data Management

        #region Transaction Management
        /// <summary>
        /// Validates all previous write access' on Backup Data files, value files and record files within one application.
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///							NxpRdLibNet.alMfdfLight.TmcTmv.TMC_TMV_NOT_RETURNED
        ///							NxpRdLibNet.alMfdfLight.TmcTmv.TMC_TMV_RETURNED
        ///
        ///							Can be ored with one of the above values.
        ///							NxpRdLibNet.alMfdfLight.TmcTmv.INCLUDE_OPTION</param>
        /// <param name="bTMC">The increased value as stored in FileType.TransactionMAC.
        ///						If Standard AES: 4 byte Transaction MAC counter value.
        ///						If LRP         : 2 byte of Actual TMC followed by 2 bytes of Session TMC</param>
        /// <param name="bTMAC">8 bytes Transaction MAC value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CommitTransaction ( byte bOption, out byte[] bTMC, out byte[] bTMAC )
        {
            bTMC = new byte[4];
            bTMAC = new byte[8];

            Status_t oStatus =  phalMfdfLight_CommitTransaction ( m_pDataParams, bOption, bTMC, bTMAC );

            /* user needs to check for null before dereferencing these variable*/
            if ( ( bOption & 0x01 ) == 0x00 )
            {
                bTMC = null;
                bTMAC = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Invalidates all previous write access' on Backup Data files, value files and record files within one application.
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t AbortTransaction ()
        {

            return phalMfdfLight_AbortTransaction ( m_pDataParams );
        }

        /// <summary>
        /// Secures the transaction by commiting the application to ReaderID specified
        /// </summary>
        ///
        /// <param name="bTMRI">16 bytes of Transaction MAC reader ID.</param>
        /// <param name="bEncTMRI">16 bytes of Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CommitReaderID ( byte[] bTMRI, out byte[] bEncTMRI )
        {
            Status_t oStatus = null, oStatusTmp;
            byte[] bTMRIenc = new byte[16];
            ushort wLength = 0, wVal = 0;

            /*Do a Get Config of  ADDITIONAL_INFO to have the backup of current value*/
            oStatusTmp = GetConfig ( 0x00A1, ref wVal );
            oStatus = phalMfdfLight_CommitReaderID ( m_pDataParams, bTMRI, bTMRIenc );

            if ( oStatus.Equals ( Error_Gen.SUCCESS ) )
            {
                /* Do a Get Config of  ADDITIONAL_INFO to read the length(wLength) of the TMRI */
                oStatusTmp = GetConfig ( 0x00A1, ref wLength );

                if ( wLength.Equals ( 0x00 ) )
                {
                    bEncTMRI = null;
                }
                else
                {
                    bEncTMRI = new byte[wLength];
                    /* Copy the version string only to the length retreived */
                    Array.Copy ( bTMRIenc, bEncTMRI, wLength );
                }
            }
            else
            {
                /* In Error case, set the enTMRI to NULL */
                bEncTMRI = null;
            }

            /*Do a Set Config of  ADDITIONAL_INFO to retain the backup value*/
            oStatusTmp = SetConfig ( 0x00A1, wVal );

            return oStatus;
        }
        #endregion Transaction Management

        #region ISO7816
        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///							NxpRdLibNet.alMfdfLight.FCI.RETURNED
        ///							NxpRdLibNet.alMfdfLight.FCI.NOT_RETURNED</param>
        /// <param name="bSelector">The selector to be used.
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_CHILD_DF
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_EF_CURRENT_DF
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_PARENT_DF_CURRENT_DF
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_BY_DF_NAME</param>
        /// <param name="pFid">The ISO File number to be selected.</param>
        /// <param name="pDFname">The ISO DFName to be selected. Valid only when bOption = 0x04.</param>
        /// <param name="pFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, out byte[] pFCI )
        {
            /*
			 * Modified API is provided immediately after this API under the commented section "ISOSELECTFILE".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API.
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc/Le is passed as short APDU */

            return IsoSelectFile ( bOption, bSelector, pFid, pDFname, bExtendedLenApdu, out pFCI );
        }

        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///							NxpRdLibNet.alMfdfLight.FCI.RETURNED
        ///							NxpRdLibNet.alMfdfLight.FCI.NOT_RETURNED</param>
        /// <param name="bSelector">The selector to be used.
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_CHILD_DF
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_EF_CURRENT_DF
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_PARENT_DF_CURRENT_DF
        ///								NxpRdLibNet.alMfdfLight.Selector.SELECT_BY_DF_NAME</param>
        /// <param name="pFid">The ISO File number to be selected.</param>
        /// <param name="pDFname">The ISO DFName to be selected. Valid only when bOption = 0x04.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU. This should be always cleared.</param>
        /// <param name="pFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] pFid, byte[] pDFname, byte bExtendedLenApdu, out byte[] pFCI )
        {
            IntPtr ppRxBuffer = IntPtr.Zero;
            ushort wBytesRead = 0;
            byte bDfNameLen = 0x00;

            if ( pDFname != null )
            {
                bDfNameLen = ( byte ) pDFname.Length;
            }

            Status_t oStatus = phalMfdfLight_IsoSelectFile ( m_pDataParams, bOption, bSelector, pFid, pDFname, bDfNameLen, bExtendedLenApdu, ref ppRxBuffer,
                ref wBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( wBytesRead != 0 ) )
            {
                pFCI = new byte[wBytesRead];
                Marshal.Copy ( ppRxBuffer, pFCI, 0, wBytesRead );
            }
            else
            {
                pFCI = null;
            }

            return oStatus;
        }

        /// <summary>
        /// ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT    : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RX_CHAINING: To exchange the chaining command to PICC in
        ///																	 case PICC returns AF as the status.</param>
        /// <param name="bOffset">The offset from where the data should be read.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied. Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bBytesToRead">Number of bytes to read. If 0, then entire file to be read.</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, byte bBytesToRead, out byte[] pRxBuffer )
        {
            /*
			 * Modified API is provided immediately after this API under the commented section "ISOREADBINARY".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Le is passed as short APDU */

            return IsoReadBinary ( wOption, bOffset, bSfid, bBytesToRead, bExtendedLenApdu, out pRxBuffer );
        }

        /// <summary>
        /// ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///							NxpRdLibNet.ExchangeOptions.DEFAULT    : To exchange the initial command to the PICC.
        ///							NxpRdLibNet.ExchangeOptions.RX_CHAINING: To exchange the chaining command to PICC in
        ///																	 case PICC returns AF as the status.</param>
        /// <param name="bOffset">The offset from where the data should be read.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied. Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bBytesToRead">Number of bytes to read. If 0, then entire file to be read.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU. This should be always cleared.</param>
        /// <param name="pRxBuffer">The data retuned by the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Returns Success_Chaining status for successfull chaining operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, UInt32 bBytesToRead, byte bExtendedLenApdu, out byte[] pRxBuffer )
        {
            IntPtr ppRxBuffer = IntPtr.Zero;
            UInt32 dwBytesRead = 0x00;

            Status_t oStatus = phalMfdfLight_IsoReadBinary ( m_pDataParams, wOption, bOffset, bSfid, bBytesToRead, bExtendedLenApdu, ref ppRxBuffer, ref dwBytesRead );

            if ( ( ppRxBuffer != IntPtr.Zero ) && ( dwBytesRead != 0 ) )
            {
                pRxBuffer = new byte[dwBytesRead];
                Marshal.Copy ( ppRxBuffer, pRxBuffer, 0, ( int ) dwBytesRead );
            }
            else
            {
                pRxBuffer = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte[] pData )
        {
            /*
			 * Modified API is provided immediately after this API under the commented section "ISOUPDATEBINARY".
			 * The hardcoding of bExtendedLenApdu is made to avoid the PTB compilation error. This is a temporary fix.
			 * Permanent solution is changing the wrapper API signature and test cases should be modified to adapt
			 * to this modified API
			 */
            byte bExtendedLenApdu = 0x00; /* 0x00 indicates Length Lc is passed as short APDU */

            return IsoUpdateBinary ( bOffset, bSfid, bExtendedLenApdu, pData );
        }

        /// <summary>
        /// Iso Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.</param>
        /// <param name="bSfid">Short ISO File Id. Bit 7 should be 1 to indicate Sfid is supplied.
        ///						Else it is treated as MSB of 2Byte offset.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU. This should be always cleared.</param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] pData )
        {
            return phalMfdfLight_IsoUpdateBinary ( m_pDataParams, bOffset, bSfid, bExtendedLenApdu, pData,
                ( byte ) ( ( pData != null ) ? pData.Length : 0 ) );
        }
        #endregion ISO7816

        #region Originality Check
        /// <summary>
        /// Performs the originality check to verify the genuineness of chip.
        /// </summary>
        ///
        /// <param name="bAddr">Value is always 00. Present for forward compatibility reasons.</param>
        /// <param name="pSignature">The plain 56 bytes originality signature of the PICC.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ReadSign ( byte bAddr, out byte[] pSignature )
        {
            IntPtr ppRxdata = IntPtr.Zero;
            Status_t status = phalMfdfLight_ReadSign ( m_pDataParams, bAddr, ref ppRxdata );

            if ( status.Equals ( Error_Gen.SUCCESS ) )
            {
                pSignature = new byte[SIG_LENGTH];
                Marshal.Copy ( ppRxdata, pSignature, 0, SIG_LENGTH );
            }
            else
            {
                pSignature = null;
            }

            return status;
        }
        #endregion Originality Check

        #region Miscellaneous
        /// <summary>
        /// Reset the authentication
        /// </summary>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ResetAuthentication ()
        {
            return phalMfdfLight_ResetAuthentication ( m_pDataParams );
        }

        /// <summary>
        /// Reset the authentication
        /// </summary>
        /// <param name="bval">If set the PICC level is reset.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t ResetAuthentication ( byte bval )
        {
            Status_t oStatusTmp, oStatus;
            ushort wVal = 0, wResetVal = 0xFFFF;

            if ( bval == 0x01 )
            {
                /*Do a Get Config of  ADDITIONAL_INFO to have the backup of current value*/
                oStatusTmp = GetConfig ( 0x00A1, ref wVal );

                /*Do a Set Config to set value 0xFFFF for PICC level Reset*/
                oStatusTmp = SetConfig ( 0x00A1, wResetVal );
            }

            oStatus = ResetAuthentication ();

            /*Do a Set Config of  ADDITIONAL_INFO to retain the backup value*/
            oStatusTmp = SetConfig ( 0x00A1, wVal );

            return oStatus;
        }

        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read. Will be one of the below values.
        ///							NxpRdLibNet.alMfdfLight.CONFIG.PHAL_MFDFLIGHT_ADDITIONAL_INFO
        ///
        ///						  Support for Software mode only including the above ones.
        ///							NxpRdLibNet.alMfdfLight.CONFIG.PHAL_MFDFLIGHT_WRAPPED_MODE
        ///							NxpRdLibNet.alMfdfLight.CONFIG.PHAL_MFDFLIGHT_SHORT_LENGTH_APDU</param>
        /// <param name="pValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t GetConfig ( ushort wConfig, ref ushort pValue )
        {
            return phalMfdfLight_GetConfig ( m_pDataParams, wConfig, ref pValue );
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to set. Will be one of the below values.
        ///							NxpRdLibNet.alMfdfLight.CONFIG.PHAL_MFDFLIGHT_ADDITIONAL_INFO
        ///
        ///						  Support for Software mode only including the above ones.
        ///							NxpRdLibNet.alMfdfLight.CONFIG.PHAL_MFDFLIGHT_WRAPPED_MODE
        ///							NxpRdLibNet.alMfdfLight.CONFIG.PHAL_MFDFLIGHT_SHORT_LENGTH_APDU</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phalMfdfLight_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Computed the Transaction MAC input as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option. 0xFFFF is for No Diversification.</param>
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="pDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and shold be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pTMI">Transaction MAC Input.</param>
        /// <param name="pTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CalculateTMV ( ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, byte[] pDivInput, byte[] pTMC, byte[] pUid,
            byte[] pTMI, out byte[] pTMV )
        {
            pTMV = new byte[8];
            return phalMfdfLight_CalculateTMV ( m_pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, 0, 0, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pTMC, pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ),
                pTMI, ( uint ) ( ( pTMI == null ) ? 0 : pTMI.Length ), pTMV );
        }

        /// <summary>
        /// Computed the Transaction MAC input as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and shold be LSB first.</param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aTMI">Transaction MAC Input.</param>
        /// <param name="aTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t CalculateTMV ( ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, ushort wRamKeyNo, ushort wRamKeyVer,
            byte[] aTMC, byte[] aUid, byte[] aTMI, out byte[] aTMV )
        {
            aTMV = new byte[8];
            return phalMfdfLight_CalculateTMV ( m_pDataParams, 0xFFFF, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, null,
                0, aTMC, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aTMI, ( uint ) ( ( aTMI == null ) ? 0 : aTMI.Length ), aTMV );
        }

        /// <summary>
        /// Decrypted the reader ID as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option. 0xFFFF is for No Diversification.</param>
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="pDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and shold be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="pTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecryptReaderID ( ushort wOption, ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, byte[] pDivInput, byte[] pTMC, byte[] pUid,
            byte[] pEncTMRI, out byte[] pTMRIPrev )
        {
            pTMRIPrev = new byte[16];
            return phalMfdfLight_DecryptReaderID ( m_pDataParams, wOption, wKeyNoTMACKey, wKeyVerTMACKey, 0, 0, pDivInput,
                ( byte ) ( ( pDivInput == null ) ? 0 : pDivInput.Length ), pTMC, pUid, ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ),
                pEncTMRI, pTMRIPrev );
        }

        /// <summary>
        /// Decrypted the reader ID as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wKeyNoTMACKey">Key number in key store of Software.</param>
        /// <param name="wKeyVerTMACKey">Key version in key store of Software.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session TMAC key will be stored.
        ///							To be used for SAM AV3 only.</param>
        /// <param name="pTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///					   the actual value and shold be LSB first.</param>
        /// <param name="pUid">UID of the card.</param>
        /// <param name="pEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="pTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t DecryptReaderID ( ushort wKeyNoTMACKey, ushort wKeyVerTMACKey, ushort wRamKeyNo, ushort wRamKeyVer, byte[] pTMC, byte[] pUid,
            byte[] pEncTMRI, out byte[] pTMRIPrev )
        {
            pTMRIPrev = new byte[16];
            return phalMfdfLight_DecryptReaderID ( m_pDataParams, 0xFFFF, wKeyNoTMACKey, wKeyVerTMACKey, wRamKeyNo, wRamKeyVer, null, 0, pTMC, pUid,
                ( byte ) ( ( pUid == null ) ? 0 : pUid.Length ), pEncTMRI, pTMRIPrev );
        }
        #endregion Miscellaneous
        #endregion

        #region Memory Maping
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return this.m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion
    }
    #endregion Generic

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    /// </summary>
    [ClassInterface ( ClassInterfaceType.AutoDual )]
    public class Sw : alMfdfLight.Generic
    {
        #region Constants
        private const byte AID_LENGTH = 3;
        private const byte IV_LENGTH = 16;
        private const byte SESSION_KEY_LENGTH = 24;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE DESFire Light Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary>Layer ID for this component, NEVER MODIFY!.</summary>
            public ushort wId;

            /// <summary>Pointer to the parameter structure of the palMifare component.</summary>
            public IntPtr pPalMifareDataParams;

            /// <summary>Pointer to the parameter structure of the KeyStore layer.</summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary>Pointer to the parameter structure of the Crypto layer for encryption.</summary>
            public IntPtr pCryptoDataParamsEnc;

            /// <summary>Pointer to the parameter structure of the CryptoRng layer.</summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary>Pointer to the parameter structure of the CryptoMAC.</summary>
            public IntPtr pCryptoMACDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pHalDataParams;

            /// <summary>Session key for this authentication</summary>
            public fixed byte bSessionKey[SESSION_KEY_LENGTH];

            /// <summary>key number against which this authentication is done</summary>
            public byte bKeyNo;

            /// <summary>Max size of IV can be 16 bytes</summary>
            public fixed byte bIv[IV_LENGTH];

            /// <summary>Authenticate (0x0A), AuthISO (0x1A), AuthAES (0xAA)</summary>
            public byte bAuthMode;

            /// <summary>Aid of the currently selected application</summary>
            public fixed byte pAid[AID_LENGTH];

            /// <summary>DES,3DES, 3K3DES or AES</summary>
            public byte bCryptoMethod;

            /// <summary>Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 apdus.</summary>
            public byte bWrappedMode;

            /// <summary>2 Byte CRC initial value in Authenticate mode.</summary>
            public ushort wCrc;

            /// <summary>4 Byte CRC initial value in 0x1A, 0xAA mode.</summary>
            public uint dwCrc;

            /// <summary>Additional info.</summary>
            public ushort wAdditionalInfo;

            /// <summary>Amount of data to be read. Required for Enc read to verify CRC.</summary>
            public uint dwPayLoadLen;

            /// <summary>Command count within transaction.</summary>
            public ushort wCmdCtr;

            /// <summary>Transaction Identifier.</summary>
            public fixed byte bTI[4];

            /// <summary>Authentication MAC key for the session.</summary>
            public fixed byte bSesAuthMACKey[16];

            /// <summary>Session Auth master key.</summary>
            public fixed byte bKeySessionAuthMaster[16];

            /// <summary>Buffer containing unprocessed bytes for read mac answer stream.</summary>
            public fixed byte pUnprocByteBuff[16];

            /// <summary>Amount of data in the pUnprocByteBuff.</summary>
            public byte bNoUnprocBytes;

            /// <summary>Buffer to store last Block of encrypted data in case of chaining.</summary>
            public fixed byte bLastBlockBuffer[16];

            /// <summary>Last Block Buffer Index.</summary>
            public byte bLastBlockIndex;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pTMIDataParams;

            /// <summary>Parameter for force set Short Length APDU in case of BIG ISO read.</summary>
            public byte bShortLenApdu;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_Sw_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams, IntPtr pKeyStoreDataParams,
            IntPtr pCryptoDataParamsEnc, IntPtr pCryptoMACDataParams, IntPtr pCryptoRngDataParams, IntPtr pTMIDataParams, IntPtr pHalDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE DESFire Light software component.
        /// </summary>
        ///
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore component context.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a Crypto component context for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pTMIUtils">Pointer to TMIUtils parameter sturcture.</param>
        /// <param name="pHalparams">Pointer to Hal parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoSym.Generic pCryptoMAC,
            CryptoRng.Generic pCryptoRng, NxpRdLibNet.TMIUtils pTMIUtils, Hal.Generic pHalparams )
        {
            return phalMfdfLight_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams, ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams,
                ( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams, ( pCryptoMAC == null ) ? IntPtr.Zero : pCryptoMAC.m_pDataParams,
                ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams, ( pTMIUtils == null ) ? IntPtr.Zero : pTMIUtils.m_pDataParams,
                ( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE DESFire Light software component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to a KeyStore component context.</param>
        /// <param name="pCryptoEnc">Pointer to a Crypto component context for encryption.</param>
        /// <param name="pCryptoMAC">Pointer to a Crypto component context for Macing.</param>
        /// <param name="pCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="pTMIUtils">Pointer to TMIUtils parameter sturcture.</param>
        /// <param name="pHalparams">Pointer to Hal parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoSym.Generic pCryptoEnc, CryptoSym.Generic pCryptoMAC,
            CryptoRng.Generic pCryptoRng, NxpRdLibNet.TMIUtils pTMIUtils, Hal.Generic pHalparams )
        {
            return phalMfdfLight_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) wDataParamSize, ( pPalMifare == null ) ? IntPtr.Zero : pPalMifare.m_pDataParams,
                ( pKeyStore == null ) ? IntPtr.Zero : pKeyStore.m_pDataParams, ( pCryptoEnc == null ) ? IntPtr.Zero : pCryptoEnc.m_pDataParams,
                ( pCryptoMAC == null ) ? IntPtr.Zero : pCryptoMAC.m_pDataParams, ( pCryptoRng == null ) ? IntPtr.Zero : pCryptoRng.m_pDataParams,
                ( pTMIUtils == null ) ? IntPtr.Zero : pTMIUtils.m_pDataParams, ( pHalparams == null ) ? IntPtr.Zero : pHalparams.m_pDataParams );
        }
#endif
        #endregion Initialization

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion Memory Maping

        #region Parameter Access
        public byte[] pAid
        {
            get
            {
                byte[] bValue = new byte[AID_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < AID_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        public byte[] bIv
        {
            get
            {
                byte[] bValue = new byte[IV_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < IV_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->bIv[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > IV_LENGTH )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->bIv[i] = value[i];
                        }
                    }
                }
            }
        }
        public byte[] bSessionKey
        {
            get
            {
                byte[] bValue = new byte[SESSION_KEY_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->bSessionKey[i];
                        }
                    }
                }
                return bValue;
            }
        }
        public ushort wCmdCtr
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        return pDataParams->wCmdCtr;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        pDataParams->wCmdCtr = value;
                    }
                }
            }
        }
        public byte AuthMode
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        return pDataParams->bAuthMode;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        pDataParams->bAuthMode = value;
                    }
                }
            }
        }
        #endregion Parameter Access
    }
    #endregion Software

#if PACKAGE_INTERNAL || PACKAGE_EXPORT_CONTROLLED
    #region Sam_NonX
    #region SamAV3
    /// <summary>
    /// Class for Sam AV3 NonX Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV3_NonX : alMfdfLight.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE DESFire Light SamAV3 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pTMIDataParams;

            /// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
            public byte bAuthMode;

            /// <summary> AuthType. Whether its EV2 or LRP. </summary>
            public byte bAuthType;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte pAid[3];

            /// <summary> Specific error codes for Desfire generic errors. </summary>
            public ushort wAdditionalInfo;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_SamAV3_NonX_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
            IntPtr pPalMifareDataParams, IntPtr pTMIDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialization API for AL MIFARE DESFire Light to communicate with SamAV3 in NonX Mode.
        /// </summary>
        ///
        /// <param name="oHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to Pal Mifare parameter structure</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 oHalSam, palMifare.Generic oPalMifare, TMIUtils oTMIUtils )
        {
            return phalMfdfLight_SamAV3_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oHalSam == null ) ? IntPtr.Zero : oHalSam.m_pDataParams, ( oPalMifare == null ) ? IntPtr.Zero : oPalMifare.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL MIFARE DESFire Light to communicate with SamAV3 in NonX Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        /// <param name="oPalMifare">Pointer to Pal Mifare parameter structure</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 oHalSam, palMifare.Generic oPalMifare, TMIUtils oTMIUtils )
        {
            return phalMfdfLight_SamAV3_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize, ( oHalSam == null ) ? IntPtr.Zero : oHalSam.m_pDataParams,
                ( oPalMifare == null ) ? IntPtr.Zero : oPalMifare.m_pDataParams, ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }
#endif
        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_NonX ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
            }
        }

        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte AuthType
        {
            get { return m_DataParamsInt[0].bAuthType; }
            set { m_DataParamsInt[0].bAuthType = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion SamAV3	`
    #endregion Sam_NonX

    #region Sam_X
    #region SamAV3
    /// <summary>
    /// Class for Sam AV3 X Mode layer initialization interface and data params.
    /// </summary>
    public class SamAV3_X : alMfdfLight.Generic
    {
        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE DESFire Light SamAV3 X layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component. </summary>
            public ushort wId;

            /// <summary> Pointer to SamAV3 parameter structure. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary>Pointer to the HAL parameters structure.</summary>
            public IntPtr pTMIDataParams;

            /// <summary> Auth Mode. 0x0A or 0x1A or 0xAA. </summary>
            public byte bAuthMode;

            /// <summary> AuthType. Whether its EV2 or LRP. </summary>
            public byte bAuthType;

            /// <summary> Key number against which authenticated. </summary>
            public byte bKeyNo;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte pAid[3];

            /// <summary> Specific error codes for Desfire generic errors. </summary>
            public ushort wAdditionalInfo;
        };
        #endregion Data Structure

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfdfLight_SamAV3_X_Init ( ref DataParams_t m_pDataParams, ushort wSizeOfDataParams, IntPtr pHalSamDataParams,
            IntPtr pTMIDataParams );
        #endregion DLL Imports

        #region Initialization
        /// <summary>
        /// Initialization API for AL MIFARE DESFire Light to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( Hal.SamAV3 pHalSam, TMIUtils oTMIUtils )
        {
            ushort wStatus = phalMfdfLight_SamAV3_X_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams, ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
            return wStatus;
        }

#if DEBUG
        /// <summary>
        /// Initialization API for AL MIFARE DESFire Light to communicate with SamAV3 in X Mode.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pHalSam">Pointer to Hal SamAV3 parameter structure.</param>
        ///
        /// <returns>Returns Success status for successfull operation.
        ///			 Other Depending on implementation and underlaying component.</returns>
        public Status_t Init ( ushort wDataParamSize, Hal.SamAV3 pHalSam, TMIUtils oTMIUtils )
        {
            return phalMfdfLight_SamAV3_X_Init ( ref m_DataParamsInt[0], wDataParamSize, ( pHalSam == null ) ? IntPtr.Zero : pHalSam.m_pDataParams,
                ( oTMIUtils == null ) ? IntPtr.Zero : oTMIUtils.m_pDataParams );
        }
#endif

        #endregion

        #region Memory Maping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public SamAV3_X ()
        {
            // Allocate internal data parameters and pointer to them
            this.m_DataParamsInt = new DataParams_t[1];
            this.m_pDataParamsInt = GCHandle.Alloc ( this.m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~SamAV3_X ()
        {
            // Free allocated pointer to data params
            if ( this.m_pDataParamsInt.IsAllocated )
            {
                this.m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter & Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                this.m_DataParamsInt[0] = value;
            }
            get
            {
                return this.m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public NxpRdLibNet.Hal.Generic HalSamDataParams
        {
            get
            {
                GCHandle Handle = ( GCHandle ) m_DataParamsInt[0].pHalSamDataParams;
                return ( Handle.Target as NxpRdLibNet.Hal.SamAV3 );
            }
        }

        public ushort ID
        {
            get { return m_DataParamsInt[0].wId; }
            set { m_DataParamsInt[0].wId = value; }
        }

        public byte AuthMode
        {
            get { return m_DataParamsInt[0].bAuthMode; }
            set { m_DataParamsInt[0].bAuthMode = value; }
        }

        public byte KeyNo
        {
            get { return m_DataParamsInt[0].bKeyNo; }
            set { m_DataParamsInt[0].bKeyNo = value; }
        }

        public ushort AdditionalInfo
        {
            get { return m_DataParamsInt[0].wAdditionalInfo; }
            set { m_DataParamsInt[0].wAdditionalInfo = value; }
        }

        public byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[3];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 3; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &this.m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion SamAV3	`
    #endregion Sam_X
#endif
}
