/*
 * Copyright 2021 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

using System;
using System.Runtime.InteropServices;

namespace NxpRdLibNet.alMfDuoX
{
    #region Enumerations
    #region Error Codes
    /// <summary>
    /// Custom error codes equivalent to C library error codes.
    /// </summary>
    public enum Error : byte
    {
        /// <summary>
        /// Field to indicate No Changes error.
        /// No changes done to backup files, CommitTransaction / AbortTransaction not necessary. or
        /// Format of the command is not correct (e.g. too many or too few bytes).
        /// </summary>
        NO_CHANGES = CustomCodes.ERROR_BEGIN,

        /// <summary>
        /// Field to indicate Out Of Memory error.
        /// Insufficient NV-Memory to complete command.
        /// </summary>
        OUT_OF_EEPROM_ERROR,

        /// <summary>
        /// Field to indicate No Suck Key error.
        /// Invalid key number specified.
        /// </summary>
        NO_SUCH_KEY,

        /// <summary>
        /// Field to indicate Permission Denied error.
        /// Current configuration / Status does not allow the requested command.
        /// </summary>
        PERMISSION_DENIED,

        /// <summary>
        /// Field to indicate Application Not Found error.
        /// Requested AID not present on PICC.
        /// </summary>
        APPLICATION_NOT_FOUND,

        /// <summary>
        /// Field to indicate Boundary Error.
        /// Attempt to read/write data from/to beyond the files/records limits. Attempt to
        /// exceed the limits of a value file.
        /// </summary>
        BOUNDARY_ERROR,

        /// <summary>
        /// Field to indicate Command Aborted error.
        /// Previous Command was not fully completed. Not all Frames were requested or provided by the PCD.
        /// </summary>
        COMMAND_ABORTED,

        /// <summary>
        /// Field to indicate Duplicate error.
        /// Creation of file/application failed because file/application with same number already exists.
        /// </summary>
        DUPLICATE,

        /// <summary>
        /// Field to indicate File Not Found error.
        /// Specified file number does not exist.
        /// </summary>
        FILE_NOT_FOUND,

        /// <summary>
        /// Field to indicate Integrity error.
        /// CRC or MAC does not match data. Padding bytes not valid.
        /// </summary>
        PICC_CRYPTO,

        /// <summary>
        /// Field to indicate Parameter error.
        /// Value of the parameter(s) invalid.
        /// </summary>
        PARAMETER_ERROR,

        /// <summary>
        /// Field to indicate PICC's General error. Call <see cref="Generic.GetConfig"/>
        /// interface with option as <see cref="Config.ADDITIONAL_INFO"/>.
        /// General Errors might consists of the below following
        /// <list type="bullet">
        ///     <item><description>Application Integrity Error (0xA1).</description></item>
        ///     <item><description>PICC Integrity Error (0xC1).</description></item>
        ///     <item><description>EEPROM Error (0xEE).</description></item>
        ///     <item><description>PICC Disabled Error (0xCD).</description></item>
        ///     <item><description>Illegal Command Code (0x1C).</description></item>
        /// </list>
        /// </summary>
        DF_GEN_ERROR,

        /// <summary>
        /// Field to indicate PICC's ISO7816 error(s). Call <see cref="Generic.GetConfig"/>
        /// interface with option as <see cref="Config.ADDITIONAL_INFO"/>.
        /// General Errors might consists of the below following
        /// <list type="bullet">
        ///     <item><description>Correct Operation with maximum supported protocol version (0x9Fxx).</description></item>
        ///     <item><description>Memory Failure (0x6581).</description></item>
        ///     <item><description>Wrong Length (0x6700).</description></item>
        ///     <item><description>Wrong Le (0x6C00).</description></item>
        ///     <item><description>Incorrect parameters in the command data field (0x6A80).</description></item>
        ///     <item><description>File not found (0x6A82).</description></item>
        ///     <item><description>Wrong Params (0x6A86).</description></item>
        ///     <item><description>Wrong LC (0x6A87).</description></item>
        ///     <item><description>No Precise Diagnostics (0x6F00).</description></item>
        ///     <item><description>EOF Reached (0x6282).</description></item>
        ///     <item><description>File Access (0x6982).</description></item>
        ///     <item><description>File Empty (0x6985).</description></item>
        ///     <item><description>Incorrect Params (0x6B00).</description></item>
        ///     <item><description>Wrong CLA (0x6E00).</description></item>
        ///     <item><description>Unsupported INS (0x6D00).</description></item>
        ///     <item><description>Limited Functionality INS (0x6283).</description></item>
        /// </list>
        /// </summary>
        DF_7816_GEN_ERROR,

        /// <summary>
        /// Field to indicate Command Invalid error.
        /// Command is received in a state where this command is not supported, or a totally
        /// unknown command is received.
        /// </summary>
        CMD_INVALID,

        /// <summary>
        /// Field to indicate Not Supported error.
        /// </summary>
        NOT_SUPPORTED,

        /// <summary>
        /// Field to indicate Limited Operation error.
        /// Successful operation with limited functionality.
        /// </summary>
        OPERATION_OK_LIM,

        /// <summary>
        /// Field to indicate Command Overflow error.
        /// Too many commands in the session or transaction.
        /// </summary>
        CMD_OVERFLOW,

        /// <summary>
        /// Field to indicate General Failure error.
        /// Failure in the operation of the PD.
        /// </summary>
        GEN_FAILURE,

        /// <summary>
        /// Field to indicate Invalid Block number error.
        /// Invalid Block number: not existing in the implementation or not valid to target with this command.
        /// </summary>
        BNR,

        /// <summary>
        /// Field to indicate Format error.
        /// Format of the command is not correct (e.g. too many or too few bytes).
        /// </summary>
        FORMAT,

        /// <summary>
        /// Field to indicate Certificate error.
        /// Reader certificate or CertAccessRights related error.
        /// </summary>
        CERTIFICATE,

        /// <summary>
        /// Field to indicate Weak Field error.
        /// Field strength not sufficient to enable power harvesting for the targeted current/voltage level.
        /// </summary>
        WEAK_FIELD
    };
    #endregion Error Codes

    #region Secondary Application
    /// <summary>
    /// Option to set the Secondary Application Indicator bit for most of the commands.
    /// </summary>
    public enum TargetApp : byte
    {
        /// <summary> Target primary application.</summary>
        PRIMARY = 0x00,

        /// <summary> Target secondary application.</summary>
        SECONDARY = 0x80
    }
    #endregion

    #region ISOFileInfo
    /// <summary>
    /// Options for application / file creation interfaces.
    /// </summary>
    public enum ISOFileInfo : byte
    {
        /// <summary> Option to indicate no ISO File ID or ISODFName are present. </summary>
        NOT_AVAILABLE = 0x00,

        /// <summary> Option to indicate the presence of ISO FileID. </summary>
        ISO_FILE_ID_AVAILABLE = 0x01,

        /// <summary> Option to indicate the presence of ISO DFName. </summary>
        ISO_DF_NAME_AVAILABLE = 0x02,

        /// <summary> Option to indicate the presence of both ISO FileID and ISO DFName. </summary>
        ISO_FILE_ID_DF_NAME_AVAILABLE = 0x03
    }
    #endregion

    #region Key Type
    /// <summary>
    /// Option to be used for below mentioned interfaces.
    /// <see cref="Generic.InitializeKeySet"/>
    /// <see cref="Generic.CreateTransactionMacFile"/>
    /// </summary>
    public enum KeyType : byte
    {
        /// <summary> Option for Invalid key type. </summary>
        INVALID = 0xFF,

        /// <summary> Option for AES 128 key type. </summary>
        AES128 = 0x02,

        /// <summary> Option for AES 256 key type. </summary>
        AES256 = 0x03
    }
    #endregion

    #region Communication Options
    /// <summary>
    /// The communication mode to be used for Command / Response.
    /// </summary>
    public enum CommOption : byte
    {
        /// <summary> Plain mode of communication. The Command / Response will be is plain format.</summary>
        PLAIN = 0x00,

        /// <summary> MAC mode of communication. The Command / Response will have MAC appended at last. </summary>
        MAC = 0x10,

        /// <summary>
        /// Enciphered mode of communication. The Command Data / Response Data will be encrypted
        /// and will have MAC appended at last.
        /// </summary>
        FULL = 0x30,
    }
    #endregion

    #region Secure Messaging Options
    #region Auth Types
    /// <summary>
    /// Type of ISOGeneralAuthentication or EV2 Authentication to perform. To be used with below mentioned interfaces.
    /// <see cref="Generic.ISOGeneralAuthenticate"/>
    /// <see cref="Generic.ISOGeneralAuthenticate_Final"/>
    /// <see cref="Generic.AuthenticateEv2"/>
    /// </summary>
    public enum AuthType : byte
    {
        /// <summary> Authentication type as EV2 NonFirst. </summary>
        EV2_NON_FIRST = 0x00,

        /// <summary> Authentication type as EV2 First. </summary>
        EV2_FIRST = 0x01,

        /// <summary>
        /// Authentication type as ISOGeneralAuthentication. Option to perform Part 1 and Part
        /// 2 exchange.
        /// </summary>
        ISOGENERAL_AUTHENTICATION = 0x02,

        /// <summary>
        /// Authentication type as ISOGeneralAuthenticate_Final. Option to perform second part of
        /// authentication initiated by ISOSelectFile.
        /// </summary>
        ISOGENERAL_AUTHENTICATION_FINAL = 0x04,

        /// <summary> Authentication type as ISOGeneral Mutual Authentication with Certificate available (Cert.A). </summary>
        MUTUAL_AUTH_WITH_CERT = 0x80,

        /// <summary> Authentication type as ISOGeneral Mutual Authentication with no Certificate available (Cert.A). </summary>
        MUTUAL_AUTH_WITHOUT_CERT = 0xA0,

        /// <summary> Authentication type as ISOGeneral Reader Uni-Lateral Authentication with Certificate available (Cert.A). </summary>
        READER_UNILATERAL_AUTH_WITH_CERT = 0x40,

        /// <summary> Authentication type as ISOGeneral Reader Uni-Lateral Authentication with Certificate available (Cert.A). </summary>
        READER_UNILATERAL_AUTH_WITHOUT_CERT = 0x60,

        /// <summary> Authentication type as Not Authenticated. </summary>
        NOT_AUTHENTICATED = 0xFF
    }
    #endregion Auth Types

    #region HashAlgorithm
    /// <summary>
    /// Hashing Algorithm options to be used with <see cref="Generic.ISOGeneralAuthenticate_Verify"/> interface.
    /// </summary>
    public enum HashAlgo : byte
    {
        /// <summary> None. Hashing will not be performed for the message. </summary>
        NO_HASH = 0x00,

        /// <summary> The SHA-224 hashing algorithm will be applied on Message. </summary>
        SHA224 = 0x03,

        /// <summary> The SHA-256 hashing algorithm will be applied on Message. </summary>
        SHA256 = 0x04,

        /// <summary> The SHA-384 hashing algorithm will be applied on Message. </summary>
        SHA384 = 0x05,

        /// <summary> The SHA-512 hashing algorithm will be applied on Message. </summary>
        SHA512 = 0x06,
    }
    #endregion HashAlgorithm

    #region Diversification Options
    /// <summary>
    /// Diversification method to be used for key diversification. To be used with <see cref="Generic.AuthenticateEv2"/>
    /// interface.
    /// </summary>
    public enum DivOption_Auth : ushort
    {
        /// <summary> No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary>
        /// Encryption based method of diversification.
        /// Also known as DESFire mode of diversification.
        /// </summary>
        ENCR = 0x0000,

        /// <summary>
        /// CMAC based method of diversification.
        /// Also known as PLUS mode of diversification.
        /// </summary>
        CMAC = 0x0001
    }
    #endregion Auth Types
    #endregion

    #region Memory and Configuration
    #region SetConfiguration
    /// <summary>
    /// Options for <see cref="Generic.SetConfiguration"/> interface.
    /// </summary>
    public enum SetConfig : byte
    {
        /// <summary> Option for updating the PICC Configuration. </summary>
        PICC_CONFIGURATION = 0x00,

        /// <summary> Option for updating the Default Keys. </summary>
        DEFAULT_KEYS_UPDATE = 0x01,

        /// <summary> Option for updating the ATS. </summary>
        ATS_UPDATE = 0x02,

        /// <summary> Option for updating the SAK. </summary>
        SAK_UPDATE = 0x03,

        /// <summary> Option for updating the Secure Messaging. </summary>
        SM_CONFIGURATION = 0x04,

        /// <summary> Option for updating the Capability Data. </summary>
        CAPABILITY_DATA = 0x05,

        /// <summary> Option for updating the VC Installation Identifier. </summary>
        VCIID = 0x06,

        /// <summary> Option for updating the ATQA Information. </summary>
        ATQA_UPDATE = 0x0C,

        /// <summary> Option for updating the NFC Management. </summary>
        NFC_MANAGEMENT = 0x0F,

        /// <summary> Option for updating the I2C Management. </summary>
        I2C_MANAGEMENT = 0x10,

        /// <summary> Option for updating the GPIO Management. </summary>
        GPIO_MANAGEMENT = 0x11,

        /// <summary> Option for updating the Crypto API Management. </summary>
        CRYPTO_API_MANAGEMENT = 0x15,

        /// <summary> Option for updating the Authentication Counter and Limit Configuration. </summary>
        AUTH_COUNTR_LIMIT_CONFIGURATION = 0x16,

        /// <summary> Option for updating the HALT and Wake-Up Configuration. </summary>
        HALT_WAKEUP_CONFIGURATION = 0x17,

        /// <summary> Option for updating the Lock Configuration. </summary>
        LOCK_CONFIGURATION = 0xFF
    }
    #endregion

    #region GetVersion
    /// <summary>
    /// Options for <see cref="Generic.GetVersion"/> interface.
    /// </summary>
    public enum FabID : byte
    {
        /// <summary>
        /// Option for not exchanging the Option information in <see cref="Generic.GetVersion">GetVersion</see>
        /// command. If used, the FabID will not be available in the response.
        /// </summary>
        DO_NOT_RETURN = 0x00,

        /// <summary>
        /// Option for exchanging the Option information in <see cref="Generic.GetVersion">GetVersion</see>
        /// command to retrieve the FabID information. If used, the FabID will be available in the response.
        /// </summary>
        RETURN = 0x01
    }
    #endregion
    #endregion

    #region Symmetric Key Management
    #region Diversification Options
    /// <summary>
    /// Diversification options to be used with below mentioned interfaces.
    /// <see cref="Generic.ChangeKey"/>
    /// <see cref="Generic.ChangeKeyEV2"/>
    /// </summary>
    public enum DivOption_ChangeKey : ushort
    {
        /// <summary> No diversification. </summary>
        NO_DIVERSIFICATION = 0xFFFF,

        /// <summary> Indicating diversification of new key required. </summary>
        NEW_KEY = 0x0002,

        /// <summary> Indicating diversification of old key required. </summary>
        OLD_KEY = 0x0004,

        /// <summary>
        /// Encryption based method of diversification.
        /// Also known as DESFire mode of diversification.
        /// </summary>
        ENCR = 0x0000,

        /// <summary>
        /// CMAC based method of diversification.
        /// Also known as PLUS mode of diversification.
        /// </summary>
        CMAC = 0x0001
    }
    #endregion

    #region KeySetting
    /// <summary>
    /// Option to be used for <see cref="Generic.GetKeySettings"/> interface.
    /// </summary>
    public enum KeySetting : byte
    {
        /// <summary> Option for Unknown Key settings option. </summary>
        UNKNOWN = 0xFF,

        /// <summary>
        /// Option for retrieval of PICC or Application Key settings.
        /// If this option is used, the option byte will not be exchanged to PICC.
        /// </summary>
        PICC_APPLICATION = 0x00,

        /// <summary> Option for retrieval of ECC Private Key MetaData. </summary>
        ECC_PRIVATE_KEY_METADATA = 0x01,

        /// <summary> Option for retrieval of CA Root Key MetaData. </summary>
        CA_ROOT_KEY_METADATA = 0x02
    }
    #endregion
    #endregion

    #region ASymmetric Key Management
    #region TargetAction
    /// <summary>
    /// Target Action options to be used with <see cref="Generic.ManageKeyPair"/> interface.
    /// </summary>
    public enum TargetAction
    {
        /// <summary> Option for Key Pair generation. </summary>
        GENERATE_KEY_PAIR = 0x00,

        /// <summary> Option for Private Key Import. </summary>
        IMPORT_PRIVATE_KEY = 0x01,

        /// <summary> Option for Meta-Data update. </summary>
        UPDATE_META_DATA = 0x02
    }
    #endregion

    #region TargetCurve
    /// <summary>
    /// Target Curve IS to be used with <see cref="Generic.ManageKeyPair"/> and
    /// <see cref="Generic.ManageCARootKey"/> interface.
    /// </summary>
    public enum TargetCurve
    {
        /// <summary> Option for NIST P-256 Curve ID. </summary>
        NIST_P256 = 0x0C,

        /// <summary> Option for BrainPool P-256 R1 Curve ID. </summary>
        BRAINPOOL_P256R1 = 0x0D
    }
    #endregion

    #region KeyPolicy
    /// <summary>
    /// KeyPolicy Options to be used with <see cref="Generic.ManageKeyPair"/> interface.
    /// </summary>
    public enum KeyPolicy
    {
        /// <summary> Option for Key Policy as disabled. </summary>
        DISABLED = 0x0000,

        /// <summary> Option for Key Policy as Freeze Key Usage Counter Limit. </summary>
        FREEZE_KUC_LIMIT = 0x8000,

        /// <summary> Option for Key Policy as ECC Based Card-Unilateral with ISOInternalAuthenticate. </summary>
        ECC_CARD_UNILATERAL_AUTHENTICATION = 0x0100,

        /// <summary> Option for Key Policy as ECC Based Mutual Authentication. </summary>
        ECC_MUTUAL_AUTHENTICATION = 0x0080,

        /// <summary> Option for Key Policy as ECC Based Transaction Signature. </summary>
        ECC_TRANSACTION_SIGNATURE = 0x0040,

        /// <summary> Option for Key Policy as ECC Based Secure Dynamic Messaging. </summary>
        ECC_SECURE_DYNAMINC_MESSAGING = 0x0020,

        /// <summary> Option for Key Policy as ECC Based CryptoRequest ECC Sign. </summary>
        CRYPTO_REQUEST_ECC_SIGN = 0x0010,
    }
    #endregion
    #endregion

    #region Application Management
    #region Application Type
    /// <summary>
    /// Options for Application Management and File Management commands.
    /// </summary>
    public enum SelectApplication : byte
    {
        /// <summary> Option for Primary application selection.</summary>
        PRIMARY = 0x00,

        /// <summary> Option for Secondary application selection.</summary>
        SECONDARY = 0x01
    }
    #endregion
    #endregion

    #region File Management
    #region File Options
    /// <summary>
    /// The File Options to be used for all the File management commands.
    /// </summary>
    public enum FileOption : byte
    {
        /// <summary> Option for File communication mode as Plain.</summary>
        PLAIN = 0x00,

        /// <summary> Option for File communication mode as Plain. </summary>
        PLAIN_1 = 0x02,

        /// <summary> Option for File communication mode as Mac. </summary>
        MAC = 0x01,

        /// <summary> Option for File communication mode as Full. </summary>
        FULL = 0x03,

        /// <summary> Option to Enable CRL (Certificate Revocation List) File. </summary>
        CRL_FILE = 0x10,

        /// <summary> Option to indicate TMCLimit configuration is enabled. </summary>
        TMCLIMIT_PRESENT = 0x20,

        /// <summary> Option to Enable Secure Dynamic Messaging and Mirroring support. </summary>
        SDM_MIRRORING_ENABLED = 0x40,

        /// <summary> Option to Enable TMI Exclusion file map. </summary>
        TMI_EXCLUSION_FILEMAP = 0x40,

        /// <summary> Option to Enable Additional Access Rights. </summary>
        ADDITIONAL_AR_PRESENT = 0x80
    }
    #endregion

    #region Limited Credit Options
    /// <summary>
    /// The File Options to be used for <see cref="Generic.CreateValueFile"/> interface.
    /// </summary>
    public enum LimitedCredit : byte
    {
        /// <summary> Option to disable Limited credit support.</summary>
        DISABLED = 0x00,

        /// <summary> Option to enable Limited credit support. </summary>
        ENABLED = 0x01
    }
    #endregion

    #region GetValue Options
    /// <summary>
    /// The File Options to be used for <see cref="Generic.CreateValueFile"/> interface.
    /// </summary>
    public enum GetValue : byte
    {
        /// <summary> Option to indicate No Free access to GetValue command.</summary>
        FREE_ACCESS_DISABLED = 0x00,

        /// <summary> Option to indicate Free access to GetValue command. </summary>
        FREE_ACCESS_ENABLED = 0x02
    }
    #endregion

    #region Diversification Options
    /// <summary>
    /// The Diversification Options to be used for <see cref="Generic.CreateTransactionMacFile"/> interface.
    /// </summary>
    public enum DivOption_TMKey
    {
        /// <summary> Option to disable key diversification. </summary>
        DIVERSIFICATION_OFF = 0x00,

        /// <summary> Option to enable key diversification. </summary>
        DIVERSIFICATION_ON = 0x01
    }
    #endregion

    #region TMKey Options
    /// <summary>
    /// The File Options to be used for <see cref="Generic.CreateTransactionMACFile"/> interface.
    /// </summary>
    public enum TMKeyOption : byte
    {
        /// <summary> Option to indicate the mode for TMKey as Transaction MAC.</summary>
        MODE_TRANSACTION_MAC = 0x00,

        /// <summary> Option to indicate the mode for TMKey as Transaction Signature. </summary>
        MODE_TRANSACTION_SIGNATURE = 0x80
    }
    #endregion
    #endregion

    #region DataManagement
    /// <summary>
    /// Chaining format to be used for DataManagement commands.
    /// </summary>
    public enum Chaining : byte
    {
        /// <summary>
        /// Option to represent the native chaining format.
        /// Here AF will be available in command and response.
        /// </summary>
        NATIVE = 0x00,

        /// <summary>
        /// Option to represent the ISO/IEC 14443-4 chaining format.
        /// Here AF will not be available in command and response
        /// rather the chaining will be done using the ISO14443-4 protocol.
        /// In case of Wrapped Mode, Extended Length APDU will be used.
        /// </summary>
        ISO = 0x01,

        /// <summary>
        /// Option to represent the ISO/IEC 14443-4 chaining format
        /// when ISO7816-4 wrapping is enabled. By using this option
        /// in wrapped mode, APDU framing is performed according to
        /// short length format.
        /// </summary>
        ISO_SHORT_LEN_APDU = 0x03
    }

    /// <summary>
    /// CRLFile indication used for <see cref="Generic.WriteData"/> command.
    /// </summary>
    public enum IsCRLFile : ushort
    {
        /// <summary> Option to represent the targeted file is not a CRL file. </summary>
        NO = 0x0000,

        /// <summary> Option to represent the targeted file is a CRL file. </summary>
        YES = 0x4000,

        /// <summary>
        /// Option to represent the generation of CRLSignature for
        /// the complete WriteData payload.
        /// </summary>
        GENERATE_CRL_SIGNATURE = 0x8000
    }
    #endregion

    #region Transaction Management
    /// <summary>
    /// The Options to be used for <see cref="Generic.CommitTransaction"/> interface.
    /// </summary>
    public enum Option
    {
        /// <summary> Option byte is not exchanged to the PICC. </summary>
        NOT_EXCHANGED = 0x80,

        /// <summary> Option byte is exchanged to PICC and represent no return of TMC and TMV / TSV. </summary>
        TRANSACTION_INFO_NOT_RETURNED = 0x00,

        /// <summary> Option byte is exchanged to PICC and represent return of TMC and TMV / TSV. </summary>
        TRANSACTION_INFO_RETURNED = 0x01
    }
    #endregion

    #region Cryptographic Support
    /// <summary>
    /// The Options to be used for <see cref="Generic.CryptoRequest"/> interface.
    /// </summary>
    public enum Action
    {
        /// <summary> Option for CryptoRequest action as Sign. </summary>
        ECC_SIGN = 0x03,

        /// <summary> Option for CryptoRequest as Echo </summary>
        ECHO = 0xFD
    }

    /// <summary>
    /// The Options to be used for <see cref="Generic.CryptoRequest_ECCSign"/> "CryptoRequest_ECCSign" interface.
    /// </summary>
    public enum Operation
    {
        /// <summary> Option for Operation as Initialize Signature. </summary>
        INITIALIZE_SIGNATURE = 0x01,

        /// <summary> Option for Operation as Update data to be signed. </summary>
        UPDATE_DATA = 0x02,

        /// <summary> Option for Operation as Finalize data to be signed. </summary>
        FINALIZE_DATA = 0x03,

        /// <summary> Option for Operation as One-Shot Operation with Raw Data. </summary>
        ONE_SHOT_RAW_DATA = 0x04,

        /// <summary> Option for Operation as One-Shot Operation with Pre-Computed Hash. </summary>
        ONE_SHOT_HASH_DATA = 0x05
    }

    /// <summary>
    /// The Options to be used for <see cref="Generic.CryptoRequest_ECCSign"/> "CryptoRequest_ECCSign" interface.
    /// </summary>
    public enum TargetAlgo
    {
        /// <summary> Option for Algorithm as ECDSA with SHA-256. </summary>
        ECDSA_SIGN = 0x00
    }

    /// <summary>
    /// The Options to be used for <see cref="Generic.CryptoRequest_ECCSign"/> "CryptoRequest_ECCSign" interface.
    /// </summary>
    public enum InputSource
    {
        /// <summary> Option for InputSource as Command Buffer with explicit length. </summary>
        COMMAND_BUFFER_EXPLICIT_LEN = 0xF0
    }
    #endregion

    #region GPIO Management
    #region GPIO Number
    /// <summary>
    /// The Options for representing GPIO number. To be used with <see cref="ManageGPIO"/> interface.
    /// </summary>
    public enum GPIONo
    {
        /// <summary> Option to indicate GPIO number as 1. </summary>
        GPIO_1 = 0x00,

        /// <summary> Option to indicate GPIO number as 2. </summary>
        GPIO_2 = 0x01
    }
    #endregion

    #region Operation
    /// <summary>
    /// The Options for representing operation to perform for the respective GPIO.
    /// To be used with <see cref="ManageGPIO"/> interface.
    /// </summary>
    public static class GPIO_Operation
    {
        #region GPIOxMode Output
        /// <summary> The Options for representing operation to perform for the GPIO configured as output. </summary>
        public enum Output
        {
            /// <summary>
            /// Option to indicate GPIO Control as CLEAR.
            /// Clear the GPIO state to LOW(not driven) or stop power harvesting depending on the mode.
            /// </summary>
            GPIO_CONTROL_CLEAR = 0x00,

            /// <summary>
            /// Option to indicate GPIO Control as SET.
            /// Set the GPIO State to HIGH (driven) or start power harvesting depending on the mode.
            /// </summary>
            GPIO_CONTROL_SET = 0x01,

            /// <summary>
            /// Option to indicate GPIO Control as TOGGLE.
            /// Toggle the GPIO State or power harvesting state depending on the mode.
            /// </summary>
            GPIO_CONTROL_TOGGLE = 0x02,

            /// <summary>
            /// Combined option
            ///     - Option to indicate Pause NFC (only accepted on NFC).
            ///     - Option to indicate Release NFC Pause (only accepted on I2C)
            /// </summary>
            PAUSE_RELEASE_NFC = 0x80
        }
        #endregion

        #region GPIOxMode Down-Stream Power Out
        /// <summary> The Options for representing operation to perform for the GPIO configured as Down-Stream Power Out. </summary>
        public static class DownStreamPowerOut
        {
            /// <summary> Option to indicate GPIO Control to Enable Power Harvesting. </summary>
            public static readonly byte ENABLE_POWER_HARVEST = 0x01;

            /// <summary> Option to indicate GPIO Measurement Control to Execute Measurement. </summary>
            public static readonly byte EXECUTE_MEASUREMENT = 0x02;

            #region 1.8V
            /// <summary> Option to indicate Power downstream voltage of 1.8V. </summary>
            public enum Voltage_1_8
            {
                /// <summary> Option to indicate Power downstream current of 100uA. </summary>
                CURRENT_100uA = 0x04,

                /// <summary> Option to indicate Power downstream current of 300uA. </summary>
                CURRENT_300uA = 0x08,

                /// <summary> Option to indicate Power downstream current of 500uA. </summary>
                CURRENT_500uA = 0x0C,

                /// <summary> Option to indicate Power downstream current of 1mA. </summary>
                CURRENT_1mA = 0x10,

                /// <summary> Option to indicate Power downstream current of 2mA. </summary>
                CURRENT_2mA = 0x14,

                /// <summary> Option to indicate Power downstream current of 3mA. </summary>
                CURRENT_3mA = 0x18,

                /// <summary> Option to indicate Power downstream current of 5mA. </summary>
                CURRENT_5mA = 0x1C,

                /// <summary> Option to indicate Power downstream current of 7mA. </summary>
                CURRENT_7mA = 0x20,

                /// <summary> Option to indicate Power downstream current of 10mA. </summary>
                CURRENT_10mA = 0x24,

                /// <summary> Option to indicate Power downstream voltage of 1.8V and maximum available current. </summary>
                CURRENT_MAX_AVAILABLE = 0x3C,
            }
            #endregion

            #region 2V
            /// <summary> Option to indicate Power downstream voltage of 1.8V. </summary>
            public enum Voltage_2
            {
                /// <summary> Option to indicate Power downstream current of 100uA. </summary>
                CURRENT_100uA = 0x44,

                /// <summary> Option to indicate Power downstream current of 300uA. </summary>
                CURRENT_300uA = 0x48,

                /// <summary> Option to indicate Power downstream current of 500uA. </summary>
                CURRENT_500uA = 0x4C,

                /// <summary> Option to indicate Power downstream current of 1mA. </summary>
                CURRENT_1mA = 0x50,

                /// <summary> Option to indicate Power downstream current of 2mA. </summary>
                CURRENT_2mA = 0x54,

                /// <summary> Option to indicate Power downstream current of 3mA. </summary>
                CURRENT_3mA = 0x58,

                /// <summary> Option to indicate Power downstream current of 5mA. </summary>
                CURRENT_5mA = 0x5C,

                /// <summary> Option to indicate Power downstream current of 7mA. </summary>
                CURRENT_7mA = 0x60,

                /// <summary> Option to indicate Power downstream current of 10mA. </summary>
                CURRENT_10mA = 0x64,

                /// <summary> Option to indicate Power downstream voltage of 1.8V and maximum available current. </summary>
                CURRENT_MAX_AVAILABLE = 0x7C,
            }
            #endregion
        }
        #endregion
    }
    #endregion
    #endregion

    #region ISO7816 - 4
    #region ISOSelect Options
    /// <summary>
    /// Definitions for ISO/IEC 7816-4 FCI modes.
    /// To be used for <see cref="Generic.IsoSelectFile"/> interface.
    /// </summary>
    public enum FCI : byte
    {
        /// <summary> Option to indicate the return of FCI. </summary>
        RETURNED = 0x00,

        /// <summary> Option to indicate the no return of FCI. </summary>
        NOT_RETURNED = 0x0C
    }

    /// <summary>
    /// Definitions for ISO/IEC 7816-4 Selection Controls.
    /// To be used for <see cref="Generic.IsoSelectFile"/> interface.
    /// </summary>
    public enum Selector : byte
    {
        /// <summary> Option to indicate Selection by 2 byte file Id. </summary>
        SELECT_MF_DF_EF_FILE_IDENTIFIER = 0x00,

        /// <summary> Option to indicate Selection by child DF. </summary>
        SELECT_CHILD_DF = 0x01,

        /// <summary> Option to indicate Select EF under current DF. FID = EF id. </summary>
        SELECT_EF_CURRENT_DF = 0x02,

        /// <summary> Option to indicate Select parent DF of the current DF. </summary>
        SELECT_PARENT_DF_CURRENT_DF = 0x03,

        /// <summary> Option to indicate Selection by DF Name. DFName and length is then valid. </summary>
        SELECT_BY_DF_NAME = 0x04
    }
    #endregion

    #region APDU Format
    /// <summary>
    /// Definitions for ISO/IEC 7816-4 APDU Format.
    /// To be used all ISO7816 - 4 interface.
    /// </summary>
    public enum ApduFormat : byte
    {
        /// <summary>
        /// Option to indicate ISO7816-4 APDU format is Short Length
        /// format where LC and LE are of 1 byte.
        /// </summary>
        SHORT_LEN = 0x00,

        /// <summary>
        /// Option to indicate ISO7816-4 APDU format is Extended Length
        /// format where LC is 3 bytes and LE is either 2 or 3 bytes.
        /// </summary>
        EXTENDED_LEN = 0x01
    }
    #endregion

    #region Encoding
    /// <summary>
    /// Definitions for ISO/IEC 7816-4 P1-P2 Encoding of ShortFile identifier or Offset.
    /// To be used with below mentioned interfaces.
    ///     <see cref="Generic.IsoReadBinary"/>
    ///     <see cref="Generic.IsoUpdateBinary"/>
    /// </summary>
    public enum SFID : byte
    {
        /// <summary> Field to indicate Encoding as offset </summary>
        DISABLED = 0x00,

        /// <summary> Field to indicate Encoding as Short File Identifier </summary>
        ENABLED = 0x80
    }
    #endregion

    #region Record Usage
    /// <summary>
    /// Definitions for ISO/IEC 7816-4 Record usage. This is required for P2 information.
    /// To be used with below mentioned interfaces.
    ///     <see cref="Generic.IsoReadRecords"/>
    ///     <see cref="Generic.IsoAppendBinary"/>
    /// </summary>
    public enum RecordUsage : byte
    {
        /// <summary> Field to indicate Reading of Single record. </summary>
        SINGLE = 0x00,

        /// <summary> Field to indicate Reading of all records. </summary>
        ALL = 0x01
    }
    #endregion
    #endregion

    #region EV Charging
    #region Operation
    /// <summary>
    /// Definitions for EV Charging Write Data operation. This is required for P2 information.
    /// To be used with below mentioned interface(s).
    ///     <see cref="Generic.VdeWriteData"/>
    /// </summary>
    public enum EV_Operation : byte
    {
        /// <summary> Field to indicate Write operation on Standard File. </summary>
        WRITE = 0x00,

        /// <summary> Field to indicate Lock operation on Standard File. </summary>
        LOCK = 0x01
    }
    #endregion
    #endregion

    #region Utility Options
    #region Configuration
    /// <summary>
    /// Option to configure some special operations. To be used with the below mentioned interfaces.
    /// <see cref="Generic.GetConfig"/>
    /// <see cref="Generic.SetConfig"/>
    /// </summary>
    public enum Config : ushort
    {
        /// <summary>
        /// Option for GetConfig/SetConfig to get/set additional info of a generic error or some length exposed by interfaces.
        /// </summary>
        ADDITIONAL_INFO = 0x00A1,

        /// <summary>
        /// Option for GetConfig/SetConfig to get/set current Status of command wrapping in ISO 7816-4 APDUs.
        /// </summary>
        WRAPPED_MODE = 0x00A2,

        /// <summary>
        /// Option for GetConfig/SetConfig to get/set current Status of Short Length APDU wrapping in ISO 7816-4 APDUs.
        /// 1: The commands will follow ISO7816 wrapped format with LC and LE as 1 byte.
        /// 0: The commands will follow ISO7816 wrapped format with LC as 3 bytes and LE as 2 or 3 bytes.
        /// </summary>
        SHORT_LENGTH_APDU = 0x00A3,

        /// <summary>
        /// Option for GetConfig/SetConfig to get/set current Status Authentication
        /// </summary>
        AUTH_STATE = 0x00A4
    };
    #endregion

    #region Verification
    /// <summary>
    /// Definitions for <see cref="Generic.ISOGeneralAuthenticate_Verify"/> utility interfaces.
    /// </summary>
    public enum Verify
    {
        /// <summary>
        /// Option for performing the verification of Message received during <see cref="Generic.ISOGeneralAuthenticate"/> or
        /// <see cref="Generic.ISOGeneralAuthenticate_Final"/> command exchange with authentication type as
        /// <see cref="AuthType.MUTUAL_AUTH_WITH_CERT"/>.
        ///     - Generic ECDSA Verification will be performed using the message, signature and Public key provided by the user.
        ///     - User needs to provide Message (OptsB || E.Pub.B), PublicKey and Signature as inputs. Constants and E.Pub.A
        ///       will be taken up internally as part of message.
        ///     - User also needs to provide the Signature and public key for verification.
        ///     - The Authentication state will be updated to AUTHENTICATED after successful verification.
        /// </summary>
        ISO_GENERAL_AUTH_MESSAGE = 0x0000,

        /// <summary>
        /// Option for performing the verification of certificate contents received during <see cref="Generic.ISOGeneralAuthenticate"/> or
        /// <see cref="Generic.ISOGeneralAuthenticate_Final"/> command exchange with authentication type as <see cref="AuthType.MUTUAL_AUTH_WITH_CERT"/>.
        ///     - Here the contents that needs to be verified should be provided by the user. In this case, the inputs from the user should
        ///       be Signature, Message and Public Key.
        ///     - This option will not validate the certificate.
        ///     - This option will not update any Authentication states.
        /// </summary>
        CERTIFICATE_MESSAGE = 0x0001
    }
    #endregion

    #region DAMMAC options
    /// <summary>
    /// Options for DAMMAC information.
    /// </summary>
    public enum DAMMAC : byte
    {
        /// <summary>
        /// Option to indicate the DAMMAC generation is for <see cref="Generic.CreateDelegatedApplication"/> command.
        /// </summary>
        CREATE_DELEGATED_APPLICATION = 0x00,

        /// <summary>
        /// Option to indicate the DAMMAC generation is for <see cref="Generic.DeleteApplication"/> command.
        /// </summary>
        DELETE_APPLICATION = 0x80
    };
    #endregion

    #region Diversification
    /// <summary>
    /// Macro Definitions for below mentioned utility interfaces.
    /// <see cref="Generic.CalculateTMV"/>
    /// <see cref="Generic.DecryptReaderID"/>
    /// </summary>
    public enum Diversification
    {
        /// <summary> Option to disable key diversification. </summary>
        DIVERSIFICATION_OFF = 0x0000,

        /// <summary> Option to enable key diversification. </summary>
        DIVERSIFICATION_ON = 0x0001
    }
    #endregion

    #region SDM Options
    /// <summary>
    /// Macro Definitions for below mentioned utility interfaces.
    /// <see cref="Generic.CalculateMACSDM"/>
    /// <see cref="Generic.DecryptSDMENCFileData"/>
    /// </summary>
    public enum SDMOption : ushort
    {
        /// <summary>
        /// Both VCUID and SDM Read Counter is considered for SDM Session key calculation.
        /// </summary>
        VCUID_RDCTR_PRESENT = 0xC0,

        /// <summary>
        /// Only VCUID is considered for SDM Session key calculation.
        /// </summary>
        VCUID_PRESENT = 0x80,

        /// <summary>
        /// Only RDCTR  is considered for SDM Session key calculation.
        /// </summary>
        RDCTR_PRESENT = 0x40
    };
    #endregion
    #endregion
    #endregion

    #region Generic
    /// <summary>
    /// Class having the wrapper for C command.
    ///
    /// Note:
    ///     Refer below interface for initialization.
    ///         <see cref="Sw.Init">Software Initialization</see>
    /// </summary>
    [NxpRdLibAttribute
        (
            MIFARE_Products.DUOX,
            Message = "Products that support EV2 Authentication, EV2 SecureMessaging and EV2 Secure Dynamic Messaging. " +
                      "Products that support ECC Authentication, ECC SecureMessaging and ECC Secure Dynamic Messaging.",
            OtherInfo = "Backward compatible to Previous version of MIFARE DUOX Products that supports EV2 Secure Messaging."
        )
    ]
    public abstract class Generic
    {
        #region DLL Imports
        #region Secure Messaging
#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ISOGeneralAuthenticate ( IntPtr pDataParams, ushort wOption, byte bCARootKeyNo, ushort wKeyNo_PrivA,
            ushort wKeyPos_PrivA, byte[] pOptsA, byte bOptsALen, byte[] pCertA, ushort wCertALen, byte[] pExpRspLen, ushort wExpRspLen, byte[] pE_PubB,
            ref ushort pE_PubBLen, ref IntPtr ppResponse, ref ushort pRspLen );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ISOInternalAuthenticate ( IntPtr pDataParams, byte bPrivKeyNo, byte bCurveID, byte[] pPubBKey,
            ushort wPubBKeyLen, byte[] pOptsA, byte bOptsALen, byte[] pExpRspLen, byte bExpRspLen );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_AuthenticateEv2 ( IntPtr pDataParams, byte bFirstAuth, ushort wOption, ushort wKeyNo,
            ushort wKeyVer, byte bKeyNoCard, byte[] pDivInput, byte bDivLen, byte[] pPcdCapsIn, byte bLenPcdCapsIn, byte[] pPcdCapsOut,
            byte[] pPdCapsOut );
#endif
        #endregion

        #region Memory and Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_FreeMem ( IntPtr pDataParams, ref IntPtr ppMemInfo, ref ushort pMemInfoLen );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_Format ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_SetConfiguration ( IntPtr pDataParams, byte bOption, byte[] pData, byte bDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetConfiguration ( IntPtr pDataParams, byte bOption, byte bExchangeOption, ref IntPtr ppData,
            ref ushort pDataLen );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetVersion ( IntPtr pDataParams, byte bOption, ref IntPtr ppVersion, ref ushort pVerLen );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetCardUID ( IntPtr pDataParams, ref IntPtr ppUid, ref ushort pUidLen );
#endif
        #endregion

        #region Symmetric Key Management
#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ChangeKey ( IntPtr pDataParams, ushort wOption, ushort wCurrKeyNo, ushort wCurrKeyVer,
            ushort wNewKeyNo, ushort wNewKeyVer, byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ChangeKeyEv2 ( IntPtr pDataParams, ushort wOption, ushort wCurrKeyNo, ushort wCurrKeyVer,
            ushort wNewKeyNo, ushort wNewKeyVer, byte bKeySetNo, byte bKeyNoCard, byte[] pDivInput, byte bDivLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_InitializeKeySet ( IntPtr pDataParams, byte bKeySetNo, byte bKeyType );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_FinalizeKeySet ( IntPtr pDataParams, byte bKeySetNo, byte bKeySetVersion );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_RollKeySet ( IntPtr pDataParams, byte bKeySetNo );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetKeySettings ( IntPtr pDataParams, byte bOption, ref IntPtr ppResponse, ref ushort pRespLen );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ChangeKeySettings ( IntPtr pDataParams, byte bKeySettings );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetKeyVersion ( IntPtr pDataParams, byte bKeyNo, byte bKeySetNo, ref IntPtr ppResponse,
            ref ushort pRespLen );
#endif
        #endregion

        #region ASymmetric Key Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ManageKeyPair ( IntPtr pDataParams, byte bComOption, byte bKeyNo, byte bOption, byte bCurveID,
            byte[] pKeyPolicy, byte bWriteAccess, uint dwKUCLimit, ushort wPrivKey_No, ushort wPrivKey_Pos, ref IntPtr ppResponse,
             ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ManageCARootKey ( IntPtr pDataParams, byte bComOption, byte bKeyNo, byte bCurveID, byte[] pAccessRights,
            byte bWriteAccess, byte bReadAccess, byte bCRLFile, byte[] pCRLFileAID, ushort wPubKey_No, ushort wPubKey_Pos, byte[] pIssuer,
            byte bIssuerLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ExportKey ( IntPtr pDataParams, byte bComOption, byte bOption, byte bKeyNo, ref IntPtr ppResponse,
            ref ushort pRspLen );
        #endregion

        #region Application Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateApplication ( IntPtr pDataParams, byte bOption, byte[] pAid, byte bKeySettings1,
            byte bKeySettings2, byte bKeySettings3, byte[] pKeySetValues, byte bKeySetValuesLen, byte[] pISOFileId, byte[] pISODFName,
            byte bISODFNameLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_DeleteApplication ( IntPtr pDataParams, byte[] pAid, byte[] pDAMMAC, byte bDAMMAC_Len );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateDelegatedApplication ( IntPtr pDataParams, byte bOption, byte[] pAid, byte[] pDamParams,
            byte bDamParamsLen, byte bKeySettings1, byte bKeySettings2, byte bKeySettings3, byte[] bKeySetValues, byte bKeySetValuesLen,
            byte[] pISOFileId, byte[] pISODFName, byte bISODFNameLen, byte[] pEncK, byte bEncKLen, byte[] pDAMMAC, byte bDAMMACLen,
            byte[] pInitK, byte bInitK );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_SelectApplication ( IntPtr pDataParams, byte bOption, byte[] pAid1, byte[] pAid2 );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetApplicationIDs ( IntPtr pDataParams, byte bOption, ref IntPtr ppAidBuff, ref ushort pAidLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetDFNames ( IntPtr pDataParams, byte bOption, ref IntPtr ppDFBuffer, ref ushort wDFBufLen );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetDelegatedInfo ( IntPtr pDataParams, byte[] pDAMSlot, byte[] bDamSlotVer, byte[] pQuotaLimit,
            byte[] pFreeBlocks, byte[] pAid );
#endif
        #endregion

        #region File Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateStdDataFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pISOFileId,
            byte bFileOption, byte[] pAccessRights, byte[] pFileSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateBackupDataFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pISOFileId,
            byte bFileOption, byte[] pAccessRights, byte[] pFileSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateValueFile ( IntPtr pDataParams, byte bFileNo, byte bFileOption, byte[] pAccessRights,
            byte[] pLowerLmit, byte[] pUpperLmit, byte[] pValue, byte bLimitedCredit );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateLinearRecordFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pIsoFileId,
            byte bFileOption, byte[] pAccessRights, byte[] pRecordSize, byte[] pMaxNoOfRec );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateCyclicRecordFile ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pIsoFileId,
            byte bFileOption, byte[] pAccessRights, byte[] pRecordSize, byte[] pMaxNoOfRec );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CreateTransactionMacFile ( IntPtr pDataParams, ushort wOption, byte bFileNo, byte bFileOption,
            byte[] pAccessRights, ushort wKeyNo, byte bTMKeyOption, byte[] pTMKey, byte bTMKeyVer, byte[] pDivInput, byte bDivInputLen,
            byte[] pTMIExclFileMap, byte bTSIGKeyNo );
#endif

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_DeleteFile ( IntPtr pDataParams, byte bFileNo );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetFileIDs ( IntPtr pDataParams, ref IntPtr ppFileId, ref ushort pFileIdLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetISOFileIDs ( IntPtr pDataParams, ref IntPtr ppISOFileId, ref ushort pISOFileIdLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetFileSettings ( IntPtr pDataParams, byte bFileNo, ref IntPtr ppFSBuffer, ref ushort pFSBufLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetFileCounters ( IntPtr pDataParams, byte bOption, byte bFileNo, ref IntPtr ppFileCounters,
            ref ushort pFileCounterLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ChangeFileSettings ( IntPtr pDataParams, byte bOption, byte bFileNo, byte bFileOption,
            byte[] pAccessRights, byte[] pAddInfo, byte bAddInfoLen );
        #endregion

        #region Data Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ReadData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset,
            byte[] pLength, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_WriteData ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, ushort wCRLVer,
            byte[] pOffset, byte[] pData, byte[] pLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetValue ( IntPtr pDataParams, byte bOption, byte bFileNo, ref IntPtr ppResponse,
            ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_Credit ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_Debit ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_LimitedCredit ( IntPtr pDataParams, byte bOption, byte bFileNo, byte[] pData );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ReadRecords ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pRecNo,
            byte[] pRecCount, byte[] pRecSize, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_WriteRecord ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pOffset,
            byte[] pData, byte[] pLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_UpdateRecord ( IntPtr pDataParams, byte bOption, byte bIns, byte bFileNo, byte[] pRecNo,
            byte[] pOffset, byte[] pData, byte[] pLength );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ClearRecordFile ( IntPtr pDataParams, byte bFileNo );
        #endregion

        #region Transaction Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CommitTransaction ( IntPtr pDataParams, byte bOption, ref IntPtr ppTMC, ref ushort pTMCLen,
            ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_AbortTransaction ( IntPtr pDataParams );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CommitReaderID ( IntPtr pDataParams, byte[] pTMRI, byte bTMRILen, ref IntPtr ppEncTMRI,
            ref ushort pEncTMRILen );
        #endregion

        #region Cryptographic Support
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CryptoRequest ( IntPtr pDataParams, byte bComOption, byte bAction, byte[] pInputData, ushort wInputLen,
            ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CryptoRequestECCSign ( IntPtr pDataParams, byte bComOption, byte bOperation, byte bAlgo, byte bKeyNo,
            byte bInputSource, byte[] pInputData, byte bInputLen, ref IntPtr ppSign, ref ushort pSignLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CryptoRequestEcho ( IntPtr pDataParams, byte bComOption, byte[] pInputData, byte bInputLen,
            ref IntPtr ppResponse, ref ushort pRspLen );
        #endregion

        #region GPIO Management
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ManageGPIO ( IntPtr pDataParams, ushort wOption, byte bGPIONo, byte bOperation, byte[] pNFCPauseRspData,
            ushort wNFCPauseRspDataLen, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ReadGPIO ( IntPtr pDataParams, ushort wOption, ref IntPtr ppResponse, ref ushort pRspLen );
        #endregion

        #region ISO7816-4
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_IsoSelectFile ( IntPtr pDataParams, byte bOption, byte bSelector, byte[] pFid, byte[] pDFname,
            byte bDFnameLen, byte bExtendedLenApdu, ref IntPtr ppFCI, ref ushort pFCILen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_IsoReadBinary ( IntPtr pDataParams, ushort wOption, byte bOffset, byte bSfid,
            uint dwBytesToRead, byte bExtendedLenApdu, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_IsoUpdateBinary ( IntPtr pDataParams, byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] pData,
            ushort wDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_IsoReadRecords ( IntPtr pDataParams, ushort wOption, byte bRecNo, byte bReadAllRecords, byte bSfid,
            uint dwBytesToRead, byte bExtendedLenApdu, ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_IsoAppendRecord ( IntPtr pDataParams, byte bSfid, byte bExtendedLenApdu, byte[] pData, ushort wDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_IsoGetChallenge ( IntPtr pDataParams, byte bExpRsp, byte bExtendedLenApdu, ref IntPtr ppResponse,
            ref ushort pRspLen );
        #endregion

        #region EV Charging
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_VdeReadData ( IntPtr pDataParams, ushort wOption, byte bFileNo, ushort wBytesToRead, byte bExtendedLenApdu,
            ref IntPtr ppResponse, ref ushort pRspLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_VdeWriteData ( IntPtr pDataParams, ushort bOperation, byte bExtendedLenApdu, byte[] pData, ushort wDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_VdeECDSASign ( IntPtr pDataParams, ushort wBytesToRead, byte bExtendedLenApdu, byte[] pData, ushort wDataLen,
            ref IntPtr ppResponse, ref ushort pRspLen );
        #endregion

#if !PACKAGE_PUBLIC
        #region Post-Delivery Configuration
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_AuthenticatePDC ( IntPtr pDataParams, ushort wKeyNoCard, ushort wKeyNo, ushort wKeyVer, byte bUpgradeInfo );
        #endregion
#endif

        #region Utility
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GetConfig ( IntPtr pDataParams, ushort wConfig, ref ushort pValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_SetConfig ( IntPtr pDataParams, ushort wConfig, ushort wValue );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ResetAuthentication ( IntPtr pDataParams );

#if !PACKAGE_PUBLIC
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_ISOGeneralAuthenticateVerify ( IntPtr pDataParams, byte[] pPubBKey,
            ushort wPubBKeyLen, byte[] pOptsB, byte bOptsBLen, byte[] pE_PubBKey, ushort wE_PubBKeyLen, byte[] pSignature,
            ushort wSignLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GenerateDAMEncKey ( IntPtr pDataParams, ushort wKeyNo_PICCDAMEncKey,
            ushort wKeyVer_PICCDAMEncKey, ushort wKeyNo_AppDAMDefault, ushort wKeyVer_AppDAMDefault, byte bAppDAMDefaultKeyVer,
            ref IntPtr ppDAMEncKey, ref ushort pDAMEncKeyLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GenerateDAMMAC ( IntPtr pDataParams, byte bOption, ushort wKeyNo_PICCDAMMAC,
            ushort wKeyVer_PICCDAMMAC, byte[] pAid, byte[] pDamParams, byte bDamParamsLen, byte bKeySettings1, byte bKeySettings2,
            byte bKeySettings3, byte[] pKeySetValues, byte bKeySetValuesLen, byte[] pISOFileId, byte[] pISODFName, byte bISODFNameLen,
            byte[] pEncK, byte bEncKLen, ref IntPtr ppDAMMAC, ref ushort pDAMMACLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_GenerateDAMMACSetConfig ( IntPtr pDataParams, ushort wKeyNo_PICCDAMMAC, ushort wKeyVer_PICCDAMMAC,
            byte[] pOld_ISODFName, byte bOld_DFNameLen, byte[] pNew_ISODFName, byte bNew_DFNameLen, ref IntPtr ppDAMMAC, ref ushort pDAMMACLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CalculateTMV ( IntPtr pDataParams, ushort wOption, ushort wKeyNo_TMACKey, ushort wKeyVer_TMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pTMI,
            ushort wTMILen, ref IntPtr ppTMV, ref ushort pTMVLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_VerifyTransationSignature ( IntPtr pDataParams, byte bCurveID, byte[] pPubKey, ushort wPubKey_Len,
            byte[] pTMI, ushort wTMILen, byte[] pTMC, byte[] pTSV, ushort wTSVLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_VerifyTransationSignature ( IntPtr pDataParams, ushort wPubKeyNo_TSigKey, ushort wPubKeyPos_TSigKey,
            byte[] pInput, ushort wInputLen, byte[] pTSV, ushort wTSVLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_DecryptReaderID ( IntPtr pDataParams, ushort wOption, ushort wKeyNo_TMACKey, ushort wKeyVer_TMACKey,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pDivInput, byte bDivInputLen, byte[] pTMC, byte[] pUid, byte bUidLen, byte[] pEncTMRI,
            byte bEncTMRILen, ref IntPtr ppTMRIPrev, ref ushort pTMRIPrevLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_CalculateMACSDM ( IntPtr pDataParams, byte bSdmOption, ushort wKeyNo_SDMMac, ushort wKeyVer_SDMMac,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pInData, ushort wInDataLen,
            ref IntPtr ppSDMMAC, ref ushort pSDMMACLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_VerifySDMSignature ( IntPtr pDataParams, ushort wPubKeyNo_SDMSig, ushort wPubKeyPos_SDMSig, byte[] pInData,
            ushort wInDataLen, byte[] pSignature, ushort pSigLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_DecryptSDMENCFileData ( IntPtr pDataParams, byte bSdmOption, ushort wKeyNo_SDMEnc, ushort wKeyVer_SDMEnc,
            ushort wRamKeyNo, ushort wRamKeyVer, byte[] pUid, byte bUidLen, byte[] pSDMReadCtr, byte[] pEncData, ushort wEncDataLen,
            ref IntPtr ppPlainData, ref ushort pPlainDataLen );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_DecryptSDMPICCData ( IntPtr pDataParams, ushort wKeyNo, ushort wKeyVer, byte[] pEncData, ushort wEncDataLen,
            ref IntPtr ppPlainData, ref ushort pPlainDataLen );
#endif
        #endregion
        #endregion

        #region Properties
        /// <summary>
        /// Get, Sets the Application Identifier.
        /// </summary>
        public virtual byte[] Aid
        {
            get { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
            set { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Get, Sets the Session Encryption Key.
        /// </summary>
        public virtual byte[] SesAuthENCKey
        {
            get { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
            set { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
        }

        /// <summary>
        /// Get, Sets the Session MAC Key.
        /// </summary>
        public virtual byte[] SesAuthMACKey
        {
            get { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
            set { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
        }

        /// <summary>
        /// Get, Sets the Transaction Identifier.
        /// </summary>
        public virtual byte[] TI
        {
            get { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
            set { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
        }
#endif

        /// <summary>
        /// Get, Sets the Command Counter.
        /// </summary>
        public virtual ushort CmdCtr
        {
            get { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
            set { throw new NotImplementedException ( "Not implemented or supported by Derived Class" ); }
        }
        #endregion

        #region Wrapper Functions
        #region Secure Messaging
#if !PACKAGE_PUBLIC
        /// <summary>
        /// Performs Asymmetric Mutual or Reader-Unilateral authentication. This interfaces performs Cmd.ISOGeneralAuthenticate.
        /// Type of Authentication (i.e Mutual or Reader-UniLateral) is extracted from the Option information (\b aOptsA ).
        /// Ephemeral Key Pair (E.Pub.A) is generated by the Library.
        /// Interface provides Ephemeral Key Pair (E.Pub.B) and Decrypted response from PICC after successful execution
        /// of ISOGeneralAuthenticate Part 2.
        ///
        /// Note:
        ///         - Authenticate State will be NOT_AUTHENTICATED if AuthMethod is Mutual Authentication for Cmd.ISOGeneralAuthenticate
        ///           command execution. To update the state to AUTHENTICATED, call <see cref="ISOGeneralAuthenticate_Verify"/>
        ///           interface.
        ///         - Authenticate State will be AUTHENTICATED if AuthMethod is Reader UniLateral Authentication. Here call of
        ///           <see cref="ISOGeneralAuthenticate_Verify"/> interface is not required.
        /// </summary>
        ///
        /// <param name="bCARootKeyNo">Key number of the targeted CA Root Key. Depending on available keys.</param>
        /// <param name="wKeyNo_PrivA">Key number in KeyStore to use, known Private A Key (Priv.A).</param>
        /// <param name="wKeyPos_PrivA">Key position in KeyStore to use, known Private A Key (Priv.A).</param>
        /// <param name="aOptsA">Complete PCD Options in TLV format.</param>
        /// <param name="aCertA">Certificate information of Reader (Cert.A).</param>
        /// <param name="aExpRspLen">Length of expected response from Device.
        ///                             - This parameter is for exchanging the LE information.
        ///                             - If NULL is provided, then the expected Response length will be taken
        ///                               as 0x00 (1 byte) by default.
        ///                             - Possible values are NULL, Array consisting of 1 byte, 2 bytes or 3 bytes.
        /// </param>
        /// <param name="aE_PubB">Ephemeral Public Key (E.Pub.B) received from PICC.</param>
        /// <param name="aResponse">The Decrypted Response received from the Device. Will consists of Decrypted Message B
        ///                             - Mutual Authentication            : Constant || OptsB || [Cert.B] || Sig.B
        ///                             - Reader Uni-lateral Authentication: Constant || OptsB
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If Buffers as null.
        ///         Values not supported for Authentication (wOption parameter)
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If Tag information is not proper for AuthDOHdr of E.Pub.B and Msg.B.enc.
        ///         If there is no Response from PICC in case of ISOGeneralAuthenticate Part 2.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = @"Only ECC (256 bit) key should be used for Authentication. Type of Authentication (i.e Mutual or
                            Reader-UniLateral) is extracted from Option (aOptsA) parameter. Here Only ISOGeneralAuthenticate
                            (Part1 and Part2) will be performed.",
                Message = @"Performs Asymmetric Mutual or Reader-Unilateral authentication."
            )
        ]
        public Status_t ISOGeneralAuthenticate ( byte bCARootKeyNo, ushort wKeyNo_PrivA, ushort wKeyPos_PrivA,
            byte[] aOptsA, byte[] aCertA, byte[] aExpRspLen, out byte[] aE_PubB, out byte[] aResponse )
        {
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;
            ushort wE_PubBLen = 0;

            aE_PubB = new byte[256];
            aResponse = null;

            Status_t oStatus = phalMfDuoX_ISOGeneralAuthenticate ( m_pDataParams, ( ushort ) AuthType.ISOGENERAL_AUTHENTICATION,
                bCARootKeyNo, wKeyNo_PrivA, wKeyPos_PrivA, aOptsA, ( byte ) ( ( aOptsA == null ) ? 0 : aOptsA.Length ), aCertA,
                ( ushort ) ( ( aCertA == null ) ? 0 : aCertA.Length ), aExpRspLen, ( ushort ) ( ( aExpRspLen == null ) ? 0 : aExpRspLen.Length ),
                aE_PubB, ref wE_PubBLen, ref ppResponse, ref wRspLen );

            aE_PubB = Resize ( oStatus, aE_PubB, wE_PubBLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Performs Asymmetric Mutual or Reader-Unilateral authentication. This interfaces performs Cmd.ISOGeneralAuthenticate_Final.
        /// Type of Authentication (i.e Mutual or Reader-UniLateral) is extracted from the Option information (\b aOptsA ).
        /// Ephemeral Key Pair (E.Pub.A) is generated by the Library.
        /// Interface provides Ephemeral Key Pair (E.Pub.A) and Decrypted response from PICC after successful execution
        /// of ISOGeneralAuthenticate Part 2.
        ///
        /// Note:
        ///         - Authenticate State will be NOT_AUTHENTICATED if AuthMethod is Mutual Authentication> for
        ///           ISOGeneralAuthenticateFinal command execution. To update the state to AUTHENTICATED, call
        ///           <see cref="ISOGeneralAuthenticate_Verify"/> interface.
        ///         - Authenticate State will be AUTHENTICATED if AuthMethod is Reader UniLateral Authentication.
        ///           Here call of <see cref="ISOGeneralAuthenticate_Verify"/> interface is not required.
        /// </summary>
        ///
        /// <param name="bCARootKeyNo">Key number of the targeted CA Root Key. Depending on available keys.</param>
        /// <param name="wKeyNo_PrivA">Key number in KeyStore to use, known Private A Key (Priv.A).</param>
        /// <param name="wKeyPos_PrivA">Key position in KeyStore to use, known Private A Key (Priv.A).</param>
        /// <param name="aOptsA">Complete PCD Options in TLV format.</param>
        /// <param name="aCertA">Certificate information of Reader (Cert.A).</param>
        /// <param name="aExpRspLen">Length of expected response from Device.
        ///                             - This parameter is for exchanging the LE information.
        ///                             - If NULL is provided, then the expected Response length will be taken
        ///                               as 0x00 (1 byte) by default.
        ///                             - Possible values are NULL, Array consisting of 1 byte, 2 bytes or 3 bytes.
        /// </param>
        /// <param name="aE_PubB">Ephemeral Public Key received from the Device (E.PubB). param>
        /// <param name="aResponse">The Decrypted Response received from the Device. Will consists of Decrypted Message B
        ///                             - Mutual Authentication            : Constant || OptsB || [Cert.B] || Sig.B
        ///                             - Reader Uni-lateral Authentication: Constant || OptsB
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If Buffers as null.
        ///         Values not supported for Authentication (wOption parameter)
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If Tag information is not proper for AuthDOHdr of E.Pub.B and Msg.B.enc.
        ///         If there is no Response from PICC in case of ISOGeneralAuthenticate_Final.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = @"Only ECC (256 bit) key should be used for Authentication. Type of Authentication (i.e Mutual or
                            Reader-UniLateral) is provided as part bAuthType parameter. Here Only ISOGeneralAuthenticate (Final)
                            will be performed.",
                Message = @"Performs Asymmetric Mutual or Reader-Unilateral authentication."
            )
        ]
        public Status_t ISOGeneralAuthenticate_Final ( byte bCARootKeyNo, ushort wKeyNo_PrivA, ushort wKeyPos_PrivA,
            byte[] aOptsA, byte[] aE_PubB, byte[] aCertA, byte[] aExpRspLen, out byte[] aResponse )
        {
            IntPtr ppResponse = IntPtr.Zero;
            ushort wE_PubKeyBLen = 0;
            ushort wRspLen = 0;

            wE_PubKeyBLen = ( ushort ) ( ( aE_PubB == null ) ? 0 : aE_PubB.Length );

            Status_t oStatus = phalMfDuoX_ISOGeneralAuthenticate ( m_pDataParams, ( ushort ) AuthType.ISOGENERAL_AUTHENTICATION_FINAL,
                bCARootKeyNo, wKeyNo_PrivA, wKeyPos_PrivA, aOptsA, ( byte ) ( ( aOptsA == null ) ? 0 : aOptsA.Length ), aCertA,
                ( ushort ) ( ( aCertA == null ) ? 0 : aCertA.Length ), aExpRspLen, ( ushort ) ( ( aExpRspLen == null ) ? 0 :
                aExpRspLen.Length ), aE_PubB, ref wE_PubKeyBLen, ref ppResponse, ref wRspLen );

            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
#endif

        /// <summary>
        /// Performs Asymmetric Card-Unilateral Authentication.
        /// </summary>
        ///
        /// <param name="bPrivKeyNo">Private Key number for signing the response.
        ///                             At PICC level, two keys are supported.
        ///                             At application level, up to five keys are supported.
        /// </param>
        /// <param name="bCurveID">The targeted curve for the public key provided in \b aPubBKey parameter.
        ///                        Should be one of the below values.
        ///                         - <see cref="TargetCurve.NIST_P256"/>
        ///                         - <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aPubBKey">Public Key (Pub.B) to be used for verification.</param>
        /// <param name="aOptsA">Complete PCD Options in TLV format.</param>
        /// <param name="aExpRspLen">Length of expected response from Device.
        ///                             - This parameter is for exchanging the LE information.
        ///                             - If NULL is provided, then the expected Response length will be taken
        ///                               as 0x00 (1 byte) by default.
        ///                             - Possible values are NULL, Array consisting of 1 byte, 2 bytes or 3 bytes.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If Buffers as null.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If Tag information is not proper for AuthDOHdr, RndB and Signature.
        ///     Returns <see cref="CryptoASym.Error.VERIFICATION_FAILED"/>
        ///         If Verification of Message / Signature combination failed.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = "Performs Asymmetric Card-Unilateral Authentication."
            )
        ]
        public Status_t ISOInternalAuthenticate ( byte bPrivKeyNo, byte bCurveID, byte[] aPubBKey, byte[] aOptsA,
            byte[] aExpRspLen )
        {
            return phalMfDuoX_ISOInternalAuthenticate ( m_pDataParams, bPrivKeyNo, bCurveID, aPubBKey,
                ( byte ) ( ( aPubBKey == null ) ? 0 : aPubBKey.Length ), aOptsA, ( byte ) ( ( aOptsA == null ) ? 0 :
                aOptsA.Length ), aExpRspLen, ( byte ) ( ( aExpRspLen == null ) ? 0 : aExpRspLen.Length ) );
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Performs an EV2 First or Non First Authentication depending upon bFirstAuth Parameter. This will
        /// be using the AES128 keys and will generate and verify the contents based on generic AES algorithm.
        /// </summary>
        ///
        /// <param name="bFirstAuth">One of the below options.
        ///                          <list type="bullet">
        ///                             <item><see cref="AuthType.EV2_FIRST"/></item>
        ///                             <item><see cref="AuthType.EV2_NON_FIRST"/></item>
        ///                          </list>
        /// </param>
        /// <param name="wOption">One of the below options.
        ///                         <see cref="DivOption_Auth.NO_DIVERSIFICATION"/>
        ///                         <see cref="DivOption_Auth.ENCR"/>
        ///                         <see cref="DivOption_Auth.CMAC"/>
        /// </param>
        /// <param name="wKeyNo">Key number in KeyStore of software.</param>
        /// <param name="wKeyVer">Key version in the key store of software.</param>
        /// <param name="bKeyNoCard">Key number on card. ORed with <see cref="TargetApp.SECONDARY"/> to indicate
        ///                          secondary application indicator.</param>
        /// <param name="aDivInput">Diversification input. Can be NULL.</param>
        /// <param name="aPcdCapsIn">PCD Capabilities. Upto 6 bytes.</param>
        /// <param name="aPcdCapsOut">PCD Capabilities.
        ///                           <list type="bullet">
        ///                             <item>If First_Auth    : Information will be available and buffer length will be 6 bytes long.</item>
        ///                             <item>If Non_First_Auth: Information will not be available and buffer will be null.</item>
        ///                           </list>
        /// </param>
        /// <param name="aPdCapsOut">PD Capabilities.
        ///                          <list type="bullet">
        ///                             <item>If First_Auth    : Information will be available and buffer length will be 6 bytes long.</item>
        ///                             <item>If Non_First_Auth: Information will not be available and buffer will be null.</item>
        ///                          </list>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If Buffers as null.
        ///         If Invalid AuthType (bFirstAuth), Invalid Diversification Options (wOption).
        ///         If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Param.KEY"/> If KeyType is not of AES128.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> Status If Response length is not of AES128 Block Size (Single or multiple).
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> If Received RndA do not matches.
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/> If Tag information is not proper for AuthDOHdr and E.Pub.B.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Only AES (128 bits) key should be used for Authentication. After Authentication, EV2 Secure Messaging is used.",
                Message = @"The Secure Messaging depends on FirstAuth parameter.\n
                          EV2_FIRST: The Authentication is intended to be first in transaction.\n
                          EV2_NON_FIRST: The Authentication is intended for any subsequent authentication after Authenticate First."
            )
        ]
        public Status_t AuthenticateEv2 ( byte bFirstAuth, ushort wOption, ushort wKeyNo, ushort wKeyVer, byte bKeyNoCard,
            byte[] aDivInput, byte[] aPcdCapsIn, out byte[] aPcdCapsOut, out byte[] aPdCapsOut )
        {
            aPcdCapsOut = new byte[6];
            aPdCapsOut = new byte[6];

            Status_t oStatus = phalMfDuoX_AuthenticateEv2 ( m_pDataParams, bFirstAuth, wOption, wKeyNo, wKeyVer, bKeyNoCard, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aPcdCapsIn,
                ( byte ) ( ( aPcdCapsIn == null ) ? 0 : aPcdCapsIn.Length ),
                aPcdCapsOut, aPdCapsOut );

            if ( !oStatus.Equals ( new Status_t () ) || bFirstAuth.Equals ( ( byte ) AuthType.EV2_NON_FIRST ) )
            {
                aPcdCapsOut = null;
                aPdCapsOut = null;
            }

            return oStatus;
        }
#endif
        #endregion

        #region Memory and Configuration
        /// <summary>
        /// Returns free memory available on the PICC
        /// </summary>
        ///
        /// <param name="aMemInfo">Current free memory available. Response Will be of 3 bytes with LSB first.
        ///                        If the free memory available is 7592 bytes, then 7592 in Hex will be 0x001F10
        ///                        aMemInfo will contain 10 1F 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = @"Response Will be of 3 bytes with LSB first. Ex: If the free memory available is 7592 bytes, then
                              7592 in Hex will be 0x001F10. The output buffer will have 10 1F 00.",
                Message = "Returns the free memory available on the card."
            )
        ]
        public Status_t FreeMem ( out byte[] aMemInfo )
        {
            Status_t oStatus;
            IntPtr ppMemInfo = IntPtr.Zero;
            ushort wMemInfoLen = 0;

            oStatus = phalMfDuoX_FreeMem ( m_pDataParams, ref ppMemInfo, ref wMemInfoLen );
            aMemInfo = MarshalCopy ( oStatus, ppMemInfo, wMemInfoLen );

            return oStatus;
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Releases the PICC user memory
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Deletes the Files / Applications and releases the memory for reuse.",
                Message = "Releases the PICC user memory."
            )
        ]
        public Status_t Format ()
        {
            return phalMfDuoX_Format ( m_pDataParams );
        }

        /// <summary>
        /// Performs SetConfiguration to configures several aspects of the card or the application.
        /// </summary>
        ///
        /// <param name="bOption">Configuration Option to configure. One of the below values.
        ///                         <see cref="SetConfig.PICC_CONFIGURATION"/>
        ///                         <see cref="SetConfig.DEFAULT_KEYS_UPDATE"/>
        ///                         <see cref="SetConfig.ATS_UPDATE"/>
        ///                         <see cref="SetConfig.SAK_UPDATE"/>
        ///                         <see cref="SetConfig.SM_CONFIGURATION"/>
        ///                         <see cref="SetConfig.CAPABILITY_DATA"/>
        ///                         <see cref="SetConfig.VCIID"/>
        ///                         <see cref="SetConfig.ATQA_UPDATE"/>
        ///                         <see cref="SetConfig.NFC_MANAGEMENT"/>
        ///                         <see cref="SetConfig.I2C_MANAGEMENT"/>
        ///                         <see cref="SetConfig.GPIO_MANAGEMENT"/>
        ///                         <see cref="SetConfig.CRYPTO_API_MANAGEMENT"/>
        ///                         <see cref="SetConfig.AUTH_COUNTR_LIMIT_CONFIGURATION"/>
        ///                         <see cref="SetConfig.HALT_WAKEUP_CONFIGURATION"/>
        ///                         <see cref="SetConfig.LOCK_CONFIGURATION"/>
        /// </param>
        /// <param name="aData">Data for the option specified.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffer is null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Performs SetConfiguration to configures several aspects of the card or the application."
            )
        ]
        public Status_t SetConfiguration ( byte bOption, byte[] aData )
        {
            byte[] aDataTmp = new byte[aData.Length];
            aData.CopyTo ( aDataTmp, 0 );

            return phalMfDuoX_SetConfiguration ( m_pDataParams, bOption, aDataTmp, ( byte ) aDataTmp.Length );
        }

        /// <summary>
        /// Performs GetConfiguration to retrieve configuration aspects of the card or the application.
        /// </summary>
        ///
        /// <param name="bOption">Configuration Option to configure. One of the below values.
        ///                         <see cref="SetConfig.PICC_CONFIGURATION"/>
        ///                         <see cref="SetConfig.ATS_UPDATE"/>
        ///                         <see cref="SetConfig.SAK_UPDATE"/>
        ///                         <see cref="SetConfig.SM_CONFIGURATION"/>
        ///                         <see cref="SetConfig.CAPABILITY_DATA"/>
        ///                         <see cref="SetConfig.VCIID"/>
        ///                         <see cref="SetConfig.ATQA_UPDATE"/>
        ///                         <see cref="SetConfig.NFC_MANAGEMENT"/>
        ///                         <see cref="SetConfig.I2C_MANAGEMENT"/>
        ///                         <see cref="SetConfig.GPIO_MANAGEMENT"/>
        ///                         <see cref="SetConfig.CRYPTO_API_MANAGEMENT"/>
        ///                         <see cref="SetConfig.AUTH_COUNTR_LIMIT_CONFIGURATION"/>
        ///                         <see cref="SetConfig.HALT_WAKEUP_CONFIGURATION"/>
        ///                         <see cref="SetConfig.LOCK_CONFIGURATION"/>
        /// </param>
        /// <param name="bExchangeOption">Whether or not to exchange Option (\b bOption) information to PICC.
        ///                                 <see cref="Value.OFF"/>: Do not exchange Option information to PICC.
        ///                                 <see cref="Value.ON"/> : Exchange Option information to PICC.
        /// </param>
        /// <param name="aData">One of the following information.
        ///                         If bExchangeOption = <see cref="Value.OFF"/> then, Manufacturer Data. will be returned.
        ///                         If bExchangeOption = <see cref="Value.ON"/> then, information based on the Option data will be returned.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For Invalid ExchangeOption (bExchangeOption) value.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = "Performs GetConfiguration to retrieve configuration aspects of the card or the application."
            )
        ]
        public Status_t GetConfiguration ( byte bOption, byte bExchangeOption, out byte[] aData )
        {
            Status_t oStatus;
            IntPtr ppData = IntPtr.Zero;
            ushort wDataLen = 0;

            oStatus = phalMfDuoX_GetConfiguration ( m_pDataParams, bOption, bExchangeOption, ref ppData, ref wDataLen );
            aData = MarshalCopy ( oStatus, ppData, wDataLen );

            return oStatus;
        }
#endif

        /// <summary>
        /// Returns manufacturing related data of the PICC.
        /// </summary>
        ///
        /// <param name="aVerInfo">Returns the complete version information of the PICC. The information includes
        ///                         - Hardware Information
        ///                         - Software Information
        ///                         - Production Related Information
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = @"Response Information includes Hardware, Software and Production Related Information.",
                Message = "Returns manufacturing related data of the PICC."
            )
        ]
        public Status_t GetVersion ( out byte[] aVerInfo )
        {
            Status_t oStatus;
            IntPtr ppVersion = IntPtr.Zero;
            ushort wVerLen = 0;

            oStatus = phalMfDuoX_GetVersion ( m_pDataParams, ( byte ) FabID.DO_NOT_RETURN, ref ppVersion, ref wVerLen );
            aVerInfo = MarshalCopy ( oStatus, ppVersion, wVerLen );

            return oStatus;
        }

        /// <summary>
        /// Returns manufacturing related data of the PICC. Also returns the FabID based on the information provided
        /// to bOption parameter.
        /// </summary>
        ///
        /// <param name="bOption">Option information to be exchanged. Will be one of the following,
        ///                         - <see cref="FabID.DO_NOT_RETURN"/>
        ///                         - <see cref="FabID.RETURN"/>
        /// </param>
        /// <param name="aVerInfo">Returns the complete version information of the PICC. The information includes
        ///                         - Hardware Information
        ///                         - Software Information
        ///                         - Production Related Information
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Response Information includes Hardware, Software and Production Related Information.\n" +
                             "Response includes FabID based on bOption information",
                Message = "Returns manufacturing related data of the PICC."
            )
        ]
        public Status_t GetVersion ( byte bOption, out byte[] aVerInfo )
        {
            Status_t oStatus;
            IntPtr ppVersion = IntPtr.Zero;
            ushort wVerLen = 0;

            oStatus = phalMfDuoX_GetVersion ( m_pDataParams, bOption, ref ppVersion, ref wVerLen );
            aVerInfo = MarshalCopy ( oStatus, ppVersion, wVerLen );

            return oStatus;
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Returns the Unique Identifier of the PICC.
        /// </summary>
        ///
        /// <param name="aUID">Returns the complete UID information of the PICC.</param>
        /// <param name="boOnlyUID">When true, return only UID else return complete information received from PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for Buffers as null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Returns the Unique Identifier of the PICC."
            )
        ]
        public Status_t GetCardUID ( out byte[] aUID, bool boOnlyUID = true )
        {
            Status_t oStatus;
            IntPtr ppUid = IntPtr.Zero;
            ushort wUidLen = 0;

            oStatus = phalMfDuoX_GetCardUID ( m_pDataParams, ref ppUid, ref wUidLen );
            aUID = MarshalCopy ( oStatus, ppUid, wUidLen );

            /* Extract UID */
            if ( boOnlyUID && ( aUID != null ) )
            {
                /* Remove UID Header */
                if ( aUID.Length != 0x07 )
                {
                    Array.Copy ( aUID, 2, aUID, 0, aUID.Length - 2 );
                    Array.Resize ( ref aUID, aUID.Length - 2 );
                }
            }

            return oStatus;
        }
#endif
        #endregion

        #region Symmetric Key Management
#if !PACKAGE_PUBLIC
        /// <summary>
        /// Depending on the currently selected AID, this command update a key of the PICC or
        /// of an application AKS ( Active Key Set).
        /// </summary>
        ///
        /// <param name="wOption">Option to be used for diversifying the Current and New key.
        ///                         - <see cref="DivOption_ChangeKey.NO_DIVERSIFICATION"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        /// </param>
        /// <param name="wCurrKeyNo">Current key number in KeyStore of software.</param>
        /// <param name="wCurrKeyVer">Current key version in KeyStore of software.</param>
        /// <param name="wNewKeyNo">New key number in KeyStore of software.</param>
        /// <param name="wNewKeyVer">New key version in KeyStore of software.</param>
        /// <param name="bKeyNoCard">Key number of the key to be changed. To be ORed with
        ///                             - If at PICC level,
        ///                                 - Bit[7 - 6]: With the type of the key.
        ///                                 - Bit[5 - 0]: PICC level keys.
        ///                             - If at application level,
        ///                                 - Bit[7 - 6]: #PHAL_MFECC_APP_SECONDARY to indicate secondary application
        ///                                               indicator.
        ///                                 - Bit[5 - 0]: Application level keys (0x00 - 0x0D).
        /// </param>
        /// <param name="aDivInput">Diversification input to be used for diversifying the key. Can be NULL.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For invalid card key numbers (bKeyNoCard) are PICC or Application level
        ///         If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Param.KEY"/> KeyType not supported
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> If command is called without prior authentication.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Depending on the currently selected AID, this command updates a key of the PICC or of an application AKS (Active Key Set)."
            )
        ]
        public Status_t ChangeKey ( ushort wOption, ushort wCurrKeyNo, ushort wCurrKeyVer, ushort wNewKeyNo, ushort wNewKeyVer,
            byte bKeyNoCard, byte[] aDivInput )
        {
            return phalMfDuoX_ChangeKey ( m_pDataParams, wOption, wCurrKeyNo, wCurrKeyVer, wNewKeyNo, wNewKeyVer,
                bKeyNoCard, aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Depending on the currently selected AID, this command update a key of the PICC or
        /// of an specified application key set.
        /// </summary>
        ///
        /// <param name="wOption">Option to be used for diversifying the Current and New key.
        ///                         - <see cref="DivOption_ChangeKey.NO_DIVERSIFICATION"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.ENCR"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.NEW_KEY"/>
        ///                         - <see cref="DivOption_ChangeKey.CMAC"/> | <see cref="DivOption_ChangeKey.OLD_KEY"/> |
        ///                           <see cref="DivOption_ChangeKey.NEW_KEY"/>
        /// </param>
        /// <param name="wCurrKeyNo">Current key number in KeyStore of software.</param>
        /// <param name="wCurrKeyVer">Current key version in KeyStore of software.</param>
        /// <param name="wNewKeyNo">New key number in KeyStore of software.</param>
        /// <param name="wNewKeyVer">New key version in KeyStore of software.</param>
        /// <param name="bKeySetNo">Key set number within targeted application.</param>
        /// <param name="bKeyNoCard">Key number of the key to be changed. To be ORed with
        ///                             - If at PICC level,
        ///                                 - Bit[7 - 6]: With the type of the key.
        ///                                 - Bit[5 - 0]: PICC level keys.
        ///                             - If at application level,
        ///                                 - Bit[7 - 6]: #PHAL_MFECC_APP_SECONDARY to indicate secondary application
        ///                                               indicator.
        ///                                 - Bit[5 - 0]: Application level keys (0x00 - 0x0D).
        /// </param>
        /// <param name="aDivInput">Diversification input to be used for diversifying the key. Can be NULL.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For invalid card key numbers (bKeyNoCard) are PICC or Application level
        ///         If Diversification is higher than 31 bytes.
        ///     Returns <see cref="Error_Param.KEY"/> KeyType not supported
        ///     Returns <see cref="Error_Comm.AUTH_ERROR"/> If command is called without prior authentication.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Depending on the currently selected AID, this command updates a key of the PICC or of an specified application key set."
            )
        ]
        public Status_t ChangeKey_Ev2 ( ushort wOption, ushort wCurrKeyNo, ushort wCurrKeyVer, ushort wNewKeyNo, ushort wNewKeyVer,
            byte bKeySetNo, byte bKeyNoCard, byte[] aDivInput )
        {
            return phalMfDuoX_ChangeKeyEv2 ( m_pDataParams, wOption, wCurrKeyNo, wCurrKeyVer, wNewKeyNo, wNewKeyVer, bKeySetNo,
                bKeyNoCard, aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ) );
        }

        /// <summary>
        /// Depending on the currently selected application, initialize the key set with specific index.
        /// </summary>
        ///
        /// <param name="bKeySetNo">Key set number within targeted application.ORed with <see cref="TargetApp.SECONDARY"/>
        ///                         to indicate secondary application indicator.
        /// </param>
        /// <param name="bKeyType">One of the below mentioned options.
        ///                         <see cref="KeyType.AES128"/>
        ///                         <see cref="KeyType.AES256"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for
        ///         Invalid KeySet number (bKeySetNo).
        ///         Invalid Key Type (bKeyType).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Depending on the currently selected application, initialize the key set with specific index."
            )
        ]
        public Status_t InitializeKeySet ( byte bKeySetNo, byte bKeyType )
        {
            return phalMfDuoX_InitializeKeySet ( m_pDataParams, bKeySetNo, bKeyType );
        }

        /// <summary>
        /// Finalizes KeySet targeted by specified KeySet number.
        /// </summary>
        ///
        /// <param name="bKeySetNo">Key set number within targeted application.ORed with <see cref="TargetApp.SECONDARY"/>
        ///                         to indicate secondary application indicator.
        /// </param>
        /// <param name="bKeySetVersion">KeySet version of the key set to be finalized.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for invalid KeySet number (bKeySetNo).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Depending on the currently selected application, finalize the key set with specified number."
            )
        ]
        public Status_t FinalizeKeySet ( byte bKeySetNo, byte bKeySetVersion )
        {
            return phalMfDuoX_FinalizeKeySet ( m_pDataParams, bKeySetNo, bKeySetVersion );
        }

        /// <summary>
        /// RollKeySet changes the ActiveKeySet to key set currently targeted with specified key set number.
        /// </summary>
        ///
        /// <param name="bKeySetNo">Key set number within targeted application.ORed with <see cref="TargetApp.SECONDARY"/>
        ///                         to indicate secondary application indicator.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for invalid KeySet number (bKeySetNo).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Depending on the currently selected application, roll to the key set with specified number."
            )
        ]
        public Status_t RollKeySet ( byte bKeySetNo )
        {
            return phalMfDuoX_RollKeySet ( m_pDataParams, bKeySetNo );
        }
#endif

        /// <summary>
        /// Gets PICC Key Settings of the PICC or APP Key Setting of the application. In addition it returns the number
        /// of keys which are configured for the selected application and if applicable the AppKeySetSettings.
        ///
        /// Note:
        ///     The Option information will not be exchanged to PICC in case of <see cref="KeySetting.PICC_APPLICATION"/>.
        /// </summary>
        ///
        /// <param name="bOption">Option to be used for information retrieval. One of the below values
        ///                         <see cref="KeySetting.PICC_APPLICATION"/>
        ///                         <see cref="KeySetting.ECC_PRIVATE_KEY_METADATA"/>
        ///                         <see cref="KeySetting.CA_ROOT_KEY_METADATA"/>
        /// </param>
        /// <param name="aResponse">Returns the key settings. Can be 2 or 3 bytes.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Depending on the currently selected AID, PICC KeySettings of PICC or the AppKeySettings of the (primary) application is returned. " +
                          "In addition it returns the number of keys which are configured for the selected application and if applicable the AppKeySetSettings."
            )
        ]
        public Status_t GetKeySettings ( byte bOption, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_GetKeySettings ( m_pDataParams, bOption, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Changes the PICC Key settings of the PICC or APP Key Settings of the Application.
        /// </summary>
        ///
        /// <param name="bKeySettings">New key settings to be updated.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Depending on the currently selected AID, this command changes the PICCKeySettings of the PICC or the AppKeySettings of the application."
            )
        ]
        public Status_t ChangeKeySettings ( byte bKeySettings )
        {
            return phalMfDuoX_ChangeKeySettings ( m_pDataParams, bKeySettings );
        }

        /// <summary>
        /// Reads out the current key version of any key stored on the PICC
        /// </summary>
        ///
        /// <param name="bKeyNo">Key number of the targeted key.</param>
        /// <param name="bKeySetNo">Key set number to be used for retrieval.</param>
        /// <param name="aResponse">Key set versions of the selected application ordered by ascending
        ///                         key set number, i.e. starting with the AKS.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Depending on the currently selected AID and given key number parameter, return key version of the key targeted or
                            return all key set versions of the selected application."
            )
        ]
        public Status_t GetKeyVersion ( byte bKeyNo, byte bKeySetNo, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_GetKeyVersion ( m_pDataParams, bKeyNo, bKeySetNo, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
#endif
        #endregion

        #region ASymmetric Key Management
        /// <summary>
        /// Creates or updates a private key entry by generating a key pair or importing a private key.
        ///
        /// Note:
        ///     bComOption will be
        ///         Communication mode of the targeted key or
        ///         If targeting not yet existing key, default CommMode defined by Cmd.SetConfiguration for
        ///         option ECC Key Management should be used. Default is <see cref="CommOption.FULL"/>.
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bKeyNo">Key number of the key to be managed.
        ///                         At PICC level, two keys are supported.
        ///                         At application level, up to five keys are supported.
        /// </param>
        /// <param name="bOption">Target action to perform. Should be one of the below values.
        ///                         <see cref="TargetAction.GENERATE_KEY_PAIR"/>
        ///                         <see cref="TargetAction.IMPORT_PRIVATE_KEY"/>
        ///                         <see cref="TargetAction.UPDATE_META_DATA"/>
        /// </param>
        /// <param name="bCurveID">The targeted curve. Should be one of the below values.
        ///                         <see cref="TargetCurve.NIST_P256"/>
        ///                         <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aKeyPolicy">Defines the allowed crypto operations with the targeted key.
        ///                             Should be two bytes as follows,
        ///                                 Byte 0 => Bit 7 - 0
        ///                                 Byte 1 => Bit 15 - 8
        ///
        ///                             Supported values are, should be ORed
        ///                                 <see cref="KeyPolicy.DISABLED"/>
        ///                                 <see cref="KeyPolicy.FREEZE_KUC_LIMIT"/>
        ///                                 <see cref="KeyPolicy.ECC_CARD_UNILATERAL_AUTHENTICATION"/>
        ///                                 <see cref="KeyPolicy.ECC_MUTUAL_AUTHENTICATION"/>
        ///                                 <see cref="KeyPolicy.ECC_TRANSACTION_SIGNATURE"/>
        ///                                 <see cref="KeyPolicy.ECC_SECURE_DYNAMINC_MESSAGING"/>
        ///                                 <see cref="KeyPolicy.CRYPTO_REQUEST_ECC_SIGN"/>
        /// </param>
        /// <param name="bWriteAccess">Defines the CommMode and access right required to update the key with
        ///                            Cmd.ManageKeyPair. Should contain below information.
        ///                                 Bits[7 - 6]: RFU
        ///                                 Bits[5 - 4]: Communication Modes, One of the below values.
        ///                                     <see cref="CommOption.PLAIN"/>
        ///                                     <see cref="CommOption.MAC"/>
        ///                                     <see cref="CommOption.FULL"/>
        ///                                 Bits[3 - 0]: Access Rights, One of the below values.
        ///                                     At PICC Level
        ///                                         0x00       : PICC Master Key
        ///                                         0x01       : VC Configuration Key
        ///                                         0x02       : ECC-based Delegated Application Management
        ///                                         0x03 - 0x0D: ECC-specific access rights
        ///                                         0x0E       : Free Access
        ///                                         0x0F       : No Access or RFU
        ///                                     At Application Level
        ///                                         0x00 - 0x0D: Authentication Required
        ///                                         0x0D       : [Optional] Free Access over I2C, Authentication required over NFC
        ///                                         0x0E       : Free Access
        ///                                         0x0F       : No Access or RFU
        /// </param>
        /// <param name="dwKUCLimit">Defines the key usage limit of the targeted key.
        ///                             0x00000000: Key Usage Counter Limit is disabled
        ///                             Any other value: Key Usage Counter Limit enabled with the given value ( LSB first ).
        /// </param>
        /// <param name="wPrivKey_No">Key number in KeyStore of Private Key.</param>
        /// <param name="wPrivKey_Pos">Key position in KeyStore of Private Key.</param>
        /// <param name="aResponse">The Public Key in uncompressed point representation format.
        ///                             Present if bOption = <see cref="TargetAction.GENERATE_KEY_PAIR"/>
        ///                             NULL otherwise
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         The values provided in bComOption is not supported.
        ///     Returns <see cref="Error_Param.KEY"/>
        ///         If Key type is not ECC.
        ///         The Key format is not Binary (Uncompressed Point Representation).
        ///         The Key pair is not Private type.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = @"CommMode of targeted key, or if targeting not yet existing key, default CommMode of the
                              command as defined by Cmd.SetConfiguration.",
                Message = @"Creates or updates a private key entry by generating a key pair or importing a private key."
            )
        ]
        public Status_t ManageKeyPair ( byte bComOption, byte bKeyNo, byte bOption, byte bCurveID, byte[] aKeyPolicy, byte bWriteAccess,
            uint dwKUCLimit, ushort wPrivKey_No, ushort wPrivKey_Pos, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_ManageKeyPair ( m_pDataParams, bComOption, bKeyNo, bOption, bCurveID, aKeyPolicy, bWriteAccess,
                dwKUCLimit, wPrivKey_No, wPrivKey_Pos, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Creates or updates a public key entry for storing a CARootKey.
        ///
        /// Note:
        ///     bComOption will be
        ///         Communication mode of the targeted key or
        ///         If targeting not yet existing key, default CommMode defined by Cmd.SetConfiguration for
        ///         option ECC Key Management should be used. Default is <see cref="CommOption.FULL"/>.
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bKeyNo">Key number of the key to be managed.
        ///                         At PICC level, two keys are supported.
        ///                         At application level, up to five keys are supported.
        /// </param>
        /// <param name="bCurveID">The targeted curve. Should be one of the below values.
        ///                         <see cref="TargetCurve.NIST_P256"/>
        ///                         <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aAccessRights">Access rights associated with the KeyID.CARootKey. Should be 2 byte.</param>
        /// <param name="bWriteAccess">Defines the CommMode and access right required to update the key with
        ///                            Cmd.ManageKeyPair. Should contain below information.
        ///                                 Bits[7 - 6]: RFU
        ///                                 Bits[5 - 4]: Communication Modes, One of the below values.
        ///                                     <see cref="CommOption.PLAIN"/>
        ///                                     <see cref="CommOption.MAC"/>
        ///                                     <see cref="CommOption.FULL"/>
        ///                                 Bits[3 - 0]: Access Rights, One of the below values.
        ///                                     At PICC Level
        ///                                         0x00       : PICC Master Key
        ///                                         0x01       : VC Configuration Key
        ///                                         0x02       : ECC-based Delegated Application Management
        ///                                         0x03 - 0x0D: ECC-specific access rights
        ///                                         0x0E       : Free Access
        ///                                         0x0F       : No Access or RFU
        ///                                     At Application Level
        ///                                         0x00 - 0x0D: Authentication Required
        ///                                         0x0D       : [Optional] Free Access over I2C, Authentication required over NFC
        ///                                         0x0E       : Free Access
        ///                                         0x0F       : No Access or RFU
        /// </param>
        /// <param name="bReadAccess">Defines the CommMode and access right required to read the
        ///                           key with Cmd.ExportKey. Should contain below information.
        ///                                 Bits[7 - 6]: RFU
        ///                                 Bits[5 - 4]: Communication Modes, One of the below values.
        ///                                     <see cref="CommOption.PLAIN"/>
        ///                                     <see cref="CommOption.MAC"/>
        ///                                     <see cref="CommOption.FULL"/>
        ///                                 Bits[3 - 0]: Access Rights, One of the below values.
        ///                                     At PICC Level
        ///                                         0x00       : PICC Master Key
        ///                                         0x01       : VC Configuration Key
        ///                                         0x02       : ECC-based Delegated Application Management
        ///                                         0x03 - 0x0D: ECC-specific access rights
        ///                                         0x0E       : Free Access
        ///                                         0x0F       : No Access or RFU
        ///                                     At Application Level
        ///                                         0x00 - 0x0D: Authentication Required
        ///                                         0x0D       : [Optional] Free Access over I2C, Authentication required over NFC
        ///                                         0x0E       : Free Access
        ///                                         0x0F       : No Access or RFU
        /// </param>
        /// <param name="bCRLFile">Defines if certificate revocation is enabled and what file holds the CRL.
        ///                         Bit[7]    : Certificate Revocation
        ///                             0x00  : Disabled
        ///                             0x01  : Enabled
        ///                         Bit[6 - 5]: RFU
        ///                         Bit[6 - 5]: CRL File
        ///                             0x0000: RFU, if Bit7 is 0.
        ///                             Others: File Number, if Bit7 is 1.
        /// </param>
        /// <param name="aCRLAid">The application identifier holding the CRL file. Will be of 3 bytes with LSB first.
        ///                       If application 01 need to be created, then the Aid will be 01 00 00.
        /// </param>
        /// <param name="wPubKey_No">Key number in KeyStore of Public Key.</param>
        /// <param name="wPubKey_Pos">Key position in KeyStore of Public Key.</param>
        /// <param name="aIssuer">The Trusted issuer name. Should be one of the following.
        ///                         NULL in case if No trusted issuer name check required.
        ///                         The Trusted issuer information otherwise. Ranging from 1 - 255 bytes
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         The values provided in bComOption is not supported.
        ///     Returns <see cref="Error_Param.KEY"/>
        ///         If Key type is not ECC.
        ///         The Key format is not Binary (Uncompressed Point Representation).
        ///         The Key pair is not Public type.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = @"CommMode of targeted key, or if targeting not yet existing key, default CommMode of the
                              command as defined by Cmd.SetConfiguration.",
                Message = @"Creates or updates a public key entry for storing a CARootKey."
            )
        ]
        public Status_t ManageCARootKey ( byte bComOption, byte bKeyNo, byte bCurveID, byte[] aAccessRights, byte bWriteAccess,
            byte bReadAccess, byte bCRLFile, byte[] aCRLAid, ushort wPubKey_No, ushort wPubKey_Pos, byte[] aIssuer )
        {
            return phalMfDuoX_ManageCARootKey ( m_pDataParams, bComOption, bKeyNo, bCurveID, aAccessRights, bWriteAccess, bReadAccess,
                bCRLFile, aCRLAid, wPubKey_No, wPubKey_Pos, aIssuer, ( byte ) ( ( aIssuer == null ) ? 0 : aIssuer.Length ) );
        }

        /// <summary>
        /// Exports the public key value of a ECCPrivateKey or CARootKey.
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOption">Target Key Type for Exporting.</param>
        /// <param name="bKeyNo">Key number of the key to be exported.
        ///                         At PICC level, two keys are supported.
        ///                         At application level, up to five keys are supported.
        /// </param>
        /// <param name="aResponse">The Public Key in uncompressed point representation format.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         The values provided in bComOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = @"CommMode of targeted key.",
                Message = @"Exports the public key value of a ECCPrivateKey or CARootKey."
            )
        ]
        public Status_t ExportKey ( byte bComOption, byte bOption, byte bKeyNo, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_ExportKey ( m_pDataParams, bComOption, bOption, bKeyNo, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region Application Management
        /// <summary>
        /// Creates a New Application on the PICC. The application is initialized according to the given settings.
        /// The application key of the active key set are initialized with the Default Application Key.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///                         <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_DF_NAME_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_DF_NAME_AVAILABLE"/>
        /// </param>
        /// <param name="aAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///                    If application 01 need to be created, then the Aid will be 01 00 00.
        /// </param>
        /// <param name="bKeySettings1">Application Key settings. Refer Application Key Settings from DataSheet. </param>
        /// <param name="bKeySettings2">Several other key settings.
        ///                                 Bit[7 - 6]: KeyType of the application keys of the initial AKS
        ///                                     00: Reserved
        ///                                     01: Reserved
        ///                                     10: AES128 KeyType
        ///                                     11: AES256 KeyType
        ///                                 Bit[5]    : Use of 2 byte ISO/IEC 7816-4 File Identifiers
        ///                                     0: No 2 byte File Identifiers for files within the application
        ///                                     1: 2 byte File Identifiers for files within the application required
        ///                                 Bit[4]    : KeySett3 presence
        ///                                     0: Disabled
        ///                                     1: Enabled
        ///                                 Bit[3 - 0]: Number of application keys ( n)
        ///                                     0x00 - 0x0E: Maximum 14 Keys
        ///                                     0x0F       : Enable Application Master Temp Key
        /// </param>
        /// <param name="bKeySettings3">Additional optional key settings.
        ///                                 Bit[7 - 5]: RFU
        ///                                 Bit[4]    : Application Deletion with Application Master Key
        ///                                     0: Depending on PICC Master Key
        ///                                     1: Always Enabled
        ///                                 Bit[3]    : Reserved
        ///                                 Bit[2]    : Application specific Capability data
        ///                                     0: Disabled
        ///                                     4: Enabled
        ///                                 Bit[1]    : Application Specific VC Proximity Key
        ///                                     0: Disabled
        ///                                     2: Enabled
        ///                                 Bit[0]    : Application KeySet
        ///                                     0: Disabled
        ///                                     1: Enabled
        /// </param>
        /// <param name="aKeySetValues">The Key set values for the application. Should as mentioned below.
        ///                                 Byte0 = Application Key Set Version ( ASKVersion)
        ///                                 Byte1 = Number of Key Sets ( NoKeySets)
        ///                                 Byte2 = Maximum Key Size ( MaxKeySize)
        ///                                     Only AES128 Key Type is allowed (upto 16 bytes).
        ///                                     Both AES128 and AES256 Key Types are allowed (upto 32 bytes).
        ///                                 Byte3 = Application KeySet Settings ( AppKeySetSett)
        ///                                     Bit[7 - 4]: RFU
        ///                                     Bit[3 - 0]: Roll Key Access Rights.
        ///                                         0x00 - 0x(n - 1): Active Authentication with specified Application
        ///                                                           Roll Key. (n: number of keys in Active KeySet)
        ///                                         0x(n) - 0x0F    : RFU
        /// </param>
        /// <param name="aISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="aISODFName">ISO DF Name to be used. Should one of the following
        ///                             If bOption = <see cref="ISOFileInfo.ISO_DF_NAME_AVAILABLE"/>, Provided information
        ///                             should be upto 16 bytes.
        ///                             If bOption = <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>, Should be NULL.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Creates a New Application on the PICC. The application is initialized according to the given settings.
                           The application key of the active key set are initialized with the Default Application Key."
            )
        ]
        public Status_t CreateApplication ( byte bOption, byte[] aAid, byte bKeySettings1, byte bKeySettings2, byte bKeySettings3,
            byte[] aKeySetValues, byte[] aISOFileId, byte[] aISODFName )
        {
            return phalMfDuoX_CreateApplication ( m_pDataParams, bOption, aAid, bKeySettings1, bKeySettings2, bKeySettings3,
                aKeySetValues, ( byte ) ( ( aKeySetValues == null ) ? 0 : aKeySetValues.Length ), aISOFileId, aISODFName,
                ( byte ) ( ( aISODFName == null ) ? 0 : aISODFName.Length ) );
        }

        /// <summary>
        /// Permanently deletes the applications on the PICC.
        /// </summary>
        ///
        /// <param name="aAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///                    If application 01 need to be created, then the Aid will be 01 00 00.
        /// </param>
        /// <param name="aDAMMAC">[Optional, present if PICCDAMAuthKey or NXPDAMAuthKey is used for authentication]
        ///                         - The MAC calculated by the card issuer to allow delegated application deletion.
        ///                         - NULL if not targeting a DAM Auth key.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                Message = @"Deletes the Delegated APllication. After deletion, the memory is release for creation of new
                            Delegated Application of Same memory size and targeting the same memory slot."
            )
        ]
        public Status_t DeleteApplication ( byte[] aAid, byte[] aDAMMAC )
        {
            return phalMfDuoX_DeleteApplication ( m_pDataParams, aAid, aDAMMAC, ( byte ) ( ( aDAMMAC == null ) ? 0 : aDAMMAC.Length ) );
        }

        /// <summary>
        /// Permanently deletes the applications on the PICC.
        /// </summary>
        ///
        /// <param name="aAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///                    If application 01 need to be created, then the Aid will be 01 00 00.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffer is null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Permanently deletes application on the PICC. The AID is released and new Application can be created."
            )
        ]
        public Status_t DeleteApplication ( byte[] aAid )
        {
            return DeleteApplication ( aAid, null );
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Creates delegated applications on the PICC with limited memory consumption. The application is
        /// initialized according to the given settings.The application keys of the active key set are
        /// initialized with the provided Application DAM Default Key.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the present of ISO information.
        ///                         <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_DF_NAME_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_DF_NAME_AVAILABLE"/>
        /// </param>
        /// <param name="aAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///                    If application 01 need to be created, then the Aid will be 01 00 00.
        /// </param>
        /// <param name="aDamParams">The DAM information. Should as mentioned below.
        ///                             - Byte0 = DAM Slot No[LSB]
        ///                             - Byte1 = DAM Slot No[MSB]
        ///                             - Byte2 = DAM Slot Version
        ///                             - Byte3 = Quota Limit [LSB]
        ///                             - Byte4 = Quota Limit [MSB]
        /// </param>
        /// <param name="bKeySettings1">Application Key settings. Refer Application Key Settings from DataSheet.</param>
        /// <param name="bKeySettings2">Several other key settings.
        ///                                 Bit[7 - 6]: KeyType of the application keys of the initial AKS
        ///                                     00: Reserved
        ///                                     01: Reserved
        ///                                     10: AES128 KeyType
        ///                                     11: AES256 KeyType
        ///                                 Bit[5]    : Use of 2 byte ISO/IEC 7816-4 File Identifiers
        ///                                     0: No 2 byte File Identifiers for files within the application
        ///                                     1: 2 byte File Identifiers for files within the application required
        ///                                 Bit[4]    : KeySett3 presence
        ///                                     0: Disabled
        ///                                     1: Enabled
        ///                                 Bit[3 - 0]: Number of application keys ( n)
        ///                                     0x00 - 0x0E: Maximum 14 Keys
        ///                                     0x0F       : Enable Application Master Temp Key
        /// </param>
        /// <param name="bKeySettings3">Additional optional key settings.
        ///                                 Bit[7 - 5]: RFU
        ///                                 Bit[4]    : Application Deletion with Application Master Key
        ///                                     0: Depending on PICC Master Key
        ///                                     1: Always Enabled
        ///                                 Bit[3]    : Reserved
        ///                                 Bit[2]    : Application specific Capability data
        ///                                     0: Disabled
        ///                                     4: Enabled
        ///                                 Bit[1]    : Application Specific VC Proximity Key
        ///                                     0: Disabled
        ///                                     2: Enabled
        ///                                 Bit[0]    : Application KeySet
        ///                                     0: Disabled
        ///                                     1: Enabled
        /// </param>
        /// <param name="aKeySetValues">The Key set values for the application. Should as mentioned below.
        ///                                 Byte0 = Application Key Set Version ( ASKVersion)
        ///                                 Byte1 = Number of Key Sets ( NoKeySets)
        ///                                 Byte2 = Maximum Key Size ( MaxKeySize)
        ///                                     Only AES128 Key Type is allowed (upto 16 bytes).
        ///                                     Both AES128 and AES256 Key Types are allowed (upto 32 bytes).
        ///                                 Byte3 = Application KeySet Settings ( AppKeySetSett)
        ///                                     Bit[7 - 4]: RFU
        ///                                     Bit[3 - 0]: Roll Key Access Rights.
        ///                                         0x00 - 0x(n - 1): Active Authentication with specified Application
        ///                                                           Roll Key. (n: number of keys in Active KeySet)
        ///                                         0x(n) - 0x0F    : RFU
        /// </param>
        /// <param name="aISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="aISODFName">ISO DF Name to be used. Should one of the following
        ///                             - If bOption = <see cref="ISOFileInfo.ISO_DF_NAME_AVAILABLE"/>, Provided information
        ///                               should be upto 16 bytes.
        ///                             - If bOption = <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>, Should be NULL.
        /// </param>
        /// <param name="aEncK">Encrypted initial application key for the DAM application.
        ///                     Should be 32 bytes long.
        ///                         - Applicable on if in EV2 Authenticated State.
        ///                         - NULL if in ECC Authenticated State.
        /// </param>
        /// <param name="aDAMMAC">The MAC calculated by the card issuer to allow delegated application creation.
        ///                       Should be 8 bytes long.
        ///                         - Applicable on if in EV2 Authenticated State.
        ///                         - NULL if in ECC Authenticated State.
        /// </param>
        /// <param name="aInitK">The initial application key for the DAM application.
        ///                      Should be 25 bytes long.
        ///                         - Applicable on if in ECC Authenticated State.
        ///                         - NULL if in EV2 Authenticated State.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Creates delegated applications on the PICC with limited memory consumption. The application is initialized
                            according to the given settings. The application keys of the active key set are initialized with the
                            provided Application DAM Default Key."
            )
        ]
        public Status_t CreateDelegatedApplication ( byte bOption, byte[] aAid, byte[] aDamParams, byte bKeySettings1, byte bKeySettings2,
            byte bKeySettings3, byte[] aKeySetValues, byte[] aISOFileId, byte[] aISODFName, byte[] aEncK, byte[] aDAMMAC, byte[] aInitK )
        {
            return phalMfDuoX_CreateDelegatedApplication ( m_pDataParams, bOption, aAid, aDamParams, ( byte ) ( ( aDamParams == null ) ? 0 : aDamParams.Length ),
                bKeySettings1, bKeySettings2, bKeySettings3, aKeySetValues, ( byte ) ( ( aKeySetValues == null ) ? 0 : aKeySetValues.Length ), aISOFileId,
                aISODFName, ( byte ) ( ( aISODFName == null ) ? 0 : aISODFName.Length ), aEncK, ( byte ) ( ( aEncK == null ) ? 0 : aEncK.Length ),
                aDAMMAC, ( byte ) ( ( aDAMMAC == null ) ? 0 : aDAMMAC.Length ), aInitK, ( byte ) ( ( aInitK == null ) ? 0 : aInitK.Length ) );
        }
#endif

        /// <summary>
        /// Selects one particular application on the PICC for further access.
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///                         <see cref="SelectApplication.PRIMARY"/>
        ///                         <see cref="SelectApplication.SECONDARY"/>
        /// </param>
        /// <param name="aAid1">The primary application identifier to be used. Will be of 3 bytes with LSB first.
        ///                     If application 01 need to be selected, then the Aid will be 01 00 00.
        /// </param>
        /// <param name="aAid2">The secondary application identifier to be used. Will be of 3 bytes with LSB first.
        ///                     If application 01 need to be selected, then the Aid will be 01 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffer is null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Select 1 or 2 applications or the PICC level specified by their application identifier."
            )
        ]
        public Status_t SelectApplication ( byte bOption, byte[] aAid1, byte[] aAid2 )
        {
            return phalMfDuoX_SelectApplication ( m_pDataParams, bOption, aAid1, aAid2 );
        }

        /// <summary>
        /// Returns application identifiers of all applications on the PICC.
        ///
        /// <remarks>
        /// The status will be <see cref="Error_Gen.SUCCESS"/> if all the application ids can be obtained in one call.
        /// If not, then <see cref="Error_Gen.SUCCESS_CHAINING"/> is returned. The user has to call this interface
        /// with bOption = <see cref="ExchangeOptions.RXCHAINING"/> to get the remaining AIDs.
        /// </remarks>
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : Exchanges the command and received the application ID's.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive remaining Application ID's.
        /// </param>
        /// <param name="aAidBuff">The available identifiers of the application(s).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> for successful operation with chaining response.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For Invalid buffering options (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Returns the Application IDentifiers of all active applications on a PICC."
            )
        ]
        public Status_t GetApplicationIDs ( byte bOption, out byte[] aAidBuff )
        {
            Status_t oStatus;
            IntPtr ppAidBuff = IntPtr.Zero;
            ushort wAidLen = 0;

            oStatus = phalMfDuoX_GetApplicationIDs ( m_pDataParams, bOption, ref ppAidBuff, ref wAidLen );
            aAidBuff = MarshalCopy ( oStatus, ppAidBuff, wAidLen );

            return oStatus;
        }

        /// <summary>
        /// Returns the Application IDentifiers together with a File ID and (optionally) a DF
        /// Name of all active applications with ISO/IEC 7816-4 support.
        ///
        /// <remarks>
        /// The status will be <see cref="Error_Gen.SUCCESS"/> if all the application DFName's can be obtained in one call.
        /// If not, then <see cref="Error_Gen.SUCCESS_CHAINING"/> is returned. The user has to call this interface with
        /// bOption = <see cref="ExchangeOptions.RXCHAINING"/> to get the remaining DFName's.
        /// </remarks>
        /// </summary>
        ///
        /// <param name="bOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : Exchanges the command and received the application DFName's.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive remaining Application DFName's.
        /// </param>
        /// <param name="aDFBuffer">The ISO information about the application.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> for successful operation with chaining response.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         For Invalid buffering options (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Returns the Application IDentifiers together with a File ID and (optionally) a DF Name of all active
                            applications with ISO/IEC 7816-4 support."
            )
        ]
        public Status_t GetDFNames ( byte bOption, out byte[] aDFBuffer )
        {
            Status_t oStatus;
            IntPtr ppDFBuffer = IntPtr.Zero;
            ushort wDFBufLen = 0;

            oStatus = phalMfDuoX_GetDFNames ( m_pDataParams, bOption, ref ppDFBuffer, ref wDFBufLen );
            aDFBuffer = MarshalCopy ( oStatus, ppDFBuffer, wDFBufLen );

            return oStatus;
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Returns the DAMSlotVersion and QuotaLimit of a target DAM slot on the card.
        /// </summary>
        ///
        /// <param name="aDAMSlot">Slot number associated with the memory space of the targeted
        ///                        delegated application. LSB first.
        /// </param>
        /// <param name="bDamSlotVer">Slot version associated with the memory space of the targeted
        ///                           delegated application.
        /// </param>
        /// <param name="aQuotaLimit">Maximal memory consumption of the targeted delegated application
        ///                           (in 32-byte blocks).
        /// </param>
        /// <param name="aFreeBlocks">Amount of unallocated memory of the targeted delegated application
        ///                           (in 32-byte blocks).
        /// </param>
        /// <param name="aAid">Application Identifier of the targeted DAM slot. This parameter is set to
        ///                    0x000000 if the DAM slot targeted by DAMSlotNo exists on the card, but is
        ///                    currently not occupied with an application, i.e. the latest application has
        ///                    been deleted.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> if the buffers are null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Returns the DAMSlotVersion and QuotaLimit of a target DAM slot on the card."
            )
        ]
        public Status_t GetDelegatedInfo ( byte[] aDAMSlot, out byte bDamSlotVer, out byte[] aQuotaLimit, out byte[] aFreeBlocks,
            out byte[] aAid )
        {
            bDamSlotVer = 0;
            byte[] aQuotaLimit_Tmp = new byte[2];
            byte[] aFreeBlocks_Tmp = new byte[2];
            byte[] aAid_Tmp = new byte[3];
            byte[] aDamSlotVer_Tmp = new byte[1];

            bDamSlotVer = 0;
            aQuotaLimit = null;
            aFreeBlocks = null;
            aAid = null;

            Status_t oStatus = phalMfDuoX_GetDelegatedInfo ( m_pDataParams, aDAMSlot, aDamSlotVer_Tmp, aQuotaLimit_Tmp, aFreeBlocks_Tmp,
                aAid_Tmp );

            if ( oStatus.Equals ( new Status_t () ) )
            {
                bDamSlotVer = aDamSlotVer_Tmp[0];
                aQuotaLimit = ( byte[] ) aQuotaLimit_Tmp.Clone ();
                aFreeBlocks = ( byte[] ) aFreeBlocks_Tmp.Clone ();
                aAid = ( byte[] ) aAid_Tmp.Clone ();
            }

            return oStatus;
        }
#endif
        #endregion

        #region File Management
        /// <summary>
        /// Creates files for the storage of plain unformatted user data within an existing application
        /// on the PICC.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the presence of ISO information.
        ///                         <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.ADDITIONAL_AR_PRESENT"/>
        ///                                 <see cref="FileOption.SDM_MIRRORING_ENABLED"/>
        ///                                 <see cref="FileOption.CRL_FILE"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : Write
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     0x0 - 0xD: Authentication Required
        ///                                     0xD      : [Optional] Free access over I2C, authentication required over NFC
        ///                                     0xE      : Free Access
        ///                                     0xF      : No Access or RFU
        /// </param>
        /// <param name="aFileSize">The size of the file. Will be of 3 bytes with LSB first.
        ///                         If size 0x10 need to be created, then the FileSize will be 10 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid Option (bOption) information.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Creates files for the storage of plain unformatted user data within an existing application on the PICC."
            )
        ]
        public Status_t CreateStdDataFile ( byte bOption, byte bFileNo, byte[] aISOFileId, byte bFileOption, byte[] aAccessRights,
            byte[] aFileSize )
        {
            return phalMfDuoX_CreateStdDataFile ( m_pDataParams, bOption, bFileNo, aISOFileId, bFileOption, aAccessRights, aFileSize );
        }

        /// <summary>
        /// Creates files for the storage of plain unformatted user data within an existing application
        /// on the PICC, additionally supporting the feature of an integrated backup mechanism.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the presence of ISO information.
        ///                         <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.ADDITIONAL_AR_PRESENT"/>
        ///                                 <see cref="FileOption.CRL_FILE"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : Write
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     0x0 - 0xD: Authentication Required
        ///                                     0xD      : [Optional] Free access over I2C, authentication required over NFC
        ///                                     0xE      : Free Access
        ///                                     0xF      : No Access or RFU
        /// </param>
        /// <param name="aFileSize">The size of the file. Will be of 3 bytes with LSB first.
        ///                         If size 0x10 need to be created, then the FileSize will be 10 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid Option (bOption) information.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Creates files for the storage of plain unformatted user data within an existing application
                            on the PICC, additionally supporting the feature of an integrated backup mechanism."
            )
        ]
        public Status_t CreateBackupDataFile ( byte bOption, byte bFileNo, byte[] aISOFileId, byte bFileOption, byte[] aAccessRights,
            byte[] aFileSize )
        {
            return phalMfDuoX_CreateBackupDataFile ( m_pDataParams, bOption, bFileNo, aISOFileId, bFileOption, aAccessRights, aFileSize );
        }

        /// <summary>
        /// Creates files for the storage and manipulation of 32bit signed integer values within
        /// an existing application on the PICC.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.ADDITIONAL_AR_PRESENT"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : Write
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     0x0 - 0xD: Authentication Required
        ///                                     0xD      : [Optional] Free access over I2C, authentication required over NFC
        ///                                     0xE      : Free Access
        ///                                     0xF      : No Access or RFU
        /// </param>
        /// <param name="aLowerLmit">The lower limit for the file. Will be of 4 bytes with LSB first.
        ///                          If size 0x10 need to be created, then the value will be 10 00 00 00.
        /// </param>
        /// <param name="aUpperLmit">The upper limit for the file. Will be of 4 bytes with LSB first.
        ///                          If size 0x20 need to be created, then the value will be 20 00 00 00.
        /// </param>
        /// <param name="aValue">The initial value. Will be of 4 bytes with LSB first.
        ///                      If size 0x10 need to be created, then the value will be 10 00 00 00.
        /// </param>
        /// <param name="bLimitedCredit">Encodes if LimitedCredit and free GetValue are allowed for this file.
        ///                                 Limited Credit Support
        ///                                     <see cref="LimitedCredit.DISABLED"/>
        ///                                     <see cref="LimitedCredit.ENABLED"/>
        ///
        ///                                 Access to GetValue. Should be ORed with above values.
        ///                                     <see cref="GetValue.FREE_ACCESS_DISABLED"/>
        ///                                     <see cref="GetValue.FREE_ACCESS_ENABLED"/>
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Creates files for the storage and manipulation of 32bit signed integer values within an existing application on the PICC."
            )
        ]
        public Status_t CreateValueFile ( byte bFileNo, byte bFileOption, byte[] aAccessRights, byte[] aLowerLmit, byte[] aUpperLmit,
            byte[] aValue, byte bLimitedCredit )
        {
            return phalMfDuoX_CreateValueFile ( m_pDataParams, bFileNo, bFileOption, aAccessRights, aLowerLmit,
                aUpperLmit, aValue, bLimitedCredit );
        }

        /// <summary>
        /// Creates files for multiple storage of structural similar data, for example for loyalty programs
        /// within an existing application. Once the file is filled, further writing is not possible
        /// unless it is cleared.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the presence of ISO information.
        ///                         <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.ADDITIONAL_AR_PRESENT"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : Write
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     0x0 - 0xD: Authentication Required
        ///                                     0xD      : [Optional] Free access over I2C, authentication required over NFC
        ///                                     0xE      : Free Access
        ///                                     0xF      : No Access or RFU
        /// </param>
        /// <param name="aRecordSize">The size of the file. Will be of 3 bytes with LSB first.
        ///                           If size 0x10 need to be created, then the RecordSize will be 10 00 00.
        /// </param>
        /// <param name="aMaxNoOfRec">The maximum number of record in the file. Will be of 3 bytes with LSB first.
        ///                           If size 0x04 need to be created, then the value will be 04 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid Option (bOption) information.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Creates files for multiple storage of structural similar data, for example for loyalty programs, within an existing
                            application on the PICC. Once the file is filled completely with data records, further writing to the file is not
                            possible unless it is cleared."
            )
        ]
        public Status_t CreateLinearRecordFile ( byte bOption, byte bFileNo, byte[] aIsoFileId, byte bFileOption, byte[] aAccessRights,
            byte[] aRecordSize, byte[] aMaxNoOfRec )
        {
            return phalMfDuoX_CreateLinearRecordFile ( m_pDataParams, bOption, bFileNo, aIsoFileId, bFileOption, aAccessRights,
                aRecordSize, aMaxNoOfRec );
        }

        /// <summary>
        /// Creates files for multiple storage of structural similar data, for example for logging transactions, within an
        /// existing application. Once the file is filled, the PICC automatically overwrites the oldest record with the
        /// latest written one.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the presence of ISO information.
        ///                         <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                         <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aISOFileId">ISO File ID to be used. Will be two bytes.</param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.ADDITIONAL_AR_PRESENT"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : Write
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     0x0 - 0xD: Authentication Required
        ///                                     0xD      : [Optional] Free access over I2C, authentication required over NFC
        ///                                     0xE      : Free Access
        ///                                     0xF      : No Access or RFU
        /// </param>
        /// <param name="aRecordSize">The size of the file. Will be of 3 bytes with LSB first.
        ///                           If size 0x10 need to be created, then the RecordSize will be 10 00 00.
        /// </param>
        /// <param name="aMaxNoOfRec">The maximum number of record in the file. Will be of 3 bytes with LSB first.
        ///                           If size 0x04 need to be created, then the value will be 04 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid Option (bOption) information.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Creates files for multiple storage of structural similar data, for example for logging transactions, within an
                            existing application. Once the file is filled, the PICC automatically overwrites the oldest record with the
                            latest written one."
            )
        ]
        public Status_t CreateCyclicRecordFile ( byte bOption, byte bFileNo, byte[] aIsoFileId, byte bFileOption, byte[] aAccessRights,
            byte[] aRecordSize, byte[] aMaxNoOfRec )
        {
            return phalMfDuoX_CreateCyclicRecordFile ( m_pDataParams, bOption, bFileNo, aIsoFileId, bFileOption, aAccessRights,
                aRecordSize, aMaxNoOfRec );
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Creates a Transaction MAC File and enables the Transaction MAC or Signature feature for the targeted application.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.TMI_EXCLUSION_FILEMAP"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : RFU (0xF)
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                     0x0 - 0xD: CommitReaderID enabled, requiring authentication with the specified
        ///                                                key index or granting equivalent access rights via asymmetric authentication
        ///                                     0xE      : CommitReaderID enabled with Free Access
        ///                                     0xF      : Commit Reader ID Disabled
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     0x0 - 0xD: Authentication Required
        ///                                     0xD      : [Optional] Free access over I2C, authentication required over NFC
        ///                                     0xE      : Free Access
        ///                                     0xF      : No Access or RFU
        /// </param>
        /// <param name="bTMKeyOption">Options for Transaction MAC Key.
        ///                                 The key type that need to be used.
        ///                                     <see cref="KeyType.AES128"/>
        ///                                     <see cref="KeyType.AES256"/>
        ///
        ///                                 The mode that needs to be used.To be ORed with above values.
        ///                                     <see cref="TMKeyOption.MODE_TRANSACTION_MAC"/>
        ///                                     <see cref="TMKeyOption.MODE_TRANSACTION_SIGNATURE"/>
        /// </param>
        /// <param name="aTMKey">The 16 byte key value to be stored in the PICC. This is applicable for Software only.</param>
        /// <param name="wKeyVer">The version AppTransactionMAC Key.</param>
        /// <param name="aTMIExclFileMap">TMI exclusion file-map. Should be 4 byte.</param>
        /// <param name="bTSIGKeyNo">Key Number pointing to AppTransactionSIGKey. Here the key number (\b TSIGKeyNo ) will be
        ///                          used directly for command exchange.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Creates a Transaction MAC File and enables the Transaction MAC or Signature feature for the targeted application."
            )
        ]
        public Status_t CreateTransactionMacFile ( byte bFileNo, byte bFileOption, byte[] aAccessRights, byte bTMKeyOption, byte[] aTMKey,
            byte bTMKeyVer, byte[] aTMIExclFileMap = null, byte bTSIGKeyNo = 0xFF )
        {
            byte[] aTMKey_Tmp = new byte[aTMKey.Length];
            aTMKey.CopyTo ( aTMKey_Tmp, 0 );

            return phalMfDuoX_CreateTransactionMacFile ( m_pDataParams, 0x00, bFileNo, bFileOption, aAccessRights, 0x00, bTMKeyOption,
                aTMKey_Tmp, bTMKeyVer, null, 0x00, aTMIExclFileMap, bTSIGKeyNo );
        }

        /// <summary>
        /// Creates a Transaction MAC File and enables the Transaction MAC or Signature feature for the targeted application.
        /// </summary>
        ///
        /// <param name="wOption">Options to be used for diversifying TransactionMAC Key. One of the below values.
        ///                         <see cref="DivOption_TMKey.DIVERSIFICATION_OFF"/>
        ///                         <see cref="DivOption_TMKey.DIVERSIFICATION_ON"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.TMI_EXCLUSION_FILEMAP"/>
        /// </param>
        /// <param name="aAccessRights">The new access right to be applied for the file. Should be 2 byte.
        ///                                 Bit[15 - 12]: Read
        ///                                 Bit[11 - 8] : RFU (0xF)
        ///                                 Bit[7 - 4]  : ReadWrite
        ///                                     0x0 - 0xD: CommitReaderID enabled, requiring authentication with the specified
        ///                                                key index or granting equivalent access rights via asymmetric authentication
        ///                                     0xE      : CommitReaderID enabled with Free Access
        ///                                     0xF      : Commit Reader ID Disabled
        ///                                 Bit[3 - 0]  : Change or RFU.Change for the 1st mandatory set of access
        ///                                               condition else RFU ( 0xF)
        ///
        ///                                 Below are the values for the above bits.
        ///                                     0x0 - 0xD: Authentication Required
        ///                                     0xD      : [Optional] Free access over I2C, authentication required over NFC
        ///                                     0xE      : Free Access
        ///                                     0xF      : No Access or RFU
        /// </param>
        /// <param name="wKeyNo">Key number of the KeyStore in Software. Can be one of the following.
        ///                         KeyID of KeyStore pointing to AppTransactionMACKey.Here the Key information
        ///                         will retrieved from KeyStore and will be used for (TMKey) command exchange.
        ///
        ///                         KeyID of KeyStore pointing to AppTransactionSIGKey. Here the key number (TSIGKeyNo)
        ///                         will be used directly for command exchange.
        /// </param>
        /// <param name="bTMKeyOption">Options for Transaction MAC Key.
        ///                                 The key type that need to be used.
        ///                                     <see cref="KeyType.AES128"/>
        ///                                     <see cref="KeyType.AES256"/>
        ///
        ///                                 The mode that needs to be used.To be ORed with above values.
        ///                                     <see cref="TMKeyOption.MODE_TRANSACTION_MAC"/>
        ///                                     <see cref="TMKeyOption.MODE_TRANSACTION_SIGNATURE"/>
        /// </param>
        /// <param name="aTMKey">The 16 byte key value to be stored in the PICC. This is applicable for Software only.</param>
        /// <param name="bTMKeyVer">The version AppTransactionMAC Key.</param>
        /// <param name="aDivInput">Diversification input. Can be NULL.</param>
        /// <param name="aTMIExclFileMap">TMI exclusion file-map. Should be 4 byte.</param>
        /// <param name="bTSIGKeyNo">Key Number pointing to AppTransactionSIGKey. Here the key number (\b TSIGKeyNo ) will be
        ///                          used directly for command exchange.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For Invalid File numbers (bFileNo).
        ///         For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Creates a Transaction MAC File and enables the Transaction MAC or Signature feature for the targeted application."
            )
        ]
        public Status_t CreateTransactionMacFile ( ushort wOption, byte bFileNo, byte bFileOption, byte[] aAccessRights, ushort wKeyNo,
            byte bTMKeyOption, byte[] aTMKey, byte bTMKeyVer, byte[] aDivInput, byte[] aTMIExclFileMap, byte bTSIGKeyNo )
        {
            return phalMfDuoX_CreateTransactionMacFile ( m_pDataParams, wOption, bFileNo, bFileOption, aAccessRights, wKeyNo, bTMKeyOption,
                aTMKey, bTMKeyVer, aDivInput, ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aTMIExclFileMap, bTSIGKeyNo );
        }
#endif

        /// <summary>
        /// Permanently deactivates a file within the file directory of the currently selected application.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Permanently deactivates a file within the file directory of the currently selected application."
            )
        ]
        public Status_t DeleteFile ( byte bFileNo )
        {
            return phalMfDuoX_DeleteFile ( m_pDataParams, bFileNo );
        }

        /// <summary>
        /// Returns the file IDs of all active files within the currently selected application.
        /// </summary>
        ///
        /// <param name="aFileId">The buffer containing the available File ID(s).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Returns the File IDentifiers of all active files within the currently selected application."
            )
        ]
        public Status_t GetFileIDs ( out byte[] aFileId )
        {
            Status_t oStatus;
            IntPtr ppFileId = IntPtr.Zero;
            ushort wFileIdLen = 0;

            oStatus = phalMfDuoX_GetFileIDs ( m_pDataParams, ref ppFileId, ref wFileIdLen );
            aFileId = MarshalCopy ( oStatus, ppFileId, wFileIdLen );

            return oStatus;
        }

        /// <summary>
        /// Get the ISO File IDs.
        /// </summary>
        ///
        /// <param name="aISOFileId">The buffer containing the available ISO File ID(s).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Returns the ISO File IDentifiers of all active files within the currently selected application."
            )
        ]
        public Status_t GetISOFileIDs ( out byte[] aISOFileId )
        {
            Status_t oStatus;
            IntPtr ppISOFileId = IntPtr.Zero;
            ushort wISOFileIdLen = 0;

            oStatus = phalMfDuoX_GetISOFileIDs ( m_pDataParams, ref ppISOFileId, ref wISOFileIdLen );
            aISOFileId = MarshalCopy ( oStatus, ppISOFileId, wISOFileIdLen );

            return oStatus;
        }

        /// <summary>
        /// Get information on the properties of a specific file
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aFSBuffer">The buffer containing the settings.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Get information on the properties of a specific file."
            )
        ]
        public Status_t GetFileSettings ( byte bFileNo, out byte[] aFSBuffer )
        {
            Status_t oStatus;
            IntPtr ppFSBuffer = IntPtr.Zero;
            ushort wFSBufLen = 0;

            oStatus = phalMfDuoX_GetFileSettings ( m_pDataParams, bFileNo, ref ppFSBuffer, ref wFSBufLen );
            aFSBuffer = MarshalCopy ( oStatus, ppFSBuffer, wFSBufLen );

            return oStatus;
        }

        /// <summary>
        /// Get file related counters used for Secure Dynamic Messaging.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                         <see cref="CommOption.PLAIN"/>
        ///                         <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="aFileCounters">The SDMReadCounter information returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                Message = "Get file related counters used for Secure Dynamic Messaging."
            )
        ]
        public Status_t GetFileCounters ( byte bOption, byte bFileNo, out byte[] aFileCounters )
        {
            Status_t oStatus;
            IntPtr ppFileCounters = IntPtr.Zero;
            ushort wFileCounterLen = 0;

            oStatus = phalMfDuoX_GetFileCounters ( m_pDataParams, bOption, bFileNo, ref ppFileCounters, ref wFileCounterLen );
            aFileCounters = MarshalCopy ( oStatus, ppFileCounters, wFileCounterLen );

            return oStatus;
        }

        /// <summary>
        /// Changes the access parameters of an existing file.
        /// </summary>
        ///
        /// <param name="bOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                         <see cref="CommOption.PLAIN"/>
        ///                         <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to be created. ORed with <see cref="TargetApp.SECONDARY"/> to
        ///                       indicate secondary application indicator.
        /// </param>
        /// <param name="bFileOption">Option for the targeted file.
        ///                             Communication settings for the file.
        ///                                 <see cref="FileOption.PLAIN"/>
        ///                                 <see cref="FileOption.MAC"/>
        ///                                 <see cref="FileOption.FULL"/>
        ///
        ///                             ORed with one of the above options.
        ///                                 <see cref="FileOption.ADDITIONAL_AR_PRESENT"/>: For all files other TransactionMAC File.
        ///                                 <see cref="FileOption.SDM_MIRRORING_ENABLED"/>: If Standard File is targeted.
        ///                                 <see cref="FileOption.TMI_EXCLUSION_FILEMAP"/>: If Transaction MAC file is targeted.
        ///                                 <see cref="FileOption.TMCLIMIT_PRESENT"/>: If Transaction MAC file is targeted.
        /// </param>
        /// <param name="aAccessRights">Set of access conditions for the 1st set in the file. Should be 2 byte.</param>
        /// <param name="aAddInfo">Buffer should contain the following information.
        ///                         [NrAddARs] || [Additional access rights] || [SDMOption || SDM AccessRights || VCUIDOffset ||
        ///                         SDMReadCtrOffset || PICCDataOffset || GPIOStatusOffset || SDMMACInputOffset || SDMENCOffset ||
        ///                         SDMENCLength || SDMMACOffset || SDMReadCtrLimit] || [TMIExclFileMap] || [TMCLimit] || [CRLOptions ||
        ///                         CNSSize] || CRLSigKey]
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File numbers (bFileNo).
        ///          For Invalid File communication mode (bFileOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = "Changes the access parameters of an existing file."
            )
        ]
        public Status_t ChangeFileSettings ( byte bOption, byte bFileNo, byte bFileOption, byte[] aAccessRights, byte[] aAddInfo )
        {
            return phalMfDuoX_ChangeFileSettings ( m_pDataParams, bOption, bFileNo, bFileOption, aAccessRights, aAddInfo,
                ( byte ) ( ( aAddInfo == null ) ? 0 : aAddInfo.Length ) );
        }
        #endregion

        #region Data Management
        /// <summary>
        /// Reads data from Standard data files, Backup data files or TransactionMAC File.
        ///
        /// Note:
        ///     - Chaining upto the size of the HAL Response buffer is handled within this function. If more data is to be read, the user
        ///       has to call this function again with bOption = <see cref="ExchangeOptions.RXCHAINING"/> | one of the communication options
        ///     - In either of the exchange options, its must to provide the communication mode also.
        ///     - If reading of data is performed using ISO / IEC 14443-4 chaining mode with wrapped enabled, make sure to disable
        ///       <see cref="Config.SHORT_LENGTH_APDU"/> configuration for data larger than frame size.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing ofSecure Messaging and reading of data.
        ///                         Reading data.
        ///                             <see cref="ExchangeOptions.DEFAULT"/>   : Exchanges the command and received the Data.
        ///                             <see cref="ExchangeOptions.RXCHAINING"/>: To Receive remaining Data.
        ///
        ///                         Communication modes. To be ORed with above values
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bIns">Type of chaining needs to be applied. One of the below values.
        ///                         <see cref="Chaining.NATIVE"/>
        ///                         <see cref="Chaining.ISO"/>
        /// </param>
        /// <param name="bFileNo">The file number from where the data to be read.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aOffset">The offset from where the data should be read. Will be of 3 bytes with LSB first.
        ///                       If 0x10 need to be offset, then it will be 10 00 00.
        ///                         0 to ( FixeSize - 1): Starting position of Read operation.
        ///                         0xFFFFFFFF          : Return CRLFile meta-data
        /// </param>
        /// <param name="aLength">The number of bytes to be read. Will be of 3 bytes with LSB first.
        ///                         If 0x10 bytes need to be read, then it will be 10 00 00.
        ///                         If complete file need to be read, then it will be 00 00 00.
        /// </param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Chaining value (bIns).
        ///          For Invalid Communication option value (bOption).
        ///          For Invalid Exchange option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Both Application Chaining and ISO/IEC 14443-4 Chaining can be performed using the INS parameter information.",
                Message = "Reads data from FileType.StandardData, FileType.BackupData or FileType.TransactionMAC files."
            )
        ]
        public Status_t ReadData ( byte bOption, byte bIns, byte bFileNo, byte[] aOffset, byte[] aLength, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_ReadData ( m_pDataParams, bOption, bIns, bFileNo, aOffset, aLength, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Writes data to standard data files or backup data files. This method is for backward compatibility which do not include
        /// writing information to CRLFile.
        ///
        /// Note:
        ///     - If writing of data is performed using ISO / IEC 14443-4 chaining mode with wrapped enabled, make sure to disable
        ///       <see cref="Config.SHORT_LENGTH_APDU"/> configuration for data larger than frame size.
        ///     - Implements chaining to the card. The data provided on aData will be chained to the card by sending data upto
        ///       the frame size of the MIFARE DUOX PICC, at a time.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging and writing of data.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bIns">Type of chaining needs to be applied. One of the below values.
        ///                         <see cref="Chaining.NATIVE"/>
        ///                         <see cref="Chaining.ISO"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the data to be written.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 need to be offset, then it will be 10 00 00.
        /// </param>
        /// <param name="aData">The data to be written to the PICC.</param>
        /// <param name="aLength">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 bytes need to be written, then it will be 10 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Chaining value (bIns).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Both Application Chaining and ISO/IEC 14443-4 Chaining can be performed using the INS parameter information.",
                Message = "Writes data to standard data files or backup data files."
            )
        ]
        public Status_t WriteData ( byte bOption, byte bIns, byte bFileNo, byte[] aOffset, byte[] aData, byte[] aLength )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            byte bOptionTmp = ( byte ) ( bOption & 0xF0 );
            return phalMfDuoX_WriteData ( m_pDataParams, bOptionTmp, bIns, bFileNo, 0xFFFF, aOffset, aDataTmp, aLength );
        }

        /// <summary>
        /// Writes data to standard data files or backup data files. The Write operation includes writing information to CRLFile also.
        ///
        /// Note:
        ///     - If writing of data is performed using ISO / IEC 14443-4 chaining mode with wrapped enabled, make sure to disable
        ///       <see cref="Config.SHORT_LENGTH_APDU"/> configuration for data larger than frame size.
        ///     - Implements chaining to the card. The data provided on aData will be chained to the card by sending data upto
        ///       the frame size of the MIFARE DUOX PICC, at a time.
        ///     - CRLSignature should be computed externally and provided as part of \b pData parameter along with Data.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging and writing of data.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bIns">Type of chaining needs to be applied. One of the below values.
        ///                         <see cref="Chaining.NATIVE"/>
        ///                         <see cref="Chaining.ISO"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the data to be written.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="wCRLVer">CRLVersion is a 16-bit value encoding the current version of the CRLFile</param>
        /// <param name="aOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 need to be offset, then it will be 10 00 00.
        /// </param>
        /// <param name="aData">The data to be written to the PICC which includes Data followed by CRL Signature.</param>
        /// <param name="aLength">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 bytes need to be written, then it will be 10 00 00.
        ///
        ///                     Note:
        ///                         - The length will be Length of Data + Length of CRL Signature.
        ///                         - In this case if Data Length is 10 bytes and CRL Signature Length is 64 bytes, then
        ///                           \b aLength will be  (10 + 40) 00 00 => 50 00 00
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Chaining value (bIns).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "Both Application Chaining and ISO/IEC 14443-4 Chaining can be performed using the INS parameter information.",
                Message = "Writes data to Standard or Backup File configured as a CRL File."
            )
        ]
        public Status_t WriteData ( byte bOption, byte bIns, byte bFileNo, ushort wCRLVer, byte[] aOffset, byte[] aData,
            byte[] aLength )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            return phalMfDuoX_WriteData ( m_pDataParams, ( byte ) ( bOption | 0x01 ), bIns, bFileNo, wCRLVer,
                aOffset, aDataTmp, aLength );
        }

        /// <summary>
        /// Reads the currently stored value from value files.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging while retrieving Value information.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number from which the value to be retrieved.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// <param name="aValue">The value returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffer is null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = @"Communication mode for Command is handled inside library and for Response, required communication mode needs
                              to be provided to bOption parameter.",
                Message = "Reads the currently stored value from value files."
            )
        ]
        public Status_t GetValue ( byte bOption, byte bFileNo, out byte[] aValue )
        {
            Status_t oStatus;
            IntPtr ppValue = IntPtr.Zero;
            ushort wValueLen = 0;

            oStatus = phalMfDuoX_GetValue ( m_pDataParams, bOption, bFileNo, ref ppValue, ref wValueLen );
            aValue = MarshalCopy ( oStatus, ppValue, wValueLen );

            return oStatus;
        }

        /// <summary>
        /// Increases a value stored in a Value File.
        /// </summary>
        ///
        /// <param name="bOption"> Communication settings for the file.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the value should be credited.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aData">The value to be credited. Will be of 4 bytes with LSB first.
        ///                     If value 0x10 need to be credited, then it will be 10 00 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffer is null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = @"Communication mode for Response is handled inside library and for Command, required communication mode needs
                              to be provided to bOption parameter.",
                Message = "Increases a value stored in a Value File"
            )
        ]
        public Status_t Credit ( byte bOption, byte bFileNo, byte[] aData )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            return phalMfDuoX_Credit ( m_pDataParams, bOption, bFileNo, aDataTmp );
        }

        /// <summary>
        /// Decreases a value stored in a Value File.
        /// </summary>
        ///
        /// <param name="bOption"> Communication settings for the file.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the value should be debited.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aData">The value to be debited. Will be of 4 bytes with LSB first.
        ///                     If value 0x10 need to be debited, then it will be 10 00 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffer is null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = @"Communication mode for Response is handled inside library and for Command, required communication mode needs
                              to be provided to bOption parameter.",
                Message = "Decreases a value stored in a Value File"
            )
        ]
        public Status_t Debit ( byte bOption, byte bFileNo, byte[] aData )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            return phalMfDuoX_Debit ( m_pDataParams, bOption, bFileNo, aDataTmp );
        }

        /// <summary>
        /// Allows a limited increase of a value stored in a Value file without having
        /// full Cmd.Credit permissions to the file.
        /// </summary>
        ///
        /// <param name="bOption"> Communication settings for the file.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the value should be credited.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aData">The value to be credited. Will be of 4 bytes with LSB first.
        ///                     If value 0x10 need to be credited, then it will be 10 00 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffer is null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = @"Communication mode for Response is handled inside library and for Command, required communication mode needs
                              to be provided to bOption parameter.",
                Message = @"Allows a limited increase of a value stored in a Value file without having full Cmd.Credit permissions to the file."
            )
        ]
        public Status_t LimitedCredit ( byte bOption, byte bFileNo, byte[] aData )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            return phalMfDuoX_LimitedCredit ( m_pDataParams, bOption, bFileNo, aDataTmp );
        }

        /// <summary>
        /// Reads out a set of complete records from a Cyclic or Linear Record File.
        ///
        /// Note:
        ///     - Chaining upto the size of below mentioned buffers are handled within this interface. If more data is to be read,
        ///       the user has to call this function again with bOption = <see cref="ExchangeOptions.RXCHAINING"/> | one of the communication options.
        ///         - In case of ISO/IEX 14443-4 Chaining, HAL response buffer is utilized.The buffer can be updated during HAL initialization.
        ///         - In case of Native Chaining, Processing buffer is utilized.The buffer can be updated during the layer initialization.
        ///     - In either of the exchange options, its must to provide the communication mode also.
        ///     - If reading of data is performed using ISO / IEC 14443-4 chaining mode with wrapped enabled, make sure to disable
        ///       <see cref="Config.SHORT_LENGTH_APDU"/> configuration for data larger than frame size.
        ///     - If TMI collection is ON and
        ///         - If aRecCount is zero then aRecSize is a mandatory parameter.
        ///         - If aRecSize and aRecCount are zero, then <see cref="Error_Param.INVALID_PARAMETER"/> error will be returned.
        ///         - If wrong aRecSize is provided, then wrong aRecCount value will be calculated and updated for TMI collection.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing ofSecure Messaging and reading of data.
        ///                         Reading data.
        ///                             <see cref="ExchangeOptions.DEFAULT"/>   : Exchanges the command and received the Data.
        ///                             <see cref="ExchangeOptions.RXCHAINING"/>: To Receive remaining Data.
        ///
        ///                         Communication modes. To be ORed with above values
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bIns">Type of chaining needs to be applied. One of the below values.
        ///                         <see cref="Chaining.NATIVE"/>
        ///                         <see cref="Chaining.ISO"/>
        /// </param>
        /// <param name="bFileNo">The file number from where the data to be read.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aRecNo">Record number of the newest record targeted, starting to count
        ///                      from the latest record written. Will be of 3 bytes with LSB first.
        ///                      If 0x10 need to be record number, then it will be 10 00 00.
        /// </param>
        /// <param name="aRecCount">Number of records to be read. If 0x10 need to be record number, then it will be 10 00 00. </param>
        /// <param name="aRecSize">The number of bytes to be read. Will be of 3 bytes with LSB first.</param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Chaining value (bIns).
        ///          For Invalid Communication option value (bOption).
        ///          For Invalid Exchange option value (bOption).
        ///          If aRecSize and aRecCount are zero and TMI Collection is enabled.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Both Application Chaining and ISO/IEC 14443-4 Chaining can be performed using the INS parameter information.",
                Message = "Reads out a set of complete records from a Cyclic or Linear Record File."
            )
        ]
        public Status_t ReadRecords ( byte bOption, byte bIns, byte bFileNo, byte[] aRecNo, byte[] aRecCount, byte[] aRecSize, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_ReadRecords ( m_pDataParams, bOption, bIns, bFileNo, aRecNo, aRecCount, aRecSize, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Writes data to a record in a Cyclic or Linear Record File.
        ///
        /// Note:
        ///     - If writing of data is performed using ISO / IEC 14443-4 chaining mode with wrapped enabled, make sure to disable
        ///       <see cref="Config.SHORT_LENGTH_APDU"/> configuration for data larger than frame size.
        ///     - Implements chaining to the card. The data provided on aData will be chained to the card by sending data upto
        ///       the frame size of the MIFARE PICC, at a time.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging and writing of data.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bIns">Type of chaining needs to be applied. One of the below values.
        ///                         <see cref="Chaining.NATIVE"/>
        ///                         <see cref="Chaining.ISO"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the data to be written.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aOffset">The offset from where the data should be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 need to be offset, then it will be 10 00 00.
        /// </param>
        /// <param name="aData">The data to be written to the PICC.</param>
        /// <param name="aLength">The number of bytes to be written. Will be of 3 bytes with LSB first.
        ///                       If 0x10 bytes need to be written, then it will be 10 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Chaining value (bIns).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Both Application Chaining and ISO/IEC 14443-4 Chaining can be performed using the INS parameter information.",
                Message = "Writes data to standard data files or backup data files."
            )
        ]
        public Status_t WriteRecord ( byte bOption, byte bIns, byte bFileNo, byte[] aOffset, byte[] aData, byte[] aLength )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            return phalMfDuoX_WriteRecord ( m_pDataParams, bOption, bIns, bFileNo, aOffset, aDataTmp, aLength );
        }

        /// <summary>
        /// Updates data of an existing record in a LinearRecord or CyclicRecord file.
        ///
        /// Note:
        ///     - If updating of data is performed using ISO / IEC 14443-4 chaining mode with wrapped enabled, make sure to disable
        ///       <see cref="Config.SHORT_LENGTH_APDU"/> configuration for data larger than frame size.
        ///     - Implements chaining to the card. The data provided on aData will be chained to the card by sending data upto
        ///       the frame size of the MIFARE PICC, at a time.
        /// </summary>
        ///
        /// <param name="bOption">Options for processing of Secure Messaging and updating of data.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bIns">Type of chaining needs to be applied. One of the below values.
        ///                         <see cref="Chaining.NATIVE"/>
        ///                         <see cref="Chaining.ISO"/>
        /// </param>
        /// <param name="bFileNo">The file number to which the data to be updated.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        /// <param name="aRecNo">Record number of the newest record targeted, starting to count
        ///                      from the latest record updated. Will be of 3 bytes with LSB first.
        ///                      If 0x10 need to be record number, then it will be 10 00 00.
        /// </param>
        /// <param name="aOffset">The offset from where the data should be updated. Will be of 3 bytes with LSB first.
        ///                       If 0x10 need to be offset, then it will be 10 00 00.
        /// </param>
        /// <param name="aData">The data to be updated to the PICC.</param>
        /// <param name="aLength">The number of bytes to be updated. Will be of 3 bytes with LSB first.
        ///                       If 0x10 bytes need to be updated, then it will be 10 00 00.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          If the buffers are null.
        ///          For Invalid File Number (bFileNo).
        ///          For Invalid Chaining value (bIns).
        ///          For Invalid Communication option value (bOption).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Both Application Chaining and ISO/IEC 14443-4 Chaining can be performed using the INS parameter information.",
                Message = "Updates data of an existing record in a LinearRecord or CyclicRecord file."
            )
        ]
        public Status_t UpdateRecord ( byte bOption, byte bIns, byte bFileNo, byte[] aRecNo, byte[] aOffset, byte[] aData, byte[] aLength )
        {
            byte[] aDataTmp =  ( byte[] ) aData?.Clone ();
            return phalMfDuoX_UpdateRecord ( m_pDataParams, bOption, bIns, bFileNo, aRecNo, aOffset, aDataTmp, aLength );
        }

        /// <summary>
        /// Resets a Cyclic or Linear Record File.
        /// </summary>
        ///
        /// <param name="bFileNo">The file number to which the data to be written.
        ///                       ORed with <see cref="TargetApp.SECONDARY"/> to indicate secondary application indicator.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///          For Invalid File Number (bFileNo).
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Both Application Chaining and ISO/IEC 14443-4 Chaining can be performed using the INS parameter information.",
                Message = "Resets a Cyclic or Linear Record File."
            )
        ]
        public Status_t ClearRecordFile ( byte bFileNo )
        {
            return phalMfDuoX_ClearRecordFile ( m_pDataParams, bFileNo );
        }
        #endregion

        #region Transaction Management
        /// <summary>
        /// Validates all previous write access on Backup Data files, Value files and Record files within selected
        /// application. If applicable, the TransactionMAC file is updated with the calculated Transaction MAC or
        /// Transaction Signature.
        ///
        /// Note:
        ///     With respect to command parameter bOption, PICC expects
        ///         No Option byte in command frame.
        ///         Option byte with zero as value in command frame along with command code.
        ///         Option byte with one as value in command frame along with command code.
        ///
        ///     If Option byte is required to be exchanged to PICC along with command code, user needs to pass bOption
        ///     value with MSB set.
        ///         If bOption = <see cref="Option.NOT_EXCHANGED"/>, only command is exchanged and Option byte is not
        ///         exchanged to PICC.
        ///
        ///         If bOption = other than <see cref="Option.NOT_EXCHANGED"/>,
        ///             Both command and option bytes are exchanged to PICC.
        ///             While exchanging the option byte to PICC, the MSB bit is masked out.
        /// </summary>
        ///
        /// <param name="bOption">Calculated Transaction MAC/Signature requested on response.
        ///                       Should be one of the below values.
        ///                         <see cref="Option.NOT_EXCHANGED"/>
        ///                         <see cref="Option.TRANSACTION_INFO_NOT_RETURNED"/>
        ///                         <see cref="Option.TRANSACTION_INFO_RETURNED"/>
        /// </param>
        /// <param name="aTMC">The increased Transaction MAC Counter (TMC) as stored in FileType.TransactionMAC.</param>
        /// <param name="aResponse">Returns one of the following information based on the functionality that is enabled
        ///                         during TransactionMAC File creation.
        ///                             If Transaction MAC is Enabled, Transaction MAC Value ( TMV) will be available.
        ///                             If Transaction Signature is Enabled, Transaction Signature Value ( TSV) will be available.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> For Invalid option value. (bOption).
        ///     Returns <see cref="Error_Comm.PROTOCOL_ERROR"/>
        ///         If the response is less than 8 bytes for the case when Return of TMC
        ///         and TMV / TSV is selected.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Validates all previous write access on Backup Data files, Value files and Record files within selected
                            application. If applicable, the TransactionMAC file is updated with the calculated Transaction MAC or
                            Transaction Signature."
            )
        ]
        public Status_t CommitTransaction ( byte bOption, out byte[] aTMC, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppTMC = IntPtr.Zero;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wTMCLen = 0;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_CommitTransaction ( m_pDataParams, bOption, ref ppTMC, ref wTMCLen, ref ppResponse, ref wRspLen );
            aTMC = MarshalCopy ( oStatus, ppTMC, wTMCLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Aborts all previous write accesses on Backup Data files, Value files and Record files within the selected
        /// application ( s). If applicable, the Transaction MAC calculation is aborted.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Aborts all previous write accesses on Backup Data files, Value files and Record files within the selected
                            application ( s). If applicable, the Transaction MAC calculation is aborted."
            )
        ]
        public Status_t AbortTransaction ()
        {
            return phalMfDuoX_AbortTransaction ( m_pDataParams );
        }

        /// <summary>
        /// Commits a ReaderID for the ongoing transaction. This will allow a back-end to identify the
        /// attacking merchant in case of fraud detected.
        /// </summary>
        ///
        /// <param name="aTMRI">Transaction MAC ReaderID information.</param>
        /// <param name="aEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffer is null.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                Message = @"Commits a ReaderID for the ongoing transaction. This will allow a back-end to identify the
                            attacking merchant in case of fraud detected."
            )
        ]
        public Status_t CommitReaderID ( byte[] aTMRI, out byte[] aEncTMRI )
        {
            Status_t oStatus;
            IntPtr ppEncTMRI = IntPtr.Zero;
            ushort wEncTMRILen = 0;

            oStatus = phalMfDuoX_CommitReaderID ( m_pDataParams, aTMRI, ( byte ) ( ( aTMRI == null ) ? 0 : aTMRI.Length ),
                ref ppEncTMRI, ref wEncTMRILen );
            aEncTMRI = MarshalCopy ( oStatus, ppEncTMRI, wEncTMRILen );

            return oStatus;
        }
        #endregion

        #region Cryptographic Support
        /// <summary>
        /// Executes a cryptographic operation. This is the generic API definition, including common error codes
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bAction">Targeted action to perform. One of the below values.
        ///                         <see cref="Action.ECC_SIGN"/>
        ///                         <see cref="Action.ECHO"/>
        /// </param>
        /// <param name="aInputData">Input data for which the cryptographic operation needs to be performed.</param>
        /// <param name="aResponse">Cryptographic output for the provided input.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         The values provided in bComOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = @"Executes a cryptographic operation. This is the generic API definition, including common error codes"
            )
        ]
        public Status_t CryptoRequest ( byte bComOption, byte bAction, byte[] aInputData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_CryptoRequest ( m_pDataParams, bComOption, bAction, aInputData, ( byte ) ( ( aInputData == null ) ?
                0 : aInputData.Length ), ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Executes an ECC signature generation
        /// </summary>
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bOperation">Target Operation to use. One of the below values.
        ///                             <see cref="Operation.INITIALIZE_SIGNATURE"/>
        ///                             <see cref="Operation.UPDATE_DATA"/>
        ///                             <see cref="Operation.FINALIZE_DATA"/>
        ///                             <see cref="Operation.ONE_SHOT_RAW_DATA"/>
        ///                             <see cref="Operation.ONE_SHOT_HASH_DATA"/>
        /// </param>
        /// <param name="bAlgo">Target Algorithm to use. One of the below values
        ///                         <see cref="TargetAlgo.ECDSA_SIGN"/>
        /// </param>
        /// <param name="bKeyNo">Key number of the targeted key. One of the following
        ///                         At PICC Level: 0x00 - 0x01 keys are supported.
        ///                         At APP Level : 0x00 - 0x04 keys are supported.
        /// </param>
        /// <param name="bInputSource">Input Source to use. One of the below values.
        ///                             <see cref="InputSource.COMMAND_BUFFER_EXPLICIT_LEN"/>
        /// </param>
        /// <param name="aInputData">Input data to be signed. Can be one of the following.
        ///                             <see cref="Operation.ONE_SHOT_RAW_DATA"/>
        ///                             <see cref="Operation.ONE_SHOT_HASH_DATA"/>
        /// </param>
        /// <param name="aSign">ECDSA Signature for the provided input.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         If the InputType is not supported (bInputType).
        ///         The values provided in bComOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = @"Executes an ECC signature generation"
            )
        ]
        public Status_t CryptoRequest_ECCSign ( byte bComOption, byte bOperation, byte bAlgo, byte bKeyNo, byte bInputSource,
            byte[] aInputData, out byte[] aSign )
        {
            Status_t oStatus;
            IntPtr ppSign = IntPtr.Zero;
            ushort wSignLen = 0;

            oStatus = phalMfDuoX_CryptoRequestECCSign ( m_pDataParams, bComOption, bOperation, bAlgo, bKeyNo, bInputSource,
                aInputData, ( byte ) ( ( aInputData == null ) ? 0 : aInputData.Length ), ref ppSign, ref wSignLen );
            aSign = MarshalCopy ( oStatus, ppSign, wSignLen );

            return oStatus;
        }

        /// <summary>
        /// Performs echoing of the data being transmitted. This allows to easily test the communication interface.
        /// </summary>
        ///
        ///
        /// <param name="bComOption">Indicates the mode of communication to be used while exchanging the data to PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="aInputData">Input data to be echoed</param>
        /// <param name="aResponse">Cryptographic output for the provided input.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         The values provided in bComOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = @"Executes a cryptographic operation to Echo the transmitted data.",
                OtherInfo = "This allows to easily test the communication interface."
            )
        ]
        public Status_t CryptoRequestEcho ( byte bComOption, byte[] aInputData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_CryptoRequestEcho ( m_pDataParams, bComOption, aInputData, ( byte ) ( ( aInputData == null ) ?
                0 : aInputData.Length ), ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region GPIO Management
        /// <summary>
        /// Perform GPIO Management
        /// </summary>
        ///
        /// <param name="wOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="bGPIONo">GPIO Number to use for management. One of the below values.
        ///                         <see cref="GPIONo.GPIO_1"/>
        ///                         <see cref="GPIONo.GPIO_2"/>
        /// </param>
        /// <param name="bOperation">Targeted Operation perform on the selected GPIO. One of the below values.
        ///                             - GPIOxMode as Output
        ///                                 - <see cref="GPIO_Operation.Output.GPIO_CONTROL_CLEAR"/>
        ///                                 - <see cref="GPIO_Operation.Output.GPIO_CONTROL_SET"/>
        ///                                 - <see cref="GPIO_Operation.Output.GPIO_CONTROL_TOGGLE"/>
        ///                                 - <see cref="GPIO_Operation.Output.PAUSE_RELEASE_NFC"/>
        ///
        ///                             - GPIOxMode as Down-Stream Power Out
        ///                                 - GPIO Control
        ///                                     -<see cref="GPIO_Operation.DownStreamPowerOut.ENABLE_POWER_HARVEST"/>
        ///
        ///                                 - GPIO Measurement Control
        ///                                     -<see cref="GPIO_Operation.DownStreamPowerOut.EXECUTE_MEASUREMENT"/>
        ///
        ///                                 - Target Voltage / Current Level
        ///                                     - Power downstream voltage of 1.8V
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_100uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_300uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_500uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_1mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_2mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_3mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_5mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_7mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_10mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_1_8.CURRENT_MAX_AVAILABLE"/>
        ///                                     - Power downstream voltage of 2V
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_100uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_300uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_500uA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_1mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_2mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_3mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_5mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_7mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_10mA"/>
        ///                                         -<see cref="GPIO_Operation.DownStreamPowerOut.Voltage_2.CURRENT_MAX_AVAILABLE"/>
        /// </param>
        /// <param name="aNFCPauseRspData">NFC Pause Response Data: data to be returned to NFC host in the case of
        ///                               Release NFC Pause
        /// </param>
        /// <param name="aResponse">Response from PICC as follows.
        ///                             - If \b bOperation = #PHAL_MFECC_OPERATION_TOGGLE_PAUSE_NFC then,
        ///                               NFC Pause Response Data: data received from the I2C interface
        ///                             - NULL for others.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         If the InputType is not supported (bInputType).
        ///         The values provided in wOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = @"Perform GPIO Management"
            )
        ]
        public Status_t ManageGPIO ( ushort wOption, byte bGPIONo, byte bOperation, byte[] aNFCPauseRspData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_ManageGPIO ( m_pDataParams, wOption, bGPIONo, bOperation, aNFCPauseRspData,
                ( ushort ) ( ( aNFCPauseRspData == null ) ? 0 : aNFCPauseRspData.Length ), ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform GPIO read.
        /// </summary>
        ///
        /// <param name="wOption">Indicates the mode of communication to be used while exchanging the data from PICC.
        ///                             <see cref="CommOption.PLAIN"/>
        ///                             <see cref="CommOption.MAC"/>
        ///                             <see cref="CommOption.FULL"/>
        /// </param>
        /// <param name="aResponse"></param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffer is null.
        ///         If the InputType is not supported (bInputType).
        ///         The values provided in wOption is not supported.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = @"Perform GPIO read."
            )
        ]
        public Status_t ReadGPIO ( ushort wOption, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0;

            oStatus = phalMfDuoX_ReadGPIO ( m_pDataParams, wOption, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

        #region ISO7816-4
        /// <summary>
        /// Perform File or Application selection. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///                             <see cref="FCI.NOT_RETURNED"/>
        ///                             <see cref="FCI.RETURNED"/>
        /// </param>
        /// <param name="bSelector">The selector to be used.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_PARENT_DF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_BY_DF_NAME"/>
        /// </param>
        /// <param name="aFid">The ISO File number to be selected.
        ///                     Valid only if bSelector is one of the following.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        /// </param>
        /// <param name="aDFname">The ISO DFName to be selected.
        ///                         Valid only when bSelector = <see cref="Selector.SELECT_BY_DF_NAME"/>.
        ///                         NULL for other bSelector options.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="aFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         DFName Length is greater than 16 (aDFname).
        ///         Invalid FCI ( File Control Identifier) (bOption).
        ///         Invalid Selector option (bSelector).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Select an application or file."
            )
        ]
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] aFid, byte[] aDFname, byte bExtendedLenApdu,
            out byte[] aFCI )
        {
            Status_t oStatus;
            IntPtr ppFCI = IntPtr.Zero;
            ushort wFCILen = 0;

            oStatus = phalMfDuoX_IsoSelectFile ( m_pDataParams, bOption, bSelector, aFid, aDFname,
                ( byte ) ( ( aDFname == null ) ? 0 : aDFname.Length ), bExtendedLenApdu,
                ref ppFCI, ref wFCILen );

            aFCI = MarshalCopy ( oStatus, ppFCI, wFCILen );

            return oStatus;
        }

        /// <summary>
        /// ISO Select. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOption">Option for return / no return of FCI.
        ///                             <see cref="FCI.NOT_RETURNED"/>
        ///                             <see cref="FCI.RETURNED"/>
        /// </param>
        /// <param name="bSelector">The selector to be used.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_PARENT_DF_CURRENT_DF"/>
        ///                             <see cref="Selector.SELECT_BY_DF_NAME"/>
        /// </param>
        /// <param name="aFid">The ISO File number to be selected.
        ///                     Valid only if bSelector is one of the following.
        ///                             <see cref="Selector.SELECT_MF_DF_EF_FILE_IDENTIFIER"/>
        ///                             <see cref="Selector.SELECT_CHILD_DF"/>
        ///                             <see cref="Selector.SELECT_EF_CURRENT_DF"/>
        /// </param>
        /// <param name="aDFname">The ISO DFName to be selected.
        ///                         Valid only when bSelector = <see cref="Selector.SELECT_BY_DF_NAME"/>.
        ///                         NULL for other bSelector options.
        /// </param>
        /// <param name="pFCI">The FCI information returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         DFName Length is greater than 16 (aDFname).
        ///         Invalid FCI ( File Control Identifier) (bOption).
        ///         Invalid Selector option (bSelector).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is not supported",
                Message = "Select an application or file."
            )
        ]
        public Status_t IsoSelectFile ( byte bOption, byte bSelector, byte[] aFid, byte[] aDFname, out byte[] aFCI )
        {
            return IsoSelectFile ( bOption, bSelector, aFid, aDFname, 0x00, out aFCI );
        }

        /// <summary>
        /// Perform ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : To exchange command to the PICC and receive the response.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive pending response from PICC.
        /// </param>
        /// <param name="bOffset">The offset from where the data should be read.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="dwBytesToRead">The number of bytes to be read from the file.
        ///                                 - If zero is provided, then entire file data is returned by PICC.
        ///                                 - If non-zero is provided, then data starting from offset is returned.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///         For Invalid Buffering Options (wOption).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Read from a data file."
            )
        ]
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, uint dwBytesToRead, byte bExtendedLenApdu,
            out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0x00;

            oStatus = phalMfDuoX_IsoReadBinary ( m_pDataParams, wOption, bOffset, bSfid, dwBytesToRead, bExtendedLenApdu,
                ref ppResponse, ref wRspLen );

            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform ISO Read Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : To exchange command to the PICC and receive the response.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive pending response from PICC.
        /// </param>
        /// <param name="bOffset">The offset from where the data should be read.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="dwBytesToRead">The number of bytes to be read from the file.
        ///                                 - If zero is provided, then entire file data is returned by PICC.
        ///                                 - If non-zero is provided, then data starting from offset is returned.
        /// </param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///         For Invalid Buffering Options (wOption).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is not supported",
                Message = "Read from a data file."
            )
        ]
        public Status_t IsoReadBinary ( ushort wOption, byte bOffset, byte bSfid, uint dwBytesToRead, out byte[] aResponse )
        {
            return IsoReadBinary ( wOption, bOffset, bSfid, dwBytesToRead, 0, out aResponse );
        }

        /// <summary>
        /// Perform ISO Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Write to a data file."
            )
        ]
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte bExtendedLenApdu, byte[] aData )
        {
            return phalMfDuoX_IsoUpdateBinary ( m_pDataParams, bOffset, bSfid, bExtendedLenApdu, aData,
                 ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Perform ISO Update Binary. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOffset">The offset from where the data should be updated.
        ///                       Regardless of bSfid value, the encoding of offset will be from 0 - 255.
        ///                       This will be part of P2 information.
        /// </param>
        /// <param name="bSfid">Indication to use either Short ISO File Id or Offset.
        ///                         - If <see cref="SFID.ENABLED"/>, then bit 7 is set and bits 0-4
        ///                           indicates short file identifier.
        ///                         - If <see cref="SFID.DISABLED"/>, then bits 0-6 indicates MSB of
        ///                           offset information.
        ///                         - This will be part of P1 information.
        ///                         - Ex.If actual Offset = 8063 ( 1F7F ), then bSfid will be 1F and bOffset will be 7F.
        /// </param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Write to a data file."
            )
        ]
        public Status_t IsoUpdateBinary ( byte bOffset, byte bSfid, byte[] aData )
        {
            return IsoUpdateBinary ( bOffset, bSfid, 0x00, aData );
        }

        /// <summary>
        /// Perform ISO Read Record. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : To exchange command to the PICC and receive the response.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive pending response from PICC.
        /// </param>
        /// <param name="bRecNo">Record to read / from where to read.</param>
        /// <param name="bReadAllRecords">Whether to read all records from P1 or just one.
        ///                                 <see cref="RecordUsage.SINGLE"/>
        ///                                 <see cref="RecordUsage.ALL"/>
        /// </param>
        /// <param name="bSfid">Indication to use Short ISO File Id.
        ///                     File Identifiers from 0x00 - 0x1F.
        /// </param>
        /// <param name="dwBytesToRead">The number of bytes to be read from the file.
        ///                                 - If zero is provided, then entire file starting from the record specified.
        ///                                 - If non-zero is provided, then
        ///                                     - The number of bytes to be read.
        ///                                     - If bigger than number of bytes available in the file, after subtracting
        ///                                       MAC length if MAC is to be returned, the entire data file starting from
        ///                                       the offset position is returned. If smaller, this number of bytes is
        ///                                       returned ( possibly containing partial record ).
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///         For Invalid Buffering Options (wOption).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Read from a record file."
            )
        ]
        public Status_t IsoReadRecords ( ushort wOption, byte bRecNo, byte bReadAllRecords, byte bSfid, uint dwBytesToRead,
            byte bExtendedLenApdu, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0x00;

            oStatus = phalMfDuoX_IsoReadRecords ( m_pDataParams, wOption, bRecNo, bReadAllRecords, bSfid, dwBytesToRead,
                bExtendedLenApdu, ref ppResponse, ref wRspLen );

            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Perform ISO Read Record. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : To exchange command to the PICC and receive the response.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive pending response from PICC.
        /// </param>
        /// <param name="bRecNo">Record to read / from where to read.</param>
        /// <param name="bReadAllRecords">Whether to read all records from P1 or just one.
        ///                                 <see cref="RecordUsage.SINGLE"/>
        ///                                 <see cref="RecordUsage.ALL"/>
        /// </param>
        /// <param name="bSfid">Indication to use Short ISO File Id.
        ///                     File Identifiers from 0x00 - 0x1F.
        /// </param>
        /// <param name="dwBytesToRead">The number of bytes to be read from the file.
        ///                                 - If zero is provided, then entire file starting from the record specified.
        ///                                 - If non-zero is provided, then
        ///                                     - The number of bytes to be read.
        ///                                     - If bigger than number of bytes available in the file, after subtracting
        ///                                       MAC length if MAC is to be returned, the entire data file starting from
        ///                                       the offset position is returned. If smaller, this number of bytes is
        ///                                       returned ( possibly containing partial record ).
        /// </param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///         For Invalid Buffering Options (wOption).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Read from a record file."
            )
        ]
        public Status_t IsoReadRecords ( ushort wOption, byte bRecNo, byte bReadAllRecords, byte bSfid, uint dwBytesToRead,
            out byte[] aResponse )
        {
            return IsoReadRecords ( wOption, bRecNo, bReadAllRecords, bSfid, dwBytesToRead, 0x00, out aResponse );
        }

        /// <summary>
        /// Perform ISO Append record. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bSfid">Indication to use Short ISO File Id.
        ///                     File Identifiers from 0x00 - 0x1F.
        /// </param>
        /// <param name="pData">Data to be updated.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Write a new record to a record file."
            )
        ]
        public Status_t IsoAppendRecord ( byte bSfid, byte bExtendedLenApdu, byte[] aData )
        {
            return phalMfDuoX_IsoAppendRecord ( m_pDataParams, bSfid, bExtendedLenApdu, aData,
                 ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Perform ISO Append record. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bSfid">Indication to use Short ISO File Id.
        ///                     File Identifiers from 0x00 - 0x1F.
        /// </param>
        /// <param name="pData">Data to be updated.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For invalid Short File identifier (bSfid).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Write to a data file."
            )
        ]
        public Status_t IsoAppendRecord ( byte bSfid, byte[] aData )
        {
            return IsoAppendRecord ( bSfid, 0x00, aData );
        }

        /// <summary>
        /// Perform ISOGetChallenge. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bExpRsp">Length of expected challenge RPICC1.</param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where
        ///                                                                        LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where
        ///                                                                        LC and LE are of 3 byte.
        /// </param>
        /// <param name="pRPICC1">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is supported",
                Message = "Get a challenge (first part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoGetChallenge ( byte bExpRsp, byte bExtendedLenApdu, out byte[] aRPICC1 )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0x00;

            oStatus = phalMfDuoX_IsoGetChallenge ( m_pDataParams, bExpRsp, bExtendedLenApdu, ref ppResponse, ref wRspLen );

            if ( ( ppResponse != IntPtr.Zero ) && ( wRspLen != 0 ) )
            {
                aRPICC1 = new byte[wRspLen];
                Marshal.Copy ( ppResponse, aRPICC1, 0, ( int ) wRspLen );
            }
            else
            {
                aRPICC1 = null;
            }

            return oStatus;
        }

        /// <summary>
        /// Perform ISOGetChallenge. This command is implemented in compliance with ISO/IEC 7816-4.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bExpRsp">Length of expected challenge RPICC1.</param>
        /// <param name="pRPICC1">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "Extended Length APDU is not supported",
                Message = "Get a challenge (first part of ISO7816-4 authentication)."
            )
        ]
        public Status_t IsoGetChallenge ( byte bExpRsp, out byte[] aRPICC1 )
        {
            return IsoGetChallenge ( bExpRsp, 0x00, out aRPICC1 );
        }
        #endregion

        #region EV Charging
        /// <summary>
        /// Reads data from Standard data File.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wOption">One of the below options.
        ///                         <see cref="ExchangeOptions.DEFAULT"/>   : To exchange command to the PICC and receive the response.
        ///                         <see cref="ExchangeOptions.RXCHAINING"/>: To Receive pending response from PICC.
        /// </param>
        /// <param name="bFileNo">The file number from where the data to be read.</param>
        /// <param name="wBytesToRead">The number of bytes to be read from the file.
        ///                             - If zero, any amount of data stating from zero upto 256 / 65536 bytes.
        ///                               LE will be exchanged as zero based on \b bExtendedLenApdu value.
        ///                             - If non zero, any amount of data stating from zero upto \b wBytesToRead will be returned.
        ///                               LE will be exchanged as \b wBytesToRead based on \b bExtendedLenApdu value.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where LE is of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where LE is of 3 byte.
        /// </param>
        /// <param name="aResponse">The data returned by the PICC.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Gen.SUCCESS_CHAINING"/> Indicating more data to be read.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> for the below ones.
        ///         If the buffers are null.
        ///         For Invalid Buffering Options (wOption).
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = "Reads data from Standard data File."
            )
        ]
        public Status_t VdeReadData ( ushort wOption, byte bFileNo, ushort wBytesToRead, byte bExtendedLenApdu, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0x00;

            oStatus = phalMfDuoX_VdeReadData ( m_pDataParams, wOption, bFileNo, wBytesToRead, bExtendedLenApdu, ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }

        /// <summary>
        /// Writes data to Standard data File and eventually lock the file.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="bOperation">The operation to perform on the file.
        ///                             - <see cref="EV_Operation.WRITE"/>: Perform Write operation on the file.
        ///                             - <see cref="EV_Operation.LOCK"/>: Perform Lock operation on the file.
        /// </param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where LC and LE are of 3 byte.
        /// </param>
        /// <param name="aData">The data to be written to the PICC</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = "Writes data to Standard data File and eventually lock the file."
            )
        ]
        public Status_t VdeWriteData ( ushort bOperation, byte bExtendedLenApdu, byte[] aData )
        {
            return phalMfDuoX_VdeWriteData ( m_pDataParams, bOperation, bExtendedLenApdu, aData, ( ushort ) ( ( aData == null ) ? 0 : aData.Length ) );
        }

        /// <summary>
        /// Generates and ECDSA signature over a 32-byte challenge.
        ///
        /// Note
        ///     For all ISO7816 errors, library returns a command error code <see cref="Error.DF_7816_GEN_ERROR"/>. To know
        ///     the exact error returned by PICC call <see cref="SetConfig"/> with Configuration as
        ///     <see cref="Config.ADDITIONAL_INFO"/>.
        /// </summary>
        ///
        /// <param name="wBytesToRead"></param>
        /// <param name="bExtendedLenApdu">Flag for Extended Length APDU.
        ///                                 <see cref="ApduFormat.SHORT_LEN"/>   : Short Length APDU is used where LC and LE are of 1 byte.
        ///                                 <see cref="ApduFormat.EXTENDED_LEN"/>: Extended Length APDU is used where LC and LE are of 3 byte.
        /// </param>
        /// <param name="aData">Message to be signed.</param>
        /// <param name="aResponse">The signature of the message.
        ///                             - The Signature will be in R and S integer format. \b ppResponse = R data followed by S data.
        ///                             - Here R and S length will be based on the curve length.
        ///                             - Ex: If curve length is 256 bit, then R and S length will be 32 bytes each.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="Error.DF_7816_GEN_ERROR"/> for standard ISO7816 - 4 PICC errors.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = "Generates and ECDSA signature over a 32-byte challenge."
            )
        ]
        public Status_t VdeECDSASign ( ushort wBytesToRead, byte bExtendedLenApdu, byte[] aData, out byte[] aResponse )
        {
            Status_t oStatus;
            IntPtr ppResponse = IntPtr.Zero;
            ushort wRspLen = 0x00;

            oStatus = phalMfDuoX_VdeECDSASign ( m_pDataParams, wBytesToRead, bExtendedLenApdu, aData, ( ushort ) ( ( aData == null ) ? 0 : aData.Length ),
                ref ppResponse, ref wRspLen );
            aResponse = MarshalCopy ( oStatus, ppResponse, wRspLen );

            return oStatus;
        }
        #endregion

#if !PACKAGE_PUBLIC
        #region Post-Delivery Configuration
        /// <summary>
        /// Perform Post Delivery Authentication.
        /// </summary>
        ///
        /// <param name="wKeyNoCard">PICC Key number to be used for authentication.</param>
        /// <param name="wKeyNo">Key number to used from software.</param>
        /// <param name="wKeyVer">Key version to used from software.</param>
        /// <param name="bUpgradeInfo">The upgrade info input.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     XXXX
        ///         Depending on status codes return by PICC.
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                Message = "Perform Post Delivery Authentication."
            )
        ]
        public Status_t AuthenticatePDC ( ushort wKeyNoCard, ushort wKeyNo, ushort wKeyVer, byte bUpgradeInfo )
        {
            return phalMfDuoX_AuthenticatePDC ( m_pDataParams, wKeyNoCard, wKeyNo, wKeyVer, bUpgradeInfo );
        }
        #endregion
#endif

        #region Utility
        /// <summary>
        /// Perform a GetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to read. Will be one of the below values.
        ///                       Refer <see cref="Config"/> for supported configuration items.</param>
        /// <param name="pValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success Status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Get the configuration like WrappedMode, Force usage of ShortLengthAPDU and to retrieve additional information."
            )
        ]
        public Status_t GetConfig ( ushort wConfig, ref ushort pValue )
        {
            return phalMfDuoX_GetConfig ( m_pDataParams, wConfig, ref pValue );
        }

        /// <summary>
        /// Perform a SetConfig command.
        /// </summary>
        ///
        /// <param name="wConfig">Configuration to set. Will be one of the below values.
        ///                       Refer <see cref="Config"/> for supported configuration items.</param>
        /// <param name="wValue">The value for the mentioned configuration.</param>
        ///
        /// <returns>Returns Success Status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Set the configuration like WrappedMode, Force usage of ShortLengthAPDU and to update additional information."
            )
        ]
        public Status_t SetConfig ( ushort wConfig, ushort wValue )
        {
            return phalMfDuoX_SetConfig ( m_pDataParams, wConfig, wValue );
        }

        /// <summary>
        /// Reset the authentication
        /// </summary>
        ///
        /// <returns>Returns Success Status for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                MIFARE_Products.DESFIRE_EV3,
                MIFARE_Products.DESFIRE_EV2,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Reset the authentication state. Clears existing session keys and set the state to not authenticated."
            )
        ]
        public Status_t ResetAuthentication ()
        {
            return phalMfDuoX_ResetAuthentication ( m_pDataParams );
        }

#if !PACKAGE_PUBLIC
        /// <summary>
        /// Performs verification of Device side Certificate contents or Message B ( Msg.B.ext). This interface needs to be called after performing
        /// <see cref="ISOGeneralAuthenticate"/> or <see cref="ISOGeneralAuthenticate_Final"/> if Authentication type is Mutual Authentication.
        ///
        /// Note: This is not an actual command of MIFARE DUOX. Its an utility interface to verify message.
        ///       There is no exchange performed with PICC.
        /// </summary>
        ///
        /// <param name="aPubBKey">Device Public Key (Pub.B) to be used for verification.</param>
        /// <param name="aOptsB">Options received from the PICC.</param>
        /// <param name="aE_PubBKey">Device Ephemeral Public Key (E.Pub.B) received from PICC</param>
        /// <param name="aSignature">The signature of the message. The Signature should be in R and S integer format.
        ///                          Format should be aSignature = R data followed by S data.
        ///                          Here R and S length should be based on the curve length.
        ///                          Ex: If curve length is 256 bit, then R and S length will be 32 bytes each.
        /// </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="CryptoASym.Error.VERIFICATION_FAILED"/> Verification of Message / Signature combination failed.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Performs verification of Message B (Msg.B.ext)."
            )
        ]
        public Status_t ISOGeneralAuthenticate_Verify ( byte[] aPubBKey, byte[] aOptsB, byte[] aE_PubBKey, byte[] aSignature )
        {
            return phalMfDuoX_ISOGeneralAuthenticateVerify ( m_pDataParams, aPubBKey, ( ushort ) ( ( aPubBKey == null ) ? 0 : aPubBKey.Length ),
                aOptsB, ( byte ) ( ( aOptsB == null ) ? 0 : aOptsB.Length ), aE_PubBKey, ( ushort ) ( ( aE_PubBKey == null ) ? 0 : aE_PubBKey.Length ),
                aSignature, ( ushort ) ( ( aSignature == null ) ? 0 : aSignature.Length ) );
        }

        /// <summary>
        /// Generates Encrypted Key for Delegated application management.
        /// </summary>
        ///
        /// <param name="wKeyNo_PICCDAMEncKey">Key number in key store of PICC DAM Encryption key.</param>
        /// <param name="wKeyVer_PICCDAMEncKey">Key version in key store of PICC DAM Encryption key.</param>
        /// <param name="wKeyNo_AppDAMDefault">Key number in key store of APP DAM default key.</param>
        /// <param name="wKeyVer_AppDAMDefault">Key version in key store of APP DAM default key.</param>
        /// <param name="bAppDAMDefaultKeyVer">APP DAM Default Key version.</param>
        /// <param name="aDAMEncKey">The initial application key for the DAM application.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffer is null.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is of not a support key type.
        ///     Returns <see cref="CryptoASym.Error.VERIFICATION_FAILED"/> Verification of Message / Signature combination failed.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Generates Encrypted Key for Delegated application management."
            )
        ]
        public Status_t GenerateDAMEncKey ( ushort wKeyNo_PICCDAMEncKey, ushort wKeyVer_PICCDAMEncKey, ushort wKeyNo_AppDAMDefault, ushort wKeyVer_AppDAMDefault,
            byte bAppDAMDefaultKeyVer, out byte[] aDAMEncKey )
        {
            Status_t oStatus;
            IntPtr ppDAMEncKey = IntPtr.Zero;
            ushort wDAMEncKeyLen = 0;

            oStatus = phalMfDuoX_GenerateDAMEncKey ( m_pDataParams, wKeyNo_PICCDAMEncKey, wKeyVer_PICCDAMEncKey, wKeyNo_AppDAMDefault, wKeyVer_AppDAMDefault,
                bAppDAMDefaultKeyVer, ref ppDAMEncKey, ref wDAMEncKeyLen );
            aDAMEncKey = MarshalCopy ( oStatus, ppDAMEncKey, wDAMEncKeyLen );

            return oStatus;
        }

        /// <summary>
        /// Generate MAC for Delegated Application Creation and Delete Application.
        /// </summary>
        ///
        /// <param name="bOption">Option to represent the
        ///                         Presence of ISO information.
        ///                             <see cref="ISOFileInfo.NOT_AVAILABLE"/>
        ///                             <see cref="ISOFileInfo.ISO_FILE_ID_AVAILABLE"/>
        ///                             <see cref="ISOFileInfo.ISO_DF_NAME_AVAILABLE"/>
        ///                             <see cref="ISOFileInfo.ISO_FILE_ID_DF_NAME_AVAILABLE"/>
        ///
        ///                         To be ORed with the above ones.
        ///                             <see cref="DAMMAC.CREATE_DELEGATED_APPLICATION"/>
        ///                             <see cref="DAMMAC.DELETE_APPLICATION"/>
        /// </param>
        /// <param name="wKeyNo_PICCDAMMAC">Key number in key store of PICC DAM MAC key.</param>
        /// <param name="wKeyVer_PICCDAMMAC">Key version in key store of PICC DAM MAC key.</param>
        /// <param name="aAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///                    If application 01 need to be created, then the Aid will be 01 00 00.
        /// </param>
        /// <param name="aDamParams">The DAM information. Should as mentioned below.
        ///                             Byte0 = DAM Slot No[LSB]
        ///                             Byte1 = DAM Slot No[MSB]
        ///                             Byte2 = DAM Slot Version
        ///                             Byte3 = Quota Limit [LSB]
        ///                             Byte4 = Quota Limit [MSB]
        /// </param>
        /// <param name="bKeySettings1">Application Key settings.</param>
        /// <param name="bKeySettings2">Several other key settings.</param>
        /// <param name="bKeySettings3">Additional optional key settings.</param>
        /// <param name="aKeySetValues">The Key set values for the application. Should as mentioned below.
        ///                                 Byte0 = AKS version
        ///                                 Byte1 = Number of KeySets
        ///                                 Byte2 = MaxKeysize
        ///                                 Byte3 = Application KeySet Settings.
        /// </param>
        /// <param name="aISOFileId">ISO File ID to be used. Should be two bytes.</param>
        /// <param name="aISODFName">ISO DF Name to be used. Should be upto 16 bytes and can also be NULL based on the option.
        ///                             If bOption = <see cref="ISOFileInfo.ISO_DF_NAME_AVAILABLE"/>, then upto 16 bytes of ISO application
        ///                             name should be provided.
        ///                             NULL for other option types.
        /// </param>
        /// <param name="aEncK">The initial application key for the DAM application.</param>
        /// <param name="aDAMMAC">The MAC calculated by the card issuer with PICCDAMMACKey to allow delegated application creation.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         - If the buffers are null.
        ///         - For Invalid options information. (bOption)
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is of not a support key type.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Generate MAC for Delegated application and Delete Application."
            )
        ]
        public Status_t GenerateDAMMAC ( byte bOption, ushort wKeyNo_PICCDAMMAC, ushort wKeyVer_PICCDAMMAC, byte[] aAid, byte[] aDamParams, byte bKeySettings1,
            byte bKeySettings2, byte bKeySettings3, byte[] aKeySetValues, byte[] aISOFileId, byte[] aISODFName, byte[] aEncK, out byte[] aDAMMAC )
        {
            Status_t oStatus;
            IntPtr ppDAMMAC = IntPtr.Zero;
            ushort wDAMMACKeyLen = 0;

            oStatus = phalMfDuoX_GenerateDAMMAC ( m_pDataParams, bOption, wKeyNo_PICCDAMMAC, wKeyVer_PICCDAMMAC, aAid, aDamParams,
                ( byte ) ( ( aDamParams == null ) ? 0 : aDamParams.Length ), bKeySettings1, bKeySettings2, bKeySettings3,
                aKeySetValues, ( byte ) ( ( aKeySetValues == null ) ? 0 : aKeySetValues.Length ), aISOFileId, aISODFName,
                ( byte ) ( ( aISODFName == null ) ? 0 : aISODFName.Length ), aEncK, ( byte ) ( ( aEncK == null ) ? 0 : aEncK.Length ),
                ref ppDAMMAC, ref wDAMMACKeyLen );
            aDAMMAC = MarshalCopy ( oStatus, ppDAMMAC, wDAMMACKeyLen );

            return oStatus;
        }

        /// <summary>
        /// Generate MAC for Delete application.
        /// </summary>
        ///
        /// <param name="wKeyNo_PICCDAMMAC">Key number in key store of PICC DAM MAC key.</param>
        /// <param name="wKeyVer_PICCDAMMAC">Key version in key store of PICC DAM MAC key.</param>
        /// <param name="pAid">The application identifier to be used. Will be of 3 bytes with LSB first.
        ///                    If application 01 need to be created then the Aid will be 01 00 00.</param>
        /// <param name="pDAMMAC">8 byte DAMMAC.</param>
        ///
        /// <returns>Returns Success oStatus for successful operation.
        ///          Other Depending on implementation and underlaying component.</returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DESFIRE_EV2, MIFARE_Products.DESFIRE_EV3,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Generate DAMMAC for Delete Application."
            )
        ]
        public Status_t GenerateDAMMAC ( ushort wKeyNo_PICCDAMMAC, ushort wKeyVer_PICCDAMMAC, byte[] aAid, out byte[] aDAMMAC )
        {
            return GenerateDAMMAC ( 0x80, wKeyNo_PICCDAMMAC, wKeyVer_PICCDAMMAC, aAid, null, 0, 0, 0, null,
                null, null, null, out aDAMMAC );
        }

        /// <summary>
        /// Generates DAMMAC for Set Configuration with option 0x06 for Delegated Application
        /// </summary>
        ///
        /// <param name="wKeyNo_PICCDAMMAC">Key number in key store of PICC DAM MAC key.</param>
        /// <param name="wKeyVer_PICCDAMMAC">Key version in key store of PICC DAM MAC key.</param>
        /// <param name="aOld_ISODFName">This means already created delegated application ISO DF Name. Maximum 16 bytes</param>
        /// <param name="aNew_ISODFName">This means new delegated application ISO DF Name which will replace the existing one. Maximum 16 bytes</param>
        /// <param name="aDAMMAC">Generated 8 bytes DAM MAC for SetConfig option 0x06</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is of not a support key type.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Generates DAMMAC for Set Configuration with option 0x06 for Delegated Application."
            )
        ]
        public Status_t GenerateDAMMACSetConfig ( ushort wKeyNo_PICCDAMMAC, ushort wKeyVer_PICCDAMMAC, byte[] aOld_ISODFName,
            byte[] aNew_ISODFName, out byte[] aDAMMAC )
        {
            Status_t oStatus;
            IntPtr ppDAMMAC = IntPtr.Zero;
            ushort wDAMMACKeyLen = 0;

            oStatus = phalMfDuoX_GenerateDAMMACSetConfig ( m_pDataParams, wKeyNo_PICCDAMMAC, wKeyVer_PICCDAMMAC, aOld_ISODFName,
                ( byte ) ( ( aOld_ISODFName == null ) ? 0 : aOld_ISODFName.Length ), aNew_ISODFName,
                ( byte ) ( ( aNew_ISODFName == null ) ? 0 : aNew_ISODFName.Length ),
                ref ppDAMMAC, ref wDAMMACKeyLen );
            aDAMMAC = MarshalCopy ( oStatus, ppDAMMAC, wDAMMACKeyLen );

            return oStatus;
        }

        /// <summary>
        /// Computes the Transaction MAC value based on the Transaction MAC Input as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option to be used for Key diversification.
        ///                         <see cref="Diversification.DIVERSIFICATION_OFF"/>
        ///                         <see cref="Diversification.DIVERSIFICATION_ON"/>
        /// </param>
        /// <param name="wKeyNo_TMACKey">Key number to be used from software KeyStore.</param>
        /// <param name="wKeyVer_TMACKey">Key version to be used from software KeyStore.</param>
        /// <param name="aDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from the actual
        ///                    value and should be LSB first.
        /// </param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aTMI">Transaction MAC Input.</param>
        /// <param name="aTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For invalid diversification options (wOption).
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is of not a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If TMC value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Computes the Transaction MAC value based on the Transaction MAC Input as performed on the PICC."
            )
        ]
        public Status_t CalculateTMV ( ushort wOption, ushort wKeyNo_TMACKey, ushort wKeyVer_TMACKey, byte[] aDivInput,
            byte[] aTMC, byte[] aUid, byte[] aTMI, out byte[] aTMV )
        {
            Status_t oStatus;
            IntPtr ppTMV = IntPtr.Zero;
            ushort wTMVLen = 0;

            oStatus = phalMfDuoX_CalculateTMV ( m_pDataParams, wOption, wKeyNo_TMACKey, wKeyVer_TMACKey, 0, 0, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aTMC, aUid, ( byte ) ( ( aUid == null ) ?
                0 : aUid.Length ), aTMI, ( ushort ) ( ( aTMI == null ) ? 0 : aTMI.Length ), ref ppTMV, ref wTMVLen );
            aTMV = MarshalCopy ( oStatus, ppTMV, wTMVLen );

            return oStatus;
        }

        /// <summary>
        /// Computes the Transaction MAC value based on the Transaction MAC Input as performed on the PICC.
        /// </summary>
        ///
        /// <param name="wKeyNo_TMACKey">Key number to be used from hardware KeyStore.</param>
        /// <param name="wKeyVer_TMACKey">Key version to be used from hardware KeyStore.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session
        ///                         TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session
        ///                          TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from the actual
        ///                    value and should be LSB first.
        /// </param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aTMI">Transaction MAC Input.</param>
        /// <param name="aTMV">The computed Transaction MAC Value.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For invalid diversification options (wOption).
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is of not a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If TMC value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Computes the Transaction MAC value based on the Transaction MAC Input as performed on the PICC."
            )
        ]
        public Status_t CalculateTMV ( ushort wKeyNo_TMACKey, ushort wKeyVer_TMACKey, ushort wRamKeyNo, ushort wRamKeyVer,
            byte[] aTMC, byte[] aUid, byte[] aTMI, out byte[] aTMV )
        {
            Status_t oStatus;
            IntPtr ppTMV = IntPtr.Zero;
            ushort wTMVLen = 0;

            oStatus = phalMfDuoX_CalculateTMV ( m_pDataParams, 0, wKeyNo_TMACKey, wKeyVer_TMACKey, wRamKeyNo, wRamKeyVer,
                null, 0, aTMC, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aTMI, ( ushort ) ( ( aTMI == null )
                ? 0 : aTMI.Length ), ref ppTMV, ref wTMVLen );
            aTMV = MarshalCopy ( oStatus, ppTMV, wTMVLen );

            return oStatus;
        }

        /// <summary>
        /// Computes the Transaction Signature value based on the Transaction MAC Input as performed on the PICC.
        /// </summary>
        ///
        /// <param name="bCurveID">The targeted curve for the public key provided in \b aPubKey parameter.
        ///                        Should be one of the below values.
        ///                         - <see cref="TargetCurve.NIST_P256"/>
        ///                         - <see cref="TargetCurve.BRAINPOOL_P256R1"/>
        /// </param>
        /// <param name="aPubKey">Public Key to be used for signature verification..</param>
        /// <param name="aTMI">Transaction MAC Input.</param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter received from PICC during <seealso cref="CommitTransaction"/>.
        ///                    If 0x00000001 is the TMC received, then it will be 01 00 00 00.
        /// </param>
        /// <param name="aTSV">The Transaction Signature Value received from PICC. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="CryptoASym.Error.VERIFICATION_FAILED"/> Verification of Message / Signature combination failed.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Verifies the Transaction MAC Signature provided by PICC."
            )
        ]
        public Status_t VerifyTransationSignature ( byte bCurveID, byte[] aPubKey, byte[] aTMI, byte[] aTMC, byte[] aTSV )
        {
            return phalMfDuoX_VerifyTransationSignature ( m_pDataParams, bCurveID, aPubKey, ( ushort ) ( ( aPubKey != null ) ? aPubKey.Length : 0 ),
                aTMI, ( ushort ) ( ( aTMI == null ) ? 0 : aTMI.Length ), aTMC, aTSV, ( ushort ) ( ( aTSV != null ) ? aTSV.Length : 0 ) );
        }

        /// <summary>
        /// Provides the previous Decrypted Reader ID. Supports decryption using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="wOption">Diversification option to be used for Key diversification.
        ///                         <see cref="Diversification.DIVERSIFICATION_OFF"/>
        ///                         <see cref="Diversification.DIVERSIFICATION_ON"/>
        /// </param>
        /// <param name="wKeyNo_TMACKey">Key number to be used from software KeyStore.</param>
        /// <param name="wKeyVer_TMACKey">Key position to be used from software KeyStore.</param>
        /// <param name="aDivInput">Diversification input to diversify TMACKey.</param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///                    the actual value and should be LSB first.
        /// </param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="aTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For invalid diversification options (wOption).
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If TMC value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Provides the previous Decrypted Reader ID. Supports decryption using AES128 and AES 256 key type."
            )
        ]
        public Status_t DecryptReaderID ( ushort wOption, ushort wKeyNo_TMACKey, ushort wKeyVer_TMACKey, byte[] aDivInput,
            byte[] aTMC, byte[] aUid, byte[] aEncTMRI, out byte[] aTMRIPrev )
        {
            Status_t oStatus;
            IntPtr ppTMRIPrev = IntPtr.Zero;
            ushort wTMRIPrevLen = 0;

            oStatus = phalMfDuoX_DecryptReaderID ( m_pDataParams, wOption, wKeyNo_TMACKey, wKeyVer_TMACKey, 0, 0, aDivInput,
                ( byte ) ( ( aDivInput == null ) ? 0 : aDivInput.Length ), aTMC, aUid, ( byte ) ( ( aUid == null ) ?
                0 : aUid.Length ), aEncTMRI, ( byte ) ( ( aEncTMRI == null ) ? 0 : aEncTMRI.Length ),
                ref ppTMRIPrev, ref wTMRIPrevLen );
            aTMRIPrev = MarshalCopy ( oStatus, ppTMRIPrev, wTMRIPrevLen );

            return oStatus;
        }

        /// <summary>
        /// Provides the previous Decrypted Reader ID. Supports decryption using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="wKeyNo_TMACKey">Key number to be used from hardware KeyStore.</param>
        /// <param name="wKeyVer_TMACKey">Key position to be used from hardware KeyStore.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session
        ///                         TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session
        ///                          TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="aTMC">4 bytes Transaction MAC Counter. It should be 1 time subtracted from
        ///                    the actual value and should be LSB first.
        /// </param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aEncTMRI">Encrypted Transaction MAC ReaderID of the latest successful transaction.</param>
        /// <param name="aTMRIPrev">Decrypted Reader ID of the last successful transaction.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         For invalid diversification options (wOption).
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If TMC value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Provides the previous Decrypted Reader ID. Supports decryption using AES128 and AES 256 key type."
            )
        ]
        public Status_t DecryptReaderID ( ushort wKeyNo_TMACKey, ushort wKeyVer_TMACKey, ushort wRamKeyNo, ushort wRamKeyVer,
            byte[] aTMC, byte[] aUid, byte[] aEncTMRI, out byte[] aTMRIPrev )
        {
            Status_t oStatus;
            IntPtr ppTMRIPrev = IntPtr.Zero;
            ushort wTMRIPrevLen = 0;

            oStatus = phalMfDuoX_DecryptReaderID ( m_pDataParams, 0, wKeyNo_TMACKey, wKeyVer_TMACKey, wRamKeyNo, wRamKeyVer,
                null, 0, aTMC, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aEncTMRI, ( byte ) ( ( aEncTMRI == null )
                ? 0 : aEncTMRI.Length ), ref ppTMRIPrev, ref wTMRIPrevLen );
            aTMRIPrev = MarshalCopy ( oStatus, ppTMRIPrev, wTMRIPrevLen );

            return oStatus;
        }

        /// <summary>
        /// Performs computation of SDM MAC information. Supports MAC Calculation using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///                             <see cref="SDMOption.VCUID_RDCTR_PRESENT"/>
        ///                             <see cref="SDMOption.VCUID_PRESENT"/>
        ///                             <see cref="SDMOption.RDCTR_PRESENT"/>
        /// </param>
        /// <param name="wKeyNo_SDMMac">Key number to be used from software KeyStore.</param>
        /// <param name="wKeyVer_SDMMac">Key version to be used from software KeyStore.</param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aSDMReadCtr">SDM Read Counter Input. Should be 4 bytes in Length.</param>
        /// <param name="aInData">Data read out from PICC that is between SDMMacInputoffset until SDMMACOffset.</param>
        /// <param name="aSDMMAC">Secure Dynamic MAC information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If SDMOptions (bSdmOption) is not supported.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If SDM Read Counter value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.This method is not supported if Library context is SAM.",
                Message = "Performs computation of SDM MAC information. Supports MAC Calculation using AES128 and AES 256 key type."
            )
        ]
        public Status_t CalculateMACSDM ( byte bSdmOption, ushort wKeyNo_SDMMac, ushort wKeyVer_SDMMac, byte[] aUid,
            byte[] aSDMReadCtr, byte[] aInData, out byte[] aSDMMAC )
        {
            Status_t oStatus;
            IntPtr ppSDMMACPrev = IntPtr.Zero;
            ushort wSDMMACPrevLen = 0;

            oStatus = phalMfDuoX_CalculateMACSDM ( m_pDataParams, bSdmOption, wKeyNo_SDMMac, wKeyVer_SDMMac, 0, 0, aUid,
                ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aSDMReadCtr, aInData, ( ushort ) ( ( aInData == null ) ?
                0 : aInData.Length ), ref ppSDMMACPrev, ref wSDMMACPrevLen );
            aSDMMAC = MarshalCopy ( oStatus, ppSDMMACPrev, wSDMMACPrevLen );

            return oStatus;
        }

        /// <summary>
        /// Performs computation of SDM MAC information. Supports MAC Calculation using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///                             <see cref="SDMOption.VCUID_RDCTR_PRESENT"/>
        ///                             <see cref="SDMOption.VCUID_PRESENT"/>
        ///                             <see cref="SDMOption.RDCTR_PRESENT"/>
        /// </param>
        /// <param name="wKeyNo_SDMMac">Key number to be used from hardware KeyStore.</param>
        /// <param name="wKeyVer_SDMMac">Key version to be used from hardware KeyStore.</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session
        ///                         TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session
        ///                          TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aSDMReadCtr">SDM Read Counter Input. Should be 4 bytes in Length.</param>
        /// <param name="aInData">Data read out from PICC that is between SDMMacInputoffset until SDMMACOffset.</param>
        /// <param name="aSDMMAC">Secure Dynamic MAC information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If SDMOptions (bSdmOption) is not supported.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If SDM Read Counter value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Performs computation of SDM MAC information. Supports MAC Calculation using AES128 and AES 256 key type."
            )
        ]
        public Status_t CalculateMACSDM ( byte bSdmOption, ushort wKeyNo_SDMMac, ushort wKeyVer_SDMMac, ushort wRamKeyNo,
            ushort wRamKeyVer, byte[] aUid, byte[] aSDMReadCtr, byte[] aInData, out byte[] aSDMMAC )
        {
            Status_t oStatus;
            IntPtr ppSDMMACPrev = IntPtr.Zero;
            ushort wSDMMACPrevLen = 0;

            oStatus = phalMfDuoX_CalculateMACSDM ( m_pDataParams, bSdmOption, wKeyNo_SDMMac, wKeyVer_SDMMac, wRamKeyNo,
                wRamKeyVer, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aSDMReadCtr, aInData,
                ( ushort ) ( ( aInData == null ) ? 0 : aInData.Length ), ref ppSDMMACPrev,
                ref wSDMMACPrevLen );
            aSDMMAC = MarshalCopy ( oStatus, ppSDMMACPrev, wSDMMACPrevLen );

            return oStatus;
        }

        /// <summary>
        /// Performs computation of SDM Signature information.
        /// </summary>
        ///
        /// <param name="wPubKeyNo_SDMSig">Key number to be used from software KeyStore.</param>
        /// <param name="wPubKeyPos_SDMSig">Key position to be used from software KeyStore.</param>
        /// <param name="aInData">Data read out from PICC that is between SDMMacInputoffset until SDMMACOffset.</param>
        /// <param name="aSignature">Secure Dynamic Signature information.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="CryptoASym.Error.VERIFICATION_FAILED"/> Verification of Message / Signature combination failed.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Performs verification of SDM Signature information."
            )
        ]
        public Status_t VerifySDMSignature ( ushort wPubKeyNo_SDMSig, ushort wPubKeyPos_SDMSig, byte[] aInData, byte[] aSignature )
        {
            return phalMfDuoX_VerifySDMSignature ( m_pDataParams, wPubKeyNo_SDMSig, wPubKeyPos_SDMSig, aInData,
                ( ushort ) ( ( aInData == null ) ? 0 : aInData.Length ), aSignature,
                ( ushort ) ( ( aSignature != null ) ? aSignature.Length : 0 ) );
        }

        /// <summary>
        /// Performs decryption of SDM File Data. Supports decryption using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///                             <see cref="SDMOption.VCUID_RDCTR_PRESENT"/>
        ///                             <see cref="SDMOption.VCUID_PRESENT"/>
        ///                             <see cref="SDMOption.RDCTR_PRESENT"/>
        /// </param>
        /// <param name="wKeyNo_SDMEnc">Key number to be used from software KeyStore</param>
        /// <param name="wKeyVer_SDMEnc">Key version to be used from software KeyStore</param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="aEncData">Encrypted NonASCII SDM Encrypted File data.</param>
        /// <param name="aPlainData">The decrypted SDMFile data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If SDMOptions (bSdmOption) is not supported.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If SDM Read Counter value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is not supported if Library context is SAM.",
                Message = "Performs decryption of SDM File Data. Supports decryption using AES128 and AES 256 key type."
            )
        ]
        public Status_t DecryptSDMENCFileData ( byte bSdmOption, ushort wKeyNo_SDMEnc, ushort wKeyVer_SDMEnc, byte[] aUid,
            byte[] aSDMReadCtr, byte[] aEncData, out byte[] aPlainData )
        {
            Status_t oStatus;
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            oStatus = phalMfDuoX_DecryptSDMENCFileData ( m_pDataParams, bSdmOption, wKeyNo_SDMEnc, wKeyVer_SDMEnc, 0, 0,
                aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aSDMReadCtr, aEncData,
                ( ushort ) ( ( aEncData == null ) ? 0 : aEncData.Length ),
                ref ppPlainData, ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );

            return oStatus;
        }

        /// <summary>
        /// Performs decryption of SDM File Data. Supports decryption using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="bSdmOption">SDM Option to indicate which parameters to be considered.
        ///                             <see cref="SDMOption.VCUID_RDCTR_PRESENT"/>
        ///                             <see cref="SDMOption.VCUID_PRESENT"/>
        ///                             <see cref="SDMOption.RDCTR_PRESENT"/>
        /// </param>
        /// <param name="wKeyNo_SDMEnc">Key number to be used from hardware KeyStore</param>
        /// <param name="wKeyVer_SDMEnc">Key version to be used from hardware KeyStore</param>
        /// <param name="wRamKeyNo">Key number of Destination Key where the computed session
        ///                         TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="wRamKeyVer">Key version of Destination Key where the computed session
        ///                          TMAC key will be stored. To be used for SAM only.
        /// </param>
        /// <param name="aUid">UID of the card.</param>
        /// <param name="aSDMReadCtr">SDM Read Counter Input.</param>
        /// <param name="aEncData">Encrypted NonASCII SDM Encrypted File data.</param>
        /// <param name="aPlainData">The decrypted SDMFile data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///         If SDMOptions (bSdmOption) is not supported.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     Returns <see cref="Error_Param.PARAMETER_OVERFLOW"/> If SDM Read Counter value reached maximum.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library. This method is supported if Library context is SAM.",
                Message = "Performs decryption of SDM File Data. Supports decryption using AES128 and AES 256 key type."
            )
        ]
        public Status_t DecryptSDMENCFileData ( byte bSdmOption, ushort wKeyNo_SDMEnc, ushort wKeyVer_SDMEnc, ushort wRamKeyNo,
            ushort wRamKeyVer, byte[] aUid, byte[] aSDMReadCtr, byte[] aEncData, out byte[] aPlainData )
        {
            Status_t oStatus;
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            oStatus = phalMfDuoX_DecryptSDMENCFileData ( m_pDataParams, bSdmOption, wKeyNo_SDMEnc, wKeyVer_SDMEnc,
                wRamKeyNo, wRamKeyVer, aUid, ( byte ) ( ( aUid == null ) ? 0 : aUid.Length ), aSDMReadCtr, aEncData,
                ( ushort ) ( ( aEncData == null ) ? 0 : aEncData.Length ), ref ppPlainData, ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );

            return oStatus;
        }

        /// <summary>
        /// Performs decryption of SDM PICC Data. Supports decryption using AES128 and AES 256 key type.
        /// </summary>
        ///
        /// <param name="wKeyNo">Key number to be used from software KeyStore.</param>
        /// <param name="wKeyVer">Key version to be used from software KeyStore.</param>
        /// <param name="aEncData">Encrypted NonASCII SDM PICC data. </param>
        /// <param name="aPlainData">The decrypted SDMPICC data.</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/> If the buffers are null.
        ///     Returns <see cref="Error_Param.KEY"/> If key number passed is not of a support key type.
        ///     XXXX Other Depending on implementation and underlying component.
        /// </returns>
        [NxpRdLibAttribute
            (
                MIFARE_Products.DUOX,
                OtherInfo = "This is an utility method for internal purpose of the library.",
                Message = "Performs decryption of SDM PICC Data. Supports decryption using AES128 and AES 256 key type."
            )
        ]
        public Status_t DecryptSDMPICCData ( ushort wKeyNo, ushort wKeyVer, byte[] aEncData, out byte[] aPlainData )
        {
            Status_t oStatus;
            IntPtr ppPlainData = IntPtr.Zero;
            ushort wPlainDataLen = 0;

            oStatus = phalMfDuoX_DecryptSDMPICCData ( m_pDataParams, wKeyNo, wKeyVer, aEncData,
                ( ushort ) ( ( aEncData == null ) ? 0 : aEncData.Length ), ref ppPlainData,
                ref wPlainDataLen );
            aPlainData = MarshalCopy ( oStatus, ppPlainData, wPlainDataLen );

            return oStatus;
        }
#endif
        #endregion
        #endregion

        #region Memory Mapping
        /// <summary>
        /// Handle to store the specific implementation context.
        /// </summary>
        protected GCHandle m_pDataParamsInt;

        /// <summary>
        /// Retrieve private data storage of underlying C Object.
        /// </summary>
        public IntPtr m_pDataParams
        {
            get
            {
                return m_pDataParamsInt.AddrOfPinnedObject ();
            }
        }
        #endregion

        #region Private Methods
        private byte[] MarshalCopy ( Status_t oStatus, IntPtr pBuffer, int dwLength )
        {
            if ( ( oStatus.Equals ( Error_Gen.SUCCESS ) || oStatus.Equals ( Error_Gen.SUCCESS_CHAINING ) ||
                oStatus.Error.Equals ( Error.DF_7816_GEN_ERROR ) ) && dwLength > 0 )
            {
                byte[] aBuffer = null;
                if ( ( pBuffer != IntPtr.Zero ) && ( dwLength != 0 ) )
                {
                    aBuffer = new byte[dwLength];
                    Marshal.Copy ( pBuffer, aBuffer, 0, dwLength );
                }
                return aBuffer;
            }
            else
                return null;
        }

        private byte[] Resize ( Status_t oStatus, byte[] aBuffer, int dwBufferLen )
        {
            Array.Resize ( ref aBuffer, dwBufferLen );
            return aBuffer;
        }
        #endregion Support Methods
    }
    #endregion

    #region Software
    /// <summary>
    /// Class for software layer initialization interface and data params.
    ///
    /// Note:
    ///     - The AL component uses two internal buffers for processing. One is named as <see cref="DataParams_t.pCmdBuf">Command Buffer</see>
    ///       and another as <see cref="DataParams_t.pProcessBuf">Processing Buffer"</see>.
    ///     - Memory for these internal buffers needs to be passed by the user during initialization.
    ///     - The internal buffer size should not be less than minimum.Minimum sizes are,
    ///         <see cref="DataParams_t.pCmdBuf">Command Buffer</see> should be >= <see cref="MIN_CMD_BUFFER_SIZE"/>
    ///         <see cref="DataParams_t.pProcessBuf">Processing Buffer</see> should be >= <see cref="MIN_PRS_BUFFER_SIZE"/>
    ///         If the sizes are less than minimum, <see cref="Error_Param.PARAMETER_SIZE">Parameter Size</see> error will be returned.
    ///     - Processing buffer size should not be less than Command buffer size else <see cref="Error_Param.PARAMETER_SIZE">Parameter Size</see>
    ///       error will be returned.
    ///     - After completion of the application, call below interfaces to clear all the internal buffers, its sizes, dependent components
    ///       like CryptoSym and CryptASym etc...
    ///         <see cref="DeInit"/>
    ///     - Its must to Initialize the component again after calling De-Initialization.
    ///     - During Initialization of HAL component, make sure the transmit and response buffer size are not less than PICC frame size.
    ///     - Call <see cref="Generic.GetConfig">GetConfig</see> with <see cref="Config.ADDITIONAL_INFO">Additional Information</see> as
    ///       Configuration identifier when any of the interface returns <see cref="Error.DF_GEN_ERROR">General Failure</see>
    ///       <see cref="Error.DF_7816_GEN_ERROR">ISO7816 General Failure</see>.
    /// </summary>
    [ClassInterface ( ClassInterfaceType.AutoDual )]
    public class Sw : Generic
    {
        #region Constants
        /// <summary> Minimum size for allocating the command buffer during initializing. </summary>
        public const ushort MIN_CMD_BUFFER_SIZE = 256;

        /// <summary> Minimum size for allocating the response / SM processing buffer during initializing. </summary>
        public const ushort MIN_PRS_BUFFER_SIZE = 512;

        private const byte AID_LENGTH = 3;

#if !PACKAGE_PUBLIC
        private const byte TI_LENGTH = 4;
        private const byte SESSION_KEY_LENGTH = 32;
#endif
        #endregion

        #region Variables
        private byte[] aCmdBuf = null;
        private byte[] aProcessBuf = null;
        private GCHandle m_pCmdBuf;
        private GCHandle m_pProcessBuf;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE DUOX Software layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY!. </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure of the KeyStore layer. </summary>
            public IntPtr pKeyStoreDataParams;

            /// <summary> Pointer to the parameter structure of the ASymmetric Crypto component. </summary>
            public IntPtr pCryptoDataParamsASym;

            /// <summary> Pointer to the parameter structure of the Symmetric Crypto layer for encryption. </summary>
            public IntPtr pCryptoDataParamsEnc;

            /// <summary> Pointer to the parameter structure of the Symmetric Crypto layer for MACing. </summary>
            public IntPtr pCryptoDataParamsMac;

            /// <summary> Pointer to the parameter structure of the Crypto layer for Random number generation. </summary>
            public IntPtr pCryptoRngDataParams;

            /// <summary> Pointer to the parameter structure for collecting TMI. </summary>
            public IntPtr pTMIDataParams;

            /// <summary> Pointer to the parameter structure for Virtual Card. </summary>
            public IntPtr pVCADataParams;

            /// <summary> Pointer to global buffer for processing the command. </summary>
            public IntPtr pCmdBuf;

            /// <summary> Size of global command buffer.</summary>
            public ushort wCmdBufSize;

            /// <summary> Length of bytes available in command buffer (pCmdBuf) for processing.</summary>
            public ushort wCmdBufLen;

            /// <summary> Command Buffer offset while performing crypto or exchange operations.</summary>
            public ushort wCmdBufOffset;

            /// <summary> Pointer to global buffer for processing the response / secure messaging information. </summary>
            public IntPtr pProcessBuf;

            /// <summary> Size of global response / secure messaging information buffer.</summary>
            public ushort wPrsBufSize;

            /// <summary> Length of bytes available in response / secure messaging information buffer (pProcessBuf) for processing.</summary>
            public ushort wPrsBufLen;

            /// <summary> Processing Buffer offset while performing crypto or exchange operations.</summary>
            public ushort wPrsBufOffset;

            /// <summary> Aid of the currently selected application. </summary>
            public fixed byte pAid[AID_LENGTH];

#if !PACKAGE_PUBLIC
            /// <summary> Authentication Encryption key for the session. </summary>
            public fixed byte aSesAuthENCKey[SESSION_KEY_LENGTH];

            /// <summary> Authentication MAC key for the session. </summary>
            public fixed byte aSesAuthMACKey[SESSION_KEY_LENGTH];

            /// <summary> Transaction Identifier. </summary>
            public fixed byte aTI[TI_LENGTH];
#endif

            /// <summary> Command count within transaction. </summary>
            public ushort wCmdCtr;

            /// <summary> Specific error codes for MIFARE DUOX generic errors or To get the response length of some commands. </summary>
            public ushort wAdditionalInfo;

            /// <summary> Wrapped APDU mode. All native commands need to be sent wrapped in ISO 7816 APDUs. </summary>
            public byte bWrappedMode;

            /// <summary>
            /// One of the following,
            ///     - Force Short Length APDU in case of BIG ISO read.
            ///     - Exchange Native commands in Short Length APDU format.
            /// </summary>
            public byte bShortLenApdu;

            /// <summary> Command code. This will be used for differentiating the common error codes between different commands. </summary>
            public byte bCmdCode;

            /// <summary> Authenticate Command used. Can be one of the following,
            ///     0x01: EV2_Authenticated
            ///     0x02: ECC_Authenticated
            ///     0xFF: NOT_Authenticated
            /// </summary>
            public byte bAuthState;

            /// <summary> Key number against which this authentication is done. </summary>
            public byte bKeyNo;

            /// <summary> Key Type being used for Authentication. </summary>
            public ushort wKeyType;

            /// <summary>
            /// Flag to Indicate if PICC data Status. Indicates the following.
            ///     PICC Data is complete but there is still more data that needs to be provided to user.
            ///     PICC Data is complete and there is no data to be given to user, but last encrypted chunk needs to be verified.
            /// </summary>
            public byte bPICCDataComplete;

#if !PACKAGE_PUBLIC
            /// <summary> Flag to indicate if MAC needs to be verified. </summary>
            public byte bMACVerified;

            /// <summary> Flag to indicate if Encryption of Data is complete or still pending. </summary>
            public byte bIsENCPending;
#endif
        };
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_Sw_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams, IntPtr pPalMifareDataParams,
            IntPtr pKeyStoreDataParams, IntPtr pCryptoDataParamsASym, IntPtr pCryptoDataParamsEnc, IntPtr pCryptoDataParamsMac,
            IntPtr pCryptoRngDataParams, IntPtr pTMIDataParams, IntPtr pVCADataParams, IntPtr pCmdBuf, ushort wCmdBufSize,
            IntPtr pPrsBuf, ushort wPrsBufSize );

        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_Sw_DeInit ( ref DataParams_t pDataParams );
        #endregion

        #region Initialization
        /// <summary>
        /// Initialization API for MIFARE DUOX software component.
        /// </summary>
        ///
        /// <param name="oPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="oKeyStore">Pointer to Key Store data parameters.</param>
        /// <param name="oCryptoASym">Pointer to the parameter structure of the ASymmetric Crypto component.</param>
        /// <param name="oCryptoEnc">Pointer to Symmetric Crypto component context for encryption.</param>
        /// <param name="oCryptoMAC">Pointer to Symmetric Crypto component context for MACing.</param>
        /// <param name="oCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="oTMIUtils">Pointer to a TMI component.</param>
        /// <param name="oAlVcaParams">Pointer to a VCA component.</param>
        /// <param name="wCmdBufSize">Size of global command buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alMfDuoX.Sw.CMD_BUFFER_SIZE).</param>
        /// <param name="wPrsBufSize">Size of global response / secure messaging information buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alMfDuoX.Sw.PRS_BUFFER_SIZE).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>
        ///         If the buffer size is less than the expected one.
        /// </returns>
        public Status_t Init ( palMifare.Generic oPalMifare, KeyStore.Generic oKeyStore, CryptoASym.Generic oCryptoASym, CryptoSym.Generic oCryptoEnc,
            CryptoSym.Generic oCryptoMAC, CryptoRng.Generic oCryptoRng, TMIUtils oTMIUtils, alVca.Generic oAlVcaParams,
            ushort wCmdBufSize = MIN_CMD_BUFFER_SIZE, ushort wPrsBufSize = MIN_PRS_BUFFER_SIZE )
        {
            if ( m_pCmdBuf.IsAllocated ) { m_pCmdBuf.Free (); aCmdBuf = null; }
            if ( m_pProcessBuf.IsAllocated ) { m_pProcessBuf.Free (); aProcessBuf = null; }

            aCmdBuf = new byte[wCmdBufSize];
            m_pCmdBuf = GCHandle.Alloc ( aCmdBuf, GCHandleType.Pinned );

            aProcessBuf = new byte[wPrsBufSize];
            m_pProcessBuf = GCHandle.Alloc ( aProcessBuf, GCHandleType.Pinned );

            return phalMfDuoX_Sw_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero,
                ( oKeyStore != null ) ? oKeyStore.m_pDataParams : IntPtr.Zero,
                ( oCryptoASym != null ) ? oCryptoASym.m_pDataParams : IntPtr.Zero,
                ( oCryptoEnc != null ) ? oCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( oCryptoMAC != null ) ? oCryptoMAC.m_pDataParams : IntPtr.Zero,
                ( oCryptoRng != null ) ? oCryptoRng.m_pDataParams : IntPtr.Zero,
                ( oTMIUtils != null ) ? oTMIUtils.m_pDataParams : IntPtr.Zero,
                ( oAlVcaParams != null ) ? oAlVcaParams.m_pDataParams : IntPtr.Zero,
                aCmdBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pCmdBuf.AddrOfPinnedObject (),
                wCmdBufSize,
                aProcessBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pProcessBuf.AddrOfPinnedObject (),
                wPrsBufSize );
        }

#if DEBUG
        /// <summary>
        /// Initialization API for MIFARE DUOX software component.
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="pPalMifare">Pointer to a palMifare component context.</param>
        /// <param name="pKeyStore">Pointer to Key Store data parameters.</param>
        /// <param name="oCryptoASym">Pointer to the parameter structure of the ASymmetric Crypto component.</param>
        /// <param name="oCryptoEnc">Pointer to Symmetric Crypto component context for encryption.</param>
        /// <param name="oCryptoMAC">Pointer to Symmetric Crypto component context for MACing.</param>
        /// <param name="oCryptoRng">Pointer to a CryptoRng component context.</param>
        /// <param name="oTMIUtils">Pointer to a TMI component.</param>
        /// <param name="oAlVcaParams">Pointer to a VCA component.</param>
        /// <param name="wCmdBufSize">Size of global command buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alMfDuoX.Sw.CMD_BUFFER_SIZE).</param>
        /// <param name="wPrsBufSize">Size of global response / secure messaging information buffer.
        ///                            Should not be less than the default specified one (NxpRdLibNet.alMfDuoX.Sw.PRS_BUFFER_SIZE).</param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>
        ///         If the buffer size is less than the expected one.
        /// </returns>
        public Status_t Init ( ushort wDataParamSize, palMifare.Generic pPalMifare, KeyStore.Generic pKeyStore, CryptoASym.Generic oCryptoASym,
            CryptoSym.Generic oCryptoEnc, CryptoSym.Generic oCryptoMAC, CryptoRng.Generic oCryptoRng, TMIUtils oTMIUtils, alVca.Generic oAlVcaParams,
            ushort wCmdBufSize = MIN_CMD_BUFFER_SIZE, ushort wPrsBufSize = MIN_PRS_BUFFER_SIZE )
        {
            if ( m_pCmdBuf.IsAllocated ) { m_pCmdBuf.Free (); aCmdBuf = null; }
            if ( m_pProcessBuf.IsAllocated ) { m_pProcessBuf.Free (); aProcessBuf = null; }

            aCmdBuf = new byte[wCmdBufSize];
            m_pCmdBuf = GCHandle.Alloc ( aCmdBuf, GCHandleType.Pinned );

            aProcessBuf = new byte[wPrsBufSize];
            m_pProcessBuf = GCHandle.Alloc ( aProcessBuf, GCHandleType.Pinned );

            return phalMfDuoX_Sw_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( pPalMifare != null ) ? pPalMifare.m_pDataParams : IntPtr.Zero,
                ( pKeyStore != null ) ? pKeyStore.m_pDataParams : IntPtr.Zero,
                ( oCryptoASym != null ) ? oCryptoASym.m_pDataParams : IntPtr.Zero,
                ( oCryptoEnc != null ) ? oCryptoEnc.m_pDataParams : IntPtr.Zero,
                ( oCryptoMAC != null ) ? oCryptoMAC.m_pDataParams : IntPtr.Zero,
                ( oCryptoRng != null ) ? oCryptoRng.m_pDataParams : IntPtr.Zero,
                ( oTMIUtils != null ) ? oTMIUtils.m_pDataParams : IntPtr.Zero,
                ( oAlVcaParams != null ) ? oAlVcaParams.m_pDataParams : IntPtr.Zero,
                aCmdBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pCmdBuf.AddrOfPinnedObject (),
                wCmdBufSize,
                aProcessBuf.Length.Equals ( 0 ) ? IntPtr.Zero : m_pProcessBuf.AddrOfPinnedObject (),
                wPrsBufSize );
        }
#endif
        #endregion

        #region De-Initialization
        /// <summary>
        /// De Initialized MIFARE DUOX Software component.
        /// </summary>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     XXXX
        ///         Other Depending on implementation and underlying component.
        /// </returns>
        public Status_t DeInit ()
        {
            /* Clear all memory. */
            if ( m_pCmdBuf.IsAllocated ) { m_pCmdBuf.Free (); aCmdBuf = null; }
            if ( m_pProcessBuf.IsAllocated ) { m_pProcessBuf.Free (); aProcessBuf = null; }

            return phalMfDuoX_Sw_DeInit ( ref m_DataParamsInt[0] );
        }
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sw ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sw ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter and Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set
            {
                m_DataParamsInt[0] = value;
            }
            get
            {
                return m_DataParamsInt[0];
            }
        }
        #endregion

        #region Parameter Access
        public override byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[AID_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < AID_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }

#if !PACKAGE_PUBLIC
        public override byte[] SesAuthENCKey
        {
            get
            {
                byte[] bValue = new byte[SESSION_KEY_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->aSesAuthENCKey[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > SESSION_KEY_LENGTH )
                    throw new ArgumentException ( string.Format ( "Expected Length is {0} but received {1}.", SESSION_KEY_LENGTH, value.Length ) );

                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aSesAuthENCKey[i] = value[i];
                        }
                    }
                }
            }
        }

        public override byte[] SesAuthMACKey
        {
            get
            {
                byte[] bValue = new byte[SESSION_KEY_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < SESSION_KEY_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->aSesAuthMACKey[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > SESSION_KEY_LENGTH )
                    throw new ArgumentException ( string.Format ( "Expected Length is {0} but received {1}.", SESSION_KEY_LENGTH, value.Length ) );
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aSesAuthMACKey[i] = value[i];
                        }
                    }
                }
            }
        }

        public override byte[] TI
        {
            get
            {
                byte[] bValue = new byte[4];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < 4; i++ )
                        {
                            bValue[i] = pDataParams->aTI[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 4 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->aTI[i] = value[i];
                        }
                    }
                }
            }
        }
#endif

        public override ushort CmdCtr
        {
            get
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        return pDataParams->wCmdCtr;
                    }
                }
            }
            set
            {
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        pDataParams->wCmdCtr = value;
                    }
                }
            }
        }
        #endregion
    }
    #endregion

#if PACKAGE_INTERNAL
    #region NonX
    #region Sam AV4
    /// <summary>
    /// Class for SAM AV4 NonX layer initialization interface and data params.
    ///
    /// Note:
    ///     - During Initialization of HAL component, make sure the transmit and response buffer size are not less than SAM frame size.
    ///     - Call <see cref="Generic.GetConfig">GetConfig</see> with <see cref="Config.ADDITIONAL_INFO">Additional Information</see> as
    ///       Configuration identifier when any of the interface returns <see cref="Error.DF_GEN_ERROR">General Failure</see>
    ///       <see cref="Error.DF_7816_GEN_ERROR">ISO7816 General Failure</see>.
    /// </summary>
    [ClassInterface ( ClassInterfaceType.AutoDual )]
    public class Sam_NonX : Generic
    {
        #region Constants
        /// <summary> Minimum size for allocating the temporary buffer during initializing. </summary>
        public const ushort MIN_TMP_BUFFER_SIZE = 256;

        /// <summary> Minimum size for allocating the remaining buffer during initializing. </summary>
        public const ushort MIN_REM_BUFFER_SIZE = 256;

        private const byte AID_LENGTH = 3;
        #endregion

        #region Variables
        private byte[] aTmpBuf = null;
        private GCHandle pTmpBuf;

        private byte[] aRemDataBuf = null;
        private GCHandle pRemDataBuf;
        #endregion

        #region Data Structure
        /// <summary>
        /// Data structure for MIFARE DUOX SAM AV4 NonX layer implementation.
        /// </summary>
        [StructLayout ( LayoutKind.Sequential, Pack = 1 )]
        public unsafe struct DataParams_t
        {
            /// <summary> Layer ID for this component, NEVER MODIFY! </summary>
            public ushort wId;

            /// <summary> Pointer to the parameter structure of the Hal of the SAM layer. </summary>
            public IntPtr pHalSamDataParams;

            /// <summary> Pointer to the parameter structure of the palMifare component. </summary>
            public IntPtr pPalMifareDataParams;

            /// <summary> Pointer to the parameter structure for collecting TMI. </summary>
            public IntPtr pTMIDataParams;

            /// <summary> Temporary buffer for framing the command buffer. </summary>
            public IntPtr pTmpBuffer;

            /// <summary> Size of Temporary buffer. </summary>
            public ushort wTmpBufSize;

            /// <summary> Buffer for saving the remaining data to be provided to user. </summary>
            public IntPtr pRemainingData;

            /// <summary> Length of bytes available in \b pRemainingData buffer. </summary>
            public ushort wRemDataLen;

            /// <summary> Authenticate Command used. Can be one of the following,
            ///     0x01: EV2_Authenticated
            ///     0x02: ECC_Authenticated
            ///     0xFF: NOT_Authenticated
            /// </summary>
            public byte bAuthState;

            /// <summary> Key number against which this authentication is done. </summary>
            public byte bKeyNo;

            /// <summary>
            /// Wrapped APDU mode. All native commands need to be sent wrapped in
            /// ISO 7816 APDU's
            /// </summary>
            public byte bWrappedMode;

            /// <summary>
            /// One of the following,
            ///     - Force Short Length APDU in case of BIG ISO read.
            ///     - Exchange Native commands in Short Length APDU format.
            /// </summary>
            public byte bShortLenApdu;

            /// <summary> Currently selected application Id. </summary>
            public fixed byte pAid[AID_LENGTH];

            /// <summary>
            /// Specific error codes for DESFire generic errors or error codes
            /// returned by the card.
            /// </summary>
            public ushort wAdditionalInfo;

            /// <summary>
            /// Command code. This will be used for differentiating the conman error
            /// codes between different commands.
            /// </summary>
            public byte bCmdCode;

            /// <summary>
            /// Flag to Indicate if PICC data Status. Indicates the following.
            ///     PICC Data is complete but there is still more data that needs to be provided to user.
            ///     PICC Data is complete and there is no data to be given to user, but last encrypted chunk needs to be verified.
            /// </summary>
            public byte bPICCDataComplete;
        }
        #endregion

        #region DLL Imports
        [DllImport ( Common.IMPORT_LIBRARY_NAME )]
        private static extern ushort phalMfDuoX_Sam_NonX_Init ( ref DataParams_t pDataParams, ushort wSizeOfDataParams,
            IntPtr pHalSamDataParams, IntPtr pPalMifareDataParams, IntPtr pTMIDataParams, IntPtr pTmpBuffer,
            ushort wTmpBufSize, IntPtr pRemainingDataBuff );
        #endregion

        #region Initialization
        /// <summary>
        /// Initializes the DUOX SAM AV4 component in Non X mode (S mode)
        /// </summary>
        ///
        /// <param name="oSamHal">Pointer to the HAL parameter structure of the SAM.</param>
        /// <param name="oPalMifare">Pointer to Pal MIFARE parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        /// <param name="wTmpBufSize">Temporary buffer size for framing the command buffer. </param>
        /// <param name="wRemDataBufSize"></param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>
        ///         If the buffer size is less than the expected one.
        /// </returns>
        public Status_t Init ( Hal.Generic oSamHal, palMifare.Generic oPalMifare, TMIUtils oTMIUtils, ushort wTmpBufSize = MIN_TMP_BUFFER_SIZE,
            ushort wRemDataBufSize = MIN_REM_BUFFER_SIZE )
        {
            if ( pTmpBuf.IsAllocated ) { pTmpBuf.Free (); aTmpBuf = null; }
            if ( pRemDataBuf.IsAllocated ) { pRemDataBuf.Free (); aRemDataBuf = null; }

            aTmpBuf = new byte[wTmpBufSize];
            pTmpBuf = GCHandle.Alloc ( aTmpBuf, GCHandleType.Pinned );

            aRemDataBuf = new byte[wRemDataBufSize];
            pRemDataBuf = GCHandle.Alloc ( aRemDataBuf, GCHandleType.Pinned );

            return phalMfDuoX_Sam_NonX_Init ( ref m_DataParamsInt[0], ( ushort ) Marshal.SizeOf ( typeof ( DataParams_t ) ),
                ( oSamHal != null ) ? oSamHal.m_pDataParams : IntPtr.Zero,
                ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero,
                ( oTMIUtils != null ) ? oTMIUtils.m_pDataParams : IntPtr.Zero,
                aTmpBuf.Length.Equals ( 0 ) ? IntPtr.Zero : pTmpBuf.AddrOfPinnedObject (),
                wTmpBufSize, pRemDataBuf.AddrOfPinnedObject () );
        }

#if DEBUG
        /// <summary>
        /// Initializes the DUOX SAM AV4 component in Non X mode (S mode)
        /// </summary>
        ///
        /// <param name="wDataParamSize">Specifies the size of the data parameter structure.</param>
        /// <param name="oSamHal">Pointer to the HAL parameter structure of the SAM.</param>
        /// <param name="oPalMifare">Pointer to Pal MIFARE parameter structure.</param>
        /// <param name="oTMIUtils">Pointer to the parameter structure for collecting TMI.</param>
        /// <param name="wTmpBufSize">Temporary buffer size for framing the command buffer. </param>
        ///
        /// <returns>
        ///     Returns <see cref="Error_Gen.SUCCESS"/> for successful operation.
        ///     Returns <see cref="Error_Param.INVALID_PARAMETER"/>
        ///         If the buffers are null.
        ///     Returns <see cref="Error_Param.PARAMETER_SIZE"/>
        ///         If the buffer size is less than the expected one.
        /// </returns>
        public Status_t Init ( ushort wDataParamSize, Hal.Generic oSamHal, palMifare.Generic oPalMifare, TMIUtils oTMIUtils,
            ushort wTmpBufSize = MIN_TMP_BUFFER_SIZE )
        {
            if ( pTmpBuf.IsAllocated ) { pTmpBuf.Free (); aTmpBuf = null; }

            aTmpBuf = new byte[wTmpBufSize];
            pTmpBuf = GCHandle.Alloc ( aTmpBuf, GCHandleType.Pinned );

            return phalMfDuoX_Sam_NonX_Init ( ref m_DataParamsInt[0], wDataParamSize,
                ( oSamHal != null ) ? oSamHal.m_pDataParams : IntPtr.Zero,
                ( oPalMifare != null ) ? oPalMifare.m_pDataParams : IntPtr.Zero,
                ( oTMIUtils != null ) ? oTMIUtils.m_pDataParams : IntPtr.Zero,
                aTmpBuf.Length.Equals ( 0 ) ? IntPtr.Zero : pTmpBuf.AddrOfPinnedObject (),
                wTmpBufSize, pRemDataBuf.AddrOfPinnedObject () );
        }
#endif
        #endregion

        #region Memory Mapping
        private DataParams_t[] m_DataParamsInt;

        /// <summary>
        /// Constructor
        /// </summary>
        public Sam_NonX ()
        {
            // Allocate internal data parameters and pointer to them
            m_DataParamsInt = new DataParams_t[1];
            m_pDataParamsInt = GCHandle.Alloc ( m_DataParamsInt, GCHandleType.Pinned );
        }

        /// <summary>
        /// Destructor
        /// </summary>
        ~Sam_NonX ()
        {
            // Free allocated pointer to data params
            if ( m_pDataParamsInt.IsAllocated )
            {
                m_pDataParamsInt.Free ();
            }
        }

        /// <summary>
        /// Setter and Getter for DataParams structure
        /// </summary>
        public DataParams_t DataParams
        {
            set { m_DataParamsInt[0] = value; }
            get { return m_DataParamsInt[0]; }
        }
        #endregion

        #region Parameter Access
        public override byte[] Aid
        {
            get
            {
                byte[] bValue = new byte[AID_LENGTH];
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < AID_LENGTH; i++ )
                        {
                            bValue[i] = pDataParams->pAid[i];
                        }
                    }
                }
                return bValue;
            }
            set
            {
                if ( value.Length > 3 )
                    throw new ArgumentException ();
                unsafe
                {
                    fixed ( DataParams_t* pDataParams = &m_DataParamsInt[0] )
                    {
                        for ( int i = 0; i < value.Length; i++ )
                        {
                            pDataParams->pAid[i] = value[i];
                        }
                    }
                }
            }
        }
        #endregion
    }
    #endregion
    #endregion
#endif
}