/*
 * Copyright 2013, 2016 - 2020, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic MIFARE(R) Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHALMFP_H
#define PHALMFP_H

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalMifare.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHAL_MFP

/** \defgroup phalMfp MIFARE Plus (R)
 * \brief These Components implement the MIFARE Plus (R) commands.
 * @{
 */

/** \name Authentication Parameters */
/** @{ */
#define PHAL_MFP_KEYA												0x0AU	/**< MIFARE(R) Key A. */
#define PHAL_MFP_KEYB												0x0BU	/**< MIFARE(R) Key B. */
/** @} */

/** \name Custom Error Codes */
/** @{ */
#define PHAL_MFP_ERR_AUTH						(PH_ERR_CUSTOM_BEGIN + 0)	/*< MFP Authentication Error. */
#define PHAL_MFP_ERR_CMD_OVERFLOW				(PH_ERR_CUSTOM_BEGIN + 1)	/*< MFP Command Overflow Error. */
#define PHAL_MFP_ERR_MAC_PCD					(PH_ERR_CUSTOM_BEGIN + 2)	/*< MFP MAC Error. */
#define PHAL_MFP_ERR_BNR						(PH_ERR_CUSTOM_BEGIN + 3)	/*< MFP Blocknumber Error. */
#define PHAL_MFP_ERR_EXT						(PH_ERR_CUSTOM_BEGIN + 4)	/*< MFP Extension Error. */
#define PHAL_MFP_ERR_CMD_INVALID				(PH_ERR_CUSTOM_BEGIN + 5)	/*< MFP Invalid Command Error. */
#define PHAL_MFP_ERR_FORMAT						(PH_ERR_CUSTOM_BEGIN + 6)	/*< MFP Authentication Error. */
#define PHAL_MFP_ERR_GEN_FAILURE				(PH_ERR_CUSTOM_BEGIN + 7)	/*< MFP Generic Error. */
/** @} */

/** \name Sizes */
/** @{ */
#define PHAL_MFP_SIZE_TI												4U	/**< Size of Transaction Identifier. */
#define PHAL_MFP_SIZE_KEYMODIFIER										6U	/**< Size of MIFARE KeyModifier. */
#define PHAL_MFP_SIZE_MAC												16U	/**< Size of (untruncated) MAC. */
/** @} */

/**
 * end of group phalMfp
 * @}
 */
#endif /* NXPBUILD__PHAL_MFP */



/***************************************************************************************************************************************/
/* Software Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFP_SW

/** \defgroup phalMfp_Sw Component : Software
 * @{
 */

#define PHAL_MFP_SW_ID          0x01    /**< ID for Software MIFARE Plus layer. */

/** \brief MIFARE Plus Software parameter structure */
typedef struct
{
	uint16_t wId;															/**< Layer ID for this component, NEVER MODIFY! */
	void * pPalMifareDataParams;											/**< Pointer to the parameter structure of the palMifare component. */
	void * pKeyStoreDataParams;												/**< Pointer to the parameter structure of the KeyStore layer. */
	void * pCryptoDataParamsEnc;											/**< Pointer to the parameter structure of the Crypto layer for encryption. */
	void * pCryptoDataParamsMac;											/**< Pointer to the parameter structure of the Crypto layer for macing. */
	void * pCryptoRngDataParams;                            				/**< Pointer to the parameter structure of the CryptoRng layer. */
	void * pCryptoDiversifyDataParams;                      				/**< Pointer to the parameter structure of the CryptoDiversify layer (can be NULL). */
	uint8_t bKeyModifier[PHAL_MFP_SIZE_KEYMODIFIER];        				/**< Key Modifier for MIFARE Plus SL2 authentication. */
	uint16_t wRCtr;                                         				/**< R_CTR (read counter); The PICC's read counter is used for a following authentication. */
	uint16_t wWCtr;                                         				/**< W_CTR (write counter); The PICC's write counter is used for a following authentication. */
	uint8_t bTi[PHAL_MFP_SIZE_TI];                          				/**< Transaction Identifier; unused if 'bFirstAuth' = 1; uint8_t[4]. */
	uint8_t bNumUnprocessedReadMacBytes;                    				/**< Amount of data in the pUnprocessedReadMacBuffer. */
	uint8_t pUnprocessedReadMacBuffer[PHAL_MFP_SIZE_MAC];   				/**< Buffer containing unprocessed bytes for read mac answer stream. */
	uint8_t pIntermediateMac[PHAL_MFP_SIZE_MAC];            				/**< Intermediate MAC for Read Calculation. */
	uint8_t bFirstRead;                                     				/**< Indicates whether the next read is a first read in a read (MACed) sequence or not. */
} phalMfp_Sw_DataParams_t;

/**
 * \brief Initialise this layer.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phalMfp_Sw_Init(
		phalMfp_Sw_DataParams_t * pDataParams,								/**< [In] Pointer to this layer's parameter structure. */
		uint16_t wSizeOfDataParams,											/**< [In] Specifies the size of the data parameter structure. */
		void * pPalMifareDataParams,										/**< [In] Pointer to a palMifare component context. */
		void * pKeyStoreDataParams,											/**< [In] Pointer to a KeyStore component context. */
		void * pCryptoDataParamsEnc,										/**< [In] Pointer to a Crypto component context for encryption. */
		void * pCryptoDataParamsMac,										/**< [In] Pointer to a Crypto component context for Macing. */
		void * pCryptoRngDataParams,										/**< [In] Pointer to a CryptoRng component context. */
		void * pCryptoDiversifyDataParams									/**< [In] Pointer to the parameter structure of the CryptoDiversify layer (can be NULL). */
	);

/**
 * end of group phalMfp_Sw
 * @}
 */
#endif /* NXPBUILD__PHAL_MFP_SW */



/***************************************************************************************************************************************/
/* Sam NonX Dataparams and Initialization Interface.																				   */
/***************************************************************************************************************************************/




/***************************************************************************************************************************************/
/* Sam X Dataparams and Initialization Interface.																				       */
/***************************************************************************************************************************************/




/***************************************************************************************************************************************/
/* Mifare Plus EV1 Generic command declarations.     																				   */
/***************************************************************************************************************************************/

#ifdef NXPBUILD__PHAL_MFP

/** \addtogroup phalMfp
 * @{
 */

/** \name Options to indicate the ISO14443 protocol layer to be used. */
/** @{ */
#define PHAL_MFP_ISO14443_L3										0x00U	/**< Option to use Iso14443 Layer 3 protocol. */
#define PHAL_MFP_ISO14443_L4										0x01U	/**< Option to use Iso14443 Layer 4 protocol. */
/** @} */

/** \name Options to indicate the communication mode. */
/** @{ */
#define PHAL_MFP_ENCRYPTION_OFF										0x00U	/**< Option to indicate the communication between PCD and PICC is plain. */
#define PHAL_MFP_ENCRYPTION_ON										0x01U	/**< Option to indicate the communication between PCD and PICC is encrypted. */
/** @} */

/** \name Options to indicate the communication mode as maced for PCD to PICC transfer. */
/** @{ */
#define PHAL_MFP_MAC_ON_COMMAND_OFF									0x00U	/**< Option to indicate the communication is not maced for PCD to PICC transfer. */
#define PHAL_MFP_MAC_ON_COMMAND_ON									0x01U	/**< Option to indicate the communication is maced for PCD to PICC transfer. */
/** @} */

/** \name Options to indicate the communication mode as maced for PICC to PCD transfer. */
/** @{ */
#define PHAL_MFP_MAC_ON_RESPONSE_OFF								0x00U	/**< Option to indicate the communication is not maced for PICC to PCD transfer. */
#define PHAL_MFP_MAC_ON_RESPONSE_ON									0x01U	/**< Option to indicate the communication is maced for PICC to PCD transfer. */
/** @} */

/* Mifare Plus personalization commands. --------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_Personalization Commands_Personalization
 * \brief These Components implements the MIFARE Plus personalization commands.
 * @{
 */

/**
 * \brief Performs a MIFARE Plus Write Perso command.
 *
 * The Write Perso command can be executed using the ISO14443-3 communication protocol (after layer 3 activation)\n
 * or using the ISO14443-4 protocol (after layer 4 activation)\n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_WritePerso(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bLayer4Comm,												/**< [In] ISO14443 protocol to be used.
																			 *			\arg #PHAL_MFP_ISO14443_L3
																			 *			\arg #PHAL_MFP_ISO14443_L4
																			 */
		uint16_t wBlockNr,													/**< [In] Block number to be personalized. */
		uint8_t * pValue													/**< [In] The value for the block mentioned in BlockNr parameter. */
	);

/**
 * \brief Performs a MIFARE Plus Commit Perso command.
 *
 * The Commit Perso command can be executed using the ISO14443-3 communication protocol (after layer 3 activation)\n
 * or using the ISO14443-4 protocol (after layer 4 activation).\n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_CommitPerso(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bLayer4Comm													/**< [In] ISO14443 protocol to be used.
																			 *			\arg #PHAL_MFP_ISO14443_L3
																			 *			\arg #PHAL_MFP_ISO14443_L4
																			 */
	);

/**
 * end of group phalMfp_Personalization
 * @}
 */


/* Mifare Plus authentication commands. ---------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_Authenticate Commands_Authenticate
 * \brief These Components implement the MIFARE Plus authentication commands.
 * @{
 */

/**
 * \brief Perform MIFARE(R) Authenticate command in Security Level 2 with MIFARE Picc.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_AuthenticateClassicSL2(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bBlockNo,													/**< [In] PICC block number to be used for authentication. */
		uint8_t bKeyType,													/**< [In] Authentication key type to be used.
																			 *			\arg #PHAL_MFP_KEYA
																			 *			\arg #PHAL_MFP_KEYB
																			 */
		uint16_t wKeyNumber,     											/**< [In] Key number to used from software or hardware keystore. */
		uint16_t wKeyVersion,    											/**< [In] Key version to used from software or hardware keystore. */
		uint8_t * pUid,														/**< [In] UID of the PICC received during anti-collision sequence. */
		uint8_t bUidLength													/**< [In] Length of the UID buffer. */
	);


/**
 * end of group phalMfp_Authenticate
 * @}
 */


/* Mifare Plus data operation commands. ---------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_DataOperation Commands_DataOperations
 * \brief These Components implement the MIFARE Plus data operation commands.
 * @{
 */

/**
 * \brief Performs a Multi Block Write command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_MultiBlockWrite(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bBlockNr,													/**< [In] PICC block number to which the data should be written. */
		uint8_t bNumBlocks,													/**< [In] Number of blocks to write (must not be more than 3). */
		uint8_t * pBlocks													/**< [In] Block(s) (16 * bNumBlocks bytes). */
	);

/**
 * \brief Performs a Multi Block Read command. This command is for to use in Security Level 2 only and with MFC Authenticated state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_MultiBlockRead(
		void * pDataParams,													/**< [In] Pointer to this layer's parameter structure. */
		uint8_t bBlockNr,													/**< [In] PICC block number from which the data should be read. */
		uint8_t bNumBlocks,													/**< [In] Number of blocks to read (must not be more than 3). */
		uint8_t * pBlocks													/**< [Out] Block(s) (16 * bNumBlocks bytes). */
	);



/**
 * end of group phalMfp_DataOperation
 * @}
 */




/* Mifare Plus special commands. ----------------------------------------------------------------------------------------------------- */
/** \defgroup phalMfp_Special Commands_Special
 * \brief These Components implement the MIFARE Plus additional feature commands.
 * @{
 */

/**
 * \brief Performs a Reset Auth command.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phalMfp_ResetAuth(
		void * pDataParams													/**< [In] Pointer to this layer's parameter structure. */
	);

/**
 * \brief Reset the libraries internal secure messaging state.
 *
 * This function must be called before interacting with the PICC to set the libraries internal card-state back to default.\n
 * E.g. when an error occurred or after a reset of the field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phalMfp_ResetSecMsgState(
		void * pDataParams													/**< [In] Pointer to this layer's parameter structure. */
	);


/**
 * end of group phalMfp_Special
 * @}
 */

/**
 * end of group phalMfp
 * @}
 */

#endif /* NXPBUILD__PHAL_MFP */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHMFPLIB_H */
