/*
 * Copyright 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example program to demonstrate Symmetric features.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/** Header for this file */
#include "Example-CryptoLibFootprint_Sym.h"

#ifdef NXPBUILD__PH_CRYPTOSYM

static void * pCryptoSym = NULL;

phStatus_t Demo_SymmetricCrypto_Main()
{
    uint32_t dwChoice = 0;
    char aChoice[5];
    uint8_t bEndLoop = FALSE;

    uint8_t aAddData_Buffer[256];

    do
    {
        dwChoice = 0;

        /* Clear the console. */
        system("cls");

        if(CRYPTOSYM_MAX > 2)
        {
#ifdef NXPBUILD__PH_CRYPTOSYM_SW
            printf("Press %d to demonstrate Symmetric Crypto functionality using Software implementation. \n", CRYPTOSYM_SOFTWARE);
#endif /* NXPBUILD__PH_CRYPTOSYM_SW */

#ifdef NXPBUILD__PH_CRYPTOSYM_MBEDTLS
            printf("Press %d to demonstrate Symmetric Crypto functionality using external mBedTLS implementation. \n", CRYPTOSYM_MBEDTLS);
#endif /* NXPBUILD__PH_CRYPTOSYM_MBEDTLS */

            printf("\n");
            printf("Press x to exit the application.\n");
            printf("Press r to return to previous demo description.\n");
            printf("Enter the option and press Enter to perform the demo - ");
            scanf("%s", aChoice);

            /* Convert to Integer value. */
            ToInt(aChoice, dwChoice);

            /* Return to previous demo description if requested */
            RETURN_IF_REQUESTED(dwChoice);

            /* Exit if requested by user. */
            EXIT_IF_REQUESTED(dwChoice);
        }
        else
        {
            dwChoice = 1;
        }

        printf("\n\n");

        switch(dwChoice)
        {
#ifdef NXPBUILD__PH_CRYPTOSYM_SW
            case CRYPTOSYM_SOFTWARE:
                /* Initialize CryptoSym component. */
                CHECK_SUCCESS(phCryptoSym_Sw_Init(&stCryptoSym_Sw, (uint16_t) sizeof(phCryptoSym_Sw_DataParams_t), pKeyStore), TRUE);

                pCryptoSym = &stCryptoSym_Sw;

                /* End the loop */
                bEndLoop = TRUE;
                break;
#endif /* NXPBUILD__PH_CRYPTOSYM_SW */

#ifdef NXPBUILD__PH_CRYPTOSYM_MBEDTLS
            case CRYPTOSYM_MBEDTLS:
                /* Initialize CryptoSym component. */
                CHECK_SUCCESS(phCryptoSym_mBedTLS_Init(&stCryptoSym_mBedTLS, (uint16_t) sizeof(phCryptoSym_mBedTLS_DataParams_t), pKeyStore,
                    aAddData_Buffer, (uint16_t) sizeof(aAddData_Buffer)), TRUE);

                pCryptoSym = &stCryptoSym_mBedTLS;

                /* End the loop */
                bEndLoop = TRUE;
                break;
#endif /* NXPBUILD__PH_CRYPTOSYM_MBEDTLS */

            default:
                printf("Invalid option selected.\n");
                printf("--------------------------------------------------------------------------------------------------- \n");
                printf("\n\n");
                printf("Press any key to continue with the demonstration.\n");
                printf("Press r to return to previous demo description.\n");
                printf("Press x to exit the demonstration.\n");
                dwChoice = _getch();

                /* Return to previous demo description if requested */
                RETURN_IF_REQUESTED(dwChoice);

                /* Exit if requested by user. */
                EXIT_IF_REQUESTED(dwChoice);
                break;
        }
    } while(!bEndLoop);

    CHECK_SUCCESS(Demo_SymmetricCrypto_Features(), FALSE);

    return PH_ERR_SUCCESS;
}

phStatus_t Demo_SymmetricCrypto_Features()
{
    uint32_t dwChoice = 0;
    char aChoice[5];
    uint8_t aMessage[] = "Symmetric Crypto Demonstration";
    uint16_t wMesgLen = (uint16_t)sizeof(aMessage);

    uint8_t IS_KEY_LOADED = FALSE;

    /* Initialize Crypto Symmetric component for logging. */
#ifdef NXPBUILD__PH_LOG
    CHECK_SUCCESS(phLog_Register(pCryptoSym, aLogEntries, (uint16_t) (sizeof(aLogEntries) / sizeof(phLog_LogEntry_t))), TRUE);
#endif /* NXPBUILD__PH_LOG */

    do
    {
        dwChoice = 0;

        /* Clear the console. */
        system("cls");

        printf("Press %d for Demo on Loading a Key directly. \n", LOAD_KEY_DIRECT);
#ifdef NXPBUILD__PH_KEYSTORE
        printf("Press %d for Demo on Loading a Key from KeyStore. \n", LOAD_KEY_KEYSTORE);
#endif /* NXPBUILD__PH_KEYSTORE */

        printf("Press %d for Demo on Message Encryption & Decryption. \n", ENCRYPT_DECRYPT);
        printf("Press %d for Demo on Diversification of Key directly. \n", DIVERSIFICATION_DIRECT_KEY);

#ifdef NXPBUILD__PH_KEYSTORE
        printf("Press %d for Demo on Diversification of Key from KeyStore. \n", DIVERSIFICATION_KEYSTORE);
#endif /* NXPBUILD__PH_KEYSTORE */

        printf("Press %d for Demo on Computing the MAC for the Message. \n", MAC);

        printf("\n");
        printf("Press x to exit the application.\n");
        printf("Press r to return to previous demo description.\n");
        printf("Enter the option and press Enter to perform the demo - ");
        scanf("%s", aChoice);

        /* Convert to Integer value. */
        ToInt(aChoice, dwChoice);

        printf("\n\n");

        /* Return to previous demo description if requested */
        RETURN_IF_REQUESTED(dwChoice);

        /* Exit if requested by user. */
        EXIT_IF_REQUESTED(dwChoice);

        switch(dwChoice)
        {
            case LOAD_KEY_DIRECT:
                printf("Demo on Loading a Key directly -------------------------------------------------------------------- \n");

                IS_KEY_LOADED = FALSE;
                CHECK_SUCCESS(LoadKeyDirect(), TRUE);
                IS_KEY_LOADED = TRUE;

                printf("\tDirect Key Loading Successful.\n");
                break;

#ifdef NXPBUILD__PH_KEYSTORE
            case LOAD_KEY_KEYSTORE:
                printf("Demo on Loading a Key from KeyStore --------------------------------------------------------------- \n");

                IS_KEY_LOADED = FALSE;
                CHECK_SUCCESS(LoadKeyKeystore(), TRUE);
                IS_KEY_LOADED = TRUE;

                printf("\tKey Loading from KeyStore Successfull.\n");
                break;
#endif /* NXPBUILD__PH_KEYSTORE */

            case ENCRYPT_DECRYPT:
                printf("Demo on Message Encryption / Decryption --------------------------------------------------------- \n\n");

                /* Check if Key Loaded. */
                if(!IS_KEY_LOADED)
                {
                    printf("\tNo Key available for Message Encryption / Decryption.\n");
                    printf("\tPerform Load Key features prior performing Message Encryption / Decryption.\n\n");
                    break;
                }

                CHECK_SUCCESS(Message_Encrypt_Decrypt(aMessage, wMesgLen), TRUE);

                printf("\tMessage Encryption & Decryption Successful\n");
                break;

            case DIVERSIFICATION_DIRECT_KEY:
                printf("Demo on Diversification of Key directly ----------------------------------------------------------- \n");

                CHECK_SUCCESS(Diversification(FALSE), TRUE);

                printf("\tKey Diversification Successful.\n");
                break;

#ifdef NXPBUILD__PH_KEYSTORE
            case DIVERSIFICATION_KEYSTORE:
                printf("Demo on Diversification of Key from KeyStore ------------------------------------------------------ \n");

                CHECK_SUCCESS(Diversification(TRUE), TRUE);

                printf("\tKey Diversification Successfull.\n");
                break;
#endif /* NXPBUILD__PH_KEYSTORE */

            case MAC:
                printf("Demo on Computing the MAC for the Message --------------------------------------------------------- \n");

                /* Check if Key Loaded. */
                if(!IS_KEY_LOADED)
                {
                    printf("\tNo Key available for Macing.\n");
                    printf("\tPerform Load Key features prior performing MAC for Message.\n\n");
                    break;
                }

                CHECK_SUCCESS(ComputeMAC(aMessage, wMesgLen), TRUE);

                printf("\tSuccessfully Generated MAC for the Message.\n");
                break;

            default:
                printf("Invalid option selected.\n");
                break;
        }

        printf("--------------------------------------------------------------------------------------------------- \n");
        printf("\n\n");
        printf("Press any key to continue with the demonstration.\n");
        printf("Press x to exit the demonstration.\n");
        dwChoice = _getch();
    } while(END_LOOP(dwChoice));

    /* Return to previous demo description if requested */
    RETURN_IF_REQUESTED(dwChoice);

    /* Exit if requested by user. */
    EXIT_IF_REQUESTED(dwChoice);

    return PH_ERR_SUCCESS;
}

#ifdef NXPBUILD__PH_KEYSTORE
phStatus_t AddKeyToKeyStore_Sym(uint8_t bKeyNo, uint8_t bKeyVer, uint16_t wKeyType, uint8_t * pKey)
{
    phStatus_t wStatus = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_FormatKeyEntry(pKeyStore, bKeyNo, wKeyType));
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_SetKeyAtPos(pKeyStore, bKeyNo, 0x00, wKeyType, pKey, bKeyVer));

    return PH_ERR_SUCCESS;
}
#endif /* NXPBUILD__PH_KEYSTORE */

phStatus_t LoadKeyDirect()
{
    phStatus_t wStatus = 0;

    uint32_t dwChoice = 0;
    char aChoice[5];
    uint8_t aKey[32];
    uint8_t bEndLoop = FALSE;

    do
    {
        dwChoice = 0;

#ifndef MBEDTLS_DES_ALT
        printf("Press %d to Load DES (56 Bit) Key. \n", LOAD_KEY_TYPE_DES);
#endif /* MBEDTLS_DES_ALT */
        printf("Press %d to Load 2K2DES (112 Bit) Key. \n", LOAD_KEY_TYPE_2K3DES);
        printf("Press %d to Load 3K3DES (168 Bit) Key. \n", LOAD_KEY_TYPE_3K3DES);

        printf("Press %d to Load AES128 (128 Bit) Key. \n", LOAD_KEY_TYPE_AES128);
#ifndef MBEDTLS_AES_ALT
        printf("Press %d to Load AES192 (192 Bit) Key. \n", LOAD_KEY_TYPE_AES192);
#endif /* MBEDTLS_AES_ALT */
        printf("Press %d to Load AES256 (256 Bit) Key. \n", LOAD_KEY_TYPE_AES256);

        printf("\n");
        printf("Press x to exit the application.\n");
        printf("Enter the option and press Enter to perform the demo - ");
        scanf("%s", aChoice);

        /* Convert to Integer value. */
        ToInt(aChoice, dwChoice);

        printf("\n\n");

        /* Exit if requested by user. */
        EXIT_IF_REQUESTED(dwChoice);

        /* End the loop */
        bEndLoop = TRUE;

        switch(dwChoice)
        {
#ifndef MBEDTLS_DES_ALT
            case LOAD_KEY_TYPE_DES:
                aKey[0x00] = 0x00; aKey[0x01] = 0x01; aKey[0x02] = 0x02; aKey[0x03] = 0x03; aKey[0x04] = 0x04; aKey[0x05] = 0x05; aKey[0x06] = 0x06; aKey[0x07] = 0x07;
                wStatus = phCryptoSym_LoadKeyDirect(pCryptoSym, aKey, PH_CRYPTOSYM_KEY_TYPE_DES);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 8, "%02X ", "\n\n\n");
                break;
#endif /* MBEDTLS_DES_ALT */

            case LOAD_KEY_TYPE_2K3DES:
                aKey[0x00] = 0x00; aKey[0x01] = 0x01; aKey[0x02] = 0x02; aKey[0x03] = 0x03; aKey[0x04] = 0x04; aKey[0x05] = 0x05; aKey[0x06] = 0x06; aKey[0x07] = 0x07;
                aKey[0x08] = 0x08; aKey[0x09] = 0x09; aKey[0x0A] = 0x0A; aKey[0x0B] = 0x0B; aKey[0x0C] = 0x0C; aKey[0x0D] = 0x0D; aKey[0x0E] = 0x0E; aKey[0x0F] = 0x0F;
                wStatus = phCryptoSym_LoadKeyDirect(pCryptoSym, aKey, PH_CRYPTOSYM_KEY_TYPE_2K3DES);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 16, "%02X ", "\n\n\n");
                break;

            case LOAD_KEY_TYPE_3K3DES:
                aKey[0x00] = 0x00; aKey[0x01] = 0x01; aKey[0x02] = 0x02; aKey[0x03] = 0x03; aKey[0x04] = 0x04; aKey[0x05] = 0x05; aKey[0x06] = 0x06; aKey[0x07] = 0x07;
                aKey[0x08] = 0x08; aKey[0x09] = 0x09; aKey[0x0A] = 0x0A; aKey[0x0B] = 0x0B; aKey[0x0C] = 0x0C; aKey[0x0D] = 0x0D; aKey[0x0E] = 0x0E; aKey[0x0F] = 0x0F;
                aKey[0x10] = 0x10; aKey[0x11] = 0x11; aKey[0x12] = 0x12; aKey[0x13] = 0x13; aKey[0x14] = 0x14; aKey[0x15] = 0x15; aKey[0x16] = 0x16; aKey[0x17] = 0x17;
                wStatus = phCryptoSym_LoadKeyDirect(pCryptoSym, aKey, PH_CRYPTOSYM_KEY_TYPE_3K3DES);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 24, "%02X ", "\n\n\n");
                break;

            case LOAD_KEY_TYPE_AES128:
                aKey[0x00] = 0x00; aKey[0x01] = 0x01; aKey[0x02] = 0x02; aKey[0x03] = 0x03; aKey[0x04] = 0x04; aKey[0x05] = 0x05; aKey[0x06] = 0x06; aKey[0x07] = 0x07;
                aKey[0x08] = 0x08; aKey[0x09] = 0x09; aKey[0x0A] = 0x0A; aKey[0x0B] = 0x0B; aKey[0x0C] = 0x0C; aKey[0x0D] = 0x0D; aKey[0x0E] = 0x0E; aKey[0x0F] = 0x0F;
                wStatus = phCryptoSym_LoadKeyDirect(pCryptoSym, aKey, PH_CRYPTOSYM_KEY_TYPE_AES128);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 16, "%02X ", "\n\n\n");
                break;

#ifndef MBEDTLS_AES_ALT
            case LOAD_KEY_TYPE_AES192:
                aKey[0x00] = 0x00; aKey[0x01] = 0x01; aKey[0x02] = 0x02; aKey[0x03] = 0x03; aKey[0x04] = 0x04; aKey[0x05] = 0x05; aKey[0x06] = 0x06; aKey[0x07] = 0x07;
                aKey[0x08] = 0x08; aKey[0x09] = 0x09; aKey[0x0A] = 0x0A; aKey[0x0B] = 0x0B; aKey[0x0C] = 0x0C; aKey[0x0D] = 0x0D; aKey[0x0E] = 0x0E; aKey[0x0F] = 0x0F;
                aKey[0x10] = 0x10; aKey[0x11] = 0x11; aKey[0x12] = 0x12; aKey[0x13] = 0x13; aKey[0x14] = 0x14; aKey[0x15] = 0x15; aKey[0x16] = 0x16; aKey[0x17] = 0x17;
                wStatus = phCryptoSym_LoadKeyDirect(pCryptoSym, aKey, PH_CRYPTOSYM_KEY_TYPE_AES192);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 24, "%02X ", "\n\n\n");
                break;
#endif /* MBEDTLS_AES_ALT */

            case LOAD_KEY_TYPE_AES256:
                aKey[0x00] = 0x00; aKey[0x01] = 0x01; aKey[0x02] = 0x02; aKey[0x03] = 0x03; aKey[0x04] = 0x04; aKey[0x05] = 0x05; aKey[0x06] = 0x06; aKey[0x07] = 0x07;
                aKey[0x08] = 0x08; aKey[0x09] = 0x09; aKey[0x0A] = 0x0A; aKey[0x0B] = 0x0B; aKey[0x0C] = 0x0C; aKey[0x0D] = 0x0D; aKey[0x0E] = 0x0E; aKey[0x0F] = 0x0F;
                aKey[0x10] = 0x10; aKey[0x11] = 0x11; aKey[0x12] = 0x12; aKey[0x13] = 0x13; aKey[0x14] = 0x14; aKey[0x15] = 0x15; aKey[0x16] = 0x16; aKey[0x17] = 0x17;
                aKey[0x18] = 0x18; aKey[0x19] = 0x19; aKey[0x1A] = 0x1A; aKey[0x1B] = 0x1B; aKey[0x1C] = 0x1C; aKey[0x1D] = 0x1D; aKey[0x1E] = 0x1E; aKey[0x1F] = 0x1F;
                wStatus = phCryptoSym_LoadKeyDirect(pCryptoSym, aKey, PH_CRYPTOSYM_KEY_TYPE_AES256);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 32, "%02X ", "\n\n\n");
                break;

            default:
                printf("Invalid option selected.\n");
                printf("--------------------------------------------------------------------------------------------------- \n");
                printf("\n\n");
                printf("Press any key to continue with the demonstration.\n");
                printf("Press x to exit the demonstration.\n");
                dwChoice = _getch();

                /* Continue the loop */
                bEndLoop = FALSE;
                break;
        }

    } while(!bEndLoop);

    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        printf("\tPerform Encrypt and Decrypt operations.\n\n");

    return wStatus;
}

phStatus_t LoadKeyKeystore()
{
    phStatus_t wStatus = 0;

    uint32_t dwChoice = 0;
    char aChoice[5];
    uint8_t aKey[32];
    uint8_t bEndLoop = FALSE;

    do
    {
        dwChoice = 0;

#ifndef MBEDTLS_DES_ALT
        printf("Press %d to Load DES (56 Bit) Key. \n", LOAD_KEY_TYPE_DES);
#endif /* MBEDTLS_DES_ALT */
        printf("Press %d to Load 2K2DES (112 Bit) Key. \n", LOAD_KEY_TYPE_2K3DES);
        printf("Press %d to Load 3K3DES (168 Bit) Key. \n", LOAD_KEY_TYPE_3K3DES);

        printf("Press %d to Load AES128 (128 Bit) Key. \n", LOAD_KEY_TYPE_AES128);
#ifndef MBEDTLS_AES_ALT
        printf("Press %d to Load AES192 (192 Bit) Key. \n", LOAD_KEY_TYPE_AES192);
#endif /* MBEDTLS_AES_ALT */
        printf("Press %d to Load AES256 (256 Bit) Key. \n", LOAD_KEY_TYPE_AES256);

        printf("\n");
        printf("Press x to exit the application.\n");
        printf("Enter the option and press Enter to perform the demo - ");
        scanf("%s", aChoice);

        /* Convert to Integer value. */
        ToInt(aChoice, dwChoice);

        printf("\n\n");

        /* Exit if requested by user. */
        EXIT_IF_REQUESTED(dwChoice);

        /* End the loop */
        bEndLoop = TRUE;

        switch(dwChoice)
        {
#ifndef MBEDTLS_DES_ALT
            case LOAD_KEY_TYPE_DES:
                aKey[0x00] = 0x50; aKey[0x01] = 0x51; aKey[0x02] = 0x52; aKey[0x03] = 0x53; aKey[0x04] = 0x54; aKey[0x05] = 0x55; aKey[0x06] = 0x56; aKey[0x07] = 0x57;

                printf("\tLoading Key to KeyStore\n");
                PH_CHECK_SUCCESS_FCT(wStatus, AddKeyToKeyStore_Sym(DES_KEY_ADDRESS, DES_KEY_VERSION, PH_KEYSTORE_KEY_TYPE_DES, aKey));
                wStatus = phCryptoSym_LoadKey(pCryptoSym, DES_KEY_ADDRESS, DES_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_DES);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 8, "%02X ", "\n\n\n");
                break;
#endif /* MBEDTLS_DES_ALT */

            case LOAD_KEY_TYPE_2K3DES:
                aKey[0x00] = 0x50; aKey[0x01] = 0x51; aKey[0x02] = 0x52; aKey[0x03] = 0x53; aKey[0x04] = 0x54; aKey[0x05] = 0x55; aKey[0x06] = 0x56; aKey[0x07] = 0x57;
                aKey[0x08] = 0x58; aKey[0x09] = 0x59; aKey[0x0A] = 0x5A; aKey[0x0B] = 0x5B; aKey[0x0C] = 0x5C; aKey[0x0D] = 0x5D; aKey[0x0E] = 0x5E; aKey[0x0F] = 0x5F;

                printf("\tLoading Key to KeyStore\n");
                PH_CHECK_SUCCESS_FCT(wStatus, AddKeyToKeyStore_Sym(DES2K3_KEY_ADDRESS, DES2K3_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_2K3DES, aKey));
                wStatus = phCryptoSym_LoadKey(pCryptoSym, DES2K3_KEY_ADDRESS, DES2K3_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_2K3DES);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 16, "%02X ", "\n\n\n");
                break;

            case LOAD_KEY_TYPE_3K3DES:
                aKey[0x00] = 0x50; aKey[0x01] = 0x51; aKey[0x02] = 0x52; aKey[0x03] = 0x53; aKey[0x04] = 0x54; aKey[0x05] = 0x55; aKey[0x06] = 0x56; aKey[0x07] = 0x57;
                aKey[0x08] = 0x58; aKey[0x09] = 0x59; aKey[0x0A] = 0x5A; aKey[0x0B] = 0x5B; aKey[0x0C] = 0x5C; aKey[0x0D] = 0x5D; aKey[0x0E] = 0x5E; aKey[0x0F] = 0x5F;
                aKey[0x10] = 0x60; aKey[0x11] = 0x61; aKey[0x12] = 0x62; aKey[0x13] = 0x63; aKey[0x14] = 0x64; aKey[0x15] = 0x65; aKey[0x16] = 0x66; aKey[0x17] = 0x67;

                printf("\tLoading Key to KeyStore\n");
                PH_CHECK_SUCCESS_FCT(wStatus, AddKeyToKeyStore_Sym(DES3K3_KEY_ADDRESS, DES3K3_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_3K3DES, aKey));
                wStatus = phCryptoSym_LoadKey(pCryptoSym, DES3K3_KEY_ADDRESS, DES3K3_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_3K3DES);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 24, "%02X ", "\n\n\n");
                break;

            case LOAD_KEY_TYPE_AES128:
                aKey[0x00] = 0x50; aKey[0x01] = 0x51; aKey[0x02] = 0x52; aKey[0x03] = 0x53; aKey[0x04] = 0x54; aKey[0x05] = 0x55; aKey[0x06] = 0x56; aKey[0x07] = 0x57;
                aKey[0x08] = 0x58; aKey[0x09] = 0x59; aKey[0x0A] = 0x5A; aKey[0x0B] = 0x5B; aKey[0x0C] = 0x5C; aKey[0x0D] = 0x5D; aKey[0x0E] = 0x5E; aKey[0x0F] = 0x5F;

                printf("\tLoading Key to KeyStore\n");
                PH_CHECK_SUCCESS_FCT(wStatus, AddKeyToKeyStore_Sym(AES128_KEY_ADDRESS, AES128_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_AES128, aKey));
                wStatus = phCryptoSym_LoadKey(pCryptoSym, AES128_KEY_ADDRESS, AES128_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_AES128);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 16, "%02X ", "\n\n\n");
                break;

#ifndef MBEDTLS_AES_ALT
            case LOAD_KEY_TYPE_AES192:
                aKey[0x00] = 0x50; aKey[0x01] = 0x51; aKey[0x02] = 0x52; aKey[0x03] = 0x53; aKey[0x04] = 0x54; aKey[0x05] = 0x55; aKey[0x06] = 0x56; aKey[0x07] = 0x57;
                aKey[0x08] = 0x58; aKey[0x09] = 0x59; aKey[0x0A] = 0x5A; aKey[0x0B] = 0x5B; aKey[0x0C] = 0x5C; aKey[0x0D] = 0x5D; aKey[0x0E] = 0x5E; aKey[0x0F] = 0x5F;
                aKey[0x10] = 0x60; aKey[0x11] = 0x61; aKey[0x12] = 0x62; aKey[0x13] = 0x63; aKey[0x14] = 0x64; aKey[0x15] = 0x65; aKey[0x16] = 0x66; aKey[0x17] = 0x67;

                printf("\tLoading Key to KeyStore\n");
                PH_CHECK_SUCCESS_FCT(wStatus, AddKeyToKeyStore_Sym(AES192_KEY_ADDRESS, AES192_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_AES192, aKey));
                wStatus = phCryptoSym_LoadKey(pCryptoSym, AES192_KEY_ADDRESS, AES192_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_AES192);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 24, "%02X ", "\n\n\n");
                break;
#endif /* MBEDTLS_AES_ALT */

            case LOAD_KEY_TYPE_AES256:
                aKey[0x00] = 0x50; aKey[0x01] = 0x51; aKey[0x02] = 0x52; aKey[0x03] = 0x53; aKey[0x04] = 0x54; aKey[0x05] = 0x55; aKey[0x06] = 0x56; aKey[0x07] = 0x57;
                aKey[0x08] = 0x58; aKey[0x09] = 0x59; aKey[0x0A] = 0x5A; aKey[0x0B] = 0x5B; aKey[0x0C] = 0x5C; aKey[0x0D] = 0x5D; aKey[0x0E] = 0x5E; aKey[0x0F] = 0x5F;
                aKey[0x10] = 0x60; aKey[0x11] = 0x61; aKey[0x12] = 0x62; aKey[0x13] = 0x63; aKey[0x14] = 0x64; aKey[0x15] = 0x65; aKey[0x16] = 0x66; aKey[0x17] = 0x67;
                aKey[0x18] = 0x68; aKey[0x19] = 0x69; aKey[0x1A] = 0x6A; aKey[0x1B] = 0x6B; aKey[0x1C] = 0x6C; aKey[0x1D] = 0x6D; aKey[0x1E] = 0x6E; aKey[0x1F] = 0x6F;

                printf("\tLoading Key to KeyStore\n");
                PH_CHECK_SUCCESS_FCT(wStatus, AddKeyToKeyStore_Sym(AES256_KEY_ADDRESS, AES256_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_AES256, aKey));
                wStatus = phCryptoSym_LoadKey(pCryptoSym, AES256_KEY_ADDRESS, AES256_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_AES256);

                printf("\tSuccessfully loaded the below Key.\n");
                printf("\t\t");
                PrintData(aKey, 32, "%02X ", "\n\n\n");
                break;

            default:
                printf("Invalid option selected.\n");
                printf("--------------------------------------------------------------------------------------------------- \n");
                printf("\n\n");
                printf("Press any key to continue with the demonstration.\n");
                printf("Press x to exit the demonstration.\n");
                dwChoice = _getch();

                /* Continue the loop */
                bEndLoop = FALSE;
                break;
        }

    } while(!bEndLoop);

    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        printf("\tPerform Encrypt, Decrypt or Key Diversification operations.\n\n");

    return wStatus;
}

phStatus_t Message_Encrypt_Decrypt(uint8_t * pMessage, uint16_t wMesgLen)
{
    phStatus_t wStatus = 0;
    uint16_t wBlockSize = 0;

    uint8_t aIv[16] = { 0x00, 0x00, 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 , 0x00 };
    uint8_t aPaddedData[64];
    uint8_t aEncData[64];
    uint16_t wPaddedDataLen = 0;

    /* Get the BlockSize. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_GetConfig(pCryptoSym, PH_CRYPTOSYM_CONFIG_BLOCK_SIZE, &wBlockSize));

    printf("\tEncryption & Decryption using ECB Cipher mode ################################################### \n\n");
    printf("\t\tPlain Message\n");
    PrintASCII(pMessage, wMesgLen, TRUE);
    PrintHEX(pMessage, wMesgLen);
    printf("\n");

    /* Perform Data Padding. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_ApplyPadding(PH_CRYPTOSYM_PADDING_MODE_2, pMessage, wMesgLen, (uint8_t) wBlockSize,
        (uint16_t) sizeof(aPaddedData), aPaddedData, &wPaddedDataLen));

    /* Encrypt the data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(pCryptoSym, PH_CRYPTOSYM_CIPHER_MODE_ECB, aPaddedData, wPaddedDataLen, aEncData));
    printf("\t\tEncrypted Data         : ");
    PrintData(aEncData, wPaddedDataLen, "%02X ", "\n");

    /* Decrypt the data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(pCryptoSym, PH_CRYPTOSYM_CIPHER_MODE_ECB, aEncData, wPaddedDataLen, aPaddedData));

    /* Preform Padding removal. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_RemovePadding(PH_CRYPTOSYM_PADDING_MODE_2, aPaddedData, wPaddedDataLen, (uint8_t) wBlockSize,
        (uint16_t) sizeof(aPaddedData), aPaddedData, &wPaddedDataLen));
    printf("\t\tDecrypted Data\n");
    PrintASCII(aPaddedData, wPaddedDataLen, TRUE);
    PrintHEX(aPaddedData, wPaddedDataLen);
    printf("\n\n");





    /*printf("\tEncryption & Decryption using CBC_DF4 Cipher mode ############################################### \n\n");
    printf("\t\tPlain Message\n");
    PrintASCII(pMessage, wMesgLen, TRUE);
    PrintHEX(pMessage, wMesgLen);
    printf("\n");*/

    /* Perform Data Padding. */
    /*PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_ApplyPadding(PH_CRYPTOSYM_PADDING_MODE_2, pMessage, wMesgLen, (uint8_t) wBlockSize,
        (uint16_t) sizeof(aPaddedData), aPaddedData, &wPaddedDataLen));
    printf("\t\tPadded Data            : ");
    PrintData(aPaddedData, wPaddedDataLen, "%02X ", "\n\n");

    printf("\t\tInitialization Vector  : ");
    PrintData(aIv, wBlockSize, "%02X ", "\n");
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(pCryptoSym, aIv, (uint8_t) wBlockSize));*/

    /* Encrypt the data. */
    /*PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(pCryptoSym, PH_CRYPTOSYM_CIPHER_MODE_CBC_DF4, aPaddedData, wPaddedDataLen, aEncData));
    printf("\t\tEncrypted Data         : ");
    PrintData(aEncData, wPaddedDataLen, "%02X ", "\n\n");

    printf("\t\tInitialization Vector  : ");
    PrintData(aIv, wBlockSize, "%02X ", "\n");
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(pCryptoSym, aIv, (uint8_t) wBlockSize));*/

    /* Decrypt the data. */
    /*PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(pCryptoSym, PH_CRYPTOSYM_CIPHER_MODE_CBC_DF4, aEncData, wPaddedDataLen, aPaddedData));
    printf("\t\tDecrypted Data         : ");
    PrintData(aPaddedData, wPaddedDataLen, "%02X ", "\n\n");*/

    /* Perform Padding removal. */
    /*PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_RemovePadding(PH_CRYPTOSYM_PADDING_MODE_2, aPaddedData, wPaddedDataLen, (uint8_t) wBlockSize,
        (uint16_t) sizeof(aPaddedData), aPaddedData, &wPaddedDataLen));
    printf("\t\tDecrypted Data\n");
    PrintASCII(aPaddedData, wPaddedDataLen, TRUE);
    PrintHEX(aPaddedData, wPaddedDataLen);
    printf("\n\n\n");*/





    printf("\tEncryption & Decryption using CBC Cipher mode ################################################### \n\n");
    printf("\t\tPlain Message\n");
    PrintASCII(pMessage, wMesgLen, TRUE);
    PrintHEX(pMessage, wMesgLen);
    printf("\n");

    /* Perform Data Padding. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_ApplyPadding(PH_CRYPTOSYM_PADDING_MODE_2, pMessage, wMesgLen, (uint8_t) wBlockSize,
        (uint16_t) sizeof(aPaddedData), aPaddedData, &wPaddedDataLen));

    printf("\t\tInitialization Vector  : ");
    PrintData(aIv, wBlockSize, "%02X ", "\n");
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(pCryptoSym, aIv, (uint8_t) wBlockSize));

    /* Encrypt the data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(pCryptoSym, PH_CRYPTOSYM_CIPHER_MODE_CBC, aPaddedData, wPaddedDataLen, aEncData));
    printf("\t\tEncrypted Data         : ");
    PrintData(aEncData, wPaddedDataLen, "%02X ", "\n\n");

    printf("\t\tInitialization Vector  : ");
    PrintData(aIv, wBlockSize, "%02X ", "\n");
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(pCryptoSym, aIv, (uint8_t) wBlockSize));

    /* Decrypt the data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(pCryptoSym, PH_CRYPTOSYM_CIPHER_MODE_CBC, aEncData, wPaddedDataLen, aPaddedData));

    /* Perform Padding removal. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_RemovePadding(PH_CRYPTOSYM_PADDING_MODE_2, aPaddedData, wPaddedDataLen, (uint8_t) wBlockSize,
        (uint16_t) sizeof(aPaddedData), aPaddedData, &wPaddedDataLen));
    printf("\t\tDecrypted Data\n");
    PrintASCII(aPaddedData, wPaddedDataLen, TRUE);
    PrintHEX(aPaddedData, wPaddedDataLen);
    printf("\n\n\n");

    return wStatus;
}

phStatus_t Diversification(uint8_t bUseKeyStore)
{
    phStatus_t wStatus = 0;

    uint8_t aDivKey[16];
    uint8_t aKey[16];
    uint8_t aDivInput[16] = { 0xA7, 0x38, 0x73, 0xBF, 0x58, 0x0A, 0x6C, 0x60, 0x4D, 0x3D, 0x16, 0x46, 0x0F, 0x80, 0xC0, 0xBD };

    if(bUseKeyStore)
    {
        aKey[0x00] = 0x50; aKey[0x01] = 0x51; aKey[0x02] = 0x52; aKey[0x03] = 0x53; aKey[0x04] = 0x54; aKey[0x05] = 0x55; aKey[0x06] = 0x56; aKey[0x07] = 0x57;
        aKey[0x08] = 0x58; aKey[0x09] = 0x59; aKey[0x0A] = 0x5A; aKey[0x0B] = 0x5B; aKey[0x0C] = 0x5C; aKey[0x0D] = 0x5D; aKey[0x0E] = 0x5E; aKey[0x0F] = 0x5F;
        PH_CHECK_SUCCESS_FCT(wStatus, AddKeyToKeyStore_Sym(AES128_KEY_ADDRESS, AES128_KEY_VERSION, PH_CRYPTOSYM_KEY_TYPE_AES128, aKey));
    }
    else
    {
        aKey[0x00] = 0x00; aKey[0x01] = 0x01; aKey[0x02] = 0x02; aKey[0x03] = 0x03; aKey[0x04] = 0x04; aKey[0x05] = 0x05; aKey[0x06] = 0x06; aKey[0x07] = 0x07;
        aKey[0x08] = 0x08; aKey[0x09] = 0x09; aKey[0x0A] = 0x0A; aKey[0x0B] = 0x0B; aKey[0x0C] = 0x0C; aKey[0x0D] = 0x0D; aKey[0x0E] = 0x0E; aKey[0x0F] = 0x0F;
    }

    printf("\tDiversification of AES128 Key using DESFire Mode ###################################### \n\n");
    printf("\t\tPlain Key            : ");
    PrintData(aKey, 16, "%02X ", "\n");

    printf("\t\tDiversification Input: ");
    PrintData(aDivInput, 16, "%02X ", "\n");

    if(bUseKeyStore)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyKey(pCryptoSym, PH_CRYPTOSYM_DIV_MODE_DESFIRE, AES128_KEY_ADDRESS, AES128_KEY_VERSION,
            aDivInput, 16, aDivKey));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(pCryptoSym, PH_CRYPTOSYM_DIV_MODE_DESFIRE, aKey, PH_CRYPTOSYM_KEY_TYPE_AES128,
            aDivInput, 16, aDivKey));
    }
    printf("\t\tDiversified Key      : ");
    PrintData(aDivKey, 16, "%02X ", "\n");
    printf("\n\n");

    printf("\tDiversification of AES128 Key using PLUS Mode ######################################### \n\n");
    printf("\t\tPlain Key            : ");
    PrintData(aKey, 16, "%02X ", "\n");

    printf("\t\tDiversification Input: ");
    PrintData(aDivInput, 16, "%02X ", "\n");

    if(bUseKeyStore)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyKey(pCryptoSym, PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS, AES128_KEY_ADDRESS, AES128_KEY_VERSION,
            aDivInput, 16, aDivKey));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(pCryptoSym, PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS, aKey, PH_CRYPTOSYM_KEY_TYPE_AES128,
            aDivInput, 16, aDivKey));
    }
    printf("\t\tDiversified Key      : ");
    PrintData(aDivKey, 16, "%02X ", "\n");
    printf("\n\n");

    printf("\tDiversification of AES128 Key using Ultralight Mode ################################### \n\n");
    printf("\t\tPlain Key            : ");
    PrintData(aKey, 16, "%02X ", "\n");

    printf("\t\tDiversification Input: ");
    PrintData(aDivInput, 16, "%02X ", "\n");

    if(bUseKeyStore)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyKey(pCryptoSym, PH_CRYPTOSYM_DIV_MODE_MIFARE_ULTRALIGHT, AES128_KEY_ADDRESS, AES128_KEY_VERSION,
            aDivInput, 16, aDivKey));
    }
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_DiversifyDirectKey(pCryptoSym, PH_CRYPTOSYM_DIV_MODE_MIFARE_ULTRALIGHT, aKey, PH_CRYPTOSYM_KEY_TYPE_AES128,
            aDivInput, 16, aDivKey));
    }
    printf("\t\tDiversified Key      : ");
    PrintData(aDivKey, 16, "%02X ", "\n");
    printf("\n\n");

    return wStatus;
}

phStatus_t ComputeMAC(uint8_t * pMessage, uint16_t wMesgLen)
{
    phStatus_t wStatus = 0;

    uint16_t wKeyType = 0;

    uint8_t aMAC[16];
    uint8_t bMACLen = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_GetConfig(pCryptoSym, PH_CRYPTOSYM_CONFIG_KEY_TYPE, &wKeyType));

    printf("\tMAC Computation using CBC-MAC Mode ############################################################## \n\n");

    printf("\t\tPlain Message\n");
    PrintASCII(pMessage, wMesgLen, TRUE);
    PrintHEX(pMessage, wMesgLen);
    printf("\n");

    PH_CHECK_SUCCESS_FCT(wStatus, DispalyAddStatus(pCryptoSym, phCryptoSym_CalculateMac(pCryptoSym, PH_CRYPTOSYM_MAC_MODE_CBCMAC,
        pMessage, wMesgLen, aMAC, &bMACLen)));

    printf("\t\tMAC                    : ");
    PrintData(aMAC, bMACLen, "%02X ", "\n\n");


    printf("\tMAC Computation using CMAC Mode ################################################################# \n\n");
    printf("\t\tPlain Message\n");
    PrintASCII(pMessage, wMesgLen, TRUE);
    PrintHEX(pMessage, wMesgLen);
    printf("\n");

    PH_CHECK_SUCCESS_FCT(wStatus, DispalyAddStatus(pCryptoSym, phCryptoSym_CalculateMac(pCryptoSym, PH_CRYPTOSYM_MAC_MODE_CMAC,
        pMessage, wMesgLen, aMAC, &bMACLen)));

    printf("\t\tMAC                    : ");
    PrintData(aMAC, bMACLen, "%02X ", "\n\n");

    return wStatus;
}

#endif /* NXPBUILD__PH_CRYPTOSYM */
