/*
 * Copyright 2013, 2016 - 2017, 2019, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAMAV2 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <phhalHw.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHHAL_HW_SAMAV2

#include "phhalHw_SamAV2.h"
#include "phhalHw_SamAV2_Int.h"
#include "phhalHw_SamAV2_HcUtils.h"
#include <phhalHw_SamAV2_Cmd.h>


phStatus_t phhalHw_SamAV2_Init(phhalHw_SamAV2_DataParams_t * pDataParams, uint16_t wSizeOfDataParams, void * pBalDataParams,
	void * pReaderHalDataParams, void * pKeyStoreDataParams, void * pCryptoENCDataParams, void * pCryptoMACDataParams,
	void * pCryptoRngDataParams, uint8_t bOpMode, uint8_t bLogicalChannel, uint8_t* pTxBuffer, uint16_t wTxBufSize,
	uint8_t* pRxBuffer, uint16_t wRxBufSize)
{
	if(sizeof(phhalHw_SamAV2_DataParams_t) != wSizeOfDataParams)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_HAL);
	}
	PH_ASSERT_NULL_DATA_PARAM(pDataParams, PHHAL_HW_SAMAV2_ID);
	PH_ASSERT_NULL_PARAM(pBalDataParams, PHHAL_HW_SAMAV2_ID);
	PH_ASSERT_NULL_PARAM(pTxBuffer, PHHAL_HW_SAMAV2_ID);
	PH_ASSERT_NULL_PARAM(pRxBuffer, PHHAL_HW_SAMAV2_ID);

	/* Init. private data */
	pDataParams->wId = PH_COMP_HAL | PHHAL_HW_SAMAV2_ID;
	pDataParams->pBalDataParams = pBalDataParams;
	pDataParams->pReaderHalDataParams = pReaderHalDataParams;
	pDataParams->pKeyStoreDataParams = pKeyStoreDataParams;
	pDataParams->pENCCryptoDataParams = pCryptoENCDataParams;
	pDataParams->pMACCryptoDataParams = pCryptoMACDataParams;
	pDataParams->pCryptoRngDataParams = pCryptoRngDataParams;
	pDataParams->Cmd_Ctr = 0;
	pDataParams->bHostMode = PHHAL_HW_SAMAV2_HC_AV1_MODE;
	pDataParams->bAuthType = 0x00;
	pDataParams->bPendingEncCmdDataLength = 0;
	pDataParams->bPendingMacCmdDataLength = 0;
	pDataParams->bPendingMacRespDataLength = 0;
	pDataParams->bCmdSM = PHHAL_HW_SAMAV2_HC_AV2_NO_SM;
	pDataParams->bRespSM = PHHAL_HW_SAMAV2_HC_AV2_NO_SM;
	pDataParams->bCommandChaining = PHHAL_HW_SAMAV2_HC_AV2_NO_CHAINING;
	pDataParams->bResponseChaining = PHHAL_HW_SAMAV2_HC_AV2_NO_CHAINING;
	pDataParams->bMasterKeyCmacMode = PH_OFF;
	pDataParams->bChangeEntryPlain = PH_OFF;
	pDataParams->bOpMode = bOpMode;
	pDataParams->bLogicalChannel = bLogicalChannel;
	pDataParams->pTxBuffer = pTxBuffer;
	pDataParams->wTxBufSize = wTxBufSize;
	pDataParams->wTxBufLen = 0;
	pDataParams->pRxBuffer = pRxBuffer;
	pDataParams->wRxBufSize = wRxBufSize;
	pDataParams->wRxBufLen = 0;
	pDataParams->wRxBufStartPos = 0;
	pDataParams->wTxBufStartPos = 0;
	pDataParams->bCardType = PHHAL_HW_CARDTYPE_ISO14443A;
	pDataParams->bTimeoutUnit = PHHAL_HW_TIME_MICROSECONDS;
	pDataParams->wFieldOffTime = PHHAL_HW_FIELD_OFF_DEFAULT;
	pDataParams->wFieldRecoveryTime = PHHAL_HW_FIELD_RECOVERY_DEFAULT;
	pDataParams->wAdditionalInfo = 0;
	pDataParams->wTimingMode = PHHAL_HW_TIMING_MODE_OFF;
	pDataParams->dwTimingUs = 0;
	pDataParams->bMifareCryptoDisabled = PH_ON;
	pDataParams->bRfResetAfterTo = PH_OFF;
	pDataParams->bDisableNonXCfgMapping = PH_OFF;

	/* Verify exchange buffers */
	if((wTxBufSize <= PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN) || (wRxBufSize <= PHHAL_HW_SAMAV2_RESERVED_RX_BUFFER_LEN))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
	}

	/* Verify operation mode */
	if(bOpMode > PHHAL_HW_SAMAV2_OPMODE_X_RC663)
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
	}

	/* Verify NonX reader HAL pointer */
	if((bOpMode == PHHAL_HW_SAMAV2_OPMODE_NON_X) && (pReaderHalDataParams == NULL))
	{
		return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}



phStatus_t phhalHw_SamAV2_DetectMode(phhalHw_SamAV2_DataParams_t * pDataParams)
{
	phStatus_t  PH_MEMLOC_REM statusTmp;
	uint8_t     PH_MEMLOC_REM bCompleteUid[PHHAL_HW_SAMAV2_CMD_GET_VERSION_RESPONSE_LENGTH];
	uint8_t     PH_MEMLOC_REM bCompleteUidLength;
	uint8_t     PH_MEMLOC_REM pKeyEntryBuffer[PHHAL_HW_SAMAV2_KEYENTRY_SIZE];
	uint8_t     PH_MEMLOC_REM bKeyEntryLength;
	uint8_t     PH_MEMLOC_REM bSet1;

	/* Issue GetVersion command */
	PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SamAV2_Cmd_SAM_GetVersion(
		pDataParams,
		bCompleteUid,
		&bCompleteUidLength));

	/* Check length of received response */
	if(bCompleteUidLength == PHHAL_HW_SAMAV2_CMD_GET_VERSION_RESPONSE_LENGTH)
	{
		/* Retrieve Host-Mode */
		switch(bCompleteUid[PHHAL_HW_SAMAV2_CMD_GET_VERSION_RESPONSE_HOSTMODE_POS])
		{
			case 0xA1:
				pDataParams->bHostMode = PHHAL_HW_SAMAV2_HC_AV1_MODE;
				break;
			case 0xA2:
				pDataParams->bHostMode = PHHAL_HW_SAMAV2_HC_AV2_MODE;
				break;
			default:
				return PH_ADD_COMPCODE(PH_ERR_INTERFACE_ERROR, PH_COMP_HAL);
		}
	}
	/* Older SAM AV1 versions do not send the AV1/AV2 byte at the end */
	else if(bCompleteUidLength == (PHHAL_HW_SAMAV2_CMD_GET_VERSION_RESPONSE_LENGTH - 1))
	{
		pDataParams->bHostMode = PHHAL_HW_SAMAV2_HC_AV1_MODE;
	}
	else
	{
		return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
	}

	/* Store the UID globally */
	memcpy(   /* PRQA S 3200 */
		pDataParams->bUid,
		&bCompleteUid[PHHAL_HW_SAMAV2_CMD_GET_VERSION_RESPONSE_UID_OFFSET],
		PHHAL_HW_SAMAV2_HC_SAM_UID_SIZE);

	/* Retrieve status of MasterKey */
	PH_CHECK_SUCCESS_FCT(statusTmp, phhalHw_SamAV2_Cmd_SAM_GetKeyEntry(
		pDataParams,
		0x00,
		pKeyEntryBuffer,
		&bKeyEntryLength
	));

	/* check for AV1 mode */
	if(pDataParams->bHostMode == PHHAL_HW_SAMAV2_HC_AV1_MODE)
	{
		bSet1 = pKeyEntryBuffer[bKeyEntryLength - 1];
	}
	/* AV2 mode */
	else
	{
		bSet1 = pKeyEntryBuffer[bKeyEntryLength - 2];
	}

	/* Check if CMAC mode is enabled */
	if(bSet1 & 0x01)
	{
		pDataParams->bMasterKeyCmacMode = PH_ON;
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHw_SamAV2_GetConfig(phhalHw_SamAV2_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
	/* In case of Non-X mode, the GetConfig is directly redirected to the Reader IC if not disabled. */
	/* Exceptions are: RxLastbits, Crypto1 and custom configs. */
	if(pDataParams->bDisableNonXCfgMapping == PH_OFF)
	{
		if((pDataParams->bOpMode == PHHAL_HW_SAMAV2_OPMODE_NON_X) &&
			(wConfig != PHHAL_HW_CONFIG_RXLASTBITS) &&
			(wConfig != PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1) &&
			(wConfig != PHHAL_HW_SAMAV2_CONFIG_HOSTMODE) &&
			(wConfig != PHHAL_HW_SAMAV2_CONFIG_DISABLE_NONX_CFG_MAPPING))
		{
			/* Also do not forward if TxBufferLen is requested and MfCrypto is enabled (buffering is done in this HAL) */
			if(!((wConfig == PHHAL_HW_CONFIG_TXBUFFER_LENGTH) && (!pDataParams->bMifareCryptoDisabled)))
			{
				return phhalHw_GetConfig(pDataParams->pReaderHalDataParams, wConfig, pValue);
			}
		}
	}

	switch(wConfig)
	{
		case PHHAL_HW_CONFIG_PARITY:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_TXCRC:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_RXCRC:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_TXLASTBITS:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_ADDITIONAL_INFO:
		case PHHAL_HW_CONFIG_RXLASTBITS:

			*pValue = pDataParams->wAdditionalInfo;
			break;

		case PHHAL_HW_CONFIG_RXDEAFBITS:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_CLEARBITSAFTERCOLL:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_TXDATARATE:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_RXDATARATE:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_MODINDEX:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_ASK100:

			/* Read config from shadow */
			*pValue = pDataParams->wCfgShadow[wConfig];
			break;

		case PHHAL_HW_CONFIG_TIMEOUT_VALUE_US:

			if(pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MICROSECONDS)
			{
				*pValue = pDataParams->wCfgShadow[wConfig];
			}
			else
			{
				if(pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] > (0xFFFF / 1000))
				{
					return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
				}
				*pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS] * 1000;
			}
			break;

		case PHHAL_HW_CONFIG_TIMEOUT_VALUE_MS:

			if(pDataParams->bTimeoutUnit == PHHAL_HW_TIME_MILLISECONDS)
			{
				*pValue = pDataParams->wCfgShadow[wConfig];
			}
			else
			{
				*pValue = pDataParams->wCfgShadow[PHHAL_HW_CONFIG_TIMEOUT_VALUE_US] / 1000;
			}
			break;

		case PHHAL_HW_CONFIG_TIMING_MODE:

			*pValue = pDataParams->wTimingMode;
			break;

		case PHHAL_HW_CONFIG_TIMING_US:

			if(pDataParams->dwTimingUs > 0xFFFF)
			{
				return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
			}

			*pValue = (uint16_t) pDataParams->dwTimingUs;
			pDataParams->dwTimingUs = 0;
			break;

		case PHHAL_HW_CONFIG_TIMING_MS:

			if(pDataParams->dwTimingUs > (0xFFFF * 1000))
			{
				pDataParams->dwTimingUs = 0;
				return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_HAL);
			}

			*pValue = (uint16_t) (pDataParams->dwTimingUs / 1000);
			pDataParams->dwTimingUs = 0;
			break;

		case PHHAL_HW_CONFIG_FIELD_OFF_TIME:

			*pValue = pDataParams->wFieldOffTime;
			break;

		case PHHAL_HW_CONFIG_FIELD_RECOVERY_TIME:

			*pValue = pDataParams->wFieldRecoveryTime;
			break;

		case PHHAL_HW_CONFIG_DISABLE_MF_CRYPTO1:

			*pValue = pDataParams->bMifareCryptoDisabled;
			break;

		case PHHAL_HW_CONFIG_RXBUFFER_STARTPOS:

			/* Return parameter */
			*pValue = pDataParams->wRxBufStartPos;
			break;

		case PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE:

			/* Return parameter */
			*pValue = pDataParams->wRxBufSize - PHHAL_HW_SAMAV2_RESERVED_RX_BUFFER_LEN;
			break;

		case PHHAL_HW_CONFIG_TXBUFFER_BUFSIZE:

			/* Return parameter */
			*pValue = pDataParams->wTxBufSize - (PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN + pDataParams->wTxBufStartPos);
			break;

		case PHHAL_HW_CONFIG_TXBUFFER_LENGTH:

			/* Needed for Mifare Encrypted buffered data */
			if(pDataParams->wTxBufLen_Cmd >= (PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1))
			{
				*pValue = pDataParams->wTxBufLen_Cmd - (PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1);
			}
			else
			{
				/* Normal Exchange */
				if(pDataParams->wTxBufLen >= PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN)
				{
					*pValue = pDataParams->wTxBufLen - PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN;
				}
				else
				{
					*pValue = 0;
				}
			}
			break;

		case PHHAL_HW_CONFIG_TXBUFFER:

			/* Needed for Mifare Encrypted buffered data */
			if(pDataParams->wTxBufLen_Cmd >= (PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1))
			{
				/* Check additional info parameter */
				if(((PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1) + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
				{
					return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
				}

				/* Return TxBuffer byte */
				*pValue = (uint16_t) pDataParams->pTxBuffer[(PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN - 1) + pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
			}
			/* Normal Exchange */
			else
			{
				/* Check additional info parameter */
				if((PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN + pDataParams->wAdditionalInfo) >= pDataParams->wTxBufSize)
				{
					return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
				}

				/* Return TxBuffer byte */
				*pValue = (uint16_t) pDataParams->pTxBuffer[PHHAL_HW_SAMAV2_RESERVED_TX_BUFFER_LEN + pDataParams->wTxBufStartPos + pDataParams->wAdditionalInfo];
			}
			break;

		case PHHAL_HW_CONFIG_RFRESET_ON_TIMEOUT:

			*pValue = (uint16_t) pDataParams->bRfResetAfterTo;
			break;

		case PHHAL_HW_SAMAV2_CONFIG_HOSTMODE:

			*pValue = pDataParams->bHostMode;
			break;

		case PHHAL_HW_SAMAV2_CONFIG_DISABLE_NONX_CFG_MAPPING:

			*pValue = pDataParams->bDisableNonXCfgMapping;
			break;

		case PHHAL_HW_CONFIG_SET_READER_IC:
			*pValue = pDataParams->bOpMode;
			break;

		default:
			return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
	}

	return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_SAMAV2 */
