/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Internal functions of Generic NTAG X DNA Application Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7462 $
 * $Date: 2025-08-29 14:09:06 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *  Rajendran Kumar: Generated 22 Aug 2024
 *
 */

#ifndef PHALNTAGXDNA_INT_H
#define PHALNTAGXDNA_INT_H

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_NTAGXDNA

#include <phalNtagXDna.h>

/* Validate Certificate Depth */
#define PHAL_NTAGXDNA_VALIDATE_CERTIFICATE_DEPTH(Depth)                             \
    switch(Depth)                                                                   \
    {                                                                               \
        case PHAL_NTAGXDNA_CERT_DEPTH_END_LEAF_ONLY:                                \
        case PHAL_NTAGXDNA_CERT_DEPTH_END_LEAF_PARENT:                              \
        case PHAL_NTAGXDNA_CERT_DEPTH_END_LEAF_PARENT_GRAND_PARENT:                 \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);  \
    }

/* Validate Session Key Size */
#define PHAL_NTAGXDNA_VALIDATE_SESSION_KEY_SIZE(KeySize)                            \
    switch(KeySize)                                                                 \
    {                                                                               \
        case PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES128:                                 \
        case PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES256:                                 \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);  \
    }

#define PHAL_NTAGXDNA_ABS(Val1, Val2)                                               \
    ((Val1) > (Val2) ? (Val1 - Val2) : (Val2 - Val1))

 /* Validate Curve ID. */
#define PHAL_NTAGXDNA_VALIDATE_CURVE(CurveID)                                       \
    switch(CurveID)                                                                 \
    {                                                                               \
        case PHAL_NTAGXDNA_TARGET_CURVE_ID_NIST_P256:                               \
        case PHAL_NTAGXDNA_TARGET_CURVE_ID_BRAINPOOL_P256R1:                        \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_NTAGXDNA);                \
    }

/* Validate Diversification Options */
#define PHAL_NTAGXDNA_VALIDATE_DIVERSIFICATION_OPTIONS(Option)                      \
    switch(Option)                                                                  \
    {                                                                               \
        case PHAL_NTAGXDNA_NO_DIVERSIFICATION:                                      \
        case PH_CRYPTOSYM_DIV_MODE_DESFIRE:                                         \
        case PH_CRYPTOSYM_DIV_MODE_MIFARE_PLUS:                                     \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_NTAGXDNA);                \
    }

/* Validate Diversification Length */
#define PHAL_NTAGXDNA_VALIDATE_DIVERSIFICATION_LENGTH(Option, Length)               \
    if ((wOption != PHAL_NTAGXDNA_NO_DIVERSIFICATION) &&                            \
        (Length > PHAL_NTAGXDNA_DIV_INPUT_LEN_MAX))                                 \
    {                                                                               \
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);      \
    }

/* Validate Keytype. */
#define PHAL_NTAGXDNA_VALIDATE_KEYTYPE(KeyType)                                     \
    switch(KeyType)                                                                 \
    {                                                                               \
        case PH_KEYSTORE_KEY_TYPE_AES128:                                           \
        case PH_KEYSTORE_KEY_TYPE_AES256:                                           \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_KEY, PH_COMP_AL_NTAGXDNA);                \
    }

#define PHAL_NTAGXDNA_NEAREST_MULTIPLE(Number, OutVar)                              \
    OutVar = ( ( ( Number + PH_CRYPTOSYM_AES_BLOCK_SIZE - 1U ) /                    \
                PH_CRYPTOSYM_AES_BLOCK_SIZE ) * PH_CRYPTOSYM_AES_BLOCK_SIZE )

#define PHAL_NTAGXDNA_IS_NOT_MULTIPLE_AES_BLOCK_SIZE(Number)                        \
    (Number == 0) ? 1U : (Number / PH_CRYPTOSYM_AES_BLOCK_SIZE)

#define PHAL_NTAGXDNA_IS_MULTIPLE_AES_BLOCK_SIZE(Number)                            \
    !(Number % PH_CRYPTOSYM_AES_BLOCK_SIZE)

#define PHAL_MFCC_AES_BLOCK_SIZE_DIFF(Value)                                        \
    (((Value > PH_CRYPTOSYM_AES_BLOCK_SIZE) ? PH_CRYPTOSYM_AES256_KEY_SIZE          \
        : PH_CRYPTOSYM_AES128_KEY_SIZE) - Value)

#define PHAL_NTAGXDNA_PREVIOUS_MULTIPLE(Number, OutVar)                             \
    OutVar = (PHAL_NTAGXDNA_IS_MULTIPLE_AES_BLOCK_SIZE(Number) ? Number :           \
             ((((Number + PH_CRYPTOSYM_AES_BLOCK_SIZE - 1U) /                       \
             PH_CRYPTOSYM_AES_BLOCK_SIZE) * PH_CRYPTOSYM_AES_BLOCK_SIZE) -          \
             PH_CRYPTOSYM_AES_BLOCK_SIZE))

#define PHAL_NTAGXDNA_ABS(Val1, Val2)                                               \
    ((Val1) > (Val2) ? (Val1 - Val2) : (Val2 - Val1))

/* Validate APDU Format. */
#define PHAL_NTAGXDNA_VALIDATE_APDU_FORMAT(ApduForamt)                              \
    switch(ApduForamt)                                                              \
    {                                                                               \
        case PHAL_NTAGXDNA_APDU_FORMAT_SHORT_LEN:                                   \
        case PHAL_NTAGXDNA_APDU_FORMAT_EXTENDED_LEN:                                \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);  \
    }

/* Validate File Number. */
#define PHAL_NTAGXDNA_IS_VALID_FILE_NO(FileNo)                                      \
    if((FileNo & 0x7FU) > 0x1FU)                                                    \
    {                                                                               \
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);      \
    }

/* Validate Keys */
#define PHAL_NTAGXDNA_IS_INVALID_KEYNO(KeyNo)                                       \
    (((KeyNo) & 0x0FU) > 0x04U) &&                                                  \
    ((((KeyNo) & 0x1FU) < 0x10U) || (((KeyNo) & 0x1FU) > 0x17U))

/* Validate Keys */
#define PHAL_NTAGXDNA_IS_CRYPTOREQUEST_KEY(KeyNo)                                   \
    (((KeyNo) >= 0x10U) && ((KeyNo) <= 0x17U))

/* Validate File Options. */
#define PHAL_NTAGXDNA_VALIDATE_FILE_OPTIONS(FileOptions)                            \
    /* Validate communication modes. */                                             \
    switch(FileOptions & 0x03U)                                                     \
    {                                                                               \
        case PHAL_NTAGXDNA_FILE_OPTION_PLAIN:                                       \
        case PHAL_NTAGXDNA_FILE_OPTION_MAC:                                         \
        case PHAL_NTAGXDNA_FILE_OPTION_FULL:                                        \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);  \
    }

/* Validate Exchange Options. */
#define PHAL_NTAGXDNA_VALIDATE_RX_EXCHANGE_OPTIONS(Option)                          \
    /* Validate communication modes. */                                             \
    switch(Option)                                                                  \
    {                                                                               \
        case PH_EXCHANGE_DEFAULT:                                                   \
        case PH_EXCHANGE_RXCHAINING:                                                \
            break;                                                                  \
                                                                                    \
        default:                                                                    \
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);  \
    }

#define PHAL_NTAGXDNA_CMD_INVALID                                               0xFFU   /**< INVALID Command code. */
#define PHAL_NTAGXDNA_COMMUNICATION_INVALID                                     0xFFU   /**< Communication mode invalid or not known. */

#define PHAL_NTAGXDNA_PICC_STATUS_INVALID                                       0xFFU   /**< INVALID Status Code to update the PICC response parameter. This is for internal purpose. */
#define PHAL_NTAGXDNA_ADDITIONAL_FRAME                                          0xAFU   /**< Command / Response code for Additional Frame. */
#define PHAL_NTAGXDNA_WRAPPED_HDR_LEN_NORMAL                                    0x05U   /**< ISO7816 Header length. */
#define PHAL_NTAGXDNA_WRAPPED_HDR_LEN_EXTENDED                                  0x07U   /**< ISO7816 Header length in Extended Mode. */

/** NTAG X DNA command options. This flag will be used to compute the response. */
#define PHAL_NTAGXDNA_OPTION_NONE                                               0x00U   /**< Command option as None. This flag is used to discard
                                                                                         *   the processing of reception from PICC.
                                                                                         */
#define PHAL_NTAGXDNA_OPTION_COMPLETE                                           0x01U   /**< Command option as complete. This flag is used to check the response other than AF. */
#define PHAL_NTAGXDNA_OPTION_PENDING                                            0x02U   /**< Command option as complete. This flag is used to check for AF response. */
#define PHAL_NTAGXDNA_OPTION_WRAPPED_EXCHANGE                                   0x00U   /**< Command option to exchange the information to tag in wrapped Format. */
#define PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE                                    0x04U   /**< Command option to exchange the information to tag directly. */

#define PHAL_NTAGXDNA_RETURN_PLAIN_DATA                                         0x04U   /**< Return plain data from SM application interface in case if communication mode is PLAIN. */
#define PHAL_NTAGXDNA_RETURN_PICC_STATUS                                        0x10U   /**< Return the PICC status to the caller. */
#define PHAL_NTAGXDNA_EXCLUDE_PICC_STATUS                                       0x40U   /**< Exclude removal of status code from actual response length. */

#define PHAL_NTAGXDNA_TRUNCATED_MAC_LEN                                         8U      /**< Size of truncated MAC information. */
#define PHAL_NTAGXDNA_DATA_TO_READ_UNKNOWN                                      0U      /**< For all the internal Read Operation calls where Data to be read is not known. */

#define PHAL_NTAGXDNA_ISO7816_GENERIC_CLA                                       0x00U   /**< Class for Generic ISO7816 commands. */
#define PHAL_NTAGXDNA_WRAPPEDAPDU_CLA                                           0x90U   /**< Wrapped APDU code for class. */
#define PHAL_NTAGXDNA_WRAPPEDAPDU_P1                                            0x00U   /**< Wrapped APDU code for default P1. */
#define PHAL_NTAGXDNA_WRAPPEDAPDU_P2                                            0x00U   /**< Wrapped APDU code for default P2. */
#define PHAL_NTAGXDNA_WRAPPEDAPDU_LC                                            0x00U   /**< Wrapped APDU code for default LC. */
#define PHAL_NTAGXDNA_WRAPPEDAPDU_LE                                            0x00U   /**< Wrapped APDU code for default LE. */

#define PHAL_NTAGXDNA_LC_POS                                                    0x04U   /**< Position of LC in ISO7816 format. */
#define PHAL_NTAGXDNA_EXCLUDE_LC_LE                                             0x00U   /**< Option to indicate LC and LE will not be exchange to PICC. */
#define PHAL_NTAGXDNA_EXCHANGE_LC_ONLY                                          0x01U   /**< Option to indicate only LC should be exchanged to PICC and LE should not be exchanged. */
#define PHAL_NTAGXDNA_EXCHANGE_LE_ONLY                                          0x02U   /**< Option to indicate only LE should be exchanged to PICC and LC should not be exchanged. */
#define PHAL_NTAGXDNA_EXCHANGE_LC_LE_BOTH                                       0x03U   /**< Option to indicate both LC and LE should be exchanged to PICC. */
#define PHAL_NTAGXDNA_PRODUCT_CMD                                               0x00U   /**< Option to indicate NTAG X DNA product commands. */
#define PHAL_NTAGXDNA_ISO7816_APDU_CMD                                          0x01U   /**< Option to indicate NTAG X DNA product's Standard ISO7816 APDU commands. */

#define PHAL_NTAGXDNA_COMM_OPTIONS_MASK                                         0xF0U   /**< Masking out communication options. */

#define PHAL_NTAGXDNA_PUBLIC_KEY_LEN                                            65U     /**< Length of Public Key. */
#define PHAL_NTAGXDNA_SHARED_SECRET_LEN                                         32U     /**< Length of Shared Secret. */
#define PHAL_NTAGXDNA_HASH_LEN                                                  32U     /**< Length of SHA256 Hash. */
#define PHAL_NTAGXDNA_SIGNATURE_LEN                                             64U     /**< Length of Signature. */

#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL                            0x01U   /**< IsoGeneralAuthenticate Protocol option as SIGMA_I. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_MASK                     0x03U   /**< IsoGeneralAuthenticate Certificate Depth as EndLeaf. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0                        0x80U   /**< IsoGeneralAuthenticate Certificate Depth as EndLeaf. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_1                        0x81U   /**< IsoGeneralAuthenticate Certificate Depth as Parent. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_2                        0x82U   /**< IsoGeneralAuthenticate Certificate Depth as GrandParent. */

#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128                    0x0001U /**< Iteration count for AES128 Key Size. This will be used while generating Session Keys. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256                    0x0002U /**< Iteration count for AES256 Key Size. This will be used while generating Session Keys. */

#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_L2_AES128                           0x0080U /**< L[2] for AES128 Key Size. This will be used while generating Session Keys. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_L2_AES256                           0x0100U /**< L[2] for AES256 Key Size. This will be used while generating Session Keys. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_L2_IV                               0x0068U /**< L[2] for Initialization Vector. This will be used while generating Session Keys. */

#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_CERT_REQUEST       0x8000U /**< Payload tag for Certificate Request with Length. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE        0x8301U /**< Payload tag for AES Key Size Option with Length. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_CERT_HASH          0x8420U /**< Payload tag for Certificate Hash with Length. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB              0x8641U /**< Payload tag for Ephemeral Public Key with Length. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_ENC                0x8700U /**< Payload tag for Encrypted Payload without Length. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_UNCOMPRESSED_CERT           0x7F21U /**< Payload tag for UnCompressed Certificate without Length. */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_COMPRESSED_CERT             0x7F22U /**< Payload tag for Compressed Certificate without Length. */

#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A0                     0xA0U   /**< Message Type tag for Protocol Options byte and Ephemeral
                                                                                         *   ECDH public key (xP) in plain text exchanged by initiator.
                                                                                         */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A1                     0xA1U   /**< Message Type tag for Cert hash (or full certificate) and signature
                                                                                         *   encrypted with <K_e1, IV_e1>.
                                                                                         */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A2                     0xA2U   /**< Message Type tag for Cert request message
                                                                                         *   encrypted with <K_e1, IV_e1>.
                                                                                         */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A3                     0xA3U   /**< Message Type tag for Certificate (optionally compressed),
                                                                                         *   encrypted with <K_e1, IV_e1>.
                                                                                         */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_AF                     0xAFU   /**< Message Type tag for Protocol Abort by Initiator. */

#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B0                     0xB0U   /**< Message Type tag for Protocol as None */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B1                     0xB1U   /**< Message Type tag for Session symmetric key size and Public
                                                                                         *   key (yP) in plain text, cert hash[1] and signature encrypted with
                                                                                         *   <K_e1, IV_e1>.
                                                                                         */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B2                     0xB2U   /**< Message Type tag for Certificate request message encrypted with <K_e1, IV_e1> */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B3                     0xB3U   /**< Message Type tag for Certificate (optionally compressed),
                                                                                         *   encrypted with <K_e1, IV_e1>.
                                                                                         */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B4                     0xB4U   /**< Message Type tag for Authentication Successful by Device as responder and returns
                                                                                         *   control to Host for Secure tunnel rules application.
                                                                                         */
#define PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_BF                     0xBFU   /**< Message Type tag for Protocol Abort by Responder. */

#define PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_AUTH_DO_HDR_TAG                         0x7CU   /**< Tag data for ISOInternal Authenticate AuthDOHdr information. */
#define PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_RND_TAG                                 0x81U   /**< Tag data for ISOInternal Authenticate RndA / RndB information. */
#define PHAL_NTAGXDNA_AUTH_ISO_INTERNAL_SIGNATURE_TAG                           0x82U   /**< Tag data for ISOInternal Authenticate Signature information. */

#define PHAL_NTAGXDNA_SESSION_MAC                                               0x01U   /**< Session key option for Macing of data. */
#define PHAL_NTAGXDNA_SESSION_ENC                                               0x02U   /**< Session key option for Encryption / Decryption of data. */

#define PHAL_NTAGXDNA_PROCESS_SM_ACTION_APPLY                                   0x01U   /**< Process SM action as Apply. */
#define PHAL_NTAGXDNA_PROCESS_SM_ACTION_REMOVE                                  0x02U   /**< Process SM action as Remove. */
#define PHAL_NTAGXDNA_PROCESS_SM_OPERATION_ONE_SHOT                             0x04U   /**< Process SM operation as one-shot. */

/* BER-TLV Length Constants. */
#define PHAL_NTAGXDNA_ISO7816_BER_TLV_L_NO_CONST                                0x01U   /**< BER-TLV constant length formats to total of 1 bytes which includes only length information upto 127 bytes. */

#define PHAL_NTAGXDNA_ISO7816_BER_TLV_C_81                                      0x81U   /**< BER-TLV constant length data if information to be exchanged is between 00 to 255 bytes. */
#define PHAL_NTAGXDNA_ISO7816_BER_TLV_L_81                                      0x02U   /**< BER-TLV constant length formats to total of 2 bytes which includes Constant and Actual Length. */

#define PHAL_NTAGXDNA_ISO7816_BER_TLV_C_82                                      0x82U   /**< BER-TLV constant length data if information to be exchanged is between 0000 to 65535 bytes. */
#define PHAL_NTAGXDNA_ISO7816_BER_TLV_L_82                                      0x03U   /**< BER-TLV constant length formats to total of 3 bytes which includes Constant and Actual Length. */

/* CryptoAPI */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_COMMAND_BUFFER                              0x00U   /**< CryptoRequest Source or Destination buffer as Command Buffer. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_VERIFY_SUCCESS                              0x5AU   /**< CryptoRequest Verify with response for successful signature verification. */

#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_SHA                                 0x01U   /**< CryptoRequest Action as SHA. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_RNG                                 0x02U   /**< CryptoRequest Action as RNG. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECC_SIGN                            0x03U   /**< CryptoRequest Action as ECC Sign. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECC_VERIFY                          0x04U   /**< CryptoRequest Action as ECC Verify. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECC_DH                              0x05U   /**< CryptoRequest Action as ECC DH. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES                                 0x06U   /**< CryptoRequest Action as AES Encryption / Decryption. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_WRITE_INTERNAL_BUFFER               0x07U   /**< CryptoRequest Action as Write Internal Buffer. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_HMAC                                0x08U   /**< CryptoRequest Action as HMAC. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_HKDF                                0x09U   /**< CryptoRequest Action as HKDF. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_CMAC_SIGN                       0x0AU   /**< CryptoRequest Action as AES CMAC generation. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_CMAC_VERIFY                     0x0BU   /**< CryptoRequest Action as AES CMAC verification. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_AEAD_ENCRYPT_SIGN               0x0CU   /**< CryptoRequest Action as AES AEAD Encrypt / Sign. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_AES_AEAD_DECRYPT_VERIFY             0x0DU   /**< CryptoRequest Action as AES AEAD Decrypt / Verify. */
#define PHAL_NTAGXDNA_CRYPTOREQUEST_ACTION_ECHO                                0xFDU   /**< CryptoRequest Action as ECHO. */

/**
* \addtogroup phalNtagXDna_SecureMessaging_Defines
* @{
*/

/**
 * \defgroup phalNtagXDna_SecureMessaging_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL                              0x86U   /**< NTAG X DNA ISOGeneral Authenticate command Code. */
#define PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_INTERNAL                             0x88U   /**< NTAG X DNA ISOInternal Authenticate command Code. */
#ifdef NXPBUILD__PH_CRYPTOSYM
#define PHAL_NTAGXDNA_CMD_AUTHENTICATE_EV2_FIRST                                0x71U   /**< NTAG X DNA Authenticate EV2 First command Code. */
#define PHAL_NTAGXDNA_CMD_AUTHENTICATE_EV2_NON_FIRST                            0x77U   /**< NTAG X DNA Authenticate EV2 Non First command Code. */
#endif /* NXPBUILD__PH_CRYPTOSYM */
#define PHAL_NTAGXDNA_CMD_PROCESS_SM                                            0xE5U   /**< NTAG X DNA Process SM command Code. */
/**
 * end of group phalNtagXDna_SecureMessaging_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_SecureMessaging_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_MemoryConfiguration_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_MemoryConfiguration_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_FREE_MEM                                              0x6EU   /**< NTAG X DNA Free Memory command Code. */
#ifdef NXPBUILD__PH_CRYPTOSYM
#define PHAL_NTAGXDNA_CMD_SET_CONFIGURATION                                     0x5CU   /**< NTAG X DNA Set Configuration command code. */
#define PHAL_NTAGXDNA_CMD_GET_CONFIGURATION                                     0x65U   /**< NTAG X DNA Get Configuration command code. */
#endif /* NXPBUILD__PH_CRYPTOSYM */
#define PHAL_NTAGXDNA_CMD_ACTIVATE_CONFIGURATION                                0x66U   /**< NTAG X DNA Get Configuration command code. */
#define PHAL_NTAGXDNA_CMD_GET_VERSION                                           0x60U   /**< NTAG X DNA GetVersion command code. */
#ifdef NXPBUILD__PH_CRYPTOSYM
#define PHAL_NTAGXDNA_CMD_GET_CARD_UID                                          0x51U   /**< NTAG X DNA GetUID command code. */
#endif /* NXPBUILD__PH_CRYPTOSYM */
/**
 * end of group phalNtagXDna_MemoryConfiguration_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_MemoryConfiguration_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_Symm_KeyManagement_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_Symm_KeyManagement_Defines_CommandCodes CommandCodes
 * @{
 */
#ifdef NXPBUILD__PH_CRYPTOSYM
#define PHAL_NTAGXDNA_CMD_CHANGE_KEY                                            0xC4U   /**< NTAG X DNA ChangeKey command code. */
#endif /* NXPBUILD__PH_CRYPTOSYM */
#define PHAL_NTAGXDNA_CMD_GET_KEY_SETTINGS                                      0x45U   /**< NTAG X DNA GetKeySettings command code. */
#define PHAL_NTAGXDNA_CMD_GET_KEY_VERSION                                       0x64U   /**< NTAG X DNA GetKeyVersion command code. */
/**
 * end of group phalNtagXDna_Symm_KeyManagement_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_Symm_KeyManagement_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_ASymm_KeyManagement_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_ASymm_KeyManagement_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_MANAGE_KEY_PAIR                                       0x46U   /**< NTAG X DNA ManageKeyPair command code. */
#define PHAL_NTAGXDNA_CMD_MANAGE_CA_ROOT_KEY                                    0x48U   /**< NTAG X DNA ManageCARootKey command code. */
#define PHAL_NTAGXDNA_CMD_MANAGE_EXPORT_KEY                                     0x47U   /**< NTAG X DNA ExportKey command code. */
/**
 * end of group phalNtagXDna_ASymm_KeyManagement_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_ASymm_KeyManagement_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_CertManagement_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_CertManagement_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_MANAGE_CERT_REPO                                      0x49U   /**< NTAG X DNA ManageCertRepo command code. */
#define PHAL_NTAGXDNA_CMD_READ_CERT_REPO                                        0x4AU   /**< NTAG X DNA ReadCertRepo command code. */
/**
 * end of group phalNtagXDna_CertManagement_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_CertManagement_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_FileManagement_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_FileManagement_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_CREATE_STANDARD_DATA_FILE                             0xCDU   /**< NTAG X DNA Create Standard Data File command code. */
#define PHAL_NTAGXDNA_CMD_CREATE_COUNTER_FILE                                   0xD0U   /**< NTAG X DNA Create Counter File command code. */
#define PHAL_NTAGXDNA_CMD_GET_FILE_IDS                                          0x6FU   /**< NTAG X DNA Create Get File ID's command code. */
#define PHAL_NTAGXDNA_CMD_GET_ISO_FILE_IDS                                      0x61U   /**< NTAG X DNA Create Get ISO File ID's command code. */
#define PHAL_NTAGXDNA_CMD_GET_FILE_SETTINGS                                     0xF5U   /**< NTAG X DNA Create Get File Settings command code. */
#define PHAL_NTAGXDNA_CMD_GET_FILE_COUNTERS                                     0xF6U   /**< NTAG X DNA Create Get File Counters command code. */
#define PHAL_NTAGXDNA_CMD_CHANGE_FILE_SETTINGS                                  0x5FU   /**< NTAG X DNA Create Change File Settings command code. */
/**
 * end of group phalNtagXDna_FileManagement_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_FileManagement_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_DataManagement_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_DataManagement_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_READ_DATA                                             0xADU   /**< NTAG X DNA Read Data command code in ISO/IEC 14443-4 chaining format. */
#define PHAL_NTAGXDNA_CMD_WRITE_DATA                                            0x8DU   /**< NTAG X DNA Write Data command code in ISO/IEC 14443-4 chaining format. */
#define PHAL_NTAGXDNA_CMD_INCREMENT_COUNTER_FILE                                0xF8U   /**< NTAG X DNA Increment Counter File command code. */
/**
 * end of group phalNtagXDna_DataManagement_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_DataManagement_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_CryptoAPI_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_CryptoAPI_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_CRYPTO_REQUEST                                        0x4CU   /**< NTAG X DNA CryptRequest command code. */
/**
 * end of group phalNtagXDna_CryptoAPI_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_CryptoAPI_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_GPIOManagement_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_GPIOManagement_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_MANAGE_GPIO                                           0x42U   /**< NTAG X DNA Manage GPIO command code. */
#define PHAL_NTAGXDNA_CMD_READ_GPIO                                             0x43U   /**< NTAG X DNA Read GPIO command code. */
/**
 * end of group phalNtagXDna_GPIOManagement_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_GPIOManagement_Defines
 * @}
 */

/**
 * \addtogroup phalNtagXDna_ISO7816_Defines
 * @{
 */

/**
 * \defgroup phalNtagXDna_ISO7816_Defines_CommandCodes CommandCodes
 * @{
 */
#define PHAL_NTAGXDNA_CMD_ISO7816_SELECT_FILE                                   0xA4U   /**< NTAG X DNA ISOSelectFile command of ISO7816-4 Standard. */
#define PHAL_NTAGXDNA_CMD_ISO7816_READ_BINARY                                   0xB0U   /**< NTAG X DNA ISOReadBinary command of ISO7816-4 Standard. */
#define PHAL_NTAGXDNA_CMD_ISO7816_UPDATE_BINARY                                 0xD6U   /**< NTAG X DNA ISOUpdateBinary command of ISO7816-4 Standard. */
/**
 * end of group phalNtagXDna_ISO7816_Defines_CommandCodes
 * @}
 */

/**
 * end of group phalNtagXDna_ISO7816_Defines
 * @}
 */

phStatus_t phalNtagXDna_Int_ComputeErrorResponse(void * pDataParams, uint16_t wStatus);

void phalNtagXDna_Int_RotateLeft(uint8_t * pData, uint8_t bDataLen, uint8_t bTimes);

void phalNtagXDna_Int_RotateRight(uint8_t * pData, uint8_t bDataLen, uint8_t bTimes);

void phalNtagXDna_Int_GetCommMode(uint8_t bAuthState, uint8_t bOption, uint8_t * pCommMode);

phStatus_t phalNtagXDna_Int_Validate_ComOption(uint8_t bComOption);

void phalNtagXDna_Int_EncodeBER_TLV_Len(uint8_t * pBuffer, uint16_t * pBuffLen, uint16_t wAddLen);

phStatus_t phalNtagXDna_Int_DecodeBER_TLV_Len(uint8_t ** ppBuffer, uint16_t * pBER_TLV_Len, uint16_t * pRspLen);

void phalNtagXDna_Int_UpdateLC(uint8_t * pData, uint16_t wDataLen, uint8_t bLE_Len);

#endif /* NXPBUILD__PHAL_NTAGXDNA */

#endif /* PHALNTAGXDNA_INT_H */
