/*
 * Copyright 2024 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Software implementation of NTAG X DNA application layer.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7462 $
 * $Date: 2025-08-29 14:09:06 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *  Rajendran Kumar: Generated 22 Aug 2024
 *
 */

#include <ph_Status.h>

#ifdef NXPBUILD__PHAL_NTAGXDNA_SW

#ifdef NXPBUILD__PH_CRYPTOSYM
#include <phKeyStore.h>
#include <phCryptoASym.h>
#include <phCryptoSym.h>
#endif /* NXPBUILD__PH_CRYPTOSYM */

#include <phpalMifare.h>
#include <phpalI14443p4.h>

#include "phalNtagXDna_Sw_Int.h"
#include "../phalNtagXDna_Int.h"





#ifdef NXPBUILD__PH_CRYPTOSYM
uint8_t PH_MEMLOC_CONST_ROM phalNtagXDna_Sw_ZeroIv[PH_CRYPTOSYM_AES_BLOCK_SIZE];
#endif /*NXPBUILD__PH_CRYPTOSYM */

static const uint16_t PH_MEMLOC_CONST_ROM aFrameSize[] = { 16U, 24U, 32U, 40U, 48U, 64U, 96U, 128U, 256U };





phStatus_t phalNtagXDna_Sw_Int_ValidateResponse(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wStatus,
    uint16_t wPiccRetCode)
{
    uint16_t PH_MEMLOC_REM wPICCStatusCode = 0;

    /* Evaluate the response. */
    if(wStatus == PH_ERR_SUCCESS)
    {
        /* Frame PICC Status Code. */
        wPICCStatusCode = (uint16_t) ((bOption == PHAL_NTAGXDNA_ISO7816_APDU_CMD) ? wPiccRetCode : (wPiccRetCode & 0x00FFU));

        /* Validate the PICC Status. */
        PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_ComputeErrorResponse(pDataParams, wPICCStatusCode));
    }
    else
    {
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_AL_NTAGXDNA);
        }

        PH_CHECK_SUCCESS(wStatus);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_Int_CardExchange(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wBufferOption,
    uint8_t bCmdOption, uint16_t wTotDataLen, uint8_t bExchangeLE, uint8_t * pData, uint16_t wDataLen,
    uint8_t ** ppResponse, uint16_t * pRespLen, uint8_t * pPiccErrCode)
{
    phStatus_t      PH_MEMLOC_REM wStatus = 0;
    phStatus_t      PH_MEMLOC_REM wPICCStatus = 0;
    uint16_t        PH_MEMLOC_REM wLc = 0;
    uint16_t        PH_MEMLOC_REM wRspLen = 0;

    uint8_t         PH_MEMLOC_REM bPICCStatLen = 0;
    uint8_t         PH_MEMLOC_REM bISO7816HeaderLen = 4U;
    uint8_t         PH_MEMLOC_REM bCheckStatus = 0;
    uint8_t         PH_MEMLOC_REM bLcLen = 0;
    static uint8_t  PH_MEMLOC_REM bLeLen;

    uint8_t         PH_MEMLOC_REM aLe[3] = { 0x00, 0x00, 0x00 };
    uint8_t         PH_MEMLOC_REM aISO7816Header[8] =
    {
        PHAL_NTAGXDNA_WRAPPEDAPDU_CLA,
        0x00,
        PHAL_NTAGXDNA_WRAPPEDAPDU_P1,
        PHAL_NTAGXDNA_WRAPPEDAPDU_P2,
        0x00,
        0x00,
        0x00
    };
    uint8_t*        PH_MEMLOC_REM pResponse = NULL;

    /* Exchange information to tag in wrapped format */
    if((bCmdOption & PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE) == 0U)
    {
        if((wBufferOption == PH_EXCHANGE_BUFFER_FIRST) || (wBufferOption == PH_EXCHANGE_DEFAULT))
        {
            bLeLen = 1;

            /* Set the LC information. */
            wLc = (uint16_t) (wTotDataLen - 1 /* Excluding the command code. */);

            /* Update the command code to Iso7816 header */
            aISO7816Header[1] = pData[0];

            /* Add LC if available */
            if(wLc)
            {
                /* Update LC bytes according to Extended APDU option. */
                if(pDataParams->bShortLenApdu == PH_OFF)
                {
                    aISO7816Header[bISO7816HeaderLen + bLcLen++] = 0x00U;
                    aISO7816Header[bISO7816HeaderLen + bLcLen++] = (uint8_t) ((wLc & 0x0000FF00U) >> 8U);

                    /* Le length is updated to two if LC is present and the APDU is extended. */
                    bLeLen = 2U;
                }

                aISO7816Header[bISO7816HeaderLen + bLcLen++] = (uint8_t) (wLc & 0x000000FFU);

                /* Update IHeader Length */
                bISO7816HeaderLen += bLcLen;
            }
            else
            {
                /* Update Le count */
                if(pDataParams->bShortLenApdu == PH_OFF)
                {
                    bLeLen = 3U;
                }
            }

            /* Add the ISO 7816 header to layer 4 buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                pDataParams->pPalMifareDataParams,
                PH_EXCHANGE_BUFFER_FIRST,
                &aISO7816Header[0],
                bISO7816HeaderLen,
                NULL,
                NULL));

            /* Add the data to layer 4 buffer. */
            if((wDataLen - 1U) != 0U)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                    pDataParams->pPalMifareDataParams,
                    PH_EXCHANGE_BUFFER_CONT,
                    &pData[1U],  /* Exclude the command code because it is added to INS. */
                    (uint16_t) (wDataLen - 1U),
                    NULL,
                    NULL));
            }
        }

        if(wBufferOption == PH_EXCHANGE_BUFFER_CONT)
        {
            /* Add the data to layer 4 buffer. */
            PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                pDataParams->pPalMifareDataParams,
                PH_EXCHANGE_BUFFER_CONT,
                pData,
                wDataLen,
                NULL,
                NULL));
        }

        if((wBufferOption == PH_EXCHANGE_BUFFER_LAST) || (wBufferOption == PH_EXCHANGE_DEFAULT))
        {
            if(wBufferOption == PH_EXCHANGE_BUFFER_LAST)
            {
                /* Add the data to layer 4 buffer. */
                PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
                    pDataParams->pPalMifareDataParams,
                    PH_EXCHANGE_BUFFER_CONT,
                    pData,
                    wDataLen,
                    NULL,
                    NULL));
            }

            /* Add Le to L4 buffer and exchange the command. */
            wStatus = phpalMifare_ExchangeL4(
                pDataParams->pPalMifareDataParams,
                PH_EXCHANGE_BUFFER_LAST,
                &aLe[0],
                (uint8_t) (bExchangeLE ? bLeLen : 0),
                &pResponse,
                &wRspLen);

            /* Validate the Status. */
            if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
            {
                /* Update PICC Error code to INVALID. */
                if(pPiccErrCode != NULL)
                    *pPiccErrCode = PHAL_NTAGXDNA_PICC_STATUS_INVALID;

                return wStatus;
            }

            /* Should the status needs to be verified. */
            bCheckStatus = (uint8_t) ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS);

            /* Combine Sw1 and Sw2 status codes. */
            if(bCheckStatus)
                wPICCStatus = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

            /* Evaluate the Status. */
            wStatus = phalNtagXDna_Sw_Int_ValidateResponse(pDataParams, PHAL_NTAGXDNA_PRODUCT_CMD, wStatus, wPICCStatus);

            /* Create memory for updating the response of ISO 14443 format. */
            *ppResponse = pResponse;

            /* Update the response buffer length excluding SW1SW2. */
            *pRespLen = (uint16_t) (wRspLen - (bCheckStatus ? 2U : 0));

            /* Copy the second byte of response (SW2) to RxBuffer */
            if(bCmdOption & PHAL_NTAGXDNA_RETURN_PICC_STATUS)
                if(pPiccErrCode != NULL)
                    *pPiccErrCode = pResponse[wRspLen - 1U];
        }
    }

    /* Exchange information to tag directly */
    else
    {
        /* Exchange the data to the card in Native format. */
        wStatus = phpalMifare_ExchangeL4(
            pDataParams->pPalMifareDataParams,
            wBufferOption,
            pData,
            wDataLen,
            &pResponse,
            &wRspLen);

        /* Validate the Status. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            /* Update PICC Error code to INVALID. */
            if(pPiccErrCode != NULL)
                *pPiccErrCode = PHAL_NTAGXDNA_PICC_STATUS_INVALID;

            return wStatus;
        }

        /* Verify the received data and update the response buffer with received data. */
        if((bCmdOption & PHAL_NTAGXDNA_OPTION_PENDING) ||
            (bCmdOption & PHAL_NTAGXDNA_OPTION_COMPLETE))
        {
            /* Combine Sw1 and Sw2 status codes. */
            wPICCStatus = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);
            bPICCStatLen = 2U;
        }

        /* Evaluate the Status. */
        wStatus = phalNtagXDna_Sw_Int_ValidateResponse(pDataParams, PHAL_NTAGXDNA_ISO7816_APDU_CMD, wStatus,
            wPICCStatus);

        if(pRespLen != NULL)
        {
            /* Update the response buffer length excluding PICC Status Code. */
            *pRespLen = wRspLen - bPICCStatLen;

            /* Add the Response data excluding PICC Status Code. */
            *ppResponse = pResponse;
        }
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_Send7816Apdu(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint16_t wBufOption,
    uint8_t bExtendedLenApdu, uint8_t bClass, uint8_t bIns, uint8_t bP1, uint8_t bP2, uint8_t * pData,
    uint16_t wDataLen, uint32_t dwExpBytes, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM aLe[3] = { 0x00, 0x00, 0x00 };
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;

    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint8_t     PH_MEMLOC_REM bLeLen = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Standard ISO7816 - 4 Header. */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bClass;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bIns;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bP1;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bP2;

    /* Check whether LC needs to be exchanged. */
    if(bOption & PHAL_NTAGXDNA_EXCHANGE_LC_ONLY)
    {
        /* Check whether Length LC is represented in short APDU or extended APDU */
        if(bExtendedLenApdu == PH_ON)
        {
            /* First byte will be 0x00 if Ext APDU present. Next 2 byte contains actual data. */
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x00;

            /* As of now this field will be set to 0x00 since maximum data that can be sent is 16 bytes.
            * In case if data to be sent exceeds 255 bytes, this byte shall be used.
            */
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((wDataLen & 0xFF00U) >> 8U);
        }

        /* Short Length APDU. */
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (wDataLen & 0x00FFU);
    }

    /* Buffer ISO7816-4 Command Header. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        &pResponse,
        &wRspLen));

    /* Check whether Le needs to be exchanged. */
    if(bOption & PHAL_NTAGXDNA_EXCHANGE_LE_ONLY)
    {
        /* As per ISO/IEC:7816-4(2005), Section 5, An extended LE field consists of either three bytes
         * (one byte set to '00' followed by two bytes with any value) if the LC field is absent, or
         * two bytes (with any value) if an extended LC field is present.
         */

        /* Check whether Length is represented in extended APDU format and LC is present.
         * If true, then Le should represented in 2 bytes else LE should be represented in 3 bytes
         */
        if(bExtendedLenApdu == PH_ON)
        {
            if(!(bOption & PHAL_NTAGXDNA_EXCHANGE_LC_ONLY))
            {
                aLe[bLeLen++] = (uint8_t) ((dwExpBytes & 0x00FF0000U) >> 16U);
            }

            aLe[bLeLen++] = (uint8_t) ((dwExpBytes & 0x0000FF00U) >> 8U);
        }

        /* Short APDU */
        aLe[bLeLen++] = (uint8_t) (dwExpBytes & 0x000000FFU);
    }

    /* Exchange the command based on the INS. */
    switch(bIns)
    {
        case PHAL_NTAGXDNA_CMD_ISO7816_SELECT_FILE:
            wStatus = phalNtagXDna_Sw_Int_ISOSelectFile(pDataParams, pData, wDataLen, aLe, bLeLen, ppResponse, pRspLen);
            break;

        case PHAL_NTAGXDNA_CMD_ISO7816_READ_BINARY:
            wStatus = phalNtagXDna_Sw_Int_ISOReadBinary(pDataParams, wBufOption, aLe, bLeLen, ppResponse, pRspLen);
            break;

        case PHAL_NTAGXDNA_CMD_ISO7816_UPDATE_BINARY:
            wStatus = phalNtagXDna_Sw_Int_ISOUpdateBinary(pDataParams, pData, wDataLen);
            break;

            /*
            * This case cannot be achieved as the INS that are implemented are the supported ones only.
            * This case is kept for completeness of the switch statement and to avoid error while
            * implementing new command support in future.
            */
        default:
            wStatus = PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_AL_NTAGXDNA);
            break;
    }

    return wStatus;
}





#ifdef NXPBUILD__PH_CRYPTOSYM
/* ISOGeneral Authenticate With Host as Initiator and Device as Responder -------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bDeviceCert_Cached,
    uint8_t bCertRepoID, uint8_t bCert_Depth, uint8_t bKeySize_Host, uint16_t wPriv_KeyNo, uint16_t wPriv_KeyPos, uint16_t wPub_KeyNo,
    uint16_t wPub_KeyPos, uint8_t * pExpRspLen, uint8_t bExpRspLen, phalNtagXDna_Cert stHostCert, phalNtagXDna_Cert * pDeviceCert)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wTagLen = 0;
    uint8_t     PH_MEMLOC_REM bCurveID = 0;
    uint8_t     PH_MEMLOC_REM bEndLoop = PH_OFF;
    uint8_t     PH_MEMLOC_REM bKeySize_Device = 0;
    uint8_t     PH_MEMLOC_REM bDevice_EndLeafCert_Available = PH_OFF;
    uint8_t     PH_MEMLOC_REM bVerification_Complete = PH_OFF;
    uint8_t     PH_MEMLOC_REM aCertDepth[2U] = { PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0, 0x00U };

    uint8_t     PH_MEMLOC_REM * pEPub_Host = NULL;
    uint16_t    PH_MEMLOC_REM wEPub_HostLen = 0;

    uint8_t     PH_MEMLOC_REM * pEPub_Device = NULL;
    uint16_t    PH_MEMLOC_REM wEPub_DeviceLen = 0;

    uint8_t     PH_MEMLOC_REM aSharedSecret[PHAL_NTAGXDNA_SHARED_SECRET_LEN];
    uint16_t    PH_MEMLOC_REM wSharedSecretLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM * pHost_Cert = NULL;
    uint16_t    PH_MEMLOC_REM wHost_CertLen = 0;

    uint8_t     PH_MEMLOC_REM aCertHash_Device[PHAL_NTAGXDNA_HASH_LEN];
    uint16_t    PH_MEMLOC_REM wCertHash_Device_Len = 0;

    uint8_t     PH_MEMLOC_REM aSig_Device[PHAL_NTAGXDNA_SIGNATURE_LEN];
    uint16_t    PH_MEMLOC_REM wSig_Device_Len = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Set the Pointers */
    pEPub_Host = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PHAL_NTAGXDNA_PUBLIC_KEY_LEN * 2U)];
    pEPub_Device = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - PHAL_NTAGXDNA_PUBLIC_KEY_LEN];

    /* Extract Host Ephemeral Public Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_ExportKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        PHAL_NTAGXDNA_PUBLIC_KEY_LEN,
        &bCurveID,
        pEPub_Host,
        &wEPub_HostLen));

    /* Send Host Key Size and Ephemeral Public Key to Device. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A0(
        pDataParams,
        bCertRepoID,
        &bKeySize_Host,
        pEPub_Host,
        &wEPub_HostLen,
        pExpRspLen,
        bExpRspLen,
        &pResponse,
        &wRespLen));

    /* Extract Device Ephemeral Public Key and Encrypted Message from received information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_B1(
        bKeySize_Host,
        pResponse,
        wRespLen,
        &bKeySize_Device,
        pEPub_Device,
        &wEPub_DeviceLen,
        &pResponse,
        &wRespLen));

    /* Compute Shared Secret. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_SharedSecret(
        pDataParams->pCryptoDataParamsASym,
        bCurveID,
        pEPub_Device,
        wEPub_DeviceLen,
        aSharedSecret,
        &wSharedSecretLen));

    /* Generate SessionKeys */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_GenerateSessionKeys_ASym(
        pDataParams,
        bKeySize_Host,
        pEPub_Host,
        wEPub_HostLen,
        pEPub_Device,
        wEPub_DeviceLen,
        aSharedSecret,
        wSharedSecretLen));

    /* Save configured CCM Authentication Tag Length */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_GetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_CCM_TAG_LENGTH,
        &wTagLen));

    /* Decrypt Response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
        PH_CRYPTOSYM_AUTH_TAG_ON),
        pResponse,
        wRespLen,
        pResponse));

    /* Remove CCM Authentication Tag */
    wRespLen -= wTagLen;

    /* Extract Device Certificate Hash */
    wCertHash_Device_Len = PHAL_NTAGXDNA_HASH_LEN;
    (void) memcpy(aCertHash_Device, pResponse, wCertHash_Device_Len);

    /* Extract Device Signature */
    wSig_Device_Len = (uint16_t) (wRespLen - wCertHash_Device_Len);
    (void) memcpy(aSig_Device, &pResponse[wCertHash_Device_Len], wSig_Device_Len);

    /* Request for Device Certificates */
    do
    {
        /* Request for Device End-Leaf Certificate if not cached */
        if(bDeviceCert_Cached == PH_OFF)
        {
            /* Frame and exchange Request for Device Certificate */
            wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A2(
                pDataParams,
                bCertRepoID,
                wTagLen,
                aCertDepth,
                pExpRspLen,
                2U,
                &pResponse,
                &wRespLen);

            /* End the loop in case of failure */
            if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bEndLoop = PH_ON;
            }
            else
            {
                /* Decrypt the received information and extract the certificate */
                wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_B3(
                    pDataParams,
                    pResponse,
                    wRespLen,
                    &pResponse,
                    &wRespLen);

                /* End the loop in case of failure */
                if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    bEndLoop = PH_ON;
                }
                else
                {
                    /* Copy End-Leaf Certificate */
                    if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0)
                    {
                        (void) memcpy(pDeviceCert->pEndLeaf, pResponse, wRespLen);
                        pDeviceCert->wEndLeaf_Len = wRespLen;

                        bDevice_EndLeafCert_Available = PH_ON;
                    }

                    /* Copy Parent Certificate */
                    else if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_1)
                    {
                        (void) memcpy(pDeviceCert->pParent, pResponse, wRespLen);
                        pDeviceCert->wParent_Len = wRespLen;
                    }

                    /* Copy Grand-Parent Certificate */
                    else
                    {
                        (void) memcpy(pDeviceCert->pGrandParent, pResponse, wRespLen);
                        pDeviceCert->wGrandParent_Len = wRespLen;
                    }

                    /* Update Depth to receive Device Parent and Grand-Parent Certificate(s) */
                    if(bCert_Depth == (uint8_t) (aCertDepth[0] & PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_MASK))
                    {
                        bEndLoop = PH_ON;
                    }
                    else
                    {
                        aCertDepth[0]++;
                    }
                }
            }
        }
        else
        {
            bDevice_EndLeafCert_Available = PH_ON;
            bEndLoop = PH_ON;
        }

        /* Verifiy Device Certificate Hash and Signature */
        if((bVerification_Complete == PH_OFF) && (bDevice_EndLeafCert_Available == PH_ON))
        {
            /* Verify Certificate Hash received from Device */
            wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_VerifyCertHash(
                pDataParams,
                aCertHash_Device,
                wCertHash_Device_Len,
                pDeviceCert->pEndLeaf,
                pDeviceCert->wEndLeaf_Len);

            /* End the loop in case of failure */
            if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bEndLoop = PH_ON;
            }

            /* Verify Signature received from Device */
            else
            {
                wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_VerifySignature(
                    pDataParams,
                    PH_OFF,
                    wPub_KeyNo,
                    wPub_KeyPos,
                    bKeySize_Host,
                    bKeySize_Device,
                    aCertHash_Device,
                    (uint8_t) wCertHash_Device_Len,
                    pEPub_Host,
                    wEPub_HostLen,
                    pEPub_Device,
                    wEPub_DeviceLen,
                    aSig_Device,
                    wSig_Device_Len);

                /* Set verification completion flag. */
                bVerification_Complete = PH_ON;

                /* End the loop in case of failure */
                if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    bEndLoop = PH_ON;
                }
            }
        }
    } while(bEndLoop != PH_ON);

    /* Check the status */
    PH_CHECK_SUCCESS(wStatus);

    /* Clear Loop Flag */
    bEndLoop = PH_OFF;

    /* Reset Certificate Depth */
    aCertDepth[0] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0;

    /* Exchange Host End-Leaf Certificate Hash and Signature */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A1(
        pDataParams,
        bCertRepoID,
        wTagLen,
        wPriv_KeyNo,
        wPriv_KeyPos,
        bKeySize_Host,
        bKeySize_Device,
        stHostCert.pEndLeaf,
        stHostCert.wEndLeaf_Len,
        pEPub_Host,
        wEPub_HostLen,
        pEPub_Device,
        wEPub_DeviceLen,
        pExpRspLen,
        bExpRspLen,
        &pResponse,
        &wRespLen));

    /* Check for Abort signal from Device */
    if(pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_BF)
    {
        /* Return Abort to user */
        return PH_ADD_COMPCODE(PH_ERR_ABORTED, PH_COMP_AL_NTAGXDNA);
    }

    /* Exclude device Certificate request. */
    if(pResponse[0] != PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B4)
    {
        /* Exchange Host Certificate(s) */
        do
        {
            /* Decrypt Response to get the Certificate depth information */
            wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_B2(
                pDataParams,
                pResponse,
                wRespLen,
                aCertDepth);

            /* End the loop in case of failure */
            if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bEndLoop = PH_ON;
            }
            else
            {
                /* Copy End-Leaf Certificate */
                if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0)
                {
                    pHost_Cert = stHostCert.pEndLeaf;
                    wHost_CertLen = stHostCert.wEndLeaf_Len;
                }

                /* Copy Parent Certificate */
                else if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_1)
                {
                    pHost_Cert = stHostCert.pParent;
                    wHost_CertLen = stHostCert.wParent_Len;
                }

                /* Copy Grand-Parent Certificate */
                else
                {
                    pHost_Cert = stHostCert.pGrandParent;
                    wHost_CertLen = stHostCert.wGrandParent_Len;
                }

                /* Encrypt Host Certificate and Exchange */
                wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A3(
                    pDataParams,
                    bCertRepoID,
                    wTagLen,
                    pHost_Cert,
                    wHost_CertLen,
                    pExpRspLen,
                    2U,
                    &pResponse,
                    &wRespLen);

                /* End the loop in case of failure */
                if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    bEndLoop = PH_ON;
                }
                else
                {
                    /* End the Loop */
                    if((pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_BF) ||
                        (pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B4))
                    {
                        bEndLoop = PH_ON;

                        /* Return Abort to user */
                        if(pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_BF)
                        {
                            wStatus = PH_ADD_COMPCODE(PH_ERR_ABORTED, PH_COMP_AL_NTAGXDNA);
                        }
                    }
                }
            }
        } while(bEndLoop != PH_ON);

        /* Check the status */
        PH_CHECK_SUCCESS(wStatus);
    }
    else
    {
        /*
         * Request for Device certificate is not required as the certificates are
         * cached during previous authentication call.
         */
    }

    /* Reset Command Counter. */
    pDataParams->wCmdCtr = 0x00;

    /* Reset Transaction Identifier */
    (void) memset(pDataParams->aTi, 0x00, sizeof(pDataParams->aTi));

    /* Reset Nonce */
    (void) memset(pDataParams->aSesNonce, 0x00, sizeof(pDataParams->aSesNonce));

    pDataParams->bAuthState = PHAL_NTAGXDNA_ECC_AUTHENTICATED;
    pDataParams->wKeyType = (uint16_t) ((bKeySize_Host == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES128) ? PH_KEYSTORE_KEY_TYPE_AES128 :
        PH_KEYSTORE_KEY_TYPE_AES256);

    /* Load the Session ENC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pDataParams->aSesAuthENCKey,
        pDataParams->wKeyType));

    /* Load the Session MAC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pDataParams->aSesAuthMACKey,
        pDataParams->wKeyType));

    /* Set the keep Iv ON */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_KEEP_IV,
        PH_CRYPTOSYM_VALUE_KEEP_IV_ON));

    /* Clear the local buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUFFER_SIZE_MINIMUM);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;
    PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUFFER_SIZE_MINIMUM);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    PHAL_NTAGXDNA_PRS_BUF_OFFSET = 0;

    PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE = PH_OFF;
    PHAL_NTAGXDNA_HAS_MAC_PROCESSED = PH_OFF;

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A0(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint8_t *pKeySize, uint8_t * pEPubKey, uint16_t * pEPubKeyLen, uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse,
    uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Cmd.ISOGeneralAuthenticate A0 Payload and Exchange to Device -------------------------------------------------------------- */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1U)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    }

    /* Add A0 Payload tag */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A0;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (0x03U /* Options */ + (2U + *pEPubKeyLen) /* Ephemeral Public Key */);

    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0xFF00U) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0x00FFU);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = *pKeySize;

    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0xFF00U) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0x00FFU);

    /* Add Ephemeral Public Key. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pEPubKey, *pEPubKeyLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += *pEPubKeyLen;

    /* Add LE. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pExpRspLen, bExpRspLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bExpRspLen;

    /* Update LC */
    phalNtagXDna_Int_UpdateLC(PHAL_NTAGXDNA_CMD_BUF, PHAL_NTAGXDNA_CMD_BUF_LEN, bExpRspLen);

    /* Perform Exchange and receive Device KeySize, Ephemeral Public Key, Certificate Hash and Signature. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        ppResponse,
        pRespLen,
        NULL));

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A1(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint16_t wTagLen, uint16_t wPriv_KeyNo, uint16_t wPriv_KeyPos, uint8_t bKeySize_Host, uint8_t bKeySize_Device, uint8_t * pEndLeaf_Cert,
    uint16_t wEndLeaf_Cert_Len, uint8_t * pE_PubInit, uint16_t wE_PubInitLen, uint8_t * pE_PubResp, uint16_t wE_PubRespLen,
    uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wCertHashLen = 0;
    uint8_t     PH_MEMLOC_REM bData_Offset = 0;

    uint8_t     PH_MEMLOC_REM * pSignature = NULL;
    uint16_t    PH_MEMLOC_REM wSigLen = 0;

    /* Reset Buffer Length. */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Certificate Request Payload */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1U)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    }

    /* Add A1 Payload tag and Length */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A1;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (PHAL_NTAGXDNA_HASH_LEN +
        PHAL_NTAGXDNA_SIGNATURE_LEN + wTagLen);

    bData_Offset = (uint8_t) PHAL_NTAGXDNA_CMD_BUF_LEN;

    /* Generate Host End Leaf Certificate Hash and Signature */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_GenerateSignature(
        pDataParams,
        PH_OFF,
        wPriv_KeyNo,
        wPriv_KeyPos,
        bKeySize_Host,
        bKeySize_Device,
        pEndLeaf_Cert,
        wEndLeaf_Cert_Len,
        pE_PubInit,
        wE_PubInitLen,
        pE_PubResp,
        wE_PubRespLen,
        &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN],
        &wCertHashLen,
        &pSignature,
        &wSigLen));
    PHAL_NTAGXDNA_CMD_BUF_LEN += wCertHashLen;

    /* Add Host Signature */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pSignature, wSigLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += wSigLen;

    /* Increment Nonce */
    pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Encrypt Message. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
        PH_CRYPTOSYM_AUTH_TAG_ON),
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset],
        (uint16_t) (wCertHashLen + wSigLen),
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset]));

    /* Update Command Buffer Length to add authentication tag */
    PHAL_NTAGXDNA_CMD_BUF_LEN += wTagLen;

    /* Add LE. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pExpRspLen, bExpRspLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bExpRspLen;

    /* Update LC */
    phalNtagXDna_Int_UpdateLC(PHAL_NTAGXDNA_CMD_BUF, PHAL_NTAGXDNA_CMD_BUF_LEN, bExpRspLen);

    /* Perform Exchange and receive information from Device */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        ppResponse,
        pRespLen,
        NULL));

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A2(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint16_t wTagLen, uint8_t * pCert_Depth, uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bData_Offset = 0;

    /* Reset Buffer Length. */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Certificate Request Payload */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;

    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A2;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x0CU /* Length of subsequent information */;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_ENC & 0xFF00U) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (2U /* Encrypted Len */ + wTagLen );

    bData_Offset = (uint8_t) PHAL_NTAGXDNA_CMD_BUF_LEN;

    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pCert_Depth, 2U);
    PHAL_NTAGXDNA_CMD_BUF_LEN += 2U;

    /* Increment Nonce */
    pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Encrypt Message. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
        PH_CRYPTOSYM_AUTH_TAG_ON),
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset],
        2U,
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset]));

    /* Update Command Buffer Length to add authentication tag */
    PHAL_NTAGXDNA_CMD_BUF_LEN += wTagLen;

    /* Add LE. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pExpRspLen, bExpRspLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bExpRspLen;

    /* Update LC */
    phalNtagXDna_Int_UpdateLC(PHAL_NTAGXDNA_CMD_BUF, PHAL_NTAGXDNA_CMD_BUF_LEN, bExpRspLen);

    /* Perform Exchange and receive Encrypted Certificate */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        ppResponse,
        pRespLen,
        NULL));

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_A3(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint16_t wTagLen, uint8_t * pCert, uint16_t wCert_Len, uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse,
    uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBER_TLV_LEN = 0;
    uint16_t    PH_MEMLOC_REM wLc = 0;
    uint8_t     PH_MEMLOC_REM bTagLen = 0;

    /* Reset Buffer Length. */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame LC */
    wLc = (uint16_t) (1U /* Payload Tag */ + 3U /* BER-TLV of complete message */ + wCert_Len + wTagLen);

    /* Frame Certificate Request Payload because Certificate length will not fit in one frame */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((wLc & 0xFF00) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (wLc & 0x00FF);

    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A3;

    /* Encode BER_TLV Length */
    wBER_TLV_LEN = (uint16_t)
    (
        wCert_Len + /* Encrypted Certificate Length */
        wTagLen     /* Authentication Tag */
    );
    phalNtagXDna_Int_EncodeBER_TLV_Len(PHAL_NTAGXDNA_CMD_BUF, &PHAL_NTAGXDNA_CMD_BUF_LEN, wBER_TLV_LEN);

    /* Buffer Initial information to be exchanged */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Increment Nonce */
    pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Encrypt Message. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM),
        pCert,
        wCert_Len,
        pCert));

    /* Buffer and Exchange Certificate information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        pCert,
        wCert_Len,
        NULL,
        NULL,
        NULL));

    /* Get the Tag information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_GetAuthenticationTag(
        pDataParams->pCryptoDataParamsEnc,
        PHAL_NTAGXDNA_CMD_BUF,
        &bTagLen));

    /* Buffer and Exchange Authentication Tag information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        bTagLen,
        NULL,
        NULL,
        NULL));

    /* Buffer and Exchange LE information */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        pExpRspLen,
        bExpRspLen,
        ppResponse,
        pRespLen,
        NULL));

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_B1(uint8_t bKeySize_Host, uint8_t * pInput, uint16_t wInputLen,
    uint8_t * pKeySize_Device, uint8_t * pEPubKey, uint16_t * pEPubKeyLen, uint8_t ** ppEncData, uint16_t * pEncDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBER_TLV_Len = 0;

    if(pInput[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B1)
    {
        /* Increment pointer to next address. */
        pInput++;

        /* Remove BER-TLV information */
        PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_DecodeBER_TLV_Len(&pInput, &wBER_TLV_Len, &wInputLen));

        /* Check Options. */
        if((pInput[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0xFF00U) >> 8U)) &&
           (pInput[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0x00FFU)))
        {
            /* Increment pointer to next address. */
            pInput += 2U;

            /* Check KeySize */
            if(pInput[0] == bKeySize_Host)
            {
                /* Save Device KeySize */
                *pKeySize_Device = pInput[0];

                /* Increment pointer to next address. */
                pInput++;
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
            }
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }

        /* Check Ephemeral Public Key. */
        if((pInput[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0xFF00U) >> 8U)) &&
            (pInput[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0x00FFU)))
        {
            /* Increment pointer to next address. */
            pInput += 1U;

            /* Save Ephemeral Public Key */
            *pEPubKeyLen = pInput[0];
            (void) memcpy(pEPubKey, &pInput[1U], *pEPubKeyLen);

            /* Increment pointer to next address. */
            pInput += (uint16_t) (*pEPubKeyLen + 1U /* Ephemeral Public Key Length */);
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }

        /* Check Encrypted Message. */
        if(pInput[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_ENC & 0xFF00U) >> 8U))
        {
            /* Increment pointer to next address. */
            pInput += 1U;

            /* Save Encrypted Message */
            *pEncDataLen = pInput[0];
            *ppEncData = &pInput[1U];
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }
    }
    else
    {
        /* Return SUCCESS */
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_B2(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t * pInput,
    uint16_t wInputLen, uint8_t * pOutData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM *pEncData = NULL;
    uint16_t    PH_MEMLOC_REM wEncDataLen = 0;

    if(pInput[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B2)
    {
        /* Increment pointer to next address. */
        pInput += 2U;
        wInputLen -= 2U;

        /* Check if Encryption tag is received */
        if(pInput[0] == ((uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_ENC & 0xFF00U) >> 8U)))
        {
            /* Increment pointer to next address. */
            pInput++;
            wInputLen--;

            /* Extract Encryption data and Length */
            wEncDataLen = pInput[0];
            pEncData = &pInput[1U];

            /* Increment Nonce */
            pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
                pDataParams->pCryptoDataParamsEnc,
                PH_CRYPTOSYM_CIPHER_MODE_CCM,
                pDataParams->aSesNonce,
                (uint8_t) sizeof(pDataParams->aSesNonce)));

            /* Decrypt Response. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
                pDataParams->pCryptoDataParamsEnc,
                (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
                PH_CRYPTOSYM_AUTH_TAG_ON),
                pEncData,
                wEncDataLen,
                pOutData));
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }
    }
    else
    {
        /* Return SUCCESS */
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Initiator_Payload_B3(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t * pInput,
    uint16_t wInputLen, uint8_t ** ppOutData, uint16_t * pOutDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBER_TLV_Len = 0;

    if(pInput[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B3)
    {
        /* Increment pointer to next address. */
        pInput++;
        wInputLen--;

        /* Remove BER-TLV information */
        PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_DecodeBER_TLV_Len(&pInput, &wBER_TLV_Len, &wInputLen));

        /* Increment Nonce */
        pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_CIPHER_MODE_CCM,
            pDataParams->aSesNonce,
            (uint8_t) sizeof(pDataParams->aSesNonce)));

        /* Decrypt Response. */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
            pDataParams->pCryptoDataParamsEnc,
            (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
            PH_CRYPTOSYM_AUTH_TAG_ON),
            pInput,
            wInputLen,
            pInput));

        /* Update Response Length */
        wInputLen -= 8U; /* Authentication Tag (MAC) removed. */

        /* Check and remove Certificate Encoding */
        if(((pInput[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_UNCOMPRESSED_CERT & 0xFF00U) >> 8U)) &&
            (pInput[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_UNCOMPRESSED_CERT & 0x00FFU))) ||
            ((pInput[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_COMPRESSED_CERT & 0xFF00U) >> 8U)) &&
            (pInput[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_COMPRESSED_CERT & 0x00FFU))))
        {
            /* Increment pointer to next address. */
            pInput += 2U;

            /* Update Response Length */
            wInputLen -= 2U; /* Certificate Compression Type. */

            /* Remove BER-TLV information */
            PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_DecodeBER_TLV_Len(&pInput, &wBER_TLV_Len, &wInputLen));

            /* Update the response parameters */
            *ppOutData = pInput;
            *pOutDataLen = wInputLen;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }
    }
    else
    {
        /* Return SUCCESS */
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

/* ISOGeneral Authenticate with Host as Responder and Device as Initiator -------------------------------------------------------------- */
phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bDeviceCert_Cached,
    uint8_t bCertRepoID, uint8_t bCert_Depth, uint8_t bKeySize_Host, uint16_t wPriv_KeyNo, uint16_t wPriv_KeyPos, uint16_t wPub_KeyNo,
    uint16_t wPub_KeyPos, uint8_t * pExpRspLen, uint8_t bExpRspLen, phalNtagXDna_Cert stHostCert, phalNtagXDna_Cert * pDeviceCert)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wTagLen = 0;
    uint8_t     PH_MEMLOC_REM bCurveID = 0;
    uint8_t     PH_MEMLOC_REM bEndLoop = PH_OFF;
    uint8_t     PH_MEMLOC_REM bKeySize_Device = 0;
    uint8_t     PH_MEMLOC_REM bDevice_EndLeafCert_Available = PH_OFF;
    uint8_t     PH_MEMLOC_REM bVerification_Complete = PH_OFF;
    uint8_t     PH_MEMLOC_REM aCertDepth[2U] = { PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0, 0x00U };

    uint8_t     PH_MEMLOC_REM * pEPub_Host = NULL;
    uint16_t    PH_MEMLOC_REM wEPub_HostLen = 0;

    uint8_t     PH_MEMLOC_REM * pEPub_Device = NULL;
    uint16_t    PH_MEMLOC_REM wEPub_DeviceLen = 0;

    uint8_t     PH_MEMLOC_REM aSharedSecret[PHAL_NTAGXDNA_SHARED_SECRET_LEN];
    uint16_t    PH_MEMLOC_REM wSharedSecretLen = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint8_t     PH_MEMLOC_REM * pHost_Cert = NULL;
    uint16_t    PH_MEMLOC_REM wHost_CertLen = 0;

    uint8_t     PH_MEMLOC_REM aCertHash_Device[PHAL_NTAGXDNA_HASH_LEN];
    uint16_t    PH_MEMLOC_REM wCertHash_Device_Len = 0;

    uint8_t     PH_MEMLOC_REM aSig_Device[PHAL_NTAGXDNA_SIGNATURE_LEN];
    uint16_t    PH_MEMLOC_REM wSig_Device_Len = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Set the Pointers */
    pEPub_Host = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PHAL_NTAGXDNA_PUBLIC_KEY_LEN * 2U)];
    pEPub_Device = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - PHAL_NTAGXDNA_PUBLIC_KEY_LEN];

    /* Extract Host Ephemeral Public Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_ExportKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        PHAL_NTAGXDNA_PUBLIC_KEY_LEN,
        &bCurveID,
        pEPub_Host,
        &wEPub_HostLen));

    /* Transfer Control to Device and extract Device supported KeySize and Ephemeral Public Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B0(
        pDataParams,
        bCertRepoID,
        bKeySize_Host,
        pExpRspLen,
        bExpRspLen,
        &bKeySize_Device,
        pEPub_Device,
        &wEPub_DeviceLen));

    /* Compute Shared Secret. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_SharedSecret(
        pDataParams->pCryptoDataParamsASym,
        bCurveID,
        pEPub_Device,
        wEPub_DeviceLen,
        aSharedSecret,
        &wSharedSecretLen));

    /* Generate SessionKeys */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_GenerateSessionKeys_ASym(
        pDataParams,
        bKeySize_Host,
        pEPub_Device,
        wEPub_DeviceLen,
        pEPub_Host,
        wEPub_HostLen,
        aSharedSecret,
        wSharedSecretLen));

    /* Save configured CCM Authentication Tag Length */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_GetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_CCM_TAG_LENGTH,
        &wTagLen));

    /* Exchange Host KeySize, Ephemeral Public Key and Encrypted End-Leaf Certificate Hash and Signature */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B1(
        pDataParams,
        bCertRepoID,
        wTagLen,
        wPriv_KeyNo,
        wPriv_KeyPos,
        bKeySize_Host,
        bKeySize_Device,
        stHostCert.pEndLeaf,
        stHostCert.wEndLeaf_Len,
        pEPub_Device,
        wEPub_DeviceLen,
        pEPub_Host,
        wEPub_HostLen,
        pExpRspLen,
        bExpRspLen,
        &pResponse,
        &wRespLen));

    /* Exclude Host Certificate Exchange if Device has not requested. */
    if(pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A2)
    {
        /* Exchange Host Certificate(s) */
        do
        {
            /* Decrypt Response to get the Certificate depth information */
            wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_A2(
                pDataParams,
                pResponse,
                wRespLen,
                aCertDepth);

            /* End the loop in case of failure */
            if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bEndLoop = PH_ON;
            }
            else
            {
                /* Copy End-Leaf Certificate */
                if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0)
                {
                    pHost_Cert = stHostCert.pEndLeaf;
                    wHost_CertLen = stHostCert.wEndLeaf_Len;
                }

                /* Copy Parent Certificate */
                else if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_1)
                {
                    pHost_Cert = stHostCert.pParent;
                    wHost_CertLen = stHostCert.wParent_Len;
                }

                /* Copy Grand-Parent Certificate */
                else
                {
                    pHost_Cert = stHostCert.pGrandParent;
                    wHost_CertLen = stHostCert.wGrandParent_Len;
                }

                /* Encrypt Host Certificate, Exchange and receive Device Certificate Hash and Signature. */
                wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B3(
                    pDataParams,
                    bCertRepoID,
                    wTagLen,
                    pHost_Cert,
                    wHost_CertLen,
                    pExpRspLen,
                    2U,
                    &pResponse,
                    &wRespLen);

                /* End Loop if Device Certificate Hash and Signature are received */
                if(pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A1)
                {
                    bEndLoop = PH_ON;

                    /* Decrement Tag and Length */
                    pResponse += 2U;
                    wRespLen -= 2U;
                }

                /* End the loop in case of failure */
                if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    bEndLoop = PH_ON;
                }
                else
                {
                    /* Nothing to do. */
                }
            }
        } while(bEndLoop != PH_ON);

        /* Check the status */
        PH_CHECK_SUCCESS(wStatus);

        /* Reset Flag */
        bEndLoop = PH_OFF;

        /* Reset Certificate Depth */
        aCertDepth[0] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0;
    }
    else
    {
        /* Check if Device Certificate Hash and Signature are received */
        if(pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A1)
        {
            /* Decrement Tag and Length */
            pResponse += 2U;
            wRespLen -= 2U;
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }
    }

    /* Increment Nonce */
    pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Decrypt Device Certificate Hash and Signature. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
        PH_CRYPTOSYM_AUTH_TAG_ON),
        pResponse,
        wRespLen,
        pResponse));

    /* Remove CCM Authentication Tag Length from actual Response Length */
    wRespLen -= wTagLen;

    /* Extract Device Certificate Hash */
    wCertHash_Device_Len = PHAL_NTAGXDNA_HASH_LEN;
    (void) memcpy(aCertHash_Device, pResponse, wCertHash_Device_Len);

    /* Extract Device Signature */
    wSig_Device_Len = (uint16_t) (wRespLen - wCertHash_Device_Len);
    (void) memcpy(aSig_Device, &pResponse[wCertHash_Device_Len], wSig_Device_Len);

    /* Request for Device Certificates */
    do
    {
        /* Request for Device End-Leaf Certificate if not cached */
        if(bDeviceCert_Cached == PH_OFF)
        {
            /* Exchange Certificate Depth and Receive Device certificate(s) based on depth value */
            wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B2(
                pDataParams,
                bCertRepoID,
                wTagLen,
                aCertDepth,
                2U,
                pExpRspLen,
                2U,
                &pResponse,
                &wRespLen);

            /* End the loop in case of failure */
            if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bEndLoop = PH_ON;
            }
            else
            {
                /* Decrypt the received information and extract the certificate */
                wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_A3(
                    pDataParams,
                    wTagLen,
                    pResponse,
                    wRespLen,
                    &pResponse,
                    &wRespLen);

                /* End the loop in case of failure */
                if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    bEndLoop = PH_ON;
                }
                else
                {
                    /* Copy End-Leaf Certificate */
                    if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_0)
                    {
                        (void) memcpy(pDeviceCert->pEndLeaf, pResponse, wRespLen);
                        pDeviceCert->wEndLeaf_Len = wRespLen;

                        bDevice_EndLeafCert_Available = PH_ON;
                    }

                    /* Copy Parent Certificate */
                    else if(aCertDepth[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_1)
                    {
                        (void) memcpy(pDeviceCert->pParent, pResponse, wRespLen);
                        pDeviceCert->wParent_Len = wRespLen;
                    }

                    /* Copy Grand-Parent Certificate */
                    else
                    {
                        (void) memcpy(pDeviceCert->pGrandParent, pResponse, wRespLen);
                        pDeviceCert->wGrandParent_Len = wRespLen;
                    }

                    /* Update Depth to receive Device Parent and Grand-Parent Certificate(s) */
                    if(bCert_Depth == (uint8_t )(aCertDepth[0] & PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_CERT_DEPTH_MASK))
                    {
                        bEndLoop = PH_ON;
                    }
                    else
                    {
                        aCertDepth[0]++;
                    }
                }
            }
        }
        else
        {
            bDevice_EndLeafCert_Available = PH_ON;
            bEndLoop = PH_ON;
        }

        /* Verifiy Device Certificate Hash and Signature */
        if((bVerification_Complete == PH_OFF) && (bDevice_EndLeafCert_Available == PH_ON))
        {
            /* Verify Certificate Hash received from Device */
            wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_VerifyCertHash(
                pDataParams,
                aCertHash_Device,
                wCertHash_Device_Len,
                pDeviceCert->pEndLeaf,
                pDeviceCert->wEndLeaf_Len);

            /* End the loop in case of failure */
            if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
            {
                bEndLoop = PH_ON;
            }

            /* Verify Signature received from Device */
            else
            {
                wStatus = phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_VerifySignature(
                    pDataParams,
                    PH_ON,
                    wPub_KeyNo,
                    wPub_KeyPos,
                    bKeySize_Host,
                    bKeySize_Device,
                    aCertHash_Device,
                    (uint8_t) wCertHash_Device_Len,
                    pEPub_Host,
                    wEPub_HostLen,
                    pEPub_Device,
                    wEPub_DeviceLen,
                    aSig_Device,
                    wSig_Device_Len);

                /* Set verification completion flag. */
                bVerification_Complete = PH_ON;

                /* End the loop in case of failure */
                if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
                {
                    bEndLoop = PH_ON;
                }
            }
        }
    } while(bEndLoop != PH_ON);

    /* Check the status */
    PH_CHECK_SUCCESS(wStatus);

    /* Reset Response buffer and Length */
    pResponse = NULL;
    wRespLen = 0;

    /* Reset Command Counter. */
    pDataParams->wCmdCtr = 0x00;

    /* Reset Transaction Identifier */
    (void) memset(pDataParams->aTi, 0x00, sizeof(pDataParams->aTi));

    /* Reset Nonce */
    (void) memset(pDataParams->aSesNonce, 0x00, sizeof(pDataParams->aSesNonce));

    pDataParams->bAuthState = PHAL_NTAGXDNA_ECC_AUTHENTICATED;
    pDataParams->wKeyType = (uint16_t) ((bKeySize_Host == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES128) ? PH_KEYSTORE_KEY_TYPE_AES128 :
        PH_KEYSTORE_KEY_TYPE_AES256);

    /* Load the Session ENC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        pDataParams->aSesAuthENCKey,
        pDataParams->wKeyType));

    /* Load the Session MAC key. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pDataParams->aSesAuthMACKey,
        pDataParams->wKeyType));

    /* Set the keep Iv ON */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_KEEP_IV,
        PH_CRYPTOSYM_VALUE_KEEP_IV_ON));

    /* Clear the local buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUFFER_SIZE_MINIMUM);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;
    PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;

    (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUFFER_SIZE_MINIMUM);
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    PHAL_NTAGXDNA_PRS_BUF_OFFSET = 0;

    PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE = PH_OFF;
    PHAL_NTAGXDNA_HAS_MAC_PROCESSED = PH_OFF;

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_A2(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t * pInput,
    uint16_t wInputLen, uint8_t * pOutData)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM *pEncData = NULL;
    uint16_t    PH_MEMLOC_REM wEncDataLen = 0;

    if(pInput[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A2)
    {
        /* Increment pointer to next address. */
        pInput += 2U;
        wInputLen -= 2U;

        /* Check if Encryption tag is received */
        if(pInput[0] == ((uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_ENC & 0xFF00U) >> 8U)))
        {
            /* Increment pointer to next address. */
            pInput++;
            wInputLen--;

            /* Extract Encryption data and Length */
            wEncDataLen = pInput[0];
            pEncData = &pInput[1U];

            /* Increment Nonce */
            pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
                pDataParams->pCryptoDataParamsEnc,
                PH_CRYPTOSYM_CIPHER_MODE_CCM,
                pDataParams->aSesNonce,
                (uint8_t) sizeof(pDataParams->aSesNonce)));

            /* Decrypt Response. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
                pDataParams->pCryptoDataParamsEnc,
                (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
                PH_CRYPTOSYM_AUTH_TAG_ON),
                pEncData,
                wEncDataLen,
                pOutData));
        }
        else
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }
    }
    else
    {
        /* Return SUCCESS */
        wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_A3(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wTagLen,
    uint8_t * pInput, uint16_t wInputLen, uint8_t ** ppOutData, uint16_t * pOutDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBER_TLV_Len = 0;

    /* Increment pointer to next address to remove Tag. */
    pInput += 1U;
    wInputLen -= 1U;

    /* Decode Length and Extract Encrypted Data */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_DecodeBER_TLV_Len(
        &pInput,
        &wBER_TLV_Len,
        &wInputLen));

    /* Increment Nonce */
    pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Decrypt Response. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
        PH_CRYPTOSYM_AUTH_TAG_ON),
        pInput,
        wInputLen,
        pInput));

    /* Update Response Length */
    wInputLen -= wTagLen; /* Authentication Tag (MAC) removed. */

    /* Check and remove Certificate Encoding */
    if(((pInput[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_UNCOMPRESSED_CERT & 0xFF00U) >> 8U)) &&
        (pInput[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_UNCOMPRESSED_CERT & 0x00FFU))) ||
        ((pInput[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_COMPRESSED_CERT & 0xFF00U) >> 8U)) &&
        (pInput[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_COMPRESSED_CERT & 0x00FFU))))
    {
        /* Increment pointer to next address. */
        pInput += 2U;

        /* Update Response Length */
        wInputLen -= 2U; /* Certificate Compression Type. */

        /* Remove BER-TLV information */
        PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Int_DecodeBER_TLV_Len(&pInput, &wBER_TLV_Len, &wInputLen));

        /* Update the response parameters */
        *ppOutData = pInput;
        *pOutDataLen = wInputLen;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B0(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint8_t bKeySize_Host, uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t * pKeySize_Device, uint8_t * pEPubKey, uint16_t * pEPubKeyLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM * pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Clear Buffers. */
    (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Cmd.ISOGeneralAuthenticate B0 Payload and Exchange to Device -------------------------------------------------------------- */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1U)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    }

    /* Add B0 Payload tag */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B0;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x00; /* Payload as None */

    /* Add LE. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pExpRspLen, bExpRspLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bExpRspLen;

    /* Update LC */
    phalNtagXDna_Int_UpdateLC(PHAL_NTAGXDNA_CMD_BUF, PHAL_NTAGXDNA_CMD_BUF_LEN, bExpRspLen);

    /* Perform Exchange if Host is Initiator */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        &pResponse,
        &wRespLen,
        NULL));

    /* Validate the response ----------------------------------------------------------------------------------------------------------- */
    if(pResponse[0] == PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_A0)
    {
        /* Increment pointer to next address. Removes MessageType and Length */
        pResponse += 2U;

        /* Check Supported Key Size and extract the same. */
        if((pResponse[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0xFF00U) >> 8U)) &&
            (pResponse[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0x00FFU)))
        {
            /* Increment pointer to next address. */
            pResponse += 2U;

            /* Check KeySize */
            if(pResponse[0] == bKeySize_Host)
            {
                /* Save Device KeySize */
                *pKeySize_Device = pResponse[0];

                /* Increment pointer to next address. */
                pResponse++;
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
            }

            /* Check Ephemeral KeyPair and extract the same. */
            if((pResponse[0] == ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0xFF00U) >> 8U)) &&
                (pResponse[1U] == (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0x00FFU)))
            {
                /* Increment pointer to next address. */
                pResponse += 1U;

                /* Save Ephemeral Public Key */
                *pEPubKeyLen = pResponse[0];
                (void) memcpy(pEPubKey, &pResponse[1U], *pEPubKeyLen);
            }
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
            }
        }
        else
        {
            wStatus = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
        }
    }
    else
    {
        wStatus = PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B1(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint16_t wTagLen, uint16_t wPriv_KeyNo, uint16_t wPriv_KeyPos, uint8_t bKeySize_Host, uint8_t bKeySize_Device, uint8_t * pEndLeaf_Cert,
    uint16_t wEndLeaf_Cert_Len, uint8_t * pE_PubInit, uint16_t wE_PubInitLen, uint8_t * pE_PubResp, uint16_t wE_PubRespLen,
    uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wCertHashLen = 0;
    uint8_t     PH_MEMLOC_REM bData_Offset = 0;

    uint8_t     PH_MEMLOC_REM * pSignature = NULL;
    uint16_t    PH_MEMLOC_REM wSigLen = 0;

    /* Reset Buffer Length. */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Certificate Request Payload */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1U)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    }

    /* Add B1 Payload tag and Length */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B1;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x81U; /* BER-TLV Length Tag */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0xB0U; /* BER-TLV Length */

    /* Add Host Key Size */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0xFF00U) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_AES_KEYSIZE & 0x00FFU);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bKeySize_Host;

    /* Add Host Ephemeral Public Key */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0xFF00U) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_E_PUB & 0x00FFU);
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pE_PubResp, wE_PubRespLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += wE_PubRespLen;

    /* Add Encrypted Host Certificate Hash and Signature */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_ENC & 0xFF00U) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x68U;

    bData_Offset = (uint8_t) PHAL_NTAGXDNA_CMD_BUF_LEN;

    /* Generate End Leaf Cert Hash and Host Signature */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_GenerateSignature(
        pDataParams,
        PH_ON,
        wPriv_KeyNo,
        wPriv_KeyPos,
        bKeySize_Host,
        bKeySize_Device,
        pEndLeaf_Cert,
        wEndLeaf_Cert_Len,
        pE_PubInit,
        wE_PubInitLen,
        pE_PubResp,
        wE_PubRespLen,
        &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN],
        &wCertHashLen,
        &pSignature,
        &wSigLen));
    PHAL_NTAGXDNA_CMD_BUF_LEN += wCertHashLen;

    /* Add Host Signature */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pSignature, wSigLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += wSigLen;

    /* Load Nonce */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Encrypt Message. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
        PH_CRYPTOSYM_AUTH_TAG_ON),
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset],
        (uint16_t) (wCertHashLen + wSigLen),
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset]));

    /* Update Command Buffer Length to add authentication tag */
    PHAL_NTAGXDNA_CMD_BUF_LEN += wTagLen;

    /* Add LE. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pExpRspLen, bExpRspLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bExpRspLen;

    /* Update LC */
    phalNtagXDna_Int_UpdateLC(PHAL_NTAGXDNA_CMD_BUF, PHAL_NTAGXDNA_CMD_BUF_LEN, bExpRspLen);

    /* Perform Exchange if Host is Initiator */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        ppResponse,
        pRespLen,
        NULL));

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B2(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint16_t wTagLen, uint8_t * pInput, uint16_t wInputLen, uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse,
    uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bData_Offset = 0;

    /* Reset Buffer Length. */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame Certificate Request Payload */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;

    /* Frame Extended LC. */
    if(bExpRspLen > 1U)
    {
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
        PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    }

    /* Add A2 Tag and Length */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B2;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = 0x0CU /* Length of subsequent information */;

    /* Add Payload Encryption Encoding Tag and Length */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PAYLOAD_ENCODING_ENC & 0xFF00U) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (2U /* Encrypted Len */ + wTagLen);

    bData_Offset = (uint8_t) PHAL_NTAGXDNA_CMD_BUF_LEN;

    /* Add Input to Command Buffer. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pInput, wInputLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += wInputLen;

    /* Increment Nonce */
    pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Encrypt Message. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM |
        PH_CRYPTOSYM_AUTH_TAG_ON),
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset],
        2U,
        &PHAL_NTAGXDNA_CMD_BUF[bData_Offset]));

    /* Update Command Buffer Length to add authentication tag */
    PHAL_NTAGXDNA_CMD_BUF_LEN += wTagLen;

    /* Add LE. */
    (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pExpRspLen, bExpRspLen);
    PHAL_NTAGXDNA_CMD_BUF_LEN += bExpRspLen;

    /* Update LC */
    phalNtagXDna_Int_UpdateLC(PHAL_NTAGXDNA_CMD_BUF, PHAL_NTAGXDNA_CMD_BUF_LEN, bExpRspLen);

    /* Perform Exchange and retrieve Device Certificate */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_DEFAULT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        ppResponse,
        pRespLen,
        NULL));

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_Responder_Payload_B3(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCertRepoID,
    uint16_t wTagLen, uint8_t * pCert, uint16_t wCert_Len, uint8_t * pExpRspLen, uint8_t bExpRspLen, uint8_t ** ppResponse,
    uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wBER_TLV_LEN = 0;
    uint16_t    PH_MEMLOC_REM wLc = 0;
    uint8_t     PH_MEMLOC_REM bTagLen = 0;

    /* Reset Buffer Length. */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

    /* Frame LC */
    wLc = (uint16_t) (1U /* Payload Tag */ + 3U /* BER-TLV of complete message */ + wCert_Len + wTagLen);

    /* Frame Certificate Request Payload because Certificate length will not fit in one frame */
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_ISO7816_GENERIC_CLA;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_CMD_AUTHENTICATE_ISO_GENERAL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_PROTOCOL;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bCertRepoID;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_WRAPPEDAPDU_LC;
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) ((wLc & 0xFF00) >> 8U);
    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (wLc & 0x00FF);

    PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_MESSAGE_TYPE_B3;

    /* Encode BER_TLV Length */
    wBER_TLV_LEN = (uint16_t)
    (
        wCert_Len + /* Encrypted Certificate Length */
        wTagLen     /* Authentication Tag */
    );
    phalNtagXDna_Int_EncodeBER_TLV_Len(PHAL_NTAGXDNA_CMD_BUF, &PHAL_NTAGXDNA_CMD_BUF_LEN, wBER_TLV_LEN);

    /* Buffer Initial information to be exchanged */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        PHAL_NTAGXDNA_CMD_BUF_LEN,
        NULL,
        NULL,
        NULL));

    /* Increment Nonce */
    pDataParams->aSesNonce[sizeof(pDataParams->aSesNonce) - 1U]++;
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));

    /* Encrypt Message. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        (uint16_t) (PH_EXCHANGE_DEFAULT | PH_CRYPTOSYM_CIPHER_MODE_CCM),
        pCert,
        wCert_Len,
        pCert));

    /* Buffer and Exchange Certificate information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        pCert,
        wCert_Len,
        NULL,
        NULL,
        NULL));

    /* Get the Tag information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_GetAuthenticationTag(
        pDataParams->pCryptoDataParamsEnc,
        PHAL_NTAGXDNA_CMD_BUF,
        &bTagLen));

    /* Buffer and Exchange Authentication Tag information. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_NONE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        PHAL_NTAGXDNA_CMD_BUF,
        bTagLen,
        NULL,
        NULL,
        NULL));

    /* Buffer and Exchange LE information */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_OPTION_DIRECT_EXCHANGE),
        0,
        PH_ON,
        pExpRspLen,
        bExpRspLen,
        ppResponse,
        pRespLen,
        NULL));

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_GenerateSignature(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bIsHostResponder,
    uint16_t wPriv_KeyNo, uint16_t wPriv_KeyPos, uint8_t bKeySize_Host, uint8_t bKeySize_Device, uint8_t * pEndLeaf_Cert,
    uint16_t wEndLeaf_Cert_Len, uint8_t * pE_PubInit, uint16_t wE_PubInitLen, uint8_t * pE_PubResp, uint16_t wE_PubRespLen,
    uint8_t * pCertHash, uint16_t * pCertHashLen, uint8_t ** ppSignature, uint16_t * pSigLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wSig_Len = 0;
    uint8_t     PH_MEMLOC_REM bMAC_Len = 0;
    uint8_t     PH_MEMLOC_REM bCert_Offset = 0;

    /* Compute the Offset to use for certificate hashing */
    bCert_Offset = (uint8_t) (2U /* Compression Type */ + 3U /* Certificate Length in BERT-TLV */);

    /* Get the Hash of Device End-Leaf Certificate */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ComputeHash(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_DEFAULT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        &pEndLeaf_Cert[bCert_Offset],
        (uint16_t) (wEndLeaf_Cert_Len - bCert_Offset),
        pCertHash,
        pCertHashLen));

    /* Load the Public Key to use for verification */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PRIVATE_KEY,
        wPriv_KeyNo,
        wPriv_KeyPos));

    /* Frame the initial information */
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = (uint8_t) (0x02U - bIsHostResponder);
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = bKeySize_Host;
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = bKeySize_Device;

    /* Load the Message to generate the Signature */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        PHAL_NTAGXDNA_PRS_BUF,
        PHAL_NTAGXDNA_PRS_BUF_LEN,
        NULL,
        0));

    /* Append Responder and Initator Ephemeral Public Key */
    if(bIsHostResponder == PH_OFF)
    {
        /* Append Responder Ephemeral Public Key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
            pDataParams->pCryptoDataParamsASym,
            PH_EXCHANGE_BUFFER_CONT,
            PH_CRYPTOASYM_HASH_ALGO_SHA256,
            &pE_PubResp[1U],
            (uint16_t) (wE_PubRespLen - 1U),
            NULL,
            0));

        /* Append Initiator Ephemeral Public Key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
            pDataParams->pCryptoDataParamsASym,
            PH_EXCHANGE_BUFFER_CONT,
            PH_CRYPTOASYM_HASH_ALGO_SHA256,
            &pE_PubInit[1U],
            (uint16_t) (wE_PubInitLen - 1U),
            NULL,
            0));
    }

    /* Append Initator and Responder Ephemeral Public Key */
    else
    {
        /* Append Initiator Ephemeral Public Key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
            pDataParams->pCryptoDataParamsASym,
            PH_EXCHANGE_BUFFER_CONT,
            PH_CRYPTOASYM_HASH_ALGO_SHA256,
            &pE_PubInit[1U],
            (uint16_t) (wE_PubInitLen - 1U),
            NULL,
            0));

        /* Append Responder Ephemeral Public Key */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
            pDataParams->pCryptoDataParamsASym,
            PH_EXCHANGE_BUFFER_CONT,
            PH_CRYPTOASYM_HASH_ALGO_SHA256,
            &pE_PubResp[1U],
            (uint16_t) (wE_PubRespLen - 1U),
            NULL,
            0));
    }

    /* Compute AES-CMAC of Device Leaf Certificate Hash. */
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = (uint8_t) (0x02U - bIsHostResponder);
    (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN], pCertHash, *pCertHashLen);
    PHAL_NTAGXDNA_PRS_BUF_LEN += *pCertHashLen;

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        PHAL_NTAGXDNA_PRS_BUF,
        PHAL_NTAGXDNA_PRS_BUF_LEN,
        PHAL_NTAGXDNA_PRS_BUF,
        &bMAC_Len));

    /* Append AES-CMAC of Host EndLeaf Certificate and Generate the Signature */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Sign(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_LAST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        PHAL_NTAGXDNA_PRS_BUF,
        bMAC_Len,
        PHAL_NTAGXDNA_PRS_BUF,
        &wSig_Len));

    /* Copy the signature to response parameter */
    *ppSignature = PHAL_NTAGXDNA_PRS_BUF;
    *pSigLen = wSig_Len;

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_VerifyCertHash(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t * pCertHash,
    uint16_t wCertHashLen, uint8_t *pCert, uint16_t wCertLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

    uint8_t     PH_MEMLOC_REM aCertHash_Act[32U];
    uint16_t    PH_MEMLOC_REM wCertHash_Act_Len = 0;

    /* Get the Hash of Device End-Leaf Certificate */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ComputeHash(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_DEFAULT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pCert,
        wCertLen,
        aCertHash_Act,
        &wCertHash_Act_Len));

    /* Verify Device End-Leaf Certificate Hash */
    if((wCertHashLen != wCertHash_Act_Len) ||
        (memcmp(aCertHash_Act, pCertHash, wCertHashLen) != 0))
    {
        wStatus = PH_ADD_COMPCODE(PHAL_NTAGXDNA_ERR_DEVICE_LEAF_CERTIFICATE_HASH_ERROR, PH_COMP_AL_NTAGXDNA);
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOGeneralAuthenticate_VerifySignature(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bIsHostResponder,
    uint16_t wPub_KeyNo, uint16_t wPub_KeyPos, uint8_t bKeySize_Host, uint8_t bKeySize_Device, uint8_t * pCertHash, uint8_t bCertHashLen,
    uint8_t * pE_PubInit, uint16_t wE_PubInitLen, uint8_t * pE_PubResp, uint16_t wE_PubRespLen, uint8_t * pSignature, uint16_t wSigLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bMAC_Len = 0;

    /* Load the Public Key to use for verification */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_LoadKey(
        pDataParams->pCryptoDataParamsASym,
        PH_CRYPTOASYM_PUBLIC_KEY,
        wPub_KeyNo,
        wPub_KeyPos));

    /* Frame the initial information */
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = (uint8_t) (0x01U + bIsHostResponder);
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = bKeySize_Host;
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = bKeySize_Device;

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        PHAL_NTAGXDNA_PRS_BUF,
        PHAL_NTAGXDNA_PRS_BUF_LEN,
        NULL,
        0));

    /* Append Initiator Ephemeral Public Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        &pE_PubInit[1U],
        (uint16_t) (wE_PubInitLen - 1U),
        NULL,
        0));

    /* Append Responder Ephemeral Public Key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        &pE_PubResp[1U],
        (uint16_t) (wE_PubRespLen - 1U),
        NULL,
        0));

    /* Compute AES-CMAC of Device Leaf Certificate Hash. */
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = (uint8_t) (0x01U + bIsHostResponder);
    (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN], pCertHash, bCertHashLen);
    PHAL_NTAGXDNA_PRS_BUF_LEN += bCertHashLen;

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        PHAL_NTAGXDNA_PRS_BUF,
        PHAL_NTAGXDNA_PRS_BUF_LEN,
        PHAL_NTAGXDNA_PRS_BUF,
        &bMAC_Len));

    /* Append AES-CMAC of Device EndLeaf Certificate and Verify the Signature */
    wStatus = phCryptoASym_ECC_Verify(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_LAST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        PHAL_NTAGXDNA_PRS_BUF,
        bMAC_Len,
        pSignature,
        wSigLen);

    /* Assign actual error code. */
    if((wStatus & PH_ERR_MASK) == PH_ERR_VERIFICATION_FAILED)
    {
        wStatus = PH_ADD_COMPCODE(PHAL_NTAGXDNA_ERR_DEVICE_SIGNATURE_ERROR, PH_COMP_AL_NTAGXDNA);
    }
    else
    {
        /* Return the actual error */
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_GenerateSessionKeys_ASym(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bKeySize, uint8_t * pE_PubInit,
    uint16_t wE_PubInitLen, uint8_t * pE_PubResp, uint16_t wE_PubRespLen, uint8_t * pSharedSecret, uint16_t wSharedSecretLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wKeyType = 0;
    uint16_t    PH_MEMLOC_REM wL2 = 0;
    uint8_t     PH_MEMLOC_REM bKey_Len = 0;

    uint8_t     PH_MEMLOC_REM aSV[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM bSV_Len = 0;

    uint8_t     PH_MEMLOC_REM aKey[PH_CRYPTOSYM_AES256_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM *pKi = aKey;
    uint16_t    PH_MEMLOC_REM wKey_Len = 0;

    /* Clear Buffers. */
    (void) memset(aKey, 0x00, sizeof(aKey));

    /* Compute KeyType */
    wKeyType = (uint16_t) ((bKeySize == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES128) ? PH_CRYPTOSYM_KEY_TYPE_AES128 :
        PH_CRYPTOSYM_KEY_TYPE_AES256);

    /* Compute L2 to use */
    wL2 = (uint16_t) ((bKeySize == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES128) ? PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_L2_AES128 :
        PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_L2_AES256);

    /* Compute K(i) => Base Key for Session Key generation
     *              => SHA-256(trans_xy)
     *              => trans_xy = x-coordinate of shared secret | initiator's public key | responder's public
     * When AES-256 is selected, then the complete 32 bytes shall be used; for AES-128 the derived bytes shall
     * be truncated to the first 16 bytes
     */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ComputeHash(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_FIRST,
        PH_CRYPTOASYM_HASH_ALGO_SHA256,
        pSharedSecret,
        wSharedSecretLen,
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ComputeHash(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_CONT,
        PH_CRYPTOASYM_HASH_ALGO_NOT_APPLICABLE,
        &pE_PubInit[1U] /* Excluding the Initial format byte */,
        (uint16_t) (wE_PubInitLen - 1U),
        NULL,
        NULL));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoASym_ComputeHash(
        pDataParams->pCryptoDataParamsASym,
        PH_EXCHANGE_BUFFER_LAST,
        PH_CRYPTOASYM_HASH_ALGO_NOT_APPLICABLE,
        &pE_PubResp[1U],
        (uint16_t) (wE_PubRespLen - 1 /* Excluding the Initial format byte */),
        pKi,
        &wKey_Len));

    /* Load Ki ------------------------------------------------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pKi,
        wKeyType));

    /* Frame, Generate K_e2 session Key and Load to Encryption Buffer of DataParams #################################################### */
    (void) memset(pDataParams->aSesAuthENCKey, 0x00, sizeof(pDataParams->aSesAuthENCKey));
    (void) memset(aSV, 0x00, sizeof(aSV));
    bSV_Len = 0;
    bKey_Len = 0;

    aSV[bSV_Len++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0xFF00U) >> 8U);
    aSV[bSV_Len++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0x00FFU);
    aSV[bSV_Len++] = 'K'; aSV[bSV_Len++] = '_'; aSV[bSV_Len++] = 'e'; aSV[bSV_Len++] = '2';
    aSV[bSV_Len++] = 0x00;
    aSV[bSV_Len++] = 'E'; aSV[bSV_Len++] = 'V'; aSV[bSV_Len++] = '2';
    aSV[bSV_Len++] = (uint8_t) ((wL2 & 0xFF00) >> 8);
    aSV[bSV_Len++] = (uint8_t) (wL2 & 0x00FF);

    /* Update SV Length to 16 bytes */
    bSV_Len = 16U;

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        aSV,
        bSV_Len,
        &pDataParams->aSesAuthENCKey[bKey_Len],
        &bKey_Len));

    if(bKeySize == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES256)
    {
        aSV[0] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0xFF00U) >> 8U);
        aSV[1] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0x00FFU);

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsMac,
            phalNtagXDna_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            aSV,
            bSV_Len,
            &pDataParams->aSesAuthENCKey[bKey_Len],
            &bKey_Len));
    }
    /* ################################################################################################################################# */

    /* Frame, Generate K_m2 session Key and Load to Macing Buffer of DataParams ######################################################## */
    (void) memset(pDataParams->aSesAuthMACKey, 0x00, sizeof(pDataParams->aSesAuthMACKey));
    (void) memset(aSV, 0x00, sizeof(aSV));
    bSV_Len = 0;
    bKey_Len = 0;

    aSV[bSV_Len++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0xFF00U) >> 8U);
    aSV[bSV_Len++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0x00FFU);
    aSV[bSV_Len++] = 'K'; aSV[bSV_Len++] = '_'; aSV[bSV_Len++] = 'm'; aSV[bSV_Len++] = '2';
    aSV[bSV_Len++] = 0x00;
    aSV[bSV_Len++] = 'E'; aSV[bSV_Len++] = 'V'; aSV[bSV_Len++] = '2';
    aSV[bSV_Len++] = (uint8_t) ((wL2 & 0xFF00) >> 8);
    aSV[bSV_Len++] = (uint8_t) (wL2 & 0x00FF);

    /* Update SV Length to 16 bytes */
    bSV_Len = 16U;

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        aSV,
        bSV_Len,
        &pDataParams->aSesAuthMACKey[bKey_Len],
        &bKey_Len));

    if(bKeySize == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES256)
    {
        aSV[0] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0xFF00U) >> 8U);
        aSV[1] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0x00FFU);

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsMac,
            phalNtagXDna_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            aSV,
            bSV_Len,
            &pDataParams->aSesAuthMACKey[bKey_Len],
            &bKey_Len));
    }
    /* ################################################################################################################################# */

    /* Frame, Generate IV_e1 session Key and Load to CryptoSym Encryption context as nonce ############################################# */
    (void) memset(aSV, 0x00, sizeof(aSV));
    bSV_Len = 0;
    bKey_Len = 0;

    aSV[bSV_Len++] = 0x00;
    aSV[bSV_Len++] = 0x01U;
    aSV[bSV_Len++] = 'I'; aSV[bSV_Len++] = 'V'; aSV[bSV_Len++] = '_'; aSV[bSV_Len++] = 'e'; aSV[bSV_Len++] = '1';
    aSV[bSV_Len++] = 0x00;
    aSV[bSV_Len++] = 'I'; aSV[bSV_Len++] = 'V'; aSV[bSV_Len++] = 's';
    aSV[bSV_Len++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_L2_IV & 0xFF00U) >> 8U);
    aSV[bSV_Len++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_L2_IV & 0x00FFU);

    /* Update SV Length to 16 bytes */
    bSV_Len = 16U;

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        aSV,
        bSV_Len,
        &aKey[bKey_Len],
        &bKey_Len));

    /* Save Nonce to be used wile performing ISOGeneralAuthenticate. */
    (void) memcpy(pDataParams->aSesNonce, aKey, 13U);

    /* Set the Tag Length to use. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_SetConfig(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CONFIG_CCM_TAG_LENGTH,
        8U));

    /* Load Nonce to Encryption context. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadNonce(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_CCM,
        pDataParams->aSesNonce,
        (uint8_t) sizeof(pDataParams->aSesNonce)));
    /* ################################################################################################################################# */

    /* Frame, Generate K_e1 session Key and Load to CryptoSym Encryption context ####################################################### */
    (void) memset(aSV, 0x00, sizeof(aSV));
    bSV_Len = 0;
    bKey_Len = 0;

    aSV[bSV_Len++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0xFF00U) >> 8U);
    aSV[bSV_Len++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0x00FFU);
    aSV[bSV_Len++] = 'K'; aSV[bSV_Len++] = '_'; aSV[bSV_Len++] = 'e'; aSV[bSV_Len++] = '1';
    aSV[bSV_Len++] = 0x00;
    aSV[bSV_Len++] = 'S'; aSV[bSV_Len++] = 'I'; aSV[bSV_Len++] = 'G'; aSV[bSV_Len++] = 'M';
    aSV[bSV_Len++] = 'A'; aSV[bSV_Len++] = '-'; aSV[bSV_Len++] = 'I';
    aSV[bSV_Len++] = (uint8_t) ((wL2 & 0xFF00U) >> 8U);
    aSV[bSV_Len++] = (uint8_t) (wL2 & 0x00FFU);

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        aSV,
        bSV_Len,
        &aKey[bKey_Len],
        &bKey_Len));

    if(bKeySize == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES256)
    {
        aSV[0] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0xFF00U) >> 8U);
        aSV[1U] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0x00FFU);

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsMac,
            phalNtagXDna_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            aSV,
            bSV_Len,
            &aKey[bKey_Len],
            &bKey_Len));
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsEnc,
        aKey,
        wKeyType));
    /* ################################################################################################################################# */

    /* Frame, Generate K_m1 session Key and Load to CryptoSym Mac context ############################################################## */
    (void) memset(aSV, 0x00, sizeof(aSV));
    bSV_Len = 0;
    bKey_Len = 0;

    aSV[bSV_Len++] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0xFF00U) >> 8U);
    aSV[bSV_Len++] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES128 & 0x00FFU);
    aSV[bSV_Len++] = 'K'; aSV[bSV_Len++] = '_'; aSV[bSV_Len++] = 'm'; aSV[bSV_Len++] = '1';
    aSV[bSV_Len++] = 0x00;
    aSV[bSV_Len++] = 'S'; aSV[bSV_Len++] = 'I'; aSV[bSV_Len++] = 'G'; aSV[bSV_Len++] = 'M';
    aSV[bSV_Len++] = 'A'; aSV[bSV_Len++] = '-'; aSV[bSV_Len++] = 'I';
    aSV[bSV_Len++] = (uint8_t) ((wL2 & 0xFF00U) >> 8U);
    aSV[bSV_Len++] = (uint8_t) (wL2 & 0x00FFU);

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        aSV,
        bSV_Len,
        &aKey[bKey_Len],
        &bKey_Len));

    if(bKeySize == PHAL_NTAGXDNA_SESSION_KEY_SIZE_AES256)
    {
        aSV[0] = (uint8_t) ((PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0xFF00U) >> 8U);
        aSV[1U] = (uint8_t) (PHAL_NTAGXDNA_AUTH_ISO_GENERAL_AUTH_ITERATION_AES256 & 0x00FFU);

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsMac,
            phalNtagXDna_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            aSV,
            bSV_Len,
            &aKey[bKey_Len],
            &bKey_Len));
    }

    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        aKey,
        wKeyType));
    /* ################################################################################################################################# */

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_MFDUOX);
}

phStatus_t phalNtagXDna_Sw_Int_GenerateSessionKeys_Sym(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wKeyType, uint8_t * pRndA,
    uint8_t * pRndB)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM aSV[PH_CRYPTOSYM_AES256_KEY_SIZE];
    uint8_t     PH_MEMLOC_REM bSvLen = 0;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;
    uint8_t     PH_MEMLOC_REM bIsAES256 = PH_OFF;

    /* Check if the Key Type is AES256 */
    bIsAES256 = (uint8_t) (wKeyType == PH_KEYSTORE_KEY_TYPE_AES256);

    /* Frame Session Vector 1 information. */
    aSV[bSvLen++] = 0xA5U; aSV[bSvLen++] = 0x5AU; aSV[bSvLen++] = 0x00; aSV[bSvLen++] = 0x01U;
    aSV[bSvLen++] = (uint8_t) (bIsAES256 ? 0x01U : 0x00);
    aSV[bSvLen++] = (uint8_t) (bIsAES256 ? 0x00 : 0x80U);

    aSV[bSvLen++] = pRndA[0]; aSV[bSvLen++] = pRndA[1];

    aSV[bSvLen++] = pRndA[2U] ^ pRndB[0]; aSV[bSvLen++] = pRndA[3U] ^ pRndB[1U]; aSV[bSvLen++] = pRndA[4U] ^ pRndB[2U];
    aSV[bSvLen++] = pRndA[5U] ^ pRndB[3U]; aSV[bSvLen++] = pRndA[6U] ^ pRndB[4U]; aSV[bSvLen++] = pRndA[7U] ^ pRndB[5U];

    (void) memcpy(&aSV[14U], &pRndB[6U], 10U);
    (void) memcpy(&aSV[24U], &pRndA[8U], 8U);

    bSvLen = (uint8_t) sizeof(aSV);

    /* Generate Session ENC Key --------------------------------------------------------------------------------------------------------
    * AES128
    *      SV1 = 0xA5 || 0x5A || 0x00 || 0x01 || 0x00 || 0x80 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
    *      KSesAuthENC = PRF(Kx, SV1)
    *
    * AES256
    *      SV1a = 0xA5 || 0x5A || 0x00 || 0x01 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
    *      SV1b = 0xA5 || 0x5A || 0x00 || 0x02 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
    *      KSesAuthENC = PRF(Kx, SV1a)||PRF(Kx, SV1b)
    */
    /* Load Zero IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Start CMAC calculation */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aSV,
        bSvLen,
        pDataParams->aSesAuthENCKey,
        &bMacLen));

    /* Extend the computation for AES256 KeyType */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_AES256)
    {
        /* Update the Session Vector */
        aSV[0x03U] = 0x02U;

        /* Load Zero IV */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsEnc,
            phalNtagXDna_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_MAC_MODE_CMAC,
            aSV,
            bSvLen,
            &pDataParams->aSesAuthENCKey[PH_CRYPTOSYM_AES_BLOCK_SIZE],
            &bMacLen));

        /* Revert back. */
        aSV[0x03] = 0x01;
    }

    /* Generate Session MAC Key --------------------------------------------------------------------------------------------------------
    * AES128
    *      SV2 = 0x5A || 0xA5 || 0x00 || 0x01 || 0x00 || 0x80 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
    *      KSesAuthMAC = PRF(Kx, SV2)
    *
    * AES256
    *      SV2a = 0x5A || 0xA5 || 0x00 || 0x01 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
    *      SV2b = 0x5A || 0xA5 || 0x00 || 0x02 || 0x01 || 0x00 || RndA[15:14] || (RndA[13::8] XOR RndB[15::10]) || RndB[9::0] || RndA[7::0]
    *      KSesAuthMAC = PRF(Kx, SV1a)||PRF(Kx, SV1b)
    */
    aSV[0] = 0x5AU;
    aSV[1U] = 0xA5U;

    /* Load IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Start CMAC calculation */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_MAC_MODE_CMAC,
        aSV,
        bSvLen,
        pDataParams->aSesAuthMACKey,
        &bMacLen));

    /* Extend the computation for AES256 KeyType */
    if(wKeyType)
    {
        aSV[0x03U] = 0x02U;

        /* Load Zero IV */
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
            pDataParams->pCryptoDataParamsEnc,
            phalNtagXDna_Sw_ZeroIv,
            PH_CRYPTOSYM_AES_BLOCK_SIZE));

        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsEnc,
            PH_CRYPTOSYM_MAC_MODE_CMAC,
            aSV,
            bSvLen,
            &pDataParams->aSesAuthMACKey[PH_CRYPTOSYM_AES_BLOCK_SIZE],
            &bMacLen));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}
#endif /* NXPBUILD__PH_CRYPTOSYM */





phStatus_t phalNtagXDna_Sw_Int_ApplySM(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bIsFirstFrame, uint8_t bIsLastFrame,
    uint8_t bCommMode, uint8_t * pCmdHeader, uint16_t wCmdHeaderLen, uint8_t * pCmdData, uint16_t wCmdDataLen,
    uint8_t ** ppSMBuf, uint16_t * pSMBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;

#ifdef NXPBUILD__PH_CRYPTOSYM
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wOption = 0;
    uint16_t    PH_MEMLOC_REM wBlockSize = 0;
    uint16_t    PH_MEMLOC_REM wOffset = 0;
    uint16_t    PH_MEMLOC_REM wPaddedDataLen = 0;
    uint16_t    PH_MEMLOC_REM wBuffOptions = 0;
    uint16_t    PH_MEMLOC_REM wUnProcessedDataLen_ENC = 0U;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;
    uint8_t     PH_MEMLOC_REM bMacProcessed = PH_OFF;
    uint8_t     PH_MEMLOC_REM bUnProcessedDataLen_MAC = 0U;

    uint8_t     PH_MEMLOC_REM aMac[PH_CRYPTOSYM_AES_BLOCK_SIZE];
#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* Set the Status to SUCCESS. */
    wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);

#ifdef NXPBUILD__PH_CRYPTOSYM
    /* Perform Secure Messaging on data. */
    if(bCommMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN)
    {
        /* Compute Encryption IV and Frame initial MAC --------------------------------------------------------------------------------- */
        if(bIsFirstFrame)
        {
            /* Compute IV for encryption. */
            if(bCommMode == PHAL_NTAGXDNA_COMMUNICATION_FULL)
            {
                PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_GenerateIv(
                    pDataParams,
                    PH_OFF,
                    pDataParams->aTi,
                    pDataParams->wCmdCtr));
            }

            /* Load zero IV for MAC Computation. */
            PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_LoadIv(
                pDataParams->pCryptoDataParamsMac,
                phalNtagXDna_Sw_ZeroIv,
                PH_CRYPTOSYM_AES_BLOCK_SIZE));

            /* Copy Command code to processing buffer. */
            PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = pCmdHeader[0U];

            /* Add command counter. */
            PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr);
            PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr >> 8U);

            /* Copy TI information to processing buffer. */
            (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN], pDataParams->aTi, PHAL_NTAGXDNA_SIZE_TI);
            PHAL_NTAGXDNA_PRS_BUF_LEN += (uint16_t) PHAL_NTAGXDNA_SIZE_TI;

            /* Copy Command Header to processing buffer. */
            (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN], &pCmdHeader[1U], (wCmdHeaderLen - 1U));
            PHAL_NTAGXDNA_PRS_BUF_LEN += (uint16_t) (wCmdHeaderLen - 1U);

            /* Update Processing buffer offset. */
            PHAL_NTAGXDNA_PRS_BUF_OFFSET = PHAL_NTAGXDNA_PRS_BUF_LEN;

            /* Clear Command buffer length and offset */
            PHAL_NTAGXDNA_CMD_BUF_LEN = 0U;
            PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0U;
        }
        else
        {
            /* Shift processing buffer for next operation. */
            if((PHAL_NTAGXDNA_IS_ENC_PENDING == PH_OFF) && (PHAL_NTAGXDNA_PRS_BUF_OFFSET > 0U))
            {
                (void) memcpy(PHAL_NTAGXDNA_PRS_BUF, &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_OFFSET], PHAL_NTAGXDNA_PRS_BUF_LEN);
                PHAL_NTAGXDNA_PRS_BUF_OFFSET = PHAL_NTAGXDNA_PRS_BUF_LEN;
            }

            /* Copy plain data to processing */
            if(PHAL_NTAGXDNA_CMD_BUF_LEN > 0U)
            {
                (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_OFFSET], PHAL_NTAGXDNA_CMD_BUF,
                    PHAL_NTAGXDNA_CMD_BUF_LEN);

                /* Update Processing buffer length. */
                PHAL_NTAGXDNA_PRS_BUF_LEN += PHAL_NTAGXDNA_CMD_BUF_LEN;
                PHAL_NTAGXDNA_PRS_BUF_OFFSET += PHAL_NTAGXDNA_CMD_BUF_LEN;

                /* Set unprocessed plain data. */
                wUnProcessedDataLen_ENC = PHAL_NTAGXDNA_CMD_BUF_LEN;

                /*Clear command buffer length and offset.*/
                PHAL_NTAGXDNA_CMD_BUF_LEN = 0U;
                PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0U;
            }

            /* Update Processing buffer offset */
            PHAL_NTAGXDNA_PRS_BUF_OFFSET = PHAL_NTAGXDNA_PRS_BUF_LEN;

            /* Clear Processing buffer Length */
            PHAL_NTAGXDNA_PRS_BUF_LEN = wUnProcessedDataLen_ENC;
        }

        /* Save the starting offset of data to be returned. */
        wOffset = (uint16_t) (PHAL_NTAGXDNA_PRS_BUF_OFFSET - wUnProcessedDataLen_ENC);

        /* Copy command data to processing buffer. */
        (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_OFFSET], pCmdData, wCmdDataLen);

        /* Set length of bytes to be encrypted. */
        wUnProcessedDataLen_ENC += wCmdDataLen;

        /* Update Processing buffer length and offset. */
        PHAL_NTAGXDNA_PRS_BUF_LEN += (uint16_t) (wCmdDataLen + (bIsFirstFrame ? 0U : wOffset /* UnProcessed MAC */));
        PHAL_NTAGXDNA_PRS_BUF_OFFSET += wCmdDataLen;

        /* Encrypt the Data */
        if(bCommMode == PHAL_NTAGXDNA_COMMUNICATION_FULL)
        {
            /* Apply padding. */
            if(bIsLastFrame)
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_ApplyPadding(
                    PH_CRYPTOSYM_PADDING_MODE_2,
                    &PHAL_NTAGXDNA_PRS_BUF[wOffset],
                    wUnProcessedDataLen_ENC,
                    PH_CRYPTOSYM_AES_BLOCK_SIZE,
                    PHAL_NTAGXDNA_PRS_BUF_SIZE,
                    &PHAL_NTAGXDNA_PRS_BUF[wOffset],
                    &wPaddedDataLen));

                /* Update Processing buffer length to including padding bytes. */
                PHAL_NTAGXDNA_PRS_BUF_LEN += (uint16_t) (wPaddedDataLen - wUnProcessedDataLen_ENC);
                PHAL_NTAGXDNA_PRS_BUF_OFFSET = PHAL_NTAGXDNA_PRS_BUF_LEN;

                /* Update un processed encrypted data length. */
                wUnProcessedDataLen_ENC = wPaddedDataLen;
            }

            /* Set the buffering option and Cipher mode .*/
            wBuffOptions = (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC |
                (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT));

            /* Get the Block size to encrypt. */
            PHAL_NTAGXDNA_PREVIOUS_MULTIPLE(wUnProcessedDataLen_ENC, wBlockSize);
            wBlockSize = (uint16_t) (bIsLastFrame ? wUnProcessedDataLen_ENC : wBlockSize);

            /* Encrypt the CmdData information. */
            if(wBlockSize != 0U)
            {
                PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_Encrypt(
                    pDataParams->pCryptoDataParamsEnc,
                    wBuffOptions,
                    &PHAL_NTAGXDNA_PRS_BUF[wOffset],
                    wBlockSize,
                    &PHAL_NTAGXDNA_PRS_BUF[wOffset]));

                PHAL_NTAGXDNA_IS_ENC_PENDING = PH_OFF;

                /* Copy un processed data to command buffer. */
                if((wUnProcessedDataLen_ENC - wBlockSize) > 0U)
                {
                    (void) memcpy(PHAL_NTAGXDNA_CMD_BUF, &PHAL_NTAGXDNA_PRS_BUF[wOffset + wBlockSize],
                        (wUnProcessedDataLen_ENC - wBlockSize));
                    PHAL_NTAGXDNA_CMD_BUF_LEN = (uint16_t) (wUnProcessedDataLen_ENC - wBlockSize);

                    /* Revert back processing buffer offset and length. */
                    PHAL_NTAGXDNA_PRS_BUF_LEN -= (uint16_t) (wUnProcessedDataLen_ENC - wBlockSize);
                    PHAL_NTAGXDNA_PRS_BUF_OFFSET -= (uint16_t) (wUnProcessedDataLen_ENC - wBlockSize);
                }
            }
            else
            {
                /* Copy plain data to command buffer. */
                (void) memcpy(PHAL_NTAGXDNA_CMD_BUF, &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN - wUnProcessedDataLen_ENC],
                    wUnProcessedDataLen_ENC);
                PHAL_NTAGXDNA_CMD_BUF_LEN = wUnProcessedDataLen_ENC;

                /* Revert back processing buffer offset and length. */
                PHAL_NTAGXDNA_PRS_BUF_LEN -= wUnProcessedDataLen_ENC;
                PHAL_NTAGXDNA_PRS_BUF_OFFSET -= wUnProcessedDataLen_ENC;

                PHAL_NTAGXDNA_IS_ENC_PENDING = PH_ON;
            }
        }

        /* Compute size of information to be MACed. */
        PHAL_NTAGXDNA_PREVIOUS_MULTIPLE(PHAL_NTAGXDNA_PRS_BUF_LEN, wBlockSize);
        wBlockSize = (uint16_t) (bIsLastFrame ? PHAL_NTAGXDNA_PRS_BUF_LEN : wBlockSize);

        /* Compute MAC. */
        if(((wBlockSize != 0U) || (bIsLastFrame == PH_ON)) && (PHAL_NTAGXDNA_IS_ENC_PENDING == PH_OFF))
        {
            /* Update Buffer Options */
            wOption = (uint16_t) (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT);
            wOption = (uint16_t) (wOption | PH_CRYPTOSYM_MAC_MODE_CMAC);

            PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_CalculateMac(
                pDataParams->pCryptoDataParamsMac,
                wOption,
                PHAL_NTAGXDNA_PRS_BUF,
                wBlockSize,
                aMac,
                &bMacLen));

            /* Compute unprocessed MAC */
            bUnProcessedDataLen_MAC = (uint8_t) (PHAL_NTAGXDNA_PRS_BUF_LEN - wBlockSize);

            /* Set MAC processed flag. */
            bMacProcessed = PH_ON;
        }
        else
        {
            /* Set unprocessed MAC */
            bUnProcessedDataLen_MAC = (uint8_t) PHAL_NTAGXDNA_PRS_BUF_LEN;

            /* Set MAC processed flag. */
            bMacProcessed = PH_OFF;
        }

        /* Add final data for MAC computation and get 16 bytes of MAC */
        if(bIsLastFrame)
        {
            /* Truncate MAC. */
            phalNtagXDna_Sw_Int_TruncateMac(aMac);

            /* Copy Truncated MAC to Processing buffer. */
            (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_OFFSET], aMac, PHAL_NTAGXDNA_TRUNCATED_MAC_LEN);
            PHAL_NTAGXDNA_PRS_BUF_LEN += PHAL_NTAGXDNA_TRUNCATED_MAC_LEN;

            /* Update Status */
            wStatus = PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
        }

        /* Copy the SM information to be exchanged. */
        *ppSMBuf = &PHAL_NTAGXDNA_PRS_BUF[wOffset];
        *pSMBufLen = (uint16_t) (PHAL_NTAGXDNA_PRS_BUF_LEN - wOffset);

        /* Update Offset */
        wOffset = (uint16_t) (bMacProcessed ? wBlockSize : 0U);

        if(PHAL_NTAGXDNA_IS_ENC_PENDING == PH_OFF)
        {
            /* Update processing buffer offset for next operation. */
            PHAL_NTAGXDNA_PRS_BUF_OFFSET = (uint16_t) (wOffset - (wBlockSize - wOffset));

            /* Update processing buffer length with unprocessed MAC bytes. */
            PHAL_NTAGXDNA_PRS_BUF_LEN = bUnProcessedDataLen_MAC;
        }
    }
    else
#endif /* NXPBUILD__PH_CRYPTOSYM */
    {
        /* Mark the parameters that are not used. */
        PH_UNUSED_VARIABLE(pDataParams);
        PH_UNUSED_VARIABLE(bIsFirstFrame);
        PH_UNUSED_VARIABLE(bIsLastFrame);
        PH_UNUSED_VARIABLE(bCommMode);
        PH_UNUSED_VARIABLE(pCmdHeader);
        PH_UNUSED_VARIABLE(wCmdHeaderLen);

        *ppSMBuf = pCmdData;
        *pSMBufLen = wCmdDataLen;
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_RemoveSM(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bIsISOChained, uint8_t bIsFirstFrame,
    uint8_t bIsLastFrame, uint8_t bCommMode, uint8_t * pResponse, uint16_t wRespLen, uint8_t bPiccStat, uint8_t ** ppOutBuffer,
    uint16_t * pOutBufLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wRemData = 0;
    uint16_t    PH_MEMLOC_REM wBuffOption = 0;
    uint16_t    PH_MEMLOC_REM wAESBlockSize = 0;
    uint16_t    PH_MEMLOC_REM wDataLen = 0;
    uint8_t     PH_MEMLOC_REM bHasMoreData = PH_OFF;
    uint8_t     PH_MEMLOC_REM bDecryptData = PH_ON;

#ifdef NXPBUILD__PH_CRYPTOSYM
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint16_t    PH_MEMLOC_REM wOffset = 0;
    uint16_t    PH_MEMLOC_REM wBuffEncLen = 0;
    uint16_t    PH_MEMLOC_REM wBuffMacLen = 0;
    uint8_t     PH_MEMLOC_REM bFinished = PH_ON;
    uint8_t     PH_MEMLOC_REM bMacLen = 0;

    uint8_t     PH_MEMLOC_REM aMac[PH_CRYPTOSYM_AES_BLOCK_SIZE];
    uint8_t     PH_MEMLOC_REM * pBufferEnc = NULL;
    uint8_t     PH_MEMLOC_REM * pBufferMac = NULL;
#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* Compute DataLen excluding MAC information. */
    wDataLen = (uint16_t) (wRespLen ? (wRespLen - (bIsLastFrame ? 8 /* MAC excluded */ : 0)) : 0);
    wDataLen = (uint16_t) ((bCommMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN) ? wDataLen : wRespLen);

    if(bIsFirstFrame)
    {
        /* Clear Buffers. */
        (void) memset(PHAL_NTAGXDNA_CMD_BUF, 0x00, PHAL_NTAGXDNA_CMD_BUF_SIZE);
        PHAL_NTAGXDNA_CMD_BUF_LEN = 0;

        (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
        PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

        /* Increment the command counter. */
        if(pDataParams->bAuthState != PHAL_NTAGXDNA_NOT_AUTHENTICATED)
            pDataParams->wCmdCtr++;

#ifdef NXPBUILD__PH_CRYPTOSYM
        if(bCommMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN)
        {
            /* Add PICC Response Code and Command Counter information. */
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = bPiccStat;
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr);
            PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN++] = (uint8_t) (pDataParams->wCmdCtr >> 8U);

            /* Add TI (Transaction Identifier) information. */
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pDataParams->aTi, PHAL_NTAGXDNA_SIZE_TI);
            PHAL_NTAGXDNA_CMD_BUF_LEN += (uint16_t) PHAL_NTAGXDNA_SIZE_TI;

            /* Compute the Remaining bytes to be filled for obtaining AES block size. */
            wAESBlockSize = PHAL_MFCC_AES_BLOCK_SIZE_DIFF(PHAL_NTAGXDNA_CMD_BUF_LEN);

            /* Check if data higher than AES block size is available. */
            bHasMoreData = (wDataLen > wAESBlockSize);
            bHasMoreData = bIsISOChained;
            bHasMoreData = (wDataLen > (PHAL_NTAGXDNA_CMD_BUF_SIZE - PHAL_NTAGXDNA_CMD_BUF_LEN));

            /* Update Index information. */
            wRemData = (uint16_t) (bHasMoreData ? wAESBlockSize : wDataLen);

            /* Copy the initial information from Response Data to fill Command buffer upto Single AES block Size. */
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pResponse, wRemData);
            PHAL_NTAGXDNA_CMD_BUF_LEN += (uint16_t) wRemData;

            /* Set the offset to pick the data. */
            wOffset = (uint16_t) (bHasMoreData ? wRemData : 0);

            /* Clear the flag that in case of ISOChainned exchanged. */
            bHasMoreData = (uint8_t) (bIsISOChained ? PH_OFF : bHasMoreData);

            /* Load Zero IV for Macing. */
            PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
                pDataParams->pCryptoDataParamsMac,
                phalNtagXDna_Sw_ZeroIv,
                PH_CRYPTOSYM_AES_BLOCK_SIZE));

            /* Compute and Load IV for Decryption. */
            PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_GenerateIv(
                pDataParams,
                PH_ON,
                pDataParams->aTi,
                pDataParams->wCmdCtr));
        }
#endif /* NXPBUILD__PH_CRYPTOSYM */
    }

    /* Performs the below code operation for,
     *  - DataManagement commands with where PICC data do not fit in one single frame.
     *  - All the commands where No Chaining is involved. The data fits in one complete PICC frame.
     *  - For commands that involve ISO/IEC 14443-4 Chaining, whether the frame fits in one PICC frame
     *    or chains into multiple PICC frames,
     *      - Processing of data is performed in if condition where processing for last frame
     *        of data is performed (Refer the if Statement if(bIsLastFrame...)).
     */

    if(bIsISOChained == PH_OFF)
    {
    /* Copy data from command buffer to processing buffer.
    * The below procedure is required in case if there is more PICC data that
    * could not fit into Processing buffer. In the next call the previously
    * remained PICC data needs to be copied to processing buffer.
    */
        if(PHAL_NTAGXDNA_CMD_BUF_OFFSET)
        {
            /* Compute the remaining data to be copied form offset. */
            wRemData = (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - PHAL_NTAGXDNA_CMD_BUF_OFFSET);

            /* Clear Buffers. */
            (void) memset(PHAL_NTAGXDNA_PRS_BUF, 0x00, PHAL_NTAGXDNA_PRS_BUF_SIZE);
            PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

            /* Compute the data. */
            (void) memcpy(PHAL_NTAGXDNA_PRS_BUF, &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_OFFSET], wRemData);
            PHAL_NTAGXDNA_PRS_BUF_LEN = wRemData;

            /* Reset Offset and Remaining Data. */
            wRemData = 0;
            PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;

            /* Set Remaining data decryption if its last frame. */
            if(bIsLastFrame)
                bDecryptData = PH_ON;
        }

        /* Buffer response information to command buffer for macing. */
        if((PHAL_NTAGXDNA_CMD_BUF_LEN + wDataLen) < PHAL_NTAGXDNA_CMD_BUF_SIZE)
        {
            (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], &pResponse[wRemData], (wDataLen - wRemData));
            PHAL_NTAGXDNA_CMD_BUF_LEN += (uint16_t) (wDataLen - wRemData);

            bHasMoreData = PH_ON;
        }

        /* Perform MACing of data based on the command buffer length and size.
         * Here if the command buffer is unable to copy the newly received PICC data, MACing of data based on AES
         * block size is first performed.
         * Then the remaining data that is not processed for macing is shifted to beginning of the command buffer
         * and then newly received PICC data is copied.
         * Additional, if there is room to allocate PICC data to processing buffer, required amount of PICC data
         * is copied to processing buffer. The remaining PICC data that is not copied will be taken from command
         * buffer for the next call.
         */
        else
        {
            /* Frame Buffering Option. */
            wBuffOption = (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_CONT);

            /* Perform MAC for Multiple of AES block size. */
            PHAL_NTAGXDNA_PREVIOUS_MULTIPLE(PHAL_NTAGXDNA_CMD_BUF_LEN, wAESBlockSize);
#ifdef NXPBUILD__PH_CRYPTOSYM
            if(bCommMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN)
            {
                /* Compute MAC if command buffer is multiple of AES block size. */
                PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
                    pDataParams->pCryptoDataParamsMac,
                    wBuffOption,
                    PHAL_NTAGXDNA_CMD_BUF,
                    wAESBlockSize,
                    aMac,
                    &bMacLen));
            }
#endif /* NXPBUILD__PH_CRYPTOSYM */

        /* Shift the remaining information. */
            if(PHAL_NTAGXDNA_IS_NOT_MULTIPLE_AES_BLOCK_SIZE(PHAL_NTAGXDNA_CMD_BUF_LEN))
            {
                /* Compute Remaining data to be shifted. */
                wRemData = (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_LEN - wAESBlockSize);

                /* Shift the remaining data. */
                (void) memcpy(PHAL_NTAGXDNA_CMD_BUF, &PHAL_NTAGXDNA_CMD_BUF[wAESBlockSize], wRemData);
                PHAL_NTAGXDNA_CMD_BUF_LEN = wRemData;

                /* Reset command buffer. */
                (void) memset(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], 0x00, (PHAL_NTAGXDNA_CMD_BUF_SIZE - PHAL_NTAGXDNA_CMD_BUF_LEN));

                /* Backup the offset from command buffer. */
                PHAL_NTAGXDNA_CMD_BUF_OFFSET = wRemData;

                /* Copy the data. */
                (void) memcpy(&PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_LEN], pResponse, wDataLen);
                PHAL_NTAGXDNA_CMD_BUF_LEN += wDataLen;

                /* Compute Remaining data to be copied. */
                wRemData = (uint16_t) (PHAL_NTAGXDNA_PRS_BUF_SIZE - PHAL_NTAGXDNA_PRS_BUF_LEN);
                wRemData = (uint16_t) ((wRemData < wDataLen) ? wRemData : wDataLen);
                wRemData = (uint16_t) ((wRemData > PHAL_NTAGXDNA_CMD_BUF_LEN) ? (PHAL_NTAGXDNA_CMD_BUF_LEN - PHAL_NTAGXDNA_CMD_BUF_OFFSET) : wRemData);

                /* Copy the data. */
                (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN], &PHAL_NTAGXDNA_CMD_BUF[PHAL_NTAGXDNA_CMD_BUF_OFFSET], wRemData);
                PHAL_NTAGXDNA_PRS_BUF_LEN += wRemData;

                /* Update Offset. */
                PHAL_NTAGXDNA_CMD_BUF_OFFSET = (uint16_t) (PHAL_NTAGXDNA_CMD_BUF_OFFSET + wRemData);

                /* Reset command buffer offset. */
                if(PHAL_NTAGXDNA_CMD_BUF_OFFSET == PHAL_NTAGXDNA_CMD_BUF_LEN)
                {
                    /* Check if more data needs to be copied to processing buffer. */
                    bHasMoreData = PH_OFF;
                    PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;
                    PHAL_NTAGXDNA_PRS_BUF_OFFSET = PHAL_NTAGXDNA_CMD_BUF_LEN;
                }
                else
                    bHasMoreData = PH_ON;

                wRemData = 0;
            }
            else
            {
                /* Reset command buffer length. */
                PHAL_NTAGXDNA_CMD_BUF_LEN = 0;
            }
        }

        /* Add PICC data to processing buffer only if required. */
        if(bHasMoreData)
        {
            /* Move the data to processing buffer. */
            if((PHAL_NTAGXDNA_PRS_BUF_LEN + wDataLen) < PHAL_NTAGXDNA_PRS_BUF_SIZE)
            {
                (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN], pResponse, wDataLen);
                PHAL_NTAGXDNA_PRS_BUF_LEN += (uint16_t) wDataLen;

                /* Reset Offset. */
                PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;
            }
            else
            {
                /* Compute Remaining data to be copied. */
                wRemData = (uint16_t) (PHAL_NTAGXDNA_PRS_BUF_SIZE - PHAL_NTAGXDNA_PRS_BUF_LEN);

                /* Copy the remaining data. */
                (void) memcpy(&PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_LEN], pResponse, wRemData);
                PHAL_NTAGXDNA_PRS_BUF_LEN += (uint16_t) wRemData;

                /* Backup the offset from command buffer. */
                if(PHAL_NTAGXDNA_PRS_BUF_OFFSET)
                {
                    PHAL_NTAGXDNA_CMD_BUF_OFFSET = (uint16_t) (wRemData ? (PHAL_NTAGXDNA_CMD_BUF_LEN - (wDataLen - wRemData)) : PHAL_NTAGXDNA_CMD_BUF_OFFSET);
                    PHAL_NTAGXDNA_PRS_BUF_OFFSET = 0;
                }
                else
                    PHAL_NTAGXDNA_CMD_BUF_OFFSET += (uint16_t) (wRemData + PHAL_NTAGXDNA_PRS_BUF_OFFSET);

                /* Update Has More Flag.*/
                bHasMoreData = (uint8_t) ((wDataLen - wRemData) ? PH_ON : PH_OFF);
#ifdef NXPBUILD__PH_CRYPTOSYM
            /* Decrypt the data. */
                if(bCommMode == PHAL_NTAGXDNA_COMMUNICATION_FULL)
                {
                    /* Frame Buffering Option. */
                    wBuffOption = (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | PH_EXCHANGE_BUFFER_CONT);

                    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Decrypt(
                        pDataParams->pCryptoDataParamsEnc,
                        wBuffOption,
                        PHAL_NTAGXDNA_PRS_BUF,
                        PHAL_NTAGXDNA_PRS_BUF_LEN,
                        PHAL_NTAGXDNA_PRS_BUF));

                    /* Remove Padding. */
                    if(!bHasMoreData && bIsLastFrame)
                        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_RemovePadding(
                            PH_CRYPTOSYM_PADDING_MODE_2,
                            PHAL_NTAGXDNA_PRS_BUF,
                            PHAL_NTAGXDNA_PRS_BUF_LEN,
                            PH_CRYPTOSYM_AES_BLOCK_SIZE,
                            PHAL_NTAGXDNA_PRS_BUF_LEN,
                            PHAL_NTAGXDNA_PRS_BUF,
                            &PHAL_NTAGXDNA_PRS_BUF_LEN));

                    /* Set if Decryption for remaining data needs to be performed in this call. */
                    if(bIsLastFrame)
                        bDecryptData = PH_OFF;
                }
#endif /* NXPBUILD__PH_CRYPTOSYM */

            /* Update status for Chaining. */
                wStatus = PH_ADD_COMPCODE((uint16_t) (bHasMoreData ? PH_ERR_SUCCESS_CHAINING : PH_ERR_SUCCESS), PH_COMP_AL_NTAGXDNA);
                wStatus = (uint16_t) (((bPiccStat == PHAL_NTAGXDNA_RESP_OPERATION_OK) && bIsLastFrame && !bHasMoreData) ?
                    PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA) : wStatus);
            }
        }
    }

    /* Compute MAC and verify. */
    if(bIsLastFrame || bIsISOChained)
    {
#ifdef NXPBUILD__PH_CRYPTOSYM
        if(bCommMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN)
        {
            /* Get the Buffer to use. */
            pBufferMac = PHAL_NTAGXDNA_CMD_BUF;
            wBuffMacLen = PHAL_NTAGXDNA_CMD_BUF_LEN;

            /* Set the Remaining Data  */
            wRemData = (uint16_t) ((wDataLen == wRemData) ? 0 : wDataLen);
            wRemData = (uint16_t) ((wRemData > wOffset) ? (wRemData - wOffset) : wRemData);

            do
            {
                /* Compute and Verify MAC if still pending. */
                if(PHAL_NTAGXDNA_HAS_MAC_PROCESSED == PH_OFF)
                {
                    /* Frame Buffering Option. */
                    wBuffOption = (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_BUFFER_LAST);

                    /* Update Buffering state. */
                    if(bIsISOChained)
                    {
                        wBuffOption = (uint16_t) (PH_CRYPTOSYM_MAC_MODE_CMAC | (((wRemData <= (PHAL_NTAGXDNA_CMD_BUF_SIZE - wBuffMacLen)) &&
                            bIsLastFrame) ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT));

                        /* Copy Remaining data to command buffer to fill up AES block Size. */
                        if(!PHAL_NTAGXDNA_IS_MULTIPLE_AES_BLOCK_SIZE(wBuffMacLen))
                        {
                            /* Compute the Remaining bytes to be filled for obtaining one AES block size. */
                            wAESBlockSize = PHAL_MFCC_AES_BLOCK_SIZE_DIFF(wBuffMacLen);
                            wAESBlockSize = (uint16_t) ((wRemData < (PHAL_NTAGXDNA_CMD_BUF_SIZE - wBuffMacLen)) ? wRemData : wAESBlockSize);
                            if(wBuffOption != PH_EXCHANGE_BUFFER_LAST)
                                wAESBlockSize = PHAL_MFCC_AES_BLOCK_SIZE_DIFF(wBuffMacLen);

                            /* Copy Remaining data to command buffer. */
                            (void) memcpy(&pBufferMac[wBuffMacLen], &pResponse[wOffset], wAESBlockSize);
                            wBuffMacLen += wAESBlockSize;

                            /* Update Remaining data and offset. */
                            wRemData -= wAESBlockSize;
                            wOffset += wAESBlockSize;
                        }

                        /* Check if More data is available for Macing. */
                        bFinished = (uint8_t) (wRemData < PH_CRYPTOSYM_AES_BLOCK_SIZE);
                        bHasMoreData = bFinished;

                        /* Update Finished Settings. */
                        bFinished = (uint8_t) ((wRemData == 0) && bIsLastFrame);

                        /* Update Buffering option. */
                        wBuffOption = (uint16_t) (wRemData ? PH_EXCHANGE_BUFFER_CONT : wBuffOption);
                    }

                    /* Compute the Final MAC */
                    PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_CalculateMac(
                        pDataParams->pCryptoDataParamsMac,
                        wBuffOption,
                        pBufferMac,
                        wBuffMacLen,
                        aMac,
                        &bMacLen));

                    /* Reset command buffer length and Offset. */
                    if(!bHasMoreData)
                    {
                        PHAL_NTAGXDNA_CMD_BUF_LEN = 0;
                        PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;
                    }

                    /* Move the next set of data for MAC calculation. */
                    if(bIsISOChained)
                    {
                        if(!bFinished)
                        {
                            /* Get AES block size length from remaining data length. */
                            PHAL_NTAGXDNA_PREVIOUS_MULTIPLE(wRemData, wAESBlockSize);
                            wAESBlockSize = (uint16_t) (bIsLastFrame ? wRemData : wAESBlockSize);
                            wRemData -= wAESBlockSize;

                            /* Copy the data that needs to be maced. */
                            pBufferMac = &pResponse[wOffset];
                            wBuffMacLen = wAESBlockSize;

                            /* Update Offset */
                            wOffset += wAESBlockSize;
                        }

                        /* Copy Remaining data to Command buffer length. */
                        else
                        {
                            (void) memcpy(PHAL_NTAGXDNA_CMD_BUF, &pResponse[wOffset], wRemData);
                            PHAL_NTAGXDNA_CMD_BUF_LEN = wRemData;
                        }
                    }

                    /* Verify MAC if its the last frame. */
                    if((bIsLastFrame == PH_ON) && (bFinished == PH_ON))
                    {
                        /* Truncate MAC. */
                        phalNtagXDna_Sw_Int_TruncateMac(aMac);

                        /* Verify the MAC. */
                        if(memcmp(&pResponse[wRespLen - PHAL_NTAGXDNA_TRUNCATED_MAC_LEN], aMac, PHAL_NTAGXDNA_TRUNCATED_MAC_LEN) != 0)
                        {
                            phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
                            return PH_ADD_COMPCODE(PH_ERR_INTEGRITY_ERROR, PH_COMP_AL_NTAGXDNA);
                        }
                        else
                            PHAL_NTAGXDNA_HAS_MAC_PROCESSED = PH_ON;
                    }
                }
            } while(!bFinished);

            /* Decrypt the data. */
            if(bCommMode == PHAL_NTAGXDNA_COMMUNICATION_FULL)
            {
                /* Perform Decryption of data only if the processing buffer is not holding data based on AES block size. */
                if(bDecryptData == PH_ON)
                {
                    /* Get the Actual Buffer that needs to be decrypted. */
                    pBufferEnc = pResponse;
                    wBuffEncLen = wDataLen;

                    /* Frame Buffering Option. */
                    wBuffOption = (uint16_t) (PH_CRYPTOSYM_CIPHER_MODE_CBC | (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST : PH_EXCHANGE_BUFFER_CONT));

                    /* Decrypt the information. */
                    PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_Decrypt(
                        pDataParams->pCryptoDataParamsEnc,
                        wBuffOption,
                        pBufferEnc,
                        wBuffEncLen,
                        pBufferEnc));

                    /* Remove Padding. */
                    if(bIsLastFrame)
                        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_RemovePadding(
                            PH_CRYPTOSYM_PADDING_MODE_2,
                            pBufferEnc,
                            wBuffEncLen,
                            PH_CRYPTOSYM_AES_BLOCK_SIZE,
                            wBuffEncLen,
                            pBufferEnc,
                            &wBuffEncLen));

                    /* Copy back the decrypted data to respective buffers. */
                    (void) memcpy(pResponse, pBufferEnc, wBuffEncLen);
                    wDataLen = wBuffEncLen;
                }
            }
        }
#endif /* NXPBUILD__PH_CRYPTOSYM */
    }

    if(bIsLastFrame || ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING))
    {
        /* Copy the data to response parameter. */
        if(pOutBufLen != NULL)
        {
            *ppOutBuffer = bIsISOChained ? pResponse : PHAL_NTAGXDNA_PRS_BUF;
            *pOutBufLen = (uint16_t) (bIsISOChained ? wDataLen : PHAL_NTAGXDNA_PRS_BUF_LEN);
        }
    }
    else
    {
        if(bIsISOChained == PH_ON)
        {
            *ppOutBuffer = pResponse;
            *pOutBufLen = wDataLen;
        }
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ReadData(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t bIsISOChained,
    uint8_t bCmd_ComMode, uint8_t bResp_ComMode, uint8_t * pCmdHeader, uint16_t wCmdHeaderLen, uint32_t dwDataToRead,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_PICC = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_SM = 0;

    uint32_t    PH_MEMLOC_REM dwBlockSize = 0;
    uint32_t    PH_MEMLOC_REM dwTotalReadLen = 0;

    uint16_t    PH_MEMLOC_REM wTotalFrameLen = 0;
    uint16_t    PH_MEMLOC_REM wSMBufLen = 0;
    uint16_t    PH_MEMLOC_REM wRspLen_PICC = 0;
    uint16_t    PH_MEMLOC_REM wRspLen_SM = 0;
    uint16_t    PH_MEMLOC_REM wBuffOption = PH_EXCHANGE_BUFFER_LAST;

    uint8_t     PH_MEMLOC_REM bCmdOptions = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;
    uint8_t     PH_MEMLOC_REM bShortLenApdu = PH_OFF;
    uint8_t     PH_MEMLOC_REM bFirstFrame = PH_ON;
    uint8_t     PH_MEMLOC_REM bLastFrame = PH_OFF;
    uint8_t     PH_MEMLOC_REM bHasError = PH_OFF;
    uint8_t     PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t     PH_MEMLOC_REM bHasChainedRsp = PH_OFF;
    uint8_t     PH_MEMLOC_REM bChainnedCmd = PHAL_NTAGXDNA_ADDITIONAL_FRAME;

    uint8_t *   PH_MEMLOC_REM pCmdHeader_Tmp = NULL;
    uint8_t *   PH_MEMLOC_REM pSMBuf = NULL;
    uint8_t *   PH_MEMLOC_REM pResponse_PICC = NULL;
    uint8_t *   PH_MEMLOC_REM pResponse_SM = NULL;

    PH_UNUSED_VARIABLE(dwDataToRead);

    if((bResp_ComMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN) && (pDataParams->bAuthState == PHAL_NTAGXDNA_NOT_AUTHENTICATED))
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_NTAGXDNA);
    }

    /* Move the Command Header to local variable. */
    pCmdHeader_Tmp = pCmdHeader;

    /* Reset Processing Buffer Length. */
    PHAL_NTAGXDNA_PRS_BUF_LEN = 0;

    /* Frame options for PICC Exchange. */
    bCmdOptions = (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | PHAL_NTAGXDNA_RETURN_PICC_STATUS);

    /* Back up current state of ShortLenAPDU information. */
    bShortLenApdu = pDataParams->bShortLenApdu;

    /* Compute Total frame information length.
     * This includes Command Header + Secure Messaging (Based on Communication modes)
     */
    if((wOption & 0xFF0F) == PH_EXCHANGE_DEFAULT)
    {
#ifdef NXPBUILD__PH_CRYPTOSYM
        wTotalFrameLen = (uint16_t) (wCmdHeaderLen + ((bCmd_ComMode == PHAL_NTAGXDNA_COMMUNICATION_MAC) ? 8U : 0));
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifndef NXPBUILD__PH_CRYPTOSYM
        wTotalFrameLen = (uint16_t) wCmdHeaderLen;
#endif /* NXPBUILD__PH_CRYPTOSYM */

            /* Manipulate for Read commands. */
        switch(pCmdHeader[0])
        {
            case PHAL_NTAGXDNA_CMD_READ_DATA:
                /* Compute Total Read Length */
                PHAL_NTAGXDNA_NEAREST_MULTIPLE(dwDataToRead, dwBlockSize);
                dwBlockSize = (uint16_t) ((dwDataToRead == dwBlockSize) ? (dwBlockSize + PH_CRYPTOSYM_AES_BLOCK_SIZE /* Padding */) : dwBlockSize);
#ifdef NXPBUILD__PH_CRYPTOSYM
                dwTotalReadLen = (uint16_t) ((bResp_ComMode == PHAL_NTAGXDNA_COMMUNICATION_PLAIN) ? dwDataToRead :
                    (((bResp_ComMode == PHAL_NTAGXDNA_COMMUNICATION_MAC) ? dwDataToRead : dwBlockSize) + 8U /* MAC */));
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifndef NXPBUILD__PH_CRYPTOSYM
                dwTotalReadLen = (uint16_t) dwDataToRead;
#endif /* NXPBUILD__PH_CRYPTOSYM */

                    /* Set the format of data to be sent as short APDU when,
                     * In case data to be read is not BIGISO (Less than 256 bytes).
                     */

                /* Enable Short Length APDU. */
                pDataParams->bShortLenApdu = (uint8_t) (((dwTotalReadLen > 0xFFU) && (bShortLenApdu == PH_ON)) ? PH_OFF : bShortLenApdu);
                pDataParams->bShortLenApdu = (uint8_t) ((dwDataToRead == 0) ? PH_OFF : pDataParams->bShortLenApdu);
                break;

            default:
                break;
        }
    }
    else
    {
        wTotalFrameLen = (uint16_t) (bIsISOChained ? 0 : 1U);

        pCmdHeader_Tmp = &bChainnedCmd;
        wCmdHeaderLen = wTotalFrameLen;

        wBuffOption = (uint16_t) (bIsISOChained ? wOption : PH_EXCHANGE_BUFFER_LAST);
        bFirstFrame = PH_OFF;

        /* Frame options for PICC Exchange. */
        bCmdOptions = (uint8_t) (PHAL_NTAGXDNA_OPTION_COMPLETE | (bIsISOChained ? PHAL_NTAGXDNA_EXCLUDE_PICC_STATUS : bCmdOptions));
    }

    /* Buffer Command Header for Exchange to PICC -------------------------------------------------------------------------------------- */
    if((PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE == PH_OFF) && (wTotalFrameLen != 0))
    {
        PH_CHECK_SUCCESS_FCT(wStatus_PICC, phalNtagXDna_Sw_Int_CardExchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_FIRST,
            PHAL_NTAGXDNA_OPTION_NONE,
            wTotalFrameLen,
            PH_ON,
            pCmdHeader_Tmp,
            wCmdHeaderLen,
            NULL,
            NULL,
            NULL));
    }

    /* Apply MAC on Command ------------------------------------------------------------------------------------------------------------ */
    if((wOption & 0xFF0FU) == PH_EXCHANGE_DEFAULT)
    {
        PH_CHECK_SUCCESS_FCT(wStatus_SM, phalNtagXDna_Sw_Int_ApplySM(
            pDataParams,
            PH_ON,
            PH_ON,
            bCmd_ComMode,
            pCmdHeader,
            wCmdHeaderLen,
            NULL,
            0,
            &pSMBuf,
            &wSMBufLen));
    }

    /* Exchange SM Information to PICC ------------------------------------------------------------------------------------------------- */
    do
    {
        if(PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE == PH_OFF)
        {
            wTotalFrameLen = (uint16_t) (bHasChainedRsp ? 1U : 0);
            wStatus_PICC = phalNtagXDna_Sw_Int_CardExchange(
                pDataParams,
                wBuffOption,
                bCmdOptions,
                wTotalFrameLen,
                PH_ON,
                bHasChainedRsp ? &bChainnedCmd : pSMBuf,
                (uint16_t) (bHasChainedRsp ? 1 : wSMBufLen),
                &pResponse_PICC,
                &wRspLen_PICC,
                &bPiccErrCode);
        }

        /* Update the PICC status to generic status variable. */
        wStatus = wStatus_PICC;

        /* Set the finished flag to end the loop. */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
            bHasError = PH_ON;
        }

        /* Check if response has Success status. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
        {
            bFinished = PH_ON;
            bLastFrame = PH_ON;
            PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE = PH_ON;
        }

        /* Verify and Remove Secure Messaging ---------------------------------------------------------------------------------------------- */
        if(!bHasError)
        {
            wStatus_SM = phalNtagXDna_Sw_Int_RemoveSM(
                pDataParams,
                bIsISOChained,
                bFirstFrame,
                bLastFrame,
                bResp_ComMode,
                pResponse_PICC,
                wRspLen_PICC,
                (uint8_t) (bFirstFrame ? PHAL_NTAGXDNA_RESP_OPERATION_OK : bPiccErrCode),
                &pResponse_SM,
                &wRspLen_SM);
        }

        /* Verify Status of SM */
        if(((wStatus_SM & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus_SM & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
            bFinished = PH_ON;

        /* Update the SM status to generic status variable. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            wStatus = wStatus_SM;

        /* Check if response has chaining status. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
        {
            wBuffOption = PH_EXCHANGE_DEFAULT;
            bHasChainedRsp = PH_ON;
            bFirstFrame = PH_OFF;

            /* Complete the loop in case if ISO Chained command. */
            if(bIsISOChained == PH_ON)
                bFinished = PH_ON;
        }

        /* Process response from SM. */
        if((wStatus_SM & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING)
        {
            /* End the loop. */
            bFinished = PH_ON;

            /* Update the Generic Status. */
            wStatus = wStatus_SM;
        }
        else
        {
            PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE = PH_OFF;
#ifdef NXPBUILD__PH_CRYPTOSYM
            PHAL_NTAGXDNA_HAS_MAC_PROCESSED = PH_OFF;
#endif /* NXPBUILD__PH_CRYPTOSYM */
            PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0;
        }

        /* Copy the response to parameter. */
        if(pRespLen != NULL)
        {
            *ppResponse = pResponse_SM;
            *pRespLen = wRspLen_SM;
        }
    } while(!bFinished);

    /* Perform Reset Authentication. */
    if(bHasError)
    {
        /* Reset Authentication State. */
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
    }

    /* Revert back state of ShortLenAPDU information. */
    pDataParams->bShortLenApdu = bShortLenApdu;

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_WriteData(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bCmd_ComMode, uint8_t bResp_ComMode,
    uint8_t bResetAuth, uint8_t * pCmdHeader, uint16_t wCmdHeaderLen, uint8_t * pCmdData, uint32_t dwCmdDataLen,
    uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_PICC = 0;
    phStatus_t  PH_MEMLOC_REM wStatus_SM = 0;

    uint32_t    PH_MEMLOC_REM dwRemData = 0;
    uint32_t    PH_MEMLOC_REM dwBlockSize = 0;
    uint32_t    PH_MEMLOC_REM dwOffset = 0;

    uint16_t    PH_MEMLOC_REM wFrameLen = 0;
    uint16_t    PH_MEMLOC_REM wTotalFrameLen = 0;
    uint16_t    PH_MEMLOC_REM wExchangeLen = 0;
    uint16_t    PH_MEMLOC_REM wTotalBytesExchanged = 0;
    uint16_t    PH_MEMLOC_REM wBytesExchanged = 0;
    uint16_t    PH_MEMLOC_REM wDataLen = 0;
    uint16_t    PH_MEMLOC_REM wRemData_Exchange = 0U;
    uint16_t    PH_MEMLOC_REM wBuffOption_PICC = PH_EXCHANGE_BUFFER_LAST;

    uint8_t     PH_MEMLOC_REM bShortLenApdu = PH_OFF;
    uint8_t     PH_MEMLOC_REM bIsFirstFrame = PH_OFF;
    uint8_t     PH_MEMLOC_REM bIsLastFrame = PH_OFF;
    uint8_t     PH_MEMLOC_REM bFinished = PH_OFF;
    uint8_t     PH_MEMLOC_REM bHasError = PH_OFF;
    uint8_t     PH_MEMLOC_REM bCmdOptions = 0;
    uint8_t     PH_MEMLOC_REM bPiccErrCode = 0;

    uint8_t *   PH_MEMLOC_REM pSMBuf = NULL;
    uint16_t    PH_MEMLOC_REM wSMBufLen = 0;

    uint8_t *   PH_MEMLOC_REM pResponse_PICC = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen_PICC = 0;

    if((bCmd_ComMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN) && (pDataParams->bAuthState == PHAL_NTAGXDNA_NOT_AUTHENTICATED))
    {
        return PH_ADD_COMPCODE(PH_ERR_USE_CONDITION, PH_COMP_AL_NTAGXDNA);
    }

    /* Clear Processing buffer. */
    if(PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE == PH_OFF)
    {
        PHAL_NTAGXDNA_PRS_BUF_LEN = 0;
    }

    /* Get PICC Frame size. */
    PH_CHECK_SUCCESS_FCT(wStatus, phalNtagXDna_Sw_Int_GetFrameLen(pDataParams, &wFrameLen));

    /* Get Encrypted Data Block Size. */
    PHAL_NTAGXDNA_NEAREST_MULTIPLE(dwCmdDataLen, dwBlockSize);
    dwBlockSize = (uint16_t) ((dwCmdDataLen == dwBlockSize) ? (dwBlockSize + PH_CRYPTOSYM_AES_BLOCK_SIZE /* Padding */) : dwBlockSize);

    /* Compute Total frame information length.
     * This includes Command Header + Command Data (Based on Communication modes)
     */
#ifdef NXPBUILD__PH_CRYPTOSYM
    wTotalFrameLen = (uint16_t) (wCmdHeaderLen + ((bCmd_ComMode == PHAL_NTAGXDNA_COMMUNICATION_PLAIN) ? dwCmdDataLen :
        (((bCmd_ComMode == PHAL_NTAGXDNA_COMMUNICATION_MAC) ? dwCmdDataLen : dwBlockSize) + 8U /* MAC */)));
#endif /* NXPBUILD__PH_CRYPTOSYM */
#ifndef NXPBUILD__PH_CRYPTOSYM
    wTotalFrameLen = (uint16_t) (wCmdHeaderLen + dwCmdDataLen);
#endif /* NXPBUILD__PH_CRYPTOSYM */

    /* Compute the maximum bytes that be transferred in one frame. */
    wExchangeLen = wTotalFrameLen;

    /* Back up current state of ShortLenAPDUinformation. */
    bShortLenApdu = pDataParams->bShortLenApdu;

    /* Enable Extended length APDU format in case if CommandLen + DataLen + ISO7816 Header exceeds frame size. */
    if(PHAL_NTAGXDNA_ABS(wExchangeLen, 5U) > wFrameLen)
        pDataParams->bShortLenApdu = PH_OFF;

    /* Buffer Command Header to PICC exchange buffer ----------------------------------------------------------------------------------- */
    PH_CHECK_SUCCESS_FCT(wStatus_PICC, phalNtagXDna_Sw_Int_CardExchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHAL_NTAGXDNA_OPTION_NONE,
        wExchangeLen,
        PH_ON,
        pCmdHeader,
        wCmdHeaderLen,
        NULL,
        NULL,
        NULL));

    /* Update Remaining data for SM and. */
    dwRemData = dwCmdDataLen;

    /* Set the Frame options. */
    bIsFirstFrame = PH_ON;
    bIsLastFrame = PH_OFF;

    /* Set Command header as exchanged information  */
    wTotalBytesExchanged = wCmdHeaderLen;
    wBytesExchanged = (uint16_t) (wCmdHeaderLen - 1U);

    do
    {
        /* Exclude if all data is processed but PICC exchange is pending. */
        if(bIsLastFrame == PH_OFF)
        {
            /* Update the data to be used for SM Application. */
            wDataLen = (uint16_t) ((dwRemData > wFrameLen) ? wFrameLen : dwRemData);
            wDataLen = (uint16_t) (((wCmdHeaderLen + wDataLen) > wFrameLen) ? PHAL_NTAGXDNA_ABS(wFrameLen, wCmdHeaderLen) : wDataLen);

            /* Update Remaining data to be used. */
            dwRemData -= wDataLen;

            /* Set Framing Options. */
            bIsLastFrame = (uint8_t) (dwRemData <= 0U);

            /* Apply Secure Messaging on Command ------------------------------------------------------------------------------------------- */
            wStatus_SM = phalNtagXDna_Sw_Int_ApplySM(
                pDataParams,
                bIsFirstFrame,
                bIsLastFrame,
                bCmd_ComMode,
                pCmdHeader,
                wCmdHeaderLen,
                &pCmdData[dwOffset],
                wDataLen,
                &pSMBuf,
                &wSMBufLen);

            /* Clear Framing Option */
            bIsFirstFrame = PH_OFF;

            /* Update the PICC status to generic status variable. */
            wStatus = wStatus_SM;

            /* Update the Offset */
            dwOffset += wDataLen;
        }

        /* Verify Status of SM */
        if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
        {
            bFinished = PH_ON;
            bHasError = PH_ON;
        }

        /* Set the finished flag to end the loop. */
        bFinished = bIsLastFrame;

        /* Exchange data to PICC. */
        if(bHasError == PH_OFF)
        {
            /* Backup data to be exchanged. */
            wRemData_Exchange = wSMBufLen;
            wDataLen = wSMBufLen;

            do
            {
                wBuffOption_PICC = PH_EXCHANGE_BUFFER_CONT;

                /* Update remaining data to exchange */
                wRemData_Exchange -= wDataLen;

                /* Update Buffer Options */
                wBuffOption_PICC = (uint16_t) (bIsLastFrame ? PH_EXCHANGE_BUFFER_LAST : wBuffOption_PICC);

                /* Update options for exchange interface. */
                bCmdOptions = (uint8_t) ((wBuffOption_PICC == PH_EXCHANGE_BUFFER_CONT) ? PHAL_NTAGXDNA_OPTION_NONE : PHAL_NTAGXDNA_OPTION_COMPLETE);
                bCmdOptions |= PHAL_NTAGXDNA_RETURN_PICC_STATUS;

                /* Exchange data. */
                if((wDataLen > 0U) || (bIsLastFrame == PH_ON))
                {
                    wStatus_PICC = phalNtagXDna_Sw_Int_CardExchange(
                        pDataParams,
                        wBuffOption_PICC,
                        bCmdOptions,
                        wExchangeLen,
                        PH_ON,
                        pSMBuf,
                        wDataLen,
                        &pResponse_PICC,
                        &wRspLen_PICC,
                        &bPiccErrCode);

                    /* Set bytes exchanged fro current frame. */
                    wBytesExchanged += wDataLen;

                    /* Set bytes exchanged */
                    wTotalBytesExchanged += wDataLen;
                }

                /* Update the SM status to generic status variable. */
                wStatus = wStatus_PICC;

                /* Set the finished flag to end the loop. */
                if(bPiccErrCode != PHAL_NTAGXDNA_RESP_OPERATION_OK)
                {
                    bHasError = PH_ON;
                    bFinished = PH_ON;
                    wRemData_Exchange = 0U;
                }

                /* Clear Command Header Length */
                wCmdHeaderLen = 0;

                /* Clear PICC Response Code */
                bPiccErrCode = 0U;

            } while(wRemData_Exchange != 0U);
        }
        else
        {
            /* Set the finished flag to end the loop. */
            bFinished = PH_ON;
        }
    } while(!bFinished);

    /* Clear Offsets and length. */
    PHAL_NTAGXDNA_CMD_BUF_LEN = 0U;
    PHAL_NTAGXDNA_CMD_BUF_OFFSET = 0U;

    PHAL_NTAGXDNA_PRS_BUF_LEN = 0U;
    PHAL_NTAGXDNA_PRS_BUF_OFFSET = 0U;

    /* Verify and Remove Secure Messaging ---------------------------------------------------------------------------------------------- */
    if(!bHasError)
    {
        if(bResp_ComMode != PHAL_NTAGXDNA_COMMUNICATION_PLAIN)
        {
            PH_CHECK_SUCCESS_FCT(wStatus_SM, phalNtagXDna_Sw_Int_RemoveSM(
                pDataParams,
                PH_ON,
                PH_ON,
                PH_ON,
                bResp_ComMode,
                pResponse_PICC,
                wRspLen_PICC,
                bPiccErrCode,
                ppResponse,
                pRespLen));
        }
        else
        {
            /* In case of SUCCESS and Communication mode as PLAIN, increment the command counter. */
            if(wStatus == PH_ERR_SUCCESS)
                pDataParams->wCmdCtr++;

            /* Copy response data to parameter */
            if(pRespLen != NULL)
            {
                *ppResponse = pResponse_PICC;
                *pRespLen = wRspLen_PICC;
            }
        }
    }

    /* Perform Reset Authentication. */
    if(bResetAuth || bHasError)
    {
        /* Clear Reset Authentication flag. */
        bResetAuth = PH_OFF;

        /* Reset Authentication State. */
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
    }

#ifdef NXPBUILD__PH_CRYPTOSYM
    PHAL_NTAGXDNA_HAS_MAC_PROCESSED = PH_OFF;
#endif /* NXPBUILD__PH_CRYPTOSYM */
    PHAL_NTAGXDNA_IS_PICC_DATA_COMPLETE = PH_OFF;

    /* Revert back state of ShortLenAPDU information. */
    pDataParams->bShortLenApdu = bShortLenApdu;

    return wStatus;
}





phStatus_t phalNtagXDna_Sw_Int_ISOSelectFile(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t * pData, uint16_t wDataLen,
    uint8_t * pLe, uint8_t bLeLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Buffer Command Data. */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatus = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatus = phalNtagXDna_Int_ComputeErrorResponse(pDataParams, wStatus);

    /*  Check for Success and for LIMITED FUNCTIONALITY error. In both cases, FCI would be returned */
    if(wStatus == PH_ERR_SUCCESS)
    {
        if(wRspLen >= 2U)
        {
            *pRspLen = wRspLen - 2U;
        }

        if(ppResponse != NULL)
        {
            *ppResponse = pResponse;
        }
    }
    else
    {
        /* Nothing to do here */
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOReadBinary(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pLe,
    uint8_t bLeLen, uint8_t ** ppResponse, uint16_t * pRspLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;
    uint16_t    PH_MEMLOC_REM wOptions_Tmp = 0;

    if((wOption & PH_EXCHANGE_MODE_MASK) == PH_EXCHANGE_RXCHAINING)
    {
        bLeLen = 0;
        wOptions_Tmp = PH_EXCHANGE_RXCHAINING;
    }
    else
    {
        wOptions_Tmp = PH_EXCHANGE_BUFFER_LAST;
    }

    /* Buffer LE and Exchange the command to PICC.  */
    wStatus = phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        wOptions_Tmp,
        pLe,
        bLeLen,
        &pResponse,
        &wRspLen);

    /* Reset Authentication state. */
    if(((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS) && ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING))
    {
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Combine Sw1 and Sw2 status codes. */
        wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

        /* Evaluate the Status. */
        wStatus = phalNtagXDna_Sw_Int_ValidateResponse(pDataParams, PHAL_NTAGXDNA_ISO7816_APDU_CMD, wStatus, wStatus_Rsp);

        /* Copy the response to parameter. */
        *ppResponse = pResponse;
        *pRspLen = wRspLen;

        /* Decrement Status code. */
        if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS)
            *pRspLen = *pRspLen - 2U;
    }

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_ISOUpdateBinary(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t * pData, uint16_t wDataLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = PH_ERR_SUCCESS;
    phStatus_t  PH_MEMLOC_REM wStatus_Rsp = PH_ERR_SUCCESS;
    uint8_t     PH_MEMLOC_REM *pResponse = NULL;
    uint16_t    PH_MEMLOC_REM wRspLen = 0;

    /* Buffer LE and Exchange the command to PICC.  */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_ExchangeL4(
        pDataParams->pPalMifareDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        pData,
        wDataLen,
        &pResponse,
        &wRspLen));

    /* Combine Sw1 and Sw2 status codes. */
    wStatus_Rsp = (uint16_t) ((pResponse[wRspLen - 2U] << 8U) | pResponse[wRspLen - 1U]);

    /* Evaluate the Status. */
    wStatus = phalNtagXDna_Int_ComputeErrorResponse(pDataParams, wStatus_Rsp);

    /* Reset Authentication state. */
    if((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS)
    {
        phalNtagXDna_Sw_Int_ResetAuthStatus(pDataParams);
    }
    else
    {
        /* Do Nothing */
    }

    return wStatus;
}






phStatus_t phalNtagXDna_Sw_Int_ResetAuthStatus(phalNtagXDna_Sw_DataParams_t * pDataParams)
{
    phStatus_t PH_MEMLOC_REM wStatus = PH_ERR_USE_CONDITION;

    pDataParams->wCmdBufLen = 0;
    pDataParams->wCmdBufOffset = 0;
    pDataParams->wPrsBufLen = 0;
    pDataParams->wPrsBufOffset = 0;
    pDataParams->wCmdCtr = 0;
    pDataParams->bAuthState = PHAL_NTAGXDNA_NOT_AUTHENTICATED;
    pDataParams->bKeyNo = 0xFFU;
    pDataParams->bPICCDataComplete = PH_OFF;
#ifdef NXPBUILD__PH_CRYPTOSYM
    pDataParams->bHasMACProcessed = PH_OFF;
    pDataParams->bIsENCPending = PH_OFF;
#endif /* NXPBUILD__PH_CRYPTOSYM */

    (void) memset(pDataParams->pCmdBuf, 0x00, PHAL_NTAGXDNA_CMD_BUFFER_SIZE_MINIMUM);
    (void) memset(pDataParams->pPrsBuf, 0x00, PHAL_NTAGXDNA_PRS_BUFFER_SIZE_MINIMUM);

#ifdef NXPBUILD__PH_CRYPTOSYM
    (void) memset(pDataParams->aSesAuthENCKey, 0x00, sizeof(pDataParams->aSesAuthENCKey));
    (void) memset(pDataParams->aSesAuthMACKey, 0x00, sizeof(pDataParams->aSesAuthMACKey));
    (void) memset(pDataParams->aSesNonce, 0x00, sizeof(pDataParams->aSesNonce));
    (void) memset(pDataParams->aTi, 0x00, PHAL_NTAGXDNA_SIZE_TI);

    phCryptoASym_InvalidateKey(pDataParams->pCryptoDataParamsASym);
#endif /* NXPBUILD__PH_CRYPTOSYM */

    return wStatus;
}

phStatus_t phalNtagXDna_Sw_Int_GetFrameLen(phalNtagXDna_Sw_DataParams_t * pDataParams, uint16_t * pFrameLen)
{
    uint16_t    PH_MEMLOC_REM wStatus = 0;
    uint16_t    PH_MEMLOC_REM wFSI = 0;

    PH_CHECK_SUCCESS_FCT(wStatus, phpalMifare_GetConfig(pDataParams->pPalMifareDataParams,
        PHPAL_I14443P4_CONFIG_FSI, &wFSI));

    /* Extract FSCI (PICC Frame Size) and Update the parameter. */
    *pFrameLen = aFrameSize[(uint8_t) (wFSI & 0x00FFU)];

    /* Remove the ISO header. */
    *pFrameLen -= 4;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

#ifdef NXPBUILD__PH_CRYPTOSYM
phStatus_t phalNtagXDna_Sw_Int_GenerateIv(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bIsResponse, uint8_t * pTi,
    uint16_t wCmdCtr)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint8_t     PH_MEMLOC_REM bIndex = 0;
    uint8_t     PH_MEMLOC_REM bCmdCtrMsb = (uint8_t) (wCmdCtr >> 8U);
    uint8_t     PH_MEMLOC_REM bCmdCtrLsb = (uint8_t) (wCmdCtr & 0x00FFU);
    uint8_t     PH_MEMLOC_REM aIv[PH_CRYPTOSYM_AES_BLOCK_SIZE];

    memset(aIv, 0, PH_CRYPTOSYM_AES_BLOCK_SIZE); /* PRQA S 3200 */

    if(bIsResponse)
    {
        /* Frame the IV for RespData as 0x5A||0xA5||TI||CmdCtr||0x0000000000000000 */
        aIv[bIndex++] = 0x5AU;
        aIv[bIndex++] = 0xA5U;
    }
    else
    {
        /* Frame the IV for CmdData as 0xA5||0x5A||TI||CmdCtr||0x0000000000000000  */
        aIv[bIndex++] = 0xA5U;
        aIv[bIndex++] = 0x5AU;
    }

    aIv[bIndex++] = pTi[0];
    aIv[bIndex++] = pTi[1U];
    aIv[bIndex++] = pTi[2U];
    aIv[bIndex++] = pTi[3U];
    aIv[bIndex++] = bCmdCtrLsb;
    aIv[bIndex++] = bCmdCtrMsb;

    /* Load Zero IV. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Encrypt IV */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_Encrypt(
        pDataParams->pCryptoDataParamsEnc,
        PH_CRYPTOSYM_CIPHER_MODE_ECB,
        aIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE,
        aIv));

    /* Load Encrypted Iv. */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsEnc,
        aIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

phStatus_t phalNtagXDna_Sw_Int_ComputeSessionKey_SDM(phalNtagXDna_Sw_DataParams_t * pDataParams, uint8_t bOption, uint8_t bSdmOption,
    uint16_t wKeyNo_SDM, uint16_t wKeyVer_SDM, uint8_t * pUid, uint8_t bUidLen, uint8_t * pSDMReadCtr)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    uint32_t    PH_MEMLOC_REM dwSDMReadCtr = 0;
    uint16_t    PH_MEMLOC_REM wKeyType;
    uint8_t     PH_MEMLOC_REM bKeyLen = 0;
    uint8_t     PH_MEMLOC_REM bSV1_ALen = 0;
    uint8_t     PH_MEMLOC_REM bSV1_BLen = 0;

    uint8_t     PH_MEMLOC_REM *pKey = NULL;

    uint8_t     PH_MEMLOC_REM *pSV_A = NULL;
    uint8_t     PH_MEMLOC_REM *pSV_B = NULL;

    /* Formation of SDM Read Counter as double word value - SDM Read Counter shall be communicated LSB first. */
    if(pSDMReadCtr != NULL)
    {
        dwSDMReadCtr = pSDMReadCtr[0] | (pSDMReadCtr[1U] << 8U) | (pSDMReadCtr[2U] << 16U) | (pSDMReadCtr[3U] << 24U);

        /* If SDMRead Counter is 0x00FFFFFF, then return error */
        if(dwSDMReadCtr >= 0x00FFFFFFU)
            return PH_ADD_COMPCODE(PH_ERR_PARAMETER_OVERFLOW, PH_COMP_AL_NTAGXDNA);
    }

    /* Set the pointer for the Random Numbers. */
    pKey = &PHAL_NTAGXDNA_PRS_BUF[PHAL_NTAGXDNA_PRS_BUF_SIZE - (PH_CRYPTOSYM_AES256_KEY_SIZE * 2U)];

    pSV_A = &PHAL_NTAGXDNA_CMD_BUF[PH_CRYPTOSYM_AES256_KEY_SIZE * 0];
    pSV_B = &PHAL_NTAGXDNA_CMD_BUF[PH_CRYPTOSYM_AES256_KEY_SIZE * 1U];

    /* Get Key out of the key store object */
    PH_CHECK_SUCCESS_FCT(wStatus, phKeyStore_GetKey(
        pDataParams->pKeyStoreDataParams,
        wKeyNo_SDM,
        wKeyVer_SDM,
        PH_CRYPTOSYM_AES256_KEY_SIZE,
        pKey,
        &wKeyType));

    /* Validate KeyType. */
    PHAL_NTAGXDNA_VALIDATE_KEYTYPE(wKeyType);

    /* Frame Session Vector
    * For AES128
    *      SV1  = 0xC3 || 0x3C || 0x00 || 0x01 || 0x00 || 0x80 || [VCUID] || [SDMReadCtr] || [Zero Padding]
    *      SV2  = 0x3C || 0xC3 || 0x00 || 0x01 || 0x00 || 0x80 || [VCUID] || [SDMReadCtr] || [Zero Padding]
    *
    *      SesSDMFileReadENC   => MAC(SDMFileReadKey, SV1)
    *      SesSDMFileReadMAC   => MAC(SDMFileReadKey, SV2)
    *
    * For AES256
    *      SV1a = 0xC3 || 0x3C || 0x00 || 0x01 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
    *      SV1b = 0xC3 || 0x3C || 0x00 || 0x02 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
    *      SV2a = 0x3C || 0xC3 || 0x00 || 0x01 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
    *      SV2b = 0x3C || 0xC3 || 0x00 || 0x02 || 0x01 || 0x00 || [VCUID] || [SDMReadCtr] || [Zero Padding]
    *
    *      SesSDMFileReadENC   => PRF(SDMFileReadKey, SV1a) || PRF(SDMFileReadKey, SV1b)
    *      SesSDMFileReadMAC   => PRF(SDMFileReadKey, SV2a) || PRF(SDMFileReadKey, SV2b)
    *
    */
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((bOption == PHAL_NTAGXDNA_SESSION_MAC) ? 0x3CU : 0xC3U);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((bOption == PHAL_NTAGXDNA_SESSION_MAC) ? 0xC3U : 0x3CU);
    pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = 0x00;

    pSV_A[bSV1_ALen++] = 0x01U;
    pSV_B[bSV1_BLen++] = 0x02U;

    pSV_A[bSV1_ALen++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x00 : 0x01U);
    pSV_B[bSV1_BLen++] = 0x01U;

    pSV_A[bSV1_ALen++] = (uint8_t) ((wKeyType == PH_KEYSTORE_KEY_TYPE_AES128) ? 0x80U : 0x00);
    pSV_B[bSV1_BLen++] = 0x00;

    /* Add UID. */
    if(bSdmOption & PHAL_NTAGXDNA_VCUID_PRESENT)
    {
        if(pUid == NULL)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_AL_NTAGXDNA);
        }
        else
        {
            (void) memcpy(&pSV_A[bSV1_ALen], pUid, bUidLen);
            bSV1_ALen += bUidLen;
            (void) memcpy(&pSV_B[bSV1_BLen], pUid, bUidLen);
            bSV1_BLen += bUidLen;
        }
    }

    /* Add SDM Read Counter. */
    if(bSdmOption & PHAL_NTAGXDNA_RDCTR_PRESENT)
    {
        if(pSDMReadCtr != NULL)
        {
            pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) (dwSDMReadCtr & 0xFFU);
            pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((dwSDMReadCtr >> 8U) & 0xFFU);
            pSV_A[bSV1_ALen++] = pSV_B[bSV1_BLen++] = (uint8_t) ((dwSDMReadCtr >> 16U) & 0xFFU);
        }
    }

    /* Get AES block size for Session vector input. */
    PHAL_NTAGXDNA_NEAREST_MULTIPLE(bSV1_ALen, bSV1_ALen);
    PHAL_NTAGXDNA_NEAREST_MULTIPLE(bSV1_BLen, bSV1_BLen);

    /* load key */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
        pDataParams->pCryptoDataParamsMac,
        pKey,
        wKeyType));

    /* Load Iv */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadIv(
        pDataParams->pCryptoDataParamsMac,
        phalNtagXDna_Sw_ZeroIv,
        PH_CRYPTOSYM_AES_BLOCK_SIZE));

    /* Generate SDMFileRead MAC Session Key (SV1 or SV1a). */
    PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
        pDataParams->pCryptoDataParamsMac,
        (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
        pSV_A,
        bSV1_ALen,
        pKey,
        &bKeyLen));

    /* Generate SDMFileRead MAC Session Key (SV1b). */
    if(wKeyType == PH_KEYSTORE_KEY_TYPE_AES256)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_CalculateMac(
            pDataParams->pCryptoDataParamsMac,
            (PH_CRYPTOSYM_MAC_MODE_CMAC | PH_EXCHANGE_DEFAULT),
            pSV_B,
            bSV1_BLen,
            &pKey[bKeyLen],
            &bKeyLen));

        /* Increment Key to make it as 32 bytes. */
        bKeyLen += PH_CRYPTOSYM_AES_BLOCK_SIZE;
    }

    /* Load SDM MAC Session Key (SesSDMMAC) */
    if(bOption == PHAL_NTAGXDNA_SESSION_MAC)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsMac,
            pKey,
            wKeyType));
    }

    /* Load SDM MAC Session Key (SesSDMENC) */
    else
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoSym_LoadKeyDirect(
            pDataParams->pCryptoDataParamsEnc,
            pKey,
            wKeyType));
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_AL_NTAGXDNA);
}

void phalNtagXDna_Sw_Int_TruncateMac(uint8_t * pMac)
{
    uint8_t PH_MEMLOC_REM bIndex = 0;
    uint8_t PH_MEMLOC_REM bIndex2 = 0;

    for(bIndex = 1U, bIndex2 = 0; bIndex < 16U; bIndex += 2U, bIndex2++)
    {
        pMac[bIndex2] = pMac[bIndex];
    }
}
#endif /* NXPBUILD__PH_CRYPTOSYM */

#endif /* NXPBUILD__PHAL_NTAGXDNA_SW */
