/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * PCSC HAL Command Layer.
 *
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHHALHW_PCSC_CMD_H
#define PHHALHW_PCSC_CMD_H

#include <ph_Status.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_PCSC

/** \defgroup phhalHw_Pcsc_Cmd Instruction
* \brief PCSC command functions
* @{
*/

/**
* Get UID of detected card.\n
*
* */
#define PHHAL_HW_PCSC_UID                     0x00U

/**
* Get historical bytes of detected card.\n
*
* */
#define PHHAL_HW_PCSC_HIST                    0x01U

/**
* Card key type.\n
*
* */
#define PHHAL_HW_PCSC_KEY_CARD                0x00U

/**
* Reader key type.\n
*
* */
#define PHHAL_HW_PCSC_KEY_READER              0x01U

/**
* Key transmission type - plain.\n
*
* */
#define PHHAL_HW_PCSC_TX_PLAIN                0x00U

/**
* Key transmission type - secured.\n
*
* */
#define PHHAL_HW_PCSC_TX_SECURED              0x01U

/**
* Keys are loaded into the reader volatile memory.\n
*
* */
#define PHHAL_HW_PCSC_KEY_LOC_VOLATILE        0x00U

/**
* Keys are loaded into the reader non-volatile memory.\n
*
* */
#define PHHAL_HW_PCSC_KEY_LOC_NONVOLATILE     0x01U

/**
* Key A for Mifare.\n
*
* */
#define PHHAL_HW_PCSC_KEY_A                   0x60U

/**
* Key B for PicoTag and PicoPass Debit key or Credit key.\n
*
* */
#define PHHAL_HW_PCSC_KEY_B                   0x61U

/**
* Switch Protocol Type A (ISO 14443A).\n
*
* */
#define PHHAL_HW_PCSC_ISO14443A               0x00U

/**
* Switch Protocol ISO 15693.\n
*
* */
#define PHHAL_HW_PCSC_ISO15693                0x02U

/**
* Switch Protocol Layer 2.\n
*
* */
#define PHHAL_HW_PCSC_LAYER_2                 0x02U

/**
* Switch Protocol Layer 3.\n
*
* */
#define PHHAL_HW_PCSC_LAYER_3                 0x03U

/**
* Switch Protocol Layer 4.\n
*
* */
#define PHHAL_HW_PCSC_LAYER_4                 0x04U

/**
* \brief PCSC Get Data Command.
*
* This command can be used to get the UID or historical bytes of
* the card detected.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Communication error.
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_GetData(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bOption,                               /**< [In] #PHHAL_HW_PCSC_UID or #PHHAL_HW_PCSC_HIST. */
    uint8_t * pData,                               /**< [Out] UID or Historical bytes based on bOption. */
    uint16_t * pDataLength                         /**< [Out] Length of pData. */
    );

/**
* \brief The command will just load (write) the keys in the reader memory.
*
* The key will be of two different types; the reader key and the card key.
* Reader Key: This key will be used to protect the transmission of secured
* data e.g. card key from the application to the reader.
* Card Key: This is the card specific key (e.g. for Mifare it is Mifare key).
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_BUFFER_OVERFLOW \b Not enough TX buffer space .
* \retval #PH_ERR_INTERNAL_ERROR Communication error.
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_LoadKeys(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bKeyType,                              /**< [In] Key Type #PHHAL_HW_PCSC_KEY_CARD or #PHHAL_HW_PCSC_KEY_READER. */
    uint8_t bTxType,                               /**< [In] Transmission Type.#PHHAL_HW_PCSC_TX_PLAIN or #PHHAL_HW_PCSC_TX_SECURED */
    uint8_t bReaderEncryptionKeyNo,                /**< [In] Reader encryption key number if #PHHAL_HW_PCSC_TX_SECURED is used else 0. */
    uint8_t bKeyLocation,                          /**< [In] Location to store key in reader #PHHAL_HW_PCSC_KEY_LOC_VOLATILE or #PHHAL_HW_PCSC_KEY_LOC_NONVOLATILE. */
    uint8_t bKeyNumber,                            /**< [In] Key number to be used for current key. */
    uint8_t * pKey,                                /**< [In] Key to be loaded. */
    uint8_t bKeyLength                             /**< [In] Key length. */
    );

/**
* \brief PCSC General Authenticate command.
*
* This command shall be used to authenticate cards (like Mifare cards). The
* key number shall be same as the key number specified in LoadKeys command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Communication error.
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_GeneralAuthenticate(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bKeyType,                              /**< [In] Key Type #PHHAL_HW_PCSC_KEY_A or #PHHAL_HW_PCSC_KEY_B. */
    uint8_t bKeyNumber,                            /**< [In] Key number to be used for authentication. */
    uint16_t wAddress                              /**< [In] Block number or the starting byte number of the card to be authenticated. */
    );

/**
* \brief PCSC Verify command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_BUFFER_OVERFLOW \b Not enough TX buffer space .
* \retval #PH_ERR_INTERNAL_ERROR Communication error.
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_Verify(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bTxType,                               /**< [In] Transmission Type.#PHHAL_HW_PCSC_TX_PLAIN or #PHHAL_HW_PCSC_TX_SECURED */
    uint8_t bReaderEncryptionKeyNo,                /**< [In] Reader encryption key number if #PHHAL_HW_PCSC_TX_SECURED is used else 0. */
    uint8_t bRefData,                              /**< [In] Reference data. */
    uint8_t * pPin,                                /**< [In] Pin. */
    uint8_t bPinLength                             /**< [In] Pin length. */
    );

/**
* \brief PCSC Read Binary command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_BUFFER_OVERFLOW \b Not enough RX buffer space .
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_ReadBinary(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wAddress,                             /**< [In] Block number or the starting address to read from. */
    uint8_t bBytesToRead,                          /**< [In] Expected read data length; Le field (extended Le not supported). */
    uint8_t ** ppData,                             /**< [Out] Read data. */
    uint16_t * pDataLength                         /**< [Out] Read data length. */
    );

/**
* \brief PCSC Update Binary command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_BUFFER_OVERFLOW \b Not enough TX buffer space .
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_UpdateBinary(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wAddress,                             /**< [In] Block number or the starting address to write from. */
    uint8_t * pData,                               /**< [In] Data to write. */
    uint8_t bDataLength                            /**< [In] Data length. */
    );

/**
* \brief PCSC Increment command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_BUFFER_OVERFLOW \b Not enough TX buffer space .
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_Increment(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bBlockNo,                              /**< [In] Block Number. */
    uint8_t * pValue                               /**< [In] Increment value. */
    );

/**
* \brief PCSC Decrement command.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_BUFFER_OVERFLOW \b Not enough TX buffer space .
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
*/
phStatus_t phhalHw_Pcsc_Cmd_Decrement(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bBlockNo,                              /**< [In] Block Number. */
    uint8_t * pValue                               /**< [In] Decrement value. */
    );

/**
* \brief PCSC Start Transparent Session (Part of Manage Session Command).
*
* The application shall send this data object to inform the reader of
* preparation for a transparent session.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_StartTransparentSession(
    phhalHw_Pcsc_DataParams_t * pDataParams        /**< [In] Pointer to this layer's parameter structure. */
    );

/**
* \brief PCSC End Transparent Session (Part of Manage Session Command).
*
* The application shall send this to inform the reader of
* the end of a transparent session.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_EndTransparentSession(
    phhalHw_Pcsc_DataParams_t * pDataParams        /**< [In] Pointer to this layer's parameter structure. */
    );

/**
* \brief PCSC Version command (Part of Manage Session Command).
*
* The application can send this to get the version number of the IFD handler.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_Version(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t * pVersion                             /**< [Out] Version number of IFD handler. */
    );

/**
* \brief PCSC RF off command (Part of Manage Session Command).
*
* Application can send this to switch off the RF field. This command will only
* be accepted after the start and before the end of a transparent session.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_USE_CONDITION \b Transparent session not on.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_RfOff(
    phhalHw_Pcsc_DataParams_t * pDataParams        /**< [In] Pointer to this layer's parameter structure. */
    );

/**
* \brief PCSC RF on command (Part of Manage Session Command).
*
* Application can send this to switch on the RF field. This command will only
* be accepted after the start and before the end of a transparent session.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_USE_CONDITION \b Transparent session not on.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_RfOn(
    phhalHw_Pcsc_DataParams_t * pDataParams        /**< [In] Pointer to this layer's parameter structure. */
    );

/**
* \brief Call PCSC RF off and RF on commands with a delay in between.
*
* Application can use this to do RF field reset. This command will only
* be accepted after the start and before the end of a transparent session.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_USE_CONDITION \b Transparent session not on.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_RfReset(
    phhalHw_Pcsc_DataParams_t * pDataParams        /**< [In] Pointer to this layer's parameter structure. */
    );

/**
* \brief PCSC Get Parameter command.
*
* Application can send this to get the value of a configurable parameter.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_GetParam(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bParamType,                            /**< [In] Parameter type. */
    uint8_t * pParamValue,                         /**< [Out] Parameter value. */
    uint8_t * pParamLength                         /**< [Out] Parameter value length (in bytes). */
    );

/**
* \brief PCSC Set Parameter command.
*
* Application can send this to set the value of a configurable parameter.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_SetParam(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bParamType,                            /**< [In] Parameter type. */
    uint8_t * pParamValue,                         /**< [In] Parameter value. */
    uint8_t bParamLength                           /**< [In] Parameter value length (in bytes). */
    );

/**
* \brief PCSC Transmission and Reception (Tx/Rx) Flag command.
*
* Application can send this to set the value of framing and RF parameters for
* the following transmission.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_TxRxFlag(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wValue                                /**< [In] Transmission and Reception (Tx/Rx) Flag value. */
    );

/**
* \brief PCSC Transceive command.
*
* Application can send this to exchange (transmit and receive) data with card.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_Transceive(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wOption,                              /**< [In] Exchange option. */
    uint8_t * pTxData,                             /**< [In] Data to transmit. */
    uint16_t wTxLength,                            /**< [In] Transmit data length. */
    uint8_t ** ppRxData,                           /**< [Out] Received data. */
    uint16_t * pRxLength                           /**< [Out] Received data length. */
    );

/**
* \brief PCSC Switch Protocol command.
*
* Application can send this to switch to a specific protocol layer.
*
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval #PH_ERR_USE_CONDITION \b Transparent session not on.
* \retval #PH_ERR_INVALID_PARAMETER \b Input parameter(s) invalid.
* \retval #PH_ERR_INTERNAL_ERROR Reader returned error (error updated in "bErrorCode").
* \retval #PH_ERR_LENGTH_ERROR Invalid response length / TLV length.
* \retval #PH_ERR_PROTOCOL_ERROR Response format not as per PCSC specification.
*/
phStatus_t phhalHw_Pcsc_Cmd_SwitchProtocol(
    phhalHw_Pcsc_DataParams_t * pDataParams,       /**< [In] Pointer to this layer's parameter structure. */
    uint8_t bProtocol,                             /**< [In] Protocol to switch. */
    uint8_t bLayer                                 /**< [In] Protocol layer to switch. */
    );

/** @}
* end of phhalHw_Pcsc_Cmd group
*/

#endif /* NXPBUILD__PHHAL_HW_PCSC */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#endif  /* PHHALHW_PCSC_CMD_H */
