/*
 * Copyright 2022 - 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Hal DUT specific declaration commands.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7463 $
 * $Date: 2025-08-29 15:56:52 +0530 (Fri, 29 Aug 2025) $
 *
 * History:
 *   Created On 6 July, 2022
 */

#ifndef PHHALHW_DUT_CMD_H
#define PHHALHW_DUT_CMD_H

#include <ph_Status.h>

#ifdef NXPBUILD__PHHAL_HW_DUT
#include <phhalHw.h>

#define PHHALHW_TAG_EEPROM_USER_AREA                            0x0010U /**< Tag value to indiate EEPROM Read from User Area. */
#define PHHALHW_TAG_EEPROM_SECURE_LIB_AREA                      0x0011U /**< Tag value to indiate EEPROM Read from Secure Lib Area. */
#define PHHALHW_TAG_EEPROM_IC_CONFIG_AREA                       0x0012U /**< Tag value to indiate EEPROM Read from IC Config Area. */
#define PHHALHW_TAG_EEPROM_PROTOCOL_AREA                        0x0020U /**< Tag value to indiate EEPROM Read from RF Protocol Area. */

/** \defgroup phhalHw_DUT_Cmd Commands
 * \brief Hal DUT specific declaration for the following features
 * - Contactless. Refer \ref phhalHw_DUT_Cmd_RDOperations "Reader Operations" and \ref phhalHw_DUT_Cmd_HALOperations "HAL Operations"
 *   commands.
 * - \ref phhalHw_DUT_Cmd_ContactCard "Contact Card (SAM)"
 * - \ref phhalHw_DUT_Cmd_I2C "I2C"
 * - \ref phhalHw_DUT_Cmd_CryptoSym "Symmetric Crypto"
 * - \ref phhalHw_DUT_Cmd_CryptoRng "Random Number Generator"
 * - \ref phhalHw_DUT_Cmd_KeyStore "Hardware KeyStore"
 * @{
 */

/* Initialization Commands ------------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_Init Command_Initialization
 * \brief Reader Initialization commands.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_Init_CommandCodes Command_Codes
 * \brief Reader Initialization Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_INIT_CONTACTLESS                           0xB001U /**< Initialize the reader to perform Contactless operation. */
#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CC
#define PHHAL_HW_DUT_INIT_CONTACT_CARD                          0xB010U /**< Initialize the reader to perform Contact Card (SAM) operation
                                                                         *   via TDA interface.
                                                                         */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CC */
#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_I2C
#define PHHAL_HW_DUT_INIT_I2C                                   0x6001U /**< Initialize the reader to perform I2C operation. */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_I2C */
/**
 * end of phhalHw_DUT_Cmd_Init_CommandCodes group
 * @}
 */

/**
 * \brief Performs the initialization of different Components.
 *
 * To initialize the communication interface of the reader below information should be used for \b wCmd parameter.
 * - \ref PHHAL_HW_DUT_INIT_CONTACTLESS "Contactless via RF Interface"
 * - \ref PHHAL_HW_DUT_INIT_CONTACT_CARD "ContactCard via TDA Interface"
 * - \ref PHHAL_HW_DUT_INIT_I2C "I2C Interface"
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_InitReader(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wCmd                                                   /**< [IN] Command to initialize the Reader. Refer description for
                                                                         *        values and combination to use.
                                                                         */
    );
/**
 * end of phhalHw_DUT_Cmd_Init group
 * @}
 */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_VERSION
/* Version Commands -------------------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_Version Command_Version
 * \brief Version Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_Version_CommandCodes Command_Codes
 * \brief Version Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_GETVERSION                             0x3201U /**< Complete Version information of NFC Controller's firmware or Host
                                                                         *   Controller's firmware.
                                                                         */
#define PHHAL_HW_DUT_CMD_GETVERSION_NFC                         0x3234U /**< Complete Version information of NFC Controller IC firmware. */
#define PHHAL_HW_DUT_CMD_GETVERSION_DATE_TIME                   0x3240U /**< Get the date time of compilation the NFC Controller's firmware or
                                                                         *   Host Controller's firmware.
                                                                         */
#define PHHAL_HW_DUT_CMD_GETVERSION_READER_IC_TYPE              0x3250U /**< Get the FrontEnd (Reader IC) or NFC Controller IC for which the code
                                                                         *   is compiled.
                                                                         */
/**
 * end of phhalHw_DUT_Cmd_Version_CommandCodes group
 * @}
 */

/**
 * \brief The function gets the Version of the Firmware of DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pProductVer,                                          /**< [OUT] Product Version. */
        uint16_t * pMajor,                                              /**< [OUT] Major Version. */
        uint8_t * pMinor,                                               /**< [OUT] Minor Version. */
        uint16_t * pPatch_Dev,                                          /**< [OUT] Development/Patch Version. */
        uint8_t * pVersionString,                                       /**< [OUT] Complete version information in form of ASCII bytes.
                                                                         *         Should be minimum 32 bytes long.
                                                                         */
        uint16_t * pVersionLen                                          /**< [In] Length of bytes allocated for \b pVersionString buffer.
                                                                         *   [Out] Length of bytes available in \b pVersionString buffer.
                                                                         */
    );

/**
 * \brief The function gets the Version of the NFC Controller connected to the DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion_NFC(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pProductVer,                                          /**< [OUT] Product Version. */
        uint16_t * pMajor,                                              /**< [OUT] Major Version. */
        uint8_t * pMinor,                                               /**< [OUT] Minor Version. */
        uint16_t * pPatch_Dev,                                          /**< [OUT] Development/Patch Version. */
        uint8_t * pVersionString,                                       /**< [OUT] Complete version information in form of ASCII bytes.
                                                                         *         Should be minimum 32 bytes long.
                                                                         */
        uint16_t * pVersionLen                                          /**< [In] Length of bytes allocated for \b pVersionString buffer.
                                                                         *   [Out] Length of bytes available in \b pVersionString buffer.
                                                                         */
    );

/**
 * \brief The function gets the date and time at which the firmware code is compiled.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion_CompiledInfo(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pResponse,                                            /**< [OUT] Complete version information in form of ASCII bytes.
                                                                         *         Should be minimum 128 bytes long.
                                                                         */
        uint16_t * pRspLen                                              /**< [In] Length of bytes allocated for \b pResponse buffer.
                                                                         *   [Out] Length of bytes available in \b pResponse buffer.
                                                                         */
    );

/**
 * \brief The function gets the type of NFC Reader IC being used.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_GetVersion_ReaderIcType(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pResponse,                                            /**< [OUT] NFCController / FrontEnd type in form of ASCII.
                                                                         *         Should be minimum 16 bytes long.
                                                                         */
        uint16_t * pRspLen                                              /**< [In] Length of bytes allocated for \b pResponse buffer.
                                                                         *   [Out] Length of bytes available in \b pResponse buffer.
                                                                         */
    );
/**
 * end of phhalHw_DUT_Cmd_Version group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_VERSION */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_RDOPS
/* Reader Operation Commands ----------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_RDOperations Command_ReaderOps
 * \brief Reader Operation Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_RdOps_CommandCodes Command_Codes
 * \brief Reader Operation Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_RD_OPS_GET_READER_STATUS               0xA006U /**< Gets the Status of Reader that included Reader Firmware Version. */
#define PHHAL_HW_DUT_CMD_RD_OPS_READ_REGISTER                   0xA007U /**< Perform Register Read. */
#define PHHAL_HW_DUT_CMD_RD_OPS_WRITE_REGISTER                  0xA008U /**< Perform Register Write. */
#define PHHAL_HW_DUT_CMD_RD_OPS_FIELD_ON                        0xA009U /**< Switch on the RF-field. */
#define PHHAL_HW_DUT_CMD_RD_OPS_FIELD_OFF                       0xA00AU /**< Switch off the RF-field. */
#define PHHAL_HW_DUT_CMD_RD_OPS_FIELD_RESET                     0xA00BU /**< Perform RF-Reset. */
#define PHHAL_HW_DUT_CMD_RD_OPS_READ_EEPROM                     0xA00EU /**< Perform EEPROM Read. */
#define PHHAL_HW_DUT_CMD_RD_OPS_WRITE_EEPROM                    0xA00FU /**< Perform EEPROM Write. */
/**
 * end of phhalHw_DUT_Cmd_RdOps_CommandCodes group
 * @}
 */

/**
 * \brief Gets the Status and Firmware Version of the Reader.
 *  - A total of 9 bytes will be returned or lesser. Below is the description of bytes.
 *      - Byte[0]: ReaderIC.
 *          - 0x01: RC523
 *          - 0x02: SAM
 *          - 0x03: RC663
 *          - 0x04: PN7642
 *      - Byte[1]: Configuration
 *      - Byte[2]: PCSC Mode
 *      - Byte[3]: Pegoda Reader's Firmware Major Version
 *      - Byte[4]: Pegoda Reader's Firmware Minor Version
 *      - Byte[5]: Pegoda Reader's Firmware Build Version
 *      - Byte[6]: Unknown
 *      - Byte[7]: Unknown
 *      - Byte[8]: Unknown
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_GetReaderStatus(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t ** ppReaderStatus,                                      /**< [OUT] The Status and Firmware Version information returned by the reader. */
        uint16_t * pReaderStatusLen                                     /**< [OUT] Length of bytes available in \b ppReaderStatus buffer. */
    );

/**
 * \brief Reads the Register information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Read_Register(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint32_t dwRegAddr,                                             /**< [IN] The Register address from where the information should
                                                                         *        be read.
                                                                         */
        uint32_t * pRegData                                             /**< [OUT] The Register data for the provided Address. */
    );

/**
 * \brief Reads the Register information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Write_Register(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint32_t dwRegAddr,                                             /**< [IN] The Register address to which the information should be
                                                                         *        written.
                                                                         */
        uint32_t dwRegData                                              /**< [IN] The Register data for the provided Address. */
    );

/**
 * \brief Switches ON the RF Field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_FieldOn(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [IN] Pointer to this layer parameter structure. */
    );

/**
 * \brief Switches OFF the RF Field.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_FieldOff(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [IN] Pointer to this layer parameter structure. */
    );

/**
 * \brief Resets the RF Field
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_FieldReset(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [IN] Pointer to this layer parameter structure. */
    );

/**
 * \brief Reads the EEPROM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Read_EEPROM(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [IN] Options to be used for EEPROM Reading.
                                                                         *          - 0x0000: Only \b dwStartAddr and \b dwLength information
                                                                         *            will be exchagned to Reader.
                                                                         *          - \b dwStartAddr and \b dwLength along with the specified information
                                                                         *             to this paramter will be exchagned to Reader. These values are
                                                                         *             specific to Pegoda - 3 Reader.
                                                                         *              - 0x0010: Read information from User area
                                                                         *              - 0x0011: Read information from Secure Lib area
                                                                         *              - 0x0012: Read information from IC Configuration area
                                                                         *              - 0x0020: Read information from RF Configuration area
                                                                         *          - Values other that above specified ones.
                                                                         */
        uint32_t dwAddr_ProtIndex,                                      /**< [IN] The EEPROM address or Protocol Index from where the information
                                                                         *        should be read. One of the following,
                                                                         *          - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
                                                                         *            EEPROM address. This is supported by Pegoda - 3 Reader
                                                                         *            only.
                                                                         *          - If \b wOption = 0x0020, then 1 byte Protocol
                                                                         *            index of the RF Configuration. This is supported
                                                                         *            by Pegoda - 3 Reader only.
                                                                         *          - If \b wOption = 0x0000, then 4 byte EEPROM address.
                                                                         *            Reader other than Pegoda 3 reader.
                                                                         *
                                                                         */
        uint32_t dwNoOfBytesToRead,                                     /**< [IN] Number of bytes to be read from EEPROM starting from address specified.
                                                                         *        If Reader is Pegoda -3 and \b wOption = 0x0020, then zero should be
                                                                         *        provided.
                                                                         */
        uint8_t ** ppResponse,                                          /**< [OUT] EEPROM information from the mentioned start address (\b dwStartAddr)
                                                                         *         until the requested length. Make sure the buffer is allocated for the
                                                                         *         requested number of bytes (\b dwLength).
                                                                         */
        uint16_t * pRespLen                                             /**< [OUT] Length of bytes available in \b pResponse buffer */
    );

/**
 * \brief Writes the EEPROM information.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_RdOps_Write_EEPROM(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [IN] Options to be used for EEPROM Writing.
                                                                         *          - 0x0000: Only \b dwStartAddr and \b dwLength information
                                                                         *            will be exchagned to Reader.
                                                                         *          - \b dwStartAddr and \b dwLength along with the specified information
                                                                         *             will be exchagned to Reader. These values are specific to Pegoda - 3
                                                                         *             Reader.
                                                                         *              - 0x0010: Read information from User area
                                                                         *              - 0x0011: Read information from Secure Lib area
                                                                         *              - 0x0012: Read information from IC Configuration area
                                                                         *              - 0x0020: Read information from RF Configuration area
                                                                         */
        uint32_t dwStartAddr,                                           /**< [IN] The EEPROM address information to be written.
                                                                         *        One of the following,
                                                                         *          - If \b wOption = 0x0010, 0x0011 or 0x0012, then 2 byte
                                                                         *            EEPROM address. This is supported by Pegoda - 3 Reader
                                                                         *            only.
                                                                         *          - If \b wOption = 0x0020, zero should be provided.
                                                                         *          - If \b wOption = 0x0000, then 4 byte EEPROM address.
                                                                         *            Reader other than Pegoda 3 reader.
                                                                         *
                                                                         */
        uint8_t * pData,                                                /**< [IN] Information to be written to EERPOM.
                                                                         *        One of the following,
                                                                         *          - If \b wOption = 0x0000, 0x0010, 0x0011 or 0x0012, then
                                                                         *            EEPROM value should be provided.
                                                                         *          - If \b wOption = 0x0020, then 1 byte Protocol index followed
                                                                         *            1 byte Register address followed by 4 byte Register Value.
                                                                         *            This is supported by Pegoda - 3 Reader only. \n
                                                                         *            Protocol Index1 ( 1 Byte) ||
                                                                         *            Reg Addr1 ( 1 Byte) || Reg Value1 ( 4 Bytes) ||
                                                                         *            Reg Addr2 ( 1 Byte) || Reg Value2 ( 4 Bytes) ||
                                                                         *            ...
                                                                         *            Reg AddrN ( 1 Byte) || Reg ValueN ( 4 Bytes)
                                                                         *
                                                                         */
        uint16_t wDataLen                                               /**< [IN] Length of bytes available in \b pData buffer. */
    );

/**
 * end of phhalHw_DUT_Cmd_RDOperations group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_RDOPS */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_HAL
/* HAL Operation Commands -------------------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_HALOperations Command_HALOps
 * \brief HAL Operation Command of HAL DUT Command dispatcher frame.
 *
 * \note
 *      - These command codes will be used to communicate to a contactless device connected via RF only.
 *      - To perfom PICC / Tag exchanges via I2C use I2C sepcific commands.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_HALOps_CommandCodes Command_Codes
 * \brief HAL Operation Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_HAL_EXCHANGE                           0xB002U /**< Perform Data Exchange with PICC. */
#define PHHAL_HW_DUT_CMD_HAL_SET_CONFIG                         0xB003U /**< Set configuration parameter for the firmware. */
#define PHHAL_HW_DUT_CMD_HAL_GET_CONFIG                         0xB004U /**< Get configuration parameter of the firmware. */
#define PHHAL_HW_DUT_CMD_HAL_APPLY_PROT_SETTINGS                0xB005U /**< Perform Apply protocol settings. */
#define PHHAL_HW_DUT_CMD_HAL_WAIT                               0xB006U /**< Blocks until the time given by \c wTimeout elapsed. */
#define PHHAL_HW_DUT_CMD_HAL_MFC_AUTH                           0xB007U /**< Perform MIFARE(R) Classic contactless IC Authentication using a key. */
#define PHHAL_HW_DUT_CMD_HAL_EXECUTE                            0xB008U /**< Execute a RC command. */
#define PHHAL_HW_DUT_CMD_HAL_MFC_AUTH_KEY_NO                    0xB009U /**< Perform MIFARE(R) Classic Authentication using a key number. */
#define PHHAL_HW_DUT_CMD_HAL_TRANSMIT                           0xB00AU /**< Perform Data Transmit to PICC. */
#define PHHAL_HW_DUT_CMD_HAL_RECEIVE                            0xB00BU /**< Perform Data Receive from PICC. */
#define PHHAL_HW_DUT_CMD_HAL_I18000P3M3_INVENTORY               0xB00CU /**< Perform Inventory command for 18000p3m3. */
#define PHHAL_HW_DUT_CMD_HAL_I18000P3M3_RESUME_INVENTORY        0xB00DU /**< Perform Inventory command for 18000p3m3. */
/**
 * end of phhalHw_DUT_Cmd_HALOps_CommandCodes group
 * @}
 */

/**
 * \brief Perform Data Exchange with a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To send any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current
 *            exchange call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *        \n
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Exchange(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen,                                                /**< [In] Length of bytes available in \b pTxBuff buffer. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by PICC. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_SetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier. */
        uint16_t wValue                                                 /**< [In] Configuration Value. */
    );

/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER Configuration is not supported or invalid.
 * \retval #PH_ERR_INVALID_PARAMETER Parameter value is invalid.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_GetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier. */
        uint16_t * pValue                                               /**< [Out] Configuration Value. */
    );

/**
 * \brief Apply protocol settings.
 * To keep the current card type, use #PHHAL_HW_CARDTYPE_CURRENT for bCardType.\n
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bCardType is invalid or not supported.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_ApplyProtocolSettings(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bCardType                                               /**< [In] Type of card for which the HAL should be configured for.
                                                                         *        One for the below values.
                                                                         *          - \ref PHHAL_HW_CARDTYPE_CURRENT "Current Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_ISO14443A "ISO14443 Type-A Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_ISO14443B "ISO14443 Type-B Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_FELICA "FELICA Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_ISO15693 "ISO15693 Configuration"
                                                                         *          - \ref PHHAL_HW_CARDTYPE_I18000P3M3 "ISO18000 p3m3 Configuration"
                                                                         */
    );

/**
 * \brief Blocks until the time given by \b wTimeout elapsed.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bUnit is invalid.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Wait(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bUnit,                                                  /**< [In] Unit of given timeout value. Supported values are
                                                                         *        - \ref PHHAL_HW_TIME_MICROSECONDS "MicroSeconds"
                                                                         *        - \ref PHHAL_HW_TIME_MILLISECONDS "MilliSeconds"
                                                                         */
        uint16_t wTimeout                                               /**< [In] Timeout value based on \b bUnit information. */
    );

/**
 * \brief Perform MIFARE(R) Classic Authentication using a key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bKeyType is invalid.
 * \retval #PH_ERR_IO_TIMEOUT Error in Authentication.
 * \retval #PH_ERR_AUTH_ERROR Error in Authentication.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_MfcAuthenticate(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bBlockNo,                                               /**< [In] Block Number on card to Authenticate to. */
        uint8_t bKeyType,                                               /**< [In] Key Type to use for Authentication.
                                                                         *          - \ref PHHAL_HW_MFC_KEYA "Key-Type A"
                                                                         *          - \ref PHHAL_HW_MFC_KEYB "Key-Type B"
                                                                         */
        uint8_t * pKey,                                                 /**< [In] Key to be used in authentication. */
        uint8_t * pUid                                                  /**< [In] 4 byte Serial number of current cascade level */
    );

/**
 * \brief Perform Command Execution with a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To send any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current
 *            exchange call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *        \n
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Execute(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bCmd,                                                   /**< [IN] Command to be executed on the Reader IC. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen,                                                /**< [In] Length of bytes available in \b pTxBuff buffer. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by PICC. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief Perform MIFARE(R) Classic Authentication using a key number.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_INVALID_PARAMETER \b bKeyType, \b wKeyNo or \b wKeyVersion is invalid.
 * \retval #PH_ERR_IO_TIMEOUT Error in Authentication.
 * \retval #PH_ERR_AUTH_ERROR Error in Authentication.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_MfcAuthenticateKeyNo(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t bBlockNo,                                               /**< [In] Block Number on card to Authenticate to. */
        uint8_t bKeyType,                                               /**< [In] Key Type to use for Authentication.
                                                                         *          - \ref PHHAL_HW_MFC_KEYA "Key-Type A"
                                                                         *          - \ref PHHAL_HW_MFC_KEYB "Key-Type B"
                                                                         */
        uint16_t wKeyNo,                                                /**< [In] Key number to be used in authentication. */
        uint16_t wKeyVer,                                               /**< [In] Key version to be used in authentication. */
        uint8_t * pUid                                                  /**< [In] 4 byte Serial number of current cascade level */
    );

/**
 * \brief Perform Transmission of data with a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To send any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current
 *            exchange call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Transmit(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen                                                 /**< [In] Length of bytes available in \b pTxBuff buffer. */
    );

/**
 * \brief Perform Reception of Data from a PICC. This command is used to exchange information to PICC and at the same
 *  time receives the payload from PICC if available including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to PICC and also receives the payload from PICC
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of PICC. Here the information is not
 *        exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for PICC (if any). Here the
 *        information is not exchanged with PICC.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for PICC (if any). Here the complete
 *        buffered information is exchanged to PICC and also receives the information from PICC if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_Receive(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by PICC. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief ISO 18000p3m3 Inventory command.
 *
 *  Note: This function does
 *  ISO18000p3m3 BeginRound
 *  Sends an ACK
 *  Optionally sends a ReqRN
 *  Stores the information related to tag reply in the HAL RX Buffer
 *  Continues with NextSlot or else pauses and returns based on the option specified
 *  Can return multi card information based on the configuration option and the size of HAL RX Buffer.
 *  The bTSprocessing (Time slot processing behavior) can take the following values
 *      - #PHHAL_HW_I18000P3M3_GET_MAX_RESPS - Gets responses from more than one slot limited by the Rx Buffer size and number of slots specified in begin round.
 *      - #PHHAL_HW_I18000P3M3_ONE_TS_ONLY - Gets response for only one time slot. Here the number of time slots should be zero in the begin round command.
 *      - #PHHAL_HW_I18000P3M3_GET_TAG_HANDLE - Gets response for only one time slot. Also send a ReqRN to get the tag handle for this slot. Here the number of time
 *        slots should be zero in the begin round command.
 *
 *  The response is a Pointer to a buffer containing the result of the inventory.
 *  The buffer may contain the result of one or more timeslots. Each timeslot result has the following structure:
 *  1 Byte time slot status
 *  (
 *      0 ... Tag response available. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field' are present.
 *      1 ... Tag response available. In addition to three field above, 'Tag handle field' also present.
 *      2 ... No tag replied in timeslot. 'Tag reply length field', 'Valid bits in last byte field' and 'Tag reply field', 'Tag Handle' are not present.
 *      3 ... Two or more tags responded in the timeslot. (Collision). Tag Reply Length, valid bits in last byte and tag reply field not present.
 *  )
 *  1 Byte 'Tag reply' Length (1-66)
 *  1 Byte Valid bits in last Byte (0-7, 0 means all bits are valid)
 *  0-66 Bytes Tag reply
 *  0 or 2 Bytes Handle of the tag, in case the field
 *
 * A typical sequence in which this API is called is given below
 *  - \ref phbalReg_SerialWin_Init "BAL initialization"
 *  - \ref phhalHw_DUT_Init "HAL initialization"
 *  - \ref phhalHw_ApplyProtocolSettings "Load ISO 1800p3m3 protocol" with Card Type as #PHHAL_HW_CARDTYPE_I18000P3M3
 *  - \ref phhalHw_FieldOn "Turn ON the field"
 *  - \ref phhalHw_Wait "Wait for initial 5000us guard time"
 *  - \ref phhalHw_I18000p3m3Inventory "Start inventory"
 *  - Use \ref phpalI18000p3m3 and \ref phalI18000p3m3 APIs to further communicate with the tag.
 *  - \ref phhalHw_FieldOff "Turn OFF the field"
 *  - Termination of the application flow.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval #PH_ERR_IO_TIMEOUT When there is no tag in the field.
 * \retval #PH_ERR_INVALID_PARAMETER If wrong value specified for bNumValidBits or pBeginRound.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_I18000p3m3Inventory(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t * pSelCmd,                                              /**< [In] ISO 18000p3m3 Select command frame. */
        uint8_t bSelCmdLen,                                             /**< [In] Select command length in bytes- 1 to 39 bytes. */
        uint8_t bNumValidBitsinLastByte,                                /**< [In] Number of valid bits in last byte of \b pSelCmd. */
        uint8_t * pBeginRndCmd,                                         /**< [In] ISO 18000p3m3 BeginRound command frame. This is 17bits
                                                                         *        i.e., 3 bytes are expected. CRC5 should not be provided.
                                                                         */
        uint8_t bTSprocessing,                                          /**< [In] TimeSlot processing behavior. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by ISO18000 tag. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief ISO 18000p3m3 resume inventory command.
 *
 * Note: This function is to be called after \ref phhalHw_I18000p3m3Inventory "I18000p3m3 Inventory".
 * This internally calls the ISO 18000p3m3 NextSlot command once or multiple times to get responses from
 * remaining slots.
 *
 * Resume inventory also returns the response similar to \ref phhalHw_I18000p3m3Inventory "I18000p3m3 Inventory".
 *
 * A typical sequence in which this API is called is given below
 *  - \ref phbalReg_SerialWin_Init "BAL initialization"
 *  - \ref phhalHw_DUT_Init "HAL initialization"
 *  - \ref phhalHw_ApplyProtocolSettings "Load ISO 1800p3m3 protocol" with Card Type as #PHHAL_HW_CARDTYPE_I18000P3M3
 *  - \ref phhalHw_FieldOn "Turn ON the field"
 *  - \ref phhalHw_Wait "Wait for initial 5000us guard time"
 *  - \ref phhalHw_I18000p3m3Inventory "Start inventory"
 *  - Store the responses received for the tags.
 *  - If inventory has to be continued then issue the \ref phhalHw_I18000p3m3ResumeInventory "ResumeInventory" command
 *  - Use \ref phpalI18000p3m3 and \ref phalI18000p3m3 APIs to further communicate with the tag.
 *  - \ref phhalHw_FieldOff "Turn OFF the field"
 *  - Termination of the application flow.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval #PH_ERR_IO_TIMEOUT When there is no tag in the field.
 * \retval #PH_ERR_USE_CONDITION If called when the phhalHw_I18000p3m3 was not called previously with bTSprocessing = PHHAL_HW_I18000P3M3_GET_MAX_RESPS.
 */
phStatus_t phhalHw_DUT_Cmd_HAL_I18000p3m3ResumeInventory(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [IN] Pointer to this layer parameter structure. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided ISO18000 tag. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * end of phhalHw_DUT_Cmd_HALOperations group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_HAL */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CC
/* Contact Card Operation Commands ----------------------------------------------------------------------------------------------------- */
/** \defgroup phhalHw_DUT_Cmd_ContactCard Command_ContactCard
 * \brief Contact Card Operation Command of HAL DUT Command dispatcher frame.
 * \note
 *      - These commands will work only if Hardware supports Contact Card features.
 *      - These command codes will be used to communicate to a Contact device connected via TDA only.
 *      - When any of the Contact Card commands are used, Direct Bal should be used and not Bal SAM.
 *      - To communicate via I2C use I2C specific commands.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_ContactCard_CommandCodes Command_Codes
 * \brief Contact Card Operation Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CC_ACTIVATE                            0x8001U /**< Activates a contact card and returns ATR. */
#define PHHAL_HW_DUT_CMD_CC_COLD_RESET                          0x8002U /**< Performs cold reset in a contact card. */
#define PHHAL_HW_DUT_CMD_CC_WARM_RESET                          0x8003U /**< Performs warm reset in a contact card. */
#define PHHAL_HW_DUT_CMD_CC_STOP_CLOCK                          0x8004U /**< Stops the clock signal to the contact card. */
#define PHHAL_HW_DUT_CMD_CC_START_CLOCK                         0x8005U /**< Starts the clock signal to the contact card. */
#define PHHAL_HW_DUT_CMD_CC_DEACTIVATE                          0x8006U /**< Starts the clock signal to the contact card. */
#define PHHAL_HW_DUT_CMD_CC_PRESENCE_CHECK                      0x8007U /**< Check if the card is inserted. If inserted returns 0x01otherwise 0x00. */
#define PHHAL_HW_DUT_CMD_CC_TRANSMIT_DATA                       0x8008U /**< Transport data to contact card. */
#define PHHAL_HW_DUT_CMD_CC_SEND_PPS                            0x8009U /**< Switches the hardware to a specific baud rate and/or performs a PPS command
                                                                         *   for the contact card.
                                                                         */
/**
 * end of phhalHw_DUT_Cmd_ContactCard_CommandCodes group
 * @}
 */

/**
 * \brief The function performs contact card activation procedure according to ISO7816-3 and
 * when successful outputs the returned ATR.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_Activate(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint8_t * pAtr,                                                 /**< [Out] Buffer holding the returned ATR string from the contact card.
                                                                         *         Make sure the buffer is allocated.
                                                                         */
        uint16_t * pAtrLen                                              /**< [Out] Length of bytes available in \b pAtr buffer. */
    );

/**
 * \brief The function performs contact card cold reset.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_ColdReset(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint8_t * pAtr,                                                 /**< [Out] Buffer holding the returned ATR string from the contact card.
                                                                         *         Make sure the buffer is allocated.
                                                                         */
        uint16_t * pAtrLen                                              /**< [Out] Length of bytes available in \b pAtr buffer. */
    );

/**
 * \brief The function performs contact card warm reset.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_WarmReset(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint8_t * pAtr,                                                 /**< [Out] Buffer holding the returned ATR string from the contact card.
                                                                         *         Make sure the buffer is allocated.
                                                                         */
        uint16_t * pAtrLen                                              /**< [Out] Length of bytes available in \b pAtr buffer. */
    );

/**
 * \brief The function stops the clock of the contact card interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_StopClock(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );

/**
 * \brief The function starts the clock pulse of the contact card interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_StartClock(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );

/**
 * \brief The function deactivates the card connected to the contact interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_Deactivate(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );
/**
 * \brief The function checks if a card is present at the contact interface.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_PresenceCheck(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint8_t * pCardInserted                                         /**< [Out] Variable indicating if a card is present. One of the following
                                                                         *         will be returned.
                                                                         *         - \ref PH_OFF "Card Not Present"
                                                                         *         - \ref PH_ON "Card Present"
                                                                         */
    );

/**
 * \brief The function transmits data with the indicated length to a contact card. The response data and its length is
 * stored at the pointed addresses.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_TransmitData(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint8_t * pTxBuff,                                              /**< [In] Buffer containing the data to be transmitted to Contact Card. */
        uint16_t wTxLen,                                                /**< [In] Length of bytes available in \b pTxBuff. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Buffer containing the received data from the Contact Card. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief The function performs protocol and parameter exchange(PPS) with a contact card according to ISO7816-3.
 * Supported are the following values:
 * - 0x12,  etu=186
 * - 0x13,  etu=93
 * - 0x18,  etu=31  speed 145.161 bit/s
 * - 0x95,  etu=32
 * - 0x96,  etu=16 - speed 281.250 bit/s
 *
 * \note
 *      The PPSS identifier byte, having value 0xff is not included in the <b>pPPS</b> buffer. The first byte in the pPPS buffer,
 *      i.e. <b>pPPS[0]</b> is PPS0. In case of set exchange speed <b>bOption = 0x01</b>, the first byte in the pPPS buffer,
 *      i.e. <b>pPPS[0]</b> indicates the transmission factors Fi[b7-b4], Di[b3-b0].
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_CC_SendPPS(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint8_t bOption,                                                /**< [in] 0x00=Perform PPS, 0x01=Set only the exchange speed. */
        uint8_t * pPPS,                                                 /**< [In] Buffer holding valid PPS exchange data. The presence of the optional bytes,
                                                                         *        respectively the length the PPS message is indicated by pPPS[0].
                                                                         */
        uint8_t * pRxBuff,                                              /**< [Out] Buffer where the received data from the Contact Card shall be stored. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b pRxBuff. */
    );

/**
 * end of phhalHw_DUT_Cmd_ContactCard group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CC */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_I2C
/* I2C Commands ------------------------------------------------------------------------------------------------------------------------ */
/** \defgroup phhalHw_DUT_Cmd_I2C Command_I2C
 * \brief I2C Peripheral Command of HAL DUT Command dispatcher frame.
 *
 * \note
 *      - These commands are for exchanging the information via I2C peripheral of the hardware.
 *      - By using these commands the Interface state of PICC / Tag or ContactCard will not be altered.
 *      - These commands are specifically for the I2C device that is connected to I2C master of the Reader.
 *      - Can be used to communicate to SAM connected using I2C interface. Here user needs to provide SAM commands framed in I2C protocol
 *        header as specified in Global Platform specification for I2C.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_I2C_CommandCodes Command_Codes
 * \brief I2C Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_I2C_EXCHANGE                           0x6002U /**< Performs I2C Read and Write operation. */
#define PHHAL_HW_DUT_CMD_I2C_TRANSMIT                           0x6003U /**< Performs I2C Write operation. */
#define PHHAL_HW_DUT_CMD_I2C_RECEIVE                            0x6004U /**< Performs I2C Read operation. */
#define PHHAL_HW_DUT_CMD_I2C_START                              0x6005U /**< Performs I2C Start Sequence. */
#define PHHAL_HW_DUT_CMD_I2C_STOP                               0x6006U /**< Performs I2C Stop Sequence. */
#define PHHAL_HW_DUT_CMD_I2C_REPEAT_START                       0x6007U /**< Performs I2C Repeated Start Sequence. */
/**
 * end of phhalHw_DUT_Cmd_I2C_CommandCodes group
 * @}
 */

/**
 * \brief Perform Data Exchange with I2C slave device. This command is used to exchange information to I2C Slave device and at the same
 *  time receives the payload from I2C slave device if available, including the Status.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to I2C Slave device and also receives the payload from I2C Slave device
 *        if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of I2C Slave device. Here the information is not
 *        exchanged with I2C Slave device.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for I2C Slave device (if any). Here the
 *        information is not exchanged with I2C Slave device.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for I2C Slave device (if any). Here the complete
 *        buffered information is exchanged to I2C Slave device and also receives the information from I2C Slave device if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To send any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current
 *            exchange call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *        \n
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *        \n
 *    \n
 *  - \b wBytesToRead
 *      - To receive a known amount of bytes use \ref PH_EXCHANGE_BYTES_TO_READ "Bytes To Read Information" available flag.
 *      - Leave \ref PH_EXCHANGE_BYTES_TO_READ "Bytes To Read Information" if the amount of bytes to read is not known.
 *      - \ref PH_EXCHANGE_BYTES_TO_READ "Bytes To Read Information" flag can be ORed with one of the options mentioned in <b>Normal Exchange</b>
 *        and <b>Chained Exchange</b>
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_I2C_Exchange(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint16_t wSlaveAddr,                                            /**< [In] Slave address of the Device to perform the Exchange.
                                                                         *          - 7Bit Addressing: Will have one byte address with LSB bit
                                                                         *            representing the Read or Write operation. \n
                                                                         *            Ex: WriteAddress: 0x92, ReadAddress: 0x93
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen,                                                /**< [In] Length of bytes available in \b pTxBuff. */
        uint16_t wBytesToRead,                                          /**< [In] Number of Bytes to read from I2C Device. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by I2C Slave device. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief Perform Transmission of data to I2C slave device. This command is used to exchange information to I2C Slave device.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to I2C Slave device and also receives the payload from I2C Slave device if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of I2C Slave device. Here the information is not exchanged with
 *        I2C Slave device.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for I2C Slave device (if any). Here the information is not
 *        exchanged with I2C Slave device.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for I2C Slave device (if any). Here the complete buffered information
 *        is exchanged to I2C Slave device and also receives the information from I2C Slave device if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_TXCHAINING "Chained Transmission"
 *          - To transmit any pending payload to DUT.
 *          - If DUT requires more data, then DUT will respond with \ref PH_ERR_SUCCESS_CHAINING "Require More Data" as status for the current exchange
 *            call. The interface should be called for this flag until DUT returns \ref PH_ERR_SUCCESS "Complete" as status.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_BUFFER_OVERFLOW Transmit data length is longer than the allocated transmit buffer size.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_I2C_Transmit(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint16_t wSlaveAddr,                                            /**< [In] Slave address of the Device to perform the Exchange.
                                                                         *          - 7Bit Addressing: Will have one byte address with LSB bit
                                                                         *            representing the Read or Write operation. \n
                                                                         *            Ex: WriteAddress: 0x92, ReadAddress: 0x93
                                                                         */
        uint8_t * pTxBuff,                                              /**< [In] Payload information to be exchanged / buffers based on
                                                                         *        \b wOption information.
                                                                         */
        uint16_t wTxLen                                                 /**< [In] Length of bytes available in \b pTxBuff. */
    );

/**
 * \brief Perform Reception of Data from I2C Slave device. This command is used to receive information from I2C Slave device.
 *
 * \b wOption can be one of:
 *  - Normal Exchange
 *      - \ref PH_EXCHANGE_DEFAULT "Default"
 *        To frame and exchange the complete payload information to I2C Slave device and also receives the payload from I2C Slave device if available.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_FIRST "Buffer Initial Payload"
 *        To buffer initial payload information. This payload can have the Command header of I2C Slave device. Here the information is not exchanged with
 *        I2C Slave device.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_CONT  "Buffer Intermediate Payload"
 *        To buffer intermediate payload information. This payload contains only the Data part for I2C Slave device (if any). Here the information is not
 *        exchanged with I2C Slave device.
 *        \n
 *      - \ref PH_EXCHANGE_BUFFER_LAST  "Buffer Final and Exchange"
 *        To buffer final payload information. This payload contains only the Data part for I2C Slave device (if any). Here the complete buffered information
 *        is exchanged to I2C Slave device and also receives the information from I2C Slave device if available.
 *        \n
 *    \n
 *  - Chained Exchange
 *      - \ref PH_EXCHANGE_RXCHAINING "Chained Reception"
 *          - To receive any pending payload from DUT.
 *          - To start using this flag, DUT will return \ref PH_ERR_SUCCESS_CHAINING "More Data Available" as status during previous reception.
 *            Payload should be received until \ref PH_ERR_SUCCESS "Complete" as status is returned by DUT.
 *        \n
 *    \n
 *  - \b wBytesToRead
 *      - To receive a known amount of bytes use \ref PH_EXCHANGE_BYTES_TO_READ "Bytes To Read Information" available flag.
 *      - Leave \ref PH_EXCHANGE_BYTES_TO_READ "Bytes To Read Information" if the amount of bytes to read is not known.
 *      - \ref PH_EXCHANGE_BYTES_TO_READ "Bytes To Read Information" flag can be ORed with one of the options mentioned in <b>Normal Exchange</b>
 *        and <b>Chained Exchange</b>
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_I2C_Receive(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wOption,                                               /**< [In] Buffering options to perform exchange. Refer description
                                                                         *        for more details.
                                                                         */
        uint16_t wSlaveAddr,                                            /**< [In] Slave address of the Device to perform the Exchange.
                                                                         *          - 7Bit Addressing: Will have one byte address with LSB bit
                                                                         *            representing the Read or Write operation. \n
                                                                         *            Ex: WriteAddress: 0x92, ReadAddress: 0x93
                                                                         */
        uint16_t wBytesToRead,                                          /**< [In] Number of Bytes to read from I2C Device. */
        uint8_t ** ppRxBuff,                                            /**< [Out] Information provided by I2C Slave device. */
        uint16_t * pRxLen                                               /**< [Out] Length of bytes available in \b ppRxBuff. */
    );

/**
 * \brief Perform I2C Start Sequence.
 *
 * \note
 *      - Post this interface call any of the below interfaces to perform exchanges.
 *          - \ref phhalHw_DUT_Cmd_I2C_Exchange "Transmit and Receive"
 *          - \ref phhalHw_DUT_Cmd_I2C_Transmit "Transmit Only"
 *          - \ref phhalHw_DUT_Cmd_I2C_Receive "Receive Only"
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_I2C_Start(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );

/**
 * \brief Perform I2C Stop Sequence.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_I2C_Stop(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );

/**
 * \brief Perform I2C Repeated Start Sequence.
 *
 * \note
 *      Post this interface call any of the below interfaces to perform exchanges.
 *          - \ref phhalHw_DUT_Cmd_I2C_Exchange "Transmit and Receive"
 *          - \ref phhalHw_DUT_Cmd_I2C_Transmit "Transmit Only"
 *          - \ref phhalHw_DUT_Cmd_I2C_Receive "Receive Only"
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval #PH_ERR_INTERFACE_ERROR Communication error.
 * \retval #PH_ERR_PROTOCOL_ERROR Invalid frame or unexpected receive data length or command code mismatch.
 * \retval Other Status word returned from the reader after the exchange function.
 */
phStatus_t phhalHw_DUT_Cmd_I2C_RepeatedStart(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );

/**
 * end of phhalHw_DUT_Cmd_I2C group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_I2C */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTOSYM
/* CryptoSym Commands ------------------------------------------------------------------------------------------------------------------ */
/** \defgroup phhalHw_DUT_Cmd_CryptoSym Command_CryptoSym
 * \brief Symmetric Crypto Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_CommandCodes Command_Codes
 * \brief Symmetric Crypto Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_INIT                         0x7101U /**< Initialize DUT CryptoSym stack. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DEINIT                       0x7102U /**< Un-Initialize DUT CryptoSym stack. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_GET_STATUS                   0x7103U /**< Get the Status. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_INVALIDATE_KEY               0x7104U /**< Remove the saved key and other information. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_SET_CONFIG                   0x7105U /**< Configure Symmetric Crypto stack. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_GET_CONFIG                   0x7106U /**< Get the Symmetric Crypto stack configuration. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_ENCRYPT                      0x7107U /**< Perform Encryption. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DECRYPT                      0x7108U /**< Perform Decryption. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CALCULATE_MAC                0x7109U /**< Perform MAC Calculation. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_IV                      0x710AU /**< Perform Initialization Vector loading. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_KEY                     0x710BU /**< Perform loading of Key from KeyStore. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_LOAD_KEY_DIRECT              0x710CU /**< Perform loading of direct Key. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_KEY                0x710DU /**< Perform Diversification of Key from KeyStore. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIVERSIFY_DIRECT_KEY         0x710EU /**< Perform Diversification of Direct Key. */
/**
 * end of phhalHw_DUT_Cmd_CryptoSym_CommandCodes group
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines Defines
 * \brief Symmetric Crypto Definitions of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_KeyTypes Defines
 * \brief Supported Key Types to be used in key loading functionality.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_INVALID             0xFFFFU /**< Invalid Key */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES128              0x0000U /**< AES 128 Key [16 Bytes]. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES192              0x0001U /**< AES 192 Key [24 Bytes]. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES256              0x0002U /**< AES 256 Key [32 Bytes]. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_DES                 0x0003U /**< DES Single Key [8 Bytes]. This is basically the 56-Bit DES key. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_2K3DES              0x0004U /**< 2 Key Triple Des [16 Bytes]. This is basically the 112-Bit DES key. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_3K3DES              0x0005U /**< 3 Key Triple Des [24 Bytes]. This is basically the 168-Bit DES key. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_MIFARE              0x0006U /**< MIFARE (R) Key. */
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_KeyTypes
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize KeySize
 * \brief Supported KeySizes for AES and DES algorithms.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize_DES DES
 * \brief Supported KeySizes DES algorithms.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DES_BLOCK_SIZE                   8U  /**< Block size in DES algorithm */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DES_KEY_SIZE                     8U  /**< Key size in DES algorithm for 56 bit key*/
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_2K3DES_KEY_SIZE                  16U /**< Key size in AES algorithm for 112 bit key*/
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_3K3DES_KEY_SIZE                  24U /**< Key size in AES algorithm for 168 bit key*/
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize_DES
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize_AES AES
 * \brief Supported KeySizes AES algorithms.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_AES_BLOCK_SIZE                   16U /**< Block size in AES algorithm */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_AES128_KEY_SIZE                  16U /**< Key size in AES algorithm for 128 bit key*/
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_AES192_KEY_SIZE                  24U /**< Key size in AES algorithm for 192 bit key*/
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_AES256_KEY_SIZE                  32U /**< Key size in AES algorithm for 256 bit key*/
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize_AES
 * @}
 */

/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_CipherModes Cipher Modes
 * \brief Supported Cipher Modes
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_ECB              0x00U   /**< ECB Cipher Mode. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_CBC              0x01U   /**< CBC Cipher Mode. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_CBC_DF4          0x02U   /**< CBC Cipher Mode for D40 Secure Messaging. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_LRP              0x03U   /**< LRP Cipher Mode. */
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_CipherModes
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_MacModes MAC Modes
 * \brief Supported Mac Modes.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_MAC_MODE_CMAC                0x00U   /**< CMAC MAC Mode. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_MAC_MODE_CBCMAC              0x01U   /**< CBCMAC MAC Mode. */
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_MacModes
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_DivTypes Diversification Types
 * \brief Supported Diversification Types.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_MASK                0x00FFU /**< BitMask for diversification Mode. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_DESFIRE             0x0000U /**< DESFire Key Diversification. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_MIFARE_PLUS         0x0001U /**< MIFARE Plus Key Diversification. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_MIFARE_ULTRALIGHT   0x0002U /**< MIFARE Ultralight Key Diversification. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_OPTION_2K3DES_FULL       0x0000U /**< Option for 2K3DES full-key diversification
                                                                         * (only with \ref PH_CRYPTOSYM_DIV_MODE_DESFIRE "DESFire Key Diversification").
                                                                         */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_OPTION_2K3DES_HALF       0x8000U /**< Option for 2K3DES half-key diversification
                                                                         * (only with \ref PH_CRYPTOSYM_DIV_MODE_DESFIRE "DESFire Key Diversification").
                                                                         */
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_DivTypes
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_ConfigTypes Config Types
 * \brief CryptoSym Layer Configuration types.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_KEY_TYPE              0x0000U /**< Key Type. Read-only. Possible Values are:
                                                                         *      - #PH_CRYPTOSYM_KEY_TYPE_INVALID
                                                                         *      - #PH_CRYPTOSYM_KEY_TYPE_AES128
                                                                         *      - #PH_CRYPTOSYM_KEY_TYPE_AES192
                                                                         *      - #PH_CRYPTOSYM_KEY_TYPE_AES256
                                                                         *      - #PH_CRYPTOSYM_KEY_TYPE_DES
                                                                         *      - #PH_CRYPTOSYM_KEY_TYPE_2K3DES
                                                                         *      - #PH_CRYPTOSYM_KEY_TYPE_3K3DES
                                                                         */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_KEY_SIZE              0x0001U /**< Key Size of currently loaded key. Read-only.  */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_BLOCK_SIZE            0x0002U /**< Block Size of currently loaded key. Read-only. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_KEEP_IV               0x0003U /**< Keep init vector. Either #PH_CRYPTOSYM_VALUE_KEEP_IV_OFF or #PH_CRYPTOSYM_VALUE_KEEP_IV_ON.
                                                                         *  This flag has to be used in combination with the option flag in the Encrypt/Decrypt/CalculateMac
                                                                         *  function: If either the option in the function or this flag is set, the IV will be updated before
                                                                         *  returning of the function. R/W access possible.
                                                                         */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_ADDITIONAL_INFO       0x0006U /**<  Additional information to be provided like diversified key length. */
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_ConfigTypes
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoSym_Defines_KeepIV Keep IV
 * \brief Supported IV Updated Behavior Modes.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_VALUE_KEEP_IV_OFF            0x0000U /**< Switch off Keep-IV behavior. */
#define PHHAL_HW_DUT_CMD_CRYPTOSYM_VALUE_KEEP_IV_ON             0x0001U /**< Switch on Keep-IV behavior. */
/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines_KeepIV
 * @}
 */

/**
 * end of group phhalHw_DUT_Cmd_CryptoSym_Defines
 * @}
 */

/**
 * \brief Initialize Hardware Symmetric Crypto Component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_Init(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Un-Initialize Hardware Symmetric Crypto Component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_DeInit(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Returns the status code and respective message. This interfaces is supported only if CryptoSym component is
 *  initialized with \ref phCryptoRng_mBedTLS_Init "mBedTLS Initialization".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_GetLastStatus(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wStatusMsgLen,                                         /**< [In] Size of bytes allocated for \b pStatusMsg parameter. */
        int8_t * pStatusMsg,                                            /**< [Out] The equivalent status message for the information available in \b pStatusCode. */
        int32_t * pStatusCode                                           /**< [Out] The status code returned by the underlying Crypto library. */
    );

/**
 * \brief Invalidate the currently loaded key.
 * Resets the key, the IV, the keep IV flag and the key Type.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_InvalidateKey(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_INVALID_PARAMETER         Valid wConfig but invalid wValue for that config.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER     Invalid (Unsupported) wConfig.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_SetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier.
                                                                         *        Refer \ref phhalHw_DUT_Cmd_CryptoSym_Defines_ConfigTypes
                                                                         *        "Configuration Identifier"
                                                                         */
        uint16_t wValue                                                 /**< [In] Configuration Value for the provided configuration identifier.
                                                                         *        Refer \ref phhalHw_DUT_Cmd_CryptoSym_Defines_KeepIV "KeepIV Options" for
                                                                         *        #PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_KEEP_IV configuration identifier.
                                                                         */
    );

/**
 * \brief Get configuration parameter.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_INVALID_PARAMETER         Value behind wConfig not valid at the moment.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER     Invalid (Unsupported) wConfig.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_GetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier.
                                                                         *        Refer \ref phhalHw_DUT_Cmd_CryptoSym_Defines_ConfigTypes
                                                                         *        "Configuration Identifier"
                                                                         */
        uint16_t * pValue                                               /**< [Out] Configuration Value for the provided configuration identifier.
                                                                         *          - Refer \ref phhalHw_DUT_Cmd_CryptoSym_Defines_KeyTypes "Keytypes" for
                                                                         *            #PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_KEY_TYPE configuration identifier.
                                                                         *          - Refer \ref phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize_DES "DES Key / Block Sizes" or
                                                                         *            \ref phhalHw_DUT_Cmd_CryptoSym_Defines_KeySize_AES "AES Key / Block Sizes" for
                                                                         *            #PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_KEY_SIZE or #PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_BLOCK_SIZE
                                                                         *            configuration identifier.
                                                                         *          - Refer \ref phhalHw_DUT_Cmd_CryptoSym_Defines_KeepIV "KeepIV Options" for
                                                                         *            #PHHAL_HW_DUT_CMD_CRYPTOSYM_CONFIG_KEEP_IV configuration identifier.
                                                                         */
    );

/**
 * \brief Perform Encryption with one of the supported crypto modes
 *
 * The option word specifies the operation mode to use and the update behavior of the IV.
 * All modes of operation are coded in the LSB, the flags in the MSB.
 * The following Cipher modes are supported:
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_ECB
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_CBC
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_CBC_DF4
 *
 * The following Flags are supported:
 * - #PH_EXCHANGE_DEFAULT
 * - #PH_EXCHANGE_BUFFER_FIRST
 * - #PH_EXCHANGE_BUFFER_CONT
 * - #PH_EXCHANGE_BUFFER_LAST
 *
 * Note:  The input data length needs to be a multiple of the current block size
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_INVALID_PARAMETER        An unsupported key is loaded (or no key is loaded) or \b wBufferLength is not
 *                                          a multiple of the current block size.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    An unknown cipher option wOption is specified.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_Encrypt(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                               /**< [In] Option byte specifying the cipher mode and the update behavior of the IV */
        const uint8_t * pPlainBuff,                                     /**< [In] Plain data buffer.
                                                                         *          - Should always be in multiple of current block size.
                                                                         *          - If not of current block size then Padding needs to be applied
                                                                         *            to make it upto current block size.
                                                                         */
        uint16_t wBuffLen,                                              /**< [In] Length of plain and encrypted data buffer - needs to be a multiple of the
                                                                         *        current block size
                                                                         */
        uint8_t * pEncBuff                                              /**< [Out] Encrypted data buffer. Will always be in multiple of current block size. */
    );

/**
 * \brief Perform Decryption with one of the supported crypto modes
 *
 * The option word specifies the operation mode to use and the update behavior of the IV.
 * All modes of operation are coded in the LSB, the flags in the MSB.
 * The following Cipher modes are supported:
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_ECB
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_CBC
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_CIPHER_MODE_CBC_DF4
 *
 * The following Flags are supported:
 * - #PH_EXCHANGE_DEFAULT
 * - #PH_EXCHANGE_BUFFER_FIRST
 * - #PH_EXCHANGE_BUFFER_CONT
 * - #PH_EXCHANGE_BUFFER_LAST
 *
 * Note:  The input data length needs to be a multiple of the current block size
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_INVALID_PARAMETER        An unsupported key is loaded (or no key is loaded) or \b wBufferLength is not
 *                                          a multiple of the current block size.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    An unknown cipher option wOption is specified.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_Decrypt(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                               /**< [In] Option byte specifying the cipher mode and the update behavior of the IV */
        const uint8_t * pEncBuff,                                       /**< [In] Encrypted data buffer. Should always be in multiple of current block size. */
        uint16_t wBuffLen,                                              /**< [In] Length of plain and encrypted data buffer - needs to be a multiple of the
                                                                         *        current block size.
                                                                         */
        uint8_t * pPlainBuff                                            /**< [Out] Plain data buffer. Will always be in multiple of current block size. Plain
                                                                         *         data may be padded with zeros if not current block size.
                                                                         */
    );

/**
 * \brief Calculate MAC with one of the supported MAC modes
 *
 * The option word specifies the MAC mode to use and the update behavior of the IV as well as the completion behavior.
 * All modes of operation are coded in the LSB, the flags in the MSB.
 * The following Cipher modes are supported:
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_MAC_MODE_CMAC
 * - #PHHAL_HW_DUT_CMD_CRYPTOSYM_MAC_MODE_CBCMAC
 *
 * The following Flags are supported:
 * - #PH_EXCHANGE_DEFAULT
 * - #PH_EXCHANGE_BUFFER_FIRST
 * - #PH_EXCHANGE_BUFFER_CONT
 * - #PH_EXCHANGE_BUFFER_LAST
 *
 * Note: If #PH_EXCHANGE_BUFFERED_BIT is set, the input length needs to be a multiple of the block length!
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_INVALID_PARAMETER        An unsupported key is loaded (or no key is loaded) or wDataLength is not
 *                                          a multiple of the current block size and the option #PH_EXCHANGE_BUFFERED_BIT is set.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    An unknown mac option wOption is specified.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_CalculateMac(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                               /**< [In] Option byte specifying the MAC mode and the update behavior of
                                                                         *        the IV and the completion flag.
                                                                         */
        const uint8_t* pData,                                           /**< [In] Input data on which the MAC needs to be computed.
                                                                         *        Input will be always be in multiple of current block size if wOption is
                                                                         *          - #PH_EXCHANGE_BUFFER_FIRST
                                                                         *          - #PH_EXCHANGE_BUFFER_CONT
                                                                         */
        uint16_t wDataLen,                                              /**< [In] number of input data bytes */
        uint8_t * pMac,                                                 /**< [Out] Output MAC block; uint8_t[16] */
        uint8_t * pMacLen                                               /**< [Out] Length of MAC */
    );

/**
 * \brief Load IV
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_INVALID_PARAMETER        \b bIVLength does not match the current block size.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_LoadIv(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        const uint8_t * pIV,                                            /**< [In] Initialization vector to use. Should of current block size. */
        uint8_t bIVLen                                                  /**< [In] Length of bytes available in \b pIV buffer. */
    );

/**
 * \brief Load Key
 *
 * This function uses the key storage provided at component initialization to retrieve the key identified by wKeyNo and wKeyVersion.
 * After retrieving the key is loaded into the internal key storage array to be prepared for subsequent cipher operations.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 *                                          - No KeyStore specified at initialization.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    Key Type not supported.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_LoadKey(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number in KeyStore to be loaded. */
        uint16_t wKeyVer,                                               /**< [In] Key Version in KeyStore to be loaded. */
        uint16_t wKeyType                                               /**< [In] Type of Key to be loaded. Supported ones are
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES128
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES192
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES256
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_DES
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_2K3DES
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_3K3DES
                                                                         */
    );

/**
 * \brief Direct Load Key
 *
 * The key provided in the pKey parameter is loaded into the internal key storage array to be prepared for subsequent cipher operations.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    Key Type not supported.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_LoadKeyDirect(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        const uint8_t * pKey,                                           /**< [In] Key to be loaded. Number of bytes should be based on the key
                                                                         *        type mentioned in \b wKeyType parameter.
                                                                         */
        uint16_t wKeyType                                               /**< [In] Type of Key to be loaded. Supported ones are
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES128
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES192
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES256
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_DES
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_2K3DES
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_3K3DES
                                                                         */
    );

/**
 * \brief Diversify Key - Note: This function invalidates the currently loaded key.
 *
 * Using the key stored in the KeyStore passed at initialization of the component and identified by wKeyNo and wKeyVersion
 * this function calculates a diversified key according to the wOption specified that can be used in different applications.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 *                                          - No KeyStore specified at Initialization.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    Key Type not supported (for key diversification).
 * \retval #PH_ERR_LENGTH_ERROR             Length of diversification input is wrong.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_DiversifyKey(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                               /**< [In] Option to specify the diversification method.
                                                                         *        One of the below mentioned information.
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_DESFIRE
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_MIFARE_PLUS
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_MIFARE_ULTRALIGHT
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_OPTION_2K3DES_FULL
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_OPTION_2K3DES_HALF
                                                                         */
        uint16_t wKeyNo,                                                /**< [In] Key number in KeyStore to be loaded */
        uint16_t wKeyVer,                                               /**< [In] Key Version in KeyStore to be loaded */
        uint8_t * pDivInput,                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInLen,                                              /**< [In] Length of diversification input used to diversify the key.
                                                                         *        If 0, no diversification is performed.
                                                                         */
        uint8_t * pDivKey,                                              /**< [Out] Diversified key. Will be of current block size. */
        uint8_t * pDivKeyLen                                            /**< [Out] Length of bytes available in \b pDivKey buffer. */
    );

/**
 * \brief Diversify Direct Key - Note: This function invalidates the currently loaded key.
 *
 * Using the key passed in the pKey parameter this function calculates a diversified key according to the wOption
 * specified that can be used in different applications.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not supported by this layer.
 * \retval #PH_ERR_UNSUPPORTED_PARAMETER    Key Type not supported (for key diversification).
 * \retval #PH_ERR_LENGTH_ERROR             Length of diversification input is wrong.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoSym_DiversifyDirectKey(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wOption,                                               /**< [In] Option to specify the diversification method.
                                                                         *        One of the below mentioned information.
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_DESFIRE
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_MIFARE_PLUS
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_MODE_MIFARE_ULTRALIGHT
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_OPTION_2K3DES_FULL
                                                                         *          - #PHHAL_HW_DUT_CMD_CRYPTOSYM_DIV_OPTION_2K3DES_HALF
                                                                         */
        uint8_t * pKey,                                                 /**< [In] Key to be loaded. Number of bytes should be based on the key
                                                                         *        type mentioned in \b wKeyType parameter.
                                                                         */
        uint16_t wKeyType,                                              /**< [In] Type of Key to be loaded. Supported ones are
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES128
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES192
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_AES256
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_DES
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_2K3DES
                                                                         *      - #PHHAL_HW_DUT_CMD_CRYPTOSYM_KEY_TYPE_3K3DES
                                                                         */
        uint8_t * pDivInput,                                            /**< [In] Diversification Input used to diversify the key. */
        uint8_t bDivInLen,                                              /**< [In] Length of diversification input used to diversify the key.
                                                                         *        If 0, no diversification is performed.
                                                                         */
        uint8_t * pDivKey,                                              /**< [Out] Diversified key. Will be of current block size. */
        uint8_t * pDivKeyLen                                            /**< [Out] Length of bytes available in \b pDivKey buffer. */
    );
/**
 * end of phhalHw_DUT_Cmd_CryptoSym group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTOSYM */










#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTORNG
/* CryptoRng Commands ------------------------------------------------------------------------------------------------------------------ */
/** \defgroup phhalHw_DUT_Cmd_CryptoRng Command_CryptoRng
 * \brief Random Number Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_CryptoRng_CommandCodes Command_Codes
 * \brief Random Number Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_CRYPTORNG_INIT                         0x7201U /**< Initialize DUT CryptoRng stack. */
#define PHHAL_HW_DUT_CMD_CRYPTORNG_DEINIT                       0x7202U /**< Un-Initialize DUT CryptoRng stack. */
#define PHHAL_HW_DUT_CMD_CRYPTORNG_GET_STATUS                   0x7203U /**< Get the Status. */
#define PHHAL_HW_DUT_CMD_CRYPTORNG_SEED                         0x7204U /**< Seed the Random number generator. */
#define PHHAL_HW_DUT_CMD_CRYPTORNG_RND                          0x7205U /**< Get the Random numbers. */
/**
 * end of phhalHw_DUT_Cmd_CryptoRng_CommandCodes group
 * @}
 */

/**
 * \brief Initialize Hardware random number generator.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoRng_Init(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Un-Initialize Hardware random number generator.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoRng_DeInit(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Returns the status code and respective message. This interfaces is supported only if CryptoRng component is
 *  initialized with \ref phCryptoRng_mBedTLS_Init "mBedTLS Initialization".
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoRng_GetLastStatus(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wStatusMsgLen,                                         /**< [In] Size of bytes allocated for \b pStatusMsg parameter. */
        int8_t * pStatusMsg,                                            /**< [Out] The equivalent status message for the information available in \b pStatusCode. */
        int32_t * pStatusCode                                           /**< [Out] The status code returned by the underlying Crypto library. */
    );

/**
 * \brief Seeds the random number generator with the given seed.
 *
 * The seed length is dependent on the underlaying implementation. Further details can be found in the detailed description
 * of the underlying layers.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoRng_Seed(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pSeed,                                                /**< [In] Seed */
        uint8_t bSeedLen                                                /**< [In] Size of the seed. */
    );

/**
 * \brief Obtains random bytes from the random source.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS
 *                                          - If the interface context (\b pDataparams) is not holding this layer ID.
 *                                          - If the component context holds a different sub-component ID that is not
 *                                            supported by this layer.
 * \retval #PH_ERR_INVALID_PARAMETER        - If the buffer is null.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_CryptoRng_Rnd(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t  wNoOfRndBytes,                                        /**< [In] Number of random bytes to generate. */
        uint8_t * pRnd                                                  /**< [Out] Generated bytes; uint8_t[wNoOfRndBytes] */
    );

/**
 * end of phhalHw_DUT_Cmd_CryptoRng group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_CRYPTORNG */





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE
/* KeyStore Commands ------------------------------------------------------------------------------------------------------------------ */
/** \defgroup phhalHw_DUT_Cmd_KeyStore Command_KeyStore
 * \brief KeyStore Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_KeyStore_CommandCodes Command_Codes
 * \brief KeyStore Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_KEYSTORE_INIT                          0x7001U /**< Initialize DUT KeyStore stack. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_DEINIT                        0x7002U /**< Un-Initialize DUT KeyStore stack. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_FORMAT_KEY_ENTRY              0x7003U /**< Format a key entry to a new KeyType. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_SET_KUC                       0x7004U /**< Change the KUC of a key entry. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_GET_KUC                       0x7005U /**< Obtain a key usage counter entry. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_CHANGE_KUC                    0x7006U /**< Change a key usage counter entry. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_SET_CONFIG                    0x7007U /**< Set configuration parameter. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_SET_CONFIG_BUFFER             0x7008U /**< Set configuration buffer. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_GET_CONFIG                    0x7009U /**< Get configuration parameter. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_GET_CONFIG_BUFFER             0x700AU /**< Get configuration buffer. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY                       0x7020U /**< Change a Symmetric key entry at a given version. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_AT_POS                0x7021U /**< Change a Symmetric key entry at the specified position. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_SET_FULL_KEY_ENTRY            0x7022U /**< Change a full Symmetric key entry. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY_ENTRY                 0x7023U /**< Get a Symmetric key entry information block. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY                       0x7024U /**< Get a Symmetric key. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_SET_KEY_ASYM                  0x7040U /**< Set a ASymmetric key. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_GET_KEY_ASYM                  0x7041U /**< Get a ASymmetric key. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_INIT                0x70A1U /**< Initialize DUT Key Provisioning stack. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_DEINIT              0x70A2U /**< Un-Initialize DUT Key Provisioning stack. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_SET_CONFIG          0x70A3U /**< Set provisioning Configuration. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_GET_CONFIG          0x70A4U /**< Get provisioning Configuration. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_APP_FIXED_KEYS      0x70A5U /**< Perform Provisioning of Symmetric Keys. */
/**
 * end of phhalHw_DUT_Cmd_KeyStore_CommandCodes group
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_KeyStore_Defines Defines
 * \brief KeyStore Definitions of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_Cmd_KeyStore_Defines_KeyTypes KeyTypes
 * \brief Supported Key Types to be used in key storing functionality.
 * @{
 */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_INVALID                  0xFFFFU /**< Invalid Key */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES128                   0x0000U /**< AES 128 Key [16 Bytes]. */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES192                   0x0001U /**< AES 192 Key [24 Bytes]. */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_AES256                   0x0002U /**< AES 256 Key [32 Bytes]. */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_DES                      0x0003U /**< DES Single Key [8 Bytes]. This is basically the 56-Bit DES key. */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_2K3DES                   0x0004U /**< 2 Key Triple Des [16 Bytes]. This is basically the 112-Bit DES key. */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_3K3DES                   0x0005U /**< 3 Key Triple Des [24 Bytes]. This is basically the 168-Bit DES key. */
#define PHHAL_HW_DUT_KEYSTORE_KEY_TYPE_MIFARE                   0x0006U /**< MIFARE (R) Key. */
/**
 * end of group phhalHw_DUT_Cmd_KeyStore_Defines_KeyTypes
 * @}
 */

/** \defgroup phhalHw_DUT_Cmd_KeyStore_Defines_KeySize KeySize
 * \brief Supported KeySizes for AES and DES algorithms.
 * @{
 */
#define PHHAL_HW_DUT_KEYSTORE_DES_KEY_SIZE                          8U  /**< Key size in DES algorithm for 56 bit key*/
#define PHHAL_HW_DUT_KEYSTORE_2K3DES_KEY_SIZE                       16U /**< Key size in AES algorithm for 112 bit key*/
#define PHHAL_HW_DUT_KEYSTORE_3K3DES_KEY_SIZE                       24U /**< Key size in AES algorithm for 168 bit key*/
#define PHHAL_HW_DUT_KEYSTORE_AES128_KEY_SIZE                       16U /**< Key size in AES algorithm for 128 bit key*/
#define PHHAL_HW_DUT_KEYSTORE_AES192_KEY_SIZE                       24U /**< Key size in AES algorithm for 192 bit key*/
#define PHHAL_HW_DUT_KEYSTORE_AES256_KEY_SIZE                       32U /**< Key size in AES algorithm for 256 bit key*/
/**
 * end of group phhalHw_DUT_Cmd_KeyStore_Defines_KeySize
 * @}
 */

/**
 * end of phhalHw_DUT_Cmd_KeyStore_Defines group
 * @}
 */

/**
 * \brief Initialize the KeyStore component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_Init(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wNoOfKeyEntries                                        /**< [In] Number of key entries for storing the keys.*/
    );

/**
 * \brief DeInitialize the KeyStore component.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_DeInit(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );

/**
 * \brief The function resets to 0 all key versions and their corresponding key values for the indicated key \b wKeyNo.
 * The key type, which is valid for all key versions of \b wKeyNo is initialized through this function to the
 * specified \b wNewKeyType.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_FormatKeyEntry(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the key, which has to be formated. */
        uint16_t wNewKeyType                                            /**< [In] Key type of the key (predefined type).*/
    );

/**
 * \brief The function changes the key usage counter(KUC) pertaining to a key number \b wKeyNo to the specified value
 * \b wRefNoKUC. The current KUC gets overwritten with the indicated value.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKUC(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the key, which has to be. */
        uint16_t wRefNoKUC                                              /**< [In] Key usage counter(KUC) value for the indicated key \b wKeyNo */
    );

/**
 * \brief The function retrieves the indicated key usage counter(KUC) and stores its limit and current values at the indicated
 * \b dwLimit and \b pdwCurVal respectively.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKUC(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wRefNoKUC,                                             /**< [In] Reference number of the key usage counter. */
        uint32_t * pLimit,                                              /**< [Out] Pointer to received limit of key usage counter. */
        uint32_t * pCurVal                                              /**< [Out] Pointer to received current key usage counter. */
    );

/**
 * \brief The function initializes the indicated key usage counter(KUC) with a limit value \b dwLimit.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_ChangeKUC(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wRefNoKUC,                                             /**< [In] Reference number of the key usage counter. */
        uint32_t dwLimit                                                /**< [In] Limit of the key usage counter(KUC). */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier */
        uint16_t wValue                                                 /**< [In] Configuration Value */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetConfigStr(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier */
        uint8_t *pBuffer,                                               /**< [In] Buffer containing the configuration string. */
        uint16_t wBuffLen                                               /**< [In] Length of configuration string */
    );

/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_GetConfig(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier */
        uint16_t * pValue                                               /**< [In] Configuration Value */
    );

/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_GetConfigStr(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier */
        uint8_t ** ppBuffer,                                            /**< [In] Pointer to the buffer containing the configuration string. */
        uint16_t * pBuffLen                                             /**< [In] Amount of valid bytes in the configuration string buffer. */
    );

/**
 * \brief The function initializes a new version of the indicated \b wKeyNo, having key type \b wKeyType.
 * Note that the key type is not set here, but this done by the \ref phhalHw_DUT_Cmd_KeyStore_FormatKeyEntry
 * "Format Key" function. All key versions of \b wKeyNo are searched for the indicated \b wKeyVer and when found
 * it gets overwritten together with its key data by the new version and data.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKey(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the key, which has to be initialized. */
        uint16_t wKeyVer,                                               /**< [In] Key version of the key, which has to be initialized. */
        uint16_t wKeyType,                                              /**< [In] Key type of the key which has to be initialized. */
        uint8_t* pNewKey,                                               /**< [In] Pointer to a buffer holding the new key data for the
                                                                         * indicated key number and version.
                                                                         */
        uint16_t wNewKeyVer                                             /**< [In] New key version for the indicated key number */
    );

/**
 * \brief The function initializes a new version of the indicated \b wKeyNo, having key type \b wKeyType.
 * Note that the key type is not set here, but this is done by the \b phhalHw_DUT_Cmd_KeyStore_FormatKeyEntry "Format KeyEntry"
 * function.
 * The key version at position \b wPos gets overwritten together with its key data by the new version and data.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKeyAtPos(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the key, which has to be initialized. */
        uint16_t wPos,                                                  /**< [In] Position of the key version for the indicated key */
        uint16_t wKeyType,                                              /**< [In] Key type of the key, which has to be initialized. */
        uint8_t * pNewKey,                                              /**< [In] Pointer to a buffer holding the new key data for the indicated
                                                                         * key number and version.
                                                                         */
        uint16_t wNewKeyVer                                             /**< [In] New key version for the indicated key number */
    );

/**
 * \brief The function initializes all key versions and the corresponding key data for the indicated key \b wKeyNo.
 * The key usage counter(KUC) and the key type are set equal to the specified values.
 *
 * \return Status code
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetFullKeyEntry(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wNoOfKeys,                                             /**< [In] Number of keys versions in \b pNewKeys buffer. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the key to be initialized. */
        uint16_t wNewRefNoKUC,                                          /**< [In] Key usage counter(KUC) value for the indicated key. */
        uint16_t wNewKeyType,                                           /**< [In] Key type to be initialized */
        uint8_t * pNewKeys,                                             /**< [In] Pointer to array of key entries to load. The order of the
                                                                         *        key-data entirely corresponds to the order of the version
                                                                         *        numbers in \b pNewKeyVersionList.
                                                                         */
        uint16_t * pNewKeyVerList                                       /**< [In] Pointer to key version list of the key to be loaded. */
    );

/**
 * \brief The function retrieves the key type and the available key versions of the indicated key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKeyEntry(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the key, whose version data is retrieved. */
        uint16_t wKeyVerBufSize,                                        /**< [In] Size of the key versions buffer \b wKeyVersion. */
        uint16_t * wKeyVersion,                                         /**< [Out] Pointer to a buffer where the key versions shall be stored. */
        uint16_t * wKeyVernLen,                                         /**< [Out] Pointer to a variable holding the number of the retrieved key versions. */
        uint16_t * pKeyType                                             /**< [Out] Pointer to a variable that shall store the key type of the retrieved key. */
    );

/**
 * \brief The function retrieves the key type and the available key versions of the indicated key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Command successfully executed.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKey(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the key whose data is retrieved. */
        uint16_t wKeyVer,                                               /**< [In] Key version of the key which has to be retrieved. */
        uint8_t bKeyBufSize,                                            /**< [In] Size of the key buffer \b pKey. */
        uint8_t * pKey,                                                 /**< [Out] Pointer to a buffer receiving the key data for the indicated key number
                                                                         *         and version.
                                                                         */
        uint16_t * pKeyType                                             /**< [Out] Pointer to a variable that shall store the key type of the retrieved key. */
    );

/**
 * \brief Change a Asymmetric Private or Public key entry in Un-compressed Binary format.
 *
 * \note
 *      - This interface should be called more than ones to load Private or Public Key.
 *      - One Key number at a specific position can hold both Private and Public Key.
 *      - Public Key alone can be stored also.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetKeyASym(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the KeyStore to be loaded. */
        uint16_t wPos,                                                  /**< [In] Position at which the Key should be updated. */
        uint16_t wKeyType,                                              /**< [In] New Key type of the KeyEntry (predefined type of KeyType). */
        uint16_t wKeyInfo,                                              /**< [In] Type of Key pair and Curve to save. */
        uint8_t * pKey,                                                 /**< [In] Pointer to the key itself. */
        uint16_t wKeyLen                                                /**< [In] Length of bytes available in \b pKey buffer. */
    );

/**
 * \brief Get a ASymmetric key.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_GetKeyASym(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wKeyNo,                                                /**< [In] Key number of the KeyStore to be retrieved. */
        uint16_t wPos,                                                  /**< [In] Position at which the Key should be retrieved. */
        uint16_t wKeyPairType,                                          /**< [In] Type of Key-pair to be retrieved. */
        uint16_t * pKeyType,                                            /**< [In] Type of key being saved earlier. */
        uint8_t * pCurveID,                                             /**< [In] One of the \ref phKeyStore_ASym_Defines_CurveID "Curve ID's". */
        uint8_t * pKey,                                                 /**< [Out] Pointer to the key itself. The buffer size should be equal or greater
                                                                         *         than the Size of the stored key.
                                                                         */
        uint16_t * pKeyLen                                              /**< [Out] Length of bytes available in \b pKey buffer. */
    );





#ifdef NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE_NDA
/** \defgroup phhalHw_DUT_Cmd_KeyStore_Provision KeyStore Provisioning
 * \brief KeyStore Provisioning Command of HAL DUT Command dispatcher frame.
 * @{
 */

/** \defgroup phhalHw_DUT_KeyStore_Provision_CommandCodes Command_Codes
 * \brief KeyStore Provisioning Command Codes of HAL DUT Command dispatcher frame.
 * @{
 */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_INIT                0x70A1U /**< Initialize DUT Key Provisioning stack. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_DEINIT              0x70A2U /**< Un-Initialize DUT Key Provisioning stack. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_SET_CONFIG          0x70A3U /**< Set provisioning Configuration. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_GET_CONFIG          0x70A4U /**< Get provisioning Configuration. */
#define PHHAL_HW_DUT_CMD_KEYSTORE_PROVISION_APP_FIXED_KEYS      0x70A5U /**< Perform Provisioning of Symmetric Keys. */
/**
 * end of phhalHw_DUT_KeyStore_Provision_CommandCodes group
 * @}
 */

/**
 * \brief Initializes the KeyStore component for Key provisioning.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_Provision_Init(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wOption,                                               /**< [In] Options to perform the following. */
        uint8_t * pTransportKey_AES128,                                 /**< [In] Array containing AES 128-Bit [16 Byte] Transport Key. */
        uint8_t * pTransportKey_AES256,                                 /**< [In] Array containing AES 256-Bit [32 Byte] Transport Key. */
        uint8_t * pAppRootKey_AES128,                                   /**< [In] Array containing AES 128-Bit [16 Byte]  Application Root Key. Should not be Null. */
        uint8_t * pAppRootKey_AES256,                                   /**< [In] Array containing AES 256-Bit [32 Byte]  Application Root Key. Should not be Null. */
        uint8_t * pExpDecData,                                          /**< [In] Array containing AES 128-Bit  [16 Byte] expected Decrypted data to be used while */
        uint8_t * pDervMsg,                                             /**< [In] Array containing derivation message. Should not be NULL. */
        uint8_t * pWrapIV                                               /**< [In] IV used for wrapping the key. This information will be used while provisioning the Key. */
    );

/**
 * \brief De-Initializes PN7642 KeyStore component for Key provisioning.
 */
void phhalHw_DUT_Cmd_KeyStore_Provision_DeInit(
        phhalHw_DUT_DataParams_t * pDataParams                          /**< [In] Pointer to an initialized HAL parameter structure. */
    );

/**
 * \brief Set configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_SetConfig_Ext(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier. */
        uint8_t * pValue,                                               /**< [In] Configuration Value to update. */
        uint16_t wValueLen                                              /**< [In] Length of byte available in \b pValue buffer. */
    );

/**
 * \brief Get configuration parameter.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_GetConfig_Ext(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint16_t wConfig,                                               /**< [In] Configuration Identifier. */
        uint8_t * pValue,                                               /**< [In] Configuration Value to retrieve. */
        uint16_t * pValueLen                                            /**< [In] Length of byte available in \b pValue buffer. */
    );

/**
 * \brief Provision Application Fixed Keys.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlying component.
 */
phStatus_t phhalHw_DUT_Cmd_KeyStore_Provision_AppFixedKeys(
        phhalHw_DUT_DataParams_t * pDataParams,                         /**< [In] Pointer to an initialized HAL parameter structure. */
        uint8_t bKeyIndex,                                              /**< [In] Key number of the hardware keys store to be provisioned. */
        uint16_t wKeyType,                                              /**< [In] Key type of the key to be loaded. Should be one of the following */
        uint8_t * pNewKey                                               /**< [In] Pointer to the key itself. */
    );
/**
 * end of phhalHw_DUT_Cmd_KeyStore_Provision group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE_NDA */

/**
 * end of phhalHw_DUT_Cmd_KeyStore group
 * @}
 */
#endif /* NXPBUILD__PHHAL_HW_DUT_CMD_KEYSTORE */

/**
 * end of phhalHw_DUT_Cmd group
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_DUT */

#endif /* PHHALHW_DUT_CMD_H */
