/*
 * Copyright 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Generic Oscilloscope Device Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#ifndef PHDLPOWERSUPPLY_H
#define PHDLPOWERSUPPLY_H

#include <ph_Status.h>
#include <phbalReg.h>

#ifdef __cplusplus
extern "C" {
#endif	/* __cplusplus */

#ifdef NXPBUILD__PHDL_POWERSUPPLY_E3631A

/** \cond INCLUDE_INTERNALS \defgroup phdlPowerSupply_E3631A Component : E3631A
* @{
*/
#define PHDL_POWERSUPPLY_E3631A_ID  0x01U				/**< ID for Agilent E3631A device layer. */

/**
* \brief Oscilloscope Software parameter structure
*/
typedef struct
{
    uint16_t wId;										/**< Layer ID for this component, NEVER MODIFY! */
    phbalReg_Visa_DataParams_t * pBalRegDataParams;		  /**< Pointer to the parameter structure of the balReg component. */
    uint8_t bChannel;                                   /**< Specifies the channel on which following Get/SetConfigs will will apply. */
    uint16_t wMillivolts[3];                            /**< Storage for the Millivoltsettings. */
    uint8_t bStateOfChannels[3];                        /**< Storage for the information if the corresponding channels are PH_ON or PH_OFF */
} phdlPowerSupply_E3631A_DataParams_t;

/**
* \brief Initialise this layer.
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
*/
phStatus_t phdlPowerSupply_E3631A_Init(
    phdlPowerSupply_E3631A_DataParams_t * pDataParams,  /**< [In] Pointer to this layer's parameter structure. */
    uint16_t wSizeOfDataParams,								          /**< [In] Specifies the size of the data parameter structure. */
    phbalReg_Visa_DataParams_t * pBalRegDataParams			/**< [In] Pointer to the parameter structure of the balReg component. */
    );

/** @} \endcond */
#endif /* NXPBUILD__PHDL_POWERSUPPLY_E3631A */

#ifdef NXPBUILD__PHDL_POWERSUPPLY

/** \cond INCLUDE_INTERNALS \defgroup phdlPowerSupply Powersupply
* \brief These Components implement the Powersupply
* @{
*/

/**
* \name Osci Configs
*/
/*@{*/

#define PHDL_POWERSUPPLY_CONFIG_VOLT            0x0000U /**< Select the Volt-part of the voltage on the currently set channel */
#define PHDL_POWERSUPPLY_CONFIG_MILLIVOLT       0x0001U /**< Select the Millivolts-part of the voltage on the currently set channel */
#define PHDL_POWERSUPPLY_CONFIG_CHANNEL         0x0002U /**< Select the channel on which the following Configs will be changed */
/*@}*/

/** \name Power Supply Channels
*/
/*@{*/

#define PHDL_POWERSUPPLY_CHANNEL_ONE            0x0000U /**< Trigger setting Auto */
#define PHDL_POWERSUPPLY_CHANNEL_TWO            0x0001U /**< Trigger setting Normal */
#define PHDL_POWERSUPPLY_CHANNEL_THREE          0x0002U /**< Trigger setting Single */
/*@}*/

/**
* \brief Set Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/

phStatus_t phdlPowerSupply_SetConfig(
        void * pDataParams,			        /**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,		        /**< [In] Specifies parameter identifier. */
        uint16_t wValue				          /**< [In] Specifies parameter value. */
        );

/**
* \brief Get Config values
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlPowerSupply_GetConfig(
        void * pDataParams,			/**< [In] Pointer to this layer's parameter structure. */
        uint16_t wIdentifier,		/**< [In] Specifies parameter identifier. */
        uint16_t * pwValue			/**< [Out] Return parameter value. */
        );

/**
* \brief Set/Activate the configured power
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlPowerSupply_PowerOn(
        void * pDataParams,			    /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel				    /**< [In] Specifies the channel on which to perform the action. */
        );

/**
* \brief Set/Activate the configured power
* \return Status code
* \retval #PH_ERR_SUCCESS Operation successful.
* \retval Other Depending on implementation and underlying component.
*/
phStatus_t phdlPowerSupply_PowerOff(
        void * pDataParams,			    /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bChannel				    /**< [In] Specifies the channel on which to perform the action. */
        );


/** @} \endcond */
#endif /* NXPBUILD__PHDL_POWERSUPPLY */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHDLPOWERSUPPLY_H */
