/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for RD710 SAM non X-Mode Keystore.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header for this file
 */
#include "Example-Rd710_SamNonX_KeyStore.h"

/**
 * Reader Library Headers
 */
#include <phbalReg.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phCidManager.h>
#include <phpalMifare.h>
#include <phalMful.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <phalMfc.h>
#include <phalMfp.h>
#include <phhalHw_Rd710_Cmd.h>

/**
 * Key store
 */
#define KEYCOUNT                 0xFFU /**< number of keys */
#define KEYVERSIONS              0x01U /**< number of key versions */

#define SAM_KEY                  0x00U /**< SAM Key address */
#define KEY_VERSION_SAM          0x00U /**< SAM Key version */
#define MIFARE_CLASSIC_KEY       0x01U /**< MIFARE classic key address */
#define KEY_VERSION_MFC          0x00U /**< MIFARE classic key version */
#define MIFARE_PLUS_AES_KEY      0x02U /**< MIFARE Plus AES Key address */
#define KEY_VERSION_MFP_AES      0x00U /**< MIFARE Plus AES key version */
#define KEY_POS                  0x00U

#define SAM_AV1                  0x01U  /**< SAM AV1 */
#define SAM_AV2                  0x02U  /**< SAM AV2 */

#define SAM_LC0         0x00U /**< SAM logical channel 0 */
#define SAM_LC1         0x01U /**< SAM logical channel 1 */

#define PLAIN       0x00U    /**< plain mode */
#define MAC         0x01U    /**< CMAC mode */
#define FULL        0x02U    /**< full mode */

#define GLOBAL_BUFFER_SIZE     0x012CU  /**< global size of transmit and recieve buffer */

#define KEY_VERSION_BUFFER_SIZE 0x06 /**< Key version buffer size */

#define RETRYCOUNT 0x05U  /**< retry count */

/**
 * Pegoda 2 configurations
 */
#define PCSC_READER_P2_NON_X_NAME   "NXP Pegoda S CL 0 0"        /**< Pegoda 2: SAM in non X-Mode */

static const uint8_t gaMifareAKey[] =    /**< MIFARE classic key 0xFF */
    { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

#ifdef NXPBUILD__PH_LOG
phLog_RegisterEntry_t logRegisterEntries[3];
phLog_LogEntry_t logEntries[32];
#endif /* NXPBUILD__PH_LOG */

#define CHECK_SUCCESS(x)                                                \
if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();                                              \
    return 0;                                                           \
}

#ifdef NXPBUILD__PH_LOG
void Log_Callback(void * pDataParams, uint8_t bOption, phLog_LogEntry_t * pEntries, uint16_t wEntryCount)
{
    phStatus_t status;
    uint8_t bIsFirstString;
    uint8_t bIsFirstParam;
    uint16_t wIndex;
    uint16_t wByteIndex;
    uint16_t wOption;
    char sOutput[512];
    char sTmp[7];

    /* satisfy compiler */
    if (pDataParams);

    /* Ignore some calls */
    if ((strncmp((char*)pEntries[0].pString, "phhalHw_SetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_GetConfig", 17) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_ReadReg", 15) == 0) ||
        (strncmp((char*)pEntries[0].pString, "phhalHw_WriteReg", 16) == 0))
    {
        return;
    }

    /* Clear output string */
    sprintf(sOutput, "");

    /* Init. states */
    bIsFirstString = 1;
    bIsFirstParam = 1;

    if (bOption != PH_LOG_OPTION_CATEGORY_GEN)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }

    for (wIndex = 0; wIndex < wEntryCount; ++wIndex)
    {
        if ((pEntries[wIndex].bLogType == PH_LOG_LOGTYPE_INFO) && pEntries[wIndex].wDataLen == 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            if (bIsFirstString)
            {
                switch (bOption)
                {
                case PH_LOG_OPTION_CATEGORY_ENTER:
                    strcat_s(sOutput, sizeof(sOutput), " [ENTRY] (");
                    break;
                case PH_LOG_OPTION_CATEGORY_GEN:
                    strcat_s(sOutput, sizeof(sOutput), " [GEN] (");
                    break;
                case PH_LOG_OPTION_CATEGORY_LEAVE:
                    strcat_s(sOutput, sizeof(sOutput), " [LEAVE] (");
                    break;
                }
                bIsFirstString = 0;
            }
        }
        else
        {
            if (!bIsFirstParam)
            {
                if ((strcmp((char*)pEntries[wIndex].pString, "status") != 0) && (pEntries[wIndex].wDataLen > 0))
                {
                    strcat_s(sOutput, sizeof(sOutput), ", ");
                }
            }
            else
            {
                bIsFirstParam = 0;
            }
        }

        if ((strcmp((char*)pEntries[wIndex].pString, "wOption") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
            wOption = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            strcat_s(sOutput, sizeof(sOutput), "wOption = ");
            switch (wOption & PH_EXCHANGE_MODE_MASK)
            {
            case PH_EXCHANGE_DEFAULT:
                strcat_s(sOutput, sizeof(sOutput), "DEFAULT");
                break;
            case PH_EXCHANGE_TXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "TXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING");
                break;
            case PH_EXCHANGE_RXCHAINING_BUFSIZE:
                strcat_s(sOutput, sizeof(sOutput), "RXCHAINING_BUFSIZE");
                break;
            }
            if ((wOption & PH_EXCHANGE_BUFFERED_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | BUFFERED_BIT");
            }
            if ((wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT) != 0)
            {
                strcat_s(sOutput, sizeof(sOutput), " | LEAVE_BUFFER_BIT");
            }
        }
        else if ((strcmp((char*)pEntries[wIndex].pString, "status") == 0) && (pEntries[wIndex].wDataLen == 2))
        {
            strcat_s(sOutput, sizeof(sOutput), ") = ");
            status = (((uint8_t*)(pEntries[wIndex].pData))[0] << 8) | ((uint8_t*)(pEntries[wIndex].pData))[1];
            sprintf_s(sTmp, sizeof(sTmp), "0x%04X", status);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
        }
        else if (pEntries[wIndex].wDataLen > 0)
        {
            strcat_s(sOutput, sizeof(sOutput), (char*)pEntries[wIndex].pString);
            strcat_s(sOutput, sizeof(sOutput), "(");
            sprintf_s(sTmp, sizeof(sTmp), "%d", pEntries[wIndex].wDataLen);
            strcat_s(sOutput, sizeof(sOutput), sTmp);
            strcat_s(sOutput, sizeof(sOutput), ") = ");

            for (wByteIndex = 0; wByteIndex < pEntries[wIndex].wDataLen; ++wByteIndex)
            {
                sprintf_s(sTmp, sizeof(sTmp), "%02X", ((uint8_t*)pEntries[wIndex].pData)[wByteIndex]);
                strcat_s(sOutput, sizeof(sOutput), sTmp);
                if ((wByteIndex + 1) < pEntries[wIndex].wDataLen)
                {
                    strcat_s(sOutput, sizeof(sOutput), " ");
                }
            }
        }
    }

    if (bOption == PH_LOG_OPTION_CATEGORY_LEAVE)
    {
        strcat_s(sOutput, sizeof(sOutput), "\n");
    }
    else
    {
        strcat_s(sOutput, sizeof(sOutput), ")");
    }
    printf(sOutput);
}

#endif /* NXPBUILD__PH_LOG */

int __cdecl main()
{
    phStatus_t status;
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
    uint8_t aBalBufferTx[GLOBAL_BUFFER_SIZE];        /**< BAL Reader transmit buffer */
    uint8_t aBalBufferRx[GLOBAL_BUFFER_SIZE];        /**< BAL Reader receive buffer */
    uint8_t aAtr[256];                /**< Atr */
    uint8_t aSeed[8];                 /**< Seed */
	uint16_t aVersion[0x1F];          /**< SAM Version */
    uint8_t aHalBufferSam[261];       /**< Hal SAM Buffer array */
	uint8_t bSamHostMode = 0;         /**< SAM host mode */
	uint8_t div = 0;                  /**< diversivication */
    uint8_t authtype = PLAIN;         /**< authentication type: PLAIN, MAC, FULL */
    uint16_t wKeyVersion[KEY_VERSION_BUFFER_SIZE];
	uint16_t wKeyVersionLength[KEY_VERSION_BUFFER_SIZE];
	uint16_t pKeyType[KEY_VERSION_BUFFER_SIZE];

    /**
     * keystore
     */
    phKeyStore_Sw_KeyEntry_t aKeyEntry[KEYCOUNT];
    phKeyStore_Sw_KeyVersionPair_t aKeyVersion[KEYCOUNT * KEYVERSIONS];
    phKeyStore_Sw_KUCEntry_t aKeyUsage[KEYCOUNT];

    /**
	 * Bfl data parameter storage
	 */
    phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
	phbalReg_Rd710Sam_DataParams_t balRd710Sam;       /**< RD710Sam (Windows) BAL parameter structure */
	phhalHw_Rd710_DataParams_t halRd710;              /**< Rd710 HAL parameter structure */
    phpalI14443p3a_Sw_DataParams_t I14443p3a_Sw;      /**< Software PAL-ISO14443P3A parameter structure */
    phpalI14443p4a_Sw_DataParams_t I14443p4a_Sw;      /**< Software PAL-ISO14443P4A parameter structure */
    phpalI14443p4_Sw_DataParams_t I14443p4_Sw;        /**< Software PAL-ISO14443P4 parameter structure */
	phpalMifare_Sw_DataParams_t palMifare_Sw;         /**< Software PAL-MIFARE parameter structure */
    phCryptoSym_Sw_DataParams_t cryptoEnc;            /**< CryptoSym parameter structure for ENC */
    phCryptoSym_Sw_DataParams_t cryptoMac;            /**< CryptoSym parameter structure for MAC */
    phCryptoSym_Sw_DataParams_t cryptoSymRnd;         /**< CryptoSym parameter structure for SymRng */
    phCryptoRng_Sw_DataParams_t cryptoRng;            /**< CryptoRng parameter structure for Rng */
    phKeyStore_Sw_DataParams_t keyStoreSw;            /**< Software KeyStore parameter structure */
    phalMfc_Sw_DataParams_t alMfc;                    /**< MIFARE Classic parameter structure. */
    phhalHw_SamAV2_DataParams_t halSamLC0;            /**< SamAV2 HAL parameter structure LC0 */
    phhalHw_SamAV2_DataParams_t halSamLC1;            /**< SamAV2 HAL parameter structure LC1 */
    phKeyStore_SamAV2_DataParams_t keyStoreSAM;       /**< SAM KeyStore parameter structure */
    void * pHal = NULL;                               /**< Pointer to the parameter structure of the HAL layer. */
	void * pKeyStore;                                 /**< Pointer to a KeyStore parameter structure */
    void * pI14443p3a;                                /**< Pointer to a PAL-ISO14443P3A parameter structure */
    void * pI14443p4a;                                /**< Pointer to a PAL-ISO14443P4A parameter structure */
    void * pI14443p4;                                 /**< Pointer to a PAL-ISO14443P4 parameter structure */
	void * pBal;                                      /**< Pointer to a BAL parameter structure */
	void * pAlMfc;                                    /**< Pointer to a MIFARE Classic parameter structure */

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /* Initialize logging */
#ifdef NXPBUILD__PH_LOG
	status = phLog_Init(Log_Callback, logRegisterEntries, (uint16_t)(sizeof(logRegisterEntries) / sizeof(phLog_RegisterEntry_t)));
	CHECK_SUCCESS(status);
    status = phLog_Register(&halSamLC0, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&halSamLC1, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
    status = phLog_Register(&keyStoreSAM, logEntries, (uint16_t)(sizeof(logEntries) / sizeof(phLog_LogEntry_t)));
    CHECK_SUCCESS(status);
#endif /* NXPBUILD__PH_LOG */

    /**
	 * init. Sw keystore
	 */
    status = phKeyStore_Sw_Init(&keyStoreSw,   /**< [In] Pointer to this layer's parameter structure. */
                                sizeof(phKeyStore_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
                                aKeyEntry,     /**< [In] Pointer to a storage containing the key entries. */
                                KEYCOUNT,      /**< [In] size of pKeyEntries. */
                                aKeyVersion,   /**< [In] Pointer to a storage containing the key version pairs. */
                                KEYVERSIONS,   /**< [In] amount of key versions available in each key entry. */
                                aKeyUsage,     /**< [In] Key usage counter entry storage, size = sizeof(phKeyStore_Sw_KUCEntry_t) * wNumKUCEntries */
                                KEYCOUNT);     /**< [In] Number of Key usage counter entries. */
    CHECK_SUCCESS(status);
	pKeyStore = &keyStoreSw;  /**< set the software keystore */

    /**
	 * init. crypto
	 */
    status = phCryptoSym_Sw_Init(&cryptoEnc,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);  /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoMac,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoSymRnd, /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoRng_Sw_Init(&cryptoRng,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoRng_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 &cryptoSymRnd);  /**< [In] Pointer to the parameter structure of the symmetric crypto layer. */
    CHECK_SUCCESS(status);

    /**
     * clear seed
     */
    memset(aSeed, 0x00, 0x08);
    /**
     * seed the random number generator with the given seed
     */
    status = phCryptoRng_Seed(&cryptoRng,      /**< [In] CryptoSym parameter structure for Rng */
                              aSeed,           /**< [In] Seed */
                              sizeof(aSeed));  /**< [In] Size of the seed. */
    CHECK_SUCCESS(status);

    printf("----------------\n");
    printf("Connected Pegoda\n");
    printf("----------------\n");
    printf("[2] Pegoda 2: SAM in non X-mode\n");
	printf("\n\n");

 	/**
	 * Init. PCSC BAL
	 */
	status = phbalReg_PcscWin_Init(&balPcsc,                                /**< [In] Pointer to this layer's parameter structure. */
		                           sizeof(phbalReg_PcscWin_DataParams_t),   /**< [In] Specifies the size of the data parameter structure. */
                                   aAtr,                                    /**< [In] ATR buffer. */
                                   sizeof(aAtr));                           /**< [In] ATR buffer size. */
	CHECK_SUCCESS(status);

	pBal = &balPcsc;  /**< set the BAL PCSC */

	/**
	 * Open Pegoda 2 and init HAL Pegoda2 for the mode "SAM in non X-Mode"
	 */
	/**
	 * set the BAL Communication Configuration values
	 */
	status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
		                        PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,  /**< BAL Communication Configs */
			                    PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED); /**< BAL Communication Configuration values */
	CHECK_SUCCESS(status);
	/**
	 * set the BAL Communication Configs
	 */
	status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
		                        PHBAL_REG_PCSCWIN_CONFIG_SHARE,  /**< BAL Communication Configs */
							    PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);  /**< BAL Communication Configuration values */
	CHECK_SUCCESS(status);

	/**
	 * Select Port to be used (SAM in non X-Mode)
	 */
    status = phbalReg_SetPort(&balPcsc,  /**< [In] Pointer to this layer's parameter structure. */
		                      (uint8_t*)PCSC_READER_P2_NON_X_NAME);  /**< [In] Port Name as String. */
    CHECK_SUCCESS(status);

	/**
	 * Initialise the Rd710 HAL component:
	 *   RD710 with GLOBAL_BUFFER_SIZE bytes send/receive buffer
	 */
	status = phhalHw_Rd710_Init(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
		                        sizeof(phhalHw_Rd710_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
								&balPcsc,   /**< [In] Pointer to the lower layers parameter structure */
								0,          /**< [In] Slot number */
								aHalBufferReaderTx,  /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
								sizeof(aHalBufferReaderTx),        /**< [In] Size of the global transmit buffer. */
								aHalBufferReaderRx,  /**< [In] Pointer to global receive buffer used by the Exchange() function. */
								sizeof(aHalBufferReaderRx));       /**< [In] Size of the global receive buffer. */
	CHECK_SUCCESS(status);

	pHal = &halRd710;  /**< Set HAL pointer */

	/**
	 * Initialise the RD710SAM BAL component
	 */
	status = phbalReg_Rd710Sam_Init(&balRd710Sam,
		                            sizeof(phbalReg_Rd710Sam_DataParams_t),
									&balPcsc,
                                    aAtr,                  /**< [In] Pointer to buffer used for storing the ATR after card activation. */
                                    sizeof(aAtr),          /**< [In] Size of the ATR buffer. */
									aBalBufferTx,          /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
							        sizeof(aBalBufferTx),  /**< [In] Size of the global transmit buffer. */
							        aBalBufferRx,          /**< [In] Pointer to global receive buffer used by the Exchange() function. */
							        sizeof(aBalBufferRx));  /**< [In] Size of the global receive buffer. */
	CHECK_SUCCESS(status);

	pBal = &balRd710Sam;  /**< Set BAL pointer */

    /**
     * init the SAM HAL LC0 component (SAM in non X-Mode)
     */
    status = phhalHw_SamAV2_Init(&halSamLC0,                /**< SamAV2 HAL parameter structure LC0 */
	  					         sizeof(phhalHw_SamAV2_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
						         pBal,                      /**< PCSC (Windows) BAL parameter structure */
						         pHal,                      /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
                                 pKeyStore,                 /**< Pointer to KeyStore parameter structure */
                                 &cryptoEnc,                /**< CryptoSym parameter structure for ENC */
                                 &cryptoMac,                /**< CryptoSym parameter structure for MAC */
                                 &cryptoRng,                /**< Pointer to the parameter structure of the CryptoRng layer. */
                                 PHHAL_HW_SAMAV2_OPMODE_NON_X, /**< Operation mode */
                                 SAM_LC0,                   /**< Logical channel */
                                 aHalBufferSam,             /**< Hal SAM Buffer array TX */
                                 sizeof(aHalBufferSam),     /**< Hal SAM Buffer array length TX */
                                 aHalBufferSam,             /**< Hal SAM Buffer array RX */
                                 sizeof(aHalBufferSam));    /**< Hal SAM Buffer array length RX */
    CHECK_SUCCESS(status);

    /**
	 * Init hal of sam with HComm AV1 object
     * init the SAM HAL LC1 component
	 */
	status = phhalHw_SamAV2_Init(&halSamLC1,                    /**< SamAV2 HAL parameter structure LC1 */
							     sizeof(phhalHw_SamAV2_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
							     pBal,                          /**< PCSC (Windows) BAL parameter structure */
							     NULL,                          /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
                                 pKeyStore,                     /**< Pointer to KeyStore parameter structure */
                                 &cryptoEnc,                    /**< CryptoSym parameter structure for ENC */
                                 &cryptoMac,                    /**< CryptoSym parameter structure for MAC */
                                 &cryptoRng,                    /**< Pointer to the parameter structure of the CryptoRng layer. */
                                 PHHAL_HW_SAMAV2_OPMODE_X_RC523,   /**< Operation mode */
                                 SAM_LC1,                       /**< Logical channel */
                                 aHalBufferSam,                 /**< Hal SAM Buffer array TX */
                                 sizeof(aHalBufferSam),         /**< Hal SAM Buffer array length TX */
                                 aHalBufferSam,                 /**< Hal SAM Buffer array RX */
                                 sizeof(aHalBufferSam));        /**< Hal SAM Buffer array length RX */
	CHECK_SUCCESS(status);

    /**
	 * Init SAM as keyStore object
	 */
    status = phKeyStore_SamAV2_Init(&keyStoreSAM,  /**< [In] Pointer to this layer's parameter structure. */
								    sizeof(phKeyStore_SamAV2_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
									&halSamLC1);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
	CHECK_SUCCESS(status);

	/**
	 * Open the PCSC port
	 */
	status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	 * Open the BAL Rd710 SAM Port
	 */
	status = phbalReg_OpenPort(&balRd710Sam);  /**< [In] Pointer to this layer's parameter structure. */
    CHECK_SUCCESS(status);

	/**
	 * Detect SAM settings LC0
	 */
	status = phhalHw_SamAV2_DetectMode(&halSamLC0);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

	/**
	 * Detect SAM settings LC1
	 */
	status = phhalHw_SamAV2_DetectMode(&halSamLC1);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

    /**
	 * Get SAM hostmode
	 */
    status = phhalHw_GetConfig(&halSamLC0, PHHAL_HW_SAMAV2_CONFIG_HOSTMODE, aVersion);
	CHECK_SUCCESS(status);

    if (aVersion[0] == SAM_AV1)
    {
        /**
		 * AV1:
		 * Load SAM DES key at position 0 in KeyStore
		 * set the host mode
		 */
        bSamHostMode = SAM_AV1;
        /**
		 * format the key entry
		 */
        status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
			                               SAM_KEY,       /**< [In] KeyEntry number to be Formatted. */
										   PH_KEYSTORE_KEY_TYPE_DES);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
        CHECK_SUCCESS(status);
    }
    else
    {
        /**
		 * AV2:
		 * Load SAM AES128 key at position 0 in KeyStore
		 * set the host mode
		 */
        bSamHostMode = SAM_AV2;
        /**
		 * format the key entry
		 */
        status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
			                               SAM_KEY,    /**< [In] KeyEntry number to be Formatted. */
										   PH_KEYSTORE_KEY_TYPE_AES128);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
        CHECK_SUCCESS(status);
    }

	pKeyStore = &keyStoreSAM;  /**< set the sam keystore */
    pHal = &halSamLC0; /** Change the reader to SAM */

    /**
	 * init the 14443-3A component
	 */
    status = phpalI14443p3a_Sw_Init(&I14443p3a_Sw,  /**< [In] PAL-ISO14443P3A parameter structure */
		                            sizeof(phpalI14443p3a_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
									pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
	 * init the 14443-4A component
	 */
    status = phpalI14443p4a_Sw_Init(&I14443p4a_Sw,  /**< [In] PAL-ISO14443P4A parameter structure */
		                            sizeof(phpalI14443p4a_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
									pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
	 * init the 14443-4 component
	 */
    status = phpalI14443p4_Sw_Init(&I14443p4_Sw,  /**< [In] PAL-ISO14443P4 parameter structure */
		                           sizeof(phpalI14443p4_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
								   pHal);         /**< [In] Pointer to the parameter structure of the underlying layer.*/
    CHECK_SUCCESS(status);

    /**
	 * init. mifare pal
	 */
    status = phpalMifare_Sw_Init(&palMifare_Sw,   /**< [In] PAL-MIFARE parameter structure */
		                         sizeof(phpalMifare_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
								 pHal,            /**< [In] Pointer to the parameter structure of the underlying layer.*/
								 &I14443p4_Sw);   /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
    CHECK_SUCCESS(status);

    status = phalMfc_Sw_Init(&alMfc,        /**< [In] MIFARE Classic parameter structure. */
                             sizeof(phalMfc_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
                             &palMifare_Sw, /**< [In] Pointer to a palMifare component context. */
                             NULL);    /**< [In] Pointer to the parameter structure of the KeyStore component. */
    CHECK_SUCCESS(status);

    /**
	 * Set the generics
	 */
	pI14443p3a = &I14443p3a_Sw;
    pI14443p4a = &I14443p4a_Sw;
    pI14443p4 = &I14443p4_Sw;
	pAlMfc = &alMfc;

	/**
	 * Set ISO 14443p4 configuration parameter
	 */
    status = phpalI14443p4_SetConfig(pI14443p4,  /**< [In] Pointer to this layer's parameter structure. */
	                                 RETRYCOUNT, /**< [In] Configuration Identifier */
		    						 0);         /**< [In] Configuration Value */
    CHECK_SUCCESS(status);

    /**
	 * Authenticate with SAM on LC1 which is the Management channel
	 */
    /**
	 * AV1 or AV2?
	 */
    if (bSamHostMode == SAM_AV1)
    {
        /**
		 * Plain Mode
		 * Perform HOST AUTHENTICATION
		 */
		status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
					  					    0x00,       /**< AuthType: Default */
										    0,          /**< Host Key Number */
										    0x00,       /**< Host Key Version */
										    0x00,       /**< Sam Key Number */
										    0x00,       /**< Sam Key Version */
										    &div,       /**< No Diversification input */
											0);         /**< Diversification length */
		CHECK_SUCCESS(status);
    }
    else
    {
	    /**
		 * Perform HOST AUTHENTICATION
		 */
		status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
											authtype,   /**< AuthType: Default */
											0,          /**< Host Key Number */
											0x00,       /**< Host Key Version */
											0x00,       /**< Sam Key Number */
											0x00,       /**< Sam Key Version */
											&div,       /**< No Diversification input */
											0);         /**< Diversification length */
		CHECK_SUCCESS(status);
    }

	printf("----------------\n");
	printf("Card Example:\n");
    printf("----------------\n");
	printf("\n\n");

	/**
     * Set key class as PICC
     */
	status = phKeyStore_SetConfig(&keyStoreSAM,PH_KEYSTORE_SAMAV2_CONFIG_KEYCLASS,PH_KEYSTORE_SAMAV2_KEY_CLASS_PICC);
		CHECK_SUCCESS(status);
	/**
     * Load MIFARE key at position 1 in KeyStore
     */
    status = phKeyStore_FormatKeyEntry(pKeyStore,
		                               MIFARE_CLASSIC_KEY,
									   PH_KEYSTORE_KEY_TYPE_MIFARE);
    CHECK_SUCCESS(status);
    status = phKeyStore_SetKeyAtPos(pKeyStore,
		                            MIFARE_CLASSIC_KEY,
									KEY_POS,
									PH_KEYSTORE_KEY_TYPE_MIFARE,
									(uint8_t*)gaMifareAKey,
									KEY_VERSION_MFC);
    CHECK_SUCCESS(status);

	status = phKeyStore_GetKeyEntry(pKeyStore,
		                            MIFARE_CLASSIC_KEY,
									KEY_VERSION_BUFFER_SIZE,
									wKeyVersion,
									wKeyVersionLength,
									pKeyType);

	/**
	 * Perform RF-Reset
	 */
	status = phhalHw_FieldReset(pHal);    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
	CHECK_SUCCESS(status);

    /**
	 * Switch off the field
	 */
    status = phhalHw_FieldOff(pHal);
    CHECK_SUCCESS(status);

    /**
	 * we are finished, release bal handle
	 */
    status = phbalReg_ClosePort(&balPcsc);
    CHECK_SUCCESS(status);

    printf("-------------------------\n");
    printf("No more tests available.\n");
    printf("Press any key to continue...\n\n");
    _getch();

    /**
	 * Returns zero if the procedure was successful.
	 */
    return 0;
}

