/*
 * Copyright 2013, 2017, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * Example Source for Rd710.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

/**
 * Header for this file
 */
#include "Example-Rd710.h"

/**
 * Reader Library Headers
 */
#include <phbalReg.h>
#include <phhalHw.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4a.h>
#include <phpalI14443p4.h>
#include <phCidManager.h>
#include <phpalMifare.h>
#include <phalMful.h>
#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <phalMfc.h>
#include <phalMfp.h>
#include <phhalHw_Rd710_Cmd.h>

/**
 * Standard Header
 */
#ifdef _WIN32
#pragma warning(push)           /* PRQA S 3116 */
#pragma warning(disable:4001)   /* PRQA S 3116 */
#endif
#include <memory.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#ifdef _WIN32
#pragma warning(pop)            /* PRQA S 3116 */
#endif

/**
 * Key store
 */
#define KEYCOUNT                 0xFFU /**< number of keys */
#define KEYVERSIONS              0x01U /**< number of key versions */

#define SAM_KEY                  0x00U /**< SAM Key address */
#define KEY_VERSION_SAM          0x00U /**< SAM Key version */
#define MIFARE_CLASSIC_KEY       0x01U /**< MIFARE classic key address */
#define KEY_VERSION_MFC          0x00U /**< MIFARE classic key version */
#define MIFARE_PLUS_AES_KEY      0x02U /**< MIFARE Plus AES Key address */
#define KEY_VERSION_MFP_AES      0x00U /**< MIFARE Plus AES key version */
#define KEY_POS                  0x00U

#define SAM_AV1                  0x01U  /**< SAM AV1 */
#define SAM_AV2                  0x02U  /**< SAM AV2 */

#define SAM_LC0         0x00U /**< SAM logical channel 0 */
#define SAM_LC1         0x01U /**< SAM logical channel 1 */

#define PLAIN       0x00U    /**< plain mode */
#define MAC         0x01U    /**< CMAC mode */
#define FULL        0x02U    /**< full mode */

#define GLOBAL_BUFFER_SIZE     0x012CU  /**< global size of transmit and recieve buffer */

#define MFC_BLOCK   0x00U    /**< MFC used Block address */

/**
 * MIFARE Plus SL3
 */
#define PHYSICAL_AES_START_ADDRESS    0x4000U /**< physical start address of AES key location Address */
/**
 * flags
 */
#define ENCRYPTED_OFF     0x00U    /**< Plain */
#define ENCRYPTED_ON      0x01U    /**< Encrypted */
#define MAC_RESPONSE_OFF  0x00U    /**< No MAC on response */
#define MAC_RESPONSE_ON   0x01U    /**< MAC on response */
#define MAC_COMMAND_OFF   0x00U    /**< No MAC on command */
#define MAC_COMMAND_ON    0x01U    /**< MAC on command */
#define FOLLOWING_AUTH    0x00U    /**< Following Authentication */
#define FIRST_AUTH        0x01U    /**< First Authentication */

#define RETRYCOUNT 0x05U  /**< retry count */

/**
 * Pegoda 2 configurations
 */
#define PCSC_READER_P2_X          '1'       /**< Pegoda 2: SAM in X-Mode */
#define PCSC_READER_P2_NON_X      '2'       /**< Pegoda 2: SAM in non X-Mode */
#define PCSC_READER_P2_NO_SAM     '3'       /**< Pegoda 2: no SAM reader */
#define PCSC_READER_P2_X_NAME       "NXP Pegoda X 0 0"        /**< Pegoda 2: SAM in X-Mode */
#define PCSC_READER_P2_NON_X_NAME   "NXP Pegoda S CL 0 0"        /**< Pegoda 2: SAM in non X-Mode */
#define PCSC_READER_P2_NO_SAM_NAME  "NXP Pegoda N CL 0 0"     /**< Pegoda 2: no SAM reader */

static const uint8_t gaEmptyKeyEntry[] = /**< empty key */
    {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
static const uint8_t gaMifareAKey[] =    /**< MIFARE classic key 0xFF */
    { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
static uint8_t gaAESKey[] =        /**< AES key */
    { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

#define CHECK_SUCCESS(x)                                                \
if ((x) != PH_ERR_SUCCESS)                                              \
{                                                                       \
    printf("An error occured (0x%04X), press any key to exit...", (x)); \
    _getch();                                              \
    return 0;                                                           \
}

int __cdecl main()
{
    phStatus_t status;
    uint8_t aHalBufferReaderTx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader transmit buffer */
    uint8_t aHalBufferReaderRx[GLOBAL_BUFFER_SIZE];  /**< HAL Reader receive buffer */
    uint8_t aBalBufferTx[GLOBAL_BUFFER_SIZE];        /**< BAL Reader transmit buffer */
    uint8_t aBalBufferRx[GLOBAL_BUFFER_SIZE];        /**< BAL Reader receive buffer */
    uint8_t aSak[1];                  /**< Select Acknowledge; uint8_t. */
    uint8_t aUid[10];                 /**< complete Uid; uint8_t[4/7/10]. */
    uint8_t aSeed[8];                 /**< Seed */
	uint8_t aAtr[256];                /**< Atr */
	uint8_t bMoreCardsAvailable;      /**< more card available */
    uint8_t bLength;                  /**< Length */
	int iPegodaConfig;                /**< Pegoda 2 Configuration */
	int iCard;                        /**< example card, MIFARE classic, MIFARE Plus SL3 */
	uint8_t aReadValue[0x10];         /**< array to read values */
	uint16_t aVersion[0x1F];           /**< SAM Version */
    uint8_t aHalBufferSam[261];       /**< Hal SAM Buffer array */
	uint8_t bSamHostMode = 0;         /**< SAM host mode */
	uint8_t div = 0;                  /**< diversivication */
    uint8_t authtype = PLAIN;         /**< authentication type: PLAIN, MAC, FULL */
    uint8_t aPcdCap2Out[6];           /**< Pointer to PCDCaps sent from the card (6 bytes). */
    uint8_t aPdCap2[6];               /**< Pointer to PCDCaps. */
    uint8_t aMultiBlocks[16*0x10];    /**< Number of Blocks * 16 bytes(block size) */
    uint8_t aAts[256];                /**< Answer to Select; uint8_t[1-254] */
    uint8_t bCidEnabled;              /**< Unequal '0' if Card Identifier is enabled. */
    uint8_t bCid = 0;                 /**< Card Identifier. */
    uint8_t bNadSupported;            /**< Node Address Support; Unequal '0' if supported. */
    uint8_t bFwi;                     /**< Frame Waiting Integer. */
    uint8_t bFsdi = 8;                /**< PCD Frame Size Integer; 0-8 */
    uint8_t bFsci;                    /**< PICC Frame Size Integer; 0-8. */

    /**
     * keystore
     */
    phKeyStore_Sw_KeyEntry_t aKeyEntry[KEYCOUNT];
    phKeyStore_Sw_KeyVersionPair_t aKeyVersion[KEYCOUNT * KEYVERSIONS];
    phKeyStore_Sw_KUCEntry_t aKeyUsage[KEYCOUNT];

    /**
	 * Bfl data parameter storage
	 */
    phbalReg_PcscWin_DataParams_t balPcsc;            /**< PCSC (Windows) BAL parameter structure */
	phbalReg_Rd710Sam_DataParams_t balRd710Sam;       /**< RD710Sam (Windows) BAL parameter structure */
	phhalHw_Rd710_DataParams_t halRd710;              /**< Rd710 HAL parameter structure */
    phpalI14443p3a_Sw_DataParams_t I14443p3a_Sw;      /**< Software PAL-ISO14443P3A parameter structure */
    phpalI14443p4a_Sw_DataParams_t I14443p4a_Sw;      /**< Software PAL-ISO14443P4A parameter structure */
    phpalI14443p4_Sw_DataParams_t I14443p4_Sw;        /**< Software PAL-ISO14443P4 parameter structure */
    phpalI14443p3a_SamAV2_X_DataParams_t I14443p3a_X; /**< SAM PAL-ISO14443P3A parameter structure */
	phpalI14443p4a_SamAV2_X_DataParams_t I14443p4a_X; /**< SAM PAL-ISO14443P4A parameter structure */
    phpalI14443p4_SamAV2_X_DataParams_t I14443p4_X;   /**< SAM PAL-ISO14443P4 parameter structure */
	phpalMifare_Sw_DataParams_t palMifare_Sw;         /**< Software PAL-MIFARE parameter structure */
	phpalMifare_SamAV2_X_DataParams_t palMifare_X;    /**< SAM PAL-MIFARE parameter structure */
    phCryptoSym_Sw_DataParams_t cryptoEnc;            /**< CryptoSym parameter structure for ENC */
    phCryptoSym_Sw_DataParams_t cryptoMac;            /**< CryptoSym parameter structure for MAC */
    phCryptoSym_Sw_DataParams_t cryptoSymRnd;         /**< CryptoSym parameter structure for SymRng */
    phCryptoRng_Sw_DataParams_t cryptoRng;            /**< CryptoRng parameter structure for Rng */
    phKeyStore_Sw_DataParams_t keyStoreSw;            /**< Software KeyStore parameter structure */
    phalMfc_Sw_DataParams_t alMfc;                    /**< MIFARE Classic parameter structure. */
    phalMfc_SamAV2_X_DataParams_t alMfc_X;            /**< SamAV2 AL-MFC parameter structure */
    phhalHw_SamAV2_DataParams_t halSamLC0;            /**< SamAV2 HAL parameter structure LC0 */
    phhalHw_SamAV2_DataParams_t halSamLC1;            /**< SamAV2 HAL parameter structure LC1 */
    phKeyStore_SamAV2_DataParams_t keyStoreSAM;       /**< SAM KeyStore parameter structure */
    phalMfp_Sw_DataParams_t alMfp_Sw;                 /**< MIFARE Plus Software parameter structure */
    phalMfp_SamAV2_X_DataParams_t alMfp_X;            /**< MIFARE Plus Software parameter structure */
	phalMfp_SamAV2_DataParams_t alMfp;
    void * pHal = NULL;                               /**< Pointer to the parameter structure of the HAL layer. */
	void * pKeyStore;                                 /**< Pointer to a KeyStore parameter structure */
    void * pI14443p3a;                                /**< Pointer to a PAL-ISO14443P3A parameter structure */
    void * pI14443p4a;                                /**< Pointer to a PAL-ISO14443P4A parameter structure */
    void * pI14443p4;                                 /**< Pointer to a PAL-ISO14443P4 parameter structure */
	void * pBal;                                      /**< Pointer to a BAL parameter structure */
	void * pAlMfc;                                    /**< Pointer to a MIFARE Classic parameter structure */
	void * pAlMfp;                                    /**< Pointer to a MIFARE Plus parameter structure */

    printf("\nNxpRdLib ANSI-C Example Program V1.0a\n\n");
    printf("Please ensure that a Pegoda reader is connected and in working condition.\n\n");
    printf("Performing startup...\n\n");

    /**
	 * init. Sw keystore
	 */
    status = phKeyStore_Sw_Init(&keyStoreSw,   /**< [In] Pointer to this layer's parameter structure. */
                                sizeof(phKeyStore_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
                                aKeyEntry,     /**< [In] Pointer to a storage containing the key entries. */
                                KEYCOUNT,      /**< [In] size of pKeyEntries. */
                                aKeyVersion,   /**< [In] Pointer to a storage containing the key version pairs. */
                                KEYVERSIONS,   /**< [In] amount of key versions available in each key entry. */
                                aKeyUsage,     /**< [In] Key usage counter entry storage, size = sizeof(phKeyStore_Sw_KUCEntry_t) * wNumKUCEntries */
                                KEYCOUNT);     /**< [In] Number of Key usage counter entries. */
    CHECK_SUCCESS(status);
	pKeyStore = &keyStoreSw;  /**< set the software keystore */

    /**
	 * init. crypto
	 */
    status = phCryptoSym_Sw_Init(&cryptoEnc,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);  /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoMac,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoSym_Sw_Init(&cryptoSymRnd, /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoSym_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 pKeyStore);    /**< [In] Pointer to a key store structure (can be null).*/
    CHECK_SUCCESS(status);
    status = phCryptoRng_Sw_Init(&cryptoRng,    /**< [In] Pointer to this layer's parameter structure. */
		                         sizeof(phCryptoRng_Sw_DataParams_t),   /**< [In] Specifies the size of the data parameter structure */
								 &cryptoSymRnd);  /**< [In] Pointer to the parameter structure of the symmetric crypto layer. */
    CHECK_SUCCESS(status);

    /**
     * clear seed
     */
    memset(aSeed, 0x00, 0x08);
    /**
     * seed the random number generator with the given seed
     */
    status = phCryptoRng_Seed(&cryptoRng,      /**< [In] CryptoSym parameter structure for Rng */
                              aSeed,           /**< [In] Seed */
                              sizeof(aSeed));  /**< [In] Size of the seed. */
    CHECK_SUCCESS(status);

    printf("----------------\n");
    printf("Connected Pegoda\n");
    printf("----------------\n");
    printf("[1] Pegoda 2: SAM in X-mode\n");
    printf("[2] Pegoda 2: SAM in non X-mode\n");
    printf("[3] Pegoda 2: no SAM reader\n");
    printf("[x] Quit\n");
    printf("----------------\n");
	printf("Choice:");
    iPegodaConfig = _getch();
    printf(" %c", iPegodaConfig);
	printf("\n\n");

 	/**
	 * Init. PCSC BAL
	 */
	status = phbalReg_PcscWin_Init(&balPcsc,  /**< [In] Pointer to this layer's parameter structure. */
		                           sizeof(phbalReg_PcscWin_DataParams_t), /**< [In] Specifies the size of the data parameter structure. */
								   aAtr,   /**< [In] Pointer to buffer used for storing the ATR after card activation. */
								   256);   /**< [In] Size of the ATR buffer. */
	CHECK_SUCCESS(status);

	pBal = &balPcsc;  /**< set the BAL PCSC */

	/**
	 * Open Pegoda 2 and init HAL Pegoda2 for the modes: "SAM in non X-Mode" and "no SAM reader"
	 */
	if (iPegodaConfig != PCSC_READER_P2_X)
	{
		/**
		 * set the BAL Communication Configuration values
		 */
		status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
			                        PHBAL_REG_PCSCWIN_CONFIG_PROTOCOL,  /**< BAL Communication Configs */
				                    PHBAL_REG_PCSCWIN_VALUE_PROTOCOL_UNDEFINED); /**< BAL Communication Configuration values */
		CHECK_SUCCESS(status);
		/**
		 * set the BAL Communication Configs
		 */
		status = phbalReg_SetConfig(&balPcsc,    /**< [In] Pointer to this layer's parameter structure. */
			                        PHBAL_REG_PCSCWIN_CONFIG_SHARE,  /**< BAL Communication Configs */
								    PHBAL_REG_PCSCWIN_VALUE_SHARE_DIRECT);  /**< BAL Communication Configuration values */
		CHECK_SUCCESS(status);

		/**
		 * Pegoda2 port configuration:
		 */
		if(iPegodaConfig == PCSC_READER_P2_NO_SAM)
		{
            /**
			 * Select Port to be used (no SAM reader)
			 */
		    status = phbalReg_SetPort(&balPcsc,  /**< [In] Pointer to this layer's parameter structure. */
				                      (uint8_t*)PCSC_READER_P2_NO_SAM_NAME);  /**< [In] Port Name as String. */
		    CHECK_SUCCESS(status);
		}
		else
		{
			/**
			 * Select Port to be used (SAM in non X-Mode)
			 */
		    status = phbalReg_SetPort(&balPcsc,  /**< [In] Pointer to this layer's parameter structure. */
				                      (uint8_t*)PCSC_READER_P2_NON_X_NAME);  /**< [In] Port Name as String. */
		    CHECK_SUCCESS(status);
		}

		/**
		 * Initialise the Rd710 HAL component:
		 *   RD710 with GLOBAL_BUFFER_SIZE bytes send/receive buffer
		 */
		status = phhalHw_Rd710_Init(&halRd710,  /**< [In] Pointer to this layer's parameter structure. */
			                        sizeof(phhalHw_Rd710_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
									&balPcsc,   /**< [In] Pointer to the lower layers parameter structure */
									0,          /**< [In] Slot number */
									aHalBufferReaderTx,  /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
									sizeof(aHalBufferReaderTx),        /**< [In] Size of the global transmit buffer. */
									aHalBufferReaderRx,  /**< [In] Pointer to global receive buffer used by the Exchange() function. */
									sizeof(aHalBufferReaderRx));       /**< [In] Size of the global receive buffer. */
		CHECK_SUCCESS(status);

		pHal = &halRd710;  /**< Set HAL pointer */

		/**
		 * Initialise the RD710SAM BAL component
		 */
		status = phbalReg_Rd710Sam_Init(&balRd710Sam,
			                            sizeof(phbalReg_Rd710Sam_DataParams_t),
										&balPcsc,
                                        aAtr,                  /**< [In] Pointer to buffer used for storing the ATR after card activation. */
                                        sizeof(aAts),          /**< [In] Size of the ATR buffer. */
										aBalBufferTx,          /**< [In] Pointer to global transmit buffer used by the Exchange() function. */
 								        sizeof(aBalBufferTx),  /**< [In] Size of the global transmit buffer. */
								        aBalBufferRx,          /**< [In] Pointer to global receive buffer used by the Exchange() function. */
								        sizeof(aBalBufferRx)); /**< [In] Size of the global receive buffer. */
		CHECK_SUCCESS(status);

		pBal = &balRd710Sam;  /**< Set BAL pointer */

	}
	else /**< Pegoda2 in X-Mode */
	{
	    /**
		 * Select Port to be used (SAM in X-Mode)
		 */
        status = phbalReg_SetPort(&balPcsc, /**< [In] Pointer to this layer's parameter structure. */
			                      (uint8_t*)PCSC_READER_P2_X_NAME);  /**< [In] Port Name as String. */
		CHECK_SUCCESS(status);
	}

	if(iPegodaConfig != PCSC_READER_P2_NO_SAM)
	{
		if (iPegodaConfig == PCSC_READER_P2_NON_X)
		{
		  /**
		   * init the SAM HAL LC0 component (SAM in non X-Mode)
		   */
		  status = phhalHw_SamAV2_Init(&halSamLC0,                  /**< SamAV2 HAL parameter structure LC0 */
								       sizeof(phhalHw_SamAV2_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
								       pBal,                        /**< PCSC (Windows) BAL parameter structure */
								       pHal,                        /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
								       pKeyStore,                   /**< Pointer to KeyStore parameter structure */
								       &cryptoEnc,                  /**< CryptoSym parameter structure for ENC */
								       &cryptoMac,                  /**< CryptoSym parameter structure for MAC */
								       &cryptoRng,                  /**< Pointer to the parameter structure of the CryptoRng layer. */
                                       PHHAL_HW_SAMAV2_OPMODE_NON_X,   /**< Operation mode */
								       SAM_LC0,                     /**< Logical channel */
								       aHalBufferSam,               /**< Hal SAM Buffer array TX */
								       sizeof(aHalBufferSam),       /**< Hal SAM Buffer array length TX */
		 						       aHalBufferSam,               /**< Hal SAM Buffer array RX */
		  						       sizeof(aHalBufferSam));      /**< Hal SAM Buffer array length RX */
		  CHECK_SUCCESS(status);
		}
		else
		{
          /**
		   * init the SAM HAL LC0 component (SAM in X-Mode)
		   */
  		  status = phhalHw_SamAV2_Init(&halSamLC0,                  /**< SamAV2 HAL parameter structure LC0 */
								       sizeof(phhalHw_SamAV2_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
								       pBal,                        /**< PCSC (Windows) BAL parameter structure */
								       NULL,                        /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
								       pKeyStore,                   /**< Pointer to KeyStore parameter structure */
								       &cryptoEnc,                  /**< CryptoSym parameter structure for ENC */
								       &cryptoMac,                  /**< CryptoSym parameter structure for MAC */
								       &cryptoRng,                  /**< Pointer to the parameter structure of the CryptoRng layer. */
                                       PHHAL_HW_SAMAV2_OPMODE_X_RC523, /**< Operation mode */
								       SAM_LC0,                     /**< Logical channel */
								       aHalBufferSam,               /**< Hal SAM Buffer array TX */
								       sizeof(aHalBufferSam),       /**< Hal SAM Buffer array length TX */
								       aHalBufferSam,               /**< Hal SAM Buffer array RX */
								       sizeof(aHalBufferSam));      /**< Hal SAM Buffer array length RX */
		  CHECK_SUCCESS(status)
		}

        /**
		 * Init hal of sam with HComm AV1 object
         * init the SAM HAL LC1 component
		 */
		status = phhalHw_SamAV2_Init(&halSamLC1,     /**< SamAV2 HAL parameter structure LC1 */
								     sizeof(phhalHw_SamAV2_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
                                     pBal,                        /**< PCSC (Windows) BAL parameter structure */
                                     NULL,                        /**< Pointer to the hal data params of a reader. NULL in case of X-Mode */
                                     pKeyStore,                   /**< Pointer to KeyStore parameter structure */
                                     &cryptoEnc,                  /**< CryptoSym parameter structure for ENC */
                                     &cryptoMac,                  /**< CryptoSym parameter structure for MAC */
                                     &cryptoRng,                  /**< Pointer to the parameter structure of the CryptoRng layer. */
                                     PHHAL_HW_SAMAV2_OPMODE_X_RC523, /**< Operation mode */
                                     SAM_LC1,                     /**< Logical channel */
                                     aHalBufferSam,               /**< Hal SAM Buffer array TX */
                                     sizeof(aHalBufferSam),       /**< Hal SAM Buffer array length TX */
                                     aHalBufferSam,               /**< Hal SAM Buffer array RX */
                                     sizeof(aHalBufferSam));      /**< Hal SAM Buffer array length RX */
		CHECK_SUCCESS(status);

        /**
		 * Init SAM as keyStore object
		 */
        status = phKeyStore_SamAV2_Init(&keyStoreSAM,  /**< [In] Pointer to this layer's parameter structure. */
									    sizeof(phKeyStore_SamAV2_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
										&halSamLC1);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
		CHECK_SUCCESS(status);
	}

	/**
	 * Open the PCSC port
	 */
	status = phbalReg_OpenPort(&balPcsc);  /**< [In] Pointer to this layer's parameter structure. */
	CHECK_SUCCESS(status);

    if (iPegodaConfig != PCSC_READER_P2_X)
	{
		/**
		 * Standard Rd710 Cmd Init.
		 */

		status = (phStatus_t)phhalHw_Rd710_Cmd_InitReader(&halRd710);  /**< [In] Pointer to this layer's parameter structure. */
		CHECK_SUCCESS(status);
	}

	if (iPegodaConfig == PCSC_READER_P2_NON_X)
	{
		/**
		 * Open the BAL Rd710 SAM Port
		 */
		status = phbalReg_OpenPort(&balRd710Sam);  /**< [In] Pointer to this layer's parameter structure. */
	    CHECK_SUCCESS(status);

	}

	if(iPegodaConfig != PCSC_READER_P2_NO_SAM)
	{
		/**
		 * Detect SAM settings LC0
		 */
		status = phhalHw_SamAV2_DetectMode(&halSamLC0);  /**< [In] Pointer to this layer's parameter structure. */
		CHECK_SUCCESS(status);

		/**
		 * Detect SAM settings LC1
		 */
		status = phhalHw_SamAV2_DetectMode(&halSamLC1);  /**< [In] Pointer to this layer's parameter structure. */
		CHECK_SUCCESS(status);

        /**
		 * Get SAM hostmode
		 */
        status = phhalHw_GetConfig(&halSamLC0, PHHAL_HW_SAMAV2_CONFIG_HOSTMODE, aVersion);
		CHECK_SUCCESS(status);

        if (aVersion[0] == SAM_AV1)
        {
            /**
			 * AV1:
			 * Load SAM DES key at position 0 in KeyStore
			 * set the host mode
			 */
            bSamHostMode = SAM_AV1;
            /**
			 * format the key entry
			 */
            status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
				                               SAM_KEY,       /**< [In] KeyEntry number to be Formatted. */
											   PH_KEYSTORE_KEY_TYPE_DES);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
            CHECK_SUCCESS(status);
        }
        else
        {
            /**
			 * AV2:
			 * Load SAM AES128 key at position 0 in KeyStore
			 * set the host mode
			 */
            bSamHostMode = SAM_AV2;
            /**
			 * format the key entry
			 */
            status = phKeyStore_FormatKeyEntry(pKeyStore,  /**< [In] Pointer to the parameter structure of the KeyStore component. */
				                               SAM_KEY,    /**< [In] KeyEntry number to be Formatted. */
											   PH_KEYSTORE_KEY_TYPE_AES128);  /**< [In] New Key type of the KeyEntry (predefined type of KeyType).*/
            CHECK_SUCCESS(status);
        }

		pKeyStore = &keyStoreSAM;  /**< set the sam keystore */
        pHal = &halSamLC0; /** Change the reader to SAM */

		status = phKeyStore_SetConfig(&keyStoreSAM,
									   PH_KEYSTORE_SAMAV2_CONFIG_ALLOW_DUMP_SESSION_KEY,
										0);
		CHECK_SUCCESS(status);
		status = phKeyStore_SetConfig((phKeyStore_SamAV2_DataParams_t *)pKeyStore,
									   PH_KEYSTORE_SAMAV2_CONFIG_ALLOW_CRYPTO_SECRET_KEY,
										0);
		CHECK_SUCCESS(status);
	}

	if (iPegodaConfig != PCSC_READER_P2_X)
	{
	    /**
		 * init the 14443-3A component
		 */
	    status = phpalI14443p3a_Sw_Init(&I14443p3a_Sw,  /**< [In] PAL-ISO14443P3A parameter structure */
			                            sizeof(phpalI14443p3a_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
										pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
	    CHECK_SUCCESS(status);

	    /**
		 * init the 14443-4A component
		 */
	    status = phpalI14443p4a_Sw_Init(&I14443p4a_Sw,  /**< [In] PAL-ISO14443P4A parameter structure */
			                            sizeof(phpalI14443p4a_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
										pHal);          /**< [In] Pointer to the parameter structure of the underlying layer.*/
	    CHECK_SUCCESS(status);

	    /**
		 * init the 14443-4 component
		 */
	    status = phpalI14443p4_Sw_Init(&I14443p4_Sw,  /**< [In] PAL-ISO14443P4 parameter structure */
			                           sizeof(phpalI14443p4_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
									   pHal);         /**< [In] Pointer to the parameter structure of the underlying layer.*/
	    CHECK_SUCCESS(status);

	    /**
		 * init. mifare pal
		 */
	    status = phpalMifare_Sw_Init(&palMifare_Sw,   /**< [In] PAL-MIFARE parameter structure */
			                         sizeof(phpalMifare_Sw_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
									 pHal,            /**< [In] Pointer to the parameter structure of the underlying layer.*/
									 &I14443p4_Sw);   /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
	    CHECK_SUCCESS(status);

		/**
		 * init. MIFARE classic
		 */
		if (iPegodaConfig == PCSC_READER_P2_NO_SAM)
		{
		    status = phalMfc_Sw_Init(&alMfc,        /**< [In] MIFARE Classic parameter structure. */
		                             sizeof(phalMfc_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
		                             &palMifare_Sw, /**< [In] Pointer to a palMifare component context. */
		                             pKeyStore);    /**< [In] Pointer to the parameter structure of the KeyStore component. */
		    CHECK_SUCCESS(status);
		}
		else
		{
		    status = phalMfc_Sw_Init(&alMfc,        /**< [In] MIFARE Classic parameter structure. */
		                             sizeof(phalMfc_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
		                             &palMifare_Sw, /**< [In] Pointer to a palMifare component context. */
		                             NULL);    /**< [In] Pointer to the parameter structure of the KeyStore component. */
		    CHECK_SUCCESS(status);
		}

        /**
		 * Set the generics
		 */
		pI14443p3a = &I14443p3a_Sw;
        pI14443p4a = &I14443p4a_Sw;
        pI14443p4 = &I14443p4_Sw;
		pAlMfc = &alMfc;
	}
	else
	{
	    /**
		 * init the 14443-3A component (SAM in X-Mode)
		 */
	    status = phpalI14443p3a_SamAV2_X_Init(&I14443p3a_X,  /**< [In] Pointer to this layer's parameter structure. */
			                                  sizeof(phpalI14443p3a_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
											  (phhalHw_SamAV2_DataParams_t *)pHal);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
		CHECK_SUCCESS(status);

        /**
		 * init the 14443-4A component (SAM in X-Mode)
		 */
		status = phpalI14443p4a_SamAV2_X_Init(&I14443p4a_X,  /**< [In] Pointer to this layer's parameter structure. */
			                                  sizeof(phpalI14443p4a_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
											  (phhalHw_SamAV2_DataParams_t *)pHal);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
		CHECK_SUCCESS(status);

		/**
		 * init the 14443-4 component (SAM in X-Mode)
		 */
        status = phpalI14443p4_SamAV2_X_Init(&I14443p4_X,  /**< [In] Pointer to this layer's parameter structure. */
			                                 sizeof(phpalI14443p4_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
											 (phhalHw_SamAV2_DataParams_t *)pHal);  /**< [In] Pointer to the parameter structure of the underlying layer.*/
		CHECK_SUCCESS(status);

        /**
		 * init. mifare pal (SAM in X-Mode)
		 */
        status = phpalMifare_SamAV2_X_Init(&palMifare_X,  /**< [In] Pointer to this layer's parameter structure. */
			                               sizeof(phpalMifare_SamAV2_X_DataParams_t),  /**< [In] Specifies the size of the data parameter structure */
										   (phhalHw_SamAV2_DataParams_t *)pHal,    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
										   &I14443p4_X);  /**< [In] Pointer to the parameter structure of the underlying ISO14443-4 layer. */
		CHECK_SUCCESS(status);

		/**
		 * init. MIFARE classic (SAM in X-Mode)
		 */
		status = phalMfc_SamAV2_X_Init(&alMfc_X,        /**< [In] Pointer to this layer's parameter structure. */
									   sizeof(phalMfp_SamAV2_X_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
									   (phhalHw_SamAV2_DataParams_t *)pHal,      /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
									   &palMifare_X);   /**< [In] Pointer to the parameter structure of the palMifare component. */
		CHECK_SUCCESS(status);

        /**
		 * Now lets set the generics
		 */
		pI14443p3a = &I14443p3a_X;
        pI14443p4a = &I14443p4a_X;
        pI14443p4 = &I14443p4_X;
		pAlMfc = &alMfc_X;
	}

	/**
	 * Set ISO 14443p4 configuration parameter
	 */
	if (iPegodaConfig != PCSC_READER_P2_X)
	{
    status = phpalI14443p4_SetConfig(pI14443p4,  /**< [In] Pointer to this layer's parameter structure. */
		                             RETRYCOUNT, /**< [In] Configuration Identifier */
									 0);         /**< [In] Configuration Value */
    CHECK_SUCCESS(status);
	}

    /**
	 * Authenticate with SAM on LC1 which is the Management channel
	 */
	if ( iPegodaConfig != PCSC_READER_P2_NO_SAM)
    {
        /**
		 * AV1 or AV2?
		 */
        if (bSamHostMode == SAM_AV1)
        {
            /**
			 * Plain Mode
			 * Perform HOST AUTHENTICATION
			 */
			status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
						  					    0x00,       /**< AuthType: Default */
											    0,          /**< Host Key Number */
											    0x00,       /**< Host Key Version */
											    0x00,       /**< Sam Key Number */
											    0x00,       /**< Sam Key Version */
											    &div,       /**< No Diversification input */
												0);         /**< Diversification length */
			CHECK_SUCCESS(status);
        }
        else
        {
  		    /**
			 * Perform HOST AUTHENTICATION
			 */
			status = phhalHw_SamAV2_Cmd_SAM_AuthenticateHost(&halSamLC1,  /**< Pointer to this layer's parameter structure. */
												authtype,   /**< AuthType: Default */
												0,          /**< Host Key Number */
												0x00,       /**< Host Key Version */
												0x00,       /**< Sam Key Number */
												0x00,       /**< Sam Key Version */
												&div,       /**< No Diversification input */
												0);         /**< Diversification length */
			CHECK_SUCCESS(status);
        }

		/**
         * RC Init
         */
	if (iPegodaConfig == PCSC_READER_P2_X)
		{
			status = phhalHw_SamAV2_Cmd_RC_Init(&halSamLC1,0);
			CHECK_SUCCESS(status);
		}
	}

    if (iPegodaConfig == PCSC_READER_P2_NO_SAM)
    {
        /**
		 * Load ENC and MAC keys to have at least one default in there....
		 */
        status = phCryptoSym_LoadKeyDirect(&cryptoEnc,                   /**< [In] Pointer to this layer's parameter structure. */
			                               gaEmptyKeyEntry,              /**< [In] key pointer to the key */
										   PH_KEYSTORE_KEY_TYPE_AES128); /**< [In] ID of the Key type */
		CHECK_SUCCESS(status);
		status = phCryptoSym_LoadKeyDirect(&cryptoMac,                   /**< [In] Pointer to this layer's parameter structure. */
			                               gaEmptyKeyEntry,              /**< [In] key pointer to the key */
										   PH_KEYSTORE_KEY_TYPE_AES128); /**< [In] ID of the Key type */
		CHECK_SUCCESS(status);

		/**
		 * Configure Hardware for Type-A cards
		 */
		status = phhalHw_ApplyProtocolSettings(pHal,      /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
											   PHHAL_HW_CARDTYPE_ISO14443A);  /**< [In] Type of card for which the hal should be configured for. */
		CHECK_SUCCESS(status);

        /**
         * Perform RF-Reset
         */
        status = phhalHw_FieldReset(pHal);    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
        CHECK_SUCCESS(status);
    }

	printf("----------------\n");
	printf("Card Example:\n");
    printf("----------------\n");
    printf("[1] MIFARE Classic\n");
    printf("[2] MIFARE Plus SL3\n");
    printf("[x] Quit\n");
    printf("----------------\n");
	printf("Choice:");
    iCard = _getch();
    printf(" %c", iCard);
	printf("\n\n");

	/**
		* Set the Key settings to PICC type & Key Class as PICC
		*/
	if (iPegodaConfig != PCSC_READER_P2_NO_SAM)
    {
			status = phKeyStore_SetConfig(&keyStoreSAM,PH_KEYSTORE_SAMAV2_CONFIG_KEYCLASS,PH_KEYSTORE_SAMAV2_KEY_CLASS_PICC);
		CHECK_SUCCESS(status);
	}
	switch(iCard)
	{
		/**
		 * MIFARE Classic
         * Activate Idle + Authenticate Key A (= all bytes FF), Block 0 + Read Block 0
		 */

	    case '1':
		{

			/**
	         * Load MIFARE key at position 1 in KeyStore
	         */
            status = phKeyStore_FormatKeyEntry(pKeyStore,
				                               MIFARE_CLASSIC_KEY,
											   PH_KEYSTORE_KEY_TYPE_MIFARE);
            CHECK_SUCCESS(status);
            status = phKeyStore_SetKeyAtPos(pKeyStore,
				                            MIFARE_CLASSIC_KEY,
											KEY_POS,
											PH_KEYSTORE_KEY_TYPE_MIFARE,
											(uint8_t*)gaMifareAKey,
											KEY_VERSION_MFC);
            CHECK_SUCCESS(status);

			/**
			 * Configure Hardware for Type-A cards
			 */
			status = phhalHw_ApplyProtocolSettings(pHal,      /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
												   PHHAL_HW_CARDTYPE_ISO14443A);  /**< [In] Type of card for which the hal should be configured for. */
			CHECK_SUCCESS(status);

			/**
			 * Perform RF-Reset
			 */
			status = phhalHw_FieldReset(pHal);    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
			CHECK_SUCCESS(status);

			/**
			 * Activate Layer 3 card
			 */
			status = phpalI14443p3a_ActivateCard(pI14443p3a, NULL, 0x00, aUid, &bLength, aSak, &bMoreCardsAvailable);
			CHECK_SUCCESS(status);

			/**
			 * Authenticate Key A
			 */
			status = phalMfc_Authenticate(pAlMfc, MFC_BLOCK, PHHAL_HW_MFC_KEYA, MIFARE_CLASSIC_KEY, KEY_VERSION_MFC, aUid, bLength);
 			CHECK_SUCCESS(status);

			/**
			 * Read block 0
			 */
			status = phalMfc_Read(pAlMfc, MFC_BLOCK, aReadValue);
			CHECK_SUCCESS(status);
	    }
	    break;

		/**
		 * MIFARE Plus SL3 (AES)
         * Activate Idle + RATS + Authenticate Key A (= all bytes FF), Block 0 + Read Block 0
		 */
	    case '2':
		{

			if (iPegodaConfig == PCSC_READER_P2_NO_SAM)
			{
			   /**
                * init. MIFARE Plus no SAM
                */
               status = phalMfp_Sw_Init(&alMfp_Sw,        /**< [In] MIFARE Plus Software parameter structure */
                                        sizeof(phalMfp_Sw_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
                                        &palMifare_Sw, /**< [In] Pointer to a palMifare component context. */
                                        pKeyStore,     /**< [In] Pointer to a KeyStore component context. */
                                        &cryptoEnc,    /**< [In] Pointer to a Crypto component context for encryption. */
                                        &cryptoMac,    /**< [In] Pointer to a Crypto component context for Macing. */
                                        &cryptoRng,    /**< [In] Pointer to a CryptoRng component context. */
                                        NULL);         /**< [In] Pointer to the parameter structure of the CryptoDiversify layer (can be NULL). */
               CHECK_SUCCESS(status);
		       pAlMfp = &alMfp_Sw;
			}
			else if(iPegodaConfig == PCSC_READER_P2_X)
			{
			  /**
               * init. MIFARE Plus X-mode
               */
              status = phalMfp_SamAV2_X_Init(&alMfp_X,        /**< [In] MIFARE Plus Software parameter structure */
                                             sizeof(phalMfp_SamAV2_X_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
                                             (phhalHw_SamAV2_DataParams_t *)pHal,
											 &palMifare_X); /**< [In] Pointer to a palMifare component context. */
              CHECK_SUCCESS(status);
			  pAlMfp = &alMfp_X;
			}
			else
			{
				/**
				 * init. MIFARE Plus non X-mode
				 */
              status = phalMfp_SamAV2_Init(&alMfp,        /**< [In] MIFARE Plus Software parameter structure */
                                             sizeof(phalMfp_SamAV2_DataParams_t), /**< [In] Specifies the size of the data parameter structure */
                                             (phhalHw_SamAV2_DataParams_t *)pHal,
											 &palMifare_Sw); /**< [In] Pointer to a palMifare component context. */
              CHECK_SUCCESS(status);
			  pAlMfp = &alMfp;
			}

			/**
			 * Configure Hardware for Type-A cards
			 */
			status = phhalHw_ApplyProtocolSettings(pHal,      /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
												   PHHAL_HW_CARDTYPE_ISO14443A);  /**< [In] Type of card for which the hal should be configured for. */
			CHECK_SUCCESS(status);

			/**
			 * Perform RF-Reset
			 */
			status = phhalHw_FieldReset(pHal);    /**< [In] Pointer to the parameter structure of the underlying HAL layer. */
			CHECK_SUCCESS(status);

		    /**
			 * Activate Layer 3 card
			 */
			status = phpalI14443p3a_ActivateCard(pI14443p3a,    /**< [In] PAL-ISO14443P3A parameter structure */
											     NULL,          /**< [In] unknown UID */
												 0x00,          /**< [In] UID length 0 */
												 aUid,          /**< [Out] complete Uid; uint8_t[7]. */
												 &bLength,      /**< [Out] Length of Uid; 4/7/10. */
												 aSak,          /**< [Out] Select Acknowledge; uint8_t. */
												 &bMoreCardsAvailable);  /**< [Out] Whether there are more cards in the field or not; uint8_t. */
			CHECK_SUCCESS(status);

			/**
			 * Activate card I14443p4a, switch to ISO4
			 */
			status = phpalI14443p4a_ActivateCard(pI14443p4a,    /**< [In] PAL-ISO14443P4A parameter structure */
											     bFsdi,         /**< [In] Frame Size Integer; 0-8 */
												 bCid,          /**< [In] Card Identifier; 0-14 */
												 PHPAL_I14443P4A_DATARATE_106,          /**< [In] Divisor Receive (PCD to PICC) Integer; 0-3 */
												 PHPAL_I14443P4A_DATARATE_106,          /**< [In] Divisor Send (PICC to PCD) Integer; 0-3 */
												 aAts);         /**< [Out] Answer to Select; uint8_t[1-254] */
			CHECK_SUCCESS(status);

		    /**
		     * Retrieve the ISO14443-4A protocol parameters
		     */
		    status = phpalI14443p4a_GetProtocolParams(pI14443p4a,    /**< [In]  PAL-ISO14443P4A parameter structure */
		  											  &bCidEnabled,    /**< [Out] Unequal '0' if Card Identifier is enabled. */
		  											  &bCid,           /**< [Out] Card Identifier. */
		  											  &bNadSupported,  /**< [Out] Node Address Support; Unequal '0' if supported. */
		  											  &bFwi,           /**< [Out] Frame Waiting Integer. */
													  &bFsdi,          /**< [Out] PCD Frame Size Integer; 0-8. */
													  &bFsci);         /**< [Out] PICC Frame Size Integer; 0-8. */
		    CHECK_SUCCESS(status);

		    /**
		     * Set the ISO14443-4 protocol parameters
		     */
			if (iPegodaConfig != PCSC_READER_P2_X)
			{
		    status = phpalI14443p4_SetProtocol(pI14443p4,    /**< [In] ISO14443-4 parameter structure */
											   bCidEnabled,    /**< [In] Enable usage of Card Identifier; Unequal '0' if enabled. */
											   bCid,           /**< [In] Card Identifier; Ignored if \c bCidEnable is equal '0'. */
											   bNadSupported,  /**< [In] Enabler usage of Node Address; Unequal '0' if enabled. */
											   0,              /**< [In] Node Address; Ignored if bNadEnabled is equal '0'. */
											   bFwi,           /**< [In] Frame Waiting Integer. */
											   bFsdi,          /**< [In] PCD Frame Size Integer; 0-8 */
											   bFsci);         /**< [In] PICC Frame Size Integer; 0-8 */
		    CHECK_SUCCESS(status);
			}

			/**
		     * Load key(in KeyStore)
		     * First format the key entry
		     */
		    status = phKeyStore_FormatKeyEntry(pKeyStore,
		  									   MIFARE_PLUS_AES_KEY,
			  								   PH_KEYSTORE_KEY_TYPE_AES128);
		    CHECK_SUCCESS(status);

		    /**
		     * Second set the key at position
		     */
		    status = phKeyStore_SetKeyAtPos(pKeyStore,
										    MIFARE_PLUS_AES_KEY,
										    KEY_POS,
										    PH_KEYSTORE_KEY_TYPE_AES128,
										    gaAESKey,
										    KEY_VERSION_MFP_AES);
		    CHECK_SUCCESS(status);

            /**
             * SL3 AES Card Authentication
             */
            status = phalMfp_AuthenticateSL3(pAlMfp,       /**< [In] MIFARE Plus Software parameter structure */
                                             FIRST_AUTH,   /**< [In] \c 0: Following Authentication; \c 1: First Authentication; */
                                             PHYSICAL_AES_START_ADDRESS,    /**< [In] Key Block number. */
                                             MIFARE_PLUS_AES_KEY, /**< [In] Key Storage number. */
                                             KEY_VERSION_MFP_AES,  /**< [In] Key Storage version. */
                                             0,             /**< [In] Length of diversification input used to diversify the key. If 0, no diversification is performed. */
                                             NULL,          /**< [In] Diversification Input used to diversify the key. */
                                             0,             /**< [In] Length of the supplied PCDCaps. */
                                             NULL,          /**< [In] Pointer to PCDCaps (bLenPcdCap2 bytes), ignored if bLenPcdCap2 == 0. */
                                             aPcdCap2Out,   /**< [Out] Pointer to PCDCaps sent from the card (6 bytes). */
                                             aPdCap2);      /**< [Out] Pointer to PDCaps sent from the card (6 bytes). */
            CHECK_SUCCESS(status);

            /**
             * Block 0 read
             */
            status = phalMfp_Read(pAlMfp,  /**< [In] MIFARE Plus Software parameter structure */
                                  ENCRYPTED_ON,    /**< [In] \c 0: Plain communication; \c 1: Encrypted communication; */
                                  MAC_RESPONSE_ON,    /**< [In] \c 0: No MAC on response; \c 1: MAC on response; */
                                  MAC_COMMAND_ON,    /**< [In] \c 0: No MAC on command; \c 1: MAC on command; */
                                  0x00,    /**< [In] MIFARE block number. */
                                  0x01,    /**< [In] Number of blocks to read. */
                                  aMultiBlocks);   /**< [Out] Block(s) (16*bNumBlocks bytes).  */
            CHECK_SUCCESS(status);

		}
	    break;
	    default:

		    break;
	}

    /**
	 * Switch off the field
	 */
    status = phhalHw_FieldOff(pHal);
    CHECK_SUCCESS(status);

    if (iPegodaConfig != PCSC_READER_P2_NO_SAM)
	{
        /**
		 * we are finished, release bal handle
		 */
        status = phbalReg_ClosePort(&balPcsc);
        CHECK_SUCCESS(status);
	}

    printf("-------------------------\n");
    printf("No more tests available.\n");
    printf("Press any key to continue...\n\n");
    _getch();

    /**
	 * Returns zero if the procedure was successful.
	 */
    return 0;


}

