/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) MIFARE Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7230 $
 * $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalMifare.h>
#include <phpalI14443p3a.h>
#include <phpalI14443p4.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#ifdef NXPBUILD__PHPAL_MIFARE_SAM_X

#include "phpalMifare_Sam_X.h"
#include "../phpalMifare_Int.h"

phStatus_t phpalMifare_Sam_X_Init(phpalMifare_Sam_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_Sam_DataParams_t * pHalDataParams, void * pPalI14443p4DataParams)
{
    if(sizeof(phpalMifare_Sam_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_MIFARE);
    }

    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_MIFARE);
    PH_ASSERT_NULL_PARAM(pHalDataParams, PH_COMP_PAL_MIFARE);

    /* init private data */
    pDataParams->wId = PH_COMP_PAL_MIFARE | PHPAL_MIFARE_SAM_X_ID;

    pDataParams->pHalDataParams = pHalDataParams;
    pDataParams->pPalI14443p4DataParams = pPalI14443p4DataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_Sam_X_ExchangeL3(phpalMifare_Sam_X_DataParams_t * pDataParams, uint16_t wOption,
    uint8_t * pTxBuffer, uint16_t wTxBuffLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    phStatus_t  PH_MEMLOC_REM wStatus1;
    uint16_t    PH_MEMLOC_REM wValidBits = 0;

    uint8_t *   PH_MEMLOC_REM pRxBuffer = NULL;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    /* Check if caller has provided valid RxBuffer */
    if(ppResponse == NULL)
    {
        ppResponse = &pRxBuffer;
    }
    if(pRespLen == NULL)
    {
        pRespLen = &wRespLen;
    }

    /* Switch CRC modes in case of first part of exchange. */
    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Enable TxCrc */
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TXCRC, PH_ON));

        /* Disable RxCrc */
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_RXCRC, PH_OFF));
    }

    /* Perform Exchange */
    wStatus = phhalHw_Exchange(
        pDataParams->pHalDataParams,
        wOption,
        pTxBuffer,
        wTxBuffLen,
        ppResponse,
        pRespLen);

    /* Return if no real exchange is done */
    if(wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return wStatus;
    }

    /* Convert HAL wStatus codes to palMifare wStatus codes */
    wStatus = phpalMifare_Sam_X_ConvertNak(wStatus);

    /* ACK/NAK Handling */
    if((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_INCOMPLETE_BYTE)
    {
        /* Check for protocol error */
        if(*pRespLen != 1)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
        }

        /* Retrieve bitcount */
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_GetConfig(pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_RXLASTBITS, &wValidBits));

        /* Check for protocol error */
        if(wValidBits != 4)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
        }

        /* ACK/NAK Mapping */
        switch((*ppResponse)[0])
        {
            /* ACK -> everything OK */
            case PHPAL_MIFARE_RESP_ACK:
                wStatus = PH_ERR_SUCCESS;
                break;

                /* Mapping of NAK codes: */
            case PHPAL_MIFARE_RESP_NAK0:
                wStatus = PHPAL_MIFARE_ERR_NAK0;
                break;

            case PHPAL_MIFARE_RESP_NAK1:
                wStatus = PHPAL_MIFARE_ERR_NAK1;
                break;

            case PHPAL_MIFARE_RESP_NAK4:
                wStatus = PHPAL_MIFARE_ERR_NAK4;
                break;

            case PHPAL_MIFARE_RESP_NAK5:
                wStatus = PHPAL_MIFARE_ERR_NAK5;
                break;

            default:
                wStatus = PH_ERR_PROTOCOL_ERROR;
                break;
        }
    }
    /* Normal data stream with CRC */
    else
    {
        /* Check wStatus */
        PH_CHECK_SUCCESS(wStatus);

        /* Check length (min. 1 byte + 2 byte CRC) */
        if(*pRespLen < 3U)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_MIFARE);
        }
    }

    return PH_ADD_COMPCODE(wStatus, PH_COMP_PAL_MIFARE);
}

phStatus_t phpalMifare_Sam_X_ExchangeL4(phpalMifare_Sam_X_DataParams_t * pDataParams, uint16_t wOption,
    uint8_t * pTxBuffer, uint16_t wTxBuffLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
#ifdef NXPBUILD__PHPAL_I14443P4
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint16_t    PH_MEMLOC_REM wValue;

    /* Not available if no Layer4 has been set */
    if(pDataParams->pPalI14443p4DataParams == NULL)
    {
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
    }

    /* Retrieve TxCrc-setting */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_TXCRC, &wValue));

    /* Enable TxCrc */
    if(wValue == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_TXCRC, PH_ON));
    }

    /* Retrieve RxCrc-setting */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_GetConfig(pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_RXCRC, &wValue));

    /* Enable RxCrc */
    if(wValue == PH_OFF)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(pDataParams->pHalDataParams,
            PHHAL_HW_CONFIG_RXCRC, PH_ON));
    }

    /* Perform Exchange */
    PH_CHECK_SUCCESS_FCT(wStatus, phpalI14443p4_Exchange(
        pDataParams->pPalI14443p4DataParams,
        wOption,
        pTxBuffer,
        wTxBuffLen,
        ppResponse,
        pRespLen));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_MIFARE);
#else
    /* satisfy compiler */
    if(pDataParams || wOption || pTxBuffer || wTxLength || ppRxBuffer || pRxLength);
    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_MIFARE);
#endif
}

phStatus_t phpalMifare_Sam_X_ConvertNak(phStatus_t wStatus)
{
    switch(wStatus & PH_ERR_MASK)
    {
        case 0x90F0U:
            return PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK0, PH_COMP_PAL_MIFARE);

        case 0x90F1U:
            return PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK1, PH_COMP_PAL_MIFARE);

        case 0x90F4U:
            return PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK4, PH_COMP_PAL_MIFARE);

        case 0x90F5U:
            return PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK5, PH_COMP_PAL_MIFARE);

        case 0x90F6U:
            return PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK6, PH_COMP_PAL_MIFARE);

        case 0x90F7U:
            return PH_ADD_COMPCODE(PHPAL_MIFARE_ERR_NAK7, PH_COMP_PAL_MIFARE);

        default:
            return wStatus;
    }
}

#endif /* NXPBUILD__PHPAL_MIFARE_SAM_X */
