/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ISO14443-4A Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7230 $
 * $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalI14443p4a.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I14443P4A_SAM_X

#include "phpalI14443p4a_Sam_X.h"
#include "phpalI14443p4a_Sam_X_Int.h"

#include "../../../phhalHw/src/Sam/Commands/17_ISO14443_4/phhalHw_Sam_Cmd_ISO14443_4.h"

phStatus_t phpalI14443p4a_Sam_X_Init(phpalI14443p4a_Sam_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_Sam_DataParams_t * pHalDataParams)
{
    if (sizeof(phpalI14443p4a_Sam_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_ISO14443P4A);
    }

    PH_ASSERT_NULL_DATA_PARAM (pDataParams, PH_COMP_PAL_ISO14443P4A);
    PH_ASSERT_NULL_PARAM (pHalDataParams, PH_COMP_PAL_ISO14443P4A);

    /* init private data */
    pDataParams->wId            = PH_COMP_PAL_ISO14443P4A | PHPAL_I14443P4A_SAM_X_ID;
    pDataParams->pHalDataParams = pHalDataParams;
    pDataParams->bCidSupported  = PH_OFF;
    pDataParams->bNadSupported  = PH_OFF;
    pDataParams->bCid           = 0x00;
    pDataParams->bBitRateCaps   = 0x00;
    pDataParams->bFwi           = PHPAL_I14443P4A_SAM_X_FWI_DEFAULT;
    pDataParams->bFsci          = PHPAL_I14443P4A_SAM_X_FSCI_DEFAULT;
    pDataParams->bDri           = 0x00;
    pDataParams->bDsi           = 0x00;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Sam_X_Rats(phpalI14443p4a_Sam_X_DataParams_t * pDataParams, uint8_t bFsdi, uint8_t bCid,
    uint8_t * pAts)
{
    phStatus_t  PH_MEMLOC_REM wStatus;

    /* Parameter check */
    if((bFsdi > 8U) || (bCid > 14U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Set default protocol settings */
    pDataParams->bCidSupported = PH_ON;
    pDataParams->bCid = bCid;
    pDataParams->bNadSupported = PH_OFF;
    pDataParams->bBitRateCaps = 0x00U;
    pDataParams->bFwi = 0x04U;
    pDataParams->bDri = 0x00U;
    pDataParams->bDsi = 0x00U;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_X_ISO14443_4_RATS_PPS(
        pDataParams->pHalDataParams,
        pDataParams->bCid,
        pDataParams->bDsi,
        pDataParams->bDri,
        &pDataParams->bCid,
        &pDataParams->bDsi,
        &pDataParams->bDri,
        pAts));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Sam_X_Pps(phpalI14443p4a_Sam_X_DataParams_t * pDataParams, uint8_t bDri, uint8_t bDsi)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM cmd[3];
    uint16_t    PH_MEMLOC_REM wTxDataRate;
    uint16_t    PH_MEMLOC_REM wRxDataRate;

    uint8_t     PH_MEMLOC_REM * pResponse;
    uint16_t    PH_MEMLOC_REM wRespLen;

    wRespLen = 0;

    /* Check and convert Dri value */
    switch(bDri)
    {
        case PHPAL_I14443P4A_DATARATE_106:
            wTxDataRate = PHHAL_HW_RF_DATARATE_106;
            break;

        case PHPAL_I14443P4A_DATARATE_212:
            wTxDataRate = PHHAL_HW_RF_DATARATE_212;
            break;

        case PHPAL_I14443P4A_DATARATE_424:
            wTxDataRate = PHHAL_HW_RF_DATARATE_424;
            break;

        case PHPAL_I14443P4A_DATARATE_848:
            wTxDataRate = PHHAL_HW_RF_DATARATE_848;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Check and convert Dsi value */
    switch(bDsi)
    {
        case PHPAL_I14443P4A_DATARATE_106:
            wRxDataRate = PHHAL_HW_RF_DATARATE_106;
            break;

        case PHPAL_I14443P4A_DATARATE_212:
            wRxDataRate = PHHAL_HW_RF_DATARATE_212;
            break;

        case PHPAL_I14443P4A_DATARATE_424:
            wRxDataRate = PHHAL_HW_RF_DATARATE_424;
            break;

        case PHPAL_I14443P4A_DATARATE_848:
            wRxDataRate = PHHAL_HW_RF_DATARATE_848;
            break;

        default:
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Send Pps command */
    cmd[0U] = PHPAL_I14443P4A_SAM_X_PPSS | pDataParams->bCid;
    cmd[1U] = PHPAL_I14443P4A_SAM_X_PPS0;
    cmd[2U] = ((bDsi << 2U) & 0x0C) | (bDri & 0x03);
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Exchange(
        pDataParams->pHalDataParams,
        PH_EXCHANGE_DEFAULT,
        cmd,
        3U,
        &pResponse,
        &wRespLen));

    /* Check length and start byte of PPS */
    if((wRespLen != 1U) || (pResponse[0] != (PHPAL_I14443P4A_SAM_X_PPSS | pDataParams->bCid)))
    {
        return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
    }

    /* Store data rates */
    pDataParams->bDri = bDri;
    pDataParams->bDsi = bDsi;

    /* Apply new data rates to Pcd */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_TXDATARATE, wTxDataRate));

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_SetConfig(pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_RXDATARATE, wRxDataRate));

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Sam_X_ActivateCard(phpalI14443p4a_Sam_X_DataParams_t * pDataParams, uint8_t bFsdi,
    uint8_t bCid, uint8_t bDri, uint8_t bDsi, uint8_t * pAts)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    uint8_t     PH_MEMLOC_REM bAtsIndex;
    uint8_t     PH_MEMLOC_REM bRespLen;

    /* Parameter check */
    if ((bFsdi != 8U) || (bCid > 14U))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Check Dri value */
    switch (bDri)
    {
    case PHPAL_I14443P4A_DATARATE_106:
    case PHPAL_I14443P4A_DATARATE_212:
    case PHPAL_I14443P4A_DATARATE_424:
    case PHPAL_I14443P4A_DATARATE_848:
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Check Dri value */
    switch (bDsi)
    {
    case PHPAL_I14443P4A_DATARATE_106:
    case PHPAL_I14443P4A_DATARATE_212:
    case PHPAL_I14443P4A_DATARATE_424:
    case PHPAL_I14443P4A_DATARATE_848:
        break;
    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_PAL_ISO14443P4A);
    }

    /* Set default values */
    /* */
    pDataParams->bCidSupported  = PH_ON;
    pDataParams->bCid           = bCid;
    pDataParams->bNadSupported  = PH_OFF;
    pDataParams->bBitRateCaps   = 0x00;
    pDataParams->bFwi           = PHPAL_I14443P4A_SAM_X_FWI_DEFAULT;
    pDataParams->bFsci          = PHPAL_I14443P4A_SAM_X_FSCI_DEFAULT;
    pDataParams->bDri           = 0x00;
    pDataParams->bDsi           = 0x00;

    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_X_ISO14443_4_RATS_PPS(
        pDataParams->pHalDataParams,
        bCid,
        bDsi,
        bDri,
        &pDataParams->bCid,
        &pDataParams->bDsi,
        &pDataParams->bDri,
        pAts));

    /* Retrieve ATS information */
    /* Start parsing with T0 byte */
    bRespLen = pAts[PHPAL_I14443P4A_SAM_X_ATS_LEN];
    bAtsIndex   = PHPAL_I14443P4A_SAM_X_ATS_T0;

    /* Parse T0 */
    pDataParams->bFsci = pAts[bAtsIndex] & 0x0F;
    if (pDataParams->bFsci > 8)
    {
        pDataParams->bFsci = 8;
    }
    ++bAtsIndex;

    /* Parse TA(1) */
    if (pAts[PHPAL_I14443P4A_SAM_X_ATS_T0] & PHPAL_I14443P4A_SAM_X_ATS_TA1_PRESENT)
    {
        /* Check for protocol error */
        if (bRespLen <= bAtsIndex)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
        }
        else
        {
            /* Store Bitrate capabilities */
            pDataParams->bBitRateCaps = pAts[bAtsIndex];

            ++bAtsIndex;
        }
    }

    /* Parse TB(1) */
    if (pAts[PHPAL_I14443P4A_SAM_X_ATS_T0] & PHPAL_I14443P4A_SAM_X_ATS_TB1_PRESENT)
    {
        /* Check for protocol error */
        if (bRespLen <= bAtsIndex)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
        }
        else
        {
            /* Store Fwi */
            pDataParams->bFwi = (pAts[bAtsIndex] >> 4) & 0x0F;
            if (pDataParams->bFwi == 0x0F)
            {
                pDataParams->bFwi = PHPAL_I14443P4A_SAM_X_FWI_DEFAULT;
            }

            ++bAtsIndex;
        }
    }

    /* Parse TC(1) */
    if (pAts[PHPAL_I14443P4A_SAM_X_ATS_T0] & PHPAL_I14443P4A_SAM_X_ATS_TC1_PRESENT)
    {
        /* Check for protocol error */
        if (bRespLen <= bAtsIndex)
        {
            return PH_ADD_COMPCODE(PH_ERR_PROTOCOL_ERROR, PH_COMP_PAL_ISO14443P4A);
        }
        else
        {
            /* Check NAD Support */
            if (pAts[bAtsIndex] & PHPAL_I14443P4A_SAM_X_ATS_TC1_NAD_SUPPORT)
            {
                pDataParams->bNadSupported = 1;
            }

            /* Check CID Support */
            if (!(pAts[bAtsIndex] & PHPAL_I14443P4A_SAM_X_ATS_TC1_CID_SUPPORT))
            {
                pDataParams->bCidSupported = 0;
                pDataParams->bCid = 0;
            }
        }
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

phStatus_t phpalI14443p4a_Sam_X_GetProtocolParams(phpalI14443p4a_Sam_X_DataParams_t * pDataParams,
    uint8_t * pCidEnabled, uint8_t * pCid, uint8_t * pNadSupported, uint8_t * pFwi, uint8_t * pFsdi,
    uint8_t * pFsci)
{
    *pCidEnabled    = pDataParams->bCidSupported;
    *pCid           = pDataParams->bCid;
    *pNadSupported  = pDataParams->bNadSupported;
    *pFwi           = pDataParams->bFwi;
    *pFsdi          = 8U;
    *pFsci          = pDataParams->bFsci;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4A);
}

#endif /* NXPBUILD__PHPAL_I14443P4A_SAM_X */
