/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ISO14443-4 Component of Basic Function Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7230 $
 * $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
 */

#include <ph_Status.h>
#include <phhalHw.h>
#include <phpalI14443p4.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHPAL_I14443P4_SAM_X

#include "phpalI14443p4_Sam_X.h"
#include "../../../phhalHw/src/Sam/Commands/17_ISO14443_4/phhalHw_Sam_Cmd_ISO14443_4.h"

phStatus_t phpalI14443p4_Sam_X_Init(phpalI14443p4_Sam_X_DataParams_t * pDataParams, uint16_t wSizeOfDataParams,
    phhalHw_Sam_DataParams_t * pHalDataParams)
{
    if(sizeof(phpalI14443p4_Sam_X_DataParams_t) != wSizeOfDataParams)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_DATA_PARAMS, PH_COMP_PAL_ISO14443P4);
    }
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_PAL_ISO14443P4);
    PH_ASSERT_NULL_PARAM(pHalDataParams, PH_COMP_PAL_ISO14443P4);

    /* Init private data */
    pDataParams->wId = PH_COMP_PAL_ISO14443P4 | PHPAL_I14443P4_SAM_X_ID;
    pDataParams->pHalDataParams = pHalDataParams;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sam_X_SetProtocol(phpalI14443p4_Sam_X_DataParams_t * pDataParams, uint8_t bCidEnable, uint8_t bCid,
    uint8_t bNadEnable, uint8_t bNad, uint8_t bFwi, uint8_t bFsdi, uint8_t bFsci)
{
    /* satisfy compiler */
    if(pDataParams || bCidEnable || bCid || bNadEnable || bNad || bFwi || bFsdi || bFsci);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sam_X_ResetProtocol(phpalI14443p4_Sam_X_DataParams_t * pDataParams)
{
    /* satisfy compiler */
    if(pDataParams);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sam_X_Deselect(phpalI14443p4_Sam_X_DataParams_t * pDataParams)
{
    return phhalHw_Sam_Cmd_X_ISO14443_4_Deselect(pDataParams->pHalDataParams,
        PHHAL_HW_SAM_CMD_ISO14443_4_DESELECT_DO_NOT_FREE_CID);
}

phStatus_t phpalI14443p4_Sam_X_PresCheck(phpalI14443p4_Sam_X_DataParams_t * pDataParams)
{
    return phhalHw_Sam_Cmd_X_ISO14443_4_PresenceCheck(pDataParams->pHalDataParams);
}

phStatus_t phpalI14443p4_Sam_X_Exchange(phpalI14443p4_Sam_X_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pTxBuffer,
    uint16_t wTxLength, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus;
    phStatus_t  PH_MEMLOC_REM wStatus1;
    uint8_t *   PH_MEMLOC_REM pRxBufferTmp;
    uint16_t    PH_MEMLOC_REM wRxLengthTmp;
    uint16_t    PH_MEMLOC_REM wTxLengthTmp;
    uint16_t    PH_MEMLOC_REM wOptionSam;
    uint16_t    PH_MEMLOC_REM wRxBufferSize;
    uint16_t    PH_MEMLOC_REM wRxBufferStartPos;

	uint16_t	PH_MEMLOC_REM wRespLen;
	uint8_t *	PH_MEMLOC_REM pResponse;

	/* Check if caller has provided valid RxBuffer */
	if (ppResponse == NULL)
    {
        ppResponse = &pResponse;
	}
	if (pRespLen == NULL)
	{
        pRespLen = &wRespLen;
    }

    /* Reset receive length */
    *pRespLen = 0;

    /* Tx-Chaining loop */
    if (!(wOption & PH_EXCHANGE_RXCHAINING))
    {
        do
        {
            /* Check if chaining is neccessary */
            if (wOption & PH_EXCHANGE_TXCHAINING)
            {
                wTxLengthTmp = wTxLength;
                wOptionSam = PH_EXCHANGE_TXCHAINING;
            }
            else if (wTxLength > 0xFFU)
            {
                wTxLengthTmp = 0xFFU;
                wOptionSam = PH_EXCHANGE_TXCHAINING;
            }
            else
            {
                wTxLengthTmp = wTxLength;
                wOptionSam = PH_EXCHANGE_DEFAULT;
            }

            /* Perform actual exchange */
            wStatus = phhalHw_Sam_Cmd_X_ISO14443_4_Exchange(
                pDataParams->pHalDataParams,
                wOptionSam,
                pTxBuffer,
                (uint8_t)wTxLengthTmp,
                &pRxBufferTmp,
                &wRxLengthTmp);

            /* Remove already transmitted bytes */
            pTxBuffer += wTxLengthTmp;
            wTxLength = wTxLength - wTxLengthTmp;
        }while (wTxLength > 0);

        /* Tx-Chaining was intended -> return successful */
        if (wOption & PH_EXCHANGE_TXCHAINING)
        {
            *ppResponse = NULL;
            *pRespLen = 0;
            return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_PAL_ISO14443P4);
        }
        /* Bail out if there's nothing to chain */
        else
        {
            if ((wStatus & PH_ERR_MASK) != PH_ERR_SUCCESS_CHAINING)
            {
                /* Return received contents */
                *ppResponse = pRxBufferTmp;
                *pRespLen = wRxLengthTmp;
                return wStatus;
            }
        }
    }

    /* Retrieve receive buffer size */
    PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_RXBUFFER_BUFSIZE,
        &wRxBufferSize));

    /* Retrieve receive buffer start position */
    PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_GetConfig(
        pDataParams->pHalDataParams,
        PHHAL_HW_CONFIG_RXBUFFER_STARTPOS,
        &wRxBufferStartPos));

    /* Rx-Chaining Loop */
    do
    {
        /* Rx-Chaining: This is the first chain, so do not check buffer size */
        if (wOption & PH_EXCHANGE_RXCHAINING)
        {
            wOption &= (uint16_t)~(uint16_t)PH_EXCHANGE_RXCHAINING;
        }
        else
        {
            /* Calculate free space within the receive buffer */
            wRxBufferSize = wRxBufferSize - wRxBufferStartPos;

            /* Bail out if we do not have enough buffer space for next exchange */
            if (wRxBufferSize < PHHAL_HW_SAM_ISO7816_EXCHANGE_RESPONSE_MAX)
            {
                return PH_ADD_COMPCODE(PH_ERR_SUCCESS_CHAINING, PH_COMP_PAL_ISO14443P4);
            }

            /* Update receive buffer start position */
            PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_SetConfig(
                pDataParams->pHalDataParams,
                PHHAL_HW_CONFIG_RXBUFFER_STARTPOS,
                wRxBufferStartPos));
        }

        /* Exchange with lower layer */
        wStatus = phhalHw_Sam_Cmd_X_ISO14443_4_Exchange(
            pDataParams->pHalDataParams,
            PH_EXCHANGE_DEFAULT,
            NULL,
            0,
            ppResponse,
            pRespLen);

        /* Update receive buffer start position */
        wRxBufferStartPos = *pRespLen;
    }
    while ((wStatus & PH_ERR_MASK) == PH_ERR_SUCCESS_CHAINING);

    return wStatus;
}

phStatus_t phpalI14443p4_Sam_X_SetConfig(phpalI14443p4_Sam_X_DataParams_t * pDataParams, uint16_t wConfig, uint16_t wValue)
{
    /* satisfy compiler */
    if (pDataParams || wConfig || wValue);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
}

phStatus_t phpalI14443p4_Sam_X_GetConfig(phpalI14443p4_Sam_X_DataParams_t * pDataParams, uint16_t wConfig, uint16_t * pValue)
{
    /* satisfy compiler */
    if (pDataParams || wConfig || pValue);

    return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_COMMAND, PH_COMP_PAL_ISO14443P4);
}

#endif /* NXPBUILD__PHPAL_I14443P4_SAM_X */
