/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * MP300 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#ifdef NXPBUILD__PHHAL_HW_CONTACT_SMARTWARE

#include "phhalHwContact_Smartware.h"
#include "phhalHwContact_Smartware_Int.h"
#include <phTools.h>
#include <phToolsAtrParser.h>
#include "ex\CARD\card.h"

#include <stdio.h>              /* PRQA S 5124 */

phStatus_t phhalHwContact_Smartware_Int_ConvertError(
                                                     uint32_t status
                                                     )
{
    switch (status)
    {
    case 0:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
}

phStatus_t phhalHwContact_Smartware_Int_AddToIntBufferHexString(
                                                                phhalHwContact_Smartware_DataParams_t * pDataParams,
                                                                uint8_t * pData,
                                                                uint16_t wLen
                                                                )
{
    uint16_t PH_MEMLOC_REM wIndex;
    uint16_t PH_MEMLOC_REM wNumCharPrint;

    /* Buffer overflow check */
    if ((uint32_t)(wLen * 2) + 1 > pDataParams->dwIntBufferSize-pDataParams->dwIntBufferLen)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    if(wLen > 0)
    {
        pDataParams->pIntBuffer[pDataParams->dwIntBufferLen] = ' ';
        pDataParams->dwIntBufferLen++;
    }

    for ( wIndex = 0; wIndex < wLen; wIndex++ )
    {
        wNumCharPrint = (uint16_t)sprintf( (char*)&pDataParams->pIntBuffer[pDataParams->dwIntBufferLen], "%02X", pData[wIndex]);
        if (wNumCharPrint != 2)
        {
            return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
        }
        pDataParams->dwIntBufferLen += 2;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_Int_ApplyConfiguration(
                                                           phhalHwContact_Smartware_DataParams_t * pDataParams,
                                                           uint16_t  wConfig,
                                                           uint32_t  dwValue
                                                           )
{
    phStatus_t statusTmp;
    card_CfgXType config;
    uint32_t fdValue;
    uint16_t fValue;
    uint16_t dValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, &fdValue));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(fdValue, &fValue, &dValue));

    /* Struct Size has to be initialized */
    config.StructSize = sizeof(card_CfgXType);

    /* Set the specific value as we want */
    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
        /*  in Card Clock cycles */
        config.Mask = card_CfgxTSWWT;
        config.TSWWT = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
        /* in ETU counts */
        config.Mask = card_CfgxWWT;
        config.WWT = dwValue / (fValue / dValue);
        break;

    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
        /* in Card Clock cycles */
        config.Mask = card_CfgxCD | card_CfgxCDU;
        config.CDU = 1;
        config.CD = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
        /* in ETU counts */
        config.Mask = card_CfgxCWT;
        config.CWT = dwValue / (fValue / dValue);
        break;

    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        /* in ETU counts */
        config.Mask = card_CfgxBWT;
        config.BWT = dwValue / (fValue / dValue);
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    /* Apply configuration */
    PH_CHECK_SUCCESS_FCT(statusTmp, (phStatus_t)pDataParams->pBalDataParams->pfCardSetGetConfig(pDataParams->pBalDataParams->CardBuffer, &config));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Smartware_Int_GetConfiguration(
                                                         phhalHwContact_Smartware_DataParams_t * pDataParams,
                                                         uint16_t   wConfig,
                                                         uint32_t * pValue
                                                         )
{
    phStatus_t statusTmp;
    card_CfgXType config;
    uint32_t fdValue;
    uint16_t fValue;
    uint16_t dValue;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, &fdValue));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(fdValue, &fValue, &dValue));

    /* Struct Size has to be initialized and Mas=0 means read config */
    config.StructSize = sizeof(card_CfgXType);
    config.Mask = 0;

    /* Get the current configuration */
    PH_CHECK_SUCCESS_FCT(statusTmp, (phStatus_t)pDataParams->pBalDataParams->pfCardSetGetConfig(pDataParams->pBalDataParams->CardBuffer, &config));

    /* Set the specific value as we want */
    switch (wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR:
        /*  in Card Clock cycles */
        *pValue = config.TSWWT;
        break;

    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
        /* in ETU counts */
        *pValue = config.WWT * (fValue / dValue);
        break;

    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
        /* in Card Clock cycles */
        if(config.CDU == 0)
        {
            *pValue = config.CD * (fValue / dValue);
        }
        else
        {
            *pValue = config.CD;
        }
        break;

    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
        /* in ETU counts */
        *pValue = config.CWT * (fValue / dValue);
        break;

    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        /* in ETU counts */
        *pValue = config.BWT * (fValue / dValue);
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_UNSUPPORTED_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_SMARTWARE */
