/***********************************************************************
                                  TCUR
                             Curseur de texte
 ***********************************************************************
 On decrit un curseur logique sur du texte pour etablir une interface
 homogene entre les supports de texte (fichiers, buffer, ... ) et 
 les consommateurs et producteurs de texte (parseurs, printeurs, ...).
 ***********************************************************************
 Un texte est constitue d'une suite de lignes eventuellement vide, 
 termine par une string eventuellement vide.
  
 Le curseur contient un pointeur physique sur le texte a lire ou une 
 zone a ecrire.
 Le curseur contient les fonctions permettant d'obtenir et de fournir 
 des lignes entieres de texte. 
 Le curseur contient e'ventuellement une fonction qui permet l'emission des 
 eventuels messages d'erreur.
  
 Le curseur contient aussi des informations sur le support du texte
 manipule (contexte).
 
 ***********************************************************************
 06/04/2006 - Marc LEGOUX C6R22b: Add PUTL
 28/12/2005 - Marc LEGOUX C6R22 : Add ModeItemNoMore
 03/09/2001 - Marc LEGOUX C6R07 : Add tcur_ctype.mode and .lineoffs0
 28/08/2001 - Marc LEGOUX C6R07 : use char_0type
 09/05/2001 - Marc LEGOUX C6R07 : Add Line field and tcur_ptoffget()
 01/04/2000 - Marc LEGOUX C6R03 : Use symbols in macros (=> modif in svcur and svcurl). 
 01/05/1996 - Marc LEGOUX C4R01 : Update from old version
 ***********************************************************************/
#ifndef tcur_h
#define tcur_h

#include <err.h>
#include <char.h>
/*-------------------------- ERRORS ------------------------------------*/
#ifndef tcur__err
extern
err_type tcur__err;
#endif
#define tcur__badrequest	(tcur__err + 1)
#define tcur__eot		(tcur__err + 2)
#define tcur__AttrOk		(tcur__err + 3)

/*-------------------------- TYPES -------------------------------------*/
/*----------------------------------------------------------------------*
 Fonction generique d'obtention/fourniture de ligne
 LECTURE:
 La fonction retourne dans cur un pointeur sur la string formant 
 la ligne suivante (pas de LF) ou sur la string terminale du texte du support.
 Le status "fin de texte" est obtenu en cas d'appel avec un curseur sur
 la string terminale.
 ECRITURE:
 La fonction transfert le texte sous forme d'une string ou d'une ligne, 
 et retourne dans cur un pointeur sur une nouvelle zone inscriptible, 
 avec la taille de cette zone. 
 Un status indique eventuellement l'impossibilite d'autres ecritures.
 *----------------------------------------------------------------------*/
typedef
  err_type (* tcur_ftype)();

/* Abstract operation codes */
#define tcur_NEXTL	0	/* Get next line or NULL */
#define tcur_SAVECURL	1	/* Save cur into cur2 (same line) */
#define tcur_RESTCUR	2	/* Restore cur from cur2 */
#define tcur_ERR	3	/* Raise error (cur2 == code) */
#define tcur_LBPUTS	4	/* Put string from line buffer */
#define tcur_LBPUTL	5	/* Put line from line buffer */
#define tcur_SAVECUR	6	/* Save cur into cur2 (evt different lines) */
#define tcur_PUTS	7	/* Put string from (char *)cur2 (LF!=new line)*/
#define tcur_SAVECURN	8	/* Save cur into new allocated cur2 */
#define tcur_FREECURN	9	/* Free new allocated cur2 */
#define tcur_PUTCOV	10	/* Put char (char_type)cur2 (overwrite)*/
#define tcur_GETC	11	/* Get char to (char_type *)cur2 */
#define tcur_PUTL	12	/* Put line from (char *)cur2 */

#define tcur_ATTRSET	0x10	/* 1x: Set attribute x from cur2 to cur */ 
/*
PARAMS : 
     int opcode; 	Opcode
     tcur_ctype * cur;  (IN) ou (OUT) Pointeur sur Curseur
     tcur_ctype * cur2; (OUT) Pointeur sur Curseur ou (IN) NULL
                                       ou code d'erreur
RETURNS: err_type
     0 :    cur->pt = Pointeur sur string ou cur restaure'.
     tcur__eot : Fin de texte. 
     ... : autres.
*/
 
/* -----> Type de la fonction de message d'erreur */
typedef
  err_type (*tcur_fertype)();
 
typedef
  char * tcur_ctx_type;

/*----------------------------------------------------------------------*
                        CURSEUR GENERIQUE
 Selon l'utilisation du curseur en ecriture ou en lecture la
 signification des champs est differente:
 LECTURE:
         pt : Pointeur sur chaine a lire terminee par char_eos.
         last : !=0 si derniere ligne (string terminale).
 ECRITURE:
         pt : Pointeur sur zone inscriptible.
         last : Taille zone inscriptible. 
 Dans un curseur invalide, pt == NULL.
 *----------------------------------------------------------------------*/
typedef
  struct  
  { 
    char_0type * pt;	/* Pointeur */
    int last;		/* Derniere ligne ou taille restante */
    tcur_ftype fline;	/* Fonction de ligne */
    tcur_fertype ferror;/* Fonction de messages d'erreurs ou NULL */
    tcur_ctx_type ctx1;	/* Contexte du parsing (selon implantation) */
    tcur_ctx_type ctx2;	/* Contexte du parsing (selon implantation) */
    tcur_ctx_type ctx3;	/* Contexte du parsing (selon implantation) */
    tcur_ctx_type ctx4;	/* Contexte du parsing (selon implantation) */
    unsigned linenum;	/* Numero ligne courante 1.. ou 0 */
    long     lineoffs;	/* Offset dans le texte de .line[0] ou -1 */
    char_0type * line;	/* Ligne courante ou NULL */
    ulong	mode;	/* Mode */
    long     lineoffs0;	/* Offset dans le texte de la ligne courante ou -1 */
  } tcur_ctype; 

/* Modes (ORed) : Private */
#define tcur_ModeLinePart	1	/* Part of line  */

/* Modes (ORed) : User */
#define tcur_ModeComment	0x10000	/* Inside comment 1 */
#define tcur_ModeComment2	0x20000	/* Inside comment 2 */
#define tcur_ModeAttrOk		0x40000	/* Stop if attr is OK */
#define tcur_ModeItemNoMore	0x80000	// Last item already parsed

/*---------------------------------------------------------------------*
                Macros de manipulation du curseur (par pointeur)
 *---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*
                               Get pt
 *---------------------------------------------------------------------*/
#define tcur_gpt(cur) ((cur)->pt)
/*---------------------------------------------------------------------*
                               Set pt
 *---------------------------------------------------------------------*/
#define tcur_spt(cur, ptn) ((cur)->pt = (ptn))
 
/*---------------------------------------------------------------------*
                               Get pt offset
 *---------------------------------------------------------------------*/
#define tcur_ptoffget(cur) ((cur)->line ? (cur)->pt-(cur)->line : -1)
 
/*---------------------------------------------------------------------*
                               Lecture nouvelle ligne
 *---------------------------------------------------------------------*
 RETURNS: !=0 si fin de texte.
 *---------------------------------------------------------------------*/
#define tcur_newl(cur) ((*(cur)->fline)(tcur_NEXTL,(cur),0))
 
/*---------------------------------------------------------------------*
                               Preserve curseur
 tcur_type * cur, *cur2, ** pcur2
 *---------------------------------------------------------------------*/
#define tcur_svcur(cur,cur2) ((*(cur)->fline)(tcur_SAVECUR  ,(cur),(cur2)))
#define tcur_svcurl(cur,cur2) ((*(cur)->fline)(tcur_SAVECURL,(cur),(cur2)))
#define tcur_svcurn(cur,pcur2) ((*(cur)->fline)(tcur_SAVECURN,(cur),(tcur_ctype *)(pcur2)))

#define tcur_freecurn(cur2) ((*(cur2)->fline)(tcur_FREECURN,0,(cur2)))
 
/*---------------------------------------------------------------------*
                               Restaure curseur
 *---------------------------------------------------------------------*/
#define tcur_rtcur(cur,cur2) ((*(cur)->fline)(tcur_RESTCUR,(cur),(cur2)))
 
/*---------------------------------------------------------------------*
                               Erreur
 *---------------------------------------------------------------------*/
#define tcur_err(cur,erno)((*(cur)->fline)(tcur_ERR,(cur),(tcur_ctype *)(erno)))
 
/*---------------------------------------------------------------------*
                              String terminale ?
 *---------------------------------------------------------------------*/
#define tcur_lastl(cur) ((cur)->last)
 
/*---------------------------------------------------------------------*
                              Fin de texte ?
 *---------------------------------------------------------------------*/
#define tcur_eot(cur) ((cur)->last && *((cur)->pt)==char_eos)
 
/*---------------------------------------------------------------------*
                              Ecriture string
                              from line buffer
 *---------------------------------------------------------------------*/
#define tcur_put(cur) ((*(cur)->fline)(tcur_LBPUTS,(cur),0))
 
/*---------------------------------------------------------------------*
                              Ecriture ligne
 *---------------------------------------------------------------------*/
#define tcur_putl(cur,st) ((*(cur)->fline)(tcur_PUTL,(cur),(st)))
/*---------------------------------------------------------------------*
                              Ecriture ligne
                              from line buffer
 *---------------------------------------------------------------------*/
#define tcur_putlb(cur) ((*(cur)->fline)(tcur_LBPUTL,(cur),0))
/*---------------------------------------------------------------------*
                              Ecriture string
 *---------------------------------------------------------------------*/
#define tcur_puts(cur,st) ((*(cur)->fline)(tcur_PUTS,(cur),(st)))

/*---------------------------------------------------------------------*
                              Ecriture char (overwrite)
 *---------------------------------------------------------------------*/
#define tcur_putcov(cur,ch) ((*(cur)->fline)(tcur_PUTCOV,(cur),(tcur_ctype *)(ch)))
/*---------------------------------------------------------------------*
                              Get char (overwrite)
 *---------------------------------------------------------------------*/
#define tcur_getc(cur,pch) ((*(cur)->fline)(tcur_GETC,(cur),(tcur_ctype *)(pch)))
/*----------------------------------------------------------------------*
				Set attribute
 Stop if tcur_ModeAttrOk and returns tcur__AttrOk
 *----------------------------------------------------------------------*/
#define tcur_AttrSet(cur,cur2,Attr) ((*(cur)->fline)(tcur_ATTRSET+(Attr),(cur),(cur2)))

#endif
