/************************************************************************
                        SOCKET Interface
                        BSD-Like 
                        BSD 4.4 model 
 **********************************************************************
 Copyright (c) 1982, 1986, 1988, 1990, 1993, 1994
 The Regents of the University of California. All Rights reserved.
 **********************************************************************
 Copyright (c) 2003, SMARTWARE. All Rights Reserved.
 ************************************************************************
 15/07/2003 - Erwan C     - V2r03 : Add (typedef char* caddr_t;)
 30/06/2003 - Marc LEGOUX - V2r02 : Creation
 ************************************************************************/
#ifndef sockbsd_h
#define sockbsd_h

#include <config.h>

#ifdef __cplusplus
extern "C" {
#endif

#ifndef u_char 
#define u_char unsigned char
#endif
#ifndef u_int
#define u_int unsigned int
#endif
#ifndef u_short
#define u_short unsigned short
#endif
#ifndef u_long
#define u_long unsigned long
#endif

/*----------------------------------------------------------------------*
			Error Codes
 *----------------------------------------------------------------------*/
#define sock__OK              0   // no error
#define sock__2BIG            -1  // argument list too long 
#define sock__ACCES           -2  // permission denied 
#define sock__ADDRINUSE       -3  // address already in use 
#define sock__ADDRNOTAVAIL    -4  // address not available from this host 
#define sock__AFNOSUPPORT     -5  // address family not supported 
#define sock__AGAIN           -6  // resource temporarily unavailable 
#define sock__ALREADY         -7  // operation already in progress 
#define sock__AUTH            -8  // authentication error 
#define sock__BADF            -9  // bad descriptor 
#define sock__BUSY            -10 // resource busy 
#define sock__CHILD           -11 // no child process 
#define sock__CONNABORTED     -12 // connection aborted by user 
#define sock__CONNREFUSED     -13 // connection refused 
#define sock__CONNRESET       -14 // connection reseted by peer 
#define sock__DEADLK          -15 // resource deadlock avoided 
#define sock__DESTADDRREQ     -16 // destination address required 
#define sock__DOM             -17 // value out of range 
#define sock__EXIST           -18 // file exists 
#define sock__FAULT           -19 // bad address 
#define sock__FBIG            -20 // file too big 
#define sock__HOSTDOWN        -21 // host down 
#define sock__HOSTUNREACH     -22 // host unreachable 
#define sock__IDRM            -23 // identifier removed 
#define sock__INPROGRESS      -24 // operation now in progress 
#define sock__INTR            -25 // interrupted function call 
#define sock__INVAL           -26 // invalid argument 
#define sock__IO              -27 // input/output error 
#define sock__ISCONN          -28 // already connected 
#define sock__ISDIR           -29 // is a directory 
#define sock__LOOP            -30 // too many levels of symbolic links 
#define sock__MFILE           -31 // out of descriptors 
#define sock__MLINK           -32 // too many links 
#define sock__MSGSIZE         -33 // message too big 
#define sock__NEEDAUTH        -34 // need authenticator 
#define sock__NETDOWN         -35 // network down 
#define sock__NETRESET        -36 // network dropped connection on reset 
#define sock__NETUNREACH      -37 // network unreachable 
#define sock__NFILE           -38 // too many open files in system 
#define sock__NOBUFS          -39 // out of resources 
#define sock__NODEV           -40 // device not found 
#define sock__NOENT           -41 // no such file or directory 
#define sock__NOEXEC          -42 // exec format error 
#define sock__NOLCK           -43 // no locks available 
#define sock__NOMEM           -44 // cannot allocate memory 
#define sock__NOMSG           -45 // no message of desired type 
#define sock__NOPROTOOPT      -46 // not a valid protocol option 
#define sock__NOSPC           -47 // not enougth space to create object 
#define sock__NOSYS           -48 // function not implemented 
#define sock__NOTBLK          -49 // block device required 
#define sock__NOTCONN         -50 // socket not connected 
#define sock__NOTDIR          -51 // not a directory 
#define sock__NOTEMPTY        -52 // directory not empty 
#define sock__NOTSOCK         -53 // not a socket descriptor 
#define sock__NOTTY           -54 // not a tty 
#define sock__NXIO            -55 // device not configured 
#define sock__OPNOTSUPP       -56 // operation not supported 
#define sock__PERM            -57 // operation not permitted 
#define sock__PFNOSUPPORT     -58 // protocol family not supported 
#define sock__PIPE            -59 // pipe broken 
#define sock__PROTONOSUPPORT  -60 // protocol not supported 
#define sock__PROTOTYPE       -61 // wrong protocol type 
#define sock__RANGE           -62 // result too large 
#define sock__ROFS            -63 // read-only file system 
#define sock__SHUTDOWN        -64 // can't send after shutdown 
#define sock__SOCKTNOSUPPORT  -65 // socket type not supported 
#define sock__SPIPE           -66 // illegal seek 
#define sock__SRCH            -67 // no such process 
#define sock__TIMEDOUT        -68 // connection timed out 
#define sock__TOOMANYREFS     -69 // too many references 
#define sock__WOULDBLOCK      sock__AGAIN // operation would block 
#define sock__XDEV            -70 // cross-device link 
#define sock__NAMETOOLONG     -71 // component of the path name too long 

/*----------------------------------------------------------------------*
			BSD Symbols
 *----------------------------------------------------------------------*/
// protocols 
#define IPPROTO_IP          0
#define IPPROTO_ICMP        1
#define IPPROTO_IGMP        2
#define IPPROTO_GGP         3
#define IPPROTO_TCP         6
#define IPPROTO_EGP         8
#define IPPROTO_PUP         12
#define IPPROTO_UDP         17
#define IPPROTO_IDP         22
#define IPPROTO_TP          29
#define IPPROTO_EON         80
#define IPPROTO_RAW         255
#define IPPROTO_MAX         256

// udp and tcp reserved ports 
#define IPPORT_RESERVED         1024
#define IPPORT_USERRESERVED     5000

#define IPPORT_ECHO             7
#define IPPORT_DISCARD          9
#define IPPORT_SYSTAT           11
#define IPPORT_DAYTIME          13
#define IPPORT_NETSTAT          15
#define IPPORT_FTP              21
#define IPPORT_TELNET           23
#define IPPORT_SMTP             25
#define IPPORT_TIMESERVER       37
#define IPPORT_NAMESERVER       42
#define IPPORT_WHOIS            43
#define IPPORT_MTP              57
#define IPPORT_BOOTP            67
#define IPPORT_BOOTPC           68
#define IPPORT_TFTP             69
#define IPPORT_RJE              77
#define IPPORT_FINGER           79
#define IPPORT_TTYLINK          87
#define IPPORT_SUPDUP           95

// address family 
#define AF_UNSPEC           0
#define AF_INET             2

// protocol family (== address family) 
#define PF_UNSPEC           0
#define PF_INET             2

// types of socket 
#define SOCK_STREAM         1
#define SOCK_DGRAM          2
#define SOCK_RAW            3

// level of option 
#define SOL_SOCKET          0xffffU
#define SOL_IP              IPPROTO_IP
#define SOL_TCP             IPPROTO_TCP
#define SOL_UDP             IPPROTO_UDP

// socket level options 
#define SO_SNDBUF        0x1001    // send buffer size 
#define SO_RCVBUF        0x1002    // receive buffer size 
#define SO_SNDLOWAT      0x1003    // send low-water mark 
#define SO_RCVLOWAT      0x1004    // receive low-water mark 
#define SO_SNDTIMEO      0x1005    // send timeout 
#define SO_RCVTIMEO      0x1006    // receive timeout 
#define SO_ERROR         0x1007    // get error status and clear (ro) 
#define SO_TYPE          0x1008    // get socket type (ro) 

#define SO_DEBUG             0x0001  // turn on debugging info recording 
#define SO_ACCEPTCONN        0x0002  // socket has had listen() 
#define SO_REUSEADDR         0x0004  // allow local address reuse 
#define SO_KEEPALIVE         0x0008  // keep connections alive 
#define SO_DONTROUTE         0x0010  // just use interface addresses 
#define SO_BROADCAST         0x0020  // permit sending of broadcast msgs 
#define SO_RECVDSTADDR       0x0040  // receive dest addr in control data 
#define SO_LINGER            0x0080  // linger on close if data present 
#define SO_OOBINLINE         0x0100  // leave received OOB data in line 
#define SO_REUSEPORT         0x0200  // allow local address & port reuse 
#define SO_HDRINCL           0x1000  // include IP header (raw sockets) 
#define SO_MCASTLOOP         0x2000  // enable loopback of outgoing mcasts 
#define SO_DONTFRAG          0x4000  // set DF flag on outgoing datagrams 

// ip level options 
// IP level options 
#define IP_HDRINCL             0x0002  // header is included with data (raw) 
#define IP_TOS                 0x0003  // IP type of service and precedence 
#define IP_TTL                 0x0004  // IP time to live 
#define IP_MULTICAST_IF        0x0005  // Interface for outgoing multicasts 
#define IP_MULTICAST_TTL       0x0006  // time to live for outgoing multicasts 
#define IP_MULTICAST_IFP       0x0007  // interface for outgoing multicasts 
#define IP_ADD_MEMBERSHIP      0x0008  // add a mcast group membership 
#define IP_ADD_MEMBERSHIP_IFP  0x0009
#define IP_DROP_MEMBERSHIP     0x000a  // drop a mcast group membership 
#define IP_DROP_MEMBERSHIP_IFP 0x000b
#define IP_MULTICAST_LOOP      0x000c  // enable loopback of outgoing mcasts 
#define IP_DONTFRAG            0x000d  // set DF bit on outgoing datagrams 
#define IP_RECVDSTADDR         0x000e  // recv dest address in control data 

// recv/send flags 
#define MSG_DONTWAIT        0x0001
#define MSG_WAITALL         0x0002
#define MSG_OOB             0x0004
#define MSG_PEEK            0x0008

#define MSG_DONTROUTE     0x0010    // bypass routing table 
#define MSG_EOR           0x0020    // end of record 
#define MSG_TRUNC         0x0040    // rcvd data truncated 
#define MSG_CTRUNC        0x0080    // rcvd control data truncated 
#define MSG_BCAST         0x0100    // rcvd link-layer bcast 
#define MSG_MCAST         0x0200    // rcvd link-layer mcast 

// TCP
#define TCP_NODELAY      0x0001     // no delay to send coalesce packets 
#define TCP_MAXSEG       0x0002     // set maximum segment size 
#define TCP_SACK_DISABLE 0x0005

// commands for fcntlsocket() 
#define F_GETFL         3		// get flags 
#define F_SETFL         4		// set flags 
#define O_NONBLOCK      0x0004		// set/clear nonblocking io 

// commands for ioctlsocket() 
#define IOC_OUT         0x40000000L     // copy out parameters 
#define IOC_IN          0x80000000L     // copy in parameters 
#define IOCPARM_MASK    0x1fff          // parameter length, at most 13 bits 

#define _IOC(inout,group,num,len) \
        (inout | (((long)len & IOCPARM_MASK) << 16) | ((group) << 8) | (num))

#define _IOR(g,n,t)     _IOC(IOC_OUT,   (g), (n), sizeof(t))
#define _IOW(g,n,t)     _IOC(IOC_IN,    (g), (n), sizeof(t))

#define FIONBIO         _IOW('f', 126, int)     // set/clear nonblocking io 
#define FIONREAD        _IOR('f', 127, int)     // get # bytes to read 
#define SIOCATMARK      _IOR('s',  7, int)      // at oob mark? 

/*----------------------------------------------------------------------*
			BSD Structures
 *----------------------------------------------------------------------*/
// generic socket address 
struct sockaddr {
    u_char     sa_len;         // total length 
    u_char     sa_family;      // address family 
    u_char     sa_data[14];    // address value 
};

// socket address in internet domain 
struct in_addr {
    u_int      s_addr;         // 32 bits IP address, net byte order 
};

struct sockaddr_in {
    u_char     sin_len;        // total length (16) 
    u_char     sin_family;     // AF_INET 
    u_short    sin_port;       // 16 bits port number, net byte order 
    struct in_addr sin_addr;   // 32 bits IP address 
    u_char     sin_zero[8];    // unused 
};

// structure used by readv/writev/recvmsg/writemsg 
struct iovec {
    void   *iov_base;
    int     iov_len;
};

// used by SO_SNDTIMEO/SO_RCVTIMEO socket options 
struct timeval {
    long    tv_sec;
    long    tv_usec;
};

// used by SO_LINGER socket option 
struct linger {
    int     l_onoff;
    int     l_linger;
};

// used by IP_ADD_MEMBERSHIP/IP_DROP_MEMBERSHIP socket options 
struct ip_mreq {
    struct in_addr  imr_multiaddr;  // multicast group address 
    struct in_addr  imr_interface;  // local interface address 
    void        *imr_ifp;        // pointer to interface 
};

typedef char* caddr_t;

// message structure for sendmsg/recvmsg 
struct msghdr {
    caddr_t     msg_name;       // optional address 
    u_int       msg_namelen;    // size of address 
    struct iovec *msg_iov;      // scatter/gather array 
    u_int       msg_iovlen;     // # elements in msg_iov 
    caddr_t     msg_control;    // ancillary data 
    u_int       msg_controllen; // ancillary data buffer len 
    int         msg_flags;      // flags on received message 
};

struct cmsghdr {
    u_int       cmsg_len;       // data byte count, including hdr 
    int         cmsg_level;     // originating protocol 
    int         cmsg_type;      // protocol-specific type 
                                // followed by	u_char  cmsg_data[]; 
};

// definitions for selectsocket() 
#define SO_FD_MAX	32		// Maximum number of descriptors
typedef struct {
    u_long fds[(SO_FD_MAX+31)/32];
} fd_set;

#define FD_ZERO( fdset) \
  { int i; for( i=0; i<((SO_FD_MAX+31)/32); i++) (fdset)->fds[i] = 0; }
#define FD_SET( fd, fdset) \
  ((fdset)->fds[(fd)>>5] |= 1UL<<((fd) & 0x1f))
#define FD_CLR( fd, fdset) \
  ((fdset)->fds[(fd)>>5] &= ~(1UL<<((fd) & 0x1f)))
#define FD_ISSET( fd, fdset) \
  (((fdset)->fds[(fd)>>5] & (1UL<<((fd) & 0x1f))) != 0UL)

// max number of fds in one select() 
#define SO_SELECT_FD_MAX    (SO_FD_MAX/2)

/*----------------------------------------------------------------------*
                        Get last error code
 *----------------------------------------------------------------------*/
extern
int 	sock_GetErrno(void);
/*----------------------------------------------------------------------*
                        BSD Functions
 *----------------------------------------------------------------------*/
extern
int	accept(int, struct sockaddr *, int *);

extern
int	bind(int, const struct sockaddr *, int);

extern
int	closesocket(int);

extern
int	connect(int, const struct sockaddr *, int);

extern
int	fcntlsocket(int s, int cmd, ...);

extern
int	getpeername(int, struct sockaddr *, int *);

extern
int	getsockname(int, struct sockaddr *, int *);

extern
int	getsockopt(int, int, int, void *, int *);

#if config_BENDIAN
#define htonl(a)   (a)
#define htons(a)   (a)
#define ntohl(a)   (a)
#define ntohs(a)   (a)
#else
u_long	htonl(u_long a);
u_short	htons(u_short a);
u_long	ntohl(u_long a);
u_short	ntohs(u_short a);
#endif

extern
int	ioctlsocket(int s, long cmd, int *arg);

extern
int	listen(int, int);

extern
int	readsocket(int, void *, int);

extern
int	readvsocket(int, const struct iovec *, int);

extern
int	recv(int, void *, int, int);

extern
int	recvfrom(int, void *, int, int, struct sockaddr *, int *);

extern
int	recvmsg(int, struct msghdr *, int);

extern
int	selectsocket(int fdmax, fd_set *readfds, fd_set *writefds,
                  fd_set *exceptfds, struct timeval *timeout);

extern
int	send(int, const void *, int, int);

extern
int	sendto(int, const void *, int, int, const struct sockaddr *, int);

extern
int	sendmsg(int, const struct msghdr *, int);

extern
int	setsockopt(int, int, int, const void *, int);

extern
int	shutdown(int, int);

extern
int	socket(int, int, int);

extern
int	writesocket(int, void *, int);

extern
int	writevsocket(int, const struct iovec *, int);

// internet address manipulation routines 
extern
u_long  inet_addr(const char *);

extern
int	inet_aton(const char *, struct in_addr *);

extern
char   *inet_ntoa(struct in_addr);

/*----------------------------------------------------------------------*/
#endif
#ifdef __cplusplus
}
#endif
