/**********************************************************************
                        Serial Hardware Interface
 ***********************************************************************
 10/11/2003 - Marc LEGOUX C6R11b: Add _SerialSignal()
 27/09/2003 - Marc LEGOUX C6R11: Change type to void
 16/08/2003 - Marc LEGOUX C6R09: Creation from SASY
 ***********************************************************************/
#ifndef hwiser_h
#define hwiser_h

#ifdef __cplusplus
extern "C" {
#endif

#include <err.h>
/*-------------------------------------------------------------------*
			Status & Errors codes
 *-------------------------------------------------------------------*/
extern err_Type hwi__SerialErr;
#define hwi__Serial_ok			0
#define hwi__Serial_comm		(1+hwi__SerialErr)
#define hwi__Serial_timeout		(2+hwi__SerialErr)
#define hwi__Serial_iq_not_empty	(3+hwi__SerialErr)

#define hwi__SerialNoFunc		(4+hwi__SerialErr)
#define hwi__SerialType			(5+hwi__SerialErr)

// Errors from asynch (used also with direct uart access)
#define hwi__SerialAsynch	(10+hwi__SerialErr)	 // errors from asynch 
#define hwi__Serial_inv_port	(2+hwi__SerialAsynch)
#define hwi__Serial_not_open	(3+hwi__SerialAsynch)
#define hwi__Serial_inv_param	(4+hwi__SerialAsynch)	// invalid parameter 
#define hwi__Serial_port_not_found (6+hwi__SerialAsynch)
#define hwi__Serial_oq_full	(7+hwi__SerialAsynch)	// output queue full 
#define hwi__Serial_already_open (9+hwi__SerialAsynch)
#define hwi__Serial_iq_empty	(10+hwi__SerialAsynch)	// input queue empty 
#define hwi__Serial_asynch_err	(11+hwi__SerialAsynch)	// internal error 
#define hwi__Serial_inv_baud	(12+hwi__SerialAsynch)	// invalid parameter 

#define hwi__SerialOpen		hwi__Serial_already_open
/*----------------------------------------------------------------------*
			Driver object
 *----------------------------------------------------------------------*/
#define hwi_SerialType void
/*-------------------------------------------------------------------*
			Open parameters
 Used to Open the driver
 *-------------------------------------------------------------------*/
typedef
  struct {
    ulong	baud_rate;	// 110,...,9600,14400,19200,38400,56000 
                        	// 57600,115200,128000,256000 
    int		parity;		// 0:no, 1:odd, 2:even 
    int		data_bits;	// 5,6,7,8 
    int		stop_bits;	// 1,2 
    unsigned	timeout;	// ms (RFU)
    unsigned	mode;
    int		port_irq;	// 0 for standard values, or port irq. 
    
  } hwi_SerialOpenParamType;

#define hwi_SerialParityNo	0
#define hwi_SerialParityOdd	1
#define hwi_SerialParityEven	2

// Modes (ORed flags)
#define hwi_SerialModeBus	1	// Half duplex bus mode 
#define hwi_SerialModeMcuClk	0x40	// Baud from mcu clock 
#define hwi_SerialModeRS485	0x80	// RS485 else RS232 
// RTS control : default = OFF by InReady and ON by OutReady 
#define hwi_SerialModeRtsR	2	// RTS is reversed (OFF for transmit) 
#define hwi_SerialModeRtsT	4	// RTS is toggled automatically 
#define hwi_SerialModeRtsNo	8	// RTS is not controlled 

/*-------------------------------------------------------------------*
                        FIND Driver
 Id :"COM1", "COM2", ... or "<PortAddr>"
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialFind(
        hwi_SerialType ** pobj,		// OUT
        unsigned Mode, 
        char * Id			// Driver Identifier, 0 ended
        );
/*-------------------------------------------------------------------*
			OPEN PORT
 Returns an error if already open
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialOpen(
        hwi_SerialType * obj,
        hwi_SerialOpenParamType * Param	// Open parameters
        );
/*-------------------------------------------------------------------*
                        CLOSE PORT
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialClose(
        hwi_SerialType * obj
        );
/*-------------------------------------------------------------------*
                        GET CHAR
 Read a char from the port with timeout
 Errors :
 0 : At least a char is read
 hwi__Serial_timeout : No reception before timeout
 syst__Signal: Get is interrupted by _Signal()
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialGetCharT(
        hwi_SerialType * obj,
        unsigned char * pch,		// (OUT) Char 
        unsigned timeout		// Timeout (ms)
        );
/*-------------------------------------------------------------------*
                        TEST INPUT QUEUE EMPTY
 Returns 0 if empty, else 1
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialGetNotReady(
        hwi_SerialType * obj
	);
/*-------------------------------------------------------------------*
			PUT CHAR
 Directly write a char to the port.
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialPutChar(
        hwi_SerialType * obj,
	unsigned char ch
        );
/*-------------------------------------------------------------------*
                        PUT CHAR with bufferization
                        into the output queue
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialPutCharB(
        hwi_SerialType * obj,
        unsigned char ch			/* Char */
        );
/*-------------------------------------------------------------------*
                        FLUSH output queue
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialPutFlush(
        hwi_SerialType * obj
        );
/*-------------------------------------------------------------------*
			DRAIN INPUT QUEUE
 The chars received until now are lost.
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialDrainIq(
        hwi_SerialType * obj
	);
/*-------------------------------------------------------------------*
                        Get Info
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialInfoGet(
        hwi_SerialType * obj, 
        hwi_SerialOpenParamType * Param,	// 0 or (OUT) Open parameters
        void * RFU			// 0
        );
/*-------------------------------------------------------------------*
                        HALF DUPLEX FUNCTIONS
 Control is given to manage correctly a real half duplex connection
 *-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*
                        Make line ready for input
 External DTR == ON && RTS == OFF (=-12V) : receive
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialInReady(
        hwi_SerialType * obj
	);
/*-------------------------------------------------------------------*
                        Make line ready for output
 External DTR == ON && RTS == ON (+12V) : transmit
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialOutReady(
        hwi_SerialType * obj
        );
/*-------------------------------------------------------------------*
		Signal to stop waiting for _GetCharT
 *-------------------------------------------------------------------*/
extern
err_Type hwi_SerialSignal(
        hwi_SerialType * obj, 
        ulong Sig		// Signal value or 0
        );
/*-------------------------------------------------------------------*/
#ifdef __cplusplus
}
#endif

#endif
