/***********************************************************************
                HWI
                Hardware Interface
                for System kernel MLOS32
 ***********************************************************************
 10/01/2009 - Marc LEGOUX - V2r23p: Add Identifier sections to hwi_SpaceType
 21/05/2007 - Marc LEGOUX - V2r23h: Add sections to hwi_SpaceType
 04/05/2006 - Marc LEGOUX - V2r23a: Add Flush Op to CacheCtrl()
 24/03/2006 - Marc LEGOUX - V2r23a: Add CacheCtrl() and CacheState in Space
 07/03/2006 - Marc LEGOUX - V2r23a: Add hwi_FlashSizeGet()
 20/10/2005 - Marc LEGOUX - V2r22g: Add missing hwi_ICONGet()
 20/05/2005 - Marc LEGOUX - V2r22c: Add Heap start/end addrs in hwi_SpaceType
 19/05/2005 - Marc LEGOUX - V2r22c: Moved to KRN32. 
 14/05/2005 - Marc LEGOUX - CF2r23 : Moved to MLOS\CF. Include HWIMISC.H
 02/11/2003 - Marc LEGOUX - V2r09a: Increase Flash Max Size
 27/09/2003 - Marc LEGOUX - V2r08b: Add hwi_JtagAddr
 05/09/2003 - Marc LEGOUX - V2r08 : Add functions into hwi_SpaceType
 13/06/2003 - Marc LEGOUX - V2r07 : Rename mcf5307 symbols to hwi_
 17/01/2003 - Marc LEGOUX - V2r03 : hwi__MemErr from #define to extern
 13/12/2002 - Marc LEGOUX - V2r02 : Add CSi i>3
 15/11/2002 - Marc LEGOUX - V2r02 : Add WDog
 28/09/2000 - Marc LEGOUX - V1r07 : Add hwi_VectExch
 10/01/2000 - Marc LEGOUX - V1r05 : Extract from HWI
 ***********************************************************************/
#ifndef hwi32_h
#define hwi32_h

#include <err.h>
#include <hwimisc.h>
/************************************************************************
			MAIN
 ************************************************************************/
/*----------------------------------------------------------------------*
                        Init Main
 *----------------------------------------------------------------------*/
extern
err_type hwi_MainInit(
        unsigned mode
        );
/************************************************************************
                Memory, Register and API Spaces
 hwi_SpaceTypeVers : 
 1 : hwi_FlashSectorMax = 20 (+ RFU[44])
 2 : hwi_FlashSectorMax = 64 (no RFU)
 ************************************************************************/
// Max nb of sector lists
#define hwi_FlashSectorMax	20
// List of contiguous sectors of same size
typedef
  struct {
    ulong Addr;		// First sector addr
    ulong Size;		// Sector Size, SectorNb-1 (lowest byte)
  } hwi_FlashSectorType;	/* Sector descriptor */

#define hwi_FlashSectorAddr(sp, s)  ((sp)->FlashMap[s].Addr)
#define hwi_FlashSectorSizeMask	    0xFFFFFF00
#define hwi_FlashSectorSize(sp, s)   ((sp)->FlashMap[s].Size &  hwi_FlashSectorSizeMask)
#define hwi_FlashSectorNb(sp, s)    (((sp)->FlashMap[s].Size & ~hwi_FlashSectorSizeMask)+1)

#define hwi_SpaceNameMax	16

// Descriptor header (immediatly followed by data = array of entries)
typedef
  struct {
    ulong SizeOf;			// Size of descriptor (header+data)
    					// 0 if not defined
    char Name[hwi_SpaceNameMax+4];	// Descriptor name
    ulong Nb;				// Number of data entries
  } hwi_SpaceDType;

// Default size of all descriptors (but first header)
// WARNING: hwi_Space but be allocated also in SRAM
#define hwi_SpaceDSize 524		//=> sizeof(hwi_SpaceType) = 0x500

// Descriptor entry for areas in memory space
#define hwi_SpaceDNameMem	"MEMORY"
typedef
  struct {
    char Name[hwi_SpaceNameMax+4];	// Name of mem area (Class)
    char Chip[hwi_SpaceNameMax+4];	// Name of chip
    ulong Addr;		// First Address
    ulong Size;		// Area Size
    ulong Mode;		// Area Mode
  } hwi_SpaceMemType;

// Modes:
#define hwi_SpaceMemModeSizeMask	3	// Mask for bus size
#define hwi_SpaceMemModeSize8		0	// bus 8 bits
#define hwi_SpaceMemModeSize16		1	// bus 16 bits
#define hwi_SpaceMemModeSize32		2	// bus 32 bits

#define hwi_SpaceMemMax	8	// Def max nb of mem entries

// Descriptor entry for areas in memory space
#define hwi_SpaceDNameId	"IDENTIFIER"
typedef
  struct {
    char BoardName[hwi_SpaceNameMax+4];	// Name of Board
    char BoardVers[hwi_SpaceNameMax+4];	// Board version
    char BoardId[8];			// Id for Serial num (ex: "EP")
    ulong BoardCode;			// Unique Code (used in Mac address)
  } hwi_SpaceIdType;

#define hwi_SpaceMagic	0xBB421421

#define hwi_SpaceTypeVers23a	2	// V2r23a
#define hwi_SpaceTypeVers	sizeof(hwi_SpaceType)
typedef
  struct {
    ulong	SizeOf;	// Set to hwi_SpaceTypeVers
    
    // Register space
    ulong IMMAddr;
    
    // Vector space
    ulong VBRAddr;
    
    /* SDRAM: Bank0 then Bank1 if defined */
    ulong SDramAddr;
    ulong SDramSize;
    ulong SDramBk0Size;
    ulong SDramBk1Size;
    
    /* SRAM (Internal) */
    ulong SramAddr;
    ulong SramSize;
    
    /* CS */
    ulong CSAddr;
    ulong CS0Addr;
    ulong CS1Addr;
    ulong CS2Addr;
    ulong CS3Addr;
    ulong CS4Addr;
    ulong CS5Addr;
    ulong CS6Addr;
    ulong CS7Addr;
    
    /* MCU MASK */
    ulong McuMask;
    
    /* FLASH */
    ulong FlashAddr;
    ulong FlashSize;
    /* Sector lists appear in increasing address order */
    /* After last valid sector, Size==0 */
    hwi_FlashSectorType FlashMap[hwi_FlashSectorMax+1];
    
    hwi_FlashSectorType FlashRFU[44];
    
    // Boot version
    char BootVersion[32+4];	// <BoardName>:Boot<BootVersion>
    
    // Magic
    ulong	Magic;
    
    // Interface Functions
    err_Type (* CfgFlagGet)(unsigned * pFlag);
    err_Type (* NodeGet)(uchar * pnode);
    void (* McuLedToggle)();
    err_Type (* ICONPut)(uchar mask, uchar op);
    err_Type (* ICONGet)(uchar * pmask);
    err_Type (* IOPut)(uchar mask, uchar op);
    err_Type (* IOGet)(uchar * pmask);
    
    ulong	McuFreq;	// MCU Frequency KHz
    ulong	ExtFreq;	// External frequency (KHz) 0=not defined
    
    // From V2R22c
    ulong	HeapAddr;	// Max System heap (Computed at link time)
    ulong	HeapSize;
    
    ulong	UserAddr;	// Max User area if system/user mode 
    ulong	UserSize;	// 0 if none
    
    // From V2R23a
    ulong	CacheState[6];	// To Save cache registers in CacheCtrl()
    
    ulong	PPortDir;	// Parallel port dir (1=Out)
    ulong	PPortData;	// First data to output
    
    // From V2R23h
    // Named Descriptors
    hwi_SpaceDType	DHd;			// First Descriptor header
    char		DData[hwi_SpaceDSize];	// Default data
    // SizeOf of last descriptor must == 0
    
  } hwi_SpaceType;
  
#define hwi_SPACE hwi_SpaceType

extern
hwi_SpaceType * hwi_Space;		// Current Space (Inited by boot)
/*----------------------------------------------------------------------*
                        Init defaults
 *----------------------------------------------------------------------*/
extern
err_type hwi_SpaceInitDef(
        hwi_SpaceType * Space
        );
/*----------------------------------------------------------------------*
                        Init for the board
 *----------------------------------------------------------------------*/
extern
err_type hwi_SpaceInit(
        hwi_SpaceType * Space
        );
/*----------------------------------------------------------------------*
                        Get Space
 *----------------------------------------------------------------------*/
extern
hwi_SpaceType * hwi_SpaceGet();

/*----------------------------------------------------------------------*
                        Get Area
 *----------------------------------------------------------------------*/
extern
err_Type hwi_SpaceMemAreaGet(
        hwi_SpaceType * Space, // 0 for current
        char *	Name, 			// Area name
        ulong	Mode, 			// bit0:Next
        hwi_SpaceMemType * pArea	//IN/OUT
	);

/***********************************************************************
                IRQ and TRAP
 To call inside CS
 IRQ represent external hardware interrupts and are assigned to special 
 vectors. 
 ***********************************************************************/
/*--------------------------------------------------------------------*
                Enable / Disable
 *--------------------------------------------------------------------*/
extern
err_Type hwi_IrqVectCtrl(
        ulong irq, 	// irq level
        unsigned Mode	// 0: disable, 1:Enable
        );
/*----------------------------------------------------------------------*
                        Check IRQ
 returns != 0 if interrupt is pending (even if disabled)
 *----------------------------------------------------------------------*/
extern
int hwi_IrqVectIsOn(
        register ulong irq 		// irq level 1..7
        );
/*--------------------------------------------------------------------*
                Set IRQ vector
 *--------------------------------------------------------------------*/
extern
err_Type hwi_IrqVectSet(
        ulong irq, 	// Irq
        ulong func
        );
/*--------------------------------------------------------------------*
                IRQ vector offset
 *--------------------------------------------------------------------*/
extern
err_Type hwi_IrqVectOffs(
        ulong irq, 	// Irq
        ulong * pOffs	// OUT: To use with hwi_VectXXX
        );
/*--------------------------------------------------------------------*
                Set IRQ vector
 *--------------------------------------------------------------------*/
extern
err_Type hwi_VectExch(
        ulong offs, 	// 0-255 
        ulong func,	// New vector (0 for default) 
        ulong * pfunc	// null or (OUT): Previous vector 
        );
/*--------------------------------------------------------------------*
                Set trap vector
 *--------------------------------------------------------------------*/
#define hwi_TrapVectSet hwi_TrapSetVect
extern
void hwi_TrapSetVect(
        ulong num, 	/* 0-15 */
        ulong func
        );
/************************************************************************
			CACHE
 ************************************************************************/
// Operation
#define hwi_CacheOpMask		0x0F	// Operation mask
#define hwi_CacheDisable	0x00	// Disable area[s]
#define hwi_CacheEnable		0x01	// Invalidate all and Enable area[s]
#define hwi_CacheModify		0x02	// Modify size and mode 
#define hwi_CacheFlush		0x03	// Flush Data
#define hwi_CacheRestore	0x0F	// Restore state before previous Op

// Cacheable areas
#define hwi_CacheGlobal		0x10
#define hwi_CacheFlash		0x20
#define hwi_CacheSDram		0x80

// Modes
#define hwi_CacheModeWT		0x100	// Write Through (else Copy back)

extern
err_Type hwi_CacheCtrl(
        hwi_SpaceType * space, 
        ulong mask,
        ulong RFU
        );
/***********************************************************************
                        ICON
 ISA and IO control register
 ***********************************************************************/
#define hwi_IconOut7	0x80	/* Red perso side */
#define hwi_IconOut6	0x40	/* Red middle */
#define hwi_IconOut5	0x20	/* Green perso side */
#define hwi_IconOut4	0x10	/* Green middle */
#define hwi_IconOut3	0x08	/* Not connected */

#define hwi_IconClk16	0x04	/* ISAClock = mcu clock / 2 (else /4)*/

/* Wait states WS1,0 : 00=0, 01=1, 10=3, 11=5 */
#define hwi_IconWS1	0x02
#define hwi_IconWS0	0x01

#define hwi_IconIn7	0x80	/* IO7 */
#define hwi_IconIn6	0x40	/* IO6 */
#define hwi_IconIn5	0x20	/* IO5 */
#define hwi_IconIn4	0x10	/* IO4 */
#define hwi_IconIn3	0x08	/* IO3 */
#define hwi_IconIn2	0x04	/* IO2 */
#define hwi_IconIn1	0x02	/* IO1 */
#define hwi_IconIn0	0x01	/* IO0 */
/*---------------------------------------------------------------------*
                        ICON Put 
 *---------------------------------------------------------------------*/
extern
err_Type hwi_ICONPut(uchar mask, uchar op);

/*---------------------------------------------------------------------*
                        ICON Get
 *---------------------------------------------------------------------*/
extern
err_Type hwi_ICONGet(uchar * pmask);

/*--------------------------------------------------------------------*
                MCU LED
 Signal the MCU activity
 *--------------------------------------------------------------------*/
extern
void hwi_McuLedToggle();
/***********************************************************************
                        TIMERs
 ***********************************************************************/
#define hwi_McuFreq	(hwi_Space->McuFreq)		// KHz 
#define hwi_ExtFreq	(hwi_Space->ExtFreq)		// KHz 

/*--------------------------------------------------------------------*
                Timer1
 *--------------------------------------------------------------------*/
extern ulong hwi_Timer1Period;		/* us */
/* Timer counter max value */
#define hwi_Timer1Max	0x0FFF0
/*--------------------------------------------------------------------*
                Set timer and start
 *--------------------------------------------------------------------*/
extern
void hwi_Timer1Start(ulong time);
/*--------------------------------------------------------------------*
                Stop timer (keep counter values)
 *--------------------------------------------------------------------*/
extern
void hwi_Timer1Stop();
/*--------------------------------------------------------------------*
                Check Irq
 returns != 0 if Irq is raised
 *--------------------------------------------------------------------*/
extern
int hwi_Timer1IrqCheck(void);
/*--------------------------------------------------------------------*
                Get timer remaining count
 *--------------------------------------------------------------------*/
extern
ulong hwi_Timer1Get();
/*--------------------------------------------------------------------*
                Set timer period
 *--------------------------------------------------------------------*/
extern
void hwi_Timer1PeriodSet(
        ulong t		/* in us */
        );
/*--------------------------------------------------------------------*
                Set timer vector
 *--------------------------------------------------------------------*/
extern
void hwi_Timer1SetVect (ulong func);

/*--------------------------------------------------------------------*
                Timer2
 *--------------------------------------------------------------------*/
extern ulong hwi_Timer2Period;		/* us */
/*--------------------------------------------------------------------*
                Set timer and start
 *--------------------------------------------------------------------*/
extern
void hwi_Timer2Start(ulong time);
/*--------------------------------------------------------------------*
                Stop timer (keep counter values)
 *--------------------------------------------------------------------*/
extern
void hwi_Timer2Stop();
/*--------------------------------------------------------------------*
                Check Irq
 returns != 0 if Irq is raised
 *--------------------------------------------------------------------*/
extern
int hwi_Timer2IrqCheck(void);
/*--------------------------------------------------------------------*
                Get current count
 *--------------------------------------------------------------------*/
extern
ulong hwi_Timer2GetCur();
/*--------------------------------------------------------------------*
                Set timer period
 Adjust for a 8 bit prescaler
 *--------------------------------------------------------------------*/
extern
void hwi_Timer2PeriodSet(
        ulong t		/* in us */
        );
/*--------------------------------------------------------------------*
                Set timer vector
 *--------------------------------------------------------------------*/
extern
void hwi_Timer2SetVect (ulong func);
/*--------------------------------------------------------------------*
                Irq Ack
 *--------------------------------------------------------------------*/
extern
void hwi_Timer2IrqAck();

/***********************************************************************
                        FLASH
 ***********************************************************************/
extern err_Type hwi__MemErr;
#define hwi__MemAlign		(hwi__MemErr + 1)

#define hwi__FlashErr		hwi__MemErr
#define hwi__FlashAlign		hwi__MemAlign
#define hwi__FlashFailure	(hwi__FlashErr + 2)
#define hwi__FlashNotErased	(hwi__FlashErr + 3)
#define hwi__FlashBadSector	(hwi__FlashErr + 4)
#define hwi__FlashEraseTimeout	(hwi__FlashErr + 5)

/*----------------------------------------------------------------------*
                Flash Sector get and goto next
 
 *pIter (IN, OUT) == <sector list idx(0..)><sector idx in list(0..)>
 or 
 *pIter (IN) = Absolute sector index (0..)
 Returns !=0 if end of iter
 *----------------------------------------------------------------------*/
extern
err_Type hwi_FlashSectorGetNx(
        ulong * pIter,		// IN OUT: Sector number (0 to start)
        ulong * pAddr,		// 0 or OUT : Sector Addr
        ulong * pSize,		// 0 or OUT : Sector Size
        ulong * pMode,		// 0 or OUT : Sector Mode
        ulong * pSectIdx	// 0 or OUT : Absolute sector Index (0..)
        );
/*--------------------------------------------------------------------*
                Flash Erase
 More than one sector can be erased at the same time
 *--------------------------------------------------------------------*/
#define hwi_FlashEraseModeSector	0	/* bit0:sector 0, ... */
#define hwi_FlashEraseModeAddr1		1	/* One sector only */
#define hwi_FlashEraseModeAddr		2	/* All sectors in Size */

extern
err_Type hwi_FlashErase(
        ulong Area,		/* Sector mask or Address */
        ulong Size,		/* Size if Mode Addr */
        unsigned Mode
        );
/*--------------------------------------------------------------------*
                Flash Programming
                Area
 AddrTo and AddrFrom must be aligned on 16bits the same way (same bit 0)
 *--------------------------------------------------------------------*/
extern
err_Type hwi_FlashProg(
        ulong AddrTo,
        ulong AddrFrom,
        ulong Size);
/*--------------------------------------------------------------------*
                Flash Verify
                Area
 AddrTo and AddrFrom must be aligned on 16bits the same way (same bit 0)
 *--------------------------------------------------------------------*/
extern
err_Type hwi_FlashCmp(
        ulong AddrTo,
        ulong AddrFrom,
        ulong Size);
/*--------------------------------------------------------------------*
                Get size of flash
 *--------------------------------------------------------------------*/
extern
err_Type hwi_FlashSizeGet(
        unsigned Mode, 
        ulong Addr, 
        ulong *pSize	// IN: Max Size, OUT: Real size (must == 2**x)
        );
/***********************************************************************/
#endif
