/**********************************************************************
                        Driver and Class objects
                        User Interface
 **********************************************************************
 Copyright (c) SMARTWARE, 1999, 2009 All Rights Reserved.
 **********************************************************************
 **********************************************************************
 20/10/2009 - Marc LEGOUX - V2R24b: Remove links to obj_. Add drv__NoFunc
 22/06/2009 - Marc LEGOUX - V2R24a: Add drv_FindX
 03/03/2009 - Marc LEGOUX - V2R23p: Add Take values into InfoType
 03/06/2008 - Marc LEGOUX - V2R23m: Add drv_NewModeGetExisting
 29/02/2008 - Marc LEGOUX - V2R23l: Add drv_RefAdd
 10/09/2006 - Marc LEGOUX - V2R22i: Add state _StateBusy
 18/08/2006 - Marc LEGOUX - V2R22i: Add mode _FindModeNext
 12/08/2005 - Marc LEGOUX - V6R19a: Define drv_Ref
 20/02/2005 - Marc LEGOUX - V6R18a: Add mode _NewModeNoDef 
 05/07/2004 - Marc LEGOUX - V6R16a: Add Find modes
 12/02/2004 - Marc LEGOUX - V6R14a: Reorganize modes
 04/12/2003 - Marc LEGOUX - V6R13c: Include obj.h
 01/12/2003 - Marc LEGOUX - V6R13b: Add drv_Call
 15/11/2003 - Marc LEGOUX - V6R13 : Add drv_FindModeSpace
 03/11/2003 - Marc LEGOUX - V6R12 : Add drv_InfoGet
 25/09/2003 - Marc LEGOUX - V6R11 : Creation for user interface
 **********************************************************************/
#ifndef drv_h
#define drv_h

#include <err.h>
#include <syst.h>

#ifdef __cplusplus
extern "C" {
#endif

/*----------------------------------------------------------------------*
                        Errors
 *----------------------------------------------------------------------*/
extern err_Type drv__Err;
#define	drv__NotFound		(drv__Err+1)	// Driver not found
#define	drv__NotStarted		(drv__Err+2)	// Driver module not started
#define drv__NoMatch		(drv__Err+3)	// Driver does not match at Addr
#define drv__Id			(drv__Err+4)	// Wrong Id
#define drv__NotOpen		(drv__Err+5)	// Driver not open
#define drv__New		(drv__Err+6)	// Class does not create drivers
#define	drv__NotFoundX		(drv__Err+7)	// Driver not found for remote access
#define	drv__NameSyntax		(drv__Err+8)	// Wrong syntax of a name
#define	drv__NoFunc		(drv__Err+9)	// Function not found
/*----------------------------------------------------------------------*
			Generic interface type
 *----------------------------------------------------------------------*/

#define drv_ObjType	void
#define drv_Type	drv_ObjType
#define drv_Ref		drv_ObjType*	// Driver handle 
typedef
  char drv_MAType[393];			// Driver type for method access

#define drv_NameMax		64
#define drv_NameItemMax		32	// Single term length
#define drv_NameSep		':'	// Item separator

#define drv_VersionMax		64	// Max length of version names
#define drv_PathMax		255	// Max length of path names

// States Ored
#define drv_StateDef	1	// Defined (Accessible)
#define drv_StateInit	2	// Initialized (Open enabled)
#define drv_StateOpen	4	// Open (Read and Write enabled)
#define drv_StateBusy	8	// Ressource is in use (low level mutex flag)

// Modes
#define drv_ModeAuto		0x001		// Auto open when defined
#define drv_ModeIdInt		0x020		// Id is a int else a string (0 ended)

// Same Modes as NewModes
#define drv_ModeAddrSt		0x002		// Addr is a string (0 ended)
#define drv_ModeAddrObj		0x008		// Addr is an object (obj_type*)

// Specific modes
#define drv_ModeMaskX		0xFFFF0000	// Modes for specific drivers
/*----------------------------------------------------------------------*
			Modes for NEW
 Needed for sysobj Calls
 *----------------------------------------------------------------------*/
#define drv_NewModeObjdl	0x001	// Reserved (for objdl)
#define drv_NewModeAddrSt	0x002	// Addr is a string pointer (0 ended)
#define drv_NewModeNoStart	0x004	// Do not start even if autostart
#define drv_NewModeNoUndef	0x004	// Do not delete when terminate
#define drv_NewModeAddrObj	0x008	// Addr is an object 
#define drv_NewModeResident	0x010	// Install as resident (start with system)
#define drv_NewModeNoMove	0x040	// Install where it is loaded
#define drv_NewModeNoDef	0x080	// Do not define
#define drv_NewModeGetExisting	0x100	// Get an already existing drv (same Id)
/*----------------------------------------------------------------------*
                        Find Driver or Class
 Find Name && Id if Name!=0 && Id!=0
 A Name ":Network:<station>:<object>" accesses a remote driver (Id ignored).
 *----------------------------------------------------------------------*/
#define drv_FindModeNext	0x10000000	// Find next item
#define drv_FindModeClass	0x20000000	// Find class else driver
#define drv_FindModeClassName	0x40000000	// Name is class name else driver name
						// Class name selected by FindModeName
#define drv_FindModeSpace	0x80000000	// Name is a space name

#define drv_FindModeName	0x04000000	// Class by driver Name (else interface)
#define drv_FindModeId		0x02000000	// By Id

#define drv_FindModeRefCnt	0x01000000	// Reserved

#define drv_FindModeWait	0x08000000	// Wait with timeout (local only)
#define drv_FindModeOpen	0x00800000	// Find only if open

extern
err_Type drv_FindX(
        char * 		Name,		// 0 or Name
        char * 		Id, 		// 0 or Id : "nnn" for a number
        drv_Ref* 	pDrv,		// IN,OUT : Driver | Class (IN with _FindModeNext)
        unsigned	Mode,
        syst_TType	Timeout
        );
/*----------------------------------------------------------------------*/
// Default timeout is 10s
extern
err_Type drv_Find(
        char * 		Name,		// 0 or Name
        char * 		Id, 		// 0 or Id : "nnn" for a number
        drv_Ref* 	pDrv,		// IN,OUT : Driver | Class (IN with _FindModeNext)
        unsigned	Mode
        );
/*----------------------------------------------------------------------*
                Get Info
 *----------------------------------------------------------------------*/
#define drv_InfoTypeSizeV2r23m	(548)

#pragma pack(4)
typedef
  struct {
    ulong	SizeOf;
    ulong	State;			// State
    ulong	Mode;			// Mode
    char	Name[drv_NameMax+4];	// Drv Path name "class:type:num"
    char 	ClassName[drv_NameMax+4];	// Class Name
    char 	Version[drv_VersionMax+4];	// Version Name
    char 	Space[drv_NameItemMax+4];	// Space name
    char  	Id[drv_NameItemMax+4];		// Object Id (0 terminated)
    
    ulong	Len;			// Reserved length at Addr (optional)
    char	Addr[drv_PathMax+1];	// Address in space (int or string)
    
    //V2r23p
    err_Type	OpenErno;
    err_Type	CloseErno;
    
    ulong	TakeCount;	// Number of nested Take()
    ulong	TakeMode;
    syst_TType	TakeTime0;	// Start Time (us)
    syst_TType	TakeTime;	// Duration (us)
    ulong	TakeCalls;	// Number of calls
    
    err_Type	MethodErno;			// Error of last called method
    char 	MethodName[drv_NameItemMax+4];	// Name of last called method
    
  } drv_InfoType;
#if config_DIAB
#pragma pack(0)
#else
#pragma pack()
#endif

extern
err_Type drv_InfoGet(
        drv_Ref obj,
        unsigned Mode,
        drv_InfoType * pInfo		// OUT
        );

#define drv_InfoModeClear	1	// Clear counted values (Time, Calls, Erno)
/*----------------------------------------------------------------------*
                        Call driver with buffer
 *----------------------------------------------------------------------*/
extern
err_Type drv_Call(
        drv_Ref obj,
        uchar * Buf, 		// IN, OUT data buffer
        ulong Argl,		// Input buffer length
        ulong * pRespl,		// IN: Max buffer length, OUT: return length
        syst_TType Timeout	// Timeout if remote
        );
/*----------------------------------------------------------------------*
                        Delete Driver Ref
 The parameter must be writable and it is set to 0
 *----------------------------------------------------------------------*/
#define drv_RefDel(par) ((par) ? drv_RefDel0((drv_Ref)(par)),((par)=0) : 0)
/*----------------------------------------------------------------------*
                        Delete Driver Ref
 *----------------------------------------------------------------------*/
extern err_Type drv_RefDel0(drv_Ref Drv);

// WARNING: RefCnt is already decremented with a macro
extern err_Type drv_RefDelF(drv_Ref Drv);
/*----------------------------------------------------------------------*
                        Duplicate Driver Ref
 *----------------------------------------------------------------------*/
extern err_Type drv_RefAdd0(drv_Ref Drv);

#define drv_RefAdd(par) ((par) ? drv_RefAdd0((drv_Ref)(par)) : 0)
/*----------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif
#endif
