/*
 * Copyright 2022 - 2023, 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * MP300 specific HAL-Component of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <phbalReg.h>
#include <ph_RefDefs.h>

#import "ex/ContactLAB.exe" no_namespace

#ifdef NXPBUILD__PHHAL_HW_CONTACT_KEOLABS

#include "phhalHwContact_Keolabs.h"
#include "phhalHwContact_Keolabs_Int.h"
#include <phTools.h>
#include <phToolsAtrParser.h>

phStatus_t phhalHwContact_Keolabs_Int_ConvertError(
                                                   uint32_t status
                                                   )
{
    switch (status)
    {
    case 0:
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);

    default:
        return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
    }
}

phStatus_t phhalHwContact_Keolabs_Int_SetVcc(
                                             phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                             uint32_t vcc
                                             )
{
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;

    if(vcc < 0 || vcc > 7000)
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    switch(vcc)
    {
    case PHHAL_HW_CONTACT_CLASS_A:
        ContactLAB->Reader->Voltage->Class = CLASS_A;
        break;

    case PHHAL_HW_CONTACT_CLASS_B:
        ContactLAB->Reader->Voltage->Class = CLASS_B;
        break;

    case PHHAL_HW_CONTACT_CLASS_C:
        ContactLAB->Reader->Voltage->Class = CLASS_C;
        break;

    default:
        ContactLAB->Reader->Voltage->Class = CLASS_CUSTOM;
        ContactLAB->Reader->Voltage->ClassCustom->VccVoh = (float32_t)(vcc / 1000.0);
        ContactLAB->Reader->Voltage->ClassCustom->IOVoh = (float32_t)(vcc / 1000.0);
        ContactLAB->Reader->Voltage->ClassCustom->RSTVoh = (float32_t)(vcc / 1000.0);
        ContactLAB->Reader->Voltage->ClassCustom->CLKVoh = (float32_t)(vcc / 1000.0);
        ContactLAB->Reader->Voltage->ClassCustom->IOVih = (float32_t)((vcc / 1000.0) * 0.7);
        ContactLAB->Reader->Voltage->ClassCustom->IOVil = (float32_t)((vcc / 1000.0) * 0.15);
        break;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_SetClk(
                                             phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                             uint32_t clk
                                             )
{
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;

    if(clk < (200 * 1000) || clk > (30000 * 1000))
    {
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    ContactLAB->Reader->ISO7816->Clock->Frequency = (float32_t)(clk / 1000000.0);
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_SetDelay(
                                               phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                               uint32_t delay
                                               )
{
    phStatus_t statusTmp;
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Delay(delay)));
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_ResetReader(
                                                  phhalHwContact_Keolabs_DataParams_t * pDataParams
                                                  )
{
    phStatus_t statusTmp;
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Reset()));

    /* Set Timing Parameters to Transparent as we will use them later on */
    ContactLAB->Reader->ISO7816->TParameter = MODE_TSP;

    /* Disable Logging (increased performance) */
    //ContactLAB->Display->Verbose = 0;

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_Activate(
                                               phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                               uint8_t * pATRBuffer,
                                               uint16_t * pwATRLength
                                               )
{
    phStatus_t statusTmp;
    uint16_t wDelay = 0;
    uint32_t dwClk = 0;
    uint32_t dwMaxClkAtr = 0;
    phStatus_t status = PH_ERR_SUCCESS;
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;
    uint8_t bColdReset;
    unsigned long qwAtrLength;

    if(pDataParams->dwAtrLength == 0)
    {
        bColdReset = 1;
    }
    else
    {
        bColdReset = 0;
    }

    /* Reset ATR before doing anything else */
    memset(pDataParams->pAtr, 0x00, PHHAL_HW_CONTACT_MAX_ATR_SIZE);
    pDataParams->dwAtrLength = 0;

    /* Before activation bring reader to default state */
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_SetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_FD, PHHAL_HW_CONTACT_FD_DEFAULT));
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));

    if(bColdReset)
    {
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Reader->Power(0)));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Reader->Power(1)));

        /* Wait for ATR */
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_CLOCK_FREQUENCY_HZ, &dwClk));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_MAX_CLK_ATR, &dwMaxClkAtr));

        wDelay = (uint16_t)(dwMaxClkAtr * 1000 / dwClk + 0.5);
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Wait(pDataParams, PHHAL_HW_CONTACT_TIME_MILLISECONDS, wDelay));

        status = phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Reader->ISO7816->GetATR(pDataParams->pAtr, PHHAL_HW_CONTACT_MAX_ATR_SIZE, &qwAtrLength));
        pDataParams->dwAtrLength = (uint32_t)qwAtrLength;
    }
    else
    {
        status = phhalHwContact_Keolabs_Int_ConvertError(
            ContactLAB->Reader->ISO7816->WarmReset(pDataParams->pAtr, PHHAL_HW_CONTACT_MAX_ATR_SIZE, &qwAtrLength));
        pDataParams->dwAtrLength = (uint32_t)qwAtrLength;
    }

    if(status == PH_ERR_SUCCESS)
    {
        if(pDataParams->dwAtrLength > PHHAL_HW_CONTACT_MAX_ATR_SIZE)
        {
            *pwATRLength = 0;
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        memcpy(pATRBuffer, &pDataParams->pAtr[0], pDataParams->dwAtrLength);
        *pwATRLength = (uint16_t)pDataParams->dwAtrLength;

        /* Set ATR in phToolsAtrParser to parse certain values later */
        PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetInternalAtr(&pDataParams->pAtr[0], pDataParams->dwAtrLength));
        PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_ApplyProtocolSettings(pDataParams, 0, 0, PHHAL_HW_CONTACT_PROTOCOLTYPE_T0));
    }
    else
    {
        *pwATRLength = 0;
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_Deactivate(
                                                 phhalHwContact_Keolabs_DataParams_t * pDataParams
                                                 )
{
    phStatus_t statusTmp;
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;
    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Reader->Power(0)));
    pDataParams->dwAtrLength = 0;
    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_SetDefaultValues());
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_TransparentCommand(
                                                         phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                                         uint8_t    * pTxBuffer,
                                                         uint16_t     wTxLength,
                                                         uint8_t   ** pRxBuffer,
                                                         uint16_t   * wRxLength,
                                                         uint16_t     wRxBuffSize
                                                         )
{
    phStatus_t statusTmp;
    phStatus_t status;
    uint32_t protocolType;
    unsigned long wTmpRxLength = 0;
    const uint32_t wTmpRxBuffSize = 256;
    uint8_t PH_MEMLOC_BUF pTmpRxBuffer[wTmpRxBuffSize];
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;

    PH_CHECK_SUCCESS_FCT(statusTmp, phhalHwContact_GetConfig32(pDataParams, PHHAL_HW_CONTACT_CONFIG_PROTOCOLTYPE, &protocolType));

    switch(protocolType)
    {
    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T0:
        status = phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Reader->ISO7816->SendTsp_WWT(
            pTxBuffer, wTxLength, 0, ContactLAB->Reader->ISO7816->Timing->WWT, pTmpRxBuffer, wTmpRxBuffSize, &wTmpRxLength));
        break;

    case PHHAL_HW_CONTACT_PROTOCOLTYPE_T1:
        status = phhalHwContact_Keolabs_Int_ConvertError(ContactLAB->Reader->ISO7816->SendTsp_WWT(
            pTxBuffer, wTxLength, 0, ContactLAB->Reader->ISO7816->Timing->BWT, pTmpRxBuffer, wTmpRxBuffSize, &wTmpRxLength));
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    if(wTmpRxLength > 0)
    {
        if(wTmpRxLength > wRxBuffSize)
        {
            return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
        }

        memcpy(*pRxBuffer, pTmpRxBuffer, wTmpRxLength);
        *wRxLength = (uint16_t)wTmpRxLength;
    }
    else
    {
        return PH_ADD_COMPCODE(PH_ERR_IO_TIMEOUT, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(status, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_SetFdValue(
                                                 phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                                 uint8_t bFdValue
                                                 )
{
    phStatus_t statusTmp;
    uint16_t wFValue = 0;
    uint16_t wDValue = 0;
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;

    PH_CHECK_SUCCESS_FCT(statusTmp, phToolsAtrParser_ConvertFdIndexesToValues(bFdValue, &wFValue, &wDValue));

    ContactLAB->Reader->ISO7816->Timing->ETU = wFValue / wDValue;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_GetConvention(
                                                    phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                                    uint8_t * bConvention
                                                    )
{
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;

    switch(ContactLAB->Reader->ISO7816->ATR->IVC)
    {
    case 0:
        *bConvention = PHHAL_HW_CONTACT_DIRECT_CONVENTION;
        break;

    case 1:
        *bConvention = PHHAL_HW_CONTACT_INVERSE_CONVENTION;
        break;

    default:
        *bConvention = 0;
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_SetTimingParameter(
                                                         phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                                         uint16_t wConfig,
                                                         uint32_t dwValue
                                                         )
{
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;

    switch(wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
        ContactLAB->Reader->ISO7816->Timing->WT = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
        ContactLAB->Reader->ISO7816->Timing->GT = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
        ContactLAB->Reader->ISO7816->Timing->CGT = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
        ContactLAB->Reader->ISO7816->Timing->CWT = dwValue;
        break;

    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        ContactLAB->Reader->ISO7816->Timing->BWT = dwValue;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_GetTimingParameter(
                                                         phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                                         uint16_t wConfig,
                                                         uint32_t * pValue
                                                         )
{
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;

    switch(wConfig)
    {
    case PHHAL_HW_CONTACT_CONFIG_WT_CLK:
        *pValue = ContactLAB->Reader->ISO7816->Timing->WT;
        break;

    case PHHAL_HW_CONTACT_CONFIG_GT_CLK:
        *pValue = ContactLAB->Reader->ISO7816->Timing->GT;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CGT_CLK:
        *pValue = ContactLAB->Reader->ISO7816->Timing->CGT;
        break;

    case PHHAL_HW_CONTACT_CONFIG_CWT_CLK:
        *pValue = ContactLAB->Reader->ISO7816->Timing->CWT;
        break;

    case PHHAL_HW_CONTACT_CONFIG_BWT_CLK:
        *pValue = ContactLAB->Reader->ISO7816->Timing->BWT;
        break;

    default:
        return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
    }

    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

phStatus_t phhalHwContact_Keolabs_Int_GetIfscValue(
                                                   phhalHwContact_Keolabs_DataParams_t * pDataParams,
                                                   uint32_t * pValue
                                                   )
{
    IContactLABPtr ContactLAB = (IContactLAB*)pDataParams->pBalDataParams->pContactLAB;
    *pValue = ContactLAB->Reader->ISO7816->T1->IFSC;
    return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
}

#endif /* NXPBUILD__PHHAL_HW_CONTACT_KEOLABS */
