/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
* SAM (AV4 and future SAM's) Specific support interface implementation of Reader Library Framework.
* $Author: Rajendran Kumar (nxp99556) $
* $Revision: 7467 $
* $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
*/

#ifndef PHHALHW_SAM_HCUTILS_H
#define PHHALHW_SAM_HCUTILS_H

#include <ph_Status.h>
#include <ph_TypeDefs.h>

/** \addtogroup phhalHw_Sam_Utils
 * @{
 */

/**
 * \brief Get the length of LC and also check is LC and LE are present in the command frame.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Utils_GetCheckLcLe(
        uint8_t * pCmd,                                                                 /**< [In] Address of buffer that contains the command. */
        uint16_t wCmdLen,                                                               /**< [In] Length of the command. */
        uint8_t * pIsLcPresent,                                                         /**< [Out] LC byte presence indicator. */
        uint8_t * pLcLen,                                                               /**< [Out] Length of the payload of the command. */
        uint8_t * pIsLePresent                                                          /**< [Out] LE byte presence indicator. */
    );

/**
 * \brief Truncate a 16-Bytes MAC buffer into a 8-Bytes Buffer
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Utils_TruncateMacBuffer(
        uint8_t * pIoBuffer,                                                            /**< [In, Out] The 16 bytes of computed MAC as input and  8 bytes of truncated MAC as output. */
        uint8_t * pMacLen                                                               /**< [In, Out] Non-Truncated MAC length as input and Truncated MAC length as output. */
    );

/**
 * \brief Gets the current updated LC information.
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Utils_GetLc(
        phhalHw_Sam_DataParams_t * pDataParams,                                          /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pLC                                                                    /**< [Out] Provides the current updated LC information. */
    );

/**
 * \brief Update LC byte according to data already available in Buffer
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Utils_UpdateLc(
        phhalHw_Sam_DataParams_t * pDataParams                                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Update P1 byte according to data already available in Buffer
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Utils_UpdateP1(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t p1                                                                      /**< [In] Value for P1. */
    );

/**
 * \brief Update P1byte according to data already available in Buffer
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Utils_UpdateP2(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t p2                                                                      /**< [In] Value for P1. */
    );

/**
 * \brief Parse Status Word Sw1Sw2 from the SAM and assign a equivalent custom code.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phhalHw_Sam_Utils_ResolveErrorCode(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pSw1Sw2                                                               /**< [In] Pointer to the status code sent by the SAM. */
    );

/**
 * \brief Session key generation for LockUnlock command.
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phhalHw_Sam_Utils_GenerateHostAuthSessionKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyType,                                                               /**< [In] The key type to be used (AES128 or AES192) */
        uint8_t * pRnd1,                                                                /**< [In] Pointer to Rnd1 data. */
        uint8_t * pRnd2,                                                                /**< [In] Pointer to Rnd2 data. */
        uint8_t * pSessionKey,                                                          /**< [Out] The generated session key. */
        uint8_t * pKeyLen                                                               /**< [Out] Length of the Session key. */
    );

/**
 * \brief Session key generation for AuthenticateHost command.
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phhalHw_Sam_Utils_GenerateSessionKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bKeyType,                                                               /**< [In] The key type to be used (AES128 or AES192) */
        uint8_t * pRnd1,                                                                /**< [In] Pointer to Rnd1 data. */
        uint8_t * pRnd2,                                                                /**< [In] Pointer to Rnd2 data. */
        uint8_t * pSessionEncKey,                                                       /**< [Out] The generate session encryption key. */
        uint8_t * pSessionMacKey,                                                       /**< [Out] The generate session mac key. */
        uint8_t * pKeyLen                                                               /**< [Out] Length of the Session key. */
    );

/**
 * \brief Session key generation for the command PLUpload.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS Operation successful.
 */
phStatus_t phhalHw_Sam_Utils_GetSessionUploadKey(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bMode,                                                                  /**< [In] Mode to use for session key generation. Can be one of the below options.
                                                                                         *        \arg PHHAL_HW_CMD_SAM_SESSION_KEY_ENC "Session Encryption Key"
                                                                                         *        \arg PHHAL_HW_CMD_SAM_SESSION_KEY_MAC "Session MAC Key"
                                                                                         */
        uint16_t wUploadCtr,                                                            /**< [In] The upload counter value. */
        uint8_t bKeyNo,                                                                 /**< [In] Key number using which the key will be taken from keystore for macing the session vectors. */
        uint8_t bKeyVer,                                                                /**< [In] Key version to be used. */
        uint8_t * pSessionKey,                                                          /**< [Out] The generated session key. */
        uint8_t * pKeyType                                                              /**< [Out] The type of key used for session key generation. */
    );

/**
 * end of group phhalHw_Sam_Utils
 * @}
 */

#endif /* PHHALHW_SAM_HCUTILS_H */
