/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) generic command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */

#include <ph_Status.h>
#include <ph_RefDefs.h>
#include <phTools.h>

#include <phbalReg.h>
#include <phhalHw.h>

#include <phKeyStore.h>
#include <phCryptoRng.h>
#include <phCryptoSym.h>
#include <phpalMifare.h>

#ifdef NXPBUILD__PHHAL_HW_SAM

#include "phhalHw_Sam_Cmd.h"
#include "../HSM_AES/phhalHw_Sam_HSM_AES.h"

#include "01_HostCommunication/phhalHw_Sam_Cmd_HC.h"
#include "05_DataProcessing/phhalHw_Sam_Cmd_DP.h"
#include "10_Plus/phhalHw_Sam_Cmd_Plus.h"
#include "16_ISO14443_3/phhalHw_Sam_Cmd_ISO14443_3.h"

/* Global variables */
uint8_t phhalHw_Sam_Cmd_DefaultLe[1U] = { 0x00 };
const uint8_t PH_MEMLOC_CONST_ROM phhalHw_Sam_Cmd_ZeroIV[PH_CRYPTOSYM_AES_BLOCK_SIZE] =
{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };

phStatus_t phhalHw_Sam_Cmd_7816Exchange(phhalHw_Sam_DataParams_t * pDataParams, uint16_t wOption, uint8_t * pData,
    uint16_t wDataLen, uint8_t ** ppResponse, uint16_t * pRespLen)
{
    phStatus_t  PH_MEMLOC_REM wStatus = 0;
    phStatus_t  PH_MEMLOC_REM wStatus1 = 0;
    uint8_t     PH_MEMLOC_REM bCmd = 0;
    uint8_t     PH_MEMLOC_REM bP1 = 0;
    uint8_t     PH_MEMLOC_REM bP2 = 0;
    uint8_t     PH_MEMLOC_REM bFirstResp = PH_OFF;
    uint8_t     PH_MEMLOC_REM bFirstCmd = PH_OFF;
    uint8_t     PH_MEMLOC_REM bLast = PH_OFF;
    uint8_t *   PH_MEMLOC_REM pTmpBuffer = 0;
    uint16_t    PH_MEMLOC_REM wTmpBuffSize = 0;

    uint8_t *   PH_MEMLOC_REM pResponse = 0;
    uint16_t    PH_MEMLOC_REM wRespLen = 0;

    uint16_t    PH_MEMLOC_REM wTxStartPosTmp = 0;
    uint16_t    PH_MEMLOC_REM wTxLengthTmp = 0;

    PH_LOG_HELPER_ALLOCATE_PARAMNAME(BeforeHostSM);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(SentFrmHost);
    PH_LOG_HELPER_ALLOCATE_PARAMNAME(GivenToHost);

    /* reset received length */
    if(pRespLen != NULL)
    {
        *pRespLen = 0;
    }

    /* Check if caller has provided valid RxBuffer */
    if(ppResponse == NULL)
    {
        ppResponse = &pResponse;
    }
    if(pRespLen == NULL)
    {
        pRespLen = &wRespLen;
    }

    /* Get / Check command */
    if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
    {
        /* Check minimum length for first call */
        if(wDataLen < PHHAL_HW_SAM_ISO7816_HEADER_NO_LC_LENGTH)
        {
            return PH_ADD_COMPCODE(PH_ERR_INVALID_PARAMETER, PH_COMP_HAL);
        }

        bCmd = pData[PHHAL_HW_SAM_ISO7816_INS_POS];
    }
    else
    {
        /* Preserved commands(all except exchange) */
        if(pDataParams->wTxBufLen_Cmd != 0)
        {
            /* Ensure SAM command is unequal exchange */
            bCmd = PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE ^ 0x01U;
        }
        else
        {
            /* Exchange command */
            if(pDataParams->wTxBufLen != 0)
            {
                bCmd = PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE;
            }
            /* Everything else is definitely an internal error */
            /*(minimum length for first call is > 0) */
            else
            {
                return PH_ADD_COMPCODE(PH_ERR_INTERNAL_ERROR, PH_COMP_HAL);
            }
        }
    }

    /* Exchange command can use default buffers */
    if(bCmd == PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE)
    {
        /* Reset TxLength */
        if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
        {
            pDataParams->wTxBufLen = 0;
            pDataParams->wTxBufLen_Cmd = 0;
        }

        wTxStartPosTmp = pDataParams->wTxBufStartPos;
        pTmpBuffer = &pDataParams->pTxBuffer[wTxStartPosTmp + pDataParams->wTxBufLen];
        wTmpBuffSize = pDataParams->wTxBufSize - (wTxStartPosTmp + pDataParams->wTxBufLen);
    }
    /* Other commands -> Preserve all buffer data */
    else
    {
        /* Reset TxLength */
        if(!(wOption & PH_EXCHANGE_LEAVE_BUFFER_BIT))
        {
            pDataParams->wTxBufLen_Cmd = 0;
        }

        /* TxBuffer equals RxBuffer */
        if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
        {
            /* Start at TxLength if necessary */
            if((pDataParams->wTxBufStartPos + pDataParams->wTxBufLen) >=
                (pDataParams->wRxBufLen))
            {
                wTxStartPosTmp = pDataParams->wTxBufStartPos + pDataParams->wTxBufLen;
                pTmpBuffer = &pDataParams->pTxBuffer[wTxStartPosTmp + pDataParams->wTxBufLen_Cmd];
                wTmpBuffSize = pDataParams->wTxBufSize - (wTxStartPosTmp + pDataParams->wTxBufLen_Cmd);
            }
            /* Start at RxLength if necessary */
            else
            {
                wTxStartPosTmp = pDataParams->wRxBufLen;
                pTmpBuffer = &pDataParams->pTxBuffer[wTxStartPosTmp + pDataParams->wTxBufLen_Cmd];
                wTmpBuffSize = pDataParams->wTxBufSize - (wTxStartPosTmp + pDataParams->wTxBufLen_Cmd);
            }
        }
        /* Buffers are different */
        else
        {
            wTxStartPosTmp = pDataParams->wTxBufLen;
            pTmpBuffer = &pDataParams->pTxBuffer[wTxStartPosTmp + pDataParams->wTxBufLen_Cmd];
            wTmpBuffSize = pDataParams->wTxBufSize - (wTxStartPosTmp + pDataParams->wTxBufLen_Cmd);
        }
    }

    /* Check for buffer overflow */
    if(wDataLen > wTmpBuffSize)
    {
        return PH_ADD_COMPCODE(PH_ERR_BUFFER_OVERFLOW, PH_COMP_HAL);
    }

    /* Copy TxBuffer contents */
    memcpy(pTmpBuffer, pData, wDataLen);

    /* Exchange uses wTxBufLen */
    if(bCmd == PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE)
    {
        pDataParams->wTxBufLen = pDataParams->wTxBufLen + wDataLen;
        wTxLengthTmp = pDataParams->wTxBufLen;
    }
    /* Other commands use wTxBufLen_Cmd */
    else
    {
        pDataParams->wTxBufLen_Cmd = pDataParams->wTxBufLen_Cmd + wDataLen;
        wTxLengthTmp = pDataParams->wTxBufLen_Cmd;
    }

    /* Shall we already perform the Exchange? */
    if(wOption & PH_EXCHANGE_BUFFERED_BIT)
    {
        return PH_ADD_COMPCODE(PH_ERR_SUCCESS, PH_COMP_HAL);
    }

    /* Set the corresponding logical channel in the CLA Byte */
    pDataParams->pTxBuffer[wTxStartPosTmp + PHHAL_HW_SAM_ISO7816_CLA_POS] |= pDataParams->bLogicalChannel;

    /* Retrieve some command information */
    bCmd = pDataParams->pTxBuffer[wTxStartPosTmp + PHHAL_HW_SAM_ISO7816_INS_POS];
    bP1 = pDataParams->pTxBuffer[wTxStartPosTmp + PHHAL_HW_SAM_ISO7816_P1_POS];
    bP2 = pDataParams->pTxBuffer[wTxStartPosTmp + PHHAL_HW_SAM_ISO7816_P2_POS];

    /* RIGHT PLACE  */
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(BeforeHostSM), pDataParams->pTxBuffer,
        wTxLengthTmp);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Get the Host Protection to be applied for the commands. */
    PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_HSM_AES_GetFirstLastCommand(
        pDataParams,
        bCmd,
        bP1,
        bP2,
        &bFirstCmd,
        &bLast));

    /* Perform encryption of the payload data. */
    if(pDataParams->bCmdSM & PHHAL_HW_SAM_HSM_AES_ENC)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_HSM_AES_Encrypt(
            pDataParams,
            &pDataParams->pTxBuffer[wTxStartPosTmp],
            wTxLengthTmp,
            wTmpBuffSize + wTxLengthTmp,
            &wTxLengthTmp,
            bFirstCmd,
            bLast));
    }

    /* Perform macing of the payload data. */
    if(pDataParams->bCmdSM & PHHAL_HW_SAM_HSM_AES_MAC)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_HSM_AES_AppendMac(
            pDataParams,
            &pDataParams->pTxBuffer[wTxStartPosTmp],
            wTxLengthTmp,
            wTmpBuffSize + wTxLengthTmp,
            &wTxLengthTmp,
            bFirstCmd,
            bLast));
    }

    /* RIGHT PLACE  */
    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, PH_LOG_VAR(SentFrmHost), pDataParams->pTxBuffer,
        wTxLengthTmp);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Increment CmdCtr */
    if(bFirstCmd)
    {
        switch(bCmd)
        {
            case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_HOST:
            case PHHAL_HW_SAM_CMD_INS_SAM_LOCK_UNLOCK:
                break;
            default:
                /* Increment the command counter in case we have a first response */
                ++(pDataParams->wCmd_Ctr);
                break;
        }
    }

    /* Exchange and Decipher command can use default buffers */
    if((bCmd == PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE) ||
        (bCmd == PHHAL_HW_SAM_CMD_INS_SAM_DECIPHER_DATA))
    {
        *ppResponse = &pDataParams->pRxBuffer[pDataParams->wRxBufStartPos];
        wTmpBuffSize = pDataParams->wRxBufSize - pDataParams->wRxBufStartPos;
    }
    /* Other commands -> Preserve all buffer data */
    else
    {
        /* TxBuffer equals RxBuffer */
        if(pDataParams->pTxBuffer == pDataParams->pRxBuffer)
        {
            /* Start after TxBuffer contents */
            if(pDataParams->wTxBufLen > pDataParams->wRxBufLen)
            {
                *ppResponse = &pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wTxBufLen];
                wTmpBuffSize = pDataParams->wTxBufSize - (pDataParams->wTxBufStartPos + pDataParams->wTxBufLen);
            }
            /* Start after RxBuffer contents */
            else
            {
                *ppResponse = &pDataParams->pRxBuffer[pDataParams->wRxBufLen];
                wTmpBuffSize = pDataParams->wRxBufSize - pDataParams->wRxBufLen;
            }
        }
        /* Buffers are different */
        else
        {
            /* Use TxBuffer if it has more space */
            if((pDataParams->wTxBufSize - pDataParams->wTxBufLen) > (pDataParams->wRxBufSize - pDataParams->wRxBufLen))
            {
                *ppResponse = &pDataParams->pTxBuffer[pDataParams->wTxBufStartPos + pDataParams->wTxBufLen];
                wTmpBuffSize = pDataParams->wTxBufSize - (pDataParams->wTxBufStartPos + pDataParams->wTxBufLen);
            }
            /* Else use RxBuffer */
            else
            {
                *ppResponse = &pDataParams->pRxBuffer[pDataParams->wRxBufLen];
                wTmpBuffSize = pDataParams->wRxBufSize - pDataParams->wRxBufLen;
            }
        }
    }

    /* Perform command exchange */
    wStatus = PHBALREG_EXCHANGE(
        pDataParams->pBalDataParams,
        PH_EXCHANGE_DEFAULT,
        &pDataParams->pTxBuffer[wTxStartPosTmp],
        wTxLengthTmp,
        wTmpBuffSize,
        *ppResponse,
        pRespLen);

    /* Reset TxBufferLength */
    if(bCmd == PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE)
    {
        pDataParams->wTxBufLen = 0;
    }
    else
    {
        pDataParams->wTxBufLen_Cmd = 0;
    }

    /* Success check */
    PH_CHECK_SUCCESS(wStatus);

    /* We need at least 2 bytes in the answer */
    if(*pRespLen < PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH)
    {
        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
        pDataParams->bCommandChaining = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;
        pDataParams->bResponseChaining = PHHAL_HW_SAM_HSM_AES_NO_CHAINING;

        /* Remapping of return values */
        return PH_ADD_COMPCODE(PH_ERR_LENGTH_ERROR, PH_COMP_HAL);
    }

    /* catch host protocol error */
    if((*pRespLen == 2U) && ((*ppResponse)[0] == 0x6AU) && ((*ppResponse)[1] == 0x84U))
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_InvalidateKey(pDataParams->pENCCryptoDataParams));
        PH_CHECK_SUCCESS_FCT(wStatus1, phCryptoSym_InvalidateKey(pDataParams->pMACCryptoDataParams));
        pDataParams->bCmdSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
        pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_NO_SM;
    }

    PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_HSM_AES_GetFirstLastResponse(
        pDataParams,
        (*ppResponse)[*pRespLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH],
        (*ppResponse)[*pRespLen - (PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH - 1U)],
        &bFirstResp,
        &bLast));

    /*
    * Special operation for some of the Part 1 commands.
    * In case of error, the error code will be echoed back with MAC applied.
    */
    if(bFirstCmd)
    {
        if(*pRespLen == 11 /* PICC Code + MAC + Status Code */)
        {
            switch(bCmd)
            {
                case PHHAL_HW_SAM_CMD_INS_SAM_AUTHENTICATE_MFP:
                    pDataParams->bRespSM = PHHAL_HW_SAM_HSM_AES_MAC;
                    break;

                default:
                    /* Do nothing. */
                    break;
            }
        }
    }

    if(pDataParams->bRespSM & PHHAL_HW_SAM_HSM_AES_MAC)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_HSM_AES_VerifyRemoveMac(
            pDataParams,
            *ppResponse,
            *pRespLen,
            pRespLen,
            bFirstResp,
            bLast));
    }

    if(pDataParams->bRespSM & PHHAL_HW_SAM_HSM_AES_ENC)
    {
        PH_CHECK_SUCCESS_FCT(wStatus1, phhalHw_Sam_HSM_AES_Decrypt(
            pDataParams,
            *ppResponse,
            *pRespLen,
            pRespLen,
            bFirstResp,
            bLast));
    }

    PH_LOG_HELPER_ADDPARAM_BUFFER(PH_LOG_LOGTYPE_DEBUG, GivenToHost_log, *ppResponse, *pRespLen);
    PH_LOG_HELPER_EXECUTE(PH_LOG_OPTION_CATEGORY_GEN);

    /* Check the return code */
    *pRespLen = *pRespLen - PHHAL_HW_SAM_ISO7816_SW1SW2_LENGTH;
    wStatus = phhalHw_Sam_Utils_ResolveErrorCode(pDataParams, &((*ppResponse)[*pRespLen]));

    /* Increment the length by 2 if its a PL error code. */
    if((wStatus & PH_ERR_MASK) == PHHAL_HW_SAM_ERR_PROGRAMMABLE_LOGIC)
    {
        *pRespLen = 2U;
    }

    /* Always return complete buffer on exchange */
    if(bCmd == PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE)
    {
        *ppResponse = pDataParams->pRxBuffer;
        *pRespLen = *pRespLen + pDataParams->wRxBufStartPos;
        pDataParams->wRxBufLen = *pRespLen;
    }

    /* Special handling for certain wStatus codes */
    switch((wStatus & PH_ERR_MASK))
    {
        case PH_ERR_SUCCESS_INCOMPLETE_BYTE:
            /* Retrieve number of bits from second byte of wStatus code */
            pDataParams->wAdditionalInfo = ((*ppResponse)[*pRespLen + 1U]);
            break;

        case PHHAL_HW_SAM_ERR_DESFIRE_GEN:
            /* Retrieve return code from card from first data byte */
            if(*pRespLen == 1)
            {
                pDataParams->wAdditionalInfo = (*ppResponse)[0];
            }
            else
            {
                if(*pRespLen == 2U)
                {
                    pDataParams->wAdditionalInfo = ((uint16_t) ((*ppResponse)[0]) << 8U) | (uint16_t) (*ppResponse)[1U];
                }
            }
            break;
        default:
            break;
    }

    return wStatus;
}

void phhalHw_Sam_Cmd_UpdateLC(uint8_t * pData, uint8_t bDataLen, uint8_t bLE_Available)
{
    uint8_t PH_MEMLOC_REM bISO7816_Hdr_Len = 5U;
    uint8_t PH_MEMLOC_REM bLC = 0;

    /* Compute Actual LC. */
    bLC = (uint8_t) (bDataLen - (bISO7816_Hdr_Len + (bLE_Available ? 1U : 0)));

    /* Update LC. */
    pData[PHHAL_HW_SAM_ISO7816_LC_POS] = bLC;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
