/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) Originality Check command implementation of Reader
 * Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7230 $
 * $Date: 2025-03-14 10:27:23 +0530 (Fri, 14 Mar 2025) $
 */

#include "phhalHw_Sam_Cmd_OC.h"

#ifdef NXPBUILD__PHHAL_HW_SAM

#include <phCryptoRng.h>
#include "../phhalHw_Sam_Cmd.h"

phStatus_t phhalHw_Sam_Cmd_SAM_ISOInternalAuthenticate(phhalHw_Sam_DataParams_t * pDataParams, uint8_t bOption,
    uint8_t * pOptsA, uint8_t bOptsALen, uint8_t * pRndA, uint8_t * pRndALen, uint8_t ** ppResponse,
    uint16_t * pRespLen)
{
    phStatus_t PH_MEMLOC_REM wStatus = 0;
    uint8_t    PH_MEMLOC_REM bCmdLen = 0;

    uint8_t   PH_MEMLOC_REM aTL[] = { 0x00, 0x00 };

    uint8_t    PH_MEMLOC_REM * pRnd = NULL;
    uint8_t    PH_MEMLOC_REM bRndLen = 0;

    uint8_t *  PH_MEMLOC_REM pResponse = NULL;
    uint16_t   PH_MEMLOC_REM wRespLen = 0;

    /* Parameter validation. */
    PH_ASSERT_NULL_DATA_PARAM(pDataParams, PH_COMP_HAL);
    if(bOptsALen != 0)
        PH_ASSERT_NULL_PARAM(pOptsA, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRndA, PH_COMP_HAL);
    PH_ASSERT_NULL_PARAM(pRespLen, PH_COMP_HAL);

    /* Reset the command buffer and its length. */
    PHHAL_HW_SAM_CLEAR_CMD_BUFFER();

    /* Frame SAM_ISOInternalAuthenticate command information . */
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_CLA_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_CMD_INS_ISO_INTERNAL_AUTHENTICATE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_P1_BYTE;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_ORIGINALITY_KEY;
    PHHAL_HW_SAM_CMD_BUFFER[bCmdLen++] = PHHAL_HW_SAM_ISO7816_DEFAULT_LC_BYTE;

    /* Add command information to exchange buffer. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_FIRST,
        PHHAL_HW_SAM_CMD_BUFFER,
        bCmdLen,
        NULL,
        NULL));

    /* Add Tag, Length and Value (OpstA) to Exchange buffer */
    if(bOptsALen != 0)
    {
        /* Set the TL to be added */
        aTL[0U] = 0x80U;
        aTL[1U] = bOptsALen;

        /* Add Tag and Length to Exchange buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            aTL,
            (uint16_t)sizeof(aTL),
            NULL,
            NULL));

        /* Add OptsA to Exchange buffer */
        PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
            pDataParams,
            PH_EXCHANGE_BUFFER_CONT,
            pOptsA,
            bOptsALen,
            NULL,
            NULL));
    }

    /* Generate RndA */
    if(bOption == PHHAL_HW_GENERATE_CHALLENGE_INTERNAL)
    {
        PH_CHECK_SUCCESS_FCT(wStatus, phCryptoRng_Rnd(pDataParams->pCryptoRngDataParams,
            PHHAL_HW_RND_CHALLENGE_LENGTH, pRnd));
        bRndLen = PHHAL_HW_RND_CHALLENGE_LENGTH;
    }
    else
    {
        memcpy(pRnd, pRndA, *pRndALen);
        bRndLen = *pRndALen;
    }

    /* Set the AuthDOHdr to be added */
    aTL[0U] = 0x7CU;
    aTL[1U] = (uint8_t) (bRndLen + 2U /* RndA Tag and Length */);

    /* Add AuthDOHdr to Exchange buffer */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        aTL,
        (uint16_t)sizeof(aTL),
        NULL,
        NULL));

    /* Set the RndA TL to be added */
    aTL[0U] = 0x81U;
    aTL[1U] = bRndLen;

    /* Add RndA TL to Exchange buffer */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        aTL,
        (uint16_t)sizeof(aTL),
        NULL,
        NULL));

    /* Add RndA to Exchange buffer */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_CONT,
        pRnd,
        bRndLen,
        NULL,
        NULL));

    /* Update LC */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Utils_UpdateLc(pDataParams));

    /* Exchange SAM_IsoInternalAuthenticate information to SAM. */
    PH_CHECK_SUCCESS_FCT(wStatus, phhalHw_Sam_Cmd_7816Exchange(
        pDataParams,
        PH_EXCHANGE_BUFFER_LAST,
        phhalHw_Sam_Cmd_DefaultLe,
        1U,
        &pResponse,
        &wRespLen));

    /* Copy RndA to parameter */
    if(bOption == PHHAL_HW_GENERATE_CHALLENGE_INTERNAL)
    {
        memcpy(pRndA, pRnd, bRndLen);
        *pRndALen = bRndLen;
    }

    /* Remove AuthDOHdr from the response. */
    ppResponse[0] = &pResponse[2U];
    *pRespLen = (uint8_t) (wRespLen - 2U);

    return wStatus;
}

#endif /* NXPBUILD__PHHAL_HW_SAM */
