/*
 * Copyright 2025 NXP
 * NXP Confidential and Proprietary.
 * This software is owned or controlled by NXP and may only be used strictly
 * in accordance with the applicable license terms. By expressly accepting
 * such terms or by downloading, installing, activating and/or otherwise using
 * the software, you are agreeing that you have read, and that you agree to
 * comply with and are bound by, such license terms. If you do not agree to be
 * bound by the applicable license terms, then you may not retain, install,
 * activate or otherwise use the software.
 */

/** \file
 * SAM (AV4 and future SAM's) ISO14443-3 command implementation of Reader Library Framework.
 * $Author: Rajendran Kumar (nxp99556) $
 * $Revision: 7467 $
 * $Date: 2025-08-31 13:27:22 +0530 (Sun, 31 Aug 2025) $
 */


#ifndef PHHALHW_SAM_CMD_ISO14443_3_H
#define PHHALHW_SAM_CMD_ISO14443_3_H

#include <ph_Status.h>
#include <phhalHw.h>

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

#ifdef NXPBUILD__PHHAL_HW_SAM

/** \addtogroup phhalHw_Sam_Cmd
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_3 15_ISO14443_3
 * \brief SAM commands used for ISO14443 layer 3 communication in X-Mode.
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_3_Defines Defines
 * \brief Definitions for ISO14443 layer 3 commands
 * @{
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_3_Defines_CommandCodes CommandCodes
 * \brief Definitions for Instruction codes
 * @{
 */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_3_REQUEST_WAKEUP                          0x25U   /**< SAM Insturction code for ISO14443-3_Request_Wakeup command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_3_ANTICOLLSION_SELECT                     0x93U   /**< SAM Insturction code for ISO14443-3_Anticollision_Select command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_3_ACTIVATE_IDLE                           0x26U   /**< SAM Insturction code for ISO14443-3_ActivateIdle command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_3_ACTIVATE_WAKEUP                         0x52U   /**< SAM Insturction code for ISO14443-3_ActivateWakeup command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_3_HALTA                                   0x50U   /**< SAM Insturction code for ISO14443-3_HaltA command. */
#define PHHAL_HW_SAM_CMD_INS_ISO14443_3_TRANSPARENT_EXCHANGE                    0x7EU   /**< SAM Insturction code for ISO14443-3_TransparentExchange command. */
/**
 * end of group phhalHw_Sam_Cmd_ISO14443_3_Defines_CommandCodes
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_3_Defines_L3 L3Command
 * \brief Definitions for ISO14443 - 3 L3 command code.
 * To be used with \ref phhalHw_Sam_Cmd_X_ISO14443_3_RequestA_Wakeup
 * "ISO14443_3_RequestA_Wakeup" interface
 * @{
 */
#define PHHAL_HW_SAM_CMD_ISO14443_3_CMD_REQA                                    0x26U   /**<  Request (0x26) command code. */
#define PHHAL_HW_SAM_CMD_ISO14443_3_CMD_WUPA                                    0x52U   /**<  Wakeup (0x52) command code. */
/**
 * end of group phhalHw_Sam_Cmd_ISO14443_3_Defines_L3
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_3_Defines_SelectCode SelectCode
 * \brief Definitions for ISO14443 - 3 L3 Select (Cascade Level) codes.
 * To be used with \ref phhalHw_Sam_Cmd_X_ISO14443_3_AnticollisionSelect
 * "ISO14443_3_AnticollisionSelect" interface
 * @{
 */
#define PHHAL_HW_SAM_CMD_ISO14443_3_SelectCode_CASCADE_LEVEL_1                  0x93U   /**<  Option flag for Cascade Level 1 */
#define PHHAL_HW_SAM_CMD_ISO14443_3_SelectCode_CASCADE_LEVEL_2                  0x95U   /**<  Option flag for Cascade Level 2 */
#define PHHAL_HW_SAM_CMD_ISO14443_3_SelectCode_CASCADE_LEVEL_3                  0x97U   /**<  Option flag for Cascade Level 3 */
/**
 * end of group phhalHw_Sam_Cmd_ISO14443_3_Defines_SelectCode
 * @}
 */

/** \defgroup phhalHw_Sam_Cmd_ISO14443_3_Defines_ActivateIdle ActivateIdle
 * \brief Definitions for ISO14443 - 3 L3 Activate Idle.
 * To be used with \ref phhalHw_Sam_Cmd_X_ISO14443_3_ActivateIdle
 * "ISO14443_3_ActivateIdle" interface
 * @{
 */
#define PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_DEFAULT                        0x00U   /**<  Default option mask for ActivateIdle. */
#define PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_ATQA                     0x01U   /**<  Option flag for applying the ATQA filter. */
#define PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_SAK                      0x02U   /**<  Option flag for applying the SAK filter. */
#define PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_TIME                     0x04U   /**<  Option flag for setting the time to wait. */
/**
 * end of group phhalHw_Sam_Cmd_ISO14443_3_Defines_ActivateIdle
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd_ISO14443_3_Defines
 * @}
 */

/**
 * \brief Perform a request or wake-up command (type A).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pAtqa is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_RequestA_Wakeup(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bCmdCode,                                                               /**< [In] Command code.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_ISO14443_3_CMD_REQA
                                                                                         *                 "RequestA"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_ISO14443_3_CMD_WUPA
                                                                                         *                 "WupaA"
                                                                                         */
        uint8_t * pAtqa                                                                 /**< [Out] AtqA returned by the card. The buffer has to be
                                                                                         *         2 bytes long.
                                                                                         */
    );

/**
 * \brief Perform bit-wise anticollision and select. The anticollision and the following
 * select are performed according to the select code in the data field.
 *  - The selection can be carried out for a variable amount of cascade levels. The select codes
 *    have to be listed in the data field subsequently. The MIFARE SAM AV4 will take the parameters
 *    exactly and use them as select code. Therefore to fully select a card with a triple UID, the
 *    data field has to be of three bytes length indicating 93h, 95h and 97h whereas the data field
 *    has to be of one byte length indicating 93h if a single size UID card are selected.
 *  - If the select code indicates a cascade level of 93h and 95h, and the UID of the card consists
 *    only of four bytes, the MIFARE SAM AV4 will exit the command and return the SAK and the UID of
 *    the card.
 *  - If the select code indicates a cascade level of 93h, and the UID consists of more than four bytes
 *    the MIFARE SAM will also exit the command and return the SAK and the first three bytes of the UID
 *    but indicates with a special return code the incompleteness of the UID separately. The caller has
 *    then to take care about continuing the procedure on his own by calling the command once more with
 *    a higher select code. The already returned UID bytes will not be returned a second time. The same
 *    applies for a select code of 95h if the UID is of ten bytes length (suggest that a selection with
 *    code 93h is carried out previously).
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pSelCodes, \b pSak, \b pUid and
 *                                      \b pUidLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_AnticollisionSelect(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pSelCodes,                                                            /**< [In] Buffer containing the SEL sequence bytes.
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_ISO14443_3_SelectCode_CASCADE_LEVEL_1
                                                                                         *                 "Cascade Level 1 (0x93)"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_ISO14443_3_SelectCode_CASCADE_LEVEL_2
                                                                                         *                 "Cascade Level 2 (0x95)"
                                                                                         *          - \ref PHHAL_HW_SAM_CMD_ISO14443_3_SelectCode_CASCADE_LEVEL_3
                                                                                         *                 "Cascade Level 3 (0x97)"
                                                                                         */
        uint8_t bSelCodesLen,                                                           /**< [In] Length of bytes available in \b pSelCodes buffer. (01h to 03h). */
        uint8_t * pSak,                                                                 /**< [Out] Buffer to the 1 byte Select Acknowledge reveived from card. */
        uint8_t * pUid,                                                                 /**< [Out] Buffer containing the received UID. This buffer has to be
                                                                                         *         10 bytes long.
                                                                                         */
        uint8_t * pUidLen                                                               /**< [Out] Length of bytes available in \b pUid buffer. */
    );

/**
 * \brief Perform one or several request - anticollision - select sequences and returns the
 * SAK and the UID of the selected PICC(s).
 *  - The ATQA is returned for every request issued, this means for every newly activated card.
 *    Due to the fact that the resulting ATQA is the OR-function of all ATQAs, the value may
 *    change frequently.
 *  - The command offers several different possibilities for activating cards:
 *      - Wait for a definable number of cards and return if this number of cards has been activated
 *      - Either try only one activation sequence and return if no PICC is detected, wait for a
 *        defined amount of time, or wait until the expected number of PICCs was activated
 *      - Apply a filter for the ATQA response of the PICC(s)
 *      - Apply a filter for the SAK response of the PICC(s)
 *  - If more than one PICC are activated, the MIFARE SAM puts all activated PICC(s) into the Halt
 *    state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pAtqaIn, \b pSakIn and
 *                                      \b pRespLen are NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_ActivateIdle(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t bOption,                                                                /**< [In] Option to be used for P2 and internal to library.
                                                                                         *        Can be combined by using bitwise OR operator.
                                                                                         *          - Option to be used for P2 byte information.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_DEFAULT
                                                                                         *                     "Do not apply SAK and ATQA filter"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_ATQA
                                                                                         *                     "Apply SAK filter"
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_SAK
                                                                                         *                     "Apply ATQA filter"
                                                                                         *
                                                                                         *          - Option used by Libray internally.
                                                                                         *              - \ref PHHAL_HW_SAM_CMD_ISO14443_3_ACTIVATEIDLE_APPLY_TIME
                                                                                         *                     "Exchange Short Time to wait" information to SAM.
                                                                                         */
        uint8_t bNumCards,                                                              /**< [In] Holds the number of cards to activate. */
        uint16_t wTime,                                                                 /**< [In] Time to wait for a card. */
        uint8_t * pAtqaIn,                                                              /**< [In] Buffer containing the AtqA filter. Should contain the following,
                                                                                         *        Mask (1 byte) || Value (1 byte) || Mask2 (1 byte) || Value2 (1 byte)
                                                                                         */
        uint8_t * pSakIn,                                                               /**< [In] Buffer containing the Sak filter. Should contain the following,
                                                                                         *        Mask (1 byte) || Value (1 byte)
                                                                                         */
        uint8_t ** ppResponse,                                                          /**< [Out] Buffer containing the received data.
                                                                                         *          - ATQA (Answer To re-Quest type A): 2 bytes
                                                                                         *          - SAK (Select AcKnowledge)        : 1 byte
                                                                                         *          - Len (Length of card UID)        : 4, 7, 10 bytes
                                                                                         *          - UID (Unique Identified)         : Card UID
                                                                                         */
        uint16_t * pRespLen                                                             /**< [Out] Length of bytes available in \b ppResponse buffer. */
    );

/**
 * \brief Perform reactivates and selects a PICC that has previously been set to Halt state.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER    \b pUid is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_ActivateWakeUp(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pUid,                                                                 /**< [In] UID of the card to reactivate. */
        uint8_t bUidLen                                                                 /**< [In] Length of bytes available in \b pUid buffer
                                                                                         *        (4, 7 or 10 bytes).
                                                                                         */
    );

/**
 * \brief Puts a selected card into Halt state
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS              Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS  \b pDataParams is NULL.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_HaltA(
        phhalHw_Sam_DataParams_t * pDataParams                                          /**< [In] Pointer to this layer's parameter structure. */
    );

/**
 * \brief Perform exchange bytes/bits transparently with a PICC. The MIFARE SAM will take
 * the user data and send it without changing, inserting or appending any content to the contactless
 * card. Appending of a CRC, time-out settings, etc. have to be configured by directly writing the
 * RC52x/RC663 registers. Take into account that switching settings of the Reader IC influence all
 * subsequent MIFARE SAM  commands proposing the correct Reader IC settings, that is ISO14443-
 * 4_Exchange.
 *  - If bits are exchanged, parameter P1 indicates the number of bits valid in the last input byte.
 *    A value of 00h indicates that the full last byte are sent to the PICC. Consequently, for using
 *    the function as pure byte exchange, P1 has to be set to zero. In case that only a few bits of
 *    the last byte are sent, the MIFARE SAM will take the indicated number of bits beginning with
 *    the LSBit of the last input byte.
 *  - Take into account that this function cannot be used for bit wise anticollision since the MIFARE
 *    SAM will react on a detected collision with an error and not return any bytes (bits).
 *  - If the card responds with single bits, the MIFARE SAM will indicate the number of valid bits
 *    within the last returned byte in the lower nibble of SW2. SW2 = 00h indicates that the full
 *    byte was received from the PICC. The last returned data byte contains the valid bits according
 *    to the indicated amount starting with the LSBit.
 *
 * \return Status code
 * \retval #PH_ERR_SUCCESS                  Operation successful.
 * \retval #PH_ERR_INVALID_DATA_PARAMS      \b pDataParams is NULL.
 * \retval #PH_ERR_INVALID_PARAMETER        \b pTxBuf, \b pRxLen and
 *                                          \b pRxBitLen are NULL.
 * \retval #PH_ERR_SUCCESS_INCOMPLETE_BYTE  Number of bits valid in the last returned
 *                                          data byte in lower nibble of SAM status SW2.
 * \retval Other Depending on implementation and underlaying component.
 */
phStatus_t phhalHw_Sam_Cmd_X_ISO14443_3_TransparentExchange(
        phhalHw_Sam_DataParams_t * pDataParams,                                         /**< [In] Pointer to this layer's parameter structure. */
        uint8_t * pTxBuf,                                                               /**< [In] Buffer containing the data to be sent. The buffer
                                                                                         *        has to be \b bTxLen bytes long.
                                                                                         */
        uint8_t bTxLen,                                                                 /**< [In] Length of bytes available in \b pTxBuf buffer. */
        uint8_t bTxBitLen,                                                              /**< [In] Number valid bits (0x00 - 0x07) in the last byte
                                                                                         *        of the \b pTxBuf buffer. If set to 00h all bits
                                                                                         *        are valid.
                                                                                         */
        uint8_t ** ppRxBuf,                                                             /**< [Out] Buffer containing the received data from PICC. */
        uint16_t * pRxLen,                                                              /**< [Out] Length of bytes available in \b ppRxBuf buffer. */
        uint8_t * pRxBitLen                                                             /**< [Out] Amount of valid bits in the last byte in case of
                                                                                         *         an incomplete byte.
                                                                                         */
    );

/**
 * end of group phhalHw_Sam_Cmd_ISO14443_3
 * @}
 */

/**
 * end of group phhalHw_Sam_Cmd
 * @}
 */

#endif /* NXPBUILD__PHHAL_HW_SAM */

#ifdef __cplusplus
} /* Extern C */
#endif

#endif /* PHHALHW_SAM_CMD_ISO14443_3_H */
